%  Copyright (C) 2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs repair}
\begin{code}
module Repair ( repair ) where
import DarcsCommands
import DarcsArguments
import Directory
import IO
import System
import Repository
import Patch
import PatchInfo
import SlurpDirectory
import Diff
import Monad
import Test
import RepoPrefs ( filetype_function )
\end{code}

\options{repair}
\begin{code}
repair_description = "Repair a corrupted repository.
"
\end{code}
\haskell{repair_help}

\begin{code}
repair_help = "Repair attempts to fix corruption that may have entered your
repository.  You should always run repair on a copy of the
repository, since it could make things worse.
"
\end{code}

\begin{code}
repair = DarcsCommand {command_name = "repair",
                       command_help = repair_help,
                       command_description = repair_description,
                       command_extra_args = 0,
                       command_command = repair_cmd,
                       command_prereq = am_in_repo,
                       command_get_arg_possibilities = return [],
                       command_argdefaults = nodefaults,
                       command_darcsoptions = [verbose]}
\end{code}

Darcs repair works using a ``sandwich'' method.  It assumes that the state
of the tree is known both before and after a sequence of patches.  If then
proceeds from both directions seeing if the patches on the edges apply
cleanly, in which case they can be removed from the sandwich.  If neither
end patch applies cleanly, one of them will have to be fixed.  The patch
whose fix is smaller is fixed and removed from the sandwich.

FIXME: Repair isn't actually implemented yet!

\begin{code}
odir = "/tmp/darcs_repair_old"
ndir = "/tmp/darcs_repair_new"

repair_cmd opts args = do
  am_verbose <- return $ Verbose `elem` opts
  patches <- read_repo "."
  recorded <- slurp_recorded "."
  formerdir <- getCurrentDirectory
  createDirectory ndir `catch` (\e->
    do putStr $ "Error:  try removing directory `"++ndir++"'\n"
       exitWith $ ExitFailure 1)
  setCurrentDirectory odir
  slurp_write recorded
  createDirectory odir `catch` (\e->
    do putStr $ "Error:  try removing directory `"++odir++"'\n"
       exitWith $ ExitFailure 1)
  setCurrentDirectory odir
  fixed <- sandwich_patches am_verbose (reverse $ concat patches) odir ndir
  return ()
\end{code}

\begin{code}
sandwich_patches :: Bool -> [(PatchInfo, Maybe Patch)]
                 -> FilePath -> FilePath -> IO [(PatchInfo, Maybe Patch)]
sandwich_patches v [] o n = return []
sandwich_patches v [(pi,mp)] o n = do
  so <- slurp o
  sn <- slurp n
  ftf <- filetype_function
  case mp of
    Nothing -> error $ "Problem reading patch "++show pi
    Just po ->
        case apply_to_slurpy po so of
        Nothing -> fixed
        Just sn' ->
            case diff ftf sn sn' of
            Just _ -> fixed
            Nothing ->
                case diff ftf sn' sn of
                Just _ -> fixed
                Nothing -> return [(pi,mp)]
        where fixed = case diff ftf so sn of
                      Nothing -> error $ "Couldn't diff in sandwich_patches"
                      Just p -> return [(pi, Just $ infopatch pi p)]
--sandwich_patches v ps o n = do

fix_patch :: Bool -> (PatchInfo, Maybe Patch) -> FilePath
          -> IO (Maybe Patch)
-- return val of Nothing means there was nothing wrong with the patch.
fix_patch v (pi,mp) o = do
  s <- slurp o
  case mp of
    Nothing -> error $ "fix_patch couldn't read patch "++ show pi
    Just p ->
        case apply_to_slurpy p s of
        Just _ -> return Nothing
        Nothing -> return $ kludge_to_slurpy p s
\end{code}
