%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs push}
\begin{code}
module Push ( push ) where
import IO
import System
import Monad ( when, unless )
import List ( intersect, elem )

import DarcsCommands
import DarcsArguments
import Directory
import Repository
import Patch
import PatchInfo
import SlurpDirectory
import DarcsArguments
import RegexString
import RepoPrefs
import External ( signString, sendEmail, fetchFile )
import Lock ( withLock )
import Pull ( select_patches )
import Depends ( get_common_and_uncommon )
\end{code}
\begin{code}
push_description =
 "Push patches at another repo."
\end{code}

\options{push}

\haskell{push_help}
\begin{code}
push_help =
 "Push is used to prepare a bundle of patches to send to another repository.
Push accepts the URL of the repository as an argument, and when called without
an argument, pull will use the most recent repository that was either pushed
to or pulled from.
"
\end{code}
\begin{code}
push = DarcsCommand {command_name = "push",
                     command_help = push_help,
                     command_description = push_description,
                     command_extra_args = 1,
                     command_command = push_cmd,
                     command_prereq = am_in_repo,
                     command_get_arg_possibilities = get_preflist "repos",
                     command_argdefaults = lastrepo,
                     command_darcsoptions = [verbose,patchname_option,
                                             all_patches,author,
                                             target,output,sign,
                                             edit_description]}
\end{code}
\begin{code}
push_cmd opts [repodir] = withLock "./_darcs/lock" $ do
  am_verbose <- return $ Verbose `elem` opts
  repovalid <- is_repo repodir
  unless repovalid $ do putStr $ "Bad repo directory: "++repodir++"\n"
                        exitWith $ ExitFailure 1
  set_lastrepo repodir
  aminrepo <- is_repo "."
  unless aminrepo $ do putStr $ "Aaack, I'm not in a repo!\n"
                       exitWith $ ExitFailure 1
  when am_verbose $ putStr "About to read in the inventories...\n"
  them <- read_repo repodir
  us <- read_repo "."
  when am_verbose $ putStr "Finished reading in the inventories...\n"
  case get_common_and_uncommon (us, them) of
    (common, us', them') -> do
      when am_verbose $ putStr $
                          "We the following new (to them) patches:\n" ++
                          format_inventory (head us')
      when (us' == [[]]) $ do
              putStr "No recorded local changes to push!\n"
              exitWith ExitSuccess
      to_be_pushed <- select_patches "push" opts $ reverse $ head us'
      when (to_be_pushed == []) $ do
          putStr "You don't want to push any patches, and that's fine with me!\n"
          exitWith ExitSuccess
      wantfile <- wants_output opts
      author <- get_author opts
      bundle <- if Sign `elem` opts
                then signString $ make_patch_bundle common to_be_pushed
                else return $ make_patch_bundle common to_be_pushed
      when (length bundle == 0) $
           putStr "Nasty hack to wait for gpg to finish running.\n"
      if wantfile
         then do
           fname <- get_output opts
           writeFile fname bundle
         else do
           thetarget <- get_target opts repodir
           mailcontents <- get_description opts
           sendEmail author thetarget "darcs patch"
                $ make_email mailcontents bundle "aaack"
make_email contents bundle boundary =
    "Content-Type: multipart/mixed; boundary=\""++boundary++"\"\n"++
    "\n--"++boundary++"\n\n"++
    contents++"\n"++
    "\n--"++boundary++"\n"++
    "Content-Type: text/x-darcs-patch\n"++
    "Content-Description: A darcs patch for your repository!\n\n"++
    bundle ++
    "\n--"++boundary++"\n"++
    "\n.\n\n"
make_patch_bundle common to_be_pushed =
    case sequence $ map snd to_be_pushed of
    Just ps ->
      "\nContext:\n\n"++
      (unlines $ map show $ common) ++
      "\nNew patches:\n\n"++
      (unlines $ map show ps) ++ "\n\n"
\end{code}

The \verb!--output! and \verb!--email! flags determine what darcs does with
the patch bundle after creating it.  If you provide an \verb!--output!
argument, the patch bundle is saved to that file.  If you give an
\verb!--email! argument, the bundle of patches is emailed to that person.
If you don't provide either, darcs will prompt you for an email recipient.

\begin{code}
wants_output :: [DarcsFlag] -> IO Bool
wants_output (Output a:_) = return True
wants_output (_:flags) = wants_output flags
wants_output [] = return False
get_output :: [DarcsFlag] -> IO String
get_output (Output a:_) = return a
get_output (_:flags) = get_output flags
\end{code}

\begin{code}
get_target :: [DarcsFlag] -> String -> IO String
get_target (Target a:_) r = return a
get_target (_:flags) r = get_target flags r
get_target [] r = do
    email <- fetchFile $ r++"/_darcs/prefs/email"
    if '@' `elem` email then return $ head $ lines email
       else do putStr "What is the target email address? "
               hFlush stdout
               getLine
\end{code}

The \verb!--patchname! argument can be used to specify a regexp, which
should be of the extended type used by \verb!egrep!.  If this option is
used, only patches which match this regexp (along with their dependencies)
are considered for pushing.

\begin{code}
mymatch ('^':r) s = matchRegex (mkRegex ('^':r)) s /= Nothing
mymatch r s = matchRegex (mkRegex (".*"++r)) s /= Nothing
get_patchname :: [DarcsFlag] -> (String -> Bool)
get_patchname (PatchName n:_) = mymatch n
get_patchname (_:flags) = get_patchname flags
get_patchname [] = \_ -> True
\end{code}

If you want to include a description or explanation along with the bundle
of patches, you need to specify the \verb!--edit-description! flag, which
will cause darcs to open up an editor with which you can compose an email
to go along with your patches.

\begin{code}
get_description :: [DarcsFlag] -> IO String
get_description opts =
    if EditDescription `elem` opts
    then do writeFile ".darcs-temp-mail" "Insert patch description here"
            edit_file ".darcs-temp-mail"
            readFile ".darcs-temp-mail"
    else return "No patch description."
\end{code}
