%  Copyright (C) 2002,2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\begin{code}
module DarcsCommands ( DarcsCommand( DarcsCommand, command_name,
                                     command_help, command_description,
                                     command_darcsoptions, command_command,
                                     command_prereq,
                                     command_extra_args,
                                     command_argdefaults,
                                     command_get_arg_possibilities ),
                       usage, run_command, command_options, nodefaults,
                     ) where
import System.Console.GetOpt
import System

import DarcsArguments
import ArgumentDefaults ( get_default_flag )
\end{code}

The general format of a darcs command is
\begin{verbatim}
% darcs COMMAND OPTIONS ARGUMENTS ...
\end{verbatim}
Here {\tt COMMAND} is a command such as {\tt add} or {\tt record}, which of
course may have one or more arguments.  Options have the form
\verb!--option! or \verb!-o!, while arguments vary from command to
command.  There are many options which are common to a number of different
commands, which will be summarized here.

\input{DarcsArguments.lhs}

\begin{code}
data DarcsCommand =
    DarcsCommand {command_name, command_help, command_description :: String,
                  command_extra_args :: Int,
                  command_command :: [DarcsFlag] -> [String] -> IO (),
                  command_prereq :: IO Bool,
                  command_get_arg_possibilities :: IO [String],
                  command_argdefaults :: [String] -> IO [String],
                  command_darcsoptions :: [DarcsOption]}
command_options :: DarcsCommand -> [OptDescr DarcsFlag]
command_options c = map option_from_darcsoption $ command_darcsoptions c
nodefaults :: [String] -> IO [String]
nodefaults as = return as
\end{code}


\begin{code}
usage :: [DarcsCommand] -> String
usage cs = "Usage: darcs COMMAND ...\nCommands:\n" ++ usage_helper cs

usage_helper :: [DarcsCommand] -> String
usage_helper [] = ""
usage_helper (c:cs) = "  "++pad_spaces (command_name c) 14 ++
                      command_description c ++ "\n" ++ usage_helper cs

pad_spaces :: String -> Int -> String
pad_spaces s n
    | length s < n = pad_spaces (s++" ") n
    | otherwise = s
\end{code}

\begin{comment}

This is the actual heavy lifter code, which is responsible for parsing the
arguments and then running the command itself.

\end{comment}
\begin{code}
run_command :: DarcsCommand -> [String] -> IO ()

run_command cmd args =
    case getOpt RequireOrder
             (option_from_darcsoption help:
              option_from_darcsoption list_options:options) args of
    (opts,extra,ermsgs) ->
      case opts of
       [Help] -> putStr $ get_command_help cmd
       (ListOptions:_) -> do
           args <- command_get_arg_possibilities cmd
           putStr $ get_command_options cmd++unlines args++"\n"
       otherwise -> consider_running cmd opts extra
    where options = command_options cmd
          extra_argnum = command_extra_args cmd

consider_running :: DarcsCommand -> [DarcsFlag] -> [String] -> IO ()
consider_running cmd opts old_extra = do
  extra <- (command_argdefaults cmd) old_extra
  if command_extra_args cmd < 0
    then (command_command cmd) opts extra
    else if length extra /= command_extra_args cmd
         then do putStr $ "Bad argument: "++unwords extra++"\n"++
                        get_command_help cmd
                 exitWith $ ExitFailure 1
         else do specops <- add_command_defaults cmd opts
                 (command_command cmd) specops extra


add_command_defaults :: DarcsCommand -> [DarcsFlag] -> IO [DarcsFlag]
add_command_defaults cmd already =
    acd (command_name cmd) already (command_darcsoptions cmd)
acd :: String -> [DarcsFlag] -> [DarcsOption] -> IO [DarcsFlag]
acd _ fs [] = return fs
acd c fs (dao@(DarcsNoArgOption _ _ f _):dos) =
    if f `elem` fs then acd c fs dos
                   else do fs' <- get_default_flag c dao
                           acd c (fs++fs') dos
acd c fs (dao@(DarcsArgOption _ _ f _ _):dos) =
    if f `isin` fs then acd c fs dos
                   else do fs' <- get_default_flag c dao
                           acd c (fs++fs') dos
f `isin` [] = False
f `isin` (f':fs) | f' `isa` f = True
                 | otherwise = f `isin` fs

get_command_options :: DarcsCommand -> String
get_command_options cmd =
    (get_options_options options) ++ "--help\n"
    where options = command_options cmd

get_options_options :: [OptDescr DarcsFlag] -> String
get_options_options [] = ""
get_options_options (o:os) =
    get_long_option o ++"\n"++ get_options_options os

get_long_option :: OptDescr DarcsFlag -> String
get_long_option (Option _ [] _ _) = ""
get_long_option (Option a (o:os) b c) = "--"++o++
                 get_long_option (Option a os b c)


get_command_help :: DarcsCommand -> String
get_command_help cmd =
    (usageInfo
     ("Usage: darcs "++name++" options ...\n"++ description++"\nOptions:")
     (options++[option_from_darcsoption help]))
    ++ help_message
    where options = command_options cmd
          name = command_name cmd
          description = command_description cmd
          help_message = command_help cmd
\end{code}
