%  Copyright (C) 2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\begin{code}
module ArgumentDefaults ( get_default_flag,
                        ) where
import Monad ( liftM )
import Maybe ( catMaybes )

import System.Console.GetOpt
import DarcsArguments
import RepoPrefs

get_default_flag :: String -> DarcsOption -> IO [DarcsFlag]
\end{code}

\subsection{defaults}

Default values for darcs commands can be configured on a per-repository
basis by editing (and possibly creating) the \verb!_darcs/prefs/defaults!
file.  Each line of this file has the following form:
\begin{verbatim}
COMMAND FLAG VALUE
\end{verbatim}
where \verb!COMMAND! is either the name of the command to which the default
applies, or \verb!ALL! to indicate that the default applies to all commands
accepting that flag.  The \verb!FLAG! term is the name of the long argument
option without the ``\verb!--!'', i.e. \verb!verbose! rather than
\verb!--verbose!.  Finally, the \verb!VALUE! option can be omitted if the
flag is one such as \verb!verbose! that doesn't involve a value.

For example, if your system clock is bizarre, you could instruct darcs to
always ignore the file modification times by adding the following line to
your \verb!_darcs/prefs/defaults! file.  (Note that this would have to be
done for each repository!)
\begin{verbatim}
ALL ignore-times
\end{verbatim}

If you never want to run a test when recording to a particular repository
(but still want to do so when running check on that repo), and like to name
all your patches ``Stupid patch'', you could use the following:
\begin{verbatim}
record no-test
record patch-name Stupid patch
\end{verbatim}

\begin{code}
get_default_flag com flag = do
    defs <- get_defaults
    case gda com flag defs of
      [] -> return $ gda "ALL" flag defs
      f -> return $ f

gda :: String -> DarcsOption -> [(String,String,String)] -> [DarcsFlag]
gda com (DarcsNoArgOption s [fl] c h) defs =
    case filter (\ (c,f,_) -> c == com && f == fl) defs of
    [] -> []
    [(_,_,_)] -> [c]
    -- I'm not sure what I should do when the combo shows up more than
    -- once, so for now I'll crash.
gda com (DarcsArgOption s [fl] c n h) defs =
    case filter (\ (c,f,_) -> c == com && f == fl) defs of
    [] -> []
    [(_,_,d)] -> [c d]
    -- I'm not sure what I should do when the combo shows up more than
    -- once, so for now I'll crash.

get_defaults = (catMaybes . map (doline.words)) `liftM` get_preflist "defaults"
doline (c:a:r) = Just (c,a,unwords r)
doline _ = Nothing
\end{code}
