/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <stdio.h>
#include <errno.h>
#include <string.h>

#include <config.h>

#include "common.h"
#include "configfile.h"
#include "browse.h"
#include "mainwin.h"
#include "template.h"
#include "util.h"
#include "tinput.h"
#include "errorchain.h"
#include "ldif.h"
#include "formfill.h"
#include "encode.h"
#include "input.h"
#include "debug.h"

#include "../icons/warning.xpm"

extern GtkWidget *statusbar, *mainwin;
extern struct gq_config config;


void new_browsemode(GHashTable *hash)
{
     GNode *root;
     GString *cur_dn;
     GtkWidget *ctreeroot, *browsemode_vbox, *spacer;
     GtkWidget *mainpane, *pane2_vbox, *pane1_scrwin, *pane2_scrwin;

     browsemode_vbox = gtk_vbox_new(FALSE, 0);
     g_hash_table_insert(hash, "vbox", browsemode_vbox);

     spacer = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(spacer);
     gtk_box_pack_start(GTK_BOX(browsemode_vbox), spacer, FALSE, FALSE, 3);

     mainpane = gtk_hpaned_new();
     gtk_container_border_width(GTK_CONTAINER(mainpane), 2);
     gtk_widget_show(mainpane);
     gtk_box_pack_start(GTK_BOX(browsemode_vbox), mainpane, TRUE, TRUE, 0);

     root = g_node_new(new_node_entry("root node"));
     g_hash_table_insert(hash, "root", root);

     ctreeroot = gtk_ctree_new(1, 0);
     g_hash_table_insert(hash, "ctreeroot", ctreeroot);

     gtk_clist_set_selection_mode(GTK_CLIST(ctreeroot), GTK_SELECTION_BROWSE);
     gtk_clist_set_column_auto_resize(GTK_CLIST(ctreeroot), 0, TRUE);
     if (config.sort_browse)
	  gtk_clist_set_auto_sort(GTK_CLIST(ctreeroot), TRUE);
     gtk_signal_connect(GTK_OBJECT(ctreeroot), "tree-select-row", GTK_SIGNAL_FUNC(tree_row_selected), hash);
     gtk_signal_connect(GTK_OBJECT(ctreeroot), "tree-expand", GTK_SIGNAL_FUNC(tree_row_expanded), hash);
     gtk_signal_connect(GTK_OBJECT(ctreeroot), "button_press_event", GTK_SIGNAL_FUNC(button_press_on_tree_item), hash);

     add_all_servers(root);
     gnode_to_tree(hash, root, NULL);
     gtk_widget_show(ctreeroot);

     pane1_scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(pane1_scrwin),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
     gtk_widget_show(pane1_scrwin);
     gtk_widget_set_usize(pane1_scrwin, 300, -1);
gtk_paned_set_position(GTK_PANED(mainpane), 300);
     gtk_paned_add1(GTK_PANED(mainpane), pane1_scrwin);
     gtk_container_add(GTK_CONTAINER(pane1_scrwin), ctreeroot);

     pane2_scrwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(pane2_scrwin),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
     gtk_widget_show(pane2_scrwin);
     g_hash_table_insert(hash, "pane2_scrwin", pane2_scrwin);

     pane2_vbox = gtk_vbox_new(FALSE, 5);
     gtk_widget_show(pane2_vbox);
     gtk_object_set_data(GTK_OBJECT(pane2_scrwin), "vbox", pane2_vbox);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane2_scrwin), pane2_vbox);

     gtk_paned_add2(GTK_PANED(mainpane), pane2_scrwin);

     gtk_widget_show(browsemode_vbox);

     cur_dn = g_string_sized_new(128);
     g_string_assign(cur_dn, "some dummy string");
     g_hash_table_insert(hash, "cur_dn", cur_dn);

}


/*
 * as soon as a new branch has been displayed (i.e. GTK has finished drawing
 * it), make sure it's actually visible. Scroll the window if needed.
 */
int browse_idle_scroll(gpointer data)
{
     gpointer ctree_node_added;
     GHashTable *hash;
     GtkWidget *ctreeroot;

     hash = (GHashTable *) data;

     ctreeroot = g_hash_table_lookup(hash, "ctreeroot");
     ctree_node_added = g_hash_table_lookup(hash, "ctree_node_added");
     if(ctree_node_added && !gtk_ctree_node_is_visible(GTK_CTREE(ctreeroot), ctree_node_added))
	  gtk_ctree_node_moveto(GTK_CTREE(ctreeroot), ctree_node_added, 0, .5, 0);

     g_hash_table_remove(hash, "node_added_sequence");
     gtk_idle_remove_by_data(hash);

     return(TRUE);
}


/*
 * add all configured servers to the root node
 */
void add_all_servers(GNode *root)
{
     GNode *new_node;
     struct ldapserver *server;
     struct node_entry *entry;
     int server_cnt;
     char message[128];

     server_cnt = 0;
     server = config.ldapservers;
     while(server) {
	  entry = new_node_entry(server->name);
	  entry->status = IS_SERVER;
	  new_node = g_node_new(entry);
	  g_node_append(root, new_node);
	  server = server->next;
	  server_cnt++;
     }

     make_message(message, server_cnt, "server", "servers", "found");
     statusbar_msg(message);

}


/*
 * build gtk_tree from GNode
 */
void gnode_to_tree(GHashTable *hash, GNode * node, GtkCTreeNode *ctree_node)
{

/*
     g_node_children_foreach(node, G_TRAVERSE_ALL,
			     (GNodeForeachFunc) gnode_to_tree_callback,
			     (gpointer) tree_node);
*/

     node = node->children;
     while(node) {
	  register GNode *current;

	  current = node;
	  node = current->next;
	  gnode_to_tree_callback(hash, current, ctree_node);

     }


}


gboolean gnode_to_tree_callback(GHashTable *hash, GNode *node, gpointer ctree_node)
{
     GtkWidget *ctreeroot;
     GtkCTreeNode *new_item;
     struct node_entry *entry;
     int node_added_sequence;
     char *labels[2] = { 0 };
     char **exploded_dn = 0;
     char *decoded_label;

     entry = (struct node_entry *) node->data;
     ctreeroot = g_hash_table_lookup(hash, "ctreeroot");

     labels[0] = dn_by_node(node);
     /* don't do this for servers or suffixes */
     if (config.show_rdn_only && !(entry->status & IS_SERVER) &&
	 node->parent && !( ((struct node_entry *)node->parent->data)->status & IS_SERVER)) {
	  exploded_dn = ldap_explode_dn(labels[0], FALSE);
	  labels[0] = exploded_dn[0];
     }

     decoded_label = decoded_string(labels[0]);
     strcpy(labels[0], decoded_label);
     g_free(decoded_label);
  
     new_item = gtk_ctree_insert_node(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node),
				      0, labels, 0, 0, 0, 0, 0, 0, 0);

     /* this lets us check if the second node added to the new branch is visible,
	thus avoiding GTK's "a pixel is visible, that's allright then" behaviour
     */
     node_added_sequence = GPOINTER_TO_INT(g_hash_table_lookup(hash, "node_added_sequence"));
     if(node_added_sequence < 2) {
	  g_hash_table_insert(hash, "node_added_sequence", GINT_TO_POINTER(node_added_sequence + 1));
	  g_hash_table_insert(hash, "ctree_node_added", new_item);
     }

     gtk_ctree_node_set_row_data(GTK_CTREE(ctreeroot), new_item, (gpointer) node);
     if (exploded_dn)
	  free(exploded_dn);

     if (node->children) {
	  gnode_to_tree(hash, node, new_item);
     }
     else {
	  char *dummy[2] = {"DUMMY", 0};
	  /* insert dummy subtree, connect expand to onelevel search */
	  gtk_ctree_insert_node(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(new_item), 0, dummy, 0, 0, 0, 0, 0, 1, 0);

     }

     return(FALSE);
}


/*
 * remove server from visual and gnode trees
 */
void remove_server_node_tree(struct ldapserver *server)
{

     /* this will be called when a server was deleted with
	the preferences dialog */




}


/*
 * an item with a dummy subtree was selected or is about to be expanded. Find out
 * the child nodes for this DN, or (if there are none) delete the subtree
 */
void attach_children_to_node(GHashTable *hash, GtkCTreeNode *item)
{
     GNode *node;
     GtkWidget *ctreeroot;
     struct node_entry *entry;

     ctreeroot = g_hash_table_lookup(hash, "ctreeroot");
     node = (GNode *) gtk_ctree_node_get_row_data(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(item));

     set_busycursor();

     descend_onelevel(node);

     set_normalcursor();

     /* remove dummy entry */
     gtk_ctree_remove_node(GTK_CTREE(ctreeroot), GTK_CTREE_ROW(item)->children);

     entry = (struct node_entry *) node->data;

     if (entry->status == HAS_NO_CHILDREN) {
	  _gtk_ctree_toggle_is_leaf(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(item));
     }
     else {
	  attach_children_to_tree(hash, item);
     }
}


/*
 * a subtree that's known to have children in it is about to be expanded.
 * attach_children_to_node() has already run, so just need to attach them to
 * the tree here
 */
void attach_children_to_tree(GHashTable *hash, GtkCTreeNode *item)
{
     GNode *node;
     GtkWidget *ctreeroot;
     struct node_entry *entry;

     ctreeroot = g_hash_table_lookup(hash, "ctreeroot");
     node = (GNode *) gtk_ctree_node_get_row_data(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(item));

     entry = (struct node_entry *) node->data;
     if(entry->status == HAS_CHILDREN) {
	  gtk_clist_freeze(GTK_CLIST(ctreeroot));
	  gnode_to_tree(hash, node, item);
	  gtk_clist_thaw(GTK_CLIST(ctreeroot));
     }

}


/*
 * a server with a dummy subtree is about to be expanded. Run the suffix
 * discovery stuff and attach any suffixes found. If none found, delete
 * the subtree. Which makes the server totally useless :-(
 */
void attach_server_suffixes(GHashTable *hash, GtkCTreeNode *item)
{
     GNode *server_node, *new_node;
     GSList *suffixes, *slist;
     GtkWidget *ctreeroot;
     struct node_entry *entry;
     struct ldapserver *server;

     /* this function only gets called the first time a server is expanded */

     ctreeroot = g_hash_table_lookup(hash, "ctreeroot");
     server_node = (GNode *) gtk_ctree_node_get_row_data(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(item));
     entry = (struct node_entry *) server_node->data;

     /* delete dummy entry */
     gtk_ctree_remove_node(GTK_CTREE(ctreeroot), GTK_CTREE_ROW(item)->children);

     if( (server = server_by_node(server_node)) == NULL) {
	  /* server has disappeared from serverlist */
	  entry->status = HAS_NO_SUFFIXES;
	  _gtk_ctree_toggle_is_leaf(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(item));
	  return;
     }

     suffixes = get_suffix(server);	 
     if( (slist = suffixes) ) {
	  entry->status = HAS_SUFFIXES;
	  /* add suffixes to server node */
	  while(slist) {
	       new_node = g_node_new(new_node_entry(slist->data));
	       g_node_append(server_node, new_node);
	       g_free(slist->data);
	       slist = slist->next;
	  }
	  g_slist_free(suffixes);

	  gnode_to_tree(hash, server_node, item);
     }
     else {
	  entry->status = HAS_NO_SUFFIXES;
	  /* no suffixes found -- delete subtree */
	  _gtk_ctree_toggle_is_leaf(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(item));

     }

}


/*
 * do a one level search of node's DN, and append any resulting entries
 * to the node
 */
void descend_onelevel(GNode *node)
{
     GNode *new_node;
     LDAP *ld;
     LDAPMessage *res, *e;
     struct ldapserver *server;
     struct node_entry *entry;
     int msg, rc, num_children, update_counter;
     char *dn, message[MAX_DN_LEN + 21];
     char *dummy[] = { "dummy", NULL };

     entry = (struct node_entry *) node->data;
     server = server_by_node(node);
     if(!server)
	  return;

     if( (ld = open_connection(server)) == NULL)
	  return;

     sprintf(message, "onelevel search on %s", dn_by_node(node));
     statusbar_msg(message);

     msg = ldap_search(ld, dn_by_node(node), LDAP_SCOPE_ONELEVEL,
		       "objectclass=*", dummy, 1);

     if(msg == -1) {
	  statusbar_msg(ldap_err2string(msg));
	  return;
     }

     num_children = update_counter = 0;

     while( (rc = ldap_result(ld, msg, 0, NULL, &res)) == LDAP_RES_SEARCH_ENTRY) {
	  for(e = ldap_first_entry(ld, res); e != NULL; e=ldap_next_entry(ld, e)) {
	       dn = ldap_get_dn(ld, e);
	       new_node = g_node_new(new_node_entry(dn));
	       g_node_append(node, new_node);
	       free(dn);
	       num_children++;
	       update_counter++;
	       if(update_counter >= 100) {
		    make_message(message, num_children, "entry", "entries", "found (running)");
		    statusbar_msg(message);
		    update_counter = 0;
	       }
	  }
	  ldap_msgfree(res);
     }
     if (res) ldap_msgfree(res);

     make_message(message, num_children, "entry", "entries", "found (finished)");
     statusbar_msg(message);

     if(num_children)
	  entry->status = HAS_CHILDREN;
     else
	  entry->status = HAS_NO_CHILDREN;

}


void tree_row_selected(GtkCTree *tree, GList *ctree_node, gint column, GHashTable *hash)
{
     GNode *node;
     struct node_entry *entry;

     node = gtk_ctree_node_get_row_data(tree, GTK_CTREE_NODE(ctree_node));
     if (node == NULL)
	  /* in GTK_SELECTION_BROWSE mode we will get called before we set the data <sigh> */
	  return;
  
     entry = (struct node_entry *) node->data;

     switch (entry->status)
     {
     case STATUS_UNKNOWN:
     case HAS_CHILDREN:
     case HAS_NO_CHILDREN:
	  browse_edit_from_entry(hash, node, (GtkCTreeNode *) ctree_node);
	  break;
     case IS_SERVER:
     case HAS_SUFFIXES:
     case HAS_NO_SUFFIXES:
	  server_item_selected(hash, GTK_CTREE_NODE(ctree_node));
	  break;
     default:
	  g_print("selected: status=%02x\n", entry->status);
     }

}


void tree_row_expanded(GtkCTree * tree, GList * ctree_node, GHashTable *hash)
{
     GNode *node;
     struct node_entry *entry;

     node = gtk_ctree_node_get_row_data(tree, GTK_CTREE_NODE(ctree_node));
     entry = (struct node_entry *) node->data;

     switch (entry->status)
     {
     case STATUS_UNKNOWN:
	  attach_children_to_node(hash, GTK_CTREE_NODE(ctree_node));
	  break;
     case IS_SERVER:
	  attach_server_suffixes(hash, GTK_CTREE_NODE(ctree_node));
	  break;
     case HAS_CHILDREN:
     case HAS_NO_CHILDREN:
     case HAS_SUFFIXES:
     case HAS_NO_SUFFIXES:
	  break;
     default:
	  g_print("expanded: status=%02x\n", entry->status);
     }

     if(g_hash_table_lookup(hash, "node_added_sequence"))
	  gtk_idle_add(browse_idle_scroll, hash);

}
 

void _gtk_ctree_toggle_is_leaf(GtkCTree * ctree, GtkCTreeNode * node)
{
     gchar *text, *temp;
     guint8 spacing;
     GdkPixmap *pixmap_closed, *pixmap_opened;
     GdkBitmap *mask_closed, *mask_opened;
     gboolean is_leaf, expanded;

     g_return_if_fail(ctree != NULL);
     g_return_if_fail(GTK_IS_CTREE(ctree));

     gtk_ctree_get_node_info(ctree, node, &text, &spacing, &pixmap_closed, &mask_closed,
			     &pixmap_opened, &mask_opened, &is_leaf, &expanded);
     temp = g_strdup(text);
     gtk_ctree_set_node_info(ctree, node, temp, spacing, pixmap_closed, mask_closed,
			     pixmap_opened, mask_opened, is_leaf, expanded);
     g_free(temp);
}


/*
 * Button pressed on a tree item. Button 3 gets intercepted and puts up
 * a popup menu, all other buttons get passed along to the default handler
 *
 * the return values here are related to handling of button events only.
 */
gboolean button_press_on_tree_item(GtkWidget *tree, GdkEventButton *event, GHashTable *hash)
{
     GNode *node;
     GtkWidget *ctreeroot, *root_menu, *menu, *submenu, *menu_item;
     GtkCTreeNode *ctree_node;
     struct node_entry *entry;
     struct ldapserver *server;
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     GList *templatelist;
     struct gq_template *template;
#endif

     if (event->type == GDK_BUTTON_PRESS && event->button == 3
	 && event->window == GTK_CLIST(tree)->clist_window) {
	  int row, column, rc;

	  rc = gtk_clist_get_selection_info(GTK_CLIST(tree), event->x, event->y, &row, &column);
	  if (rc == 0)
	       return TRUE;
    
	  ctree_node = gtk_ctree_node_nth(GTK_CTREE(tree), row);
	  if (ctree_node == 0)
	       return TRUE;

	  ctreeroot = g_hash_table_lookup(hash, "ctreeroot");
	  node = (GNode *) gtk_ctree_node_get_row_data(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node));
	  if (node == NULL)
	       return(TRUE);

	  if( (entry = (struct node_entry *) node->data) == NULL)
	       return(TRUE);

	  if( (server = server_by_node(node)) == NULL)
	       return(TRUE);

	  /* hack, hack. Need to pass both the ctree_node and hash... */
	  g_hash_table_insert(hash, "selected_ctree_node", ctree_node);

	  root_menu = gtk_menu_item_new_with_label("Root");
	  gtk_widget_show(root_menu);
	  menu = gtk_menu_new();
	  gtk_menu_item_set_submenu(GTK_MENU_ITEM(root_menu), menu);

	  menu_item = gtk_tearoff_menu_item_new();
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_widget_set_sensitive(menu_item, FALSE);
	  gtk_widget_show(menu_item);

	  /* New submenu */
	  menu_item = gtk_menu_item_new_with_label("New");
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  submenu = gtk_menu_new();
	  gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_item), submenu);
	  gtk_widget_show(menu_item);

#ifdef HAVE_LDAP_STR2OBJECTCLASS

	  templatelist = config.templates;
	  while(templatelist) {
	       template = (struct gq_template *) templatelist->data;
	       menu_item = gtk_menu_item_new_with_label(template->name);
	       gtk_object_set_data(GTK_OBJECT(menu_item), "server", server);
	       gtk_menu_append(GTK_MENU(submenu), menu_item);
	       gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
				  GTK_SIGNAL_FUNC(browse_new_from_template_callback),
				  (gpointer) template);
	       gtk_widget_show(menu_item);

	       templatelist = templatelist->next;
	  }

#endif

	  menu_item = gtk_menu_item_new_with_label("Use current entry");
	  gtk_object_set_data(GTK_OBJECT(menu_item), "server", server);
	  gtk_menu_append(GTK_MENU(submenu), menu_item);
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     GTK_SIGNAL_FUNC(browse_new_from_entry_callback),
			     (gpointer) entry);
	  gtk_widget_show(menu_item);

	  /* Refresh */
	  menu_item = gtk_menu_item_new_with_label("Refresh");
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_widget_show(menu_item);

	  if (entry->status & IS_SERVER)
	       gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
				  GTK_SIGNAL_FUNC(refresh_server),
				  (gpointer) hash);
	  else
	       gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
				  (GtkSignalFunc) refresh_subtree,
				  (gpointer) hash);

	  /* Export to LDIF */
	  menu_item = gtk_menu_item_new_with_label("Export to LDIF");
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     (GtkSignalFunc) dump_subtree,
			     (gpointer) hash);
	  gtk_widget_show(menu_item);

	  menu_item = gtk_menu_item_new();
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_widget_show(menu_item);

	  /* Delete */
	  menu_item = gtk_menu_item_new_with_label("Delete");
	  gtk_menu_append(GTK_MENU(menu), menu_item);
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     (GtkSignalFunc) delete_browse_entry,
			     (gpointer) hash);
	  if(entry->status & IS_SERVER)
	       gtk_widget_set_sensitive(menu_item, FALSE);
	  gtk_widget_show(menu_item);

	  gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
			 event->button, event->time);

	  gtk_signal_emit_stop_by_name(GTK_OBJECT(ctreeroot), "button_press_event");
     }

     return(TRUE);
}


/*
 * refresh button in edit pane pressed
 */
void refresh_server(GtkWidget *widget, GHashTable *hash)
{
     GNode *node;
     GtkWidget *ctreeroot, *ctree_node;
     struct ldapserver *server;
     int was_selected, was_expanded;
     struct node_entry *entry;
     char *dummy[2] = {"DUMMY", 0};

     ctreeroot = g_hash_table_lookup(hash, "ctreeroot");
     ctree_node = g_hash_table_lookup(hash, "selected_ctree_node");
     node = (GNode *) gtk_ctree_node_get_row_data(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node));
     if(node == NULL)
	  return;

     if( (entry = (struct node_entry *) node->data) == NULL)
	  return;

     was_selected = (GTK_CTREE_ROW(ctree_node)->row.state == GTK_STATE_SELECTED);
     was_expanded = GTK_CTREE_ROW(ctree_node)->expanded;


     /* clear cached connection, if any */
     if( (server = server_by_node(node)) )
	  close_connection(server, TRUE);

     /* delete all old info */
     gtk_clist_freeze(GTK_CLIST(ctreeroot));

     while (GTK_CTREE_ROW(ctree_node)->children)
	  gtk_ctree_remove_node(GTK_CTREE(ctreeroot), GTK_CTREE_ROW(ctree_node)->children);
     g_node_children_foreach(node, G_TRAVERSE_ALL, (GNodeForeachFunc) node_entry_destroy, NULL);

     /* tell the system to re-query */
     entry->status = IS_SERVER;
     /* insert dummy subtree, connect expand to onelevel search */
     gtk_ctree_insert_node(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node), 0, dummy, 0, 0, 0, 0, 0, 1, 0);

     if (was_selected)
	  gtk_ctree_select(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node));

     if(was_expanded)
	  gtk_ctree_expand(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node));

     gtk_clist_thaw(GTK_CLIST(ctreeroot));

}


void refresh_subtree(GtkWidget *widget, GHashTable *hash)
{
     GNode *node;
     GtkWidget *ctreeroot, *ctree_node;
     struct node_entry *entry;
     int was_selected, was_expanded;
     char *dummy[2] = {"DUMMY", 0};

     ctreeroot = g_hash_table_lookup(hash, "ctreeroot");
     ctree_node = g_hash_table_lookup(hash, "selected_ctree_node");
     if(ctree_node == NULL) {
	  /* refresh callback from mod_entry_from_formlist() */
	  ctree_node = g_hash_table_lookup(hash, "selected_ctree_refresh");
     }
     node = (GNode *) gtk_ctree_node_get_row_data(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node));
     if (node == NULL)
	  return;

     if( (entry = (struct node_entry *) node->data) == NULL)
	  return;

     was_selected = (GTK_CTREE_ROW(ctree_node)->row.state == GTK_STATE_SELECTED);
     was_expanded = GTK_CTREE_ROW(ctree_node)->expanded;

     /* delete all old info */

     /* ermm, this doesn't do anything */
     gtk_clist_freeze(GTK_CLIST(ctreeroot));

     /* there is no gtk_ctree_freeze() so just collapse to avoid visual updates */
     if(was_expanded)
	  gtk_ctree_collapse(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node));

     while (GTK_CTREE_ROW(ctree_node)->children)
	  gtk_ctree_remove_node(GTK_CTREE(ctreeroot), GTK_CTREE_ROW(ctree_node)->children);
     g_node_children_foreach(node, G_TRAVERSE_ALL, (GNodeForeachFunc) node_entry_destroy, NULL);

     /* tell the system to re-query */
     entry->status = STATUS_UNKNOWN;
     /* insert dummy subtree, connect expand to onelevel search */
     gtk_ctree_insert_node(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node), 0, dummy, 0, 0, 0, 0, 0, 1, 0);

     if(was_selected)
	  gtk_ctree_select(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node));

     if(was_expanded)
	  gtk_ctree_expand(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node));

     gtk_clist_thaw(GTK_CLIST(ctreeroot));

}


void delete_browse_entry(GtkWidget *widget, GHashTable *hash)
{
     GNode *node;
     GtkWidget *ctreeroot, *ctree_node;
     struct ldapserver *server;
     struct node_entry *entry;
     int do_delete;

     ctreeroot = g_hash_table_lookup(hash, "ctreeroot");
     ctree_node = g_hash_table_lookup(hash, "selected_ctree_node");
     node = (GNode *) gtk_ctree_node_get_row_data(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node));
     if (node == NULL)
	  return;

     if( (entry = (struct node_entry *) node->data) == NULL)
	  return;

     if( (server = server_by_node(node)) == NULL)
	  return;

     do_delete = 0;
     if(entry->status == HAS_NO_CHILDREN) {
	  /* item is a leaf node */
	  do_delete = 1;
     }
     else if (entry->status == STATUS_UNKNOWN) {
	  /* maybe delete everything in the subtree as well?
	     should do another LDAP_SCOPE_SUBTREE search after
	     each batch of deletes, in case the server is limiting
	     the number of entries returned per search. This could
	     get hairy...

	     For now, just pop up a dialog box with a warning
	  */

	  /* make sure the subtree isn't a dummy */
	  if(node->children == NULL) {
	       /* it's a dummy subtree -- check server for children */
	       if(is_leaf_entry(server, entry->dn))
		    do_delete = 1;
	  }

     }

     if(do_delete) {
	  if(delete_entry(server, entry->dn)) {
	       GNode *parent = node->parent;
	       gtk_ctree_remove_node(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node));
	       g_node_destroy(node);

	       if (parent && parent->children == 0 && parent->data) {
		    entry = (struct node_entry *) parent->data;
		    entry->status = HAS_NO_CHILDREN;
	       }
	  }
     }
     else
	  delete_browse_warning_popup(node);
}


void do_recursive_delete(LDAP* ld, char* dn)
{
     int status;
     LDAPMessage *res, *e;
     static char* attrs [] = {"dn", NULL};
     char message[MAX_DN_LEN + 100];
     int something_to_do;

     sprintf (message, "searching: %s", dn);
     statusbar_msg(message);

     something_to_do = 1;
     while (something_to_do) {
         something_to_do = 0;
         status = ldap_search_s (ld, dn, LDAP_SCOPE_ONELEVEL, "(objectclass=*)", attrs, 1, &res);
         if(status == LDAP_SUCCESS) {
              for (e=ldap_first_entry(ld, res); e; e=ldap_next_entry(ld, e)) {
                   something_to_do = 1;
                   do_recursive_delete (ld, ldap_get_dn (ld, e));
              }
              ldap_msgfree(res);
         }
     }

     sprintf (message, "deleting: %s", dn);
     statusbar_msg(message);
     status = ldap_delete_s (ld, dn);
     if(status != LDAP_SUCCESS)
          error_popup("Error deleting entry", ldap_err2string(status));
}


void delete_non_leaf(GtkWidget *button, GNode *node)
{
     LDAP* ld;
     struct ldapserver *server;
     struct node_entry *entry;

     if( (entry = (struct node_entry *) node->data) == NULL)
          return;

     if( (server = server_by_node(node)) == NULL)
          return;

     if( (ld = open_connection(server)) == NULL)
          return;

     do_recursive_delete (ld, entry->dn);

     close_connection(server, FALSE);

     gtk_widget_destroy ((GtkWidget *) gtk_object_get_user_data (GTK_OBJECT (button)));
}


void delete_browse_warning_popup(GNode *node)
{
     GtkWidget *window, *vbox1, *vbox2, *vbox3, *label;
     GtkWidget *hbox1, *ok_button, *cancel_button;
     GtkWidget *pixmap;
     GdkPixmap *warning;
     GdkBitmap *warning_mask;

     window = gtk_dialog_new();
     gtk_container_border_width(GTK_CONTAINER(window), 12);
     gtk_window_set_title(GTK_WINDOW(window), "Warning");
     gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
     vbox1 = GTK_DIALOG(window)->vbox;
     gtk_widget_show(vbox1);

     hbox1 = gtk_hbox_new(FALSE, 0);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(vbox1), hbox1, FALSE, FALSE, 10);
     warning = gdk_pixmap_create_from_xpm_d(GTK_WIDGET(mainwin)->window,
                                            &warning_mask,
                                            &mainwin->style->white,
                                            warning_xpm);
     pixmap = gtk_pixmap_new(warning, warning_mask);
     gtk_widget_show(pixmap);
     gtk_box_pack_start(GTK_BOX(hbox1), pixmap, TRUE, TRUE, 10);

     vbox2 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox2);
     gtk_box_pack_start(GTK_BOX(hbox1), vbox2, TRUE, TRUE, 10);


     label = gtk_label_new("This entry has a subtree!");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
     gtk_widget_show(label);
     gtk_box_pack_start(GTK_BOX(vbox2), label, TRUE, TRUE, 0);
     label = gtk_label_new("Do you want to delete every entry under it as well?");
     gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
     gtk_widget_show(label);
     gtk_box_pack_start(GTK_BOX(vbox2), label, TRUE, TRUE, 0);

     /* OK button */
     vbox3 = GTK_DIALOG(window)->action_area;
     gtk_widget_show(vbox3);

     ok_button = gtk_button_new_with_label("   OK   ");
     gtk_object_set_user_data (GTK_OBJECT (ok_button), (gpointer) window);
     gtk_signal_connect(GTK_OBJECT(ok_button), "clicked",
                        (GtkSignalFunc) delete_non_leaf,
                        (gpointer) node);
     gtk_box_pack_start(GTK_BOX(vbox3), ok_button, FALSE, FALSE, 0);
     GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);
     gtk_widget_grab_default(ok_button);
     gtk_widget_show(ok_button);

     cancel_button = gtk_button_new_with_label(" CANCEL ");
     gtk_signal_connect_object(GTK_OBJECT(cancel_button), "clicked",
                               (GtkSignalFunc) gtk_widget_destroy,
                               (gpointer) window);
     gtk_box_pack_start(GTK_BOX(vbox3), cancel_button, FALSE, FALSE, 0);
     GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT);
     gtk_widget_grab_default(cancel_button);
     gtk_widget_show(cancel_button);

     gtk_signal_connect_object(GTK_OBJECT(window), "destroy",
                               (GtkSignalFunc) gtk_widget_destroy,
                               (gpointer) window);
     gtk_signal_connect_object(GTK_OBJECT(window), "key_press_event",
                               GTK_SIGNAL_FUNC(close_on_esc),
                               (gpointer) window);

     gtk_widget_show(window);
}


/*
 * a server was selected in the tree widget.
 *
 * FIXME: put up the server's editable info here? Easy to do, but
 *        does it belong here?
 */
void server_item_selected(GHashTable *hash, GtkCTreeNode *ctree_node)
{
     GNode *node;
     GString *cur_dn;
     GtkWidget *ctreeroot, *pane2_scrwin, *pane2_vbox, *label;
     char *server_name;

     ctreeroot = g_hash_table_lookup(hash, "ctreeroot");
     node = (GNode *) gtk_ctree_node_get_row_data(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node));
     if (node == NULL)
	  return;

     server_name = dn_by_node(node);

     cur_dn = g_hash_table_lookup(hash, "cur_dn");
     if(strncmp(cur_dn->str, server_name, cur_dn->len)) {
	  g_string_assign(cur_dn, server_name);
	  g_hash_table_insert(hash, "cur_dn", cur_dn);

	  pane2_scrwin = g_hash_table_lookup(hash, "pane2_scrwin");
	  pane2_vbox = gtk_object_get_data(GTK_OBJECT(pane2_scrwin), "vbox");
	  gtk_widget_destroy(pane2_vbox);

	  pane2_vbox = gtk_vbox_new(FALSE, 2);
	  gtk_object_set_data(GTK_OBJECT(pane2_scrwin), "vbox", pane2_vbox);
	  gtk_widget_show(pane2_vbox);
	  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane2_scrwin),
						pane2_vbox);

	  label = gtk_label_new(server_name);
	  gtk_widget_show(label);
	  gtk_box_pack_start(GTK_BOX(pane2_vbox), label, FALSE, FALSE, 0);

     }

}


/*
 * move up the gnode tree until root + 1 level, which is the server entry
 */
struct ldapserver *server_by_node(GNode *node)
{
     GNode *prevnode;

     prevnode = node;
     while(!G_NODE_IS_ROOT(node)) {
	  prevnode = node;
	  node = node->parent;
     }

     return(server_by_name(dn_by_node(prevnode)));
}


/*
 * returns this node's DN
 */
char *dn_by_node(GNode *node)
{
     struct node_entry *entry;

     if(node) {
	  entry = node->data;
	  if(entry && entry->dn)
	       return(entry->dn);
     }

     return("");
}


/*
 * allocate and initialize the node_entry struct which
 * lives in the node's data field
 */
struct node_entry *new_node_entry(char *dn)
{
     struct node_entry *new_entry;

     new_entry = MALLOC(sizeof(struct node_entry), "struct node_entry");
     new_entry->status = 0;
     new_entry->dn = g_strdup(dn);

     return(new_entry);
}


/*
 * get all suffixes a server considers itself authorative for.
 */
GSList *get_suffix(struct ldapserver *server)
{
     LDAP *ld;
     LDAPMessage *res, *e;
     GSList *suffixes;
     int msg, i, num_suffixes;
     char **vals, message[128];
     char *ldapv3_config[] = {
	  "namingcontexts",
	  NULL
     };

     num_suffixes = 0;

     set_busycursor();

     if( (ld = open_connection(server)) == NULL) {
	  set_normalcursor();
	  return(NULL);
     }

     suffixes = NULL;

     /* try LDAP V3 style config */
     statusbar_msg("base search on NULL DN");
     msg = ldap_search_s(ld, "", LDAP_SCOPE_BASE, "(objectclass=*)",
			 ldapv3_config, 0, &res);
     if(msg == LDAP_SUCCESS) {
	  e = ldap_first_entry(ld, res);
	  while (e) {
	       vals = ldap_get_values(ld, e, "namingcontexts");
	       if (vals) {
		    for(i = 0; vals[i]; i++) {
			 suffixes = g_slist_append(suffixes, g_strdup(vals[i]));
			 num_suffixes++;
		    }
		    ldap_value_free(vals);
	       }
	       e = ldap_next_entry(ld, e);
	  }
          ldap_msgfree(res);
     }

     if(num_suffixes == 0) {

	  /* try Umich style config */
	  statusbar_msg("base search on cn=config");
	  msg = ldap_search_s(ld, "cn=config", LDAP_SCOPE_BASE,
			      "(objectclass=*)", NULL, 0, &res);

	  if(msg == LDAP_SUCCESS) {
	       e = ldap_first_entry(ld, res);
	       while (e) {
		    char **valptr;

		    vals = ldap_get_values(ld, e, "database");
		    if (vals) {
			 for (valptr = vals; *valptr; valptr++) {
			      char *p = *valptr;
	  
			      i = 0;
			      while (p[i] && p[i] != ':')
				   i++;
			      while (p[i] && (p[i] == ':' || p[i] == ' '))
				   i++;
			      if (p[i]) {
				   int len = strlen(p + i);
					
				   while (p[i + len - 1] == ' ')
					len--;
				   p[i + len] = '\0';

				   suffixes = g_slist_append(suffixes, g_strdup(p + i));
				   num_suffixes++;
			      }
			 }
			 ldap_value_free(vals);
		    }
		    e = ldap_next_entry(ld, e);
	       }
	       ldap_msgfree(res);
	  }
     }

     if(num_suffixes == 0) {

	  /* last resort: add the configured base DN and hope something comes out */
	  if(strlen(server->basedn)) {
	       suffixes = g_slist_append(suffixes, g_strdup(server->basedn));
	       num_suffixes++;
	  }

     }

     set_normalcursor();
     close_connection(server, FALSE);

     make_message(message, num_suffixes, "suffix", "suffixes", "found");
     statusbar_msg(message);

     return(suffixes);
}


void cleanup_browse_mode(GHashTable *hash)
{
     GNode *root;
     GString *cur_dn;

     root = g_hash_table_lookup(hash, "root");
     if(root) {
	  /* free all node_entry structs in gnode tree */
	  g_node_traverse(root, G_LEVEL_ORDER, G_TRAVERSE_ALL, -1,
			  (GNodeTraverseFunc) node_entry_free_callback, NULL);

	  g_node_destroy(root);
	  g_hash_table_remove(hash, "root");
     }

     cur_dn = g_hash_table_lookup(hash, "cur_dn");
     if(cur_dn) {
	  g_string_free(cur_dn, TRUE);
	  g_hash_table_remove(hash, "cur_dn");
     }

}


/*
 * this does the real work for cleanup_browse_mode()
 */
gboolean node_entry_free_callback(GNode *node, gpointer dummy)
{
     struct node_entry *entry;

     if(node && node->data) {
	  entry = node->data;
	  if(entry->dn)
	       g_free(entry->dn);
	  FREE(entry, "struct node_entry");
	  node->data = NULL;
     }

     return(FALSE);
}


/*
 * free() all data in the node, and destroy the node for good measure
 */
void node_entry_destroy(GNode *node, gpointer dummy)
{

     node_entry_free_callback(node, dummy);
     g_node_destroy(node);

}


void dump_subtree(GtkWidget *widget, GHashTable *hash)
{
     LDAPMessage *res, *e;
     LDAP *ld;
     GNode *node;
     GString *out, *gmessage;
     GSList *bases;
     GtkWidget *ctreeroot, *ctree_node, *filesel;
     struct ldapserver *server;
     struct node_entry *entry;
     int msg, num_entries;
     char message[128];

     ctreeroot = g_hash_table_lookup(hash, "ctreeroot");
     ctree_node = g_hash_table_lookup(hash, "selected_ctree_node");
     node = (GNode *) gtk_ctree_node_get_row_data(GTK_CTREE(ctreeroot), GTK_CTREE_NODE(ctree_node));
     if (node == NULL)
	  return;

     if( (server = server_by_node(node)) == NULL)
	  return;

     if( (entry = (struct node_entry *) node->data) == NULL)
	  return;

     bases = NULL;

     out = g_string_sized_new(2048);

     if(entry->status & IS_SERVER) {

	  if (entry->status == IS_SERVER)
	  /* an unexpanded server tree hasn't been scanned for suffixes yet,
	     do that now */
	       attach_server_suffixes(hash, GTK_CTREE_NODE(ctree_node));

	  g_node_children_foreach(node, G_TRAVERSE_ALL,
				  (GNodeForeachFunc) add_dn_to_gslist, &bases);
     }
     else
	  bases = g_slist_append(bases, strdup(entry->dn));

     if(g_slist_length(bases) == 0)
	  return;

     /* AFAIK, the UMich LDIF format doesn't take comments or a version string */
     if(config.ldifformat != LDIF_UMICH)
	  prepend_ldif_header(out, server, bases);

     set_busycursor();

     if( (ld = open_connection(server)) == NULL) {
	  /* bases and suffixes/dn leak here... */
	  set_normalcursor();
	  return;
     }

     num_entries = 0;
     gmessage = g_string_sized_new(256);
     while(bases) {
	  g_string_sprintf(gmessage, "subtree search on %s", (char *) bases->data);
	  statusbar_msg(gmessage->str);
	  msg = ldap_search_s(ld, (char *) bases->data, LDAP_SCOPE_SUBTREE,
			      "(objectclass=*)", NULL, 0, &res);
	  if(msg == LDAP_SUCCESS)
	       for(e = ldap_first_entry(ld, res); e; e = ldap_next_entry(ld, e)) {
		    ldif_entry_out(out, ld, e);
		    num_entries++;
	       }

	  free(bases->data);
	  bases = bases->next;
     }

     close_connection(server, FALSE);
     make_message(message, num_entries, "entry", "entries", "found");
     statusbar_msg(message);

     set_normalcursor();

     filesel = gtk_file_selection_new("Save LDIF");
     gtk_object_set_data(GTK_OBJECT(filesel), "out", out);
     gtk_object_set_data(GTK_OBJECT(filesel), "num_entries", GINT_TO_POINTER(num_entries));
     gtk_object_set_data(GTK_OBJECT(filesel), "server", server);
     gtk_signal_connect(GTK_OBJECT(filesel), "destroy",
			(GtkSignalFunc) dump_subtree_filesel_destroy,
			filesel);
     gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button), "clicked",
			(GtkSignalFunc) dump_subtree_ok_callback,
			filesel);
     gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button), "clicked",
			       (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT(filesel));
     gtk_signal_connect_object(GTK_OBJECT(filesel), "key_press_event",
			       (GtkSignalFunc) close_on_esc, (gpointer) filesel);
     gtk_widget_show(filesel);

}


void dump_subtree_ok_callback(GtkWidget *button, GtkWidget *filesel)
{
     FILE *outfile;
     GString *out, *bigmessage;
     struct ldapserver *server;
     int written, num_entries;
     char *filename, message[128];

     out = (GString *) gtk_object_get_data(GTK_OBJECT(filesel), "out");
     server = (struct ldapserver *) gtk_object_get_data(GTK_OBJECT(filesel), "server");
     num_entries = GPOINTER_TO_INT((GString *) gtk_object_get_data(GTK_OBJECT(filesel),
								   "num_entries"));
     filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));

     if( (outfile = fopen(filename, "w")) == NULL)
	  error_popup("Save failed", strerror(errno));
     else {
	  written = fwrite(out->str, 1, out->len, outfile);
	  if(written != out->len) {
	       sprintf(message, "%d of %d bytes written", written, out->len);
	       error_popup("Save failed", message);
	  }
	  else {
	       bigmessage = g_string_sized_new(128);
	       make_message(message, num_entries, "entry", "entries", "exported");
	       g_string_sprintf(bigmessage, "%s to %s", message, filename);
	       statusbar_msg(bigmessage->str);
	       g_string_free(bigmessage, TRUE);
	  }
	  fclose(outfile);
     }


     gtk_widget_destroy(filesel);

}


void dump_subtree_filesel_destroy(GtkWidget *button, GtkWidget *filesel)
{
     GString *out;

     out = (GString *) gtk_object_get_data(GTK_OBJECT(filesel), "out");
     g_string_free(out, TRUE);

     gtk_widget_destroy(filesel);

}


void add_dn_to_gslist(GNode *node, gpointer *bases)
{
     struct node_entry *entry;

     if( (entry = (struct node_entry *) node->data) == NULL)
	  return;

     *bases = g_slist_append(*bases, strdup(entry->dn));

}


void browse_new_from_entry_callback(GtkWidget *widget, gpointer data)
{
     struct node_entry *entry;
     struct ldapserver *server;
     char *dn;

     entry = (struct node_entry *) data;
     dn = entry->dn;

     server = (struct ldapserver *) gtk_object_get_data(GTK_OBJECT(widget), "server");

     new_from_entry(server, dn);

}


#ifdef HAVE_LDAP_STR2OBJECTCLASS

void browse_new_from_template_callback(GtkWidget *widget, gpointer data)
{
     GHashTable *hash;
     GList *formlist;
     struct ldapserver *server;
     struct gq_template *template;

     template = (struct gq_template *) data;

     hash = g_hash_table_new(g_str_hash, g_str_equal);
     g_hash_table_insert(hash, "dn", NULL);

     server = (struct ldapserver *) gtk_object_get_data(GTK_OBJECT(widget), "server");
     g_hash_table_insert(hash, "server", server);

     formlist = formfill_from_template(server, template);
     if(formlist) {
	  g_hash_table_insert(hash, "formlist", formlist);

	  create_form_window(hash);
	  create_form_content(hash);
	  build_inputform(hash);
     }
     else {
	  g_hash_table_destroy(hash);
     }

}

#endif /* HAVE_LDAP_STR2OBJECTCLASS */


void browse_edit_from_entry(GHashTable *hash, GNode *node, GtkCTreeNode *ctree_node)
{
     GList *oldlist, *newlist, *tmplist;
     GHashTable *browsehash;
     GString *cur_dn;
     GtkWidget *pane2_scrwin, *pane2_vbox;
     struct ldapserver *server;
     char *dn;

     if (node == NULL)
	  return;

     if( (server = server_by_node(node)) == NULL)
	  return;

     dn = dn_by_node(node);
     cur_dn = g_hash_table_lookup(hash, "cur_dn");
     g_string_assign(cur_dn, dn);
     g_hash_table_insert(hash, "cur_dn", cur_dn);

     browsehash = g_hash_table_new(g_str_hash, g_str_equal);
     g_hash_table_insert(browsehash, "server", server);
     g_hash_table_insert(browsehash, "edit", GINT_TO_POINTER(1));

     tmplist = NULL;
     oldlist = formlist_from_entry(server, dn, 0);
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     oldlist = add_schema_attrs(server, oldlist);
#endif

     if(oldlist) {
	  newlist = dup_formlist(oldlist);
	  g_hash_table_insert(browsehash, "oldlist", oldlist);
	  g_hash_table_insert(browsehash, "formlist", newlist);
	  g_hash_table_insert(browsehash, "olddn", g_strdup(dn));
	  g_hash_table_insert(browsehash, "dn", g_strdup(dn));
	  g_hash_table_insert(browsehash, "ctree_refresh", GTK_CTREE_ROW(ctree_node)->parent);

	  /* XXX should free etc first */
	  pane2_scrwin = g_hash_table_lookup(hash, "pane2_scrwin");
	  pane2_vbox = gtk_object_get_data(GTK_OBJECT(pane2_scrwin), "vbox");
	  gtk_widget_destroy(pane2_vbox);

	  pane2_vbox = gtk_vbox_new(FALSE, 2);
	  g_hash_table_insert(browsehash, "target_vbox", pane2_vbox);
	  gtk_object_set_data(GTK_OBJECT(pane2_scrwin), "vbox", pane2_vbox);
	  gtk_widget_show(pane2_vbox);
	  gtk_widget_set_parent_window(pane2_vbox, (GdkWindow *) mainwin);
	  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane2_scrwin), pane2_vbox);

	  create_form_content(browsehash);
	  build_inputform(browsehash);
     }
     else {
	  g_hash_table_destroy(browsehash);
     }

}

