/*
For general Sqliteman copyright and licensing information please refer
to the COPYING file provided with the program. Following this notice may exist
a copyright and/or license notice that predates the release of Sqliteman
for which a new license (GPL+exception) is in place.
*/

#include <QMessageBox>
#include <QClipboard>
#include <QCheckBox>
#include <QFile>
#include <QSqlRecord>
#include <QFileDialog>
#include <QDir>
#include <QProgressDialog>

#include "dataviewer.h"
#include "dataexportdialog.h"


DataExportDialog::DataExportDialog(DataViewer * parent) : QDialog(0)
{
	m_data = parent->tableData();
	m_header = parent->tableHeader();
	cancelled = false;

	ui.setupUi(this);
	formats[tr("Comma Separated Values (CSV)")] = "csv";
	formats[tr("HTML")] = "html";
	formats[tr("MS Excel XML (XLS)")] = "xls";
	formats[tr("SQL inserts")] = "sql";
	formats[tr("Python List")] = "py";
	ui.formatBox->addItems(formats.keys());
	connect(ui.fileButton, SIGNAL(toggled(bool)), this, SLOT(fileButton_toggled(bool)));
	connect(ui.searchButton, SIGNAL(clicked()), this, SLOT(searchButton_clicked()));
}

bool DataExportDialog::doExport()
{
	progress = new QProgressDialog("Exporting...", "Abort", 0, 0, this);
	connect(progress, SIGNAL(canceled()), this, SLOT(cancel()));
	progress->setWindowModality(Qt::WindowModal);
	// export everything
	while (m_data->canFetchMore())
		m_data->fetchMore();

	progress->setMaximum(m_data->rowCount());

	QString curr(formats[ui.formatBox->currentText()]);
	bool res = openStream();
	if (curr == "csv")
		res &= exportCSV();
	if (curr == "html")
		res &= exportHTML();
	if (curr == "xls")
		res &= exportExcelXML();
	if (curr == "sql")
		res &= exportSql();
	if (curr == "py")
		res &= exportPython();

	if (res)
		res &= closeStream();

	progress->setValue(m_data->rowCount());
	delete progress;
	progress = 0;
	
	return res;
}

void DataExportDialog::cancel()
{
	cancelled = true;
}

bool DataExportDialog::setProgress(int p)
{
	if (cancelled)
		return false;
	progress->setValue(p);
	qApp->processEvents();
	return true;
}

bool DataExportDialog::openStream()
{
	// file
	exportFile = ui.fileButton->isChecked();
	// clipboard
	clipboard = QString();
	out.setString(&clipboard);
	return true;
}

bool DataExportDialog::closeStream()
{
	out.flush();
	if (exportFile)
	{
		QFile file(ui.fileEdit->text());
		if (!file.open(QFile::WriteOnly | QFile::Truncate))
		{
			QMessageBox::warning(this, tr("Export Error"),
								 tr("Cannot open file %1 for writting").arg(ui.fileEdit->text()));
			return false;
		}
		QTextStream of(&file);
		of << clipboard;
		file.close();
	}
	else
	{
		QClipboard *c = QApplication::clipboard();
		c->setText(clipboard);
	}
	return true;
}

bool DataExportDialog::exportCSV()
{
	for (int i = 0; i < m_header.size(); ++i)
		out << '"' << m_header.at(i) << '"' << ',';
	out << '\n';

	for (int i = 0; i < m_data->rowCount(); ++i)
	{
		if (!setProgress(i))
			return false;
		QSqlRecord r = m_data->record(i);
		for (int j = 0; j < m_header.size(); ++j)
		{
			out << '"' << r.value(j).toString().replace('"', "\"\"").replace('\n', "\\n") << '"';
			if (j != (m_header.size() - 1))
				out << ", ";
		}
		out << '\n';
	}
	return true;
}

bool DataExportDialog::exportHTML()
{
	out << "<html>\n<header>\n<title>Sqliteman export</title>\n</header>\n<body>\n<table border=\"1\">\n<tr>";
	for (int i = 0; i < m_header.size(); ++i)
		out << "<th>" << m_header.at(i) << "</th>";
	out << "</tr>\n";

	for (int i = 0; i < m_data->rowCount(); ++i)
	{
		if (!setProgress(i))
			return false;
		out << "<tr>";
		QSqlRecord r = m_data->record(i);
		for (int j = 0; j < m_header.size(); ++j)
			out << "<td>" << r.value(j).toString() << "</td>";
		out << "</tr>\n";
	}
	out << "</table>\n</body>\n</html>";
	return true;
}

bool DataExportDialog::exportExcelXML()
{
	out << "<?xml version=\"1.0\"?>\n"
		<< "<ss:Workbook xmlns:ss=\"urn:schemas-microsoft-com:office:spreadsheet\">\n"
		<< "<ss:Styles><ss:Style ss:ID=\"1\"><ss:Font ss:Bold=\"1\"/></ss:Style></ss:Styles>\n"
		<< "<ss:Worksheet ss:Name=\"Sqliteman Export\">\n"
		<< "<ss:Table>\n";

	for (int i = 0; i < m_header.size(); ++i)
		out << "<ss:Column ss:Width=\"100\"/>\n";

	out << "<ss:Row ss:StyleID=\"1\">\n";
	for (int i = 0; i < m_header.size(); ++i)
		out << "<ss:Cell><ss:Data ss:Type=\"String\">" << m_header.at(i) << "</ss:Data></ss:Cell>\n";
	out << "</ss:Row>\n";

	for (int i = 0; i < m_data->rowCount(); ++i)
	{
		if (!setProgress(i))
			return false;
		out << "<ss:Row>\n";
		QSqlRecord r = m_data->record(i);
		for (int j = 0; j < m_header.size(); ++j)
			out << "<ss:Cell><ss:Data ss:Type=\"String\">" << r.value(j).toString() << "</ss:Data></ss:Cell>\n";
		out << "</ss:Row>\n";
	}

	out << "</ss:Table>\n"
		<< "</ss:Worksheet>\n"
		<< "</ss:Workbook>\n";
	return true;
}

bool DataExportDialog::exportSql()
{
	out << "BEGIN TRANSACTION;\n";
	QString columns(m_header.join("\", \""));

	for (int i = 0; i < m_data->rowCount(); ++i)
	{
		if (!setProgress(i))
			return false;
		out << "insert into " << m_data->tableName() << " (\"" << columns << "\") values (";
		QSqlRecord r = m_data->record(i);
		for (int j = 0; j < m_header.size(); ++j)
		{
			out << "'" << r.value(j).toString().replace('\'', "''") << "'";
			if (j != (m_header.size() - 1))
				out << ", ";
		}
		out << ");\n";
	}
	out << "COMMIT;\n";
	return true;
}

bool DataExportDialog::exportPython()
{
	out << "[\n";

	for (int i = 0; i < m_data->rowCount(); ++i)
	{
		if (!setProgress(i))
			return false;
		out << "    { ";
		QSqlRecord r = m_data->record(i);
		for (int j = 0; j < m_header.size(); ++j)
		{
			out << "\"" << m_header.at(j) << "\" : \"" << r.value(j).toString() << "\"";
			if (j != (m_header.size() - 1))
				out << ", ";
		}
		out << " },\n";
	}
	out << "]\n";
	return true;
}

void DataExportDialog::fileButton_toggled(bool state)
{
	ui.fileEdit->setEnabled(state);
	ui.searchButton->setEnabled(state);
	ui.label_2->setEnabled(state);
}

void DataExportDialog::searchButton_clicked()
{
	QString mask;
	QString curr(formats[ui.formatBox->currentText()]);
	if (curr == "csv")
		mask = tr("Comma Separated Value (*.csv)");
	if (curr == "html")
		mask = tr("HTML (*.html)");
	if (curr == "xls")
		mask = tr("MS Excel XML (*.xml)");
	if (curr == "sql")
		mask = tr("SQL inserts (*.sql)");
	if (curr == "py")
		mask = tr("Python list (*.py)");
	
	QString fileName = QFileDialog::getSaveFileName(this,
			tr("Export to File"),
			QDir::homePath(),
			mask);
	if (!fileName.isNull())
		ui.fileEdit->setText(fileName);
}
