/*  $Id: AfternoonStalkerEngine.cpp,v 1.20 2003/01/04 11:18:16 sarrazip Exp $
    AfternoonStalkerEngine.cpp - A robot-killing video game engine.

    afternoonstalker - A robot-killing video game.
    Copyright (C) 2001, 2002, 2003 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#include "AfternoonStalkerEngine.h"

#include "RobotSprite.h"

#include <gengameng/PixmapLoadError.h>

#include <assert.h>
#include <stdlib.h>
#include <fstream>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <stdarg.h>
#include <errno.h>

#include <map>
#include <vector>
#include <string>
#include <algorithm>
#include <iomanip>

using namespace std;


/*  Tiles:
*/

#include "images/bunker_door_tile.xpm"
#include "images/bunker_tile.xpm"
#include "images/bunker_tile_0.xpm"
#include "images/bunker_tile_1.xpm"
#include "images/bunker_tile_2.xpm"
#include "images/bunker_tile_3.xpm"
#include "images/cobweb_tile.xpm"  /* foreground tile */
#include "images/floor_tile.xpm"
#include "images/wall_tile.xpm"
#include "images/corner_tile_0.xpm"
#include "images/corner_tile_1.xpm"
#include "images/corner_tile_2.xpm"
#include "images/corner_tile_3.xpm"
#include "images/wall_end_tile_0.xpm"
#include "images/wall_end_tile_1.xpm"
#include "images/wall_end_tile_2.xpm"
#include "images/wall_end_tile_3.xpm"


/*  Sprite images:
*/

#include "images/human_front_0.xpm"
#include "images/human_front_1.xpm"
#include "images/human_front_2.xpm"
#include "images/human_paralyzed_0.xpm"
#include "images/human_explosion_0.xpm"
#include "images/gray_robot_0.xpm"
#include "images/gray_robot_1.xpm"
#include "images/blue_robot_0.xpm"
#include "images/blue_robot_1.xpm"
#include "images/white_robot_0.xpm"
#include "images/white_robot_1.xpm"
#include "images/black_robot_0.xpm"
#include "images/black_robot_1.xpm"
#include "images/invisible_robot_0.xpm"
#include "images/invisible_robot_1.xpm"

#include "images/blink.xpm"
#include "images/human_bullet.xpm"
#include "images/robot_bullet.xpm"
#include "images/big_robot_bullet_0.xpm"
#include "images/big_robot_bullet_1.xpm"
#include "images/big_robot_bullet_2.xpm"

#include "images/gray_robot_explosion_0.xpm"
#include "images/gray_robot_explosion_1.xpm"
#include "images/blue_robot_explosion_0.xpm"
#include "images/blue_robot_explosion_1.xpm"
#include "images/white_robot_explosion_0.xpm"
#include "images/white_robot_explosion_1.xpm"
#include "images/black_robot_explosion_0.xpm"
#include "images/black_robot_explosion_1.xpm"
#include "images/invisible_robot_explosion_0.xpm"
#include "images/invisible_robot_explosion_1.xpm"

#include "images/gun.xpm"
#include "images/spider_right_0.xpm"
#include "images/spider_right_1.xpm"
#include "images/spider_up_0.xpm"
#include "images/spider_up_1.xpm"
#include "images/spider_left_0.xpm"
#include "images/spider_left_1.xpm"
#include "images/spider_down_0.xpm"
#include "images/spider_down_1.xpm"
#include "images/spider_explosion_0.xpm"
#include "images/bat_flying_0.xpm"
#include "images/bat_flying_1.xpm"
#include "images/bat_flying_2.xpm"
#include "images/bat_flying_3.xpm"
#include "images/bat_explosion_0.xpm"
#include "images/digit0.xpm"
#include "images/digit1.xpm"
#include "images/digit2.xpm"
#include "images/digit3.xpm"
#include "images/digit4.xpm"
#include "images/digit5.xpm"
#include "images/digit6.xpm"
#include "images/digit7.xpm"
#include "images/digit8.xpm"
#include "images/digit9.xpm"


#ifdef _MSC_VER
    #if !defined(PACKAGE) || !defined(VERSION)
    	#error PACKAGE and VERSION must be defined in the project
    #endif
    #define PKGDATADIR "."
    #define PKGSOUNDDIR "."
    #define snprintf _snprintf
#endif


///////////////////////////////////////////////////////////////////////////////
//
// CONSTANTS
//

static const int
    FRAMES_PER_SECOND = 20,
    TILE_SIDE = 32,
    SCORE_TTL = 20,
    NUM_INIT_LIVES = 5,
    TILE_MATRIX_Y_OFFSET = TILE_SIDE,
    PLAYER_SPEED_FACTOR = 4,
    GRAY_ROBOT_SPEED_FACTOR = 3,
    SPIDER_SPEED_FACTOR = 2,
    BAT_SPEED_FACTOR = 3,
    BULLET_SPEED_FACTOR = 8,
    BULLETS_PER_GUN = 6;

static const double PI = 4.0 * atan(1.0);
static const double TWO_PI = 2.0 * PI;


///////////////////////////////////////////////////////////////////////////////
//
// UTILITY FUNCTIONS
//

inline int
Rnd(int lowerLimit, int upperLimit)
{
    return rand() % (upperLimit - lowerLimit + 1) + lowerLimit;
}


static void
removeNullElementsFromSpriteList(SpriteList &slist)
{
    SpriteList::iterator it =
		    remove(slist.begin(), slist.end(), (Sprite *) NULL);
	/*  remove() has "packed" the remaining elements at the beginning
	    of the sequence, but has not shortened the list.  This must
	    be done by a call to the erase() method.  Doesn't this seem
	    unintuitive?  I thought remove() removed stuff.
	    @sarrazip 20010501
	*/
    slist.erase(it, slist.end());
}


static void
deleteSprite(Sprite *p)
{
    delete p;
}


template <class Container>
static void
deleteSprites(Container &c)
{
    for_each(c.begin(), c.end(), deleteSprite);
    c.clear();
}


static Couple
getCoupleFromDirection(int direction, int amplitude)
{
    switch (direction)
    {
	case AfternoonStalkerEngine::RIGHT:  return Couple(+amplitude, 0);
	case AfternoonStalkerEngine::UP   :  return Couple(0, -amplitude);
	case AfternoonStalkerEngine::LEFT :  return Couple(-amplitude, 0);
	case AfternoonStalkerEngine::DOWN :  return Couple(0, +amplitude);
	default:                        assert(false); return Couple(0, 0);
    }
}


static int
getDirectionFromCouple(Couple v)
/*  Returns RIGHT, UP, LEFT, DOWN according to v if it represents a
    horizontal or vertical direction.
    Otherwise, returns -1 (for diagonal directions, or if v is zero).
*/
{
    if (v.isZero())
    {
	assert(false);  // for debugging purposes
	return -1;
    }
    if (v.x != 0 && v.y != 0)  // if neither horizontal nor vertical
	return -1;

    if (v.x == 0)
	return (v.y < 0
		? AfternoonStalkerEngine::UP
		: AfternoonStalkerEngine::DOWN);
    return (v.x < 0
		? AfternoonStalkerEngine::LEFT
		: AfternoonStalkerEngine::RIGHT);
}


inline size_t
countDirections(const bool d[4])
{
    return d[0] + d[1] + d[2] + d[3];
}


///////////////////////////////////////////////////////////////////////////////


AfternoonStalkerEngine::AfternoonStalkerEngine(
			    const string &windowManagerCaption,
			    bool useSound, bool pedantic, bool fullScreen)
						throw(int, string)

  : GameEngine(Couple(640, 416), windowManagerCaption, fullScreen),
		// may throw string exception

    paused(false),
    tickCount(0),

    blackColor(0),

    tilePA(17),
    tileMatrix(),

    theScore(0),
    scorePos(3, 6),

    playerLivesPos(theScreenSizeInPixels.x - 10 * 7 - 3, 6),

    playerPA(4),
    playerSprite(NULL),
    numPlayerLives(0),
    initPlayerPos(),
    numPlayerBullets(0),
    lastPlayerDir(-1),
    lastRequestedDir(-1),
    playerParalysisTime(0),

    playerBulletPA(1),
    playerBulletSprites(),

    playerExplosionPA(1),
    playerExplosionSprites(),

    gunPA(1),
    gunSprites(),

    grayRobotPA(2),
    blueRobotPA(2),
    whiteRobotPA(2),
    blackRobotPA(2),
    invisibleRobotPA(2),
    blinkPA(1),
    robotSprites(),
    timeUntilNextRobot(0),
    initRobotPos(),
    numCreatedRobots(0),
    maxNumRobots(0),

    robotBulletPA(1),
    bigRobotBulletPA(3),
    robotBulletSprites(),

    grayRobotExplosionPA(2),
    blueRobotExplosionPA(2),
    whiteRobotExplosionPA(2),
    blackRobotExplosionPA(2),
    invisibleRobotExplosionPA(2),
    robotExplosionSprites(),

    spiderPA(8),
    batPA(4),
    spiderSprites(),
    batSprites(),
    initSpiderPos(),
    timeUntilNextAnimal(0),

    spiderExplosionPA(1),
    spiderExplosionSprites(),
    batExplosionPA(1),
    batExplosionSprites(),

    digitPA(10),
    scoreSprites(),

    escapePressed(false),
    leftArrowPressed(false),
    rightArrowPressed(false),
    upArrowPressed(false),
    downArrowPressed(false),
    spacePressed(false),
    spacePressedBefore(false),
    ctrlPressed(false),
    ctrlPressedBefore(false),
    letterPPressed(false),
    letterPPressedBefore(false),

    theSoundMixer(NULL)
{
    blackColor = SDL_MapRGB(theSDLScreen->format, 0, 0, 0);

    try
    {
	try
	{
	    loadPixmaps();
	}
	catch (PixmapLoadError &e)
	{
	    string msg = "Could not load pixmap " + e.getFilename();
	    throw msg;
	}

	doOneTimeInitializations(useSound, pedantic);
    }
    catch (string &msg)
    {
	displayErrorMessage(msg);
	throw -1;
    }
}


void
AfternoonStalkerEngine::displayErrorMessage(const string &msg) throw()
{
    fprintf(stderr, "%s\n", msg.c_str());
}


AfternoonStalkerEngine::~AfternoonStalkerEngine()
{
    delete playerSprite;
}


void
AfternoonStalkerEngine::initializeLevel()
/*  Initializations that must be done at the start of each new level.
    Assumes that all pixmap arrays have been loaded.
*/
{
    static const char *initNumCreatedRobots = getenv("INITNUMCREATEDROBOTS");

    numCreatedRobots = (initNumCreatedRobots != NULL
    			? atoi(initNumCreatedRobots) : 0);
    maxNumRobots = 3;

    initializePlayerLife();
}


void
AfternoonStalkerEngine::initializePlayerLife()
{
    playerSprite->setPos(initPlayerPos);

    numPlayerBullets = 0;
    lastPlayerDir = -1;
    lastRequestedDir = -1;
    playerParalysisTime = 0;

    deleteSprites(playerBulletSprites);
    deleteSprites(playerExplosionSprites);
    deleteSprites(gunSprites);
    deleteSprites(robotSprites);
    deleteSprites(robotBulletSprites);
    deleteSprites(robotExplosionSprites);
    deleteSprites(spiderSprites);
    deleteSprites(spiderExplosionSprites);
    deleteSprites(batSprites);
    deleteSprites(batExplosionSprites);

    setTimeUntilNextRobot();
    setTimeUntilNextAnimal();
}


void
AfternoonStalkerEngine::setTimeUntilNextRobot()
{
    timeUntilNextRobot = FRAMES_PER_SECOND * 2;
}


void
AfternoonStalkerEngine::setTimeUntilNextAnimal()
{
    timeUntilNextAnimal = FRAMES_PER_SECOND * 5;
}


static
string
getDir(const char *defaultValue, const char *envVarName)
{
    string dir;
    const char *s = getenv(envVarName);
    if (s != NULL)
	dir = s;
    else
	dir = defaultValue;
    
    if (!dir.empty() && dir[dir.length() - 1] != '/')
	dir += '/';

    return dir;
}


void
AfternoonStalkerEngine::doOneTimeInitializations(
				bool useSound, bool pedantic) throw(string)
/*  Throws an error message in a 'string' if an error occurs.
*/
{
    string pkgdatadir = getDir(PKGDATADIR, "PKGDATADIR");
    string pkgsounddir = getDir(PKGSOUNDDIR, "PKGSOUNDDIR");

    string setFilename = pkgdatadir + "level1.set";
    loadLevel(1, setFilename);
    assert(initPlayerPos.x >= 0);
    assert(initPlayerPos.y >= 0);

    playerSprite = new Sprite(playerPA, initPlayerPos,
				Couple(0, 0), Couple(0, 0), Couple(4, 4),
				playerPA.getImageSize() - Couple(8, 8));


    /*  Initialize sound system and load sound files, if desired.
	If useSound is false, then theSoundMixer remains NULL and
	the SoundMixer::Chunk objects remain uninitialized.
    */
    if (useSound)
    {
	theSoundMixer = new SoundMixer();  // may throw string

	try
	{
	    gunPickupSound.init(pkgsounddir + "gun-pickup.wav");
	    playerBulletSound.init(pkgsounddir + "player-bullet.wav");
	    shootingBlanksSound.init(pkgsounddir + "shooting-blanks.wav");
	    playerHitSound.init(pkgsounddir + "player-hit.wav");
	    robotBulletSound.init(pkgsounddir + "robot-bullet.wav");
	    robotHitSound.init(pkgsounddir + "robot-hit.wav");
	    batKilledSound.init(pkgsounddir + "bat-killed.wav");
	    spiderKilledSound.init(pkgsounddir + "spider-killed.wav");
	    newLifeSound.init(pkgsounddir + "new-life.wav");
	}
	catch (const SoundMixer::Error &e)
	{
	    throw e.what();
	}

	try
	{
	    if (pedantic)
		missedSound.init(pkgsounddir + "missed.wav");
	}
	catch (const SoundMixer::Error &e)
	{
	    // Don't use this (lame) sound if it is mysteriously absent...
	}
    }
}


void
AfternoonStalkerEngine::loadLevel(int levelNo, const string &setFilename)
								throw(string)
/*  Loads the level described in the file designated by 'setFilename'
    and associates it with the given level number.

    Replaces the contents of the members 'tileMatrix' and 'initPlayerPos'.

    The next-to-last row of a level must have at least one floor tile.

    Throws an error message in a 'string' if an error occurs.
*/
{
    ifstream setFile(setFilename.c_str());
    if (!setFile)
	throw "Could not open " + setFilename;
    
    size_t levelWidth = 0, levelHeight = 0;
    string line;


    /*  Read level width and height from the file:
    */

    if (!getline(setFile, line))
	throw "Could not read level width from set file";
    if (sscanf(line.c_str(), "%u", &levelWidth) != 1)
	throw "Level width expected";
    if (levelWidth < 3 || levelWidth > 255)  // arbitrary safety limit
	throw "Invalid level width";

    if (!getline(setFile, line))
	throw "Could not read level height from set file";
    if (sscanf(line.c_str(), "%u", &levelHeight) != 1)
	throw "Could not read level height";
    if (levelHeight < 3 || levelHeight > 255)  // arbitrary safety limit
	throw "Invalid level height";


    /*  Read tile characters from the file and initialize tileMatrix
	and initPlayerPos.  There must be exactly one 'D' character.
    */
    static const char *noBunker = getenv("NOBUNKER");  // for debugging

    tileMatrix.clear();
    initPlayerPos = Couple(-1, -1);  // must be changed by following loop

    size_t x, y;
    for (y = 0; y < levelHeight; y++)
    {
	if (!getline(setFile, line))
	    throw "Could not read level row";
	if (line.length() != levelWidth)
	    throw "Level row has wrong width";

	tileMatrix.push_back(TileMatrixRow(levelWidth));

	for (x = 0; x < levelWidth; x++)
	{
	    TileNo tileNo;
	    switch (line[x])
	    {
		case 'W':  tileNo = WALL_TILE;   break;
		case '0':  tileNo = CORNER0_TILE;  break;
		case '1':  tileNo = CORNER1_TILE;  break;
		case '2':  tileNo = CORNER2_TILE;  break;
		case '3':  tileNo = CORNER3_TILE;  break;
		case 'a':  tileNo = WALLEND0_TILE; break;
		case 'b':  tileNo = WALLEND1_TILE; break;
		case 'c':  tileNo = WALLEND2_TILE; break;
		case 'd':  tileNo = WALLEND3_TILE; break;
		case ' ':  tileNo = FLOOR_TILE;  break;
		case 'C':  tileNo = COBWEB_TILE; break;

		case 'B':  tileNo = BUNKER_TILE; break;
		case '4':  tileNo = BUNKER0_TILE; break;
		case '5':  tileNo = BUNKER1_TILE; break;
		case '6':  tileNo = BUNKER2_TILE; break;
		case '7':  tileNo = BUNKER3_TILE; break;

		case 'D':
		    tileNo = BUNKER_DOOR_TILE;

		    initPlayerPos = Couple(x, y) * TILE_SIDE
		    			+ Couple(1, 1 + TILE_MATRIX_Y_OFFSET);
		    break;

		default: throw "Invalid character in level row";
	    }

	    if (noBunker != NULL && tileNo == BUNKER_TILE)
		tileNo = FLOOR_TILE;

	    tileMatrix.back()[x] = tileNo;
	}

	assert(tileMatrix.back().size() == levelWidth);
    }

    assert(tileMatrix.size() == levelHeight);


    if (initPlayerPos == Couple(-1, -1))
	throw "no bunker door tile in level";


    /*  Find the left-most floor tile of the next-to-last row.
	This tile will be the initial position for the robots.
    */
    TileMatrixRow &lastRow = tileMatrix[levelHeight - 2];
    TileMatrixRow::iterator it =
			find(lastRow.begin(), lastRow.end(), FLOOR_TILE);
    if (it == lastRow.end())
	throw "no floor tile on next-to-last row";

    x = distance(lastRow.begin(), it);
    assert(x < levelWidth);

    // Position of tile in pixels:
    initRobotPos = Couple(x, levelHeight - 2) * TILE_SIDE;
    initRobotPos.y += TILE_MATRIX_Y_OFFSET;

    // Go to center of tile:
    initRobotPos += Couple(TILE_SIDE, TILE_SIDE) / 2;

    // Center sprite rectangle in tile:
    initRobotPos -= grayRobotPA.getImageSize() / 2;


    /*  The second cobweb tile of the second row will be the
	spider's initial position.
    */
    TileMatrixRow &firstRow = tileMatrix[1];
    it = find(firstRow.begin(), firstRow.end(), COBWEB_TILE);
    if (it == firstRow.end())
	throw "no cobweb tile on second row";
    it++;
    it = find(it, firstRow.end(), COBWEB_TILE);
    if (it == firstRow.end())
	throw "only one cobweb tile on second row";
    x = distance(firstRow.begin(), it);
    assert(x < levelWidth);

    // Position of tile in pixels:
    initSpiderPos = Couple(x, 1) * TILE_SIDE;
    initSpiderPos.y += TILE_MATRIX_Y_OFFSET;

    // Go to center of tile:
    initSpiderPos += Couple(TILE_SIDE, TILE_SIDE) / 2;

    // Center sprite rectangle in tile:
    initSpiderPos -= spiderPA.getImageSize() / 2;
}


///////////////////////////////////////////////////////////////////////////////


/*virtual*/ void
AfternoonStalkerEngine::processKey(SDLKey keysym, bool pressed)
{
    switch (keysym)
    {
	case SDLK_ESCAPE:
	    escapePressed = pressed;
	    break;
	case SDLK_LEFT:
	    leftArrowPressed = pressed;
	    if (pressed)
		rightArrowPressed = false;
	    break;
	case SDLK_RIGHT:
	    rightArrowPressed = pressed;
	    if (pressed)
		leftArrowPressed = false;
	    break;
	case SDLK_UP:
	    upArrowPressed = pressed;
	    if (pressed)
		downArrowPressed = false;
	    break;
	case SDLK_DOWN:
	    downArrowPressed = pressed;
	    if (pressed)
		upArrowPressed = false;
	    break;
	case SDLK_SPACE:
	    spacePressed = pressed;
	    break;
	case SDLK_LCTRL:
	    ctrlPressed = pressed;
	    break;
	case SDLK_RCTRL:
	    ctrlPressed = pressed;
	    break;
	case SDLK_p:
	    letterPPressed = pressed;
	    break;

	default:
	    ;  // ignore other keys
    }
}


/*virtual*/ bool
AfternoonStalkerEngine::tick()
{
    if (escapePressed)
	return false;

    if (paused)
    {
	restoreBackground();
	drawSprites();
	restoreForeground();
	drawScoreBoard();

	if (letterPPressed && !letterPPressedBefore)
	{
	    paused = false;
	    displayPauseMessage(false);
	}
	else
	    displayPauseMessage(true);
    }
    else
    {
	tickCount++;


	animateAutomaticCharacters();
	restoreBackground();
	drawSprites();
	restoreForeground();
	drawScoreBoard();


	if (numPlayerLives == 0)
	{
	    if (!spacePressed && spacePressedBefore)
	    {
		initializeLevel();
		theScore = getenv("INITSCORE") ? atol(getenv("INITSCORE")) : 0;
		addToNumPlayerLives(5);
		displayStartMessage(false);
	    }
	    else
		displayStartMessage(true);
	}
	

	if (numPlayerLives > 0)
	{
	    if (!animatePlayer())
		return false;
	}

	if (numPlayerLives > 0)
	{
	    /*  If the player is currently paralyzed, it is lying down.
		We then temporarily reduce its collision box.
	    */

	    Couple p, s;
	    bool paralyzed = (playerParalysisTime > 0);
	    if (paralyzed)
	    {
		p = playerSprite->getCollBoxPos();
		s = playerSprite->getCollBoxSize();
		Couple size = playerSprite->getSize();
		playerSprite->setCollBoxPos(Couple(2, size.y - 12));
		playerSprite->setCollBoxSize(Couple(size.x - 4, 12));
	    }
		
	    detectCollisions();

	    if (paralyzed)
	    {
		playerSprite->setCollBoxPos(p);
		playerSprite->setCollBoxSize(s);
	    }
	}
    }

    spacePressedBefore = spacePressed;
    ctrlPressedBefore = ctrlPressed;
    letterPPressedBefore = letterPPressed;
    
    return true;
}


bool
AfternoonStalkerEngine::animatePlayer()
/*  Returns true if the game must continue, or false to have it stop.
*/
{
    if (letterPPressed && !letterPPressedBefore)
    {
	paused = true;
	displayPauseMessage(true);
	return true;
    }


    if (!playerExplosionSprites.empty())  // if player is dying
	return true;


    if (playerParalysisTime > 0)
    {
	playerParalysisTime--;
	playerSprite->currentPixmapIndex = 3;
	return true;
    }


    Couple &playerPos = playerSprite->getPos();
    Couple &playerSpeed = playerSprite->getSpeed();


    /*  Register the last requested direction, which will be the direction
	of the player's bullet, when applicable.
    */
    if (rightArrowPressed)
	lastRequestedDir = RIGHT;
    if (upArrowPressed)
	lastRequestedDir = UP;
    if (leftArrowPressed)
	lastRequestedDir = LEFT;
    if (downArrowPressed)
	lastRequestedDir = DOWN;
    if (ctrlPressed && !ctrlPressedBefore)
	makePlayerShoot();


    playerSprite->currentPixmapIndex = 0;
	    // use "stopped" image unless a non-zero speed is chosen

    bool attemptedDirs[4] =
    {
	rightArrowPressed, upArrowPressed, leftArrowPressed, downArrowPressed
    };
    playerSpeed = attemptMove(*playerSprite,
				attemptedDirs,
				PLAYER_SPEED_FACTOR);
    if (playerSpeed.isZero())
        return true;

    playerSprite->currentPixmapIndex = 1 + ((tickCount & 2) != 0);

    playerPos += playerSpeed;

    // Remember the last moving direction of the player:
    lastPlayerDir = getDirectionFromCouple(playerSpeed);
    if (lastPlayerDir == -1)  // if speed is diagonal
	lastPlayerDir = getDirectionFromCouple(Couple(playerSpeed.x, 0));
						// choose horiz. component

    return true;
}


void
AfternoonStalkerEngine::makePlayerShoot()
{
    static const char *infiniteBullets = getenv("INFINITEBULLETS");

    if (infiniteBullets == NULL && numPlayerBullets == 0)
    {
	playSoundEffect(shootingBlanksSound);
	return;
    }

    playSoundEffect(playerBulletSound);

    int shotDir = lastRequestedDir;
    if (shotDir == -1)
	shotDir = UP;

    Couple displacement = getCoupleFromDirection(shotDir,
			playerSprite->getSize().x / 2 - BULLET_SPEED_FACTOR);

    Couple speed = getCoupleFromDirection(shotDir, BULLET_SPEED_FACTOR);
    Couple bsize = playerBulletPA.getImageSize();
    Couple pos = playerSprite->getCenterPos() - bsize / 2; // + displacement;
    Sprite *bullet = new Sprite(playerBulletPA,
				pos,
				speed,
				Couple(),
				Couple(1, 1),
				bsize - Couple(2, 2));
    playerBulletSprites.push_back(bullet);

    if (numPlayerBullets > 0)
	numPlayerBullets--;
}


Couple
AfternoonStalkerEngine::attemptMove(const Sprite &s,
				    const bool attemptedDirections[4],
				    int speedFactor) const
/*  Attempts a move described by the parameters.
    Returns a non-zero speed if the attempt succeeds.
    Returns a zero speed if no direction is allowed.

    's' must be the sprite that attempts to move from its current position.
    The four 'attempt*' boolean parameters indicate which directions
    are to be attempted.
    'speedFactor' must be the length of the move to try.
*/
{
    bool allowedDirections[4];

    /*  Try to move without first correcting the position.
    */
    Couple delta = determineAllowedDirections(
				s, true /* for player */,
				speedFactor, -1, allowedDirections);
    Couple speed = getSpeedFromAttemptedAndAllowedDirections(
				attemptedDirections, allowedDirections,
				speedFactor, delta);
    if (speed.isNonZero())
	return speed;


    /*  Try to move by allowing the position to be corrected by at most
	'speedFactor'.
    */
    delta = determineAllowedDirections(
				s, true /* for player */,
				speedFactor, speedFactor, allowedDirections);
    speed = getSpeedFromAttemptedAndAllowedDirections(
				attemptedDirections, allowedDirections,
				speedFactor, delta);
    return speed;
}


Couple
AfternoonStalkerEngine::getSpeedFromAttemptedAndAllowedDirections(
				const bool attemptedDirections[4],
				const bool allowedDirections[4],
				int speedFactor,
				Couple delta) const
{
    Couple speed;

    if (attemptedDirections[LEFT] && allowedDirections[LEFT])
    {
	speed.x = -speedFactor;
	speed.y = delta.y;
	return speed;
    }
    if (attemptedDirections[RIGHT] && allowedDirections[RIGHT])
    {
	speed.x = +speedFactor;
	speed.y = delta.y;
	return speed;
    }
    if (attemptedDirections[UP] && allowedDirections[UP])
    {
	speed.y = -speedFactor;
	speed.x = delta.x;
	return speed;
    }
    if (attemptedDirections[DOWN] && allowedDirections[DOWN])
    {
	speed.y = +speedFactor;
	speed.x = delta.x;
	return speed;
    }

    return speed;
}


Couple
AfternoonStalkerEngine::determineAllowedDirections(
					    const Sprite &s,
					    bool forPlayer,
					    int  speedFactor,
					    int  tolerance,
					    bool allowedDirections[4]) const
/*  Determines in what directions a move from the described parameters
    would be allowed.
    Stores boolean values in allowedDirections[], indexed by the integer
    constants RIGHT, UP, LEFT and DOWN.

    'forPlayer' must be true if the determination concerns the player.
    If it is false, the determination will be made for enemies.

    'speedFactor' must be the length of the move to try.

    'tolerance' must be the maximum number of pixels of distance allowed
    between the sprite and the "perfect position" to engage into a corridor.
    If 'tolerance' is -1, the determination is made without any regard
    for "perfect positions".

    Returns the distance between the sprite's position and a "perfect"
    position, as computed by the getDistanceToPerfectPos() method.
*/
{
    assert(tolerance >= -1);

    bool requirePerfectPos = (tolerance >= 0);

    Couple size, pos, delta;
    bool xOK = true, yOK = true;

    if (requirePerfectPos)
    {
	size  = s.getSize();
	pos   = s.getPos();
	delta = getDistanceToPerfectPos(s);
	xOK   = (abs(delta.x) <= tolerance);
	yOK   = (abs(delta.y) <= tolerance);
    }
    else
    {
	size  = Couple(TILE_SIDE, TILE_SIDE);
	pos   = s.getCenterPos() - size / 2;
    }


    Privilege priv = (forPlayer ? PLAYER_PRIV : ENEMY_PRIV);

    Couple newPos = pos + Couple(-speedFactor, delta.y);
    allowedDirections[LEFT] =
	    yOK && positionIsAllowed(LEFT, priv, newPos, size);

    newPos = pos + Couple(+speedFactor, delta.y);
    allowedDirections[RIGHT] =
	    yOK && positionIsAllowed(RIGHT, priv, newPos, size);

    newPos = pos + Couple(delta.x, -speedFactor);
    allowedDirections[UP] =
	    xOK && positionIsAllowed(UP, priv, newPos, size);

    newPos = pos + Couple(delta.x, +speedFactor);
    allowedDirections[DOWN] =
	    xOK && positionIsAllowed(DOWN, priv, newPos, size);

    return delta;
}


Couple
AfternoonStalkerEngine::getDistanceToPerfectPos(const Sprite &s) const
/*  DEFINITION: a "perfect position" for a sprite is a position where
    the sprite's center is at the center of a tile.
*/
{
    int dx = TILE_SIDE / 2 - s.getCenterPos().x % TILE_SIDE;
    int dy = TILE_SIDE / 2 - s.getCenterPos().y % TILE_SIDE;
    return Couple(dx, dy);
}


bool
AfternoonStalkerEngine::positionIsAllowed(
		int direction, Privilege priv, Couple pos, Couple size) const
/*  
    Determines if the proposed position 'pos' for a sprite that would
    have the given 'size' would allow movement in the given 'direction'.
    Looks at the tiles at the given position, which must be floor tiles.
*/
{
    pair<TileNo, TileNo> p = getSideTilesAtPos(direction, pos, size);
    TileNo t0 = p.first;
    TileNo t1 = p.second;

    switch (priv)
    {
	case ENEMY_PRIV:
	    return (t0 == FLOOR_TILE
			|| t0 == COBWEB_TILE)
		    && (t1 == FLOOR_TILE
			|| t1 == COBWEB_TILE);

	case PLAYER_PRIV:
	    return (t0 == FLOOR_TILE
			|| t0 == COBWEB_TILE
			|| t0 == BUNKER_DOOR_TILE)
		    && (t1 == FLOOR_TILE
			|| t1 == COBWEB_TILE
			|| t1 == BUNKER_DOOR_TILE);

	case BULLET_PRIV:
	    return (t0 == FLOOR_TILE /*|| t0 == BUNKER_DOOR_TILE*/ )
	    		&& (t1 == FLOOR_TILE /*|| t1 == BUNKER_DOOR_TILE*/ );

	/*  If the condition 't1 == BUNKER_DOOR_TILE' is added, then bullets
	    can go into the bunker's door.  This means that robots can kill
	    the player when the player is "stuck" in the bunker.  This seemed
	    to be an excessively difficult rule.  @sarrazip 20011230
	*/

	case PIERCING_BULLET_PRIV:
	    return (t0 == FLOOR_TILE
			|| t0 == COBWEB_TILE
			|| t0 == BUNKER_TILE
			|| t0 == BUNKER_DOOR_TILE)
		    && (t1 == FLOOR_TILE
			|| t1 == COBWEB_TILE
			|| t1 == BUNKER_TILE
			|| t1 == BUNKER_DOOR_TILE);

	default:
	    assert(false);
	    return false;
    }
}


pair<AfternoonStalkerEngine::TileNo, AfternoonStalkerEngine::TileNo>
AfternoonStalkerEngine::getSideTilesAtPos(
				int direction, Couple pos, Couple size) const
{
    Couple topLeft(pos);
    Couple botRight = topLeft + size - Couple(1, 1);
    Couple botLeft(topLeft.x, botRight.y);
    Couple topRight(botRight.x, topLeft.y);

    TileNo t0 = WALL_TILE, t1 = WALL_TILE;

    switch (direction)
    {
	case RIGHT:
	    t0 = getTileAtPos(topRight);
	    t1 = getTileAtPos(botRight);
	    break;

	case UP:
	    t0 = getTileAtPos(topLeft);
	    t1 = getTileAtPos(topRight);
	    break;
	
	case LEFT:
	    t0 = getTileAtPos(topLeft);
	    t1 = getTileAtPos(botLeft);
	    break;
	
	case DOWN:
	    t0 = getTileAtPos(botLeft);
	    t1 = getTileAtPos(botRight);
	    break;
	
	default:
	    assert(false);
    }

    return pair<TileNo, TileNo>(t0, t1);
}


AfternoonStalkerEngine::TileNo
AfternoonStalkerEngine::getTileAtPos(Couple pos) const
/*  Returns the type of tile that is found at the specified position.
    If the tile is CORNER* or WALLEND*, the value WALL_TILE is
    returned, because those types of tiles must be treated as
    ordinary wall tiles.
    If the tile is BUNKER[0-3]*, the value BUNKER_TILE is
    returned, because those types of tiles must be treated as
    ordinary bunker tiles.
    This avoids having to change the code that calls this method.
    @sarrazip 20020118
*/
{
    pos.y -= TILE_MATRIX_Y_OFFSET;
    pos /= TILE_SIDE;

    if (pos.x < 0 || pos.x >= (int) tileMatrix.back().size()
		|| pos.y < 0 || pos.y >= (int) tileMatrix.size())
	return WALL_TILE;  // for safety; not supposed to happen

    TileNo t = tileMatrix[pos.y][pos.x];
    if (t >= CORNER0_TILE && t <= WALLEND3_TILE)
	return WALL_TILE;
    if (t >= BUNKER0_TILE && t <= BUNKER3_TILE)
	return BUNKER_TILE;
    return t;
}


void
AfternoonStalkerEngine::setTileAtPos(Couple pos, TileNo t)
{
    pos.y -= TILE_MATRIX_Y_OFFSET;
    pos /= TILE_SIDE;

    if (pos.x < 0 || pos.x >= (int) tileMatrix.back().size()
		|| pos.y < 0 || pos.y >= (int) tileMatrix.size())
	return;  // for safety; not supposed to happen

    tileMatrix[pos.y][pos.x] = t;
}


void
AfternoonStalkerEngine::animateTemporarySprites(SpriteList &slist) const
/*  'slist' must be a list of sprites that die when their "time to live"
    expires.  This method removes sprites from 'slist' when they die.
    Sprites that live are advanced by adding their speed to their position.
*/
{
    for (SpriteList::iterator it = slist.begin(); it != slist.end(); it++)
    {
        Sprite *s = *it;
        assert(s != NULL);
        if (s->getTimeToLive() == 0)
        {
            delete s;
            *it = NULL;  // mark list element for deletion
        }
        else
        {
            s->decTimeToLive();
            s->addSpeedToPos();
        }
    }

    removeNullElementsFromSpriteList(slist);
}


void
AfternoonStalkerEngine::animateAutomaticCharacters()
{
    /*  Animate score sprites:
    */
    animateTemporarySprites(scoreSprites);


    /*  Animate explosions:
    */
    SpriteList::iterator it;
    for (it = robotExplosionSprites.begin();
				it != robotExplosionSprites.end(); it++)
	(*it)->currentPixmapIndex ^= 1;
    moveExplosionSprites(robotExplosionSprites);
    moveExplosionSprites(spiderExplosionSprites);
    moveExplosionSprites(batExplosionSprites);

    bool playerDying = !playerExplosionSprites.empty();
    moveExplosionSprites(playerExplosionSprites);

    if (playerExplosionSprites.empty() && playerDying)
    {
	// The player has finished dying...

	initializePlayerLife();
	return;
    }



    if (!playerExplosionSprites.empty())  // if player is dying
	return;


    /*  Create robots if needed:
    */
    static const char *noEnemies = getenv("NOENEMIES");

    maxNumRobots = 3 + numCreatedRobots / 63;

    if (timeUntilNextRobot > 0)
	timeUntilNextRobot--;
    else if (noEnemies == NULL && robotSprites.size() < maxNumRobots)
    {
	setTimeUntilNextRobot();

	bool smart = false;

	const PixmapArray *pa = NULL;

	if (numPlayerLives == 0)
	    pa = &grayRobotPA;  // only gray robots in demo mode
	else
	{
	    /*  Set the variables 'smart' and 'pa' according to game level:
	    */
	    int numRobotTypes = 1;
	    if (theScore >= 5000)
		numRobotTypes++;
	    if (theScore >= 15000)
		numRobotTypes++;
	    if (theScore >= 30000)
		numRobotTypes++;
	    if (theScore >= 80000)
		numRobotTypes++;
	    
	    int robotType = rand() % numRobotTypes;

	    // In later levels, avoid the two weakest robots:
	    if (numRobotTypes >= 4 && robotType < 2)
		robotType += 2;

	    switch (robotType)
	    {
		case 0:  pa = &grayRobotPA; break;
		case 1:  pa = &blueRobotPA; break;
		case 2:  pa = &whiteRobotPA; break;
		case 3:  pa = &blackRobotPA; break;
		case 4:  pa = &invisibleRobotPA; break;
		default: assert(false); pa = &grayRobotPA;
	    }

	    smart = (pa != &grayRobotPA);
	}

	const PixmapArray *explosionPA = NULL;
	if (pa == &grayRobotPA)
	    explosionPA = &grayRobotExplosionPA;
	else if (pa == &blueRobotPA)
	    explosionPA = &blueRobotExplosionPA;
	else if (pa == &whiteRobotPA)
	    explosionPA = &whiteRobotExplosionPA;
	else if (pa == &blackRobotPA)
	    explosionPA = &blackRobotExplosionPA;
	else if (pa == &invisibleRobotPA)
	    explosionPA = &invisibleRobotExplosionPA;
	else
	    assert(false);

	RobotSprite *robot = new RobotSprite(*pa, initRobotPos,
		    Couple(4, 4), grayRobotPA.getImageSize() - Couple(8, 8),
		    *explosionPA);
	robot->smart = smart;
	if (pa == &whiteRobotPA
			|| pa == &blackRobotPA
			|| pa == &invisibleRobotPA)
	    robot->numShotsBeforeDeath = 3;
	if (pa == &blackRobotPA || pa == &invisibleRobotPA)
	    robot->bigBullets = true;

	robotSprites.push_back(robot);

	numCreatedRobots++;
    }


    /*  Create animals if needed:
    */
    if (timeUntilNextAnimal > 0)
	timeUntilNextAnimal--;
    else if (noEnemies == NULL)
    {
	if (spiderSprites.size() < 1)
	{
	    setTimeUntilNextAnimal();

	    Sprite *spider = new Sprite(spiderPA, initSpiderPos,
					Couple(),
					Couple(),
					Couple(4, 4),
					spiderPA.getImageSize() - Couple(8, 8));
	    spiderSprites.push_back(spider);
	}
	else if (batSprites.size() < 2)
	{
	    setTimeUntilNextAnimal();

	    Couple size = batPA.getImageSize();

	    Sprite *bat = new Sprite(batPA, initSpiderPos,
				Couple(),
				Couple(),
				Couple(6, (size.y - BULLET_SPEED_FACTOR) / 2),
				Couple(18, BULLET_SPEED_FACTOR));
		/*  The height (and width) of a collision box must be
		    at least large as BULLET_SPEED_FACTOR, or else bullets
		    can go "through" the sprite entirely.
		*/

	    batSprites.push_back(bat);
	}
    }


    /*  Show a gun if the player is out of bullets:
    */
    if (numPlayerBullets == 0 && gunSprites.size() < 1 && numPlayerLives > 0)
    {
	Couple gunSize = gunPA.getImageSize();

	Couple pos;  // choose random position outside of the leftmost columns:
	do
	{
	    pos = chooseRandomAllowedTile();
	} while (pos.x < 4 * TILE_SIDE);

	pos += (Couple(1, 1) * TILE_SIDE - gunSize) / 2;
	Sprite *gun = new Sprite(gunPA,
				pos, Couple(), Couple(),
				Couple(2, 2), gunSize - Couple(4, 4));
	gunSprites.push_back(gun);
    }


    /*  Stop blinking for robots when appropriate:
    */
    for (SpriteList::const_iterator itr = robotSprites.begin();
				    itr != robotSprites.end(); itr++)
    {
	RobotSprite *robot = dynamic_cast<RobotSprite *>(*itr);
	if (robot->endOfBlinkTime <= tickCount)
	    robot->stopBlinking();
    }

    /*  Move other sprites:
    */
    moveEnemyList(robotSprites, GRAY_ROBOT_SPEED_FACTOR);

    int incr = (numCreatedRobots >= 18);
    moveEnemyList(spiderSprites, SPIDER_SPEED_FACTOR + incr);
    moveEnemyList(batSprites, BAT_SPEED_FACTOR + incr);

    moveBullets(playerBulletSprites, true);
    moveBullets(robotBulletSprites);


    /*  Make enemies attack the player:
    */
    makeRobotsShoot();
}


void
AfternoonStalkerEngine::moveExplosionSprites(SpriteList &slist)
{
    for (SpriteList::iterator it = slist.begin(); it != slist.end(); it++)
    {
	Sprite *expl = *it;
	if (expl == NULL)
	    { assert(false); continue; }
	expl->addAccelToSpeed();
	expl->addSpeedToPos();

	if (isOutOfSetting(*expl))
	{
	    *it = NULL;
	    delete expl;
	}
	else if (expl->getTimeToLive() > 0 && expl->decTimeToLive() == 0)
	{
	    *it = NULL;
	    delete expl;
	}
    }
    removeNullElementsFromSpriteList(slist);
}


bool
AfternoonStalkerEngine::isOutOfSetting(const Sprite &s) const
{
    Couple botRight = s.getLowerRightPos();
    if (botRight.x <= 0)
	return true;
    if (botRight.y <= 0 /*TILE_MATRIX_Y_OFFSET*/ )
	return true;

    Couple topLeft = s.getPos();
    int rightOfSetting = tileMatrix[0].size() * TILE_SIDE;
    if (topLeft.x >= rightOfSetting)
	return true;
    int belowSetting = tileMatrix.size() * TILE_SIDE + TILE_MATRIX_Y_OFFSET;
    if (topLeft.y >= belowSetting)
	return true;

    return false;
}

size_t
AfternoonStalkerEngine::getNumToughRobots() const
{
    size_t count = 0;
    for (SpriteList::const_iterator it = robotSprites.begin();
    				it != robotSprites.end(); it++)
    {
	const RobotSprite *robot = dynamic_cast<RobotSprite *>(*it);
	if (robot->numShotsBeforeDeath > 1)
	    count++;
    }

    return count;
}


void
AfternoonStalkerEngine::moveEnemyList(SpriteList &slist, int speedFactor)
{
    const Couple plpos = playerSprite->getPos();

    bool changePixmap = (tickCount % (FRAMES_PER_SECOND / 5) == 0);

    size_t batPixmapIndex;
    { 
	size_t b = size_t((tickCount / 3) & 1);
	size_t a = size_t(tickCount % 6);
	size_t d = 6 - a;
	batPixmapIndex = a + b * (d - a);

	assert(batPixmapIndex < batPA.getNumImages());
	    
	    /*  "Real Programmers don't document --
		if the code was hard to write, it should be hard to read."
	    */
    }

    for (SpriteList::iterator it = slist.begin(); it != slist.end(); it++)
    {
	Sprite *s = *it;
	if (s == NULL)
	    { assert(false); continue; }

	/*  Change the current pixmap of the sprite, if it has an even
	    number of them.  Thus, pixmap index 0 and 1 alternate,
	    and so do indices 2 and 3, 4 and 5, etc.
	*/
	const PixmapArray *pa = s->getPixmapArray();
	if (pa == NULL)
	    { assert(false); continue; }
	if (pa == &batPA)
	{
	    s->currentPixmapIndex = batPixmapIndex;
	}
	else  // robots or spiders:
	{
	    size_t numImages = pa->getNumImages();
	    if (numImages == 0)
		{ assert(false); continue; }
	    if (numImages % 2 != 0)
		{ assert(false); continue; }
	    if (changePixmap)
		s->currentPixmapIndex ^= 1;
	    assert(s->currentPixmapIndex < numImages);
	}


	bool allowedDirections[4];
	Couple delta = determineAllowedDirections(
			    *s, false /* for enemies */,
			    speedFactor, speedFactor - 1, allowedDirections);


	/*  If the current direction is allowed, then disallow the
	    opposite direction.
	*/
	if (s->getSpeed().isNonZero())
	{
	    int currentDir = getDirectionFromCouple(s->getSpeed());
	    if (currentDir != -1 && allowedDirections[currentDir])
		allowedDirections[currentDir ^ 2] = false;
	}


	int dir = -1;

	RobotSprite *robot = dynamic_cast<RobotSprite *>(s);
	bool smart = (robot != NULL && robot->smart);

	/*  To make the enemies less predictable, they will choose their
	    next direction at random once in a while, instead of always
	    looking at where the player is.
	*/
	if (rand() % 5 == 0)
	    smart = false;

	if (smart)
	    dir = chooseDirectionTowardTarget(s->getPos(), plpos,
	    				speedFactor, allowedDirections);


	if (dir == -1)  // if still undecided:
	{
	    // Find a random true element in allowedDirections[]:
	    dir = rand() % 4;
	    while (!allowedDirections[dir])
		dir = (dir + 1) % 4;
	}

	// Add a correction to get to a "perfect" position:
	if (dir == RIGHT || dir == LEFT)
	    delta.x = 0;
	else if (dir == UP || dir == DOWN)
	    delta.y = 0;
	s->getPos() += delta;

	// Convert the direction (RIGHT/UP/LEFT/DOWN) into a speed couple:
	Couple speed = getCoupleFromDirection(dir, speedFactor);
	s->setSpeed(speed);
	s->addSpeedToPos();


	/*  Set the pixmap index of the sprite according to its new speed.
	*/
	if (pa == &spiderPA)
	{
	    int dir = getDirectionFromCouple(speed);
	    int lowBit = (s->currentPixmapIndex & 1);
		    // preserve oscillating bit for animation
	    s->currentPixmapIndex = dir * 2 + lowBit;
	}
    }

    removeNullElementsFromSpriteList(slist);
}


void
AfternoonStalkerEngine::moveBullets(SpriteList &slist, bool playMissedSound)
/*
    If 'playMissedSound' is true, then a "missed" sound will be played
    if the bullet dies without having hit anything.
    A bullet is considered to have hit something if its "time to live"
    field is not zero.
*/
{
    for (SpriteList::iterator it = slist.begin(); it != slist.end(); it++)
    {
	Sprite *bullet = *it;
	if (bullet == NULL)
	{
	    assert(false);
	    continue;
	}

	if (bullet->getPixmapArray() == &bigRobotBulletPA)
	    bullet->currentPixmapIndex = (bullet->currentPixmapIndex + 1) % 3;

	Couple &pos = bullet->getPos();
	Couple &speed = bullet->getSpeed();

	Couple newPos = pos + speed;
	int dir = getDirectionFromCouple(speed);
	assert(dir != -1);  // speed is supposed to be horiz. or vertical
	Privilege priv = BULLET_PRIV;


	/*  Late in the game, the "big bullets" can go over the player's
	    bunker, which will destroy the tiles that comprise it.
	*/
	bool lateInGame = (theScore >= 50000);
	if (lateInGame && bullet->getPixmapArray() == &bigRobotBulletPA)
	    priv = PIERCING_BULLET_PRIV;


	bool allowed = positionIsAllowed(dir, priv, newPos, bullet->getSize());
	if (allowed)
	{
	    pos = newPos;

	    /*  Late in the game, the "big bullets" (energy bolts) destroy
		the player's bunker:
	    */
	    if (lateInGame && priv == PIERCING_BULLET_PRIV)
	    {
		TileNo t = getTileAtPos(bullet->getCenterPos());
		if (t == BUNKER_TILE || t == BUNKER_DOOR_TILE)
		{
		    setTileAtPos(bullet->getCenterPos(), FLOOR_TILE);
		    allowed = false;  // to kill this bullet
		}
	    }
	}

	if (!allowed)
	{
	    if (playMissedSound && bullet->getTimeToLive() == 0)
		playSoundEffect(missedSound);
	    *it = NULL;
	    delete bullet;
	}
    }

    removeNullElementsFromSpriteList(slist);
}


int
AfternoonStalkerEngine::chooseDirectionTowardTarget(
				    Couple startPos,
				    Couple targetPos,
				    int speedFactor,
				    const bool allowedDirections[4]) const
/*  Determines the direction that should lead to the position of 'target'
    from the starting position 'startPos'.
    'speedFactor' must be the speed of the sprite that is at the
    starting position.
    'allowedDirections' must indicate which directions are possible.

    Returns RIGHT, UP, LEFT, DOWN, or -1 if no decision was possible.
*/
{
    int dir = -1;  // should contain RIGHT, UP, LEFT or DOWN

    // Choose "preferred" directions depending on the target's position:
    const Couple toTarget = targetPos - startPos;
    int prefHorizDir = (toTarget.x >= speedFactor ? RIGHT :
			(toTarget.x <= -speedFactor ? LEFT : -1));
    int prefVertDir  = (toTarget.y >= speedFactor ? DOWN :
			(toTarget.y <= -speedFactor ? UP : -1));
    if (prefHorizDir != -1 && !allowedDirections[prefHorizDir])
	prefHorizDir =  -1;
    if (prefVertDir  != -1 && !allowedDirections[prefVertDir])
	prefVertDir  =  -1;

    if (prefHorizDir != -1 && prefVertDir != -1)
	dir = (rand() % 2 ? prefHorizDir : prefVertDir);
    else if (prefHorizDir != -1)
	dir = prefHorizDir;
    else if (prefVertDir != -1)
	dir = prefVertDir;

    return dir;
}


Couple
AfternoonStalkerEngine::chooseRandomAllowedTile()
/*  Returns the absolute position in pixels of a random tile that is
    an allowed position for the player and the enemies.
*/
{
    size_t levelWidth = tileMatrix.back().size();
    size_t levelHeight = tileMatrix.size();
    Couple pos;
    do
    {
	pos.x = rand() % levelWidth;
	pos.y = rand() % levelHeight;
    } while (tileMatrix[pos.y][pos.x] != FLOOR_TILE);

    return pos * TILE_SIDE + Couple(0, TILE_MATRIX_Y_OFFSET);
}


void
AfternoonStalkerEngine::makeRobotsShoot()
{
    SpriteList::iterator it;
    for (it = robotSprites.begin(); it != robotSprites.end(); it++)
    {
	RobotSprite *robot = dynamic_cast<RobotSprite *>(*it);
	if (robot == NULL)
	    { assert(false); continue; }

	// Require minimum time interval between robot shots:
	if (robot->timeOfLastShot != -1
		    && tickCount - robot->timeOfLastShot < 20)
	    continue;
	
	// This robot shoots if it is aligned (horiz. or vert.) with player:
	Couple robotCenter = robot->getCenterPos();
	Couple playerCenter = playerSprite->getCenterPos();
	int dx = abs(robotCenter.x - playerCenter.x);
	int dy = abs(robotCenter.y - playerCenter.y);

	if (dx < TILE_SIDE / 4 || dy < TILE_SIDE / 4)
	{
	    if (numPlayerLives > 0)
		playSoundEffect(robotBulletSound);

	    int shotDir = -1;
	    if (dx < dy)
		shotDir = (robotCenter.y < playerCenter.y ? DOWN : UP);
	    else
		shotDir = (robotCenter.x < playerCenter.x ? RIGHT : LEFT);

	    Couple displacement = getCoupleFromDirection(shotDir,
				robot->getSize().x / 2 - BULLET_SPEED_FACTOR);
	    Couple speed = getCoupleFromDirection(shotDir, BULLET_SPEED_FACTOR);

	    PixmapArray *pa =
		    (robot->bigBullets ? &bigRobotBulletPA : &robotBulletPA);
	    const Couple bsize = pa->getImageSize();
	    Couple pos = robot->getCenterPos() - bsize / 2;  // + displacement;

	    Sprite *bullet = new Sprite(*pa,
					pos,
					speed,
					Couple(),
					Couple(1, 1),
					bsize - Couple(2, 2));
	    robotBulletSprites.push_back(bullet);

	    robot->timeOfLastShot = tickCount;
	}
    }
}


void
AfternoonStalkerEngine::detectCollisions()
{
    if (!playerExplosionSprites.empty())  // if player is dying
	return;


    static const char *invinciblePlayer = getenv("INVINCIBLEPLAYER");

    SpriteList::iterator it;


    /*  Human vs gun:
    */
    if (numPlayerLives > 0)
    {
	for (it = gunSprites.begin(); it != gunSprites.end(); it++)
	{
	    if (playerSprite->collidesWithSprite(**it))
	    {
		playSoundEffect(gunPickupSound);

		numPlayerBullets += BULLETS_PER_GUN;
		delete *it;
		*it = NULL;
		continue;
	    }
	}
	removeNullElementsFromSpriteList(gunSprites);
    }


    /*  Human's bullets vs enemies:
    */
    SpriteList::iterator itb;
    for (itb = playerBulletSprites.begin();
			itb != playerBulletSprites.end(); itb++)
    {
	Sprite *bullet = *itb;
	if (bullet == NULL)
	    { assert(false); continue; }


	bool absorbedByBlinking = false;

	/*  Robots:
	*/
	SpriteList::iterator itr;
	for (itr = robotSprites.begin(); itr != robotSprites.end(); itr++)
	{
	    RobotSprite *robot = dynamic_cast<RobotSprite *>(*itr);
	    if (robot == NULL)
		continue;

	    /*  If the robot is touched by the bullet and is not blinking:
	    */
	    if (bullet->collidesWithSprite(*robot))
	    {
		if (robot->endOfBlinkTime > tickCount)  // if still blinking
		    absorbedByBlinking = true;
		else
		{
		    playSoundEffect(robotHitSound);

		    bullet->setTimeToLive(1);
			// Patch: mark this bullet as having hit something.

		    if (--robot->numShotsBeforeDeath == 0)
		    {
			createRobotExplosion(*robot);

			long score = -1;
			if (robot->getPixmapArray() == &grayRobotPA)
			    score = 400;
			else if (robot->getPixmapArray() == &blueRobotPA)
			    score = 500;
			else if (robot->getPixmapArray() == &whiteRobotPA)
			    score = 1000;
			else if (robot->getPixmapArray() == &blackRobotPA)
			    score = 2000;
			else if (robot->getPixmapArray() == &invisibleRobotPA)
			    score = 4000;
			else
			    assert(false);
			createScoreSprites(score, robot->getCenterPos());

			delete robot;
			*itr = NULL;
			setTimeUntilNextRobot();
			timeUntilNextRobot += 2 * FRAMES_PER_SECOND;
			    // add extra time to allow explosion to disappear
		    }
		    else
		    {
			robot->startBlinking(blinkPA,
				    tickCount + FRAMES_PER_SECOND * 3 / 2);
			    // initializes robot->endOfBlinkTime
		    }
		    
		    /*  No 'break' statement here, since the bullet is allowed
			to go through more than one object.
		    */
		}
	    }
	}


	/*  Spiders:
	*/
	SpriteList::iterator its;
	for (its = spiderSprites.begin(); its != spiderSprites.end(); its++)
	{
	    Sprite *spider = *its;
	    if (spider == NULL)
		continue;
	    if (bullet->collidesWithSprite(*spider))
	    {
		playSoundEffect(spiderKilledSound);

		bullet->setTimeToLive(1);
		    // Patch: mark this bullet as having hit something.

		createAnimalExplosion(spider->getCenterPos(),
				spiderExplosionPA, spiderExplosionSprites);
		createScoreSprites(100, spider->getCenterPos());
		delete spider;
		*its = NULL;
		setTimeUntilNextAnimal();
		timeUntilNextAnimal += 2 * FRAMES_PER_SECOND;
		    // add extra time to allow explosion to disappear

		/*  No 'break' statement here, since the bullet is allowed
		    to go through more than one object.
		*/
	    }
	}


	/*  Bats:
	    The following block duplicates most of the preceding one -- sorry
	*/
	for (its = batSprites.begin(); its != batSprites.end(); its++)
	{
	    Sprite *bat = *its;
	    if (bat == NULL)
		continue;
	    if (bullet->collidesWithSprite(*bat))
	    {
		playSoundEffect(batKilledSound);

		bullet->setTimeToLive(1);
		    // Patch: mark this bullet as having hit something.

		createAnimalExplosion(bat->getCenterPos(),
				    batExplosionPA, batExplosionSprites);
		createScoreSprites(300, bat->getCenterPos());
		delete bat;
		*its = NULL;
		setTimeUntilNextAnimal();
		timeUntilNextAnimal += 2 * FRAMES_PER_SECOND;
		    // add extra time to allow explosion to disappear

		/*  No 'break' statement here, since the bullet is allowed
		    to go through more than one object.
		*/
	    }
	}


	/*  Player's bullets are absorbed by "big robot bullets":
	*/
	SpriteList::const_iterator itrb;
	for (itrb = robotBulletSprites.begin();
				itrb != robotBulletSprites.end(); itrb++)
	{
	    const Sprite *robotBullet = *itrb;
	    if (robotBullet == NULL)
		{ assert(false); continue; }
	    
	    if (robotBullet->getPixmapArray() == &bigRobotBulletPA
				&& bullet->collidesWithSprite(*robotBullet))
	    {
		playSoundEffect(missedSound);

		delete bullet;
		bullet = NULL;
		*itb = NULL;
		break;
	    }
	}


	/*  CAUTION: bullet and *itb may now be null.
	*/


	if (bullet != NULL && absorbedByBlinking)
	{
	    playSoundEffect(missedSound);

	    delete bullet;
	    bullet = NULL;
	    *itb = NULL;
	}

    }   // for itb


    removeNullElementsFromSpriteList(playerBulletSprites);
    removeNullElementsFromSpriteList(robotSprites);
    removeNullElementsFromSpriteList(spiderSprites);
    removeNullElementsFromSpriteList(batSprites);


    if (numPlayerLives > 0)
    {
	/*  Robots' bullets vs human:
	*/
	for (itb = robotBulletSprites.begin();
			    itb != robotBulletSprites.end(); itb++)
	{
	    const Sprite *bullet = *itb;
	    if (bullet == NULL)
		{ assert(false); continue; }

	    if (bullet->collidesWithSprite(*playerSprite))
		break;
	}
	bool playerHit = (itb != robotBulletSprites.end());

	if (playerHit && invinciblePlayer == NULL)
	{
	    makePlayerDie();
	    return;
	}

	/*  Human vs robots:
	*/
	if (invinciblePlayer == NULL)
	{
	    SpriteList::iterator its;

	    for (its = spiderSprites.begin();
			    its != spiderSprites.end(); its++)
	    {
		Sprite *spider = *its;
		if (spider == NULL)
		    { assert(false); continue; }
		if (spider->collidesWithSprite(*playerSprite))
		{
		    paralyzePlayer();
		    break;
		}
	    }

	    if (its == spiderSprites.end())  // if player survived
	    {
		SpriteList::iterator itb;
		for (itb = batSprites.begin(); itb != batSprites.end(); itb++)
		{
		    Sprite *bat = *itb;
		    if (bat == NULL)
			{ assert(false); continue; }
		    if (bat->collidesWithSprite(*playerSprite))
		    {
			paralyzePlayer();
			break;
		    }
		}
	    }
	}

    }
}


void
AfternoonStalkerEngine::makePlayerDie()
{
    playSoundEffect(playerHitSound);

    createPlayerExplosion(*playerSprite);


    playerSprite->setPos(initPlayerPos);
    playerSprite->currentPixmapIndex = 0;
    numPlayerBullets = 0;

    addToNumPlayerLives(-1);

    if (numPlayerLives == 0)
    {
	// The "demo" mode will only show the gray robots.
	numCreatedRobots = 0;
    }
}


void
AfternoonStalkerEngine::paralyzePlayer()
{
    playerParalysisTime = FRAMES_PER_SECOND * 3;
}


void
AfternoonStalkerEngine::createRobotExplosion(const RobotSprite &r)
{
    const PixmapArray *pa = r.getExplosionPA();
    Couple pos = r.getCenterPos() - pa->getImageSize() / 2;
    Couple speed[2] = { Couple(-2, -8), Couple(+2, -8) };
    Couple accel = Couple(0, +1);  // gravity goes down (duh)

    for (size_t i = 0; i < 2; i++)
    {
	Sprite *expl = new Sprite(*pa,
				pos, speed[i], accel, Couple(), Couple());
	expl->currentPixmapIndex = i;
	robotExplosionSprites.push_back(expl);
    }
}


void
AfternoonStalkerEngine::createAnimalExplosion(Couple pos,
				    const PixmapArray &pa, SpriteList &slist)
/*  Creates an explosion around the position of sprite 's', using the
    designated pixmap array, and putting the created sprites in the
    designated sprite list.
*/
{
    const size_t numFragments = 32;
    for (size_t i = 0; i < numFragments; i++)
    {
	double angle = double(i) / numFragments * TWO_PI * 2;
	double radius = rand() % 8;
	int x = (int) floor(pos.x + radius * cos(angle) + 0.5);
	int y = (int) floor(pos.y + radius * sin(angle) + 0.5);
	double da = (rand() % 21 - 10) * (PI / 180.0);
	int dx = (int) floor(4.0 * cos(angle + da) + 0.5);
	int dy = (int) floor(4.0 * sin(angle + da) + 0.5);

	Sprite *expl = new Sprite(pa, Couple(x, y), Couple(dx, dy),
				    Couple(), Couple(), Couple());
	expl->setTimeToLive(FRAMES_PER_SECOND * 3 / 2);
	slist.push_back(expl);
    }
}


void
AfternoonStalkerEngine::createPlayerExplosion(const Sprite &s)
{
    Couple pos = s.getCenterPos();

    const size_t numFragments = 128;
    for (size_t i = 0; i < numFragments; i++)
    {
	double angle = rand() * TWO_PI / RAND_MAX;
	double radius = 2.0;
	int x = (int) floor(pos.x + radius * cos(angle) + 0.5);
	int y = (int) floor(pos.y + radius * sin(angle) + 0.5);
	int speed = rand() % 6 + 2;
	int dx = (int) floor(speed * cos(angle) + 0.5);
	int dy = (int) floor(speed * sin(angle) + 0.5);

	Sprite *expl = new Sprite(playerExplosionPA,
				    Couple(x, y), Couple(dx, dy),
				    Couple(), Couple(), Couple());
	expl->setTimeToLive(FRAMES_PER_SECOND * 2);
	playerExplosionSprites.push_back(expl);
    }
}


void
AfternoonStalkerEngine::addToScore(long n)
{
    const long step = 10000;  // add new life at every <step> points

    long oldInterval = theScore / step;

    theScore += n;

    long newInterval = theScore / step;
    if (oldInterval < newInterval)
    {
	addToNumPlayerLives(1);
	playSoundEffect(newLifeSound);
    }
}


void
AfternoonStalkerEngine::createScoreSprites(long n, Couple center)
{
    addToScore(n);

    long absval = (n < 0 ? -n : n);

    char number[64];
    snprintf(number, sizeof(number), "%ld", absval);
    size_t numDigits = strlen(number);

    Couple digitSize = digitPA.getImageSize();
    Couple totalSize((digitSize.x + 2) * numDigits - 2, digitSize.y);
    Couple scorePos = center - totalSize / 2;

    for (size_t i = 0; i < numDigits; i++)
    {
        int digit = number[i] - '0';
        Sprite *s = new Sprite(digitPA,
			    scorePos + i * Couple(digitSize.x + 2, 0),
			    Couple(0, -1), Couple(0, 0),
			    Couple(0, 0), Couple(0, 0));
        s->setTimeToLive(SCORE_TTL);
	s->currentPixmapIndex = digit;
        scoreSprites.push_back(s);
    }
}


void
AfternoonStalkerEngine::loadPixmaps() throw(PixmapLoadError)
{
    /*  Tiles:
    */
    loadPixmap(wall_tile_xpm,        tilePA, WALL_TILE);
    loadPixmap(corner_tile_0_xpm,    tilePA, CORNER0_TILE);
    loadPixmap(corner_tile_1_xpm,    tilePA, CORNER1_TILE);
    loadPixmap(corner_tile_2_xpm,    tilePA, CORNER2_TILE);
    loadPixmap(corner_tile_3_xpm,    tilePA, CORNER3_TILE);
    loadPixmap(wall_end_tile_0_xpm,  tilePA, WALLEND0_TILE);
    loadPixmap(wall_end_tile_1_xpm,  tilePA, WALLEND1_TILE);
    loadPixmap(wall_end_tile_2_xpm,  tilePA, WALLEND2_TILE);
    loadPixmap(wall_end_tile_3_xpm,  tilePA, WALLEND3_TILE);
    loadPixmap(floor_tile_xpm,       tilePA, FLOOR_TILE);
    loadPixmap(cobweb_tile_xpm,      tilePA, COBWEB_TILE);
    loadPixmap(bunker_tile_xpm,      tilePA, BUNKER_TILE);
    loadPixmap(bunker_tile_0_xpm,    tilePA, BUNKER0_TILE);
    loadPixmap(bunker_tile_1_xpm,    tilePA, BUNKER1_TILE);
    loadPixmap(bunker_tile_2_xpm,    tilePA, BUNKER2_TILE);
    loadPixmap(bunker_tile_3_xpm,    tilePA, BUNKER3_TILE);
    loadPixmap(bunker_door_tile_xpm, tilePA, BUNKER_DOOR_TILE);


    /*  Sprites:
    */
    loadPixmap(human_front_0_xpm, playerPA, 0);
    loadPixmap(human_front_1_xpm, playerPA, 1);
    loadPixmap(human_front_2_xpm, playerPA, 2);
    loadPixmap(human_paralyzed_0_xpm, playerPA, 3);
    loadPixmap(human_bullet_xpm, playerBulletPA, 0);
    loadPixmap(robot_bullet_xpm, robotBulletPA, 0);
    loadPixmap(big_robot_bullet_0_xpm, bigRobotBulletPA, 0);
    loadPixmap(big_robot_bullet_1_xpm, bigRobotBulletPA, 1);
    loadPixmap(big_robot_bullet_2_xpm, bigRobotBulletPA, 2);
    loadPixmap(gun_xpm, gunPA, 0);
    loadPixmap(human_explosion_0_xpm, playerExplosionPA, 0);
    loadPixmap(gray_robot_0_xpm, grayRobotPA, 0);
    loadPixmap(gray_robot_1_xpm, grayRobotPA, 1);
    loadPixmap(blue_robot_0_xpm, blueRobotPA, 0);
    loadPixmap(blue_robot_1_xpm, blueRobotPA, 1);
    loadPixmap(white_robot_0_xpm, whiteRobotPA, 0);
    loadPixmap(white_robot_1_xpm, whiteRobotPA, 1);
    loadPixmap(black_robot_0_xpm, blackRobotPA, 0);
    loadPixmap(black_robot_1_xpm, blackRobotPA, 1);
    loadPixmap(invisible_robot_0_xpm, invisibleRobotPA, 0);
    loadPixmap(invisible_robot_1_xpm, invisibleRobotPA, 1);
    loadPixmap(blink_xpm, blinkPA, 0);

    loadPixmap(gray_robot_explosion_0_xpm, grayRobotExplosionPA, 0);
    loadPixmap(gray_robot_explosion_1_xpm, grayRobotExplosionPA, 1);
    loadPixmap(blue_robot_explosion_0_xpm, blueRobotExplosionPA, 0);
    loadPixmap(blue_robot_explosion_1_xpm, blueRobotExplosionPA, 1);
    loadPixmap(white_robot_explosion_0_xpm, whiteRobotExplosionPA, 0);
    loadPixmap(white_robot_explosion_1_xpm, whiteRobotExplosionPA, 1);
    loadPixmap(black_robot_explosion_0_xpm, blackRobotExplosionPA, 0);
    loadPixmap(black_robot_explosion_1_xpm, blackRobotExplosionPA, 1);
    loadPixmap(invisible_robot_explosion_0_xpm, invisibleRobotExplosionPA, 0);
    loadPixmap(invisible_robot_explosion_1_xpm, invisibleRobotExplosionPA, 1);


    /*  Order of directions same as enum constants RIGHT, UP, LEFT, DOWN:
    */
    loadPixmap(spider_right_0_xpm, spiderPA, 0);
    loadPixmap(spider_right_1_xpm, spiderPA, 1);
    loadPixmap(spider_up_0_xpm, spiderPA, 2);
    loadPixmap(spider_up_1_xpm, spiderPA, 3);
    loadPixmap(spider_left_0_xpm, spiderPA, 4);
    loadPixmap(spider_left_1_xpm, spiderPA, 5);
    loadPixmap(spider_down_0_xpm, spiderPA, 6);
    loadPixmap(spider_down_1_xpm, spiderPA, 7);

    loadPixmap(spider_explosion_0_xpm, spiderExplosionPA, 0);


    loadPixmap(bat_flying_0_xpm, batPA, 0);
    loadPixmap(bat_flying_1_xpm, batPA, 1);
    loadPixmap(bat_flying_2_xpm, batPA, 2);
    loadPixmap(bat_flying_3_xpm, batPA, 3);

    loadPixmap(bat_explosion_0_xpm, batExplosionPA, 0);


    /*  Score digits:
    */
    loadPixmap(digit0_xpm, digitPA, 0);
    loadPixmap(digit1_xpm, digitPA, 1);
    loadPixmap(digit2_xpm, digitPA, 2);
    loadPixmap(digit3_xpm, digitPA, 3);
    loadPixmap(digit4_xpm, digitPA, 4);
    loadPixmap(digit5_xpm, digitPA, 5);
    loadPixmap(digit6_xpm, digitPA, 6);
    loadPixmap(digit7_xpm, digitPA, 7);
    loadPixmap(digit8_xpm, digitPA, 8);
    loadPixmap(digit9_xpm, digitPA, 9);
}


void
AfternoonStalkerEngine::restoreBackground()
{
    // Make the top part black:
    SDL_Rect rect = { 0, 0, theScreenSizeInPixels.x, TILE_MATRIX_Y_OFFSET };
    (void) SDL_FillRect(theSDLScreen, &rect, blackColor);


    const size_t levelHeight = tileMatrix.size();
    const size_t levelWidth = tileMatrix[0].size();

    size_t x, y;
    for (y = 0; y < levelHeight; y++)
    {
	Couple dest(0, y * TILE_SIDE + TILE_MATRIX_Y_OFFSET);
	for (x = 0; x < levelWidth; x++, dest.x += TILE_SIDE)
	{
	    TileNo tileNo = tileMatrix[y][x];
	    if (tileNo == COBWEB_TILE)  // this is a foreground tile
		tileNo = FLOOR_TILE;  // so we don't draw it now
	    Pixmap p = tilePA.getImage(size_t(tileNo));
	    copyPixmap(p, dest);
	}
    }
}


void
AfternoonStalkerEngine::restoreForeground()
{
    const size_t levelHeight = tileMatrix.size();
    const size_t levelWidth = tileMatrix[0].size();

    Pixmap p = tilePA.getImage(COBWEB_TILE);  // only one foreground tile

    size_t x, y;
    for (y = 0; y < levelHeight; y++)
    {
	Couple dest(0, y * TILE_SIDE + TILE_MATRIX_Y_OFFSET);
	for (x = 0; x < levelWidth; x++, dest.x += TILE_SIDE)
	{
	    TileNo tileNo = tileMatrix[y][x];
	    if (tileNo == COBWEB_TILE)
		copyPixmap(p, dest);
	}
    }
}


inline void
AfternoonStalkerEngine::putSprite(const Sprite &s)
{
    copySpritePixmap(s, s.currentPixmapIndex, s.getPos());
}


void
AfternoonStalkerEngine::putSpriteList(const SpriteList &slist)
{
    SpriteList::const_iterator it;
    for (it = slist.begin(); it != slist.end(); it++)
    {
	if (*it == NULL)
	    { assert(false); continue; }
	putSprite(**it);
    }
}


void
AfternoonStalkerEngine::drawSprites()
{
    SpriteList::const_iterator it;

    // Draw the robots, possibly in blinking mode:
    for (it = robotSprites.begin(); it != robotSprites.end(); it++)
    {
	RobotSprite *robot = dynamic_cast<RobotSprite *>(*it);
	if (robot == NULL)
	    { assert(false); continue; }

	bool visible = (robot->getPixmapArray() != &invisibleRobotPA);
	unsigned long n = robot->endOfBlinkTime - tickCount;
	bool blink = (tickCount < robot->endOfBlinkTime && (n & 2) == 0);

	if (blink)
	{
	    Couple pos = robot->getCenterPos() - blinkPA.getImageSize() / 2;
	    copySpritePixmap(*robot->getBlinkSprite(), 0, pos);
	}

	if (visible || blink)
	    putSprite(*robot);
    }


    // Animals:
    putSpriteList(spiderSprites);
    putSpriteList(batSprites);


    // Explosions:
    putSpriteList(robotExplosionSprites);
    putSpriteList(spiderExplosionSprites);
    putSpriteList(batExplosionSprites);
    putSpriteList(playerExplosionSprites);


    putSpriteList(robotBulletSprites);
    putSpriteList(playerBulletSprites);

    if (!playerExplosionSprites.empty())  // if player is dying
	return;


    // Make the gun(s) blink if applicable:
    if (tickCount % FRAMES_PER_SECOND < FRAMES_PER_SECOND * 3 / 4)
	putSpriteList(gunSprites);


    // Score sprites:
    for (it = scoreSprites.begin(); it != scoreSprites.end(); it++)
	putSprite(**it);


    // The player, if alive:
    if (numPlayerLives > 0)
	putSprite(*playerSprite);
}


void
AfternoonStalkerEngine::drawScoreBoard()
{
    char temp[128];
    snprintf(temp, sizeof(temp), "Score: %7ld", theScore);
    writeString(temp, scorePos);
    snprintf(temp, sizeof(temp), "Lives: %3u\n", numPlayerLives);
    writeString(temp, playerLivesPos);
}


void
AfternoonStalkerEngine::displayPauseMessage(bool display)
/*  Displays the pause message if 'display' is true, or erases the
    corresponding region if 'display' is false.
*/
{
    const char *s = (display ? "PAUSED -- press P to resume" : "");
    displayMessage(s);
}


void
AfternoonStalkerEngine::displayStartMessage(bool display)
/*  Displays the start message if 'display' is true, or erases the
    corresponding region if 'display' is false.
*/
{
    if (!display)
	displayMessage("");
    else
    {
	static unsigned long firstTick = tickCount;

	const char *msg = "";
	switch ((tickCount - firstTick) / 32 % 3)
	{
	    case 0:
		msg = "Afternoon Stalker " VERSION " - by Pierre Sarrazin";
		break;
	    case 1:
		msg = "Move with arrow keys - shoot with Ctrl key";
		break;
	    case 2:
		msg = "Press SPACE to start";
		break;
	    default:
		assert(false);
		msg = "";
	}

	displayMessage(msg);
    }
}


void
AfternoonStalkerEngine::displayMessage(const char *msg)
/*  Displays the start message if 'display' is true, or erases the
    corresponding region if 'display' is false.
*/
{
    static const Couple fontdim = getFontDimensions();
    
    const size_t len = strlen(msg);
    int x = (theScreenSizeInPixels.x - len * fontdim.x) / 2;
    int y = 6;

    writeString(msg, Couple(x, y));
}


void
AfternoonStalkerEngine::addToNumPlayerLives(int n)
{
    numPlayerLives += n;
}


void
AfternoonStalkerEngine::playSoundEffect(SoundMixer::Chunk &wb)
{
    /*  'theSoundMixer' is still NULL if the sound effects were not wanted.
    */
    if (theSoundMixer != NULL)
    {
	try
	{
	    theSoundMixer->playChunk(wb);
	}
	catch (const SoundMixer::Error &e)
	{
	    fprintf(stderr, "playSoundEffect: %s\n", e.what().c_str());
	}
    }
}
