/********************************************************************************
*                                                                               *
*                S h u t t e r   B u g   A p p l i c a t i o n                  *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This program is free software; you can redistribute it and/or modify          *
* it under the terms of the GNU General Public License as published by          *
* the Free Software Foundation; either version 2 of the License, or             *
* (at your option) any later version.                                           *
*                                                                               *
* This program is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
* GNU General Public License for more details.                                  *
*                                                                               *
* You should have received a copy of the GNU General Public License             *
* along with this program; if not, write to the Free Software                   *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************
* $Id: ShutterBug.cpp,v 1.18 2003/09/10 03:54:37 fox Exp $                      *
********************************************************************************/
#include "fx.h"
#include "fxkeys.h"
#ifdef HAVE_PNG_H
#include "FXPNGImage.h"
#endif
#ifdef HAVE_JPEG_H
#include "FXJPGImage.h"
#endif
#ifdef HAVE_TIFF_H
#include "FXTIFImage.h"
#endif
#include "FXICOImage.h"
#include "FXTGAImage.h"
#include "FXRGBImage.h"
#include <stdio.h>
#include <stdlib.h>
#include "icons.h"
#include "ShutterBug.h"
#include "Snapper.h"


/*
  Notes:
  - In fixed size mode you can just drag the snap rectangle around;
    it stays the same size.
  - Now remembers last selected filename and file type.
*/

#define FUDGE         10        // Corner fudge for diagonal dragging
#define MINSIZE       8 // Minimum snap size
#define VERSION_MAJOR 1
#define VERSION_MINOR 0
#define VERSION_PATCH 0

/*******************************************************************************/

// Map
FXDEFMAP(ShutterBug) ShutterBugMap[]={
  FXMAPFUNC(SEL_PAINT,0,ShutterBug::onPaint),
  FXMAPFUNC(SEL_MOTION,0,ShutterBug::onMotion),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,ShutterBug::onBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,ShutterBug::onBtnRelease),
  FXMAPFUNC(SEL_MIDDLEBUTTONPRESS,0,ShutterBug::onBtnPress),
  FXMAPFUNC(SEL_MIDDLEBUTTONRELEASE,0,ShutterBug::onBtnRelease),
  FXMAPFUNC(SEL_RIGHTBUTTONPRESS,0,ShutterBug::onBtnPress),
  FXMAPFUNC(SEL_RIGHTBUTTONRELEASE,0,ShutterBug::onBtnRelease),
  FXMAPFUNC(SEL_KEYPRESS,0,ShutterBug::onKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,0,ShutterBug::onKeyRelease),
  FXMAPFUNC(SEL_COMMAND,ShutterBug::ID_SNAPSHOT,ShutterBug::onCmdSnap),
  FXMAPFUNC(SEL_COMMAND,ShutterBug::ID_ABOUT,ShutterBug::onCmdAbout),
  FXMAPFUNC(SEL_UPDATE,ShutterBug::ID_TOGGLE_LASSO,ShutterBug::onUpdLasso),
  FXMAPFUNC(SEL_COMMAND,ShutterBug::ID_TOGGLE_LASSO,ShutterBug::onCmdLasso),
  FXMAPFUNCS(SEL_LEFTBUTTONPRESS,ShutterBug::ID_SNAPPER_0,ShutterBug::ID_SNAPPER_3,ShutterBug::onPressSnapper),
  FXMAPFUNCS(SEL_LEFTBUTTONRELEASE,ShutterBug::ID_SNAPPER_0,ShutterBug::ID_SNAPPER_3,ShutterBug::onReleaseSnapper),
  FXMAPFUNCS(SEL_MOTION,ShutterBug::ID_SNAPPER_0,ShutterBug::ID_SNAPPER_3,ShutterBug::onMovedSnapper),
  FXMAPFUNCS(SEL_ENTER,ShutterBug::ID_SNAPPER_0,ShutterBug::ID_SNAPPER_3,ShutterBug::onEnterSnapper),
  FXMAPFUNCS(SEL_LEAVE,ShutterBug::ID_SNAPPER_0,ShutterBug::ID_SNAPPER_3,ShutterBug::onLeaveSnapper),
  FXMAPFUNCS(SEL_UPDATE,ShutterBug::ID_SIZE_CUSTOM,ShutterBug::ID_SIZE_512X512,ShutterBug::onUpdSize),
  FXMAPFUNCS(SEL_COMMAND,ShutterBug::ID_SIZE_CUSTOM,ShutterBug::ID_SIZE_512X512,ShutterBug::onCmdSize),
  };


// Implementation
FXIMPLEMENT(ShutterBug,FXMainWindow,ShutterBugMap,ARRAYNUMBER(ShutterBugMap))

// Pattern wildcards
const FXchar patterns[]=
    "GIF Image (*.gif)"
  "\nBMP Image (*.bmp)"
  "\nXPM Image (*.xpm)"
  "\nPCX Image (*.pcx)"
  "\nRGB Image (*.rgb)"
  "\nXBM Image (*.xbm)"
  "\nTARGA Image (*.tga)"
  "\nPPM Image (*.ppm)"
#ifdef HAVE_PNG_H
  "\nPNG Image (*.png)"
#endif
#ifdef HAVE_JPEG_H
  "\nJPEG Image (*.jpg)"
#endif
#ifdef HAVE_TIFF_H
  "\nTIFF Image (*.tif)"
#endif
  "\nPS Image (*.ps)"
  ;

// Pattern numbers
enum {
   TYPE_GIF
  ,TYPE_BMP
  ,TYPE_XPM
  ,TYPE_PCX
  ,TYPE_RGB
  ,TYPE_XBM
  ,TYPE_TGA
  ,TYPE_PPM
#ifdef HAVE_PNG_H
  ,TYPE_PNG
#endif
#ifdef HAVE_JPEG_H
  ,TYPE_JPG
#endif
#ifdef HAVE_TIFF_H
  ,TYPE_TIF
#endif
  ,TYPE_PS
  };


/*******************************************************************************/

// ShutterBug main window
ShutterBug::ShutterBug(FXApp* a):FXMainWindow(a,"ShutterBug",NULL,NULL,DECOR_NONE, 0,0,0,0, 0,0){
  flags|=FLAG_ENABLED;
  rectangle.x=100;
  rectangle.y=100;
  rectangle.w=100;
  rectangle.h=100;
  filename="image.gif";
  fileformat=TYPE_GIF;
  snapper[0]=new Snapper(getApp(),this,ID_SNAPPER_0);
  snapper[1]=new Snapper(getApp(),this,ID_SNAPPER_1);
  snapper[2]=new Snapper(getApp(),this,ID_SNAPPER_2);
  snapper[3]=new Snapper(getApp(),this,ID_SNAPPER_3);
  bigicon=new FXGIFIcon(getApp(),shutterbug);
  smallicon=new FXGIFIcon(getApp(),tinyshutterbug);
  setIcon(bigicon);
  setMiniIcon(smallicon);
  weight=3;
  size=0;
  spotx=0;
  spoty=0;
  mode=MODE_NONE;
  }


// Create and show window
void ShutterBug::create(){
  readRegistry();
  FXMainWindow::create();
  snapper[0]->create();
  snapper[1]->create();
  snapper[2]->create();
  snapper[3]->create();
  moveSnapRectangle(rectangle);
  showSnapRectangle();
  show();
  }


// Move snap rectangle
void ShutterBug::moveSnapRectangle(const FXRectangle& r){
  snapper[0]->position(r.x,r.y,weight,r.h);
  snapper[1]->position(r.x+r.w-weight,r.y,weight,r.h);
  snapper[2]->position(r.x,r.y,r.w,weight);
  snapper[3]->position(r.x,r.y+r.h-weight,r.w,weight);
  }


// Show snap rectangle
void ShutterBug::showSnapRectangle(){
  snapper[0]->show();
  snapper[1]->show();
  snapper[2]->show();
  snapper[3]->show();
  snapper[0]->raise();
  snapper[1]->raise();
  snapper[2]->raise();
  snapper[3]->raise();
  getApp()->flush(TRUE);
  }


// Hide snap rectangle
void ShutterBug::hideSnapRectangle(){
  snapper[0]->hide();
  snapper[1]->hide();
  snapper[2]->hide();
  snapper[3]->hide();
  getApp()->flush(TRUE);
  fxsleep(10000);
  }


// Is snap rectangle shown
FXbool ShutterBug::snapRectangleShown() const {
  return snapper[0]->shown();
  }



// Get default width
FXint ShutterBug::getDefaultWidth(){
  return bigicon->getWidth()+4;
  }


// Get default height
FXint ShutterBug::getDefaultHeight(){
  return bigicon->getHeight()+4;
  }


// Read registry
void ShutterBug::readRegistry(){
  filename=getApp()->reg().readStringEntry("SETTINGS","filename","image.gif");
  fileformat=getApp()->reg().readIntEntry("SETTINGS","fileformat",TYPE_GIF);
  setX(getApp()->reg().readIntEntry("SETTINGS","x",50));
  setY(getApp()->reg().readIntEntry("SETTINGS","y",50));
  rectangle.x=getApp()->reg().readIntEntry("SETTINGS","snapx",50);
  rectangle.y=getApp()->reg().readIntEntry("SETTINGS","snapy",50);
  rectangle.w=getApp()->reg().readIntEntry("SETTINGS","snapw",50);
  rectangle.h=getApp()->reg().readIntEntry("SETTINGS","snaph",50);
  size=getApp()->reg().readIntEntry("SETTINGS","size",0);
  if(size){ rectangle.w=size; rectangle.h=size; }
  }


// Write registry
void ShutterBug::writeRegistry(){
  getApp()->reg().writeStringEntry("SETTINGS","filename",filename.text());
  getApp()->reg().writeIntEntry("SETTINGS","fileformat",fileformat);
  getApp()->reg().writeIntEntry("SETTINGS","x",getX());
  getApp()->reg().writeIntEntry("SETTINGS","y",getY());
  getApp()->reg().writeIntEntry("SETTINGS","snapx",rectangle.x);
  getApp()->reg().writeIntEntry("SETTINGS","snapy",rectangle.y);
  getApp()->reg().writeIntEntry("SETTINGS","snapw",rectangle.w);
  getApp()->reg().writeIntEntry("SETTINGS","snaph",rectangle.h);
  getApp()->reg().writeIntEntry("SETTINGS","size",size);
  }


// Find out where window was grabbed
FXuchar ShutterBug::where(FXint x,FXint y) const {
  FXuchar code=MODE_NONE;
  if(x<rectangle.x+FUDGE) code|=MODE_LEFT;
  if(rectangle.x+rectangle.w-FUDGE<=x) code|=MODE_RIGHT;
  if(y<rectangle.y+FUDGE) code|=MODE_TOP;
  if(rectangle.y+rectangle.h-FUDGE<=y) code|=MODE_BOTTOM;
  return code;
  }


// Change cursor based on location over window
void ShutterBug::changeCursor(FXint which,FXuchar drag){
  switch(drag){
    case MODE_TOP:
    case MODE_BOTTOM:
      snapper[which]->setDefaultCursor(getApp()->getDefaultCursor(DEF_DRAGH_CURSOR));
      snapper[which]->setDragCursor(getApp()->getDefaultCursor(DEF_DRAGH_CURSOR));
      break;
    case MODE_LEFT:
    case MODE_RIGHT:
      snapper[which]->setDefaultCursor(getApp()->getDefaultCursor(DEF_DRAGV_CURSOR));
      snapper[which]->setDragCursor(getApp()->getDefaultCursor(DEF_DRAGV_CURSOR));
      break;
    case MODE_TOPLEFT:
    case MODE_BOTTOMRIGHT:
      snapper[which]->setDefaultCursor(getApp()->getDefaultCursor(DEF_DRAGTL_CURSOR));
      snapper[which]->setDragCursor(getApp()->getDefaultCursor(DEF_DRAGTL_CURSOR));
      break;
    case MODE_TOPRIGHT:
    case MODE_BOTTOMLEFT:
      snapper[which]->setDefaultCursor(getApp()->getDefaultCursor(DEF_DRAGTR_CURSOR));
      snapper[which]->setDragCursor(getApp()->getDefaultCursor(DEF_DRAGTR_CURSOR));
      break;
    case MODE_WHOLERECT:
      snapper[which]->setDefaultCursor(getApp()->getDefaultCursor(DEF_MOVE_CURSOR));
      snapper[which]->setDragCursor(getApp()->getDefaultCursor(DEF_MOVE_CURSOR));
      break;
    default:
      snapper[which]->setDefaultCursor(getApp()->getDefaultCursor(DEF_ARROW_CURSOR));
      snapper[which]->setDragCursor(getApp()->getDefaultCursor(DEF_ARROW_CURSOR));
      break;
    }
  }


// Handle repaint
long ShutterBug::onPaint(FXObject*,FXSelector,void* ptr){
  register FXEvent *event=(FXEvent*)ptr;
  FXDCWindow dc(this,event);
  dc.setForeground(backColor);
  dc.fillRectangle(0,0,width,height);
  dc.setForeground(FXRGB(0,0,0));
  dc.drawRectangle(0,0,width-1,height-1);
  dc.drawIcon(icon,2,2);
  return 1;
  }


// Mouse motion
long ShutterBug::onMotion(FXObject*,FXSelector,void* ptr){
  FXEvent *event=(FXEvent*)ptr;
  if(flags&FLAG_PRESSED){
    move(event->root_x-spotx,event->root_y-spoty);
    return 1;
    }
  return 0;
  }


// Left button pressed
long ShutterBug::onBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent *event=(FXEvent*)ptr;
  grab();
  spotx=event->win_x;
  spoty=event->win_y;
  flags|=FLAG_PRESSED;
  return 1;
  }


// Left button released
long ShutterBug::onBtnRelease(FXObject*,FXSelector,void* ptr){
  FXEvent *event=(FXEvent*)ptr;
  ungrab();
  flags&=~FLAG_PRESSED;
  if(event->moved) return 1;
  FXMenuPane filemenu(this);
  new FXMenuCaption(&filemenu,"ShutterBug",smallicon);
  new FXMenuSeparator(&filemenu);
  new FXMenuCommand(&filemenu,"Snap...",NULL,this,ShutterBug::ID_SNAPSHOT);
  new FXMenuCheck(&filemenu,"Show lasso",this,ShutterBug::ID_TOGGLE_LASSO);
  FXMenuPane sizemenu(this);
  new FXMenuCascade(&filemenu,"Size",NULL,&sizemenu);
  new FXMenuRadio(&sizemenu,"8x8",this,ID_SIZE_8X8);
  new FXMenuRadio(&sizemenu,"16x16",this,ID_SIZE_16X16);
  new FXMenuRadio(&sizemenu,"32x32",this,ID_SIZE_32X32);
  new FXMenuRadio(&sizemenu,"64x64",this,ID_SIZE_64X64);
  new FXMenuRadio(&sizemenu,"128x128",this,ID_SIZE_128X128);
  new FXMenuRadio(&sizemenu,"256X256",this,ID_SIZE_256X256);
  new FXMenuRadio(&sizemenu,"512X512",this,ID_SIZE_512X512);
  new FXMenuRadio(&sizemenu,"Custom",this,ID_SIZE_CUSTOM);
  new FXMenuCommand(&filemenu,"About...",NULL,this,ShutterBug::ID_ABOUT);
  new FXMenuCommand(&filemenu,"Quit",NULL,this,ShutterBug::ID_CLOSE);
  filemenu.create();
  filemenu.popup(NULL,event->root_x,event->root_y);
  getApp()->runModalWhileShown(&filemenu);
  return 1;
  }


// Keyboard press
long ShutterBug::onKeyPress(FXObject* sender,FXSelector sel,void* ptr){
  if(((FXEvent*)ptr)->code==KEY_q || ((FXEvent*)ptr)->code==KEY_Q){
    close(TRUE);
    }
  return 1;
  }


// Keyboard release
long ShutterBug::onKeyRelease(FXObject*,FXSelector,void*){
  return 1;
  }


// Close window and save registry
FXbool ShutterBug::close(FXbool notify){
  writeRegistry();
  return FXMainWindow::close(notify);
  }


// Read pixels from root to image
void ShutterBug::readPixels(FXImage* image,const FXRectangle& r){
  FXDCWindow dc(image);
  dc.clipChildren(FALSE);
  dc.setFunction(BLT_SRC);
  dc.drawArea(getRoot(),r.x,r.y,r.w,r.h,0,0);
  }


// Snapshot rectangle
FXbool ShutterBug::snapRectangle(FXColor*& data,const FXRectangle& r){
  data=NULL;
  if(1<r.w && 1<r.h){
    FXTRACE((1,"size=%d %d\n",r.w,r.h));

    // Allocate memory for pixels
    if(FXCALLOC(&data,FXColor,r.w*r.h)){

      // Hide snap rectangle
      hideSnapRectangle();

      // Make image
      FXImage image(getApp(),data,IMAGE_KEEP,r.w,r.h);

      // Create it
      image.create();

      // Blit pixels from root window
      readPixels(&image,r);

      // Read back pixels
      image.restore();

      // Show snap rectangle
      showSnapRectangle();

      return TRUE;
      }
    }
  return FALSE;
  }


// Restore image from off-screen pixmap
long ShutterBug::onCmdSnap(FXObject*,FXSelector,void*){
  FXColor *data=NULL;
  FXbool ok=FALSE;

  // Try grab pixels
  if(snapRectangle(data,rectangle)){

    // Construct file dialog
    FXFileDialog savedialog(this,"Save Image");
    savedialog.setPatternList(patterns);
    savedialog.setCurrentPattern(fileformat);
    savedialog.setFilename(FXFile::absolute(filename));

    // Run file dialog
    if(savedialog.execute()){
      filename=savedialog.getFilename();
      fileformat=savedialog.getCurrentPattern();
      if(!FXFile::exists(filename) || FXMessageBox::question(this,MBOX_YES_NO,"Overwrite Document","Overwrite existing document: %s?",filename.text())==MBOX_CLICKED_YES){
        FXFileStream outfile;
        if(outfile.open(filename,FXStreamSave)){
          switch(fileformat){
            case TYPE_GIF: ok=fxsaveGIF(outfile,data,rectangle.w,rectangle.h); break;
            case TYPE_BMP: ok=fxsaveBMP(outfile,data,rectangle.w,rectangle.h); break;
            case TYPE_XPM: ok=fxsaveXPM(outfile,data,rectangle.w,rectangle.h); break;
            case TYPE_PCX: ok=fxsavePCX(outfile,data,rectangle.w,rectangle.h); break;
            case TYPE_RGB: ok=fxsaveRGB(outfile,data,rectangle.w,rectangle.h); break;
            case TYPE_XBM: ok=fxsaveXBM(outfile,data,rectangle.w,rectangle.h); break;
            case TYPE_TGA: ok=fxsaveTGA(outfile,data,rectangle.w,rectangle.h); break;
            case TYPE_PPM: ok=fxsavePPM(outfile,data,rectangle.w,rectangle.h); break;
#ifdef HAVE_PNG_H
            case TYPE_PNG: ok=fxsavePNG(outfile,data,rectangle.w,rectangle.h); break;
#endif
#ifdef HAVE_JPEG_H
            case TYPE_JPG: ok=fxsaveJPG(outfile,data,rectangle.w,rectangle.h,75); break;
#endif
#ifdef HAVE_TIFF_H
            case TYPE_TIF: ok=fxsaveTIF(outfile,data,rectangle.w,rectangle.h,0); break;
#endif
            case TYPE_PS: ok=fxsavePS(outfile,data,rectangle.w,rectangle.h); break;
            }
          outfile.close();
          }
        }
      }
    FXFREE(&data);
    }
  return 1;
  }


// Pressed on snapper
long ShutterBug::onPressSnapper(FXObject*,FXSelector sel,void* ptr){
  register FXint which=FXSELID(sel)-ID_SNAPPER_0;
  register FXEvent *event=(FXEvent*)ptr;
  if((event->state&CONTROLMASK) || (0<size)){
    mode=MODE_WHOLERECT;
    spotx=event->root_x-rectangle.x;
    spoty=event->root_y-rectangle.y;
    }
  else{
    mode=where(event->root_x,event->root_y);
    if(mode&MODE_TOP) spoty=event->root_y-rectangle.y;
    else if(mode&MODE_BOTTOM) spoty=event->root_y-rectangle.y-rectangle.h;
    if(mode&MODE_LEFT) spotx=event->root_x-rectangle.x;
    else if(mode&MODE_RIGHT) spotx=event->root_x-rectangle.x-rectangle.w;
    }
  changeCursor(which,mode);
  snapper[0]->raise();
  snapper[1]->raise();
  snapper[2]->raise();
  snapper[3]->raise();
  raise();
  return 1;
  }


// Release on snapper
long ShutterBug::onReleaseSnapper(FXObject*,FXSelector sel,void*){
  register FXint which=FXSELID(sel)-ID_SNAPPER_0;
  mode=MODE_NONE;
  changeCursor(which,mode);
  return 1;
  }


// Moved snapper
long ShutterBug::onMovedSnapper(FXObject*,FXSelector sel,void* ptr){
  register FXint which=FXSELID(sel)-ID_SNAPPER_0;
  register FXEvent *event=(FXEvent*)ptr;
  register FXuchar m;
  FXint t;
  FXTRACE((1,"%s::onMovedSnapper %d,%d\n",getClassName(),((FXEvent*)ptr)->win_x,((FXEvent*)ptr)->win_y));
  if(mode!=MODE_NONE){

    // Move whole rectangle
    if(mode&MODE_WHOLERECT){
      rectangle.x=event->root_x-spotx;
      rectangle.y=event->root_y-spoty;
      }

    // Move corner of rectangle
    else{

      // Vertical
      if(mode&MODE_TOP){
        t=rectangle.y+rectangle.h-event->root_y+spoty;
        if(t>=MINSIZE){ rectangle.h=t; rectangle.y=event->root_y-spoty; }
        }
      else if(mode&MODE_BOTTOM){
        t=event->root_y-spoty-rectangle.y;
        if(t>=MINSIZE){ rectangle.h=t; }
        }

      // Horizontal
      if(mode&MODE_LEFT){
        t=rectangle.x+rectangle.w-event->root_x+spotx;
        if(t>=MINSIZE){ rectangle.w=t; rectangle.x=event->root_x-spotx; }
        }
      else if(mode&MODE_RIGHT){
        t=event->root_x-spotx-rectangle.x;
        if(t>=MINSIZE){ rectangle.w=t; }
        }
      }

    // Update rectangle
    moveSnapRectangle(rectangle);
    m=mode;
    }
  else{
    if((event->state&CONTROLMASK) || (0<size))
      m=MODE_WHOLERECT;
    else
      m=where(event->root_x,event->root_y);
    }
  changeCursor(which,m);
  return 1;
  }


// Entered snapper window
long ShutterBug::onEnterSnapper(FXObject*,FXSelector sel,void* ptr){
  register FXuchar m=MODE_WHOLERECT;
  if(!(((FXEvent*)ptr)->state&CONTROLMASK) && (0==size)){
    m=where(((FXEvent*)ptr)->root_x,((FXEvent*)ptr)->root_y);
    }
  changeCursor(FXSELID(sel)-ID_SNAPPER_0,m);
  return 1;
  }


// Left snapper window
long ShutterBug::onLeaveSnapper(FXObject*,FXSelector,void*){
  return 1;
  }


// About box
long ShutterBug::onCmdAbout(FXObject*,FXSelector,void*){
  FXDialogBox about(this,"About ShutterBug",DECOR_TITLE|DECOR_BORDER,0,0,0,0, 10,10,0,0, 0,0);
  new FXLabel(&about,FXString::null,bigicon,FRAME_GROOVE|LAYOUT_SIDE_LEFT|LAYOUT_CENTER_Y|JUSTIFY_CENTER_X|JUSTIFY_CENTER_Y);
  FXVerticalFrame* side=new FXVerticalFrame(&about,LAYOUT_SIDE_RIGHT|LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 10,10,10,10, 0,0);
  new FXLabel(side,"ShutterBug",NULL,JUSTIFY_LEFT|ICON_BEFORE_TEXT|LAYOUT_FILL_X);
  new FXHorizontalSeparator(side,SEPARATOR_LINE|LAYOUT_FILL_X);
  new FXLabel(side,FXStringFormat("\nFOX Screenshot Utility, version %d.%d.%d.\nShutterBug uses the FOX Toolkit version %d.%d.%d.\nCopyright (C) 2003 Jeroen van der Zijp (jeroen@fox-toolkit.org).\n ",VERSION_MAJOR,VERSION_MINOR,VERSION_PATCH,FOX_MAJOR,FOX_MINOR,FOX_LEVEL),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X|LAYOUT_FILL_Y);
  FXButton *button=new FXButton(side,"&OK",NULL,&about,FXDialogBox::ID_ACCEPT,BUTTON_INITIAL|BUTTON_DEFAULT|FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT,0,0,0,0,32,32,2,2);
  button->setFocus();
  about.execute();
  return 1;
  }


// Toggle lasso display
long ShutterBug::onCmdLasso(FXObject*,FXSelector,void*){
  if(snapRectangleShown()) hideSnapRectangle(); else showSnapRectangle();
  return 1;
  }


// Update toggle lasso
long ShutterBug::onUpdLasso(FXObject* sender,FXSelector,void*){
  sender->handle(this,snapRectangleShown()?FXSEL(SEL_COMMAND,ID_CHECK):FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
  return 1;
  }


// Change size mode
long ShutterBug::onCmdSize(FXObject*,FXSelector sel,void*){
  size=FXSELID(sel)-ID_SIZE_CUSTOM;
  if(size){
    rectangle.w=size;
    rectangle.h=size;
    moveSnapRectangle(rectangle);
    }
  return 1;
  }


// Update change size mode
long ShutterBug::onUpdSize(FXObject* sender,FXSelector sel,void*){
  sender->handle(this,size==FXSELID(sel)-ID_SIZE_CUSTOM?FXSEL(SEL_COMMAND,ID_CHECK):FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
  return 1;
  }


// Destroy main window
ShutterBug::~ShutterBug(){
  delete snapper[0];
  delete snapper[1];
  delete snapper[2];
  delete snapper[3];
  delete bigicon;
  delete smallicon;
  }

/*******************************************************************************/


// Start the whole thing
int main(int argc,char *argv[]){

  // Make application
  FXApp application("ShutterBug",FXString::null);

  // Open display
  application.init(argc,argv);

  // Main window
  new ShutterBug(&application);

  // Create app
  application.create();

  // Run
  return application.run();
  }
