/********************************************************************************
*                                                                               *
*                     H i g h l i g h t   E n g i n e                           *
*                                                                               *
*********************************************************************************
* Copyright (C) 2002,2003 by Jeroen van der Zijp.   All Rights Reserved.        *
*********************************************************************************
* This program is free software; you can redistribute it and/or modify          *
* it under the terms of the GNU General Public License as published by          *
* the Free Software Foundation; either version 2 of the License, or             *
* (at your option) any later version.                                           *
*                                                                               *
* This program is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
* GNU General Public License for more details.                                  *
*                                                                               *
* You should have received a copy of the GNU General Public License             *
* along with this program; if not, write to the Free Software                   *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************
* $Id: Hilite.h,v 1.20 2003/09/25 13:08:00 fox Exp $                            *
********************************************************************************/
#ifndef HILITE_H
#define HILITE_H

#include "FXRex.h"


// Base highlight node
class HLNode {
  friend class Hilite;
protected:
  HLNode  *parent;      // Parent
  HLNode  *alt;         // Alternative
  HLNode  *sub;         // Sub
  FXint    context;     // Context
  FXint    style;       // Style
public:

  // Construct node
  HLNode(HLNode* p,FXint cx,FXint st):
    parent(p),alt(NULL),sub(NULL),context(cx),style(st){
    }

  // Stylize text
  virtual FXbool stylize(const FXchar* text,FXchar *textstyle,FXint fm,FXint to,FXint& start,FXint& stop) const = 0;

  // Style of the pattern node
  FXint getStyle() const { return style; }

  // Context of the pattern node
  FXint getContext() const { return context; }

  // Return parent node
  HLNode *getParent() const { return parent; }

  // Get first alternative node
  HLNode *getAlt() const { return alt; }

  // Get child of pattern node
  HLNode *getSub() const { return sub; }

  // Delete node
  virtual ~HLNode();
  };


// Simple highlight node
class HLSimple : public HLNode {
  friend class Hilite;
protected:
  FXRex pat;    // Match pattern
public:

  // Construct node
  HLSimple(HLNode* p,FXint cx,FXint st,const FXchar* rex):HLNode(p,cx,st),pat(rex,REX_NEWLINE){}

  // Stylize text
  virtual FXbool stylize(const FXchar* text,FXchar *textstyle,FXint fm,FXint to,FXint& start,FXint& stop) const;
  };


// Complex highlight node
class HLComplex : public HLNode {
  friend class Hilite;
protected:
  FXRex beg;    // Beginning pattern
  FXRex end;    // Ending pattern
public:

  // Construct node
  HLComplex(HLNode* p,FXint cx,FXint st,const FXchar* brex,const FXchar* erex):HLNode(p,cx,st),beg(brex,REX_NEWLINE),end(erex,REX_NEWLINE){}

  // Stylize text
  virtual FXbool stylize(const FXchar* text,FXchar *textstyle,FXint fm,FXint to,FXint& start,FXint& stop) const;
  };


// Complex highlight node with termination
class HLSafeComplex : public HLComplex {
  friend class Hilite;
protected:
  FXRex term;   // Termination pattern
public:

  // Construct node
  HLSafeComplex(HLNode* p,FXint cx,FXint st,const FXchar* brex,const FXchar* erex,const FXchar* srex):HLComplex(p,cx,st,brex,erex),term(srex,REX_NEWLINE){}

  // Stylize text
  virtual FXbool stylize(const FXchar* text,FXchar *textstyle,FXint fm,FXint to,FXint& start,FXint& stop) const;
  };


// Highlight engine
class Hilite {
private:
  HLNode  *root;        // Root of all syntax nodes
private:
  Hilite(const Hilite&);
  Hilite &operator=(const Hilite&);
  static void removenodes(HLNode* node);
public:

  // Create syntax coloring engine
  Hilite():root(NULL){}

  // Stylize text
  void stylize(const FXchar* text,FXchar *textstyle,FXint fm,FXint to) const;

  // Append simple highlight pattern
  HLNode *append(HLNode *parent,FXint context,FXint style,const FXchar* rex);

  // Append bracketing highlight pattern
  HLNode *append(HLNode *parent,FXint context,FXint style,const FXchar* brex,const FXchar* erex);

  // Append bracketing highlight pattern with termination pattern
  HLNode *append(HLNode *parent,FXint context,FXint style,const FXchar* brex,const FXchar* erex,const FXchar* srex);

  // Remove highlight pattern
  void remove(HLNode* node);

  // Find pattern node by style
  HLNode *find(FXint style) const;

  // Get root node
  HLNode *getRoot() const { return root; }

  // Clear all pattern nodes
  void clear();

  // Clean up
  ~Hilite();
  };


/*
// Language mode
class Language {
  FXString language;    // Language name
  FXString wildcards;   // File wildcard patterns for language
  FXRex    contents;    // If wildcard fails, match contents with this
  FXString delimiters;  // Word delimiters in this language
  Hilite   hilite;      // Colorizer for this language
  FXbool   wrap;        // State of word wrapping flag
private:
  Language(const Language&);
  Language &operator=(const Language&);
public:
  Language(const FXString& name,const FXString& pats,const FXString& delims,const FXString& delims
  };
*/


#endif

