/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
|  3/22/2000 Patch from Jason Kasper <j_dot_kasper@usa.net>:
|		Added option to run in withdrawn mode for the Blackbox slit.
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"

#include <X11/Xlib.h>
#include <X11/Xmd.h>
#include <X11/Xatom.h>

#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
#include <kvm.h>

kvm_t	*kvmd = NULL;
char	errbuf[_POSIX2_LINE_MAX];
#endif
#if defined(__FreeBSD__)
extern void scan_for_sensors();
#endif

struct ThemeConfig	GK;
struct UserConfig	UC;

struct
	{
	GtkWidget	*window;		/* Top level window	*/
	GtkWidget	*vbox;

	GtkWidget	*top0_event_box;	/* Top frame event box */
	GtkWidget	*top0_vbox;			/* Top frame			*/

	GtkWidget	*middle_hbox;
	  GtkWidget	*left_event_box;	/* Holds left vbox.		*/
	  GtkWidget	*left_vbox;			/* Holds left frame.	*/

	  GtkWidget	*middle_vbox;		/* A handle for sliding shut */
	  GtkWidget	*monitor_vbox;		/* All monitors go in here.	*/
		GtkWidget	*top1_event_box;	/* First event box */
		GtkWidget	*top1_vbox;			/* hostname	*/

	  GtkWidget	*right_event_box;	/* Holds right vbox.	*/
	  GtkWidget	*right_vbox;		/* Holds right frame.	*/

	GtkWidget	*bottom_vbox;	/* Bottom frame	*/

	GdkPixmap	*frame_left_pixmap;
	GdkBitmap	*frame_left_mask;
	GdkPixmap	*frame_right_pixmap;
	GdkBitmap	*frame_right_mask;

	GdkPixmap	*frame_top_pixmap;
	GdkBitmap	*frame_top_mask;
	GdkPixmap	*frame_bottom_pixmap;
	GdkBitmap	*frame_bottom_mask;

	GdkPixmap	*window_transparency_mask;
	}
	gtree;



GList			*gkrellm_monitor_list;
GtkWidget		*top_window;				/* Make static at 1.1.0 */

static GList	*chart_list;
static GList	*panel_list;

static GtkItemFactory *item_factory;

static Pixmap	root_xpixmap	= None;
static GdkGC	*trans_gc;

gint			w_display, h_display;
gint			window_decorations;
static gchar	*geometry;

static gint		x_position, y_position;

static gint		withdrawn;
static gint		total_frame_height;
static gint		initialized;
static gint		monitor_previous_height;
static gint		monitors_visible	= TRUE;
static gint		mask_monitors_visible = TRUE;
static gint		check_rootpixmap_transparency,
				check_frame_transparency;
static gint		no_transparency;
static gint		any_transparency;

static void		apply_rootpixmap_transparency(void);
static void		apply_frame_transparency(void);

static gchar	*fail_large_font[2] =
{
"-b&h-lucida-bold-r-normal-*-*-120-*-*-*-*-*-*",
"-adobe-courier-medium-r-*-*-*-120-*-*-*-*-*-*"
};

static gchar	*fail_normal_font[2] =
{
"-b&h-lucida-medium-r-normal-*-*-100-*-*-*-*-*-*",
"-adobe-courier-medium-r-*-*-*-100-*-*-*-*-*-*"
};

static gchar	*fail_small_font[2] =
{
"-adobe-helvetica-medium-r-normal-*-*-80-*-*-*-*-*-*",
"-adobe-courier-medium-r-*-*-*-80-*-*-*-*-*-*"
};


static void
setup_fonts()
	{
	GdkFont	*font;
	gint	i;

	if ((font = gdk_font_load(GK.large_font_string)) == NULL)
		{
		if (GK.debug_level & DEBUG_GKRELLMRC)
			printf(_("Could not load large font\n  \"%s\"\nFalling back...\n"),
						GK.large_font_string);
		for (i = 0; i < 2; ++i)
			if ((font = gdk_font_load(fail_large_font[i])) != NULL)
				break;
		}
	if (GK.large_font && ! gdk_font_equal(GK.large_font, font))
		gdk_font_unref(GK.large_font);
	GK.large_font = font;

	if ((font = gdk_font_load(GK.normal_font_string)) == NULL)
		{
		if (GK.debug_level & DEBUG_GKRELLMRC)
			printf(_("Could not load label font\n  \"%s\"\nFalling back...\n"),
						GK.normal_font_string);
		for (i = 0; i < 2; ++i)
			if ((font = gdk_font_load(fail_normal_font[i])) != NULL)
				break;
		}
	if (GK.normal_font && ! gdk_font_equal(GK.normal_font, font))
		gdk_font_unref(GK.normal_font);
	GK.normal_font = font;

	if ((font = gdk_font_load(GK.small_font_string)) == NULL)
		{
		if (GK.debug_level & DEBUG_GKRELLMRC)
			printf(_("Could not load alt font\n  \"%s\"\nFalling back...\n"),
							GK.small_font_string);
		for (i = 0; i < 2; ++i)
			if ((font = gdk_font_load(fail_small_font[i])) != NULL)
				break;
		}
	if (GK.small_font && ! gdk_font_equal(GK.small_font, font))
		gdk_font_unref(GK.small_font);
	GK.small_font = font;

	if (GK.large_font == NULL || GK.normal_font == NULL || GK.small_font == NULL)
		{
		printf(_("Error: Could not load all fonts.\n"));
		exit(0);
		}
	}

void
map_color_string(gchar *color_string, GdkColor *color)
	{
	static GdkColormap	*colormap;

	if (colormap == NULL)
		colormap = gtk_widget_get_colormap(top_window);
	if (color->red || color->green || color->blue)
		gdk_colormap_free_colors(colormap, color, 1);
	gdk_color_parse(color_string, color);
	gdk_color_alloc(colormap, color);
	}

static void
setup_colors()
	{
	GtkWidget	*win	= top_window;

	map_color_string(GK.chart_in_color, &GK.in_color);
	map_color_string(GK.chart_in_color_grid, &GK.in_color_grid);

	map_color_string(GK.chart_out_color, &GK.out_color);
	map_color_string(GK.chart_out_color_grid, &GK.out_color_grid);

	map_color_string("#000000", &GK.background_color);

	map_color_string("#FFFFFF", &GK.white_color);

	if (GK.draw1_GC == NULL)
		{
		GK.draw1_GC = gdk_gc_new( win->window );
		gdk_gc_copy( GK.draw1_GC, win->style->white_gc );
		}
	if (GK.draw2_GC == NULL)
		{
		GK.draw2_GC = gdk_gc_new( win->window );
		gdk_gc_copy( GK.draw2_GC, win->style->white_gc );
		}
	if (GK.draw3_GC == NULL)
		{
		GK.draw3_GC = gdk_gc_new( win->window );
		gdk_gc_copy( GK.draw3_GC, win->style->white_gc );
		}
	if (GK.draw_stencil_GC == NULL)
		{
		GK.draw_stencil_GC = gdk_gc_new( win->window );
		gdk_gc_copy(GK.draw_stencil_GC, win->style->white_gc );
		}
	if (GK.text_GC == NULL)
		{
		GK.text_GC = gdk_gc_new( win->window );
		gdk_gc_copy( GK.text_GC, win->style->white_gc );
		}

	/* Set up the depth 1 GCs
	*/
	/* printf("white pixel = %ld\n", GK.white_color.pixel); */
	if (GK.bit1_GC == NULL)
		{
		GdkBitmap	*dummy_bitmap;
		GdkColor	bit_color;

		dummy_bitmap = gdk_pixmap_new(top_window->window, 16, 16, 1);
		GK.bit1_GC = gdk_gc_new(dummy_bitmap);
		GK.bit0_GC = gdk_gc_new(dummy_bitmap);
		bit_color.pixel = 1;
		gdk_gc_set_foreground(GK.bit1_GC, &bit_color);
		bit_color.pixel = 0;
		gdk_gc_set_foreground(GK.bit0_GC, &bit_color);
		gdk_pixmap_unref(dummy_bitmap);
		}
	}



#define	WIN_LAYER_NORMAL	4
#define	WIN_LAYER_ONTOP		6

static Atom	xa_check;
static Atom	xa_layer;

  /* I think users should configure their window managers to keep GKrellM
  |  on top if that is what they want...
  */
static void
send_on_top_hint()
	{
	XEvent				xev;
	GdkWindowPrivate	*win;
	glong				data[1];

	win = (GdkWindowPrivate *) (GTK_WIDGET(top_window)->window);
	if (GTK_WIDGET_MAPPED(top_window))
		{
		xev.type = ClientMessage;
		xev.xclient.type = ClientMessage;
		xev.xclient.window = win->xwindow;
		xev.xclient.message_type = xa_layer;
		xev.xclient.format = 32;
		xev.xclient.data.l[0] = (CARD32) WIN_LAYER_ONTOP;
		xev.xclient.data.l[1] = gdk_time_get();
		XSendEvent(GDK_DISPLAY(), GDK_ROOT_WINDOW(), False,
			SubstructureNotifyMask, (XEvent *) & xev);
		}
	else
		{
		data[0] = WIN_LAYER_ONTOP;
		XChangeProperty(GDK_DISPLAY(), win->xwindow, xa_layer,
			XA_CARDINAL, 32, PropModeReplace, (guchar *) data, 1);
		}
	}

static void
set_on_top()
	{
	Window	win;
	Atom	type;
	gint	format;
	gulong	count, pad;
	guchar	*prop;
	gint	prev_error;

	prev_error = gdk_error_warnings;
	xa_check = XInternAtom(GDK_DISPLAY(), "_WIN_SUPPORTING_WM_CHECK", False);
	xa_layer = XInternAtom(GDK_DISPLAY(), "_WIN_LAYER", False);
	if (XGetWindowProperty(GDK_DISPLAY(), GDK_ROOT_WINDOW(), xa_check,
			0, 1, False, XA_CARDINAL, &type, &format, &count, &pad, &prop)
			== Success && prop)
		{
		win = *(long *) prop;
		XFree(prop);
		if (type == XA_CARDINAL && format == 32 && count == 1)
			{
			if (XGetWindowProperty(GDK_DISPLAY(), win, xa_check, 0, 1, False,
					XA_CARDINAL, &type, &format, &count, &pad, &prop)
					== Success && prop)
				{
				XFree(prop);
				if (type == XA_CARDINAL && format == 32 && count == 1)
					send_on_top_hint();
				}
			}
		}
	gdk_error_warnings = prev_error;
	}


static gint	save_position_countdown;

static void
set_or_save_position(gint save)
	{
	FILE		*f;
	gchar		path[128];
	gint		x, y;
	static gint	x_last = -1,
				y_last = -1;

	snprintf(path, sizeof(path), "%s/%s/startup_position", gkrellm_homedir(),
				GKRELLM_DATA_DIR);
	if (save)
		{
		if (   (x_position != x_last || y_position != y_last)
			&& (f = fopen(path, "w")) != NULL
		   )
			{
			x_last = x_position;
			y_last = y_position;
			fprintf(f, "%d %d\n", x_position, y_position);
			fclose(f);
			}
		save_position_countdown = 0;
		}
	else
		{
		if ((f = fopen(path, "r")) != NULL)
			{
			x = y = 0;
			fscanf(f, "%d %d", &x, &y);
			if (x >= 0 && x < w_display - 10 && y >= 0 && y < h_display - 25)
				{
				x_position = x_last = x;
				y_position = y_last = y;
				gdk_window_move(gtree.window->window, x, y);
				}
			}
		}
	}

static gint
update_monitors()
	{
	GList			*list;
	Monitor			*mon;
	static time_t	time_prev;

	time(&time_now);
	GK.second_tick = (time_now == time_prev) ? FALSE : TRUE;
	time_prev = time_now;

	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (mon->update_monitor && mon->private->enabled)
			(*(mon->update_monitor))();
		}
	++GK.timer_ticks;

	if (save_position_countdown > 0 && --save_position_countdown == 0)
		set_or_save_position(1);

	if (check_rootpixmap_transparency)
		apply_rootpixmap_transparency();
	if (check_frame_transparency)
		{
		/* Force a redraw of the mask */
		mask_monitors_visible = 1 - monitors_visible;
		apply_frame_transparency();
		}
	check_rootpixmap_transparency = FALSE;
	check_frame_transparency = FALSE;

	return TRUE;			/* This restarts timeout */
	}

void
start_timer(gint Hz)
	{
	static gint	timeout_id	= 0;
	gint		interval;

	if (timeout_id)
		gtk_timeout_remove(timeout_id);
	timeout_id = 0;
	if (Hz > 0)
		{
		interval = 1000 / Hz;
		interval = interval * 60 / 63;	/* Compensate for overhead XXX */
		timeout_id = gtk_timeout_add(interval,
						(GtkFunction) update_monitors,NULL);
		}
	}

GtkWidget *
gkrellm_get_top_window()
	{
	return top_window;
	}


  /* Nice set of effects I have here...  Either shadow effect or none.
  |  Returning 1 means shadow effect and the return value is also used by
  |  callers to increment height fields to allow for the offset shadow draw.
  */
gint
effect_string_value(gchar *effect_string)
	{
	if (effect_string)
		return (strcmp(effect_string, "shadow") == 0) ? 1 : 0;
	return 0;
	}

  /* Before 1.0.3 text, decals, krells used top/bottom borders for placement.
  |  This is to allow themes to transition to using top/bottom_margin.
  */
void
gkrellm_get_top_bottom_margins(Style *style, gint *top, gint *bottom)
	{
	gint	t = 0,
			b = 0;

	if (style)
		{
		if (GK.use_top_bottom_margins)
			{
			t = style->top_margin;
			b = style->bottom_margin;

			/* Ugh!! If a plugin violates policy and declares a static
			|  Style struct (and it's so), then pre 1.0.3 it won't have
			|  top/bottom_margin members.  So, range check'em.
			*/
			if (t < 0 || t > 25 || b < 0 || b > 25)
				{
				t = 2;
				b = 2;
				}
			}
		else
			{
			t = style->border.top;
			b = style->border.bottom;
			}
		}
	if (top)
		*top = t;
	if (bottom)
		*bottom = b;
	}

  /* Calculate panel height required for given border, label height,
  |  krell heights, and decal heights.  Also calculate label extents
  |  and load all this info into a Label structure.
  |  After panel height is calculated, calling routine may want to adjust
  |  krell or decal y offsets for centering.  
  */
void
gkrellm_configure_panel(Panel *p, gchar *string, Style *style)
	{
	GList			*list;
	Krell			*k;
	Decal			*d;
	Label			*lbl;
	gint			h_panel, h_decal;
	gint			top_margin = 0, bottom_margin = 0;

	if (p == NULL)
		return;
	p->style = style;
	lbl = p->label;
	lbl->string = string;
	lbl->position = style ? style->label_position : LABEL_CENTER;
	if (style)
		{
		p->transparency = style->transparency;
		any_transparency |= p->transparency;
		gkrellm_get_top_bottom_margins(style, &top_margin, &bottom_margin);
		}
	if (   lbl->string && p->textstyle && p->textstyle->font
		&& lbl->position >= 0
	   )
		gdk_string_extents(p->textstyle->font, string, &lbl->lbearing,
				&lbl->rbearing, &lbl->width, &lbl->ascent, &lbl->descent);
	else
		{
		lbl->lbearing = 0;
		lbl->rbearing = 0;
		lbl->width = 0;
		lbl->ascent = 0;
		lbl->descent = 0;
		}
	h_panel = top_margin + bottom_margin;
	h_panel += lbl->ascent + lbl->descent
				+ (p->textstyle ? p->textstyle->effect : 0);

	/* Krells have their own y_off and are not placed inside of margins.
	|  Decals are assumed to fit inside of margins.  Caller must subtract
	|  off bottom_margin if this is not so.
	*/
	for (list = p->krell; list; list = list->next)
		{
		k = (Krell *) list->data;
		if (k->y0 + k->h_frame > h_panel)
			h_panel = k->y0 + k->h_frame;
		}
	for (list = p->decal; list; list = list->next)
		{
		d = (Decal *) list->data;
		h_decal = d->y + d->h + bottom_margin;
		if (h_decal > h_panel)
			h_panel = h_decal;
		}

	if (h_panel <= 0)
		h_panel = 1;
	lbl->y_baseline = top_margin + lbl->ascent;
	lbl->h_panel = h_panel;
	}

void
gkrellm_draw_string(GdkDrawable *drawable, TextStyle *ts, gint x, gint y,
			gchar *s)
	{
	if (!drawable || !ts || !s)
		return;
	if (ts->effect)
		{
		gdk_gc_set_foreground(GK.text_GC, &ts->shadow_color);
		gdk_draw_string(drawable, ts->font, GK.text_GC, x + 1, y + 1, s);
		}
	gdk_gc_set_foreground(GK.text_GC, &ts->color);
	gdk_draw_string(drawable, ts->font, GK.text_GC, x, y, s);
	}

gint
gkrellm_draw_chart_label(Chart *cp, TextStyle *ts, gint x, gint y, gchar *s)
	{
	gint		w, l, r, a, d;

	if (!cp || !ts || !s)
		return x;
	gdk_string_extents(ts->font, s, &l, &r, &w, &a, &d);

	gdk_draw_pixmap(cp->pixmap, GK.draw1_GC,cp->bg_grided_pixmap,
		x - l, y - a, x - l, y - a, l + r + ts->effect, a + d + ts->effect);
    gkrellm_draw_string(cp->pixmap, ts, x, y, s);
	gdk_draw_pixmap(cp->drawing_area->window, GK.draw1_GC, cp->pixmap,
		x - l, y - a, x - l, y - a, l + r + ts->effect, a + d + ts->effect);

	return x + r;
	}

gint
x_label_position(gint position, gint w_field, gint w_label, gint margin)
	{
	gint	x;

	x = w_field * position / LABEL_MAX;
	x -= w_label / 2;
	if (x > w_field - w_label - margin)
		x = w_field - w_label - margin;
	if (x < margin)
		x = margin;
	return x;
	}

static void
draw_rootpixmap_onto_transparent_chart(Chart *cp)
	{
	Window	child;
	gint	x, y;

	if (root_xpixmap == None)
		return;
	XTranslateCoordinates(GDK_DISPLAY(),
			GDK_WINDOW_XWINDOW(cp->drawing_area->window),
			GDK_ROOT_WINDOW(),
			0, 0, &x, &y, &child);
	XSetTSOrigin(GDK_DISPLAY(), GDK_GC_XGC(trans_gc), -x, -y);

	/* First make the chart totally transparent
	*/
	gdk_draw_rectangle(cp->bg_pixmap, trans_gc, TRUE, 0, 0, cp->w, cp->h);

	/* If mode permits, stencil on non transparent parts of bg_clean_pixmap.
	*/
	if (cp->transparency == 2 && cp->bg_mask)
		{
        gdk_gc_set_clip_mask(GK.text_GC, cp->bg_mask);
        gdk_draw_pixmap(cp->bg_pixmap, GK.text_GC, cp->bg_clean_pixmap,
                    0, 0, 0, 0, cp->w, cp->h);
        gdk_gc_set_clip_mask(GK.text_GC, NULL);
		}
	}	

static void
draw_rootpixmap_onto_transparent_panel(Panel *p)
	{
	Window	child;
	gint	x, y;

	if (root_xpixmap == None)
		return;
	XTranslateCoordinates(GDK_DISPLAY(),
			GDK_WINDOW_XWINDOW(p->drawing_area->window),
			GDK_ROOT_WINDOW(),
			0, 0, &x, &y, &child);
	XSetTSOrigin(GDK_DISPLAY(), GDK_GC_XGC(trans_gc), -x, -y);

	/* First make the panel totally transparent
	*/
	gdk_draw_rectangle(p->bg_pixmap, trans_gc, TRUE, 0, 0, p->w, p->h);

	/* If mode permits, stencil on non transparent parts of bg_clean_pixmap.
	*/
	if (p->transparency == 2 && p->bg_mask)
		{
        gdk_gc_set_clip_mask(GK.text_GC, p->bg_mask);
        gdk_draw_pixmap(p->bg_pixmap, GK.text_GC, p->bg_clean_pixmap,
                    0, 0, 0, 0, p->w, p->h);
        gdk_gc_set_clip_mask(GK.text_GC, NULL);
		}
	}	


static void
draw_panel(Panel *p, gint to_screen)
	{
	Label		*lbl	= p->label;
	TextStyle	*ts		= p->textstyle;
	gchar		*s;
	gint		xdst, margin;

	if (root_xpixmap != None && p->transparency > 0)
		draw_rootpixmap_onto_transparent_panel(p);
	else
		gdk_draw_pixmap(p->bg_pixmap, GK.draw1_GC, p->bg_clean_pixmap,
					0, 0, 0, 0, p->w, p->h);

	gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->bg_pixmap,
					0, 0, 0, 0, p->w, p->h);
	if (   lbl && ts
		&& ((s = lbl->string) != NULL)
		&& lbl->position >= 0
	   )
		{
		lbl->width = gdk_string_width(ts->font, s);
		margin = p->style ? p->style->margin : 0;
		xdst = x_label_position(lbl->position, p->w, lbl->width, margin);
		lbl->x_panel = xdst;
		gkrellm_draw_string(p->pixmap, ts, xdst, lbl->y_baseline, s);
		gkrellm_draw_string(p->bg_pixmap, ts, xdst, lbl->y_baseline, s);
		}
	if (p->drawing_area->window && to_screen)
		{
		gdk_draw_pixmap(p->drawing_area->window, GK.draw1_GC, p->pixmap,
					0, 0, 0, 0, p->w, p->h);
		gkrellm_draw_layers_force(p);
		}
	}

void
gkrellm_draw_panel_label(Panel *p, GdkImlibImage *image /* obsolete */)
	{
	draw_panel(p, TRUE);
	}

static void
apply_rootpixmap_transparency()
	{
	Atom	prop,
			ret_type	= (Atom) 0;
	GList	*list;
	Chart	*cp;
	Panel	*p;
	guchar	*prop_return = NULL;
	gint	fmt;
	gulong	nitems, bytes_after;
	gint	depth_visual;
	Window	root_return;
	guint	x_ret, y_ret, w_ret, h_ret, bw_ret, depth_ret;

	/* Some systems are getting BadDrawable errors when trying to use
	|  root_xpixmap even though I'm using what XGetWindowProperty() returns???
	*/
	if (no_transparency || !any_transparency)
		return;
	prop = XInternAtom(GDK_DISPLAY(), "_XROOTPMAP_ID", True);
	if (prop == None)
		return;
	XGetWindowProperty(GDK_DISPLAY(), GDK_ROOT_WINDOW(), prop, 0L, 1L, False,
			AnyPropertyType, &ret_type, &fmt, &nitems, &bytes_after,
			&prop_return);
	if (prop_return && ret_type == XA_PIXMAP)
		{
		root_xpixmap = *((Pixmap *) prop_return);
		XFree(prop_return);
		}
	if (root_xpixmap == None)
		{
		if (GK.debug_level & DEBUG_MISC)
			printf("Cannot get root pixmap [%lx,%d,%ld,%ld,%d]\n",
				(gulong) ret_type, fmt, nitems, bytes_after,
				(gint) sizeof(Pixmap));
		return;
		}
	if (trans_gc == NULL)
		{
		trans_gc = gdk_gc_new(top_window->window);
		gdk_gc_copy(trans_gc, GK.draw1_GC);
		}

	depth_ret = 0;
	depth_visual = gdk_window_get_visual(top_window->window)->depth;
	if (   !XGetGeometry(GDK_DISPLAY(), root_xpixmap, &root_return,
				&x_ret, &y_ret, &w_ret, &h_ret, &bw_ret, &depth_ret)
		|| depth_ret != depth_visual
	   )
		{
		if (GK.debug_level & DEBUG_MISC)
			printf("Mismatch: root depth = %d visual depth = %d\n",
					depth_ret, depth_visual);
		root_xpixmap = None;
		return;
		}

	/* I could use gdk_pixmap_foreign_new() and stay in the gdk domain,
	|  but it fails (in XGetGeometry()) if I change backgrounds.
	*/
	XSetTile(GDK_DISPLAY(), GDK_GC_XGC(trans_gc), root_xpixmap);
	XSetFillStyle(GDK_DISPLAY(), GDK_GC_XGC(trans_gc), FillTiled);
	for (list = chart_list; list; list = list->next)
		{
		cp = (Chart *) list->data;
		if (!cp->transparency)
			continue;
		draw_rootpixmap_onto_transparent_chart(cp);
		cp->scale_max = 0;
		cp->need_redraw = TRUE;	/* Charts must be redrawn by monitors */
		}
	for (list = panel_list; list; list = list->next)
		{
		p = (Panel *) list->data;
		if (!p->transparency)
			continue;
		draw_panel(p, TRUE);	/* Panels know how to redraw themselves */
		}
	}

/* ---------------------------------------------------------------------- */

  /* Destroy everything inside a panel, but caller must free Panel storage
  |  if not a static panel.
  */
void
gkrellm_destroy_panel(Panel *p)
	{
	gkrellm_destroy_krell_list(p);
	gkrellm_destroy_decal_list(p);

	if (p->label)
		g_free(p->label);

	if (p->pixmap)
		gdk_pixmap_unref(p->pixmap);
	if (p->bg_pixmap)
		gdk_pixmap_unref(p->bg_pixmap);
	if (p->bg_clean_pixmap)
		gdk_imlib_free_pixmap(p->bg_clean_pixmap);

	if (p->drawing_area)
		gtk_signal_handlers_destroy(GTK_OBJECT(p->drawing_area));
	if (p->hbox)
		gtk_widget_destroy(p->hbox);
	panel_list = g_list_remove(panel_list, p);
	memset(p, 0, sizeof(Panel));
	}

  /* Krells, LED's, and labels are drawn on the panel area drawing_area.
  */
void
gkrellm_create_panel(GtkWidget *vbox, Panel *p, GdkImlibImage *bg_image)
	{
	GtkWidget	*hbox;
	GtkWidget	*fixed;

	p->h = p->label->h_panel;
	p->w = UC.chart_width;

	if (p->hbox == NULL)
		{
		hbox = gtk_hbox_new(FALSE, 0);
		gtk_container_add (GTK_CONTAINER(vbox), hbox);
		p->hbox = hbox;

		/* And create the drawing area that the middle panel lives in and put
		|  it into a fixed widget.
		*/
		fixed = gtk_fixed_new();
		gtk_box_pack_start(GTK_BOX(hbox), fixed, TRUE, TRUE, 0);
		gtk_widget_show(fixed);
		p->fixed = fixed;
		p->drawing_area = gtk_drawing_area_new();
		gtk_widget_set_events (p->drawing_area, GDK_EXPOSURE_MASK
				| GDK_LEAVE_NOTIFY_MASK | GDK_ENTER_NOTIFY_MASK
				| GDK_BUTTON_RELEASE_MASK | GDK_BUTTON_PRESS_MASK
				| GDK_POINTER_MOTION_MASK);
		gtk_fixed_put(GTK_FIXED(fixed), p->drawing_area, 0, 0);
		gtk_widget_show(p->drawing_area);
		gtk_widget_show(hbox);
		gtk_widget_realize(hbox);
		gtk_widget_realize(p->fixed);
		gtk_widget_realize(p->drawing_area);
		panel_list = g_list_append(panel_list, p);
		}
	gtk_drawing_area_size(GTK_DRAWING_AREA (p->drawing_area), p->w, p->h);
	gkrellm_render_to_pixmap(bg_image, &p->bg_clean_pixmap,
			&p->bg_mask, p->w, p->h);
	if (p->bg_pixmap)
		gdk_pixmap_unref(p->bg_pixmap);
	p->bg_pixmap = gdk_pixmap_new(top_window->window, p->w, p->h, -1);
	if (p->pixmap)
		gdk_pixmap_unref(p->pixmap);
	p->pixmap = gdk_pixmap_new(top_window->window, p->w, p->h, -1);

	draw_panel(p, FALSE);
	}

static void
free_chart_pixmaps(Chart *cp)
	{
	gdk_pixmap_unref(cp->in_data_bitmap);
	gdk_pixmap_unref(cp->out_data_bitmap);

	gdk_imlib_free_pixmap(cp->bg_clean_pixmap);
	gdk_imlib_free_pixmap(cp->bg_pixmap);
	gdk_imlib_free_pixmap(cp->grid_pixmap);
	gdk_imlib_free_pixmap(cp->bg_grided_pixmap);
	gdk_imlib_free_pixmap(cp->in_grided_pixmap);
	gdk_imlib_free_pixmap(cp->out_grided_pixmap);
	gdk_imlib_free_pixmap(cp->pixmap);
	}

  /* Destroy everything inside a chart, but caller must free Chart storage
  |  if not a static chart.
  */
void
gkrellm_destroy_chart(Chart *cp)
	{
	free_chart_pixmaps(cp);

	if (cp->pDataIn)
		g_free(cp->pDataIn);
	if (cp->pDataOut)
		g_free(cp->pDataOut);

	gtk_signal_handlers_destroy(GTK_OBJECT(cp->drawing_area));
	gtk_widget_destroy(cp->hbox);
	chart_list = g_list_remove(chart_list, cp);
	memset(cp, 0, sizeof(Chart));
	}

void
gkrellm_chart_images_override(Chart *cp, GdkImlibImage *bg_image,
			GdkImlibImage *grid_image)
	{
	if (!cp || !bg_image || !grid_image)
		return;
	cp->bg_image = bg_image;
	cp->grid_image = grid_image;
	cp->override_images = TRUE;
	}

void
gkrellm_create_chart(GtkWidget *vbox, Chart *cp, gint style_id)
	{
	GtkWidget		*hbox;
	GdkWindow		*window;
	Style			*style;
	gint			h;

	if (!vbox || !cp)
		return;
	if (!cp->override_images)
		{
		cp->bg_image = gkrellm_bg_chart_image(style_id);
		cp->grid_image = gkrellm_bg_grid_image(style_id);
		}
	cp->override_images = FALSE;
	cp->x = 0;
/*	cp->y = 0; */
	cp->w = UC.chart_width;
	if (cp->h == 0)
		cp->h = UC.chart_height[0];

	window = top_window->window;

	if (cp->hbox == NULL)
		{
		hbox = gtk_hbox_new(FALSE, 0);
		gtk_container_add (GTK_CONTAINER(vbox), hbox);
		cp->hbox = hbox;

		/* Create the drawing area and the various pixmap layers to draw on.
		|  Must gtk_widget_set_events() immediately.
		*/
		cp->drawing_area = gtk_drawing_area_new();
		gtk_widget_set_events (cp->drawing_area, GDK_EXPOSURE_MASK
				| GDK_LEAVE_NOTIFY_MASK | GDK_ENTER_NOTIFY_MASK
				| GDK_BUTTON_RELEASE_MASK | GDK_BUTTON_PRESS_MASK
				| GDK_POINTER_MOTION_MASK);
		gtk_box_pack_start (GTK_BOX(hbox), cp->drawing_area, FALSE, FALSE, 0);
		gtk_widget_show(cp->drawing_area);
		gtk_widget_show(hbox);
		gtk_widget_realize(hbox);
		gtk_widget_realize(cp->drawing_area);
		chart_list = g_list_append(chart_list, cp);
		}
	else
		free_chart_pixmaps(cp);

	gtk_drawing_area_size(GTK_DRAWING_AREA (cp->drawing_area), cp->w, cp->h);

	/* In/out chart data is drawn on data_bitmaps which are then used
	|  as stencils for pushing in/out_grided_pixmap pixels onto expose pixmap.
	*/
	cp->in_data_bitmap = gdk_pixmap_new(window, cp->w, cp->h, 1);
	cp->out_data_bitmap = gdk_pixmap_new(window, cp->w, cp->h, 1);

	if (gdk_imlib_render(cp->bg_image, cp->w, cp->h) == 0)
		printf(_("create_chart, imlib could not render image.\n"));
	cp->bg_clean_pixmap = gdk_imlib_copy_image(cp->bg_image);
	cp->bg_mask = gdk_imlib_copy_mask(cp->bg_image);
	cp->bg_pixmap = gdk_imlib_copy_image(cp->bg_image);
	cp->bg_mask = gdk_imlib_copy_mask(cp->bg_image);
	cp->bg_grided_pixmap = gdk_imlib_copy_image(cp->bg_image);
	cp->in_grided_pixmap = gdk_imlib_copy_image(cp->bg_image);
	cp->out_grided_pixmap = gdk_imlib_copy_image(cp->bg_image);
	cp->pixmap = gdk_imlib_copy_image(cp->bg_image);

	if ((h = cp->grid_image->rgb_height) > 2)
		h = 2;
	gdk_imlib_render(cp->grid_image, cp->w, h);
	cp->grid_pixmap = gdk_imlib_copy_image(cp->grid_image);

	style = gkrellm_chart_style(style_id);
	cp->transparency = style->transparency;
	any_transparency |= cp->transparency;
	}


static gint		moving_gkrellm	= FALSE;
static gint		x_press_event, y_press_event;

void
gkrellm_motion(GtkWidget *widget, GdkEventMotion *ev, gpointer data)
	{
	gint			x_pointer, y_pointer;
	GdkModifierType	m;
	XEvent			xevent;

	if (moving_gkrellm)
		{
		m = ev->state;
		if (!(m & GDK_BUTTON1_MASK))
			{
			moving_gkrellm = FALSE;
			return;
			}
		/* Catch up to the pointer so GKrellM does not lag the pointer motion.
		*/
		XSync(GDK_DISPLAY(), False /* Don't discard events */);
		while (XCheckTypedEvent(GDK_DISPLAY(), MotionNotify, &xevent))
			{
			ev->x = xevent.xmotion.x;
			ev->y = xevent.xmotion.y;
			}

		gdk_window_get_pointer(NULL, &x_pointer, &y_pointer, &m);

		/* Moves to x,y position relative to root.  Subtract the press
		|  event coordinates to account for pointer offset into top_window.
		*/
		gdk_window_move(top_window->window,
			x_pointer - x_press_event, y_pointer - y_press_event);
		}
	}

static void
top_frame_button_release(GtkWidget *widget, GdkEventButton *ev, gpointer data)
	{
	apply_rootpixmap_transparency();
	moving_gkrellm = FALSE;
	}

static void
top_frame_button_press(GtkWidget *widget, GdkEventButton *ev, gpointer data)
	{
	gint			x_pointer, y_pointer;
	GdkModifierType	m;
	time_t			time_check;

	gdk_window_get_pointer(NULL, &x_pointer, &y_pointer, &m);

	/* If system time is changed, gtk_timeout_add() setting gets
	|  confused, so put some duct tape here ...
	*/
	time(&time_check);
	if (time_check > time_now + 2 || time_check < time_now)
		{
		printf(_("GKrellM restarting timer...\n"));
		start_timer(UC.update_HZ);
		}

	if (ev->button == 3)
		{
		gtk_menu_popup(GTK_MENU(item_factory->widget), NULL, NULL, NULL, NULL,
					ev->button, ev->time);
		return;
		}
	gdk_window_raise(top_window->window);

	/* I need pointer coords relative to top_window.  So, add in offsets
	|  to hostname window if that is where we pressed.
	*/
	x_press_event = ev->x;
	y_press_event = ev->y;
	if (widget == gtree.top1_event_box)
		{
		x_press_event += GK.frame_left_width;
		y_press_event += GK.frame_top_height;
		}
	if (!withdrawn)	/* Needed according to Wes Hopkins <weshopkins@home.com> */
		moving_gkrellm = TRUE;
	}

#define	CLOSE_LEFT	0
#define	CLOSE_RIGHT	1


  /* If any frame images have transparency (masks != NULL) use the
  |  frame masks to construct a main window sized mask.
  */
static void
apply_frame_transparency()
	{
	GtkWidget			*win;
	gint				w, h;
	static gint			w_prev, h_prev;

	if (window_decorations)		/* was > 1 */
		return;

	win = gtree.window;
	w = UC.chart_width + GK.frame_left_width + GK.frame_right_width;
	h = GK.monitor_height + total_frame_height;
	if (!gtree.window_transparency_mask || w != w_prev || h != h_prev)
		{
		if (gtree.window_transparency_mask)
			gdk_pixmap_unref(gtree.window_transparency_mask);
		gtree.window_transparency_mask = gdk_pixmap_new(win->window, w, h, 1);
		w_prev = w;
		h_prev = h;
		if (GK.debug_level & DEBUG_SIZES)
			printf(_("(new mask allocated) "));
		}
	else if (monitors_visible == mask_monitors_visible)
		{
		if (GK.debug_level & DEBUG_SIZES)
			printf(_("(re-using previous mask)\n"));
		return;
		}

	/* Set entire shape mask to 1, then write in the frame masks.
	*/
	gdk_draw_rectangle(gtree.window_transparency_mask, GK.bit1_GC, TRUE,
				0, 0, w, h);

	if (monitors_visible)
		{
		if (gtree.frame_top_mask)
			gdk_draw_pixmap(gtree.window_transparency_mask, GK.bit1_GC,
				gtree.frame_top_mask, 0, 0,  0, 0,  w, GK.frame_top_height);
		if (gtree.frame_bottom_mask)
			gdk_draw_pixmap(gtree.window_transparency_mask, GK.bit1_GC,
				gtree.frame_bottom_mask,
				0, 0, 0, h - GK.frame_bottom_height, w,GK.frame_bottom_height);

		if (gtree.frame_left_mask)
			gdk_draw_pixmap(gtree.window_transparency_mask, GK.bit1_GC,
				gtree.frame_left_mask,
				0, 0,  0, GK.frame_top_height,
				GK.frame_left_width, GK.monitor_height);

		if (gtree.frame_right_mask)
			gdk_draw_pixmap(gtree.window_transparency_mask, GK.bit1_GC,
				gtree.frame_right_mask,
				0, 0,  w - GK.frame_right_width, GK.frame_top_height,
				GK.frame_right_width, GK.monitor_height);
		}
	else	/* Top and bottom frames are not visible and GKrellM is shut */
		{
		if (gtree.frame_left_mask)
			gdk_draw_pixmap(gtree.window_transparency_mask, GK.bit1_GC,
				gtree.frame_left_mask,
				0, 0,  0, 0,
				GK.frame_left_width, GK.monitor_height);

		if (gtree.frame_right_mask)
			gdk_draw_pixmap(gtree.window_transparency_mask, GK.bit1_GC,
				gtree.frame_right_mask,
				0, 0,  GK.frame_left_width, 0,
				GK.frame_right_width, GK.monitor_height);
		}

	gdk_window_shape_combine_mask(gtree.window->window,
				gtree.window_transparency_mask, 0, 0);
	if (GK.debug_level & DEBUG_SIZES)
		printf(_("Transparency mask w=%d h=%d tf_h=%d bf_h=%d sf_h=%d\n"),
			w, h, GK.frame_top_height, GK.frame_bottom_height,
			GK.monitor_height);
	mask_monitors_visible = monitors_visible;
	}

static void
side_frame_button_press(GtkWidget *widget, GdkEventButton *ev, gpointer data)
	{
	static gint		direction;
	gint			x_gkrell, y_gkrell;

	if (ev->button == 3)
		{
		gtk_menu_popup(GTK_MENU(item_factory->widget), NULL, NULL, NULL, NULL,
					ev->button, ev->time);
		return;
		}
	if (window_decorations)
		return;
	gdk_window_get_origin(gtree.window->window, &x_gkrell, &y_gkrell);
	direction = (x_gkrell < w_display / 2) ? CLOSE_LEFT : CLOSE_RIGHT;

	if (ev->button == 2 || (GK.m2 && ev->button == 1))
		{
		monitors_visible = 1 - monitors_visible;
		if (monitors_visible)
			{
			gtk_widget_show(gtree.middle_vbox);
			gtk_widget_show(gtree.top0_vbox);
			gtk_widget_show(gtree.bottom_vbox);
			if (direction == CLOSE_RIGHT)
				gdk_window_move(gtree.window->window,
								x_gkrell - UC.chart_width, y_gkrell);
			}
		else
			{
			gtk_widget_hide(gtree.top0_vbox);
			gtk_widget_hide(gtree.bottom_vbox);
			gtk_widget_hide(gtree.middle_vbox);
			if (direction == CLOSE_RIGHT)
				gdk_window_move(gtree.window->window,
								x_gkrell + UC.chart_width, y_gkrell);
			}
		XSync(GDK_DISPLAY(), False);  /* Avoid double click race */
		}
	}


gint
gkrellm_render_to_pixmap(GdkImlibImage *im, GdkPixmap **pixmap,
			GdkBitmap **mask, gint w, gint h)
	{
	/* I want the pixmap freed even if there is no image to render back
	|  in.  Eg. theme switch to one with no data_in/out_image.
	*/
	if (*pixmap)
		gdk_imlib_free_pixmap(*pixmap);
	if (im == NULL)
		return FALSE;
	if (w <= 0)
		w = im->rgb_width;
	if (h <= 0)
		h = im->rgb_height;
	if (gdk_imlib_render(im, w, h) == 0)
		printf(_("render_to_pixmap, imlib could not render image.\n"));
	*pixmap = gdk_imlib_copy_image(im);
	if (mask)
		*mask = gdk_imlib_copy_mask(im);
	return TRUE;
	}

gint
render_to_pixmap(GdkImlibImage *im, GdkPixmap **pixmap,
						GdkBitmap **mask, gint w, gint h)
	{
	return gkrellm_render_to_pixmap(im, pixmap, mask, w, h);
	}

static void
create_frame_top(GtkWidget *vbox)
	{
	static GtkWidget	*ft_vbox;
	static GtkWidget	*ft_pixmapwid;
	GdkImlibImage		*im;
	gint				w, h;

	if (window_decorations)		/* was > 1 */
		return;
	im = GK.frame_top_image;

	w = UC.chart_width + GK.frame_left_width + GK.frame_right_width;
	h = (GK.frame_top_height > 0) ? GK.frame_top_height : im->rgb_height;
	GK.frame_top_height = h;

	total_frame_height += h;
	render_to_pixmap(im, &gtree.frame_top_pixmap, &gtree.frame_top_mask, w, h);

	if (ft_vbox == NULL)
		{
		ft_vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(vbox), ft_vbox);
		gtk_widget_show(ft_vbox);
		}
	if (ft_pixmapwid)
		gtk_container_remove(GTK_CONTAINER(ft_vbox), ft_pixmapwid);
	ft_pixmapwid = gtk_pixmap_new(gtree.frame_top_pixmap, NULL);
	gtk_box_pack_start(GTK_BOX(ft_vbox), ft_pixmapwid,
				/* expand */ FALSE, /* fill */ FALSE, /* padding */ 0);
	gtk_widget_show (ft_pixmapwid);

	if (GK.debug_level & DEBUG_SIZES)
		printf(_("frame_top: w=%d im_height=%d h=%d T=%d\n"),
				w, im->rgb_height, h, gtree.frame_top_mask ? 1 : 0);

	}

static void
create_frame_bottom(GtkWidget *vbox)
	{
	static GtkWidget	*fb_vbox;
	static GtkWidget	*fb_pixmapwid;
	GdkImlibImage	*im;
	gint			w, h;

	if (window_decorations)		/* was > 1 */
		return;
	im = GK.frame_bottom_image;

	w = UC.chart_width + GK.frame_left_width + GK.frame_right_width;
	h = (GK.frame_bottom_height > 0) ? GK.frame_bottom_height : im->rgb_height;
	GK.frame_bottom_height = h;

	total_frame_height += h;
	render_to_pixmap(im, &gtree.frame_bottom_pixmap, &gtree.frame_bottom_mask,
				w, h);
	if (fb_vbox == NULL)
		{
		fb_vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(vbox), fb_vbox);
		gtk_widget_show(fb_vbox);
		}
	if (fb_pixmapwid)
		gtk_container_remove(GTK_CONTAINER(fb_vbox), fb_pixmapwid);
	fb_pixmapwid = gtk_pixmap_new(gtree.frame_bottom_pixmap, NULL);
	gtk_box_pack_start(GTK_BOX(fb_vbox), fb_pixmapwid,
				/* expand */ FALSE, /* fill */ FALSE, /* padding */ 0);
	gtk_widget_show (fb_pixmapwid);

	if (GK.debug_level & DEBUG_SIZES)
		printf(_("frame_bottom: w=%d im_height=%d h=%d T=%d\n"),
				w, im->rgb_height, h, gtree.frame_top_mask ? 1 : 0);

	}


  /* Return TRUE if visibility is changed, FALSE if not.
  */
int
gkrellm_enable_visibility(int new_visibility, gint *current_visibility,
						GtkWidget *vbox, gint height)
	{
	if (new_visibility  && ! *current_visibility)
		{
		gtk_widget_show(vbox);
		*current_visibility  = TRUE;
		GK.monitor_height += height;
		gkrellm_pack_side_frames();
		return TRUE;
		}
	if (! new_visibility  && *current_visibility )
		{
		gtk_widget_hide(vbox);
		*current_visibility  = FALSE;
		GK.monitor_height -= height;
		gkrellm_pack_side_frames();
		return TRUE;
		}
	return FALSE;
	}

void
gkrellm_pack_side_frames()
	{
	static GtkWidget	*lf_pixmapwid, *rf_pixmapwid;
	gint				x_gkrell, y_gkrell, y_bottom;
	gint				y, w, h;
	gint				was_on_bottom;

	if (! initialized || monitor_previous_height == GK.monitor_height)
		return;

	if (window_decorations)		/* was > 1 */
		{
		GK.frame_left_width = 0;
		GK.frame_right_width = 0;
		return;
		}
	if (GK.debug_level & DEBUG_SIZES)
		printf(_("pack_side_frames mon_width=%d mon_height=%d "),
				UC.chart_width + GK.frame_left_width + GK.frame_right_width,
				GK.monitor_height + total_frame_height);

	gdk_window_get_origin(gtree.window->window, &x_gkrell, &y_gkrell);
	gdk_window_get_size(gtree.window->window, &w, &h);
	was_on_bottom = (y_gkrell + h >= h_display) ? TRUE : FALSE;

	render_to_pixmap(GK.frame_left_image, &gtree.frame_left_pixmap,
			&gtree.frame_left_mask, GK.frame_left_width, GK.monitor_height);
	if (lf_pixmapwid)
		gtk_container_remove(GTK_CONTAINER(gtree.left_vbox), lf_pixmapwid);
	lf_pixmapwid = gtk_pixmap_new(gtree.frame_left_pixmap, NULL);
	gtk_box_pack_start (GTK_BOX(gtree.left_vbox), lf_pixmapwid,
				FALSE, FALSE, 0);
	gtk_widget_show(lf_pixmapwid);

	render_to_pixmap(GK.frame_right_image, &gtree.frame_right_pixmap,
			&gtree.frame_right_mask, GK.frame_right_width, GK.monitor_height);
	if (rf_pixmapwid)
		gtk_container_remove(GTK_CONTAINER(gtree.right_vbox), rf_pixmapwid);
	rf_pixmapwid = gtk_pixmap_new(gtree.frame_right_pixmap, NULL);
	gtk_box_pack_start (GTK_BOX(gtree.right_vbox), rf_pixmapwid,
				FALSE, FALSE, 0);	
	gtk_widget_show(rf_pixmapwid);


	/* If GKrellM grows in height and bottom is moved off screen, move so
	|  that all of GKrellM is visible.
	*/
	h = GK.monitor_height + total_frame_height;
	y_bottom = h_display - (y_gkrell + h);
	if (GK.debug_level & DEBUG_SIZES)
		printf(_("y_bottom=%d h_display=%d y_gkrell=%d\n"),
				y_bottom, h_display, y_gkrell);
	if (y_bottom < 0)
		{
		if ((y = y_gkrell + y_bottom) < 0)
			y = 0;
		gdk_window_move(gtree.window->window, x_gkrell, y);
		}
	/* If GKrellM bottom edge was <= screen bottom, then move to make
	|  resized GKrellM still on bottom whether window has shrunk or grown.
	*/
	else if (was_on_bottom)
		{
		if ((y = h_display - h) < 0)
			y = 0;
		gdk_window_move(gtree.window->window, x_gkrell, y);
		}
	monitor_previous_height = GK.monitor_height;
	}

static gint	on_edge[4];		/* left, right, top, bottom */

static void
edge_record()
	{
	gint	x, y, w, h;

	gdk_window_get_origin(gtree.window->window, &x, &y);
	gdk_window_get_size(gtree.window->window, &w, &h);
	on_edge[0] = (x <= 0) ? TRUE : FALSE;
	on_edge[1] = (x + w >= w_display) ? TRUE : FALSE;
	on_edge[2] = (y <= 0) ? TRUE : FALSE;
	on_edge[3] = (y + h >= h_display) ? TRUE : FALSE;
	}

static void
fix_edges()
	{
	gint	x, y, w, h, x0, y0;

	gdk_window_get_origin(gtree.window->window, &x, &y);
	w = UC.chart_width + GK.frame_left_width + GK.frame_right_width;
	h = GK.monitor_height + total_frame_height;

	x0 = x;
	y0 = y;
	if (x < 0 || on_edge[0])
		x = 0;
	if (x > w_display - w || on_edge[1])
		x = w_display - w;
	if (y < 0 || on_edge[2])
		y = 0;
	if (y > h_display - h || on_edge[3])
		y = h_display - h;
	if (x != x0 || y != y0)
		gdk_window_move(gtree.window->window, x, y);
	}

  /* My own parsing of +x+y, -x-y geometry placements.
  */
static void
place_gkrellm(gchar *geom)
	{
	gchar	*sx, *sy, x_sign, y_sign;
	gint	x, y, w_gkrell, h_gkrell;

	w_gkrell = UC.chart_width + GK.frame_left_width + GK.frame_right_width;
	h_gkrell = GK.monitor_height + total_frame_height;
	sx = geom;
	if (sx == NULL || *sx == '\0')
		{
		printf(_("Cannot parse geometry.\n"));
		return;
		}
	if (*sx != '+' && *sx != '-')
		x_sign = '+';
	else
		x_sign = *sx++;
	for (sy = sx; *sy != '\0'; ++sy)
		if (*sy == '+' || *sy == '-')
			break;
	if (*sy == '\0')
		{
		printf(_("Cannot parse geometry.\n"));
		return;
		}
	y_sign = *sy;
	*sy++ = '\0';

	x = atoi(sx);
	if (x_sign == '-')
		x = w_display - w_gkrell - x;

	y = atoi(sy);
	if (y_sign == '-')
		y = h_display - h_gkrell - y;

	if (x > w_display - w_gkrell)
		x = w_display - w_gkrell;
	if (x < 0)
		x = 0;
	if (y > h_display - h_gkrell)
		y = h_display - h_gkrell;
	if (y < 0)
		y = 0;
	gdk_window_move(gtree.window->window, x, y);
	}

static void
create_spacers(Monitor *mon)
	{
	MonPrivate	*mp		= mon->private;
	GdkPixmap	*pixmap;
	gint		h, w;

	if (mp->top_pixmapwid)
		gtk_widget_destroy(mp->top_pixmapwid);
	mp->top_pixmapwid = NULL;
	if (mp->bottom_pixmapwid)
		gtk_widget_destroy(mp->bottom_pixmapwid);
	mp->bottom_pixmapwid = NULL;

	w = UC.chart_width;
	if (mon->private->spacer_top_image)
		{
		h = mp->spacer_top_height ? mp->spacer_top_height : 3;
		pixmap = gdk_pixmap_new(top_window->window, w, h, -1);
		gdk_imlib_paste_image(mp->spacer_top_image, pixmap, 0, 0, w, h);
		mp->top_pixmapwid = gtk_pixmap_new(pixmap, NULL);
		gtk_box_pack_start(GTK_BOX(mp->spacer_top_vbox),
				mp->top_pixmapwid, FALSE, FALSE, 0);
		gtk_widget_show_all(mp->spacer_top_vbox);
		GK.monitor_height += h;
		}
	else
		gtk_widget_hide(mp->spacer_top_vbox);

	if (mon->private->spacer_bottom_image)
		{
		h = mp->spacer_bottom_height ? mp->spacer_bottom_height : 3;
		pixmap = gdk_pixmap_new(top_window->window, UC.chart_width, h, -1);
		gdk_imlib_paste_image(mp->spacer_bottom_image, pixmap, 0, 0, w, h);
		mp->bottom_pixmapwid = gtk_pixmap_new(pixmap, NULL);
		gtk_box_pack_start(GTK_BOX(mp->spacer_bottom_vbox),
				mp->bottom_pixmapwid, FALSE, FALSE, 0);
		gtk_widget_show_all(mp->spacer_bottom_vbox);
		GK.monitor_height += h;
		}
	else
		gtk_widget_hide(mp->spacer_bottom_vbox);
	}

static void
create_widget_tree()
	{
	gtree.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_name(gtree.window, "gkrellm");

	gtree.vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.window), gtree.vbox);

	gtree.top0_event_box = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(gtree.vbox), gtree.top0_event_box);
	gtree.top0_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.top0_event_box), gtree.top0_vbox);

	/* The middle hbox has left frame, monitors & a right frame.
	*/
	gtree.middle_hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.vbox), gtree.middle_hbox);

	gtree.left_event_box = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(gtree.middle_hbox), gtree.left_event_box);
	gtree.left_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.left_event_box), gtree.left_vbox);

	gtree.middle_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.middle_hbox), gtree.middle_vbox);

	/* Hostname will go in an event box for moving gkrellm */
	gtree.top1_event_box = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(gtree.middle_vbox), gtree.top1_event_box);
	gtree.top1_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.top1_event_box), gtree.top1_vbox);

	gtree.monitor_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.middle_vbox), gtree.monitor_vbox);

	gtree.right_event_box = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(gtree.middle_hbox), gtree.right_event_box);
	gtree.right_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.right_event_box), gtree.right_vbox);

	gtree.bottom_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.vbox), gtree.bottom_vbox);

	gtk_widget_realize(gtree.window);
	if (!window_decorations)
		{
		gtk_window_set_policy(GTK_WINDOW(gtree.window),
				FALSE /*allow_shrink*/ , FALSE /*allow_grow*/,
				TRUE /*auto_shrink*/);
		gdk_window_set_decorations (gtree.window->window, 0);
		}
	else
		gtk_window_set_policy(GTK_WINDOW(gtree.window),
				TRUE /*allow_shrink*/ , TRUE /*allow_grow*/,
				TRUE /*auto_shrink*/);

	gtk_widget_show_all(gtree.vbox);

	/* Probably don't need to realize all these here. Just a little paranoia.
	*/
	gtk_widget_realize(gtree.vbox);
	gtk_widget_realize(gtree.top0_vbox);
	gtk_widget_realize(gtree.middle_hbox);
	gtk_widget_realize(gtree.left_vbox);
	gtk_widget_realize(gtree.middle_vbox);
	gtk_widget_realize(gtree.monitor_vbox);
	gtk_widget_realize(gtree.top1_vbox);
	gtk_widget_realize(gtree.right_vbox);
	gtk_widget_realize(gtree.bottom_vbox);
	}


static gint
cb_client_event(GtkWidget  *widget, GdkEventClient *event, gpointer data)
	{
	static GdkAtom	atom_gtk_read_rcfiles = GDK_NONE;
	static GdkAtom	atom_gkrellm_read_theme = GDK_NONE;

	if (!atom_gtk_read_rcfiles)
		{
		atom_gtk_read_rcfiles = gdk_atom_intern("_GTK_READ_RCFILES", FALSE);
		atom_gkrellm_read_theme = gdk_atom_intern("_GKRELLM_READ_THEME",FALSE);
		}
	if (event->message_type == atom_gtk_read_rcfiles) 
		{
		read_theme_event(0);
		/* Gtk theme resets will munge my shape combine settings, so must
		|  redraw frames.
		*/
		check_frame_transparency = TRUE;
		}
	else if (event->message_type == atom_gkrellm_read_theme) 
		read_theme_event(1);
	return FALSE;
	}

  /* Callback called when configure_event is received.  This can
  |  be for window raise, lower, resize, move & maybe others
  */
static void
cb_configure_notify(GtkWidget *widget, GdkEventConfigure *ev, gpointer data)
	{
	gboolean	size_change, position_change;
	gint        x, y, w, h, w_gkrellm, h_gkrellm;
	static gint	x_prev, y_prev, w_prev, h_prev;

	gdk_window_get_position(widget->window, &x, &y);
	gdk_window_get_size(gtree.window->window, &w, &h);

	w_gkrellm = UC.chart_width + GK.frame_left_width + GK.frame_right_width;
	h_gkrellm = GK.monitor_height + total_frame_height;

	/* If window manager decorated, I can't allow a user to resize GKrellM
	|  via the window manager.  So, try to turn it into a no-op.
	*/
	if (   window_decorations && monitors_visible
		&& (w_gkrellm != w || h_gkrellm != h)
	   )
		{
		if (GK.debug_level & DEBUG_SIZES)
			printf(_("Overriding user window resize!\n"));
		gdk_window_resize(top_window->window, w_gkrellm, h_gkrellm);
		w = w_gkrellm;
		h = h_gkrellm;
		}
	size_change = (w != w_prev) | (h != w_prev);
	position_change = (x != x_prev) | (y != y_prev);

	if (size_change)
		{
		if (GK.debug_level & DEBUG_SIZES)
			printf(_("****config_notify w=%d h=%d gk_width=%d gk_height=%d\n"),
				w, h, w_gkrellm, h_gkrellm);
		apply_frame_transparency();
		}
	/* This is abit of a hose... I have to defer applying the root pixmap
	|  because windows just appearing by gtk_widget_show() are not in the
	|  right place yet even though configure notify is sent.
	*/
	if ((size_change || position_change) && !moving_gkrellm)
		check_rootpixmap_transparency = TRUE;
	x_prev = x;
	y_prev = y;
	w_prev = w;
	h_prev = h;

	if (x >= 0 && x < w_display - 10)
		x_position = x;
	if (y >= 0 && y < h_display - 25)
		y_position = y;

	/* If GKrellM is killed (signal, X shutdown, etc) it might not get
	|  a chance to save its position.  So write the position file
	|  10 seconds after window moves stop.
	*/
	if (UC.save_position)
		save_position_countdown = 10 * UC.update_HZ;
	}

void
gkrellm_save_all()
	{
	save_inet_data();
	if (UC.save_position)
		set_or_save_position(1);
	if (GK.config_modified)
		save_user_config();
	save_theme_config();
	}

  /* Pre 0.8.0 versions data directory fix
  */
static void
fix_old_data_dir()
	{
	gchar	dir[256], fixcmd[512];

	snprintf(dir, sizeof(dir), "%s/%s", gkrellm_homedir(), ".gkrellm/+data");
	if (isdir(dir, NULL))
		{
		snprintf(fixcmd, sizeof(fixcmd), "mv %s %s/%s", dir,
					gkrellm_homedir(), GKRELLM_DATA_DIR);
		system(fixcmd);
		}
	}

static gchar	*intro_msg =
N_(	"You can configure your monitors by right clicking on\n"
	"the top frame of GKrellM or by hitting the F1 key\n"
	"with the mouse in the GKrellM window.\n"
	"Read the Info pages in the config for setup help.");

static void
check_gkrellm_directories()
	{
	gchar	*t, *u;

	if (make_home_subdir(GKRELLM_DIR))
		gkrellm_message_window(_("GKrellM Introduction"), _(intro_msg), NULL);

	make_home_subdir(GKRELLM_PLUGINS_DIR);
	make_home_subdir(GKRELLM_THEMES_DIR);
	fix_old_data_dir();
	make_home_subdir(GKRELLM_DATA_DIR);

	/* If no user_config specified, force a check for host-specific theme
	*/
	if (!GK.config_suffix)
		{
		GK.found_host_config = TRUE;
		t = gkrellm_make_config_file_name(gkrellm_homedir(),
							GKRELLM_THEME_CONFIG);
		u = gkrellm_make_config_file_name(gkrellm_homedir(),
							GKRELLM_USER_CONFIG);
		if (!isfile(u) && !isfile(t))
			GK.found_host_config = FALSE;
		g_free(t);
		g_free(u);
		}
	}


static gint
cb_delete_event(GtkWidget *widget, GdkEvent *event, gpointer data)
	{
	/* Return FALSE to get GTK to emit the "destroy" signal.
	|  Return TRUE to not destroy the window (for verify dialog pop up).
	*/
	return(FALSE);
	}

static void
cb_destroy_event()
	{
	gtk_main_quit();
	}

static gchar	*usage_string =
N_("usage: gkrellm [options]\n"
"options:\n"
"   -t, --theme theme_dir    Select a theme directory.\n"
"   -g, --geometry +x+y      Position the window on the screen.\n"
"       --wm                 Allow window manager decorations.\n"
"       --m2                 Left button side frame shading (for 2 btn mice).\n"
"       --nt                 No transparency.\n"
"   -o, --on-top             Try to set on top hint. If this does not work,\n"
"                            configure your window manager to do this job!\n"
"   -w, --withdrawn          Draw GKrellM in withdrawn mode.\n"
"   -c, --config suffix      Use alternate config files generated by\n"
"                            appending \"suffix\" to config file names.\n"
"   -f, --force-host-config  Creates config files generated by appending the\n"
"                            hostname to config file names.  Subsequent runs\n"
"                            automatically will use these configs unless a\n"
"                            specific config is specified with --config.\n"
"                            This is a convenience for allowing remote runs\n"
"                            with independent configs in a shared home dir\n"
"                            and for the hostname to appear in the X title.\n"
"       --nc                 No config mode prevents configuration changes.\n"
"   -p, --plugin plugin.so   While developing, load your plugin under test.\n"
"       --demo               Force enabling of many monitors so themers can\n"
"                            see everything. All config saving is inhibited.\n"
"   -v, --version            Print GKrellM version number and exit.\n"
"   -d, --debug-level n      Turn debugging on for selective code sections.\n"

"\ndebug-level numbers are (bitwise OR to debug multiple sections):\n"
"     0x1   misc\n"
"     0x2   images\n"
"     0x4   gkrellmrc\n"
"     0x8   styles\n"
"     0x10  mail\n"
"     0x20  net\n"
"     0x40  timer button\n"
"     0x80  size changes\n"
"     0x100 plugins\n"
"     0x200 sensors\n\n"
);

static void
usage()
	{
	printf(_(usage_string));
	}

  /* Get an argc and argv pruned of all GKrellM options with argv[0] and
  |  only the assumed session manager options which were magically trapped out.
  */
gint
gkrellm_get_sm_argc()
	{
	return GK.sm_argc;
	}

gchar **
gkrellm_get_sm_argv()
	{
	return GK.sm_argv;
	}

  /* There can be a session manager plugin, eg gkrellm-gnome, and it will
  |  need an arg list for a restart command.  Only some original options
  |  are appropriate for a restart (-t or -p are not).  Excludes argv[0].
  */
gint
gkrellm_get_restart_options(gchar **argv, gint max_argc)
	{
	gint argc	= 0;

	if (window_decorations && argc < max_argc)
		argv[argc++] = "--wm";
	if (GK.config_suffix && argc < max_argc - 1)
		{
		argv[argc++] = "-c";
		argv[argc++] = GK.config_suffix;
		}
	if (withdrawn && argc < max_argc)
		argv[argc++] = "-w";
	if (geometry && argc < max_argc - 1)
		{
		argv[argc++] = "-g";
		argv[argc++] = geometry;
		}
	if (no_transparency && argc < max_argc)
		argv[argc++] = "--nt";
	if (GK.m2 && argc < max_argc)
		argv[argc++] = "--m2";
	if (GK.no_config && argc < max_argc)
		argv[argc++] = "--nc";
	return argc;
	}


void
build_gkrellm()
	{
	Monitor		*mon;
	GList		*list;
	static gint	first_create	= TRUE;

	initialized = FALSE;
	monitor_previous_height = 0;
	total_frame_height = 0;
	root_xpixmap = None;
	any_transparency = FALSE;
	start_timer(0);				/* Stop the timer */

	if (!first_create)
		edge_record();
	destroy_decal_button_list();
	gkrellmrc_theme_config();
	if (first_create)
		read_user_config();
	setup_colors();
	setup_fonts();
	load_theme_images();

	create_frame_top(gtree.top0_vbox);

	/* Create the monitors.  Each monitor must increment GK.monitor_height
	|  by its vertical space.
	*/
	GK.monitor_height = 0;
	create_hostname(gtree.top1_vbox, first_create);
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (mon->private->vbox)
			{
			if (first_create)
				{
				gtk_box_pack_start(GTK_BOX(gtree.monitor_vbox),
							mon->private->spacer_top_vbox, FALSE, FALSE, 0);
				gtk_box_pack_start(GTK_BOX(gtree.monitor_vbox),
							mon->private->vbox, FALSE, FALSE, 0);
				gtk_box_pack_start(GTK_BOX(gtree.monitor_vbox),
							mon->private->spacer_bottom_vbox, FALSE, FALSE, 0);
				}
			if (mon->create_monitor && mon->private->enabled)
				{
				gtk_widget_show(mon->private->vbox);
				(*(mon->create_monitor))(mon->private->vbox,
							mon->private->created ? FALSE : TRUE);
				mon->private->created = TRUE;
				create_spacers(mon);
				}
			}
		}
	create_frame_bottom(gtree.bottom_vbox);

	initialized = TRUE;
	gkrellm_pack_side_frames();
	start_timer(UC.update_HZ);
	if (!first_create)
		{
		fix_edges();
		apply_rootpixmap_transparency();
		}
	first_create = FALSE;
	}

static void
add_builtin(Monitor *mon)
	{
	if (mon == NULL)
		return;
	gkrellm_monitor_list = g_list_append(gkrellm_monitor_list, mon);
	if (mon->private == NULL)		/* Won't be null if style was added */
		mon->private = g_new0(MonPrivate, 1);
	if (mon->create_monitor)
		{
		mon->private->spacer_top_vbox = gtk_vbox_new(FALSE, 0);
		mon->private->vbox = gtk_vbox_new(FALSE, 0);
		mon->private->spacer_bottom_vbox = gtk_vbox_new(FALSE, 0);
		}
	mon->private->enabled = TRUE;
	}

  /* Eventually, add a config for selecting specific monitors here so
  |  multiple gkrellm instances can be set up with distinct monitor
  |  sets.
  */
static void
load_builtin_monitors()
	{
	gkrellm_add_chart_style(NULL, "*");
	gkrellm_add_meter_style(NULL, "*");

	/* The sensors_monitor does not have a create or update, but it does
	|  have an apply which needs to be called before the cpu or proc apply.
	|  So just put sensor first.
	*/
	add_builtin(init_sensors_monitor());
	add_builtin(init_host_monitor());
	add_builtin(init_cal_monitor());
	add_builtin(init_clock_monitor());
	add_builtin(init_cpu_monitor());
	add_builtin(init_proc_monitor());
	add_builtin(init_voltage_monitor());
	add_builtin(init_disk_monitor());
	add_builtin(init_inet_monitor());
	add_builtin(init_net_monitor());
	add_builtin(init_timer_monitor());
	add_builtin(init_mem_monitor());
	add_builtin(init_swap_monitor());
	add_builtin(init_fs_monitor());
	if (GK.demo)
		add_builtin(init_demo_monitor());
	add_builtin(init_mail_monitor());
	add_builtin(init_apm_monitor());
	add_builtin(init_uptime_monitor());
	}

static void
set_window_title()
	{
	gchar	*title;

	title = gkrellm_make_config_file_name(NULL, "gkrellm");
	gtk_window_set_title(GTK_WINDOW (gtree.window), title);
	g_free(title);
	}

int
main (int argc, char *argv[])
	{
	gint		i, on_top = 0;
	gchar		*s;

#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
	/* We just ignore error, here.  Even if GKrellM doesn't have
	|  kmem privilege, it runs with available information.
	*/
	kvmd = kvm_openfiles(NULL, NULL, NULL, O_RDONLY, errbuf);
	if (setgid(getgid()) != 0)
		{
		fprintf(stderr, "Can't drop setgid privileges.");
		return 1;
		}
#endif
#if defined(__FreeBSD__)
	scan_for_sensors();
	if (setuid(getuid()) != 0)
		{
		fprintf(stderr, "Can't drop setuid privileges.");
		return 1;
		}
#endif

#ifdef ENABLE_NLS
	gtk_set_locale();
#endif
	gtk_init(&argc, &argv);		/* Will call gdk_init() */
	gdk_imlib_init();

	/* Get gdk to use imlib's visual and colormap
	*/
	gtk_widget_push_visual(gdk_imlib_get_visual());
	gtk_widget_push_colormap(gdk_imlib_get_colormap());

#ifdef ENABLE_NLS
#	ifdef PACKAGE
#		ifdef LOCALEDIR
			bindtextdomain(PACKAGE, LOCALEDIR);
#		endif	/* LOCALEDIR */
		textdomain(PACKAGE);
#	else
		textdomain("gkrellm");
#	endif	/* PACKAGE */
#endif	/* ENABLE_NLS */


#if !defined(__FreeBSD__) && !defined(__linux__) && !defined(__NetBSD__) \
	&& !defined(__OpenBSD__)
	glibtop_init();
#endif

	/* Even when I destroy images, if same im pointer turns up in image
	| reload, pixmap cacheing can be fooled. So turn off pixmap cacheing.
	*/
	gdk_imlib_set_cache_info(0, 1);		/* cache_pixmaps, cache_images	*/

	GK.start_time = time(0);
	current_tm = *(localtime(&GK.start_time));
	initialized = FALSE;

	GK.sm_argc = 1;
	GK.sm_argv[0] = argv[0];
	for (i = 1; i < argc; ++i)
		{
		s = argv[i];
		if (*s == '-')
			{
			++s;
			if (*s == '-')
				++s;
			}
		if (!strncmp(s, "sm-", 3))	/* Save session manager args */
			{
			if (GK.sm_argc < GKRELLM_SM_ARGC_MAX - 1)
				{
				GK.sm_argv[GK.sm_argc++] = argv[i];
				GK.sm_argv[GK.sm_argc++] = argv[i + 1];
				}
			++i;
			}
		else if ((!strcmp(s, "t") || !strcmp(s, "theme")) && i < argc - 1)
			{
			GK.theme_path = g_strdup(argv[++i]);
			GK.command_line_theme = g_strdup(GK.theme_path);
			}
		else if ((!strcmp(s, "p") || !strcmp(s, "plugin")) && i < argc - 1)
			GK.command_line_plugin = g_strdup(argv[++i]);
		else if ((!strcmp(s, "config") || !strcmp(s, "c")) && i < argc - 1)
			GK.config_suffix = g_strdup(argv[++i]);
		else if ((!strcmp(s, "geometry") || !strcmp(s, "g")) && i < argc - 1)
			geometry = argv[++i];
		else if (!strcmp(s, "wm"))
			++window_decorations;
		else if (!strcmp(s, "m2"))
			GK.m2 = TRUE;
		else if ( (! strcmp(s, "withdrawn")) || (! strcmp(s, "w")))
			withdrawn = TRUE;
		else if (!strcmp(s, "force-host-config") || !strcmp(s, "f"))
			{
			GK.force_host_config = TRUE;
			gkrellm_config_modified();
			}
		else if (!strcmp(s, "nt"))
			no_transparency = TRUE;
		else if (!strcmp(s, "nc"))
			GK.no_config = TRUE;
		else if (!strcmp(s, "on-top") || !strcmp(s, "o"))
			on_top = TRUE;
		else if ((!strcmp(s, "debug-level") || !strcmp(s, "d")) && i < argc-1)
			GK.debug_level = (gint) strtoul(argv[++i], NULL, 0);
		else if (!strncmp(s, "debug", 5))
			{
			if (s[5] != '\0')
				GK.debug = atoi(s + 5);
			else
				++GK.debug;
			}
		else if (!strcmp(s, "demo"))
			++GK.demo;
		else if (!strcmp(s, "test"))
			GK.test += 1;
		else if (!strcmp(s, "version") || !strcmp(s, "v"))
			{
			printf(_("gkrellm %d.%d.%d\n"), GKRELLM_VERSION_MAJOR,
					GKRELLM_VERSION_MINOR, GKRELLM_VERSION_REV);
			exit(0);
			}
		else if (strcmp(s, "help") == 0 || strcmp(s, "h") == 0)
			{
			usage();
			exit(0);
			}
		else
			{
			printf(_("Bad arg: %s\n"), argv[i]);
			usage();
			exit(0);
			}
		}
	gdk_window_get_geometry(NULL /*gdk_root_parent*/, &i, &i, &w_display,
				&h_display, &i);

	create_widget_tree();
	top_window = gtree.window;

	check_gkrellm_directories();
	load_builtin_monitors();
	load_plugin_monitors();

	build_gkrellm();
	make_themes_list();

	gtk_signal_connect(GTK_OBJECT(gtree.window), "delete_event",
			GTK_SIGNAL_FUNC(cb_delete_event), NULL);
	gtk_signal_connect(GTK_OBJECT(gtree.window), "destroy",
				GTK_SIGNAL_FUNC(cb_destroy_event), NULL);
	gtk_signal_connect(GTK_OBJECT(gtree.window), "configure_event",
			GTK_SIGNAL_FUNC(cb_configure_notify), NULL);
	gtk_signal_connect(GTK_OBJECT(gtree.window), "client_event",
			GTK_SIGNAL_FUNC(cb_client_event), NULL);

	gtk_signal_connect(GTK_OBJECT(gtree.top0_event_box), "button_press_event",
			GTK_SIGNAL_FUNC(top_frame_button_press), NULL );
	gtk_signal_connect(GTK_OBJECT(gtree.top1_event_box), "button_press_event",
			GTK_SIGNAL_FUNC(top_frame_button_press), NULL );
	gtk_signal_connect(GTK_OBJECT(gtree.top0_event_box),
			"motion_notify_event", (GtkSignalFunc) gkrellm_motion, NULL);
	gtk_signal_connect(GTK_OBJECT(gtree.top1_event_box),
			"motion_notify_event", (GtkSignalFunc) gkrellm_motion, NULL);
	gtk_signal_connect(GTK_OBJECT(gtree.top0_event_box), "button_release_event",
			GTK_SIGNAL_FUNC(top_frame_button_release), NULL );
	gtk_signal_connect(GTK_OBJECT(gtree.top1_event_box), "button_release_event",
			GTK_SIGNAL_FUNC(top_frame_button_release), NULL );

	gtk_signal_connect(GTK_OBJECT(gtree.left_event_box), "button_press_event",
			GTK_SIGNAL_FUNC(side_frame_button_press), NULL );
	gtk_signal_connect(GTK_OBJECT(gtree.right_event_box), "button_press_event",
			GTK_SIGNAL_FUNC(side_frame_button_press), NULL );

	item_factory = create_item_factory_popup();

	set_window_title();
	gtk_widget_show(gtree.window);

	if (geometry)		/* Command line placement overrides */
		place_gkrellm(geometry);
	else if (UC.save_position)
		set_or_save_position(0);

    if (withdrawn)
		{
		XWMHints mywmhints; 
		mywmhints.initial_state = WithdrawnState; 
		mywmhints.flags=StateHint;

		XSetWMHints(GDK_DISPLAY(), GDK_WINDOW_XWINDOW(gtree.window->window), 
				&mywmhints); 
		/* gdk_window_withdraw(gtree.window->window); */
		} 
	if (on_top)
		set_on_top();

	load_inet_data();

	start_timer(UC.update_HZ);
	gtk_main ();
	gkrellm_save_all();

	return 0;
	}


#if 0
/* Debugging aid */
void
exit(int code)
    {
	printf("exit: gkrellm forcing core dump.\n");
	abort();
	}
 
void
_exit(int code)
    {
	printf("_exit: gkrellm forcing core dump.\n");
	abort();
	}
#endif
