/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* 12/22/2000 Patch from Dan Maas <dmaas@dcine.com>
|		Added code to support hostname qualified config files so a remote
|		machine with a shared home directory can have independent configs.
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"
#include <unistd.h> /* needed for gethostname() */

#include	"pixmaps/frame_top.xpm"
#include	"pixmaps/frame_bottom.xpm"
#include	"pixmaps/frame_left.xpm"
#include	"pixmaps/frame_right.xpm"

#include	"pixmaps/button_panel_out.xpm"
#include	"pixmaps/button_panel_in.xpm"
#include	"pixmaps/button_meter_out.xpm"
#include	"pixmaps/button_meter_in.xpm"

#include	"pixmaps/bg_chart.xpm"
#include	"pixmaps/bg_grid.xpm"
#include	"pixmaps/bg_panel.xpm"

#include	"pixmaps/bg_meter.xpm"
#include	"pixmaps/bg_meter_mem.xpm"
#include	"pixmaps/bg_meter_swap.xpm"
#include	"pixmaps/bg_meter_fs.xpm"
#include	"pixmaps/bg_meter_host.xpm"
#include	"pixmaps/bg_meter_uptime.xpm"
#include	"pixmaps/bg_meter_timerbutton.xpm"

#include	"pixmaps/bg_timer.xpm"

#include	"pixmaps/bg_slider_panel.xpm"
#include	"pixmaps/bg_slider_meter.xpm"

  /* These data images are used only for the default theme
  */
#include	"pixmaps/data_in.xpm"
#include	"pixmaps/data_in_grid.xpm"
#include	"pixmaps/data_out.xpm"
#include	"pixmaps/data_out_grid.xpm"

#include	"pixmaps/decal_net_leds.xpm"
#include	"pixmaps/decal_misc.xpm"
#include	"pixmaps/decal_timer_button.xpm"

#include	"pixmaps/krell_panel.xpm"
#include	"pixmaps/krell_meter.xpm"

#ifdef BSD
#include	"pixmaps/krell_meter_mail_daemon.xpm"
#else
#include	"pixmaps/krell_meter_mail.xpm"
#endif
#include	"pixmaps/krell_meter_mem.xpm"
#include	"pixmaps/krell_meter_swap.xpm"
#include	"pixmaps/krell_slider.xpm"
#include	"pixmaps/krell_mini.xpm"
#include	"pixmaps/decal_mail.xpm"

#include	"pixmaps/bg_scroll_gkrellmms.xpm"
#include	"pixmaps/bg_meter_gkrellmms.xpm"
#include	"pixmaps/spacer_top_gkrellmms.xpm"
#include	"pixmaps/spacer_bottom_gkrellmms.xpm"

#include	"pixmaps/bg_meter_volume.xpm"
#include	"pixmaps/spacer_top_volume.xpm"
#include	"pixmaps/spacer_bottom_volume.xpm"

#include	"pixmaps/bg_meter_pmu.xpm"
#include	"pixmaps/spacer_top_pmu.xpm"
#include	"pixmaps/spacer_bottom_pmu.xpm"


static gchar	*image_type[] =
	{
	".png", ".jpg", ".xpm", ".gif"
	};

gchar *
gkrellm_theme_file_exists(char *name, gchar *subdir)
	{
	gint			i;
	static gchar	*path;
	struct stat		st;

	if (gkrellm_using_default_theme())
		return NULL;
	if (path)
		g_free(path);
	if (GK.theme_alternative > 0)
		{
		for (i = 0; i < sizeof(image_type) / sizeof(char *); ++i)
			{
			if (subdir)
				path = g_strdup_printf("%s/%s/%s_%d%s", GK.theme_path, subdir,
						name, GK.theme_alternative, image_type[i]);
			else
				path = g_strdup_printf("%s/%s_%d%s", GK.theme_path,
						name, GK.theme_alternative, image_type[i]);
			if (   stat(path, &st) == 0
				&& (S_ISREG(st.st_mode) || S_ISLNK(st.st_mode))
			   )
				return path;
			}
		}
	for (i = 0; i < sizeof(image_type) / sizeof(char *); ++i)
		{
		if (subdir)
			path = g_strdup_printf("%s/%s/%s%s", GK.theme_path, subdir, name,
					image_type[i]);
		else
			path = g_strdup_printf("%s/%s%s", GK.theme_path, name,
					image_type[i]);
		if (   stat(path, &st) == 0
			&& (S_ISREG(st.st_mode) || S_ISLNK(st.st_mode))
		   )
			return path;
		g_free(path);
		path = NULL;
		}
	return NULL;
	}

static void
set_border(GdkImlibBorder *border, char *string)
	{
	if (border == NULL)
		return;
	border->top = 0;
	border->bottom = 0;
	border->left = 0;
	border->right = 0;
	if (string == NULL)
		return;
	sscanf(string, "%d,%d,%d,%d", &border->left, &border->right,
				&border->top, &border->bottom);
	}


static void
assign_font(Style *style, gchar *fontname, gint AorB)
	{
	TextStyle	*ts;

	ts = (AorB == SET_TEXTFONT_A) ? &style->label_tsA : &style->label_tsB;

	if (strcmp(fontname, "large_font") == 0)
		ts->font_seed = &GK.large_font;
	else if (strcmp(fontname, "normal_font") == 0)
		ts->font_seed = &GK.normal_font;
	else if (strcmp(fontname, "small_font") == 0)
		ts->font_seed = &GK.small_font;
	}

static void
assign_textcolor(Style *style, gchar *arg, gint AorB)
	{
	TextStyle	*ts;
	gchar		*values, *s;
	gchar		*color, *shadowcolor, *effect;

	values = g_strconcat(arg, NULL);

	color = cut_quoted_string(values, &s);
	shadowcolor = cut_quoted_string(s, &s);
	effect = cut_quoted_string(s, &s);
	if (*color == '\0' || *shadowcolor == '\0' || *effect == '\0')
		{
		printf(_("Bad textcolor line %s\n"), arg);
		g_free(values);
		return;
		}
	ts = (AorB == SET_TEXTCOLOR_A) ? &style->label_tsA : &style->label_tsB;
	map_color_string(color, &(ts->color));
	map_color_string(shadowcolor, &(ts->shadow_color));
	ts->effect = effect_string_value(effect);
	g_free(values);
	}

static void
assign_style_entry(Style *style, gchar *value, gint entry_flag)
	{
	if (entry_flag == SET_KRELL_YOFF)
		style->krell_yoff = atoi(value);
	else if (entry_flag == SET_KRELL_EXPAND)
		{
		if (strcmp(value, "left") == 0)
			style->krell_expand = LEFT;
		else if (strcmp(value, "right") == 0)
			style->krell_expand = RIGHT;
		else
			style->krell_expand = NONE;
		}
	else if (entry_flag == SET_KRELL_X_HOT)
		style->krell_x_hot = atoi(value);
	else if (entry_flag == SET_KRELL_EMA_PERIOD)
		style->krell_ema_period = atoi(value);
	else if (entry_flag == SET_KRELL_DEPTH)
		style->krell_depth = atoi(value);
	else if (entry_flag == SET_LABEL_POSITION)
		{
		if (strcmp(value, "center") == 0)
			style->label_position = LABEL_CENTER;
		else if (isdigit(*value))
			style->label_position = atoi(value);
		else
			style->label_position = LABEL_NONE;
		}
	else if (entry_flag == SET_MARGIN)
		style->margin = atoi(value);
	else if (entry_flag == SET_TOP_MARGIN)
		{
		style->top_margin = atoi(value);
		GK.use_top_bottom_margins = TRUE;	/* Allow themes to adapt. */
		}
	else if (entry_flag == SET_BOTTOM_MARGIN)
		{
		style->bottom_margin = atoi(value);
		GK.use_top_bottom_margins = TRUE;	/* Allow themes to adapt. */
		}
	else if (entry_flag == SET_TRANSPARENCY)
		style->transparency = atoi(value);
	else if (entry_flag == SET_TEXTCOLOR_A)
		assign_textcolor(style, value, SET_TEXTCOLOR_A);
	else if (entry_flag == SET_TEXTCOLOR_B)
		assign_textcolor(style, value, SET_TEXTCOLOR_B);
	else if (entry_flag == SET_TEXTFONT_A)
		assign_font(style, value, SET_TEXTFONT_A);
	else if (entry_flag == SET_TEXTFONT_B)
		assign_font(style, value, SET_TEXTFONT_B);
	else if (entry_flag == SET_BORDER)
		set_border(&style->border, value);
	}


static void
assign_style(gchar *debug_name, GList *style_list, gint index,
				gchar *arg, gint entry_flag, gint override)
	{
	Style	*style;
	GList	*list;

	style = (Style *) g_list_nth_data(style_list, index);
	if (style == NULL)
		printf(_("Bad assign style for %s, index=%d, arg=%s\n"),
				debug_name, index, arg);

	/* If this is not an override assignment and this entry has already had
	|  an override assignment, then we do not assign.
	*/
	if (! override && (style->override & BIT(entry_flag)))
		return;
	if (override)
		style->override |= BIT(entry_flag);
	assign_style_entry(style, arg, entry_flag);
	if (index++ == 0)		/* style == style_list */
		{
		if (override)
			printf(_("Bad override on DEFAULT: %s %s %d\n"),
					debug_name, arg, entry_flag);
		if (GK.debug_level & DEBUG_STYLES)
			printf(_("  propagating %s %d: "), arg, entry_flag);
		for (list = style_list->next; list; list = list->next, ++index)
			{
			style = (Style *) list->data;
			if (style && !(style->override & BIT(entry_flag)))
				{
				if (GK.debug_level & DEBUG_STYLES)
					printf("%d ", index);
				assign_style_entry(style, arg, entry_flag);
				}
			}
		if (GK.debug_level & DEBUG_STYLES)
			printf("\n");
		}
	}

#if 0
static void
assign_chart_style(gint index, gchar *arg, gint entry_flag, gint override)
	{
	assign_style("StyleChart", GK.chart_style_list, index, arg,
				entry_flag, override);
	}

static void
assign_panel_style(gint index, gchar *arg, gint entry_flag, gint override)
	{
	assign_style("StylePanel", GK.panel_style_list, index, arg,
				entry_flag, override);
	}
#endif

static void
assign_meter_style(gint index, gchar *arg, gint entry_flag, gint override)
	{
	assign_style("StyleMeter", GK.meter_style_list, index, arg,
				entry_flag, override);
	}

static struct string_map
	{
	gchar	*string;
	gint	value;
	}
	entry_map[] =
	{
	{ "krell_yoff",		SET_KRELL_YOFF },
	{ "krell_expand",	SET_KRELL_EXPAND },
	{ "krell_x_hot",	SET_KRELL_X_HOT },
	{ "krell_ema_period", SET_KRELL_EMA_PERIOD },
	{ "krell_depth",	SET_KRELL_DEPTH },
	{ "label_position",	SET_LABEL_POSITION },
	{ "margin",			SET_MARGIN },
	{ "top_margin",		SET_TOP_MARGIN },
	{ "bottom_margin",	SET_BOTTOM_MARGIN },
	{ "textcolor",		SET_TEXTCOLOR_A },
	{ "alt_textcolor",	SET_TEXTCOLOR_B },
	{ "font",			SET_TEXTFONT_A },
	{ "alt_font",		SET_TEXTFONT_B },
	{ "border",			SET_BORDER },
	{ "transparency",	SET_TRANSPARENCY },
	};

static void
assign_gkrellmrc_style(gchar *source_line, gchar *area, gchar *string)
	{
	struct string_map	*sm;
	GList				*list, *style_list, *name_list;
	gchar				*target, *arg, *scope, *entry;
	gint				i, style_id, entry_flag, override;

	target = strtok(string, " \t=:");	/* *.yyy  or xxx.yyy */
	arg = strtok(NULL, "\n");
	while (*arg == ' ' || *arg == '\t' || *arg == '=' || *arg == ':')
		++arg;
	scope = strtok(target, ".");
	entry = strtok(NULL, " \n");
	if (GK.debug_level & DEBUG_STYLES)
		printf(_("Assign style: %s %s.%s = %s "), area, scope, entry, arg);

	override = TRUE;
	style_id = -1;
	entry_flag = -1;
	if (!strcmp(area, "StyleChart"))
		{
		name_list = GK.chart_name_list;
		style_list = GK.chart_style_list;
		}
	else if (!strcmp(area, "StylePanel"))
		{
		name_list = GK.chart_name_list;
		style_list = GK.panel_style_list;
		}
	else if (!strcmp(area, "StyleMeter"))
		{
		name_list = GK.meter_name_list;
		style_list = GK.meter_style_list;
		}
	else
		{
		printf(_("unknown Style in %s\n"), source_line);
		return;
		}
	for (list = name_list, i = 0; list; list = list->next, ++i)
		if (strcmp(scope, (gchar *) list->data) == 0)
			{
			style_id = i;
			break;
			}
	if (style_id == DEFAULT_STYLE_ID)
		override = FALSE;

	for (sm = &entry_map[0];
		sm < &entry_map[sizeof(entry_map) / sizeof(struct string_map)]; ++sm)
		if (strcmp(entry, sm->string) == 0)
			{
			entry_flag = sm->value;
			break;
			}
	if (GK.debug_level & DEBUG_STYLES)
		printf(" (%d %d %d)\n", style_id, entry_flag, override);

	if (entry_flag >= 0 && style_id >= 0)
		assign_style(area, style_list, style_id, arg, entry_flag, override);
	else if (GK.debug_level & DEBUG_STYLES)
		printf(_("\nUnknown %s.%s in line:\n    %s\n"),
			   scope, entry, source_line);
	}

gint
gkrellm_add_chart_style(Monitor *mon, gchar *name)
	{
	Style		*style;
	gint		id;
	static gint	style_id;

	id = style_id++;
	if (mon)
		{
		if (mon->private == NULL)
			mon->private = g_new0(MonPrivate, 1);
		mon->private->style_name = name;
		mon->private->style_type = CHART_PANEL_TYPE;
		mon->private->style_id = id;
		}
	GK.chart_name_list = g_list_append(GK.chart_name_list, (gchar *) name);
	style = gkrellm_style_new0();
	GK.chart_style_list = g_list_append(GK.chart_style_list, style);
	style = gkrellm_style_new0();
	GK.panel_style_list = g_list_append(GK.panel_style_list, style);
	GK.bg_chart_image_list = g_list_append(GK.bg_chart_image_list, NULL);
	GK.bg_grid_image_list = g_list_append(GK.bg_grid_image_list, NULL);
	GK.bg_panel_image_list = g_list_append(GK.bg_panel_image_list, NULL);
	GK.krell_panel_image_list = g_list_append(GK.krell_panel_image_list, NULL);
	return id;
	}

gint
gkrellm_add_meter_style(Monitor *mon, gchar *name)
	{
	Style		*style;
	gint		id;
	static gint	style_id;

	id = style_id++;
	if (mon)
		{
		if (mon->private == NULL)
			mon->private = g_new0(MonPrivate, 1);
		mon->private->style_name = name;
		mon->private->style_type = METER_PANEL_TYPE;
		mon->private->style_id = id;
		}
	style = gkrellm_style_new0();
	GK.meter_name_list = g_list_append(GK.meter_name_list, (gchar *) name);
	GK.meter_style_list = g_list_append(GK.meter_style_list, style);
	GK.bg_meter_image_list = g_list_append(GK.bg_meter_image_list, NULL);
	GK.krell_meter_image_list = g_list_append(GK.krell_meter_image_list, NULL);
	return id;
	}

static void
set_image_borders_in_list(GList *st_list, GList *im_list, GList *nm_list)
	{
	Style			*style;
	GdkImlibImage	*image;

	for ( ; st_list && im_list && nm_list;
			st_list = st_list->next, im_list = im_list->next,
			nm_list = nm_list->next)
		{
		style = (Style *) st_list->data;
		image = (GdkImlibImage *) im_list->data;
		if (style && image)
			{
			if (GK.debug_level & DEBUG_STYLES)
				printf(_("Setting border: %s\n"), (gchar *) nm_list->data);
			gdk_imlib_set_image_border(image, &style->border);
			}
		}
	}

static void
setup_images()
	{
	GtkWidget	*top_win = gkrellm_get_top_window();
	gint		i, w, h;

	gdk_imlib_set_image_border(GK.frame_top_image, &GK.frame_top_border);
	gdk_imlib_set_image_border(GK.frame_bottom_image, &GK.frame_bottom_border);
	if (GK.frame_left_width == 0)
		GK.frame_left_width = GK.frame_left_image->rgb_width;
	gdk_imlib_set_image_border(GK.frame_left_image, &GK.frame_left_border);
	if (GK.frame_right_width == 0)
		GK.frame_right_width = GK.frame_right_image->rgb_width;
	gdk_imlib_set_image_border(GK.frame_right_image, &GK.frame_right_border);

	gdk_imlib_set_image_border(GK.button_panel_out_image,
						&GK.button_panel_border);
	gdk_imlib_set_image_border(GK.button_panel_in_image,
						&GK.button_panel_border);

	gdk_imlib_set_image_border(GK.button_meter_out_image,
						&GK.button_meter_border);
	gdk_imlib_set_image_border(GK.button_meter_in_image,
						&GK.button_meter_border);

	set_image_borders_in_list(GK.chart_style_list, GK.bg_chart_image_list,
				GK.chart_name_list);
	set_image_borders_in_list(GK.panel_style_list, GK.bg_panel_image_list,
				GK.chart_name_list);
	set_image_borders_in_list(GK.meter_style_list, GK.bg_meter_image_list,
				GK.meter_name_list);

	if (GK.bg_timer_image)		/* Is legal for this to be NULL */
		gdk_imlib_set_image_border(GK.bg_timer_image, &GK.bg_timer_border);

	gkrellm_render_to_pixmap(GK.decal_misc_image, &GK.decal_misc_pixmap,
					&GK.decal_misc_mask, 0, 0);

	for (i = 0, h = 0; i < N_CHART_MONITORS; ++i)
		if (h < UC.chart_height[i])
			h = UC.chart_height[i];
	w = UC.chart_width;
	GK.max_chart_height = h;

	if (! gkrellm_render_to_pixmap(GK.data_in_image, &GK.data_in_pixmap,
				NULL, w, h))
		{
		GK.data_in_pixmap = gdk_pixmap_new(top_win->window, w, h, -1);
		gdk_gc_set_foreground(GK.draw1_GC, &GK.in_color);
		gdk_draw_rectangle(GK.data_in_pixmap, GK.draw1_GC, TRUE, 0, 0, w, h);
		}
	if (! gkrellm_render_to_pixmap(GK.data_out_image, &GK.data_out_pixmap,
				NULL, w, h))
		{
		GK.data_out_pixmap = gdk_pixmap_new(top_win->window, w, h, -1);
		gdk_gc_set_foreground(GK.draw1_GC, &GK.out_color);
		gdk_draw_rectangle(GK.data_out_pixmap, GK.draw1_GC, TRUE, 0, 0, w, h);
		}
	if (GK.data_in_grid_image)
		{
		if ((h = GK.data_in_grid_image->rgb_height) > 2)
			h = 2;
		gkrellm_render_to_pixmap(GK.data_in_grid_image,
					&GK.data_in_grid_pixmap, NULL, w, h);
		}
	else
		{
		GK.data_in_grid_pixmap = gdk_pixmap_new(top_win->window, w, 1, -1);
		gdk_gc_set_foreground(GK.draw1_GC, &GK.in_color_grid);
		gdk_draw_rectangle(GK.data_in_grid_pixmap, GK.draw1_GC,
						TRUE, 0, 0, w, 1);
		}
	if (GK.data_out_grid_image)
		{
		if ((h = GK.data_out_grid_image->rgb_height) > 2)
			h = 2;
		gkrellm_render_to_pixmap(GK.data_out_grid_image,
					&GK.data_out_grid_pixmap, NULL, w, h);
		}
	else
		{
		GK.data_out_grid_pixmap = gdk_pixmap_new(top_win->window, w, 1, -1);
		gdk_gc_set_foreground(GK.draw1_GC, &GK.out_color_grid);
		gdk_draw_rectangle(GK.data_out_grid_pixmap, GK.draw1_GC,
						TRUE, 0, 0, w, 1);
		}
	}


typedef struct
	{
	gchar			*name;
	gchar			**xpm;
	GdkImlibImage	**im;
	GList			**image_list;
	gchar			*name_in_list;
	}
	ImageTable;

static ImageTable base_theme_images[]	=
	{
	/* Images in this table which have a non NULL _xpm default form the
	|  minimal set of required images for a complete theme change.
	*/
{ "frame_top",		frame_top_xpm,		&GK.frame_top_image,	NULL, 0},
{ "frame_bottom",	frame_bottom_xpm,	&GK.frame_bottom_image,	NULL, 0},
{ "frame_left",  	frame_left_xpm,		&GK.frame_left_image,	NULL, 0},
{ "frame_right", 	frame_right_xpm,	&GK.frame_right_image,	NULL, 0},

{ "button_panel_out",	NULL,	&GK.button_panel_out_image,		NULL, 0},
{ "button_panel_in",	NULL,	&GK.button_panel_in_image,		NULL, 0},
{ "button_meter_out",	NULL,	&GK.button_meter_out_image,		NULL, 0},
{ "button_meter_in",	NULL,	&GK.button_meter_in_image,		NULL, 0},

{ "bg_chart",	 	bg_chart_xpm, NULL,	&GK.bg_chart_image_list,	"*"	},
{ "bg_chart_cpu", 	NULL,	NULL,	&GK.bg_chart_image_list, CPU_STYLE_NAME	},
{ "bg_chart_disk", 	NULL,	NULL,	&GK.bg_chart_image_list, DISK_STYLE_NAME	},
{ "bg_chart_net", 	NULL,	NULL,	&GK.bg_chart_image_list, NET_STYLE_NAME	},
{ "bg_chart_proc", 	NULL,	NULL,	&GK.bg_chart_image_list, PROC_STYLE_NAME	},
{ "bg_chart_inet", 	NULL,	NULL,	&GK.bg_chart_image_list, INET_STYLE_NAME	},

{ "bg_grid", 		bg_grid_xpm, NULL, &GK.bg_grid_image_list,	"*"},
{ "bg_grid_cpu",	NULL,	NULL,	&GK.bg_grid_image_list, CPU_STYLE_NAME	},
{ "bg_grid_disk", 	NULL,	NULL,	&GK.bg_grid_image_list, DISK_STYLE_NAME	},
{ "bg_grid_net",	NULL,	NULL,	&GK.bg_grid_image_list, NET_STYLE_NAME	},
{ "bg_grid_proc",	NULL,	NULL,	&GK.bg_grid_image_list, PROC_STYLE_NAME	},
{ "bg_grid_inet",	NULL,	NULL,	&GK.bg_grid_image_list, INET_STYLE_NAME	},

{ "bg_panel",		bg_panel_xpm, NULL, &GK.bg_panel_image_list,	"*" },
{ "bg_panel_cpu",	NULL,	NULL,	&GK.bg_panel_image_list, CPU_STYLE_NAME	},
{ "bg_panel_disk",	NULL,	NULL,	&GK.bg_panel_image_list, DISK_STYLE_NAME	},
{ "bg_panel_net",	NULL,	NULL,	&GK.bg_panel_image_list, NET_STYLE_NAME	},
{ "bg_panel_proc",	NULL,	NULL,	&GK.bg_panel_image_list, PROC_STYLE_NAME	},
{ "bg_panel_inet",	NULL,	NULL,	&GK.bg_panel_image_list, INET_STYLE_NAME	},

{ "bg_meter",		bg_meter_xpm, NULL, &GK.bg_meter_image_list, "*" },
{ "bg_meter_mem",	NULL, NULL,	&GK.bg_meter_image_list, MEM_STYLE_NAME },
{ "bg_meter_swap",	NULL, NULL,	&GK.bg_meter_image_list, SWAP_STYLE_NAME },
{ "bg_meter_fs",	NULL, NULL,	&GK.bg_meter_image_list, FS_STYLE_NAME },
{ "bg_meter_apm",	NULL, NULL,	&GK.bg_meter_image_list, APM_STYLE_NAME },
{ "bg_meter_mail",	NULL, NULL,	&GK.bg_meter_image_list, MAIL_STYLE_NAME },
{ "bg_meter_uptime",NULL, NULL,	&GK.bg_meter_image_list, UPTIME_STYLE_NAME },
{ "bg_meter_clock",	NULL, NULL,	&GK.bg_meter_image_list, CLOCK_STYLE_NAME },
{ "bg_meter_cal",	NULL, NULL,	&GK.bg_meter_image_list, CAL_STYLE_NAME },
{ "bg_meter_host",	NULL, NULL,	&GK.bg_meter_image_list, HOST_STYLE_NAME },
{ "bg_meter_timerbutton",NULL, NULL, &GK.bg_meter_image_list,
						TIMER_STYLE_NAME},

{ "bg_timer",		NULL,  &GK.bg_timer_image,	NULL, 0},
{ "bg_slider_panel", NULL, &GK.bg_slider_image[CHART_PANEL_TYPE], NULL, 0},
{ "bg_slider_meter", NULL, &GK.bg_slider_image[METER_PANEL_TYPE], NULL, 0},


{ "decal_timer_button", decal_timer_button_xpm,
								&GK.decal_timer_button_image, NULL, 0},
{ "decal_net_leds",	decal_net_leds_xpm,	&GK.decal_net_led_image, NULL, 0},
{ "decal_misc",		decal_misc_xpm,		&GK.decal_misc_image, NULL, 0},
{ "decal_mail",		decal_mail_xpm,		&GK.decal_mail_image, NULL, 0},

{ "data_in",	 		NULL,		&GK.data_in_image,	NULL, 0},
{ "data_in_grid", 	NULL,		&GK.data_in_grid_image,	NULL, 0},
{ "data_out",	 	NULL,		&GK.data_out_image,		NULL, 0},
{ "data_out_grid", 	NULL,		&GK.data_out_grid_image, NULL, 0},

{ "krell_panel",	krell_panel_xpm, NULL, &GK.krell_panel_image_list, "*"},
{ "krell_panel_cpu",  NULL, NULL, &GK.krell_panel_image_list, CPU_STYLE_NAME},
{ "krell_panel_disk", NULL, NULL, &GK.krell_panel_image_list, DISK_STYLE_NAME},
{ "krell_panel_net",  NULL, NULL, &GK.krell_panel_image_list, NET_STYLE_NAME},
{ "krell_panel_proc", NULL, NULL, &GK.krell_panel_image_list, PROC_STYLE_NAME},
{ "krell_panel_inet", NULL, NULL, &GK.krell_panel_image_list, INET_STYLE_NAME},

{ "krell_meter",	krell_meter_xpm, NULL, &GK.krell_meter_image_list, "*"},
{ "krell_meter_mem",  NULL,	NULL, &GK.krell_meter_image_list, MEM_STYLE_NAME},
{ "krell_meter_swap", NULL,	NULL, &GK.krell_meter_image_list, SWAP_STYLE_NAME},
{ "krell_meter_fs",	NULL,	NULL, &GK.krell_meter_image_list, FS_STYLE_NAME},
{ "krell_meter_apm",  NULL,	NULL, &GK.krell_meter_image_list, APM_STYLE_NAME},
{ "krell_meter_mail", krell_meter_mail_xpm, NULL,
								&GK.krell_meter_image_list, MAIL_STYLE_NAME },

{ "krell_slider",	krell_slider_xpm,	&GK.krell_slider_image, NULL, 0},
{ "krell_mini", 		krell_mini_xpm,		&GK.krell_mini_image, NULL, 0}
	};



static ImageTable	default_theme_images[] =
	{
{"button_panel_out", button_panel_out_xpm, &GK.button_panel_out_image,NULL, 0},
{"button_panel_in", button_panel_in_xpm, &GK.button_panel_in_image, NULL, 0},
{"button_meter_out", button_meter_out_xpm,&GK.button_meter_out_image, NULL, 0},
{"button_meter_in", button_meter_in_xpm, &GK.button_meter_in_image, NULL, 0},

{"bg_meter_fs",	bg_meter_fs_xpm, NULL, &GK.bg_meter_image_list, FS_STYLE_NAME},
{"bg_meter_mem", bg_meter_mem_xpm, NULL, &GK.bg_meter_image_list, MEM_STYLE_NAME },
{"bg_meter_swap",bg_meter_swap_xpm, NULL, &GK.bg_meter_image_list, SWAP_STYLE_NAME },
{"bg_meter_host",	bg_meter_host_xpm, NULL, &GK.bg_meter_image_list, HOST_STYLE_NAME },
{"bg_meter_uptime", bg_meter_uptime_xpm, NULL, &GK.bg_meter_image_list, UPTIME_STYLE_NAME },
{"bg_meter_timerbutton", bg_meter_timerbutton_xpm, NULL, &GK.bg_meter_image_list, TIMER_STYLE_NAME },

{"bg_timer",		bg_timer_xpm,	&GK.bg_timer_image, NULL, 0},
{"bg_slider_panel", bg_slider_panel_xpm,
							&GK.bg_slider_image[CHART_PANEL_TYPE], NULL, 0},
{"bg_slider_meter", bg_slider_meter_xpm,
							&GK.bg_slider_image[METER_PANEL_TYPE], NULL, 0},

{"data_in",			data_in_xpm,		&GK.data_in_image, NULL, 0},
{"data_in_grid",	data_in_grid_xpm,	&GK.data_in_grid_image, NULL, 0},
{"data_out",		data_out_xpm,		&GK.data_out_image,	NULL, 0},
{"data_out_grid",	data_out_grid_xpm,	&GK.data_out_grid_image, NULL, 0},

{"krell_meter_mem",	krell_meter_mem_xpm, NULL, 
							&GK.krell_meter_image_list, MEM_STYLE_NAME},
{"krell_meter_swap",	krell_meter_swap_xpm, NULL, 
							&GK.krell_meter_image_list, SWAP_STYLE_NAME},

/* Plugins */
{"bg_meter_volume",	bg_meter_volume_xpm, NULL, &GK.bg_meter_image_list,
		"volume"},		/* Track the volume plugin style name */
{"bg_meter_gkrellmms",	bg_meter_gkrellmms_xpm, NULL, &GK.bg_meter_image_list,
		"gkrellmms"},	/* Track the gkrellmms plugin style name*/
{"bg_meter_pmu",	bg_meter_pmu_xpm, NULL, &GK.bg_meter_image_list,
		"pmu"},			/* Track the pmu plugin style name */

	};


gboolean
gkrellm_load_image(gchar *name, gchar **xpm, GdkImlibImage **image,
		gchar *subdir)
	{
	GdkImlibImage	*im		= NULL;
	gchar			*fname,
					*source = "",
					*debug_action = _("Loading  ");

	if (gkrellm_using_default_theme() && name && subdir)
		{
		if (!strcmp(subdir, "gkrellmms") && !strcmp(name, "bg_scroll"))
			im = gdk_imlib_create_image_from_xpm_data(bg_scroll_gkrellmms_xpm);
		}
	else if (name && (fname = gkrellm_theme_file_exists(name, subdir)) != NULL)
		{
		source = _("file image");
		name = fname;
		im = gdk_imlib_load_image(fname);
		if (im == NULL)
			printf(_("  Cannot load file image: %s\n"), fname);
		}
	if (im == NULL && xpm)
		{
		source = _("xpm image");
		im = gdk_imlib_create_image_from_xpm_data(xpm);
		if (im == NULL)
			printf(_("  Cannot load xpm: %s\n"), name);
		}
	if (im && image)
		{
		if (*image)
			{
			debug_action = _("Replacing");
			gdk_imlib_kill_image(*image);
			}
		if (GK.debug_level & DEBUG_IMAGES)
			printf("  %s (%s) %s: %s\n", debug_action, subdir ? subdir :"BASE",
					source, name);
		*image = im;
		}
	return (im ? TRUE : FALSE);
    }


static void
load_from_image_list(gchar *name, GList *image_list, gint index, gchar *subdir)
	{
	GList			*list;
	GdkImlibImage	**im;

	list = g_list_nth(image_list, index);
	if (list)
		{
		im = (GdkImlibImage **) (&list->data);
		gkrellm_load_image(name, NULL, im, subdir);
		}
	else
		printf("Bad index %d for image list (meter/panel problem?)\n", index);
	}

static void
load_monitor_specific_images()
	{
	Monitor	*mon;
	GList	*list;
	gchar	*subdir, *type;
	gint	i;

	if (GK.debug_level & DEBUG_IMAGES)
		printf(_("====>Loading monitor specific theme images.<====\n"));

	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if ((subdir = mon->private->style_name) == NULL)
			continue;
		i = mon->private->style_id;
		type = (MONITOR_ID(mon) == MON_PLUGIN) ? "plugin " : "builtin";
		if (mon->private->style_type == CHART_PANEL_TYPE)
			{
			if (GK.debug_level & DEBUG_IMAGES)
				printf(_("Checking %s chart monitor: %s\n"), type, subdir);
			load_from_image_list("bg_chart", GK.bg_chart_image_list, i,subdir);
			load_from_image_list("bg_grid", GK.bg_grid_image_list, i, subdir);
			load_from_image_list("bg_panel", GK.bg_panel_image_list, i,subdir);
			load_from_image_list("krell", GK.krell_panel_image_list, i,subdir);
			}
		else
			{
			if (GK.debug_level & DEBUG_IMAGES)
				printf(_("Checking %s meter monitor: %s\n"), type, subdir);
			load_from_image_list("krell", GK.krell_meter_image_list, i,subdir);
			load_from_image_list("bg_panel", GK.bg_meter_image_list, i,subdir);
			load_from_image_list("bg_meter", GK.bg_meter_image_list, i,subdir);
			}
		gkrellm_load_image("spacer_top", NULL,
					&mon->private->spacer_top_image, subdir);
		gkrellm_load_image("spacer_bottom", NULL,
					&mon->private->spacer_bottom_image, subdir);
		}
	}

static Monitor *
lookup_monitor_from_style_name(gchar *style_name)
	{
	Monitor	*mon;
	GList	*list;

	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (!mon->private || !mon->private->style_name)
			continue;
		if (!strcmp(style_name, mon->private->style_name))
			return mon;
		}
	return NULL;
	}

static void
assign_gkrellmrc_spacer_height(gchar *source_line, gchar *area, gchar *string)
	{
	Monitor	*mon;
	gchar	style_name[32];
	gint	h;

	if (sscanf(string, "%32s %d", style_name, &h) != 2)
		return;
	if ((mon = lookup_monitor_from_style_name(style_name)) == NULL)
		return;
	if (!strcmp(area, "spacer_top_height"))
		mon->private->spacer_top_height = h;
	else if (!strcmp(area, "spacer_bottom_height"))
		mon->private->spacer_bottom_height = h;
	}


  /* Yeoww! I have to do something about separate chart/meter lists.
  */
static GList *
lookup_image_from_name(GList *image_list, gchar *name)
	{
	GList	*n_list, *i_list;

	for (n_list = GK.chart_name_list, i_list = image_list;
			n_list && i_list; n_list = n_list->next, i_list = i_list->next)
		if (!strcmp(name, (gchar *) n_list->data))
			return i_list;
	for (n_list = GK.meter_name_list, i_list = image_list;
			n_list && i_list; n_list = n_list->next, i_list = i_list->next)
		if (!strcmp(name, (gchar *) n_list->data))
			return i_list;
	return NULL;
	}

static void
load_image_table(ImageTable *ti, gint n_images, gchar *subdir)
	{
	GdkImlibImage	**im;
	GList			*list;
	gint			i;

	for (i = 0; i < n_images; ++i, ++ti)
		{
		if (ti->image_list)
			{
/*			list = g_list_nth(*(ti->image_list), ti->list_index); */
			list = lookup_image_from_name(*(ti->image_list), ti->name_in_list);
			if (list)
				{
				im = (GdkImlibImage **) (&list->data);
				gkrellm_load_image(ti->name, ti->xpm, im, subdir);
				}
			}
		else
			gkrellm_load_image(ti->name, ti->xpm, ti->im, subdir);
		}
    }

  /* When loading a new theme, required base level images are not cleaned
  |  so the program will not crash if a theme does not have all images yet.
  |  It will just look funny.  But all optional base level images are cleaned
  |  so they will not carry over to the new theme.  There are no optional
  |  base level images in the image_lists.
  */
static void
clean_base_image_table()
	{
	ImageTable		*ti;
	gint			i;

	ti = &base_theme_images[0];
	for (i = 0; i < sizeof(base_theme_images) / sizeof(ImageTable); ++i, ++ti)
		if (ti->xpm == NULL && ti->im && *(ti->im))	/* Is an optional image */
			{
			if (GK.debug_level & DEBUG_IMAGES)
				printf(_("Killing image from table: %s\n"), ti->name);
			gdk_imlib_kill_image(*(ti->im));
			*(ti->im) = NULL;
			}
	}

static void
kill_image(GdkImlibImage **im)
	{
	if (im && *im)
		{
		gdk_imlib_kill_image(*im);
		*im = NULL;
		}
	}

static void
kill_image_list(GList *list, GList *name_list, gchar *debug_name)
	{
	for ( ; list; list = list->next, name_list = name_list->next)
		kill_image((GdkImlibImage **) (&list->data));
	}

static void
kill_monitor_specific_images()
	{
	Monitor	*mon;
	GList	*list;

	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		kill_image(&mon->private->bg_chart_image);
		kill_image(&mon->private->bg_panel_image);
		kill_image(&mon->private->bg_grid_image);
		kill_image(&mon->private->krell_image);
		kill_image(&mon->private->spacer_top_image);
		kill_image(&mon->private->spacer_bottom_image);	
		}
	}

void
load_theme_images()
	{
	Monitor	*mon;
	gint	n_base, n_default;

	/* Free up all custom images from old theme.
	*/
	kill_image_list(GK.bg_chart_image_list, GK.chart_name_list, "bg_chart");
	kill_image_list(GK.bg_grid_image_list, GK.chart_name_list, "bg_grid");
	kill_image_list(GK.bg_panel_image_list, GK.chart_name_list, "bg_panel");
	kill_image_list(GK.bg_meter_image_list, GK.meter_name_list, "bg_meter");
	kill_image_list(GK.krell_panel_image_list, GK.chart_name_list, "krell_panel");
	kill_image_list(GK.krell_meter_image_list, GK.meter_name_list, "krell_meter");
	kill_monitor_specific_images();

	n_base = sizeof(base_theme_images) / sizeof(ImageTable);
	n_default = sizeof(default_theme_images) / sizeof(ImageTable);

	clean_base_image_table();
	if (GK.debug_level & DEBUG_IMAGES)
		printf(_("====>Loading base level theme images.<====\n"));

	/* This loads the base images in the top level of the theme directory.
	|  For backward compatibility, it also loads monitor specific name
	|  qualified images in the top level directory.  The new way is for
	|  monitor specific images to be in subdirectories, loaded below.
	*/
	load_image_table(&base_theme_images[0], n_base, NULL);

	if (gkrellm_using_default_theme())
		{
		if (GK.debug_level & DEBUG_IMAGES)
			printf(_("====>Loading additional default theme images.<====\n"));
		load_image_table(&default_theme_images[0], n_default, NULL);
		if ((mon = lookup_monitor_from_style_name("volume")) != NULL)
			{
			gkrellm_load_image(NULL, spacer_top_volume_xpm,
						&mon->private->spacer_top_image, NULL);
			gkrellm_load_image(NULL, spacer_bottom_volume_xpm,
						&mon->private->spacer_bottom_image, NULL);
			}
		if ((mon = lookup_monitor_from_style_name("gkrellmms")) != NULL)
			{
			gkrellm_load_image(NULL, spacer_top_gkrellmms_xpm,
						&mon->private->spacer_top_image, NULL);
			gkrellm_load_image(NULL, spacer_bottom_gkrellmms_xpm,
						&mon->private->spacer_bottom_image, NULL);
			}
		if ((mon = lookup_monitor_from_style_name("pmu")) != NULL)
			{
			gkrellm_load_image(NULL, spacer_top_pmu_xpm,
						&mon->private->spacer_top_image, NULL);
			gkrellm_load_image(NULL, spacer_bottom_pmu_xpm,
						&mon->private->spacer_bottom_image, NULL);
			}
		}
	else
		{
		load_monitor_specific_images();
		}
	setup_images();
	}

  /* Borders for things that are not primary background parts of a monitor,
  |  and so are not set by a style line.
  */
static gchar
			*frame_top_border,
			*frame_bottom_border,
			*frame_left_border,
			*frame_right_border,
			*button_panel_border,
			*button_meter_border,
            *bg_timer_border,
            *bg_slider_panel_border,
            *bg_slider_meter_border;

gint		krell_slider_depth,
			krell_slider_x_hot;

static struct	_config
	{
	gchar	*option;
	gint	*value;
	gchar	**arg;
	gint	minimum;
	}
	theme_config []	=
	{
	{"author",				NULL,		NULL,			-100 },

	{"theme_alternatives",	&GK.theme_n_alternatives,	NULL,			0  },

	{"frame_top_height",	&GK.frame_top_height,		NULL,			0  },
	{"frame_bottom_height",	&GK.frame_bottom_height,	NULL,			0  },
	{"frame_left_width",	&GK.frame_left_width,		NULL,			0  },
	{"frame_right_width",	&GK.frame_right_width,		NULL,			0  },
	{"chart_width_ref",		&GK.chart_width_ref,		NULL,			30 },
	{"chart_height_min",	&GK.chart_height_min,		NULL,			2 },
	{"chart_height_max",	&GK.chart_height_max,		NULL,			20 },
	{"allow_scaling",		&GK.allow_scaling,			NULL,			0 },

	{"rx_led_x",			&GK.rx_led_x,				NULL,			-99 },
	{"rx_led_y",			&GK.rx_led_y,				NULL,			0   },
	{"tx_led_x",			&GK.tx_led_x,				NULL,			-99 },
	{"tx_led_y",			&GK.tx_led_y,				NULL,			0   },

	{"decal_mail_frames",	&GK.decal_mail_frames,	NULL,			1  },
	{"decal_mail_delay",	&GK.decal_mail_delay,	NULL,			1  },

	{"large_font",			NULL,		&GK.large_font_string,		-100 },
	{"normal_font",			NULL,		&GK.normal_font_string,		-100 },
	{"small_font",			NULL,		&GK.small_font_string,		-100 },

	{"chart_in_color",		NULL,		&GK.chart_in_color,		-100 },
	{"chart_in_color_grid",	NULL,		&GK.chart_in_color_grid,	-100 },
	{"chart_out_color",		NULL,		&GK.chart_out_color,		-100 },
	{"chart_out_color_grid",NULL,		&GK.chart_out_color_grid,	-100 },

	{"bg_grid_mode",		&GK.bg_grid_mode,		NULL,			0  },

    {"frame_top_border",	NULL,		&frame_top_border,			-100 },
    {"frame_bottom_border",	NULL,		&frame_bottom_border,		-100 },
    {"frame_left_border",	NULL,		&frame_left_border,			-100 },
    {"frame_right_border",	NULL,		&frame_right_border,		-100 },
    {"button_panel_border", NULL,		&button_panel_border,		-100 },
    {"button_meter_border", NULL,		&button_meter_border,		-100 },

    {"bg_timer_border",		NULL,		&bg_timer_border,			-100 },
    {"bg_slider_panel_border", NULL,	&bg_slider_panel_border,	-100 },
    {"bg_slider_meter_border", NULL,	&bg_slider_meter_border,	-100 },

	{"krell_slider_depth",	&krell_slider_depth,	NULL,			1  },
	{"krell_slider_x_hot",	&krell_slider_x_hot,	NULL,			-1  },
	};


static gchar	*font_override_1[] =
	{
	"-b&h-lucida-bold-r-normal-*-*-140-*-*-*-*-*-*",		/* large  */
	"-b&h-lucida-medium-r-normal-*-*-120-*-*-*-*-*-*",		/* normal */
	"-adobe-helvetica-medium-r-*-*-*-100-*-*-*-*-*-*"		/* small  */
	};

static gchar	*font_override_2[] =
	{
	"-b&h-lucida-bold-r-normal-*-*-160-*-*-*-*-*-*",		/* large  */
	"-b&h-lucida-medium-r-normal-*-*-140-*-*-*-*-*-*",		/* normal */
	"-adobe-helvetica-medium-r-*-*-*-120-*-*-*-*-*-*"		/* small  */
	};

  /* Handle borders set in gkrellmrc which are not set by a style line.
  */
static void
cleanup_gkrellmrc()
	{
	if (GK.font_override == 1)
		{
		gkrellm_dup_string(&GK.large_font_string, font_override_1[0]);
		gkrellm_dup_string(&GK.normal_font_string, font_override_1[1]);
		gkrellm_dup_string(&GK.small_font_string, font_override_1[2]);
		}
	if (GK.font_override == 2)
		{
		gkrellm_dup_string(&GK.large_font_string, font_override_2[0]);
		gkrellm_dup_string(&GK.normal_font_string, font_override_2[1]);
		gkrellm_dup_string(&GK.small_font_string, font_override_2[2]);
		}
	set_border(&GK.frame_top_border, frame_top_border);
	set_border(&GK.frame_bottom_border, frame_bottom_border);
	set_border(&GK.frame_left_border, frame_left_border);
	set_border(&GK.frame_right_border, frame_right_border);

	set_border(&GK.button_panel_border, button_panel_border);
	set_border(&GK.button_meter_border, button_meter_border);
	set_border(&GK.bg_timer_border, bg_timer_border);
	set_border(&GK.bg_slider_border[CHART_PANEL_TYPE], bg_slider_panel_border);
	set_border(&GK.bg_slider_border[METER_PANEL_TYPE], bg_slider_meter_border);

	GK.krell_slider_style->krell_x_hot  = krell_slider_x_hot;
	GK.krell_slider_style->krell_depth  = krell_slider_depth;
	}

static GList	*gkrellmrc_border_list,
				*gkrellmrc_integer_list,
				*gkrellmrc_string_list;


static GdkImlibBorder	zero_border;

gboolean
gkrellm_set_image_border(gchar *image_name, GdkImlibImage *image, Style *style)
	{
	GdkImlibBorder	b;
	GList			*list;
	gchar			name[64], border_string[32];
	gchar			*s, *r;

	if (style)
		style->border = zero_border;
	if (!image || !image_name)
		return FALSE;
	for (list = gkrellmrc_border_list; list; list = list->next)
		{
		s = list->data;
		if ((r = strchr(s, '=')) != NULL)
			*r = ' ';
		sscanf(s, "%64s %32s", name, border_string);
		if (!strcmp(name, image_name))
			{
			set_border(&b, border_string);
			gdk_imlib_set_image_border(image, &b);
#if 0
printf("set_image_border %s %d %d %d %d %s\n",
name, b.left, b.right, b.top, b.bottom,
style ? "Setting style" : "Not setting style");
#endif
			if (style)
				style->border = b;
			return TRUE;
			}
		}
	return FALSE;
	}

gboolean
gkrellm_get_gkrellmrc_integer(gchar *int_name, gint *result)
	{
	GList	*list;
	gchar	name[64], string[64];
	gchar	*s, *r;

	if (!int_name || !result)
		return FALSE;
	for (list = gkrellmrc_integer_list; list; list = list->next)
		{
		s = list->data;
		if ((r = strchr(s, '=')) != NULL)
			*r = ' ';
		sscanf(s, "%64s %64s", name, string);
		if (!strcmp(name, int_name))
			{
			if (sscanf(string, "%d", result) == 1)
				return TRUE;
			break;
			}
		}
	return FALSE;
	}

gchar *
gkrellm_get_gkrellmrc_string(gchar *string_name)
	{
	GList	*list;
	gchar	name[64], string[CFG_BUFSIZE];
	gchar	*s, *r;

	if (!string_name)
		return NULL;
	for (list = gkrellmrc_string_list; list; list = list->next)
		{
		s = list->data;
		if ((r = strchr(s, '=')) != NULL)
			*r = ' ';
		sscanf(s, "%64s %[^\n]", name, string);
		if (!strcmp(name, string_name))
			{
			if ((s = cut_quoted_string(string, NULL)) != NULL)
				return g_strdup(s);
			break;
			}
		}
	return NULL;
	}

static void
parse_config_line(gchar *line, struct _config *cf, gint size)
	{
	GList			*list;
	Monitor			*mon;
	struct _config	*pc;
	gchar			*s, *arg;
	gint			i, val, foundit;
	gchar			buf[CFG_BUFSIZE], pbuf[CFG_BUFSIZE];

	strncpy(buf, line, CFG_BUFSIZE);	/* strtok() is destructive */
	buf[CFG_BUFSIZE - 1] = '\0';
	s = strtok(buf, " \t:=\n");
	if (s == NULL || *s == '#' || *s == '\n' || *s == '\0')
		return;

	/* Special case diversions.
	*/
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (!mon->config_keyword || strcmp(mon->config_keyword, s))
			continue;
		arg = strtok(NULL, "\n");
		if (mon->load_user_config && mon->private->enabled)
			(*(mon->load_user_config))(arg);
		return;
		}
	if (strncmp(s, "Style", 5) == 0)	/* StyleChart ... */
		{
		arg = strtok(NULL, "\n");
		assign_gkrellmrc_style(line, s, arg);
		return;
		}
	if (strncmp(s, "spacer_", 7) == 0)
		{
		arg = strtok(NULL, "\n");
		assign_gkrellmrc_spacer_height(line, s, arg);
		return;
		}
	if (strcmp(s, "set_image_border") == 0)
		{
		arg = strtok(NULL, "\n");
		gkrellmrc_border_list = g_list_append(gkrellmrc_border_list,
				g_strdup(arg));
		return;
		}
	if (strcmp(s, "set_integer") == 0)
		{
		arg = strtok(NULL, "\n");
		gkrellmrc_integer_list = g_list_append(gkrellmrc_integer_list,
				g_strdup(arg));
		return;
		}
	if (strcmp(s, "set_string") == 0)
		{
		arg = strtok(NULL, "\n");
		gkrellmrc_string_list = g_list_append(gkrellmrc_string_list,
				g_strdup(arg));
		return;
		}
	arg = strtok(NULL, " \t:=\n");

	/* Must allow '#' char for arguments - color setting.
	*/
	if (arg == NULL || *arg == '\n' || *arg == '\0')
		{
		printf(_("Incomplete config line:\n    %s\n"), line);
		return;
		}
	if (isdigit(*arg) || *arg == '-')
		val = atoi(arg);
	else
		val = 0;
	if (strcmp(arg, "off") == 0 || strcmp(arg, "false") == 0)
		val = 0;
	if (strcmp(arg, "on") == 0 || strcmp(arg, "true") == 0)
		val = 1;

	foundit = FALSE;
	for (i = 0; i < size; ++i)
		{
		pc = cf + i;
		if (strcmp(pc->option, s) == 0)
			{
			if (pc->value)
				*(pc->value) = val;
			if (pc->arg)
				{
				/* String args can have quotes around them.
				*/
				if (*arg == '"')
					{
					strcpy(pbuf, arg + 1);
					arg = strtok(NULL, "#\n");	/* Get rest of line */
					if (arg)
						{
						strcat(pbuf, " ");
						strcat(pbuf, arg);
						}
					arg = strchr(pbuf, (int) '"');
					if (arg)
						*arg = '\0';
					gkrellm_dup_string(pc->arg, pbuf);
					}
				else
					gkrellm_dup_string(pc->arg, arg);
				}
			if (pc->minimum > -100 && val < pc->minimum)
				*(pc->value) = pc->minimum;
			foundit = TRUE;
			if (GK.debug_level & DEBUG_GKRELLMRC)
				printf(_("parse_config %s %s\n"), s, arg);
			break;
			}
		}
	if ((GK.debug_level & DEBUG_GKRELLMRC) && !foundit)
		printf(_("parse_config cannot grok %s.\n"), s);
	}

static void
user_config_defaults()
	{
	gint	i;

	UC.enable_hostname = TRUE;
	UC.hostname_short = FALSE;

	UC.chart_width = 60;
	for (i = 0; i < N_CHART_MONITORS; ++i)
		UC.chart_height[i] = 36;

	UC.proc_load_bar_graph = FALSE;
	UC.proc_clip_processes = TRUE;

	UC.ema_period = 4;
	UC.update_HZ = 10;
/*	UC.bump_HZ = 1; */
	}

static void
clear_style_list(GList *list, GList *name_list, gchar *debug_name)
	{
	Style	*style;

	for ( ; list; list = list->next, name_list = name_list->next)
		{
		style = (Style *) list->data;
		if (style)
			{
			if (GK.debug_level & DEBUG_STYLES)
				printf(_("    Clearing style from list: %s %s\n"), debug_name,
							(gchar *) name_list->data);
			memset(style, 0, sizeof(Style));
			}
		}
	}

static gchar	*base_theme[]	=
	{
	"StyleChart *.border = 0,0,0,0",
	"StyleChart *.font = normal_font",
	"StyleChart *.alt_font = small_font",
	"StyleChart *.textcolor = #efd097 #000000 shadow",
	"StyleChart *.alt_textcolor = #c0c0c0 #181818 shadow",

	"StylePanel *.border = 0,0,2,1",
	"StylePanel *.font = normal_font",
	"StylePanel *.alt_font = normal_font",
	"StylePanel *.textcolor = white #000000 shadow",
	"StylePanel *.alt_textcolor = #d8e0c8 #000000 shadow",
	"StylePanel *.label_position = 50",
	"StylePanel *.margin = 1",
	"StylePanel *.top_margin = 2",
	"StylePanel *.bottom_margin = 1",
	"StylePanel *.krell_yoff = 0",
	"StylePanel *.krell_x_hot = 3",
	"StylePanel *.krell_depth = 4",
	"StylePanel *.krell_expand = 0",
	"StylePanel *.krell_ema_period = 3",

	"StyleMeter *.border = 3,3,3,2",
	"StyleMeter *.font = normal_font",
	"StyleMeter *.alt_font = small_font",
	"StyleMeter *.textcolor = #ffeac4 #000000 shadow",
	"StyleMeter *.alt_textcolor = wheat #000000 shadow",
	"StyleMeter *.label_position = 50",
	"StyleMeter *.margin = 2",
	"StyleMeter *.top_margin = 2",
	"StyleMeter *.bottom_margin = 2",
	"StyleMeter *.krell_yoff = 1",
	"StyleMeter *.krell_expand = 0",
	"StyleMeter *.krell_ema_period = 1",
	"StyleMeter *.krell_x_hot = -1",
	"StyleMeter *.krell_depth = 1",

	/* These have an override effect */
	"StyleMeter apm.bottom_margin = 2",
	"StyleMeter mail.krell_depth = 15",
	"StyleMeter mail.krell_yoff = 0",
	"StyleMeter mail.krell_x_hot = -1",
	"StyleMeter mail.krell_expand = 0",
	"StyleMeter mail.label_position = 70",
	"StyleChart net.alt_font = small_font",
	};


static gchar	*default_fonts[] =
	{
	"-b&h-lucida-bold-r-normal-*-*-120-*-*-*-*-*-*",		/* large  */
	"-b&h-lucida-medium-r-normal-*-*-100-*-*-*-*-*-*",		/* normal */
	"-adobe-helvetica-medium-r-normal-*-*-80-*-*-*-*-*-*"	/* small  */
	};

static void
theme_config_init()
	{
	Monitor	*mon;
	GList	*list;
	Style	*style;
	gint	style_id;
	gint	i;

	GK.theme_n_alternatives = 0;
	GK.frame_top_height = 0;	/* use image height. */
	GK.frame_bottom_height = 0;	/* use image height. */
	GK.frame_left_width = 0;
	GK.frame_right_width = 0;
	GK.chart_height_min = 2;
	GK.chart_height_max = 150;
	GK.chart_width_ref = 60;
	GK.allow_scaling = TRUE;

	GK.rx_led_x = 2;
	GK.rx_led_y = 6;
	GK.tx_led_x = -2;
	GK.tx_led_y = 6;

	GK.decal_mail_frames = 18;
	GK.decal_mail_delay = 1;

	gkrellm_dup_string(&GK.large_font_string, default_fonts[0]);
	gkrellm_dup_string(&GK.normal_font_string, default_fonts[1]);
	gkrellm_dup_string(&GK.small_font_string, default_fonts[2]);

	gkrellm_dup_string(&GK.chart_in_color, "#10d3d3");
	gkrellm_dup_string(&GK.chart_in_color_grid, "#00a3a3");
	gkrellm_dup_string(&GK.chart_out_color, "#f4ac4a");
	gkrellm_dup_string(&GK.chart_out_color_grid, "#b47c20");


	/* Setup the default styles.  Substyles may be set in gkrellmrc.  If
	|  they are not, then they will be set to point to the default after
	|  parsing the gkrellmrc file.
	*/
	clear_style_list(GK.chart_style_list, GK.chart_name_list, _("Chart"));
	clear_style_list(GK.panel_style_list, GK.chart_name_list, _("Panel"));
	clear_style_list(GK.meter_style_list, GK.meter_name_list, _("Meter"));

	for (i = 0; i < sizeof(base_theme) / sizeof(gchar *); ++i)
		parse_config_line(base_theme[i], &theme_config[0],
					sizeof(theme_config) / sizeof (struct _config));

	/* Allow themes to transition to using top/bottom margins. */
	GK.use_top_bottom_margins = FALSE;

	/* I set some base theme parameters with no override.  The idea is if
	|  a theme does not bother to set anything, these will remain in effect,
	|  but if the theme sets any "*" settings, they can wipe these out.
	|  This is probably a mistake, I am contributing to theme author
	|  laziness and should move these to the default theme.
	*/
	style_id = gkrellm_lookup_meter_style_id(APM_STYLE_NAME);
	assign_meter_style(style_id, "3,3,2,2", SET_BORDER, 0);
	assign_meter_style(style_id,"small_font", SET_TEXTFONT_A, 0);

	style_id = gkrellm_lookup_meter_style_id(CAL_STYLE_NAME);
	assign_meter_style(style_id,  "small_font", SET_TEXTFONT_A, 0);
	assign_meter_style(style_id,  "large_font", SET_TEXTFONT_B, 0);

	style_id = gkrellm_lookup_meter_style_id(CLOCK_STYLE_NAME);
	assign_meter_style(style_id,"large_font", SET_TEXTFONT_A, 0);

	style_id = gkrellm_lookup_meter_style_id(FS_STYLE_NAME);
	assign_meter_style(style_id, "0", SET_LABEL_POSITION, 0);

	if (GK.krell_slider_style)
		g_free(GK.krell_slider_style);
	GK.krell_slider_style = gkrellm_style_new0();
	style = (Style *) GK.meter_style_list->data;
	*GK.krell_slider_style = *style;

	if (GK.krell_mini_style)
		g_free(GK.krell_mini_style);
	GK.krell_mini_style = gkrellm_style_new0();
	*GK.krell_mini_style = *style;

	gkrellm_dup_string(&frame_top_border, "0,0,0,0");
	gkrellm_dup_string(&frame_bottom_border, "0,0,0,0");
	gkrellm_dup_string(&frame_left_border, "0,0,0,0");
	gkrellm_dup_string(&frame_right_border, "0,0,0,0");
	gkrellm_dup_string(&button_panel_border, "2,2,2,2");
	gkrellm_dup_string(&button_meter_border, "2,2,2,2");

	gkrellm_dup_string(&bg_timer_border, "1,1,2,2");
	gkrellm_dup_string(&bg_slider_panel_border, "1,1,1,1");
	gkrellm_dup_string(&bg_slider_meter_border, "1,1,1,1");

	krell_slider_x_hot = -1;
	krell_slider_depth = 6;

	free_glist_and_data(&gkrellmrc_border_list);
	free_glist_and_data(&gkrellmrc_integer_list);
	free_glist_and_data(&gkrellmrc_string_list);
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		mon->private->spacer_top_height = 0;
		mon->private->spacer_bottom_height = 0;
		}
	}

static gchar	*default_theme[]	=
	{
	"frame_left_border = 0,0,42,16",
	"frame_right_border = 0,0,42,16",

	"StyleMeter cal.textcolor = azure2 #000000 shadow",

	"StyleMeter clock.textcolor = wheat #000000 shadow",
	"StyleMeter clock.alt_textcolor = azure2 #000000 shadow",

	"StyleMeter fs.border = 3,3,3,1",
	"StyleMeter fs.top_margin = 2",
	"StyleMeter fs.bottom_margin = 0",
	"StyleMeter fs.label_position = 0",
	"StyleMeter fs.alt_font = normal_font",

	"StyleMeter host.textcolor = #c8d4d2 #000000 shadow",
	"StyleMeter host.top_margin = 1",
	"StyleMeter host.bottom_margin = 0",

	"StyleMeter mail.alt_textcolor = #ffc0c0 #000000 shadow",

	"StyleMeter mem.krell_yoff = 0",
	"StyleMeter mem.alt_font = normal_font",
	"StyleMeter mem.bottom_margin = 1",

	"StyleMeter swap.krell_yoff = 2",
	"StyleMeter swap.alt_font = normal_font",

	"StyleMeter sensors.alt_textcolor = #d8e0c8 #000000 shadow",
	"StyleMeter sensors.top_margin = 3",
	"StyleMeter sensors.bottom_margin = 3",
	"set_image_border sensors_bg_volt 2,1,1,1",

	"StyleMeter timer.textcolor = wheat #000000 shadow",
	"StyleMeter timer.alt_textcolor = azure2 #000000 shadow",
	"StyleMeter timer.margin = 1",
	"StyleMeter timer.top_margin = 1",

	"StyleMeter uptime.textcolor = wheat #000000 shadow",
	"StyleMeter uptime.border = 3,3,2,2",
	"StyleMeter uptime.bottom_margin = 1",

	/* plugins */
	"spacer_top_height gkrellmms 3",
	"spacer_bottom_height gkrellmms 3",
	"spacer_top_height volume 3",
	"spacer_bottom_height volume 2",
	"spacer_top_height pmu 3",
	"spacer_bottom_height pmu 2",

	/* GKrellMMS scroll bar panel */
	"set_image_border gkrellmms_bg_scroll 3,3,2,2",
	"set_integer gkrellmms_scroll_margin 2",
	"set_integer gkrellmms_scroll_top_margin 2",
	"set_integer gkrellmms_scroll_bottom_margin 2",
	/* GKrellMMS time bar and button bar panels */
	"StyleMeter gkrellmms.alt_textcolor = black #dcdccc shadow",
	"StyleMeter gkrellmms.margin = 2",
	"StyleMeter gkrellmms.top_margin = 2",
	"StyleMeter gkrellmms.bottom_margin = 0",
	"StyleMeter gkrellmms.krell_yoff = 0",

	/* All volume panels */
	"StyleMeter volume.label_position = 0",
	"StyleMeter volume.border = 26,3,1,1",
	"StyleMeter volume.top_margin = 1",
	"StyleMeter volume.bottom_margin = 0",
	"StyleMeter volume.textcolor #b8c4c2 #000000 shadow",
	};

static gchar	*default_theme_1[]	=
	{
	"StyleChart *.textcolor #a0a8c0 #000000 shadow",
	"StyleChart *.alt_textcolor #202020 #747474 shadow",
	"StylePanel *.textcolor #c0c0ca #000000 shadow",
	"StylePanel *.alt_textcolor #d2d8c0 #000000 shadow",
	"StyleMeter *.textcolor #c2c2b6 #000000 shadow",
	"StyleMeter *.alt_textcolor #d2d8c0 #000000 shadow",
	"StyleChart net.alt_textcolor #a0a8c0 #000000 shadow",
	"StyleMeter mem.textcolor #282828 #a0b0a0 shadow",
	"StyleMeter swap.textcolor #282828 #a0b0a0 shadow",
	"StyleMeter mail.alt_textcolor #e0c0c0 #000000 shadow",
	"StyleMeter timer.textcolor #d2d8c0 #000000 shadow",
	"StyleMeter timer.alt_textcolor = #d2d8e0 #000000 shadow",
	"StyleMeter uptime.textcolor #d2d8c0 #000000 shadow",
	"StyleMeter host.textcolor #c0ccc6 #000000 shadow",
	"StyleMeter clock.textcolor #d2d8c0 #000000 shadow",
	"StyleMeter clock.alt_textcolor = #d2d8e0 #000000 shadow",
	"StyleMeter cal.textcolor #d2d8e0 #000000 shadow",
	"StyleMeter sensors.textcolor = #d8e0c8 #000000 shadow",
	"StyleMeter sensors.alt_textcolor = #181818 #a0b0a0 shadow",

	/* plugins */
	"StyleMeter volume.textcolor #282828 #a0b0a0 shadow",
	};

static gchar	*default_theme_2[]	=
	{
	"StyleChart *.transparency 1",
	};

static void
default_theme_config()
	{
	gint	style_id,
			i;

	GK.theme_n_alternatives = 3;
	if (GK.theme_alternative > GK.theme_n_alternatives)
		GK.theme_alternative = GK.theme_n_alternatives;

	for (i = 0; i < sizeof(default_theme) / sizeof(gchar *); ++i)
		parse_config_line(default_theme[i], &theme_config[0],
					sizeof(theme_config) / sizeof (struct _config));

	style_id = gkrellm_lookup_meter_style_id(SWAP_STYLE_NAME);
	if (GK.font_override == 1)
		assign_meter_style(style_id, "4", SET_KRELL_YOFF, 1);
	if (GK.font_override == 2)
		assign_meter_style(style_id, "6", SET_KRELL_YOFF, 1);

	if (GK.theme_alternative & 1)
		{
		for (i = 0; i < sizeof(default_theme_1) / sizeof(gchar *); ++i)
			parse_config_line(default_theme_1[i], &theme_config[0],
						sizeof(theme_config) / sizeof (struct _config));
		}
	if (GK.theme_alternative & 2)
		{
		for (i = 0; i < sizeof(default_theme_2) / sizeof(gchar *); ++i)
			parse_config_line(default_theme_2[i], &theme_config[0],
						sizeof(theme_config) / sizeof (struct _config));
		}
	}


static void
parse_gkrellmrc(gint alternative)
	{
	FILE	*f;
	gchar	*path, lbuf[16];
	gchar	buf[CFG_BUFSIZE];

	lbuf[0] = '\0';
	if (alternative > 0)
		snprintf(lbuf, sizeof(lbuf), "_%d", alternative);
	path = g_strdup_printf("%s/%s%s", GK.theme_path, GKRELLMRC, lbuf);
	if ((f = fopen(path, "r")) != NULL)
		{
		if (GK.debug_level & DEBUG_GKRELLMRC)
			printf(_("--> Parsing %s\n"), path);
		while (fgets(buf, sizeof(buf), f))
			parse_config_line(buf, &theme_config[0],
					sizeof(theme_config) / sizeof (struct _config));
		fclose(f);
		}
	else if (GK.debug_level & DEBUG_GKRELLMRC)
		printf(_("--> %s not found\n"), path);

	g_free(path);
	}

gboolean
gkrellm_using_default_theme()
	{
	return (*(GK.theme_path) == '\0') ? TRUE : FALSE;
	}

void
gkrellmrc_theme_config()
	{
	FILE	*f;
	gchar	*path, *s;
	gchar	buf[256];

	theme_config_init();

	if (GK.theme_path == NULL)
		{
		path = gkrellm_make_config_file_name(gkrellm_homedir(),
				GKRELLM_THEME_CONFIG);
		f = fopen(path, "r");
		g_free(path);
		if (f != NULL)
			{
			fgets(buf, sizeof(buf), f);
			s = strtok(buf, " \n\t");
			if (s && *s != '#' && *s != '\0' && strcmp(s, "Default"))
				{
				if (*s == '/')
					GK.theme_path = g_strdup(s);
			  	else
					GK.theme_path = g_strdup_printf("%s/%s/%s",
							gkrellm_homedir(), GKRELLM_THEMES_DIR, s);
				if (!isdir(GK.theme_path, NULL))
					{
					g_free(GK.theme_path);
					GK.theme_path = NULL;
					}
				}
			if (fgets(buf, sizeof(buf), f))
				{
				GK.theme_alternative = strtoul(buf, NULL, 0);
				if (fgets(buf, sizeof(buf), f))
					GK.font_override = strtoul(buf, NULL, 0);
				}
			fclose(f);
			}
		}
	if (GK.theme_path == NULL)
		GK.theme_path = g_strdup("");
	if (GK.debug_level & DEBUG_GKRELLMRC)
		printf(_("GK.theme_path <%s>\n"), GK.theme_path);

	if (gkrellm_using_default_theme())
		default_theme_config();
	else
		{
		parse_gkrellmrc(0);
		if (GK.theme_alternative > GK.theme_n_alternatives)
			GK.theme_alternative = GK.theme_n_alternatives;
		if (GK.theme_alternative > 0)
			parse_gkrellmrc(GK.theme_alternative);
		}
	cleanup_gkrellmrc();
	set_theme_alternatives_label();
	}


/* --------------------------------------------------------------*/

struct	_config	user_config[] =
	{
	{"enable_hostname",	&UC.enable_hostname,	NULL,				0  },
	{"hostname_short",	&UC.hostname_short,		NULL,				0  },

	{"track_gtk_rcfiles", &UC.track_gtk_rcfiles, NULL,				0  },
	{"large_info_font", &UC.large_info_font,	NULL,				0  },
	{"save_position",	&UC.save_position,		NULL,				0  },
	{"fixed_scale",		&UC.fixed_scale,		NULL,				0  },

	{"chart_width",		&UC.chart_width,			NULL,			0  },
	{"height_cpu",		&UC.chart_height[MON_CPU],	NULL,			0  },
	{"height_proc",		&UC.chart_height[MON_PROC],	NULL,			0  },
	{"height_disk",		&UC.chart_height[MON_DISK],	NULL,			0  },
	{"height_net",		&UC.chart_height[MON_NET],	NULL,			0  },
	{"height_inet",		&UC.chart_height[MON_INET],	NULL,			0  },

	{"ema_period",		&UC.ema_period,			NULL,				0  },
	{"update_HZ",		&UC.update_HZ,			NULL,				0  },
/*	{"bump_HZ",			&UC.bump_HZ,			NULL,				0  }, */

	{"load_bar_graph",	&UC.proc_load_bar_graph, NULL,				0  },
	{"clip_processes",	&UC.proc_clip_processes, NULL,				0  },

	};


void
read_user_config()
	{
	FILE	*f;
	gchar	*config;
	gchar	buf[CFG_BUFSIZE];


	user_config_defaults();
	config = gkrellm_make_config_file_name(gkrellm_homedir(),
					GKRELLM_USER_CONFIG);
	f = fopen(config, "r");
	g_free(config);
	if (f)
		{
		while (fgets(buf, sizeof(buf), f))
			parse_config_line(buf, &user_config[0],
						sizeof(user_config) / sizeof (struct _config));
		fclose(f);
		}
	if (UC.chart_width < CHART_WIDTH_MIN)
		UC.chart_width = CHART_WIDTH_MIN;
	if (UC.chart_width > CHART_WIDTH_MAX)
		UC.chart_width = CHART_WIDTH_MAX;
	}

void
gkrellm_config_modified()
	{
	if (GK.no_config)
		return;
	GK.config_modified = TRUE;
	}


gchar *
gkrellm_make_config_file_name(gchar *dir, gchar *config)
	{
	gchar	hostname[256],
			*fname,
			*s		= NULL;

	if (GK.config_suffix || GK.found_host_config || GK.force_host_config)
		{
		if (GK.config_suffix)
			s = GK.config_suffix;
		else if (!gethostname(hostname, 256))
			s = hostname;
		if (s)
			s = g_strdup_printf("%s-%s", config, s);
		}
	if (!s)
		s = g_strdup(config);

	if (dir)
		{
		fname = g_strdup_printf("%s/%s", dir, s);
		g_free(s);
		}
	else
		fname = s;
	return fname;
	}

void
save_user_config()
	{
	FILE	*f;
	GList	*list;
	Monitor	*mon;
	gint	i;
	mode_t	mode;
	gchar	*config;

	if (GK.demo)
		return;
	config = gkrellm_make_config_file_name(gkrellm_homedir(),
								GKRELLM_USER_CONFIG);
	f = fopen(config, "w");
	g_free(config);
	if (f == NULL)
		{
		printf(_("Cannot open config file %s for writing.\n"), config);
		return;
		}
	fprintf(f, "### GKrellM user config.  Auto written, do not edit ###\n");
	fprintf(f, "### Version %d.%d.%d ###\n",
			GKRELLM_VERSION_MAJOR, GKRELLM_VERSION_MINOR, GKRELLM_VERSION_REV);
	for (i = 0; i < sizeof(user_config) / sizeof(struct _config); ++i)
		{
		if (user_config[i].value)
			fprintf(f, "%s %d\n", user_config[i].option,
							*(user_config[i].value));
		else if (user_config[i].arg)	/* Put quotes around strings */
			fprintf(f, "%s \"%s\"\n",user_config[i].option,
						*(user_config[i].arg));
		}
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (mon->save_user_config && mon->private->enabled)
			(*(mon->save_user_config))(f);
		}
#if defined (S_IRUSR)
	mode = (S_IRUSR | S_IWUSR);
#elif defined (S_IREAD)
	mode = (S_IREAD | S_IWRITE);
#else
	mode = 0600;
#endif
	fchmod(fileno(f), mode);
	fclose(f);

	GK.config_modified = 0;
	}

