/* ==========================================================================
 * mod_auth_bsd - Apache module for login_bsd BSD Authentication wrapper
 * --------------------------------------------------------------------------
 * Copyright (c) 2003, 2005  William Ahern
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to permit
 * persons to whom the Software is furnished to do so, subject to the
 * following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
 * NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 * ==========================================================================
 *
 * HMAC Implementation
 *
 * http://www.ietf.org/rfc/rfc2104.txt
 */

#if !defined USE_SHA1 && !defined USE_MD5
#define	USE_MD5
#endif

#define SHA1_DIGEST_SIZE	20
#define SHA1_BLOCK_SIZE		64

#define MD5_DIGEST_SIZE		16
#define MD5_BLOCK_SIZE		64

#ifdef USE_SHA1
#define HMAC_DIGEST_SIZE	SHA1_DIGEST_SIZE
#define HMAC_BLOCK_SIZE		SHA1_BLOCK_SIZE
#else
#define HMAC_DIGEST_SIZE	MD5_DIGEST_SIZE
#define HMAC_BLOCK_SIZE		MD5_BLOCK_SIZE
#endif


#define HMAC_IPAD	0x36
#define HMAC_OPAD	0x5C


typedef struct {
	#ifdef USE_SHA1
	AP_SHA1_CTX hash;
	#else
	AP_MD5_CTX hash;
	#endif

	void (*hash_init)(void *);
	void (*hash_update)(void *,const char *,unsigned int);
	void (*hash_final)(unsigned char[HMAC_DIGEST_SIZE],void *);

	unsigned char key[HMAC_BLOCK_SIZE];
	unsigned char ipad[HMAC_BLOCK_SIZE];
	unsigned char opad[HMAC_BLOCK_SIZE];
} hmac_ctx;


static void hmac_init(hmac_ctx *ctx, unsigned char *key, int keysz) {
	int i;

	#ifdef USE_SHA1
		ctx->hash_init
			= (void(*)(void *))ap_SHA1Init;
		ctx->hash_update
			= (void(*)(void *,const char *,unsigned int))ap_SHA1Update;
		ctx->hash_final
			= (void(*)(unsigned char[HMAC_DIGEST_SIZE],void *))ap_SHA1Final;
	#else
		ctx->hash_init
			= (void(*)(void *))ap_MD5Init;
		ctx->hash_update
			= (void(*)(void *,const char *,unsigned int))ap_MD5Update;
		ctx->hash_final
			= (void(*)(unsigned char[HMAC_DIGEST_SIZE],void *))ap_MD5Final;
	#endif

	bzero(ctx->key,sizeof(ctx->key));

	if (keysz > HMAC_BLOCK_SIZE) {
		ctx->hash_init(&ctx->hash);
		ctx->hash_update(&ctx->hash,key,keysz);
		ctx->hash_final(ctx->key,&ctx->hash);
	} else {
		memcpy(ctx->key,key,keysz);
	}

	assert(sizeof(ctx->key) == sizeof(ctx->ipad));

	for (i = 0; i < sizeof(ctx->ipad); i++)
		ctx->ipad[i]	= HMAC_IPAD ^ ctx->key[i];

	ctx->hash_init(&ctx->hash);
	ctx->hash_update(&ctx->hash,ctx->ipad,sizeof(ctx->ipad));

	return; /* void */
} /* hmac_init */


static void hmac_update(hmac_ctx *ctx, unsigned char *buf, int bufsz) {
	ctx->hash_update(&ctx->hash,buf,bufsz);

	return; /* void */
} /* hmac_update */


static void hmac_final(hmac_ctx *ctx, unsigned char digest[HMAC_DIGEST_SIZE]) {
	int i;

	ctx->hash_final(digest,&ctx->hash);

	assert(sizeof(ctx->opad) == sizeof(ctx->key));

	for (i = 0; i < sizeof(ctx->opad); i++)
		ctx->opad[i]	= HMAC_OPAD ^ ctx->key[i];

	ctx->hash_init(&ctx->hash);
	ctx->hash_update(&ctx->hash,ctx->opad,sizeof(ctx->opad));
	ctx->hash_update(&ctx->hash,digest,sizeof(digest));

	ctx->hash_final(digest,&ctx->hash);

	return; /* void */
} /* hmac_final */


static void hmac(unsigned char digest[HMAC_DIGEST_SIZE], unsigned char *key, int keysz, unsigned char *plain, int plainsz) {
	hmac_ctx ctx;

	hmac_init(&ctx,key,keysz);
	hmac_update(&ctx,plain,plainsz);
	hmac_final(&ctx,digest);

	return; /* void */
} /* hmac */

