/* ==========================================================================
 * mod_auth_bsd - Apache module for login_bsd BSD Authentication wrapper
 * --------------------------------------------------------------------------
 * Copyright (c) 2006  William Ahern
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to permit
 * persons to whom the Software is furnished to do so, subject to the
 * following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
 * NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 * ==========================================================================
 */
#include <stdlib.h>	/* malloc(3) free(3) mkstemp(3) setproctitle(3) */
#include <stddef.h>	/* offsetof() */
#include <stdbool.h>

#include <limits.h>	/* LOGIN_NAME_MAX NGROUPS_MAX */

#include <time.h>	/* time(2) */

#include <string.h>	/* strerror(3) memcpy(3) memset(3) memcpy(3) strlcpy(3) */

#include <errno.h>	/* EINTR EAGAIN ERANGE */

#include <assert.h>

#include <sys/types.h>	/* pid_t mode_t uid_t gid_t -socketpair(2) */
#include <sys/stat.h>	/* umask(2) */
#include <sys/time.h>	/* time_t */
#include <sys/queue.h>
#include <sys/tree.h>
#include <sys/socket.h>	/* AF_UNIX SOCK_STREAM SOL_SOCKET SCM_RIGHTS
			 * CMSG_DATA CMSG_SPACE CMSG_LEN CMSG_FIRSTHDR 
			 * struct msghdr struct cmsghdr socketpair(2)
			 * recvmsg(2) sendmsg(2) */
#include <sys/uio.h>	/* struct iovec */

#include <unistd.h>	/* F_LOCK pipe(2) close(2) fork(2) unlink(2) lockf(2)
			 * setresuid(3) setresgid(2) getgrouplist(3) */

#include <fcntl.h>	/* open(2) O_RDONLY */

#include <signal.h>	/* SIGPIPE SIGTERM SIGHUP SIG_BLOCK sigset_t
			 * sigemptyset(3) sigprocmask(2) */

#include <pwd.h>	/* _PASSWORD_LEN getpwnam(3) */
#include <grp.h>	/* getgrnam(3) getgrgid_r(3) */

#include <login_cap.h>
#include <bsd_auth.h>

#include <event.h>

#include "http_log.h"	/* ap_log_error() */
#include "http_config.h"


/* Last time I checked OpenBSD set it to 128. Of course, that doesn't mean
 * it's an accurate value for the maximum allowed length, but it's all we
 * have.
 */
#ifndef _PASSWORD_LEN
#define _PASSWORD_LEN	128
#endif


/*
 * Is there no LOGIN_NAME_MAX equivalent for groups names?
 */
#ifndef GROUP_NAME_MAX
#define GROUP_NAME_MAX	LOGIN_NAME_MAX
#endif


/*
 * OpenBSD doesn't have _SC_GETGR_R_SIZE_MAX yet. Nor _SC_GETPW_R_SIZE_MAX.
 * Nor getpwuid_r(3) for that matter. However, lib/libc/gen/getgrent.c
 * defines GETGR_R_SIZE_MAX, so we'll copy it.
 *
 * The logic is sound. Lines in /etc/group are limited to 1024 characters,
 * and groups are limited to 200 members.
 */
#ifndef _SC_GETGR_R_SIZE_MAX
#define _SC_GETGR_R_SIZE_MAX (1024+200*sizeof(char *))
#endif


enum auth_service {
	AUTH_SERVICE_LOGIN	= 0,
	AUTH_SERVICE_CHALLENGE	= 1,
	AUTH_SERVICE_RESPONSE	= 2,
}; /* enum auth_service */


enum auth_userinfo {
	AUTH_USERINFO_GROUPLIST	= 1,
}; /* enum auth_userinfo */


struct auth_service_login {
	char user[LOGIN_NAME_MAX + 1];
	char pass[_PASSWORD_LEN];
	char style[32];
	char type[32];
}; /* struct auth_service_login */


/* Add 1 to NGROUPS_MAX so we can include the user's default group. */
struct auth_userinfo_grouplist {
	size_t ngroups;
	char groups[NGROUPS_MAX + 1][GROUP_NAME_MAX + 1];
}; /* struct auth_userinfo_grouplist */


/*
 * Return two values: 1) Offset into the packet and 2) length of the data to
 * hash.
 */
#define AUTH_PACKET_HMAC_VEC(pkt) 					\
	(((unsigned char *)(pkt)) + offsetof(__typeof__(*(pkt)),type)),	\
	(sizeof *(pkt) - offsetof(__typeof__(*(pkt)),type))


struct auth_packet {
	unsigned char hmac[HMAC_DIGEST_SIZE];

	enum auth_service type;

	union {
		struct auth_service_login login;
	} service;

	int auth_state;	/* BSD Auth state */
	int sys_errno;

	enum auth_userinfo infotype;

	union {
		struct auth_userinfo_grouplist grouplist;
	} userinfo;
}; /* struct auth_packet */


struct auth_attempt {
	unsigned char key[HMAC_DIGEST_SIZE];

	time_t timestamp;

	union {
		struct {
			unsigned long int count;
		} failed;

		struct {
			enum auth_userinfo infotype;

			union {
				struct auth_userinfo_grouplist grouplist;
			} userinfo;
		} okayed;
	} result;

	SPLAY_ENTRY(auth_attempt) spe;
	CIRCLEQ_ENTRY(auth_attempt) cqe;
}; /* struct auth_attempt */


static const struct auth_peer {
	struct event ev;

	unsigned char secret[HMAC_BLOCK_SIZE];

	struct auth_packet pkt;

	char *buf;
	size_t buflen;
} auth_peer_initializer;


/*
 * Cache and lookup tree for failed authentications.
 */
struct auth_attempt;

static SPLAY_HEAD(mod_auth_bsd_fail,auth_attempt) auth_failed_lookup;
static CIRCLEQ_HEAD(,auth_attempt) auth_failed_cache;
static size_t auth_failed_num;

static int auth_fail_cmp(struct auth_attempt *a, struct auth_attempt *b) {
	return memcmp(a->key,b->key,sizeof a->key);
} /* auth_fail_cmp() */

SPLAY_PROTOTYPE(mod_auth_bsd_fail,auth_attempt,spe,auth_fail_cmp)
SPLAY_GENERATE(mod_auth_bsd_fail,auth_attempt,spe,auth_fail_cmp)



/*
 * Cache and lookup tree for successful authentications.
 */
static SPLAY_HEAD(mod_auth_bsd_okay,auth_attempt) auth_okayed_lookup;
static CIRCLEQ_HEAD(,auth_attempt) auth_okayed_cache;
static size_t auth_okayed_num;

static int auth_okay_cmp(struct auth_attempt *a, struct auth_attempt *b) {
	return memcmp(a->key,b->key,sizeof a->key);
} /* auth_okay_cmp() */

SPLAY_PROTOTYPE(mod_auth_bsd_okay,auth_attempt,spe,auth_okay_cmp)
SPLAY_GENERATE(mod_auth_bsd_okay,auth_attempt,spe,auth_okay_cmp)


/*
 * The proto socket pair is used by the child to request and receive an
 * authentication socket from the daemon.
 */
#define SOCKET_RECV	0
#define SOCKET_SEND	1

static int proto_socket[2]	= { -1, -1 };	/* Transmit auth_socket */
static int proto_lockfd		= -1;		/* Serialize access */

static int auth_socket		= -1;		/* For child only */
static unsigned char auth_secret[HMAC_BLOCK_SIZE];
static struct auth_packet auth_packet;

static int secret_fd		= -1;		/* For daemon only */

static unsigned char hash_secret[HMAC_BLOCK_SIZE];

static server_rec *http_server	= 0;

static const char *auth_user	= "www";
static const char *auth_group	= "www";

static unsigned int fail_delay		= 3;
static unsigned int fail_tolerance	= 3;

static unsigned int cache_size		= 1024;
static unsigned int cache_ttl		= 600;	/* 10 minutes */


static inline char *strtonull(const char *str) {
	return (str && *str == '\0')? (char *)0 : (char *)str;
} /* strtonull() */


static inline char *nulltostr(const char *str) {
	return (str == (char *)0)? "" : (char *)str;
} /* nulltostr() */


/* Returns the last grouplist we got */
static int auth_child_getgrouplist(request_rec *r,const char *user, char *groups[], int *ngroups) {
	struct auth_packet *pkt	= &auth_packet;
	size_t userlen		= strlen(user);
	int i;

	if (pkt->type != AUTH_SERVICE_LOGIN
	||  pkt->infotype != AUTH_USERINFO_GROUPLIST
	||  !(pkt->auth_state & (AUTH_OKAY|AUTH_ROOTOKAY)))
	{
		ap_log_rerror(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,r,"[AuthBSD] Stored auth packet doesn't look like our last request");

		*ngroups	= 0;

		return -1;
	}

	assert(pkt->service.login.user[sizeof pkt->service.login.user - 1] == '\0');

	if (0 != strcmp(user,pkt->service.login.user)) {
		ap_log_rerror(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,r,"[AuthBSD] Stored auth packet has different user");

		return -1;
	}

	for (i = 0; i < pkt->userinfo.grouplist.ngroups; i++) {
		assert(pkt->userinfo.grouplist.groups[i][sizeof pkt->userinfo.grouplist.groups[i] - 1] == '\0');

		if (*ngroups > i)
			groups[i]	= pkt->userinfo.grouplist.groups[i];
	}

	if (*ngroups >= pkt->userinfo.grouplist.ngroups) {
		*ngroups	= pkt->userinfo.grouplist.ngroups;

		return *ngroups;
	} else {
		*ngroups	= pkt->userinfo.grouplist.ngroups;

		return -1;	
	}
} /* auth_child_getgrouplist() */


/*
 * Construct and send authentication packet to the daemon, then receive and
 * parse the answer.
 */
static int auth_child_userokay(request_rec *r, const char *user, const char *style, const char *type, const char *pass) {
	struct auth_packet p;
	unsigned char digest[sizeof p.hmac];
	size_t len, n;
	unsigned char *bytes;
	size_t nbytes;

	/* Reset our last packet. We'll restore if/when we return success. */
	(void)memset(&auth_packet,'\0',sizeof auth_packet);

	user	= nulltostr(user);
	pass	= nulltostr(pass);
	style	= nulltostr(style);
	type	= nulltostr(type);

	(void)memset(&p,'\0',sizeof p);

	p.type	= AUTH_SERVICE_LOGIN;

	len	= strlcpy(p.service.login.user,user,sizeof p.service.login.user);

	if (len >= sizeof p.service.login.user) {
		ap_log_rerror(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,r,"[AuthBSD] Username too long: %lu",(unsigned long)len);

		return -1;
	}

	len	= strlcpy(p.service.login.pass,pass,sizeof p.service.login.pass);

	if (len >= sizeof p.service.login.pass) {
		ap_log_rerror(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,r,"[AuthBSD] Password too long: %lu",(unsigned long)len);

		return -1;
	}

	len	= strlcpy(p.service.login.style,style,sizeof p.service.login.style);

	if (len >= sizeof p.service.login.style) {
		ap_log_rerror(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,r,"[AuthBSD] Style too long: %lu",(unsigned long)len);

		return -1;
	}

	len	= strlcpy(p.service.login.type,type,sizeof p.service.login.type);

	if (len >= sizeof p.service.login.type) {
		ap_log_rerror(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,r,"[AuthBSD] Type too long: %lu",(unsigned long)len);

		return -1;
	}

	/* Request group info */
	p.infotype	= AUTH_USERINFO_GROUPLIST;

	hmac(p.hmac,auth_secret,sizeof auth_secret,AUTH_PACKET_HMAC_VEC(&p));

	bytes	= (unsigned char *)&p;
	nbytes	= sizeof p;

	ap_log_rerror(APLOG_MARK,APLOG_DEBUG|APLOG_NOERRNO,r,"[AuthBSD] Peer writing request");

	do {
		n	= write(auth_socket,bytes,nbytes);

		if (n > 0) {
			bytes	+= n;
			nbytes	-= n;
		}
	} while (nbytes && (n > 0 || errno == EINTR));

	(void)memset(p.service.login.pass,'\0',sizeof p.service.login.pass);

	if (nbytes) {
		ap_log_rerror(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,r,"[AuthBSD] Peer failed to send credentials: %s",strerror(errno));

		(void)close(auth_socket);
		auth_socket	= -1;

		return -1;
	} else
		ap_log_rerror(APLOG_MARK,APLOG_DEBUG|APLOG_NOERRNO,r,"[AuthBSD] Peer wrote request");

	bytes	= (unsigned char *)&p;
	nbytes	= sizeof p;

	do {
		n	= read(auth_socket,bytes,nbytes);

		if (n > 0) {
			bytes	+= n;
			nbytes	-= n;
		}
	} while (nbytes && (n > 0 || errno == EINTR));

	if (nbytes) {
		ap_log_rerror(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,r,"[AuthBSD] Peer failed to receive answer: %s",strerror(errno));

		(void)close(auth_socket);
		auth_socket	= -1;

		return -1;
	} else
		ap_log_rerror(APLOG_MARK,APLOG_DEBUG|APLOG_NOERRNO,r,"[AuthBSD] Peer read answer");


	hmac(digest,auth_secret,sizeof auth_secret,AUTH_PACKET_HMAC_VEC(&p));

	if (0 != memcmp(digest,p.hmac,sizeof digest)) {
		ap_log_rerror(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,r,"[AuthBSD] Peer received invalid packet");

		errno	= EINVAL;

		return -1;
	}

	if (p.sys_errno) {
		ap_log_rerror(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,r,"[AuthBSD] Peer received errno reply: %s",strerror(p.sys_errno));

		return -1;
	} else if (p.auth_state & (AUTH_OKAY|AUTH_ROOTOKAY)) {
		(void)memcpy(&auth_packet,&p,sizeof auth_packet);

		return 1;
	} else
		return 0;
} /* auth_child_userokay() */


/*
 * Request our own authentication socket.
 */
static void auth_child_init(server_rec *s) {
	struct msghdr msg;
	struct iovec vec[1];
	union { struct cmsghdr cm; unsigned char cmsgbuf[CMSG_SPACE(sizeof auth_socket)]; } cm;
	ssize_t n;

	if (0 != lockf(proto_lockfd,F_LOCK,0)) {
		ap_log_error(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,s,"[AuthBSD] Failed to acquire shared auth lock (%d): %s",proto_lockfd,strerror(errno));

		goto done;
	}

	do {
		n	= write(proto_socket[SOCKET_RECV],"a",1);
	} while(n == -1 && errno == EINTR);

	if (n == -1) {
		ap_log_error(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,s,"[AuthBSD] Failed to request auth socket: %s",strerror(errno));

		goto done;
	}

	msg.msg_control		= cm.cmsgbuf;
	msg.msg_controllen	= sizeof cm.cmsgbuf;

	msg.msg_name		= NULL;
	msg.msg_namelen		= 0;

	vec[0].iov_base		= auth_secret;
	vec[0].iov_len		= sizeof auth_secret;

	msg.msg_iov		= vec;
	msg.msg_iovlen		= sizeof vec / sizeof *vec;

	do {
		n	= recvmsg(proto_socket[SOCKET_RECV],&msg,0);
	} while (n == -1 && errno == EINTR);

	if (n == -1) {
		ap_log_error(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,s,"[AuthBSD] Peer failed to received auth socket: %s",strerror(errno));

		goto done;
	} else if ((msg.msg_flags & MSG_TRUNC) || (msg.msg_flags & MSG_CTRUNC)) {
		ap_log_error(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,s,"[AuthBSD] Peer received truncated control message");

		goto done;
	} else
		ap_log_error(APLOG_MARK,APLOG_DEBUG|APLOG_NOERRNO,s,"[AuthBSD] Peer read %d bytes from auth daemon",n);

	if (CMSG_FIRSTHDR(&msg) == NULL
	||  CMSG_FIRSTHDR(&msg)->cmsg_len != CMSG_LEN(sizeof auth_socket)
	||  CMSG_FIRSTHDR(&msg)->cmsg_level != SOL_SOCKET
	||  CMSG_FIRSTHDR(&msg)->cmsg_type != SCM_RIGHTS
	||  msg.msg_iovlen != sizeof vec / sizeof *vec
	||  n != sizeof auth_secret) {
		ap_log_error(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,s,"[AuthBSD] Peer received invalid data");

		goto done;
	}

	auth_socket	= *((int *)CMSG_DATA(CMSG_FIRSTHDR(&msg)));

	ap_log_error(APLOG_MARK,APLOG_DEBUG|APLOG_NOERRNO,s,"[AuthBSD] Peer received auth socket and shared secret");

done:
	(void)close(proto_socket[SOCKET_RECV]);
	proto_socket[SOCKET_RECV]	= -1;

	(void)close(proto_socket[SOCKET_SEND]);
	proto_socket[SOCKET_SEND]	= -1;

	/* Also releases any lock. */
	(void)close(proto_lockfd);
	proto_lockfd	= -1;

	return /* void */;
} /* auth_child_init() */


static void authd_hash_fail(unsigned char hash[HMAC_DIGEST_SIZE], struct auth_packet *pkt) {
	hmac_ctx ctx;

	assert(pkt->type == AUTH_SERVICE_LOGIN);

	hmac_init(&ctx,hash_secret,sizeof hash_secret);

	hmac_update(&ctx,pkt->service.login.user,sizeof pkt->service.login.user);
	hmac_update(&ctx,pkt->service.login.style,sizeof pkt->service.login.style);
	hmac_update(&ctx,pkt->service.login.type,sizeof pkt->service.login.type);

	hmac_final(&ctx,hash);
} /* authd_hash_fail() */


static void authd_hash_okay(unsigned char hash[HMAC_DIGEST_SIZE], struct auth_packet *pkt) {
	hmac_ctx ctx;

	assert(pkt->type == AUTH_SERVICE_LOGIN);

	hmac_init(&ctx,hash_secret,sizeof hash_secret);

	hmac_update(&ctx,pkt->service.login.user,sizeof pkt->service.login.user);
	hmac_update(&ctx,pkt->service.login.pass,sizeof pkt->service.login.pass);
	hmac_update(&ctx,pkt->service.login.style,sizeof pkt->service.login.style);
	hmac_update(&ctx,pkt->service.login.type,sizeof pkt->service.login.type);

	hmac_final(&ctx,hash);
} /* authd_hash_okay() */


static struct auth_attempt *authd_userokay_cached(struct auth_packet *pkt) {
	struct auth_attempt *a	= 0;
	struct auth_attempt k;

	authd_hash_okay(k.key,pkt);

	if ((a = SPLAY_FIND(mod_auth_bsd_okay,&auth_okayed_lookup,&k))) {
		time_t now	= time(NULL);

		/*
		 * If the timestamp expired don't unlink and call free(3). 
		 * Just setup it up for reallocation.
		 */
		if (now - a->timestamp > cache_ttl) {
			CIRCLEQ_REMOVE(&auth_okayed_cache,a,cqe);
			CIRCLEQ_INSERT_HEAD(&auth_okayed_cache,a,cqe);

			a	= NULL;
		} else {
			a->timestamp	= now;

			CIRCLEQ_REMOVE(&auth_okayed_cache,a,cqe);
			CIRCLEQ_INSERT_TAIL(&auth_okayed_cache,a,cqe);
		}
	}

	return a;
} /* authd_userokay_cached() */


static struct auth_attempt *authd_userokay_cache(struct auth_packet *pkt, int okay) {
	struct auth_attempt k, *a, *e;
	time_t now;

	if (cache_size == 0)
		return NULL;

	now	= time(NULL);

	if (okay) {
		if ((a = CIRCLEQ_FIRST(&auth_okayed_cache)) != CIRCLEQ_END(&auth_okayed_cache)
		&&  (now - a->timestamp > cache_ttl || auth_okayed_num >= cache_size)) {
			assert(SPLAY_REMOVE(mod_auth_bsd_okay,&auth_okayed_lookup,a));

			CIRCLEQ_REMOVE(&auth_okayed_cache,a,cqe);
			auth_okayed_num--;
		} else if (!(a = malloc(sizeof *a)))
			return NULL;

		ap_log_error(APLOG_MARK,APLOG_DEBUG|APLOG_NOERRNO,http_server,"[AuthBSD] Inserting user into okay cache (%lu existing)",auth_okayed_num);

		authd_hash_okay(a->key,pkt);
		a->timestamp	= now;

		if ((e = SPLAY_INSERT(mod_auth_bsd_okay,&auth_okayed_lookup,a))) {
			e->timestamp	= now;

			CIRCLEQ_REMOVE(&auth_okayed_cache,e,cqe);
			auth_okayed_num--;

			free(a);

			a	= e;			
		}

		CIRCLEQ_INSERT_TAIL(&auth_okayed_cache,a,cqe);
		auth_okayed_num++;
	} else {
		authd_hash_fail(k.key,pkt);

		if ((a = SPLAY_FIND(mod_auth_bsd_fail,&auth_failed_lookup,&k))) {
			CIRCLEQ_REMOVE(&auth_failed_cache,a,cqe);
			auth_failed_num--;

			if (now - a->timestamp > cache_ttl)
				a->result.failed.count	= 0;
		} else {
			if ((a = CIRCLEQ_FIRST(&auth_failed_cache)) != CIRCLEQ_END(&auth_failed_cache)
		        &&  (now - a->timestamp > cache_ttl || auth_failed_num >= cache_size)) {
				assert(SPLAY_REMOVE(mod_auth_bsd_fail,&auth_failed_lookup,a));

				CIRCLEQ_REMOVE(&auth_failed_cache,a,cqe);
				auth_failed_num--;
			} else if (!(a = malloc(sizeof *a)))
				return NULL;

			(void)memcpy(a->key,k.key,sizeof a->key);
			a->result.failed.count	= 0;

			assert(NULL == SPLAY_INSERT(mod_auth_bsd_fail,&auth_failed_lookup,a));
		}

		ap_log_error(APLOG_MARK,APLOG_DEBUG|APLOG_NOERRNO,http_server,"[AuthBSD] Inserting user into fail cache (%lu existing)",auth_failed_num);

		a->timestamp	= now;
		a->result.failed.count++;

		CIRCLEQ_INSERT_TAIL(&auth_failed_cache,a,cqe);
		auth_failed_num++;
	}

	return a;
} /* authd_userokay_cache() */


static bool authd_packet_okay(struct auth_peer *p, struct auth_packet *pkt) {
	unsigned char digest[sizeof pkt->hmac];

	hmac(digest,p->secret,sizeof p->secret,AUTH_PACKET_HMAC_VEC(pkt));

	if (0 != memcmp(digest,pkt->hmac,sizeof digest))
		return false;

	/* Is it clean? */
	if (pkt->type == AUTH_SERVICE_LOGIN) {
		if (pkt->service.login.user[sizeof pkt->service.login.user - 1]	  != '\0'
		||  pkt->service.login.pass[sizeof pkt->service.login.pass - 1]	  != '\0'
		||  pkt->service.login.style[sizeof pkt->service.login.style - 1] != '\0'
		||  pkt->service.login.type[sizeof pkt->service.login.type - 1]	  != '\0')
		{
			return false;
		}
	} else
		return false;

	return true;
} /* authd_packet_okay() */


static int authd_userinfo_getgrouplist(struct auth_peer *p, const char *user, struct auth_packet *pkt) {
	gid_t groups[sizeof pkt->userinfo.grouplist.groups / sizeof *pkt->userinfo.grouplist.groups];
	int ngroups	= sizeof groups / sizeof *groups;
	char gr_mem[_SC_GETGR_R_SIZE_MAX + 8192];
	struct passwd *pw;
	struct group gr_buf, *gr;
	int i, gr_errno;

	/* No getpwnam_r(3) in OpenBSD yet? */
	if (!(pw = getpwnam(user))) {
		assert(0);

		goto sysfail;
	}

	/*
	 * Should we bail when our list is too long? Will we always get a
	 * valid value? What if /etc/group is gone? How do we detect errors?
	 */
	if (-1 == getgrouplist(user,pw->pw_gid,groups,&ngroups)) {
		ap_log_error(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,http_server,"[AuthBSD] getgrouplist(user=%s,gid=%lu,ngroups=%lu -> %d)",user,(unsigned long)pw->pw_gid,(unsigned long)sizeof groups / sizeof *groups,ngroups);

		goto sysfail;
	}

	(void)memset(p->pkt.userinfo.grouplist.groups,'\0',sizeof p->pkt.userinfo.grouplist.groups);
	p->pkt.userinfo.grouplist.ngroups	= 0;

	for (i = 0; i < ngroups; i++) {
		if (0 != (gr_errno = getgrgid_r(groups[i],&gr_buf,gr_mem,sizeof gr_mem,&gr))) {
			ap_log_error(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,http_server,"[AuthBSD] getgrgid_r(gid=%lu,sizeof gr_mem=%lu): %s",(unsigned long)groups[i],(unsigned long)sizeof gr_mem,strerror(gr_errno));

			errno	= gr_errno;

			goto sysfail;
		}

		if (sizeof p->pkt.userinfo.grouplist.groups[i] <= strlcpy(p->pkt.userinfo.grouplist.groups[i],gr->gr_name,sizeof p->pkt.userinfo.grouplist.groups[i])) {
			ap_log_error(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,http_server,"[AuthBSD] Group name too long: %s",gr->gr_name);

			/* ERANGE documented by SUSv3 manual for getgruid_r(3)
			 * as relaying "Insufficient storage" type error. */
			errno	= ERANGE;	/* See SUSv3 man of getgruid_r(3) */

			goto sysfail;
		}
	}

	p->pkt.userinfo.grouplist.ngroups	= ngroups;

	return 0;
sysfail:
	gr_errno	= errno;

	(void)memset(p->pkt.userinfo.grouplist.groups,'\0',sizeof p->pkt.userinfo.grouplist.groups);
	p->pkt.userinfo.grouplist.ngroups	= 0;

	errno		= gr_errno;

	return -1;
} /* authd_userinfo_getgrouplist() */


static void authd_catch_packet(int fd, short event, void *arg) {
	struct auth_peer *p		= arg;
	struct auth_attempt *cached	= 0;
	int cached_info			= 0;
	unsigned char digest[sizeof p->pkt.hmac];
	int n;
	char *user, *style, *type, *pass;

	switch (event) {
	case EV_READ:
		if (!p->buf) {
			p->buf		= (char *)&p->pkt;
			p->buflen	= 0;
		}

		do {
			n	= read(fd,p->buf,sizeof p->pkt - p->buflen);

			if (n > 0) {
				p->buf		+= n;
				p->buflen	+= n;
			}
		} while (p->buflen < sizeof p->pkt && n != 0 && (n > 0 || errno == EINTR));

		if (n == 0) {
			ap_log_error(APLOG_MARK,APLOG_DEBUG|APLOG_NOERRNO,http_server,"[AuthBSD] Auth daemon detected demise of peer");

			goto hangup;
		} else if (p->buflen != sizeof p->pkt) {
			if (errno == EAGAIN)
				goto needmore;
		}

		p->buf		= NULL;
		p->buflen	= 0;


		if (!authd_packet_okay(p,&p->pkt)) {
			ap_log_error(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,http_server,"[AuthBSD] Auth daemon received invalid packet");

			errno	= EINVAL;

			goto sysfail;
		}

		if ((cached = authd_userokay_cached(&p->pkt))) {
			ap_log_error(APLOG_MARK,APLOG_DEBUG|APLOG_NOERRNO,http_server,"[AuthBSD] Auth cache hit for user %s",p->pkt.service.login.user);

			p->pkt.auth_state	= AUTH_OKAY;

			if (cached_info = (cached->result.okayed.infotype == p->pkt.infotype)) {
				(void)memcpy(&p->pkt.userinfo,&cached->result.okayed.userinfo,sizeof p->pkt.userinfo);
			}
		} else {
			ap_log_error(APLOG_MARK,APLOG_DEBUG|APLOG_NOERRNO,http_server,"[AuthBSD] Auth cache miss for user %s",p->pkt.service.login.user);

			user	= strtonull(p->pkt.service.login.user);
			style	= strtonull(p->pkt.service.login.style);
			type	= strtonull(p->pkt.service.login.type);
			pass	= strtonull(p->pkt.service.login.pass);

			p->pkt.auth_state	= auth_userokay(user,style,type,pass)? AUTH_OKAY : 0;
		}

		/* Some day might we only want userinfo w/o authenticating? */
		if (p->pkt.auth_state == AUTH_OKAY && !cached_info) {
			switch (p->pkt.infotype) {
			case AUTH_USERINFO_GROUPLIST:
				if (0 != authd_userinfo_getgrouplist(p,p->pkt.service.login.user,&p->pkt)) {
					ap_log_error(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,http_server,"[AuthBSD] Failed to get group list: %s",strerror(errno));

					goto sysfail;
				}

				break;
			case 0:
				break;
			default:
				ap_log_error(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,http_server,"[AuthBSD] Unknown info request type: %d",p->pkt.infotype);

				errno	= EINVAL;

				goto sysfail;
			}
		}

		if (!cached) {
			struct auth_attempt *a;

			ap_log_error(APLOG_MARK,APLOG_DEBUG|APLOG_NOERRNO,http_server,"[AuthBSD] Caching %s user %s",(p->pkt.auth_state)? "success" : "failure",p->pkt.service.login.user);

			if ((a = authd_userokay_cache(&p->pkt,(p->pkt.auth_state == AUTH_OKAY)))
			&&  p->pkt.auth_state != AUTH_OKAY && a->result.failed.count > fail_tolerance
			&&  fail_delay > 0) {
				struct timeval tv	= { fail_delay, 0 }; 

				ap_log_error(APLOG_MARK,APLOG_DEBUG|APLOG_NOERRNO,http_server,"[AuthBSD] Setting timeout of %u seconds for user %s",(unsigned)fail_delay,p->pkt.service.login.user);

				event_set(&p->ev,fd,0,authd_catch_packet,p);

				if (0 != event_add(&p->ev,&tv))
					goto sysfail;

				return /* void */;
			}
		}

		/* FALL THROUGH */
	case EV_TIMEOUT:
		if (event == EV_TIMEOUT)
			ap_log_error(APLOG_MARK,APLOG_DEBUG|APLOG_NOERRNO,http_server,"[AuthBSD] Caught delay timeout for user %s",p->pkt.service.login.user);

		event	= EV_WRITE;

		/* FALL THROUGH */
	case EV_WRITE:
		if (!p->buf) {
			(void)memset(p->pkt.service.login.pass,'\0',sizeof p->pkt.service.login.pass);

			hmac(p->pkt.hmac,p->secret,sizeof p->secret,AUTH_PACKET_HMAC_VEC(&p->pkt));

			p->buf		= (char *)&p->pkt;
			p->buflen	= sizeof p->pkt;
		}

		do {
			n	= write(fd,p->buf,p->buflen);

			if (n > 0) {
				p->buf		+= n;
				p->buflen	-= n;
			}
		} while (p->buflen && (n > 0 || errno == EINTR));

		if (p->buflen) {
			if (errno == EAGAIN)
				goto needmore;

			goto sysfail;
		}

		p->buf		= NULL;
		p->buflen	= 0;

		event	= EV_READ;

		break;
	default:
		assert(0);
	} /* switch(event) */

	assert(event == EV_READ || event == EV_WRITE);

	/* CONTINUE */
needmore:
	event_set(&p->ev,fd,event,authd_catch_packet,p);

	if (0 != event_add(&p->ev,NULL))
		goto sysfail;

	return /* void */;
sysfail:
	ap_log_error(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,http_server,"[AuthBSD] Auth daemon lost link to peer",strerror(errno));

hangup:
	free(p), p = NULL;

	(void)close(fd), fd = -1;

	return /* void */;
} /* authd_catch_packet() */


static void authd_catch_sockreq(int fd, short event, void *arg) {
	struct auth_peer *p	= 0;
	int fds[2]		= { -1, -1 };
	int flags;
	unsigned char secret[sizeof p->secret];
	struct msghdr msg;
	struct iovec vec[1];
	union { struct cmsghdr cm; unsigned char cmsgbuf[CMSG_SPACE(sizeof fds[1])]; } cm;
	char buf[1];
	ssize_t n;

	do {
		n = read(fd,buf,sizeof buf);
	} while (n == -1 && errno == EINTR);

	if (n == 0) {
		ap_log_error(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,http_server,"[AuthBSD] Auth peers gone, shutting down.");

		exit(EXIT_SUCCESS);
	}

	n	= read(secret_fd,secret,sizeof secret);

	if (n < 0 || (size_t)n != sizeof secret)
		goto sysfail;

	if (0 != socketpair(AF_UNIX,SOCK_STREAM,0,fds))
		goto sysfail;

	if (-1 == (flags = fcntl(fds[0],F_GETFL,0)))
		goto sysfail;

	if (-1 == fcntl(fds[0],F_SETFL,flags|O_NONBLOCK))
		goto sysfail;

	if (!(p = malloc(sizeof *p)))
		goto sysfail;

	*p	= auth_peer_initializer;

	(void)memcpy(p->secret,secret,sizeof p->secret);

	event_set(&p->ev,fds[0],EV_READ,authd_catch_packet,p);

	if (0 != event_add(&p->ev,NULL)) {
		free(p), p = NULL;

		goto sysfail;
	}

	memset(&msg,'\0',sizeof msg);

	msg.msg_control		= cm.cmsgbuf;
	msg.msg_controllen	= CMSG_LEN(sizeof fds[1]);

	CMSG_FIRSTHDR(&msg)->cmsg_len			= CMSG_LEN(sizeof fds[1]);
	CMSG_FIRSTHDR(&msg)->cmsg_level			= SOL_SOCKET;
	CMSG_FIRSTHDR(&msg)->cmsg_type			= SCM_RIGHTS;
	*((int *)CMSG_DATA(CMSG_FIRSTHDR(&msg)))	= fds[1];

	msg.msg_name		= NULL;
	msg.msg_namelen		= 0;

	vec[0].iov_base		= p->secret;
	vec[0].iov_len		= sizeof p->secret;

	msg.msg_iov		= vec;
	msg.msg_iovlen		= sizeof vec / sizeof *vec;

	do {
		n	= sendmsg(fd,&msg,0);
	} while (n == -1 && errno == EINTR);

	if (n == -1)
		goto sysfail;

	(void)close(fds[1]);

	return /* void */;
sysfail:
	ap_log_error(APLOG_MARK,APLOG_NOERRNO|APLOG_ERR,http_server,"[AuthBSD] Auth daemon encountered fatal error: %s",strerror(errno));

	if (p) {
		event_del(&p->ev);

		free(p), p = NULL;
	}

	(void)close(fds[0]);
	(void)close(fds[1]);

	exit(EXIT_FAILURE);
} /* authd_catch_sockreq() */


static int authd_drop_privs(void) {
	struct passwd *pw;
	struct group *gr;

	if (!(pw = getpwnam(auth_user)))
		goto sysfail;

	if (!(gr = getgrnam(auth_group)))
		goto sysfail;

	if (0 != setresgid(gr->gr_gid,gr->gr_gid,gr->gr_gid))
		goto sysfail;

	if (0 != setresuid(pw->pw_uid,pw->pw_uid,pw->pw_uid))
		goto sysfail;

	ap_log_error(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,http_server,"[AuthBSD] Auth daemon changed user/group to %s/%s",auth_user,auth_group);

	return 0;
sysfail:
	ap_log_error(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,http_server,"[AuthBSD] Auth daemon failed to drop privileges to user/group %s/%s: %s",auth_user,auth_group,strerror(errno));

	return -1;	
} /* authd_drop_privs() */


static const char *authd_set_user(cmd_parms *cmd, void *dummy, char *arg) {
	auth_user	= arg;

	return NULL;
} /* authd_set_user() */


static const char *authd_set_group(cmd_parms *cmd, void *dummy, char *arg) {
	auth_group	= arg;

	return NULL;
} /* authd_set_group() */


static const char *authd_set_fail_delay(cmd_parms *cmd, void *dummy, char *arg) {
	fail_delay	= atoi(arg);	/* Ugly, but how do we send back errors? */

	return NULL;
} /* authd_set_fail_delay() */


static const char *authd_set_fail_tolerance(cmd_parms *cmd, void *dummy, char *arg) {
	fail_tolerance	= atoi(arg);	/* Ugly, but how do we send back errors? */

	return NULL;
} /* authd_set_fail_tolerance() */


static const char *authd_set_cache_size(cmd_parms *cmd, void *dummy, char *arg) {
	cache_size	= atoi(arg);	/* Ugly, but how do we send back errors? */

	return NULL;
} /* authd_set_cache_size() */


static const char *authd_set_cache_ttl(cmd_parms *cmd, void *dummy, char *arg) {
	cache_ttl	= atoi(arg);	/* Ugly, but how do we send back errors? */

	return NULL;
} /* authd_set_cache_ttl() */


static int authd_main(server_rec *s) {
/*	struct auth_config *cfg;*/
	sigset_t block;
	struct event ev_sockreq;

	http_server	= s;

	setproctitle("mod_auth_bsd");

/*	if (!(cfg = ap_get_module_config(http_server->module_config,&bsd_auth_module)))
		goto sysfail;*/

	if (0 != authd_drop_privs())
		goto sysfail;

	if (0 != chdir("/"))
		goto sysfail;

	if (-1 == (secret_fd = open(STRINGIFY(DEV_RANDOM),O_RDONLY)))
		goto sysfail;

	if ((int)sizeof hash_secret != read(secret_fd,hash_secret,sizeof hash_secret))
		goto sysfail;

	if (!event_init())
		goto sysfail;

	if (0 != sigemptyset(&block))
		goto sysfail;

	if (0 != sigaddset(&block,SIGPIPE))
		goto sysfail;

	if (0 != sigaddset(&block,SIGTERM))
		goto sysfail;

	if (0 != sigaddset(&block,SIGHUP))
		goto sysfail;

	if (0 != sigprocmask(SIG_BLOCK,&block,NULL))
		goto sysfail;

	event_set(&ev_sockreq,proto_socket[SOCKET_SEND],EV_READ|EV_PERSIST,authd_catch_sockreq,NULL);

	if (0 != event_add(&ev_sockreq,NULL))
		goto sysfail;

	(void)event_loop(0);

	/* Only successful exit done from authd_catch_sockreq() */
sysfail:
	return EXIT_FAILURE;
} /* authd_main() */


/*
 * Initialization that needs to occur in the master httpd process before we
 * begin to fork any children.
 */
static int authd_init(server_rec *s) {
	int errsav;
	mode_t masksav;
	char filenam[]	 = "/tmp/mod_auth_bsd-lock.XXXXXXXXXX";
	pid_t pid;
	int status;

	SPLAY_INIT(&auth_failed_lookup);
	CIRCLEQ_INIT(&auth_failed_cache);

	SPLAY_INIT(&auth_okayed_lookup);
	CIRCLEQ_INIT(&auth_okayed_cache);


	ap_log_error(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,s,"[AuthBSD] Preparing auth daemon");

	/*
	 * Create our IPC link.
	 */
	if (0 != socketpair(AF_UNIX,SOCK_STREAM,0,proto_socket))
		goto sysfail;

	/*
	 * Create our lock file
	 */
	masksav	= umask(0077);

	proto_lockfd	= mkstemp(filenam);

	errsav	= errno;

	if (0 != unlink(filenam)) {
		errsav	= errno;

		(void)close(proto_lockfd);

		proto_lockfd	= -1;
	}

	(void)umask(masksav);

	errno	= errsav;

	if (proto_lockfd == -1)
		goto sysfail;

	/*
	 * Part ways and split mutually exclusive resources.
	 */
	ap_log_error(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,s,"[AuthBSD] Forking auth daemon");

	if (-1 == (pid = fork()))
		goto sysfail;

	/*
	 * Fallback to httpd process?
	 */
	if (pid > 0) {
		(void)close(proto_socket[SOCKET_SEND]);
		proto_socket[SOCKET_SEND]	= -1;

		return 0;
	}

	/*
	 * Auth daemon, go!
	 */
	(void)close(proto_socket[SOCKET_RECV]);
	proto_socket[SOCKET_RECV]	= -1;

	ap_log_error(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,s,"[AuthBSD] Auth daemon running with pid %lu",(unsigned long)getpid());

	status	= authd_main(s);

	ap_log_error(APLOG_MARK,APLOG_NOTICE|APLOG_NOERRNO,s,"[AuthBSD] Auth daemon crashing....");

	exit(status);

	/* NOT REACHED */
sysfail:
	errsav	= errno;

	(void)close(proto_socket[SOCKET_RECV]);
	proto_socket[SOCKET_RECV]	= -1;

	(void)close(proto_socket[SOCKET_SEND]);
	proto_socket[SOCKET_SEND]	= -1;

	errno	= errsav;

	ap_log_error(APLOG_MARK,APLOG_ERR|APLOG_NOERRNO,s,"[AuthBSD] Failed to start auth daemon: %s",strerror(errno));

	return -1;
} /* authd_init() */

