/* ==========================================================================
 * mod_auth_bsd - Apache module for login_bsd BSD Authentication wrapper
 * --------------------------------------------------------------------------
 * Copyright (c) 2003  William Ahern
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to permit
 * persons to whom the Software is furnished to do so, subject to the
 * following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
 * NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 * ==========================================================================
 */

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_protocol.h"
#include "ap_md5.h"
#include "ap_sha1.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>

#include <unistd.h>	/* access */

#include <sys/param.h>	/* MAXPATHLEN */

#include <string.h>	/* strerror, strlen, strcasecmp */
#include <errno.h>

#include <assert.h>



#define XSTR(x) #x
#define STR(x)  XSTR(x)



/*
 * http://www.ibiblio.org/
 * 	mdw/HOWTO/Secure-Programs-HOWTO/handle-metacharacters.html
 *
 * Ideally we should create a popen() w/ execl() semantics.
 */
bool is_clean(const char *s) {
	char dirty[]	= "&;'\"\\|*?~<>^()[]{}$\n\r!#-";
	char *c;

	for (; *s; s++)
		for (c = dirty; *c; c++)
			if (*s == *c)
				return false;

	return true;
} /* is_clean */



/*
 * HMAC Implementation
 *
 * http://www.ietf.org/rfc/rfc2104.txt
 */

#if !defined USE_SHA1 && !defined USE_MD5
#define	USE_MD5
#endif

#define SHA1_DIGEST_SIZE	20
#define SHA1_BLOCK_SIZE		64

#define MD5_DIGEST_SIZE		16
#define MD5_BLOCK_SIZE		64

#ifdef USE_SHA1
#define HMAC_DIGEST_SIZE	SHA1_DIGEST_SIZE
#define HMAC_BLOCK_SIZE		SHA1_BLOCK_SIZE
#else
#define HMAC_DIGEST_SIZE	MD5_DIGEST_SIZE
#define HMAC_BLOCK_SIZE		MD5_BLOCK_SIZE
#endif


#define HMAC_IPAD	0x36
#define HMAC_OPAD	0x5C

typedef struct {
	#ifdef USE_SHA1
	AP_SHA1_CTX hash;
	#else
	AP_MD5_CTX hash;
	#endif

	void (*hash_init)(void *);
	void (*hash_update)(void *,const char *,unsigned int);
	void (*hash_final)(unsigned char[HMAC_DIGEST_SIZE],void *);

	unsigned char key[HMAC_BLOCK_SIZE];
	unsigned char ipad[HMAC_BLOCK_SIZE];
	unsigned char opad[HMAC_BLOCK_SIZE];
} hmac_ctx;


static void hmac_init(hmac_ctx *ctx, unsigned char *key, int keysz) {
	int i;

	#ifdef USE_SHA1
		ctx->hash_init
			= (void(*)(void *))ap_SHA1Init;
		ctx->hash_update
			= (void(*)(void *,const char *,unsigned int))ap_SHA1Update;
		ctx->hash_final
			= (void(*)(unsigned char[HMAC_DIGEST_SIZE],void *))ap_SHA1Final;
	#else
		ctx->hash_init
			= (void(*)(void *))ap_MD5Init;
		ctx->hash_update
			= (void(*)(void *,const char *,unsigned int))ap_MD5Update;
		ctx->hash_final
			= (void(*)(unsigned char[HMAC_DIGEST_SIZE],void *))ap_MD5Final;
	#endif

	bzero(ctx->key,sizeof(ctx->key));

	if (keysz > HMAC_BLOCK_SIZE) {
		ctx->hash_init(&ctx->hash);
		ctx->hash_update(&ctx->hash,key,keysz);
		ctx->hash_final(ctx->key,&ctx->hash);
	} else {
		memcpy(ctx->key,key,keysz);
	}

	assert(sizeof(ctx->key) == sizeof(ctx->ipad));

	for (i = 0; i < sizeof(ctx->ipad); i++)
		ctx->ipad[i]	= HMAC_IPAD ^ ctx->key[i];

	ctx->hash_init(&ctx->hash);
	ctx->hash_update(&ctx->hash,ctx->ipad,sizeof(ctx->ipad));

	return; /* void */
} /* hmac_init */


static void hmac_update(hmac_ctx *ctx, unsigned char *buf, int bufsz) {
	ctx->hash_update(&ctx->hash,buf,bufsz);

	return; /* void */
} /* hmac_update */


static void hmac_final(hmac_ctx *ctx, unsigned char digest[HMAC_DIGEST_SIZE]) {
	int i;

	ctx->hash_final(digest,&ctx->hash);

	assert(sizeof(ctx->opad) == sizeof(ctx->key));

	for (i = 0; i < sizeof(ctx->opad); i++)
		ctx->opad[i]	= HMAC_OPAD ^ ctx->key[i];

	ctx->hash_init(&ctx->hash);
	ctx->hash_update(&ctx->hash,ctx->opad,sizeof(ctx->opad));
	ctx->hash_update(&ctx->hash,digest,sizeof(digest));

	ctx->hash_final(digest,&ctx->hash);

	return; /* void */
} /* hmac_final */


static void hmac(unsigned char digest[HMAC_DIGEST_SIZE], unsigned char *key, int keysz, unsigned char *plain, int plainsz) {
	hmac_ctx ctx;

	hmac_init(&ctx,key,keysz);
	hmac_update(&ctx,plain,plainsz);
	hmac_final(&ctx,digest);

	return; /* void */
} /* hmac */



static int trace_callback(void *data, const char *key, const char *val) {
	request_rec *r = (request_rec *)data;
	ap_rprintf(r, "Header Field %s' == %s'\n", key, val);
	return TRUE;
} /* trace_callback */



struct auth_config {
	bool use;
	bool keep_pass;
	bool req_ssl;
	char *wrapper;
	char *service;
	char *style;
	char *type;
};


/*
 * Apache module parameter declaration
 */
module MODULE_VAR_EXPORT bsd_auth_module;


/*
 * pool *p:	apache pool allocator
 * char *dn:	directory name
 */
static void *create_dir_config(pool *p, char *dn) {
	struct auth_config *cfg;

	cfg	= (struct auth_config *)ap_pcalloc(p, sizeof(*cfg));

	assert(cfg);

	cfg->use	= false;
	cfg->keep_pass	= false;
	cfg->req_ssl	= true;
	#ifdef WRAPPER
	cfg->wrapper	= STR(WRAPPER);
	#else
	cfg->wrapper	= NULL;
	#endif
	cfg->service	= NULL;
	cfg->style	= NULL;
	cfg->type	= NULL;

	return cfg;
} /* create_dir_config */


#ifndef DEV_RANDOM
#error "DEV_RANDOM must be defined as a path to a randomness device"
#endif


unsigned char secret[HMAC_BLOCK_SIZE];	/* must be unsigned, otherwise
					 * you'll overflow; you won't
					 * be guaranteed to get a strlen
					 * of 2.
					 */

static void process_init(server_rec *s, pool *p) {
	FILE *fp;
	int rd;

	fp	= fopen(STR(DEV_RANDOM),"r");

	if (!fp)
		goto fail;

	errno	= 0;
	rd	= fread(secret,1,sizeof(secret),fp);

	if (rd != sizeof(secret)) {
		if (feof(fp) && errno == 0)
			errno	= EIO;
		goto fail;
	}

	fclose(fp);

	return; /* void */
fail:
	ap_log_printf(s,"[AuthBSD] Failed to read secret from device[%s]: %s",STR(DEV_RANDOM),strerror(errno));

	if (fp)
		fclose(fp);

	bzero(secret,sizeof(secret));

	return; /* void */
} /* process_init */


static int authenticate(request_rec *r) {
	struct auth_config *cfg;
	const char *auth_type			= NULL;
	char user[256]				= {0};
	const char *pass			= NULL;
	unsigned char digest[HMAC_DIGEST_SIZE]	= {0};
	char mac[sizeof(digest)*2+1]		= {0};
	char auth[256]			= {'B','a','s','i','c',' ','\0'};
	char log[256]			= {0};
	char exec[1024]			= {0};
	int i, ln, ret;
	FILE *p;


	auth_type	= ap_auth_type(r);

	if (auth_type == NULL || strcasecmp(auth_type,"Basic") != 0) {
		ap_note_basic_auth_failure(r);
		ap_log_reason("[AuthBSD] Only `Basic' authentication supported", r->uri, r);
		ret	= HTTP_INTERNAL_SERVER_ERROR;
		goto finish;
	}


	ret	= ap_get_basic_auth_pw(r,&pass);

	if (ret != OK)
		return ret;

	if (r->connection->user == NULL || *r->connection->user == '\0') {
		ap_note_basic_auth_failure(r);
		ap_log_reason("[AuthBSD] No username provided", r->uri, r);
		ret	= HTTP_UNAUTHORIZED;
		goto finish;
	}

	if (!is_clean(r->connection->user)) {
		ap_note_basic_auth_failure(r);
		ap_log_reason("[AuthBSD] Username has illegal characters", r->uri, r);
		ret	= HTTP_UNAUTHORIZED;
		goto finish;
	}


	cfg	= ap_get_module_config(r->per_dir_config,&bsd_auth_module);

	assert(cfg);


	if (!cfg->use) {
		ret	= DECLINED;
		goto finish;
	}


	/* check for prior authorization (from internal redirect) */
	if (cfg->keep_pass == false) {
		hmac(digest,secret,sizeof(secret),r->connection->user,strlen(r->connection->user));
		for (i = 0; i < sizeof(digest); i++)
			(void)sprintf(&mac[i*2],"%.2x",digest[i]);

		if (strcmp(mac,pass) == 0) {
			ret	= OK;
			goto finish;
		}
	}


	/* check for SSL */
	if (cfg->req_ssl) {
		#ifdef EAPI
		if (ap_ctx_get(r->connection->client->ctx,"ssl") == NULL) {
			ap_note_basic_auth_failure(r);
			ap_log_reason("[AuthBSD] Will not authenticate without SSL connection", r->uri, r);
			ret	= HTTP_INTERNAL_SERVER_ERROR;
			goto finish;
		}
		#else
		ap_note_basic_auth_failure(r);
		ap_log_reason("[AuthBSD] Cannot determine if SSL is in use", r->uri, r);
		ret	= HTTP_INTERNAL_SERVER_ERROR;
		goto finish;
		#endif
	}


	if (!cfg->wrapper) {
		ap_note_basic_auth_failure(r);
		ap_log_reason("[AuthBSD] No wrapper defined", r->uri, r);
		ret	= HTTP_INTERNAL_SERVER_ERROR;
		goto finish;
	}
		
	if (cfg->service && strcmp(cfg->service,"login") != 0) {
		ap_note_basic_auth_failure(r);
		(void)snprintf(log,sizeof(log),"[AuthBSD] Unsupported service protocol: %s",cfg->type);
		ap_log_reason(log, r->uri, r);
		ret	= HTTP_UNAUTHORIZED;
		goto finish;
	}


	if (cfg->style)
		ln	= snprintf(user,sizeof(user),"%s:%s",r->connection->user,cfg->style);
	else
		ln	= snprintf(user,sizeof(user),"%s",r->connection->user);

	if (ln >= sizeof(user)) {
		ap_note_basic_auth_failure(r);
		(void)snprintf(log,sizeof(log),"[AuthBSD] Maximum username length is %d",sizeof(user));
		ap_log_reason(log, r->uri, r);
		ret	= HTTP_INTERNAL_SERVER_ERROR;
		goto finish;
	}


	if (cfg->type)
		ln	= snprintf(exec,sizeof(exec),"%s -t %s %s 2>/dev/null",cfg->wrapper,cfg->type,user);
	else
		ln	= snprintf(exec,sizeof(exec),"%s %s 2>/dev/null",cfg->wrapper,user);

	if (ln >= sizeof(exec)) {
		ap_note_basic_auth_failure(r);
		(void)snprintf(log,sizeof(log),"[AuthBSD] Length of AuthBSDWrapper + AuthBSDService + AuthBSDType + AuthBSDStyle + username > %d",sizeof(exec));
		ap_log_reason(log, r->uri, r);
		ret	= HTTP_INTERNAL_SERVER_ERROR;
		goto finish;
	}


	p	= popen(exec,"w");

	if (!p) {
		ap_note_basic_auth_failure(r);
		(void)snprintf(log,sizeof(log),"[AuthBSD] Could not execute wrapper: %s",strerror(errno));
		ap_log_reason(log, r->uri, r);
		ret	= HTTP_INTERNAL_SERVER_ERROR;
		goto finish;
	}


	/*
	 * XXX: what does the wrapper do if ln == 0?
	 */
	ln	= strlen(pass);	

	ret = fwrite(pass,1,ln,p);

	if (ret != ln) {
		ap_note_basic_auth_failure(r);
		(void)snprintf(log,sizeof(log),"[AuthBSD] Could not write passwd to wrapper: %s",strerror(errno));
		ap_log_reason(log, r->uri, r);
		ret	= HTTP_INTERNAL_SERVER_ERROR;
		goto finish;
	}

	ret	= pclose(p);

	if (ret == -1 || WEXITSTATUS(ret) == 127) {
		ap_note_basic_auth_failure(r);
		(void)snprintf(log,sizeof(log),"[AuthBSD] Wrapper execution failure: %s",strerror(errno));
		ap_log_reason(log, r->uri, r);
		ret 	= HTTP_INTERNAL_SERVER_ERROR;
		goto finish;
	} else if (WEXITSTATUS(ret) == 0) {
		ret	= OK;
	} else {
		ap_note_basic_auth_failure(r);
		(void)snprintf(log,sizeof(log),"[AuthBSD] Authentication failed for user: %s",user);
		ap_log_reason(log, r->uri, r);
		ret	= HTTP_UNAUTHORIZED;
		goto finish;
	}


	if (cfg->keep_pass == false) {
		hmac(digest,secret,sizeof(secret),r->connection->user,strlen(r->connection->user));
		for (i = 0; i < sizeof(digest); i++)
			(void)sprintf(&mac[i*2],"%.2x",digest[i]);

		ln	= snprintf(user,sizeof(user),"%s:%s",r->connection->user,mac);

		if (ln >= sizeof(user)) {
			ap_note_basic_auth_failure(r);
			ap_log_reason("[AuthBSD] Could not erase password", r->uri, r);
			ret	= HTTP_INTERNAL_SERVER_ERROR;
			goto finish;
		}

		/* XXX: "Basic " + auth */
		if (ap_base64encode_len(ln) >= (sizeof(auth) - strlen(auth))) {
			ap_note_basic_auth_failure(r);
			ap_log_reason("[AuthBSD] Could not erase password", r->uri, r);
			ret	= HTTP_INTERNAL_SERVER_ERROR;
			goto finish;
		}

		ap_base64encode(auth + strlen(auth),user,ln); /* ln comes from above */

		ap_table_set(r->headers_in,"Authorization",auth);
	}

finish:
	/*ap_table_do(trace_callback,r,r->headers_in,NULL);*/

	return ret;
} /* authenticate */



/*
 * Apache config command table
 */
static const command_rec ctable[] = {
	{
		"AuthBSD",
		ap_set_flag_slot,
		(void *)XtOffsetOf(struct auth_config,use),
		OR_AUTHCFG,
		FLAG,
		"Toggle BSD Authentication: On | Off"
	},{
		"AuthBSDKeepPass",
		ap_set_flag_slot,
		(void *)XtOffsetOf(struct auth_config,keep_pass),
		OR_AUTHCFG,
		FLAG,
		"Toggle password erasure: On | Off"
	},{
		"AuthBSDRequireSSL",
		ap_set_flag_slot,
		(void *)XtOffsetOf(struct auth_config,req_ssl),
		OR_AUTHCFG,
		FLAG,
		"Toggle secure connection verification: On | Off"
	},{
		"AuthBSDWrapper",
		ap_set_file_slot,
		(void *)XtOffsetOf(struct auth_config,wrapper),
		OR_AUTHCFG,
		TAKE1,
		"BSD Authentication wrapper: /usr/local/libexec/login_bsd"
	},{
		"AuthBSDService",
		ap_set_string_slot,
		(void *)XtOffsetOf(struct auth_config,service),
		OR_AUTHCFG,
		TAKE1,
		"BSD Authentication protocol: login | challenge"
	},{
		"AuthBSDStyle",
		ap_set_string_slot,
		(void *)XtOffsetOf(struct auth_config,style),
		OR_AUTHCFG,
		TAKE1,
		"BSD Authentication style: passwd | radius | ..."
	},{
		"AuthBSDType",
		ap_set_string_slot,
		(void *)XtOffsetOf(struct auth_config,type),
		OR_AUTHCFG,
		TAKE1,
		"BSD Authentication type: auth-www | auth-webdav | ..."
	},{NULL}

};




/*
 * Apache module parameter definitions
 *
 * NOTE:
 * 	[N]: N is the order in which the routine is called, if called
 */
module MODULE_VAR_EXPORT bsd_auth_module = {
	STANDARD_MODULE_STUFF,
	NULL,			/* module initializer */
	create_dir_config,	/* per-directory config creator */
	NULL,			/* directory config merger */
	NULL,			/* server config creator */
	NULL,			/* server config merger */
	ctable,			/* command table */
	NULL,			/* [9] list of handlers */
	NULL,			/* [2] filename-to-URI translation */
	authenticate,		/* [5] check/validate user_id */
	NULL,			/* [6] check user_id is valid here */
	NULL,			/* [4] check access by host address */
	NULL,			/* [7] MIME type checker/settler */
	NULL,			/* [8] fixups */
	NULL,			/* [10] logger */
	NULL,			/* [3] header parser */
	process_init,		/* child_init */
	NULL,			/* child_exit */
	NULL,			/* [1] post read-request */
};

