/* ==========================================================================
 * login_bsd - Shell wrapper for BSD Authentication using high-level C API
 * --------------------------------------------------------------------------
 * Copyright (c) 2003  William Ahern
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to permit
 * persons to whom the Software is furnished to do so, subject to the
 * following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
 * NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 * ==========================================================================
 */

#include <stdio.h>
#include <stdlib.h>

#include <login_cap.h>	/* auth_userokay, ... */
#include <bsd_auth.h>	/* auth_userokay, ... */

#include <sys/types.h>	/* getpwuid */
#include <pwd.h>	/* getpwuid */

#include <fcntl.h>	/* open */

#include <unistd.h>	/* getopt, getpwuid */

#include <time.h>	/* nanosleep */

#include <syslog.h>	/* openlog, syslog */

#include <errno.h>

#include <err.h>	/* err, warn, ... */

#include <string.h>	/* strcmp, strchr, strdup */


#define USAGE	\
	"bsdauth [-t type] [-s service] [-v name=value] user[:style] [class]\n"	\
	"  -t type        auth type, i.e. auth-ftp\n"				\
	"  -s service     auth protocol: login | challenge\n"			\
	"  -v name=value  name/value pairs to pass to auth script\n"		\
	"  -h             display this help and exit\n"				\
	"\n"									\
	"Report bugs to William Ahern <william@25thandClement.com>"


#if !defined PROGRAM
#define PROGRAM		login_bsd
#endif


#if !defined DELAY_SEC && !defined DELAY_NSEC
#define DELAY_SEC	1
#elif defined DELAY_NSEC
#define DELAY_SEC	0
#endif

#if !defined DELAY_NSEC
#define DELAY_NSEC	0
#endif


#define AUTH_SUCCESS	EXIT_SUCCESS
#define AUTH_FAILURE	EXIT_FAILURE


#define XSTR(x) #x
#define STR(x)  XSTR(x)


/*
 * Auth type and function handler pair
 */
struct auth {
	char *service;
	int (*authenticate)(char *, char *, char *, char *);
};


int service_login(char *user, char *style, char *type, char *class) {
	/* sizeof response in login.c, obsd 3.3 */
	char passwd[1024]	= {0};
	int passln;
	int ch;
	struct timespec delay;
	int okay;

	(void)setvbuf(stdin,NULL,_IONBF,0);

	for (passln = 0; passln < sizeof(passwd); passln++) {
		ch	= getchar();
		if (ch == EOF || ch == '\0' || ch == '\n')
			break;
		passwd[passln]	= ch;
	}

	delay.tv_sec	= DELAY_SEC;
	delay.tv_nsec	= DELAY_NSEC;

	#if DELAY_SEC || DELAY_NSEC
	if (nanosleep(&delay,NULL) != 0) {
		syslog(LOG_WARNING,"rudely woken from sleep: %m");
		return AUTH_FAILURE;
	}
	#endif

	okay	= auth_userokay(user,style,type,passwd);

	if (okay) {
		return AUTH_SUCCESS;
	} else {
		return AUTH_FAILURE;
	}
} /* service_login */


int service_challenge(char *user, char *style, char *type, char *class) {

	syslog(LOG_NOTICE,"challenge service not supported yet");

	return AUTH_FAILURE;
} /* service_challenge */


struct auth handlers[] = {
	{"login",service_login},
	{"challenge",service_challenge},
	{NULL,NULL}
};


int main(int argc, char *argv[]) {
	char *srvc	= "login";
	char *user	= NULL;
	char *style	= NULL;
	char *type	= NULL;
	char *class	= NULL;


/* get option arguments */ {

	extern char *optarg;
	extern int optind;
	int ch;


	opterr	= 0;
	while ((ch = getopt(argc,argv,"t:s:v:h")) != -1) {
		switch (ch) {
			case 't':
				type	= optarg;
				break;
			case 's':
				srvc	= optarg;
				break;
			case 'v':
				warnx("-v values not currently supported");
				break;
			case 'h':
			case '?':
			default:
				fprintf(stderr,"%s\n",USAGE);
				return AUTH_FAILURE;
		}
	}

	argc	-= optind;
	argv	+= optind;


	if (argc == 0)
		return (fprintf(stderr,"%s\n",USAGE),AUTH_FAILURE);

	user	= *argv;

	if ((style = strchr(user,':'))) {
		*style	= '\0';
		style++;
	}

	argc--;
	argv++;


	if (argc == 0)
		goto handler;

	class	= *argv;

	argc--;
	argv++;


/* end get option arguments */ }


/* call handler */ {

	extern struct auth handlers[];
	struct auth *handler;
	struct passwd *pw;
	uid_t uid;
	int auth;

handler:

	/* we're important enough to log to syslog in addition to stderr */
	openlog(STR(PROGRAM),LOG_PERROR,LOG_AUTH);

	#if FAIL_BELOW_UID
	errno	= 0;
	pw	= getpwnam(user);
	
	if (pw) {
		uid	= pw->pw_uid;
	} else if (errno) {
		syslog(LOG_ERR,"getpwnam failed for `%s': %m",user);
		return AUTH_FAILURE;
	}
	#endif

	pw	= getpwuid(getuid());

	if (pw == NULL) {
		syslog(LOG_ERR,"unable to get username of authenticator during authentication for `%s': %m",user);
		return AUTH_FAILURE;
	}

	#if FAIL_BELOW_UID
		if (uid < FAIL_BELOW_UID) {
			syslog(LOG_WARNING,"user `%s' authentication for `%s' forbidden",pw->pw_name,user);
			return AUTH_FAILURE;
		}
	#endif

	for (handler = handlers; handler->service; handler++) {
		if (strcmp(srvc,handler->service) == 0) {
			auth	= handler->authenticate(user,style,type,class);

			if (auth != AUTH_SUCCESS)
				syslog(LOG_WARNING,"user `%s' failed authentication for `%s'",pw->pw_name,user);

			#if defined DO_LOG_SUCCESS
			if (auth == AUTH_SUCCESS)
				syslog(LOG_INFO,"user `%s' passed authentication for `%s'",pw->pw_name,user);
			#endif

			return auth;
		}
	}
	
/* end call handler */ }

	syslog(LOG_WARNING,"no protocol handler for service: %s",srvc);

	return AUTH_FAILURE;
}
