/*
 * prism2dump v0.01 [fields_mgmt.c]
 * by h1kari - (c) Dachb0den Labs 2001
 */

/*
 * Copyright (c) 2001 Dachb0den Labs.
 *      David Hulton <h1kari@dachb0den.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by David Hulton.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY David Hulton AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL David Hulton OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <sys/types.h>

#include <pcap.h>

#include "frames.h"
#include "prism2dump.h"

/*
 * structures
 */
struct wi_var_s {
  u_int8_t code;
  void (*func)(u_char *, u_int8_t);
};


void
print_mgmt_algo(u_int16_t algo)
{
  printf("algo: %s ", algo ? "sh-key" : "open");

  return;
}

void
print_mgmt_seq(u_int16_t seq)
{
  if(seq)
    printf("seq: %u ", seq);

  return;
}

void
print_mgmt_interval(u_int16_t interval)
{
  if(interval)
    printf("int: %u ", interval);

  return;
}

void
print_mgmt_capinfo(u_int16_t capinfo)
{
  if(!capinfo)
    return;
  
  printf("capinfo: ");

  if(capinfo & WI_CAPINFO_ESS)
    printf("ess ");
  if(capinfo & WI_CAPINFO_IBSS)
    printf("ibss ");
  if(capinfo & WI_CAPINFO_CFPOLL)
    printf("cfpoll ");
  if(capinfo & WI_CAPINFO_CFPOLLREQ)
    printf("cfpollreq ");
  if(capinfo & WI_CAPINFO_PRIV)
    printf("priv ");

  return;
}

void
print_mgmt_currap(u_int8_t *currap)
{
  printf("currap: %x:%x:%x:%x:%x:%x ", MACIT(currap));

  return;
}

void
print_mgmt_reason(u_int16_t reason)
{
  if(!reason)
    return;
  
  printf("reason: ");

  switch(reason)
  {
    case WI_REASON_UNSPEC:
      printf("unspec ");
      break;
    case WI_REASON_AUTH_INVALID:
      printf("auth-invalid ");
      break;
    case WI_REASON_DEAUTH_LEAVE:
      printf("deauth-leave ");
      break;
    case WI_REASON_DISAS_INACT:
      printf("disas-inact ");
      break;
    case WI_REASON_DISAS_OVERL:
      printf("disas-overl ");
      break;
    case WI_REASON_CLASS2:
      printf("class2 ");
      break;
    case WI_REASON_CLASS3:
      printf("class3 ");
      break;
    case WI_REASON_DISAS_LEAVE:
      printf("disas-leave ");
      break;
    case WI_REASON_NOAUTH:
      printf("noauth ");
      break;
    default:
      break;
  }

  return;
}

void
print_mgmt_aid(u_int16_t aid)
{
   printf("aid: %u ", aid & WI_AID_MASK);

   return;
}

void
print_mgmt_status(u_int16_t status)
{
  if(!status)
    return;

  printf("status: ");

  switch(status)
  {
    case WI_STAT_UNSPEC_FAIL:
      printf("unspec-fail ");
      break;
    case WI_STAT_CAPINFO_FAIL:
      printf("capinfo-fail ");
      break;
    case WI_STAT_REAS_DENY:
      printf("reas-deny ");
      break;
    case WI_STAT_ASSOC_DENY:
      printf("assoc-deny ");
      break;
    case WI_STAT_ALGO_FAIL:
      printf("algo-fail ");
      break;
    case WI_STAT_SEQ_FAIL:
      printf("seq-fail ");
      break;
    case WI_STAT_CHAL_FAIL:
      printf("chal-fail ");
      break;
    case WI_STAT_TOUT_FAIL:
      printf("tout-fail ");
      break;
    case WI_STAT_OVERL_DENY:
      printf("overl-deny ");
      break;
    case WI_STAT_RATE_DENY:
      printf("rate-deny ");
      break;
    default:
      break;
  }

  return;
}

void
print_mgmt_ts(u_int32_t ts0, u_int32_t ts1)
{
  u_int64_t timestamp;

  timestamp = ts1;
  timestamp <<= 32;
  timestamp |= ts0;

  printf("ts: %llu.%llu ", timestamp / 1000000, timestamp % 1000000);

  return;
}

struct wi_var_s wi_vars[] = {
  { WI_VAR_SSID, &print_mgmt_var_ssid },
  { WI_VAR_SRATES, &print_mgmt_var_srates },
  { WI_VAR_FH, &print_mgmt_var_fh },
  { WI_VAR_DS, &print_mgmt_var_ds },
  { WI_VAR_CF, &print_mgmt_var_cf },
  { WI_VAR_IBSS, &print_mgmt_var_ibss },
  { WI_VAR_TIM, &print_mgmt_var_tim },
  { WI_VAR_CHAL, &print_mgmt_var_chal },
  { 0xFF, NULL }
};

void
print_mgmt_var(u_char *data, int off, u_int16_t len, u_int8_t *var, int var_len)
{
  int i, found;
  struct wi_mgmt_var_hdr wi_var_h;

  if(verbose < 2)
    return;

  while(off < len)
  {
    memcpy(&wi_var_h, data + off, 2);
    off += 2;
    memcpy(wi_var_h.wi_data, data + off, wi_var_h.wi_len);
    off += wi_var_h.wi_len;

    for(i = 0, found = 0; i < var_len; i++)
    {
      if(var[i] == wi_var_h.wi_code)
      {
        found++;
        break;
      }
    }

    if(!found)
      continue;

    for(i = 0; wi_vars[i].func != NULL; i++)
    {
      if(wi_vars[i].code == wi_var_h.wi_code)
      {
        printsol(2);
        (*wi_vars[i].func)(wi_var_h.wi_data, wi_var_h.wi_len);
        printeol(2);
        break;
      }
    }
  }

  return;
}

void
print_mgmt_var_ssid(u_char *data, u_int8_t len)
{
  data[len] = '\0';
  printf("ssid: [%s] ", data);

  return;
}

void
print_mgmt_var_srates(u_char *data, u_int8_t len)
{
  int i;

  printf("rates: ");
  for(i = 0; i < len; i++)
  {
    data[i] &= WI_VAR_SRATES_MASK; /* kill the msb */
    printf("%u.%u ", data[i] / 2, (data[i] % 2) ? 5 : 0);
  }

  return;
}

void
print_mgmt_var_fh(u_char *data, u_int8_t len)
{
  u_int16_t dtu;

  dtu = *(data);
  printf("fh dtu: %u hs: %u hp: %u hi: %u ", dtu, data[2], data[3], data[4]);

  return;
}

void
print_mgmt_var_ds(u_char *data, u_int8_t len)
{
  printf("ds ch: %u ", data[0]);

  return;
}

void
print_mgmt_var_cf(u_char *data, u_int8_t len)
{
  u_int16_t mdtu, drtu;

  mdtu = *(data + 2);
  drtu = *(data + 4);
  printf("cf c: %u p: %u mdtu: %u drtu: %u ", data[0], data[1], mdtu, drtu);

  return;
}

void
print_mgmt_var_ibss(u_char *data, u_int8_t len)
{
  u_int16_t atimwtu;

  atimwtu = *(data);
  printf("ibss atimwtu: %u ", atimwtu);

  return;
}

#define TEMPC_LEN 16
void
print_mgmt_var_tim(u_char *data, u_int8_t len)
{
  char tempc[TEMPC_LEN];

  printf("dtim c: %u p: %u bc: %u ", data[0], data[1], data[2]);
  snprintf(tempc, TEMPC_LEN - 1, "pvb: %%%ux ", (len - 3) * 2);
  printf(tempc, data + 3);

  return;
}

void
print_mgmt_var_chal(u_char *data, u_int8_t len)
{
  data[len] = '\0';
  printf("chal: %s ", data);

  return;
}
