/*
 * prism2ctl v0.1 [prism2ctl.c]
 * by h1kari - (c) Dachb0den Labs 2001
 *
 * prism2ctl is heavily based on the structure used in wicontrol by Bill Paul,
 * however does not fully copy his code in any way. The similar structure was
 * chosen for conformity in case this code is later integrated into any bsd
 * release.
 */

/*
 * Copyright (c) 2001 Dachb0den Labs.
 *      David Hulton <h1kari@dachb0den.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by David Hulton.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY David Hulton AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL David Hulton OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <stdio.h>
#include <sys/types.h>
#ifdef __OpenBSD__
#include <sys/cdefs.h>
#include <sys/param.h>
#endif
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <net/if.h>

#include <bat/common.h>

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <err.h>


#ifdef __FreeBSD__
#define le16toh(x) (x)
#define htole16(x) (x)
#else
#ifdef __OpenBSD__
#define le16toh(x) letoh16(x)
#endif
#endif




struct p2_table {
	int p2_type;
	int p2_code;
#define P2_NONE                 0x00
#define P2_STRING               0x01
#define P2_BOOL                 0x02
#define P2_WORDS		0x03
#define P2_HEX			0x04
        char *p2_label;                 /* label used to print info */
        int p2_opt;                     /* option character to set this */
        char *p2_desc;
        char *p2_optval;
};

static int p2_xtoi		__P((char *));
static void p2_getval		__P((char *, struct wi_req *));
static void p2_setval		__P((char *, struct wi_req *));
static void p2_setnone		__P((char *, int));
static void p2_setstr		__P((char *, int, char *));
static void p2_setwords		__P((char *, int, int));
static void p2_sethex		__P((char *, int, char *));
static void p2_printstr		__P((struct wi_req *));
static void p2_printbool	__P((struct wi_req *));
static void p2_printwords	__P((struct wi_req *));
static void p2_printhex		__P((struct wi_req *));
static void p2_dumpinfo		__P((char *));
int main			__P((int argc, char **argv));

#ifndef MIN
#define MIN(x, y) (x < y ? x : y);
#endif

#define HEX2INTOFF(x, y, z) { \
	x |= (y >= '0' && y <= '9' ? y - '0' : \
	    (y >= 'A' && y <= 'F' ? y - 'A' + 10 : \
	    (y >= 'a' && y <= 'f' ? y - 'a' + 10 : 0))) << ((z) * 4); \
}

static int
p2_xtoi(str)
	char	*str;
{
	/*
	 * have to limit the amount of hex to 16-bits because all of the
	 * debug commands take only 16-bit arguments.
	 */
	char	*ptr;
	int	i, hex = 0, len = MIN(strlen(str), 4);

	ptr = (strlen(str) > 4 ? str + strlen(str) - len : str);

	for(i = 0; i < len; i++)
		HEX2INTOFF(hex, ptr[i], len - i - 1);

	return (hex);
}

static void
p2_getval(iface, wreq)
	char		*iface;
	struct wi_req	*wreq;
{
	struct ifreq	ifr;
	int		s;

	bzero((char *)&ifr, sizeof(ifr));

	strcpy(ifr.ifr_name, iface);
	ifr.ifr_data = (caddr_t)wreq;

	s = socket(AF_INET, SOCK_DGRAM, 0);

	if (s == -1)
		err(1, "socket");

	if (ioctl(s, SIOCGPRISM2DEBUG, &ifr) == -1)
		err(1, "SIOCGPRISM2DEBUG");

	close(s);

	return;
}

static void
p2_setval(iface, wreq)
	char 		*iface;
	struct wi_req	*wreq;
{
	struct ifreq	ifr;
	int		s;

	bzero((char *)&ifr, sizeof(ifr));

	strcpy(ifr.ifr_name, iface);
	ifr.ifr_data = (caddr_t)wreq;

	s = socket(AF_INET, SOCK_DGRAM, 0);

	if (s == -1)
		err(1, "socket");

	if (ioctl(s, SIOCSPRISM2DEBUG, &ifr) == -1)
		err(1, "SIOCSPRISM2DEBUG");

	close(s);

	return;
}

static void
p2_setnone(iface, code)
	char	*iface;
	int	code;
{
	struct wi_req	wreq;

	bzero((char *)&wreq, sizeof(wreq));

	wreq.wi_type = code;
	wreq.wi_len = 0;

	p2_setval(iface, &wreq);

	return;
}

static void
p2_setstr(iface, code, str)
	char	*iface;
	int	code;
	char	*str;
{
	int		bool, hex;
	char		*ptr;
	struct wi_req	wreq;

	bzero((char *)&wreq, sizeof(wreq));

	/*
	 * assume string is in bool:hex format. parse for the : character
	 * and split accordingly.
	 */
	if((ptr = strchr(str, ':')) == NULL)
		err(1, "invalid argument, must be in num:hex format");

	*ptr++ = '\0';
	bool = atoi(str);
	hex = p2_xtoi(ptr);

	wreq.wi_type = code;
	wreq.wi_len = 2;
	wreq.wi_val[0] = htole16(bool);
	wreq.wi_val[1] = htole16(hex);

	p2_setval(iface, &wreq);

	return;
}

static void
p2_setwords(iface, code, words)
	char *iface;
	int code;
	int words;
{
	struct wi_req	wreq;

	bzero((char *)&wreq, sizeof(wreq));

	wreq.wi_type = code;
	wreq.wi_len = 1;
	wreq.wi_val[0] = htole16(words);

	switch (code) {
	/*
	 * a false for sleep is equivalent to wake. make sure we swap it out
	 * before it's sent to the driver so the driver doesn't have to do
	 * any extra legwork.
	 */
	case WI_DEBUG_SLEEP:
		if(!words)
			wreq.wi_type = WI_DEBUG_WAKE;
		wreq.wi_val[0] = htole16(1);
		break;
	/*
	 * Configure bits only has the Automatic Level Control implemented so
	 * far, so that's all we can enable/disable right now.. Hopefully
	 * there will be other options in the future.
	 */
	case WI_DEBUG_CONFBITS:
		wreq.wi_val[0] = htole16(wreq.wi_val[0] ? 1 : 0);
		wreq.wi_len = 2;
		wreq.wi_val[1] = htole16(0x0008);
		break;
	/*
	 * make sure the channel specified is within the range. otherwise,
	 * it could cause unpredictable results.
	 */
	case WI_DEBUG_CHAN:
		wreq.wi_val[0] = htole16(words > 14 ? 14 :
		    (wreq.wi_val[0] < 1 ? 1 : words));
		break;
	default:
		break;
	}

	p2_setval(iface, &wreq);

	return;
}

static void
p2_sethex(iface, code, hex)
	char	*iface;
	int	code;
	char	*hex;
{
	struct wi_req	wreq;

	bzero((char *)&wreq, sizeof(wreq));

	wreq.wi_type = code;
	wreq.wi_len = 1;
	wreq.wi_val[0] = htole16(p2_xtoi(hex));

	p2_setval(iface, &wreq);

	return;
}

static void
p2_printstr(wreq)
	struct wi_req	*wreq;
{
	if(le16toh(wreq->wi_val[0]))
		printf("[ %d:%04x ]", le16toh(wreq->wi_val[1]),
		    le16toh(wreq->wi_val[2]));
	else
		printf("[ ]");

	return;
}

static void
p2_printbool(wreq)
	struct wi_req	*wreq;
{
	printf(le16toh(wreq->wi_val[0]) ? "[ On ]" : "[ Off ]");

	return;
}

static void
p2_printwords(wreq)
	struct wi_req	*wreq;
{
	int	i;

	printf("[ ");
	for (i = 0; i < wreq->wi_len - 1; i++)
		printf("%d ", le16toh(wreq->wi_val[i]));
	printf("]");

	return;
}

static void
p2_printhex(wreq)
	struct wi_req	*wreq;
{
	if(wreq->wi_val[0])
		printf("[ 0x%04x ]", wreq->wi_val[1]);
	else
		printf("[ ]");

	return;
}
	

static struct p2_table p2_table[] = {
	{ WI_DEBUG_RESET, P2_NONE, NULL, 'r', "reset phy" },
	{ WI_DEBUG_INIT, P2_NONE, NULL, 'i', "init phy" },
	{ WI_DEBUG_SLEEP, P2_BOOL, "Sleep mode:\t\t\t\t", 's', "sleep mode" },
	{ WI_DEBUG_CHAN, P2_WORDS, NULL, 'f', "change channel" },
	{ WI_DEBUG_DELAYSUPP, P2_NONE, "Suppress post back-off delay:\t\t",
	    'd', "suppress delay" },
	{ WI_DEBUG_TXSUPP, P2_NONE, "Suppress Tx Exception:\t\t\t", 't',
	    "suppress tx" },
	{ WI_DEBUG_MONITOR, P2_NONE, "Monitor mode:\t\t\t\t", 'm',
	    "monitor mode" },
	{ WI_DEBUG_LEDTEST, P2_STRING, "LED Test:\t\t\t\t", 'l',
	    "led test" },
	{ WI_DEBUG_CONTTX, P2_HEX, "Continuous Tx:\t\t\t\t", 'c',
	    "continuous tx" },
	{ WI_DEBUG_STOPTEST, P2_NONE, NULL, 'h', "stop testing mode" },
	{ WI_DEBUG_CONTRX, P2_NONE, "Continuous Rx:\t\t\t\t", 'e',
	    "continuous rx" },
	{ WI_DEBUG_SIGSTATE, P2_HEX, "Signal State:\t\t\t\t", 'g',
	    "signal state" },
	{ WI_DEBUG_SIGSTATE, P2_NONE, NULL, 'a', "cal enable" },
	{ WI_DEBUG_CONFBITS, P2_BOOL, "Automatic level control:\t\t", 'b',
	    "automatic level control" },
	{ -1 }
};

static void
p2_dumpinfo(iface)
	char *iface;
{
	struct wi_req	wreq;
	struct p2_table	*table;

	for(table = p2_table; table->p2_type != -1; table++) {
		if(table->p2_label == NULL) continue;

		bzero((char *)&wreq, sizeof(wreq));

		wreq.wi_len = WI_MAX_DATALEN;
		wreq.wi_type = table->p2_type;

		p2_getval(iface, &wreq);
		printf("%s", table->p2_label);

		switch(table->p2_code) {
		case P2_STRING:
			p2_printstr(&wreq);
			break;
		case P2_NONE:
		case P2_BOOL:
			p2_printbool(&wreq);
			break;
		case P2_WORDS:
			p2_printwords(&wreq);
			break;
		case P2_HEX:
			p2_printhex(&wreq);
			break;
		}
		printf("\n");
	}

	return;
}

static void
usage(progname)
	char *progname;
{
	fprintf(stderr,
	    "usage: %s interface "
	    "[-r] [-i] [-s 0|1] [-f frequency] [-d] [-t] [-m]\n"
	    "       [-l 1|2:rate] [-c data] [-h] [-e] [-g bits] [-a] "
	    "[-b 0|1]\n", progname);
	exit(1);
}

int main(argc, argv)
	int	argc;
	char	*argv[];
{
	struct p2_table *table;
	char *iface;
	int ch, dumpinfo = 1;
	struct wi_req wreq;

	iface = NULL;

	if (argc > 1 && argv[1][0] != '-') {
		iface = argv[1];
		optind++;
	}
	else
		usage(argv[0]);

	bzero((char *)&wreq, sizeof(wreq));

	while ((ch = getopt(argc, argv, "ab:c:def:g:hil:mrs:t")) != -1) {
		/* since we have arguments, don't dump info */
		dumpinfo = 0;

		/*
		 * search for a match in the table for the specified argument.
		 * if we get a match, run a set function to send the data
		 * off to the device.
		 */
		for (table = p2_table; table->p2_type != -1; table++) {
			if (ch == table->p2_opt) {
				switch(table->p2_code) {
				case P2_NONE:
					p2_setnone(iface, table->p2_type);
					break;
				case P2_STRING:
					p2_setstr(iface, table->p2_type,
					    optarg);
					break;
				case P2_BOOL:
				case P2_WORDS:
					p2_setwords(iface, table->p2_type,
					    atoi(optarg));
					break;
				case P2_HEX:
					p2_sethex(iface, table->p2_type,
					    optarg);
					break;
				default:
					usage(argv[0]);
					break;
				}
				break;
			}
		}
	}

	if (iface == NULL)
		usage(argv[0]);

	if (dumpinfo)
		p2_dumpinfo(iface);

	exit(0);
}
