/*
 * dstumbler v1.0 [mon.c]
 * by h1kari - (c) Dachb0den Labs 2001
 */

/*
 * Copyright (c) 2001 Dachb0den Labs.
 *      David Hulton <h1kari@dachb0den.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by David Hulton.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY David Hulton AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL David Hulton OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>

#include <pcap.h>

#include "dstumbler.h"
#include "config.h"
#include "screen.h"

pcap_t *p;
u_char *buf;
struct pcap_pkthdr h;

/*
 * just grab the next packet so we can process it in both mon_parse_ap and
 * mon_parse_node
 */
int
mon_next(const char *iface)
{
  static int tset = 0;
  static struct timeval now, then;

  if(monmode && !chanlock && (CMPTVAL(now, then) > -1 || !tset))
  {
    ch %= SWITCH_CHANS;
    ch++;

    setdebugchan(iface, ch);

    gettimeofday(&now, NULL);
    ADDTVAL(now, then, MONSWSPEED);

    tset++;
  }

  if((buf = (u_char *)pcap_next(p, &h)) == NULL)
    return -1;

  return 0;
}

/*
 * parse buf for a beacon packet or proberesponse so we can get ap info
 */
#define GOT_SSID   0x1
#define GOT_SRATES 0x2
#define GOT_DS     0x4
#define GOT_ALL    (GOT_SSID | GOT_SRATES | GOT_DS)
int
mon_parse_ap(const char *iface, struct aps_s *aps)
{
  int off, len, i, mask = 0;
  u_char *data;
  struct wi_rx_frame *wi_h;
  struct wi_mgmt_beacon_hdr *beacon_h;
  struct wi_mgmt_var_hdr wi_var_h;

  if(buf == NULL)
    return -1;

  wi_h = (struct wi_rx_frame *)buf;
  data = (u_char *)buf + sizeof(struct wi_rx_frame);

  /*
   * make sure it's a beacon packet or probe response before continuing
   */
  if(le16toh(wi_h->wi_status) & WI_STAT_BADCRC ||
   !((le16toh(wi_h->wi_frame_ctl) & WI_FCTL_FTYPE) == WI_FTYPE_MGMT &&
   (((le16toh(wi_h->wi_frame_ctl) & WI_FCTL_STYPE) == WI_STYPE_MGMT_BEACON) ||
   (le16toh(wi_h->wi_frame_ctl) & WI_FCTL_STYPE) == WI_STYPE_MGMT_PROBERESP ||
   (le16toh(wi_h->wi_frame_ctl) & WI_FCTL_STYPE) == WI_STYPE_MGMT_AUTH)))
    return -1;

  data = (u_char *)buf + sizeof(struct wi_rx_frame);

  memcpy(aps->bssid, wi_h->wi_addr2, MACSIZE);

  aps->quality[2] = MAX(wi_h->wi_silence, 27);
  aps->quality[1] = MAX(wi_h->wi_signal, 27);
  aps->quality[0] = MAX(aps->quality[1] - aps->quality[2], 0);

  switch(le16toh(wi_h->wi_frame_ctl) & WI_FCTL_FTYPE)
  {
    case WI_FTYPE_MGMT:
      switch(le16toh(wi_h->wi_frame_ctl) & WI_FCTL_STYPE)
      {
        case WI_STYPE_MGMT_BEACON:
        case WI_STYPE_MGMT_PROBERESP:
          beacon_h = (struct wi_mgmt_beacon_hdr *)data;

          aps->weped = (le16toh(beacon_h->wi_capinfo) & WI_CAPINFO_PRIV) ?
           1 : 2;
          aps->adhoc = (le16toh(beacon_h->wi_capinfo) & WI_CAPINFO_IBSS) ?
           1 : 2;
          aps->interval = le16toh(beacon_h->wi_interval);

          off = sizeof(struct wi_mgmt_beacon_hdr);
          break;
        case WI_STYPE_MGMT_AUTH:
        {
          struct wi_mgmt_auth_hdr *auth_h;
          auth_h = (struct wi_mgmt_auth_hdr *)data;

          if(!(auth_h->wi_seq == 2 || auth_h->wi_seq == 4))
            return -1;

          /* set weped to on so we don't disturb the other settings */
          aps->weped = 1;
          aps->keyed = le16toh(auth_h->wi_algo) ? 1 : 2; 

          return 0;
        }
        default:
          return -1;
      }
      break;
    default:
      return -1;
  }

  len = le16toh(wi_h->wi_dat_len);

  while(off < len) 
  {
    memcpy(&wi_var_h, data + off, 2);
    off += 2;
    memcpy(wi_var_h.wi_data, data + off, wi_var_h.wi_len);
    off += wi_var_h.wi_len;

    switch(wi_var_h.wi_code)
    {
      case WI_VAR_SSID:
        wi_var_h.wi_len = MIN(wi_var_h.wi_len, SSIDLEN);
        strncpy(aps->ssid, wi_var_h.wi_data, wi_var_h.wi_len);
        aps->ssid[wi_var_h.wi_len] = '\0';
        mask |= GOT_SSID;
        break;
      case WI_VAR_SRATES:
        if(!(mask & GOT_SSID)) /* make sure the ssid is before the srate */
          break;
        for(i = 0; i < wi_var_h.wi_len; i++)
          aps->srate =
           MAX(aps->srate, (wi_var_h.wi_data[i] & WI_VAR_SRATES_MASK));
        mask |= GOT_SRATES;
        break;
      case WI_VAR_DS:
        if(!((mask & GOT_SSID) && (mask & GOT_SRATES)))
          break;
        aps->chan = wi_var_h.wi_data[0];
        mask |= GOT_DS;
        break;
      default:
        break;
    }
  }

  if(mask != GOT_ALL)
    return -1;

  return 0;
}

/*
 * parse packet for possible network node information
 */
#define NODE_BROADCAST "\xff\xff\xff\xff\xff\xff"
int
mon_parse_node(struct node_s *node, char *bssid)
{
  struct wi_rx_frame *wi_h;
  u_char *data;

  if(buf == NULL)
    return -1;

  wi_h = (struct wi_rx_frame *)buf;
  data = (u_char *)buf + sizeof(struct wi_rx_frame);

  /*
   * make sure it's not a beacon or proberequest, because they usually use
   * broadcast addresses
   */
  if(le16toh(wi_h->wi_status) & WI_STAT_BADCRC ||
   ((le16toh(wi_h->wi_frame_ctl) & WI_FCTL_FTYPE) == WI_FTYPE_MGMT &&
   ((le16toh(wi_h->wi_frame_ctl) & WI_FCTL_STYPE) == WI_STYPE_MGMT_BEACON)))
    return -1;

  /*
   * the source/dest/bssid/ra/ta is set dependant on the fromds/tods settings
   * so handle accordingly
   */
  if(le16toh(wi_h->wi_frame_ctl) & WI_FCTL_TODS &&
   !(le16toh(wi_h->wi_frame_ctl) & WI_FCTL_FROMDS))
  {
    memcpy(bssid, wi_h->wi_addr1, MACSIZE);
    memcpy(node->mac, wi_h->wi_addr2, MACSIZE);
  }
  else if((le16toh(wi_h->wi_frame_ctl) & WI_FCTL_FROMDS) && 
   !(le16toh(wi_h->wi_frame_ctl) & WI_FCTL_TODS))
  {
    memcpy(bssid, wi_h->wi_addr2, MACSIZE);
    memcpy(node->mac, wi_h->wi_addr1, MACSIZE);
  }
  else if(!(le16toh(wi_h->wi_frame_ctl) & (WI_FCTL_FROMDS | WI_FCTL_TODS)))
  {
    memcpy(bssid, wi_h->wi_addr3, MACSIZE);
    memcpy(node->mac, wi_h->wi_addr2, MACSIZE);

    if(memcmp(node->mac, bssid, MACSIZE) == 0)
      memcpy(node->mac, wi_h->wi_addr1, MACSIZE);
  }

  /* if we got a broadcast address, this packet is junk */
  if((le16toh(wi_h->wi_frame_ctl) & WI_FCTL_STYPE) != WI_STYPE_MGMT_PROBEREQ &&
   (memcmp(bssid, NODE_BROADCAST, MACSIZE) == 0 ||
   memcmp(node->mac, NODE_BROADCAST, MACSIZE) == 0 ||
   (node->mac[0] != 0x0 && node->mac[1] != 0x8)))
    return -1;

  node->quality[2] = MAX(wi_h->wi_silence, 27);
  node->quality[1] = MAX(wi_h->wi_signal, 27);
  node->quality[0] = MAX(node->quality[1] - node->quality[2], 0);

  switch(le16toh(wi_h->wi_frame_ctl) & WI_FCTL_FTYPE)
  {
    case WI_FTYPE_DATA:
      /* if this is a data packet, set the amount of data sent */
      node->datalen = le16toh(wi_h->wi_dat_len);
      node->keyid = *(u_char *)(buf + WEP_KEYID_OFF);
      node->keyid = (node->keyid & 0xc0) >> 6;
      break;
    case WI_FTYPE_MGMT:
      switch(le16toh(wi_h->wi_frame_ctl) & WI_FCTL_STYPE)
      {
        case WI_STYPE_MGMT_PROBEREQ:
        case WI_STYPE_MGMT_ASREQ:
        case WI_STYPE_MGMT_REASREQ:
        {
          int len, off, i;
          struct wi_mgmt_var_hdr wi_var_h;

          if((le16toh(wi_h->wi_frame_ctl) & WI_FCTL_STYPE) ==
           WI_STYPE_MGMT_ASREQ)
          {
            struct wi_mgmt_asreq_hdr *asreq_h;
            asreq_h = (struct wi_mgmt_asreq_hdr *)data;

            node->weped = (le16toh(asreq_h->wi_capinfo) & WI_CAPINFO_PRIV) ?
             1 : 2;
            if(le16toh(asreq_h->wi_capinfo) & WI_CAPINFO_IBSS)
              node->adhoc = 1;
            if(le16toh(asreq_h->wi_capinfo) & WI_CAPINFO_ESS)
              node->adhoc = 2;
            node->interval = le16toh(asreq_h->wi_interval);

            off = sizeof(struct wi_mgmt_asreq_hdr);
          }
          else if((le16toh(wi_h->wi_frame_ctl) & WI_FCTL_STYPE) ==
           WI_STYPE_MGMT_REASREQ)
          {
            struct wi_mgmt_reasreq_hdr *reasreq_h;
            reasreq_h = (struct wi_mgmt_reasreq_hdr *)data;

            node->weped = (le16toh(reasreq_h->wi_capinfo) & WI_CAPINFO_PRIV) ?
             1 : 2;
            if(le16toh(reasreq_h->wi_capinfo) & WI_CAPINFO_IBSS)
              node->adhoc = 1;
            if(le16toh(reasreq_h->wi_capinfo) & WI_CAPINFO_ESS)
              node->adhoc = 2;
            node->interval = le16toh(reasreq_h->wi_interval);

            off = sizeof(struct wi_mgmt_reasreq_hdr);
          }
          else
            off = 0;

          len = le16toh(wi_h->wi_dat_len);

          while(off < len)
          {
            memcpy(&wi_var_h, data + off, 2);
            off += 2;
            memcpy(wi_var_h.wi_data, data + off, wi_var_h.wi_len);
            off += wi_var_h.wi_len;

            switch(wi_var_h.wi_code)
            {
              case WI_VAR_SSID:
                if((le16toh(wi_h->wi_frame_ctl) &
                 WI_FCTL_STYPE) == WI_STYPE_MGMT_PROBEREQ)
                  node_setany(node->mac, wi_var_h.wi_len ? 0 : 1);
                break;
              case WI_VAR_SRATES:
                for(i = 0; i < wi_var_h.wi_len; i++)
                  node->srate =
                   MAX(node->srate, (wi_var_h.wi_data[i] & WI_VAR_SRATES_MASK));
                if((le16toh(wi_h->wi_frame_ctl) &
                 WI_FCTL_STYPE) == WI_STYPE_MGMT_PROBEREQ)
                  node_setsrate(node->mac, node->srate);
                break;
              default:
                break;
            }
          }

          if((le16toh(wi_h->wi_frame_ctl) & WI_FCTL_STYPE) ==
           WI_STYPE_MGMT_PROBEREQ)
            return -1;

          break;
        }
        case WI_STYPE_MGMT_AUTH:
        {
          struct wi_mgmt_auth_hdr *auth_h;
          auth_h = (struct wi_mgmt_auth_hdr *)data;

          if(le16toh(auth_h->wi_algo))
            node->keyed = 1;
          else
            node->keyed = 2;

          break;
        }
        default:
          break;
      }
      break;
    default:
      break;
  }

  return 0;
}

void
mon_start(char *iface)
{
  char errbuf[PCAP_ERRBUF_SIZE];

  if((p = pcap_open_live(iface, PCAP_SNAPLEN, PCAP_PROMISC, PCAP_TOUT, errbuf))
   == NULL)
  {
    alert("error: unable to open pcap device: %s", strerror(errno));
    exitclean(2);
  }

  buf = NULL;
}

void
mon_stop(void)
{
  if(p != NULL)
    pcap_close(p);
}
