/*
 * dstumbler v1.0 [misc.c]
 * by h1kari - (c) Dachb0den Labs 2001
 */

/*
 * Copyright (c) 2001 Dachb0den Labs.
 *      David Hulton <h1kari@dachb0den.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by David Hulton.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY David Hulton AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL David Hulton OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <sys/types.h>
#include <errno.h>

#include "dstumbler.h"
#include "screen.h"

/*
 * resolve a mac address to a manufacturer name
 */
void
mfgresolve(char *mfg, char *mac)
{
  static struct mfg_s mfgs[] = MFG_INFO;
  int i, found = 0;

  for(i = 0; i < MFG_LEN; i++)
    if(memcmp(mac, mfgs[i].mac, MFG_MACSIZE) == 0) 
    {
      strncpy(mfg, mfgs[i].string, MFG_STRLEN);
      found++;
    }

  if(!found)
    strncpy(mfg, MFG_DEFAULT, MFG_STRLEN);
}

/*
 * check to see if this ap's ssid is a default or not
 */
int
defssid(char *ssid)
{
  static char *defssid[] = DEFSSID_INFO;
  int i;

  for(i = 0; i < DEFSSID_LEN; i++)
  {
    if(strncmp(ssid, defssid[i], strlen(ssid)) == 0)
      return 1;
  }

  return 0;
}

/*
 * smart redraw function so you can mark windows for redraw and draw them when
 * ready
 */
void
smart_redraw(void)
{
  int i = 0;

  if(redraw.all && ++i)
    refresh();
  else
  {
    if(redraw.ap_scr && ++i)
      wnoutrefresh(ap_scr);

    if(redraw.info_scr && ++i)
      wnoutrefresh(info_scr);

    if(redraw.graph_scr && ++i)
      wnoutrefresh(graph_scr);

    if(redraw.menu_scr && ++i)
      wnoutrefresh(menu_scr);

    if(redraw.alert_scr && ++i)
      wnoutrefresh(alert_scr);

    if(redraw.node_scr && ++i)
      wnoutrefresh(node_scr);
  }

  if(i)
  {
    if(!redraw.all)
      doupdate();

    redraw.ap_scr = redraw.info_scr = redraw.graph_scr = redraw.menu_scr =
     redraw.alert_scr = redraw.all = 0;
  }
}

/*
 * get the max cols and lines of the screen. more reliable than getmaxyx,
 * especially with window resizes.
 */
void
getscrmaxyx(void)
{
  struct winsize ws;

  if(ioctl(STDIN_FILENO, TIOCGWINSZ, &ws) < 0)
  {
    alert("error: unable to determine screen size: %s", strerror(errno));
    exitclean(2);
  }

  max_lines = ws.ws_row;
  max_cols  = ws.ws_col;
}
    

/*
 * print dstumbler banner on the alert screen
 */
void 
printbanner(char *banner)
{
  int i;

  werase(alert_scr);

  wattrset(alert_scr, BANNER_COLOR);
  for(i = 0; i < ((ALERT_SCR_COLS - 5) - strlen(banner)); i++)
    waddch(alert_scr, '-');
  waddch(alert_scr, '[');

  wattrset(alert_scr, BANNER_COLOR_BOLD);
  wprintw(alert_scr, " %s ", banner);

  wattrset(alert_scr, BANNER_COLOR);
  waddch(alert_scr, ']');
  redraw.alert_scr++;
}

/*
 * print alert to the alert screen and wait for user input.
 */
int
alert(const char *fmt, ...)
{
  int i, ans;
  char buffer[ALERT_SCR_COLS - 6];
  va_list ap;

  /* build buffer to print to alert window */
  va_start(ap, fmt);
  vsnprintf(buffer, ALERT_SCR_COLS - 6, fmt, ap);
  va_end(ap);

  werase(alert_scr);

  /* print alert */
  wattrset(alert_scr, ALERT_COLOR);
  for(i = 0; i < ((ALERT_SCR_COLS - 5) - strlen(buffer)); i++)
    waddch(alert_scr, '-');
  waddch(alert_scr, '[');

  wattrset(alert_scr, ALERT_COLOR_BOLD);
  wprintw(alert_scr, " %s ", buffer);

  wattrset(alert_scr, ALERT_COLOR);
  waddch(alert_scr, ']');

  redraw.alert_scr++;
  smart_redraw();

  /* wait for user to press any key to continue */
  nodelay(main_scr, FALSE);
  ans = getch();
  nodelay(main_scr, TRUE);

  /* now refresh alert screen with banner */
  PRINTBANNER();
  smart_redraw();

  return ans;
}
