/*
 * dstumbler v1.0 [dstumbler.h]
 * by h1kari - (c) Dachb0den Labs 2001
 */

/*
 * Copyright (c) 2001 Dachb0den Labs.
 *      David Hulton <h1kari@dachb0den.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by David Hulton.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY David Hulton AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL David Hulton OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <sys/types.h>
#ifdef __OpenBSD__
#include <sys/cdefs.h>
#include <sys/param.h>
#endif
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <net/if.h>
#include <sys/termios.h>

#include <bat/common.h>

#ifdef __OpenBSD__
#include <curses.h>
#else
#include <ncurses.h>
#endif

#ifdef DEBUG
#include "debug.h"
#endif

#include <pcap.h>

#include "config.h"

/*
 * definitions
 */
#define MIN_SCREEN_Y    24
#define MIN_SCREEN_X    78
#define SSIDLEN         32
#define MACSIZE         6
#define QUALITYSIZE     3
#define PREALLOC_APS    128
#define PREALLOC_LOG    256
#define PREALLOC_NODE   64
#define GRAPH_NORM_MAX  192
#define GRAPH_PRISM_MAX 192
#define WEP_KEYID_OFF   63
#define BANNER          "dstumbler v1.0 by h1kari - " \
 "(c) Dachb0den Labs 2001"
#define FILE_FMT        "%15s %17s %02d %02d:%02d:%02d %1d " \
 "%010ld.%010ld %010ld.%010ld\n"
#define NOTE_PROMPT     "Note:"
#define NOTE_BUFFERLEN  (MIN_SCREEN_X - 9)
#define FILE_PROMPT     "File:"
#define FILE_BUFFERLEN  (MIN_SCREEN_X - 9)
#define MENU \
 " [ basic navigation ]----\n" \
 " [+/-]: ap up/down\n" \
 " [</>]: node up/down\n" \
 " [u/d]: page ap up/down\n" \
 " [e/h]: end/home\n" \
 " [n/s]: newest/sort\n" \
 " [a/r]: autosel/resolve\n" \
 " [o/i]: nodes/audio\n" \
 " [m/k]: menu/refresh\n" \
 " [c/.]: chanlock/comment\n\n" \
 " [ file commands ]-------\n" \
 " [l/b]: load/backup\n" \
 " [q]:   quit\n"
#define EXIT_WIERR     -1

/*
 * macros
 */
#define PRINTBANNER()\
{\
  if(nodemode && aps_len && aps[aps_cur]->node_len &&\
   strlen(aps[aps_cur]->node[aps[aps_cur]->node_cur]->note))\
    printbanner(aps[aps_cur]->node[aps[aps_cur]->node_cur]->note);\
  else if(aps_len && strlen(aps[aps_cur]->note))\
    printbanner(aps[aps_cur]->note);\
  else\
    printbanner(BANNER);\
}

#define APCHANGE(x)\
{\
  aps_cur = x;\
  apchange++;\
  apnew = 0;\
  PRINTBANNER();\
}

#define NODECHANGE(x)\
{\
  aps[aps_cur]->node_cur = x;\
  nodechange++;\
  nodenew = 0;\
  PRINTBANNER();\
}

#define APNEW(x)\
{\
  aps_new = x;\
\
  if(autosel)\
  {\
    if(x != aps_lst)\
    {\
      apnew++;\
      apchange = 0;\
    }\
    aps_lst = x;\
  }\
}

#define NODENEW(x)\
{\
  node_new = x;\
\
  if(autosel)\
  {\
    if(x != node_lst)\
    {\
      nodenew++;\
      nodechange = 0;\
    }\
    node_lst = x;\
  }\
}

#define REDRAWAP() {\
  if(aps_len > 0)\
  {\
    redraw_aps();\
    redraw_apinfo();\
\
    if(!nodemode)\
      redraw_graph();\
  }\
}

#define REDRAWNODE() {\
  if(nodemode && aps_len && aps[aps_cur]->node_len)\
  {\
    redraw_nodes();\
    redraw_node_graph();\
  }\
}

#define ADDTVAL(x, y, z)\
{\
  y.tv_usec = x.tv_usec + z;\
  y.tv_sec = x.tv_sec + (y.tv_usec / 1000000);\
  y.tv_usec %= 1000000;\
}

#define CMPTVAL(x, y)\
 (x.tv_sec == y.tv_sec ?\
 (x.tv_usec == y.tv_usec ? 0 :\
 (x.tv_usec > y.tv_usec ? -1 : 1)) :\
 (x.tv_sec > y.tv_sec ? -1 : 1))

#ifndef MAX
#define MAX(x, y) (x > y ? x : y)
#endif
#ifndef MIN
#define MIN(x, y) (x < y ? x : y)
#endif
#define SWAP(x, y) { x ^= y; y ^= x; x ^= y; }

/*
 * define all of these as macros for now, for easy implementation of other
 * card driver symantics in the future.
 */
#define getssid(x, y) wi_getstat(x, WI_RID_CURRENT_SSID, y, SSIDLEN)
#define getbssid(x, y) wi_getstat(x, WI_RID_CURRENT_BSSID, (char *)y, MACSIZE)
#define getchan(x, y) wi_getstat(x, WI_RID_CURRENT_CHAN, (char *)y, 0)
#define getquality(x, y) wi_getstat(x, WI_RID_COMMS_QUALITY, (char *)y, 0)
#define getporttype(x, y) wi_getstat(x, WI_RID_PORTTYPE, (char *)y, 0)
#define getcrypt(x, y) wi_getstat(x, WI_RID_ENCRYPTION, (char *)y, 0)
#define getcryptkeys(x, y) wi_getstat(x, WI_RID_DEFLT_CRYPT_KEYS, (char *)y, 0)
#define getdefaultssid(x, y) wi_getstat(x, WI_RID_OWN_SSID, y, SSIDLEN)
#define getmacaddr(x, y) wi_getstat(x, WI_RID_MAC_NODE, (char *)y, MACSIZE)
#define getscanres(x, y) wi_getstat(x, WI_RID_SCAN_RES, (char *)y, \
 WI_MAX_DATALEN)
#define getprism2(x, y) wi_getstat(x, WI_RID_PRISM2, (char *)y, 0)
#define getdebugmonitor(x, y) wi_getdebug(x, WI_DEBUG_MONITOR, (char *)y)
#define setporttype(x, y) wi_setstat(x, WI_RID_PORTTYPE, (char *)y, 2)
#define setcrypt(x, y) wi_setstat(x, WI_RID_ENCRYPTION, (char *)y, 2)
#define setcryptkeys(x, y) wi_setstat(x, WI_RID_DEFLT_CRYPT_KEYS, (char *)y, \
 (sizeof(struct wi_ltv_keys) / 2) + 1)
#define setdefaultssid(x, y) wi_setstat(x, WI_RID_OWN_SSID, y, SSIDLEN)
#define setmacaddr(x, y) wi_setstat(x, WI_RID_MAC_NODE, (char *)y, MACSIZE)
#define setscanreq(x) wi_setstat(x, WI_RID_SCAN_REQ, (char *)NULL, 0)
#define setdebugstop(x) wi_setdebug(x, WI_DEBUG_STOPTEST, 0)
#define setdebugmonitor(x) wi_setdebug(x, WI_DEBUG_MONITOR, 0)
#define setdebugchan(x, y) wi_setdebug(x, WI_DEBUG_CHAN, y)
#define getgps(x, y) {\
 memcpy((char *)x, (char *)&ns, sizeof(struct gps_s));\
 memcpy((char *)y, (char *)&ew, sizeof(struct gps_s));\
}

/* redefine these for compatibility with freebsd/openbsd */
#ifdef __FreeBSD__
#define htole16(x) (x)
#define le16toh(x) (x)
#define O_SYNC     O_FSYNC
#else
#ifdef __OpenBSD__
#define le16toh(x) letoh16(x)
#endif
#endif

struct gps_s {
  char dir;
  float coordinates;
};

struct log_s {
  struct gps_s ns;
  struct gps_s ew;
  int quality[QUALITYSIZE];
  struct timeval seen;
};

struct node_s {
  struct gps_s ns;
  struct gps_s ew;
  u_char mac[MACSIZE];
  int quality[QUALITYSIZE];
  int datalen;
  int weped;
  int srate;
  int adhoc;
  int keyed;
  int any;
  int interval;
  int keyid;
  char note[NOTE_BUFFERLEN];
  struct timeval firstseen;
  struct timeval lastseen;
  struct log_s **log;
  int log_len;
  int log_max;
};

struct aps_s {
  struct gps_s ns;
  struct gps_s ew;
  char ssid[SSIDLEN];
  u_char bssid[MACSIZE];
  int chan;
  int quality[QUALITYSIZE];
  int weped;
  int srate;
  int adhoc;
  int keyed;
  int interval;
  int wepid;
  char note[NOTE_BUFFERLEN];
  struct timeval firstseen;
  struct timeval lastseen;
  struct node_s **node;
  int node_len;
  int node_max;
  int node_cur;
  struct log_s **log;
  int log_len;
  int log_max;
};

struct mfg_s {
  char mac[MFG_MACSIZE];
  char *string;
};

/*
 * declarations
 */
WINDOW *main_scr, *ap_scr, *info_scr, *graph_scr, *menu_scr, *alert_scr;
WINDOW *top_border, *top_right_border, *right_border, *bottom_border;
WINDOW *node_scr, *middle_border;
int max_lines, max_cols;
int node_scr_lines, middle_border_lines, menu_scr_cols, right_border_cols;
int autosel, resolvmfg, nodemode, audiomode, menumode;
int prism2, scanmode, monmode, chanlock, macreset, macint;
int ncurse, nodevice, usegps, uselog;
int apchange, apnew;
int aps_len, aps_cur, aps_new, aps_win, aps_max, aps_lst;
int nodechange, nodenew;
int node_new, node_win, node_lst;
int gps_fd, sp_fd;
int ch, sigwinch;
char *device, *gps, *logfile;
FILE *log_fd;
struct aps_s **aps;
struct gps_s ns, ew;

struct {
  int ap_scr;
  int info_scr;
  int graph_scr;
  int menu_scr;
  int alert_scr;
  int node_scr;
  int all;
} redraw;

struct {
  int wi_started;

  int monitor;
  int porttype;
  int crypt;
  u_char macaddr[MACSIZE];
  struct wi_ltv_keys keys;
  char defaultssid[SSIDLEN];

  int gps_started;
  struct termios ttyset;
} backup;

/*
 * main.c
 */
int main(int, char **);
void print_usage(const char *);
void start_loop(const char *);
void init_curses(void);
void draw_screen(void);
void init_vars(void);
void handle_sigwinch(int);
void refreshclean(void);
void exitclean(int);

/*
 * misc.c
 */
void mfgresolve(char *, char *);
int defssid(char *);
void smart_redraw(void);
void getscrmaxyx(void);
void printbanner(char *);
int alert(const char *, ...);

/*
 * aps.c
 */
void parse_ap(const char *);
int find_ap(const char *, const char *, int);
void redraw_aps(void);
void redraw_apinfo(void);
void redraw_graph(void);
void print_graph(int *, int);
void add_ap(const struct aps_s *);
void add_ap_log(struct aps_s *, const struct aps_s *);

/*
 * nodes.c
 */
void parse_node(const char *);
int find_node_ap(const char *);
int find_node(int, const char *);
void node_setany(u_char *, int);
void node_setsrate(u_char *, int);
void redraw_nodes(void);
void redraw_node_graph(void);
void add_node(int, const struct node_s *);
void add_node_log(int, struct node_s *, const struct node_s *);

/*
 * menu.c
 */
void parse_cmd(int);
void sort_aplist(void);
int aplistcmp(const void *, const void *);
void redraw_menu(void);
void redraw_toggle(int, int, char);
void load_aplist(void);
void backup_aplist(void);
void prompt(char *, char *, int);

/*
 * wistat.c
 */
int wi_parse_ap(const char *, struct aps_s *);
int wi_parse_scan_ap(const char *, struct aps_s *, int);
void wi_ioctl(const char *, struct wi_req *, int);
void wi_getstat(const char *, int, char *, int);
void wi_setstat(const char *, int, char *, int);
void wi_getdebug(const char *, int, char *);
void wi_setdebug(const char *, int, int);
void wi_strncpyval(char *, const char *, int);
void wi_setrandmacaddr(const char *);
void wi_start(const char *);
void wi_stop(const char *);

/*
 * gps.c
 */
int gps_open(const char *);
void gps_close(int);
void gps_parse(int);
void gps_start(const char *);
void gps_stop(void);

/*
 * mon.c
 */
int mon_next(const char *);
int mon_parse_ap(const char *, struct aps_s *);
int mon_parse_node(struct node_s *, char *);
void mon_start(char *);
void mon_stop(void);

/*
 * audio.c
 */
void audio_graph(int);
void audio_play(char *);

/*
 * log.c
 */
void log_ap(FILE *, struct aps_s *, struct log_s *);
void unlog_ap(FILE *, char *, struct aps_s *);
void log_node(FILE *, struct aps_s *, struct node_s *, struct log_s *);
void unlog_node(FILE *, char *, struct node_s *, u_char *);
FILE *log_start(char *);
void log_stop(FILE *);
