/*	$eterna: bozohttpd.c,v 1.112 2004/08/23 10:56:18 mrg Exp $	*/

/*
 * Copyright (c) 1997-2004 Matthew R. Green
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer and
 *    dedication in the documentation and/or other materials provided
 *    with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/* this program is dedicated to the Great God of Processed Cheese */

/*
 * bozohttpd.c:  minimal httpd; provides only these features:
 *	- HTTP/0.9 (by virtue of ..)
 *	- HTTP/1.0
 *	- HTTP/1.1
 *	- CGI/1.1 this will only be provided for "system" scripts
 *	- automatic "missing trailing slash" redirections
 *	- configurable translation of /~user/ to ~user/public_html,
 *	  however, this does not include cgi-bin support
 *	- access lists via libwrap via inetd/tcpd
 *	- virtual hosting
 *	- not that we do not even pretend to understand MIME, but
 *	  rely only on the HTTP specification
 *	- ipv6 support
 *	- automatic `index.html' generation
 *	- configurable server name
 *	- directory index generation
 *	- daemon mode (lacks libwrap support)
 *	- .htpasswd support
 */

/*
 * requirements for minimal http/1.1 (at least, as documented in
 * <draft-ietf-http-v11-spec-rev-06> which expired may 18, 1999):
 *
 *	- 14.15: content-encoding handling. [1]
 *
 *	- 14.16: content-length handling.  this is only a SHOULD header
 *	  thus we could just not send it ever.  [1]
 *
 *	- 14.17: content-type handling. [1]
 *
 *	- 14.25/28: if-{,un}modified-since handling.  maybe do this, but
 *	  i really don't want to have to parse 3 differnet date formats
 *
 * [1] need to revisit to ensure proper behaviour
 *
 * and the following is a list of features that we do not need
 * to have due to other limits, or are too lazy.  there are more
 * of these than are listed, but these are of particular note,
 * and could perhaps be implemented.
 *
 *	- 3.5/3.6: content/transfer codings.  probably can ignore
 *	  this?  we "SHOULD"n't.  but 4.4 says we should ignore a
 *	  `content-length' header upon reciept of a `transfer-encoding'
 *	  header.
 *
 *	- 5.1.1: request methods.  only MUST support GET and HEAD,
 *	  but there are new ones besides POST that are currently
 *	  supported: OPTIONS PUT DELETE TRACE and CONNECT, plus
 *	  extensions not yet known?
 *
 * 	- 10.1: we can ignore informational status codes
 *
 *	- 10.3.3/10.3.4/10.3.8:  just use '302' codes always.
 *
 *	- 14.1/14.2/14.3/14.27: we do not support Accept: headers..
 *	  just ignore them and send the request anyway.  they are
 *	  only SHOULD.
 *
 *	- 14.5/14.16/14.35: we don't do ranges.  from section 14.35.2
 *	  `A server MAY ignore the Range header'.  but it might be nice.
 *
 *	- 14.9: we aren't a cache.
 *
 *	- 14.15: content-md5 would be nice...
 *	
 *	- 14.24/14.26/14.27: be nice to support this...
 *
 *	- 14.44: not sure about this Vary: header.  ignore it for now.
 */

/*
 * optional features on a compile-time basis.
 */
#if 0
#undef DEBUG			/* include debugging support */
#undef NO_USER_SUPPORT		/* don't support /~user requests */
#undef NO_CGIBIN_SUPPORT	/* don't support cgi-bin requests */
#undef NO_DIRINDEX_SUPPORT	/* don't support directory indexing */
#undef NO_DAEMON_MODE		/* don't support daemon mode */
#undef NO_DYNAMIC_CONTENT	/* don't support dynamic content updates */
#undef DO_HTPASSWD		/* support .htpasswd files */
#undef NO_SSL_SUPPORT		/* don't support ssl (https) */
#endif

#ifndef INDEX_HTML
#define INDEX_HTML		"index.html"
#endif
#ifndef PUBLIC_HTML
#define PUBLIC_HTML		"public_html"
#endif
#ifndef SERVER_SOFTWARE
#define SERVER_SOFTWARE		"bozohttpd/20040823"
#endif
#ifndef AUTH_FILE
#define AUTH_FILE		".htpasswd"
#endif
#ifndef DIRECT_ACCESS_FILE
#define DIRECT_ACCESS_FILE	".bzdirect"
#endif
#ifndef REDIRECT_FILE
#define REDIRECT_FILE		".bzredirect"
#endif
#ifndef ABSREDIRECT_FILE
#define ABSREDIRECT_FILE	".bzabsredirect"
#endif

/*
 * And so it begins ..
 */

#include <sys/param.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <sys/wait.h>

#include <netinet/in.h>

#include <arpa/inet.h>

#ifndef NO_SSL_SUPPORT
#include <openssl/ssl.h>
#include <openssl/err.h>
#endif /* NO_SSL_SUPPORT */

#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#ifndef NO_CGIBIN_SUPPORT
#include <paths.h>
#endif /* NO_CGIBIN_SUPPORT */
#ifndef NO_DAEMON_MODE
#include <poll.h>
#endif /* NO_DAEMON_MODE */
#include <pwd.h>
#include <grp.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#ifndef __attribute__
#define __attribute__(x)
#endif /* __attribute__ */

#include "queue.h"

#ifndef MAX_WAIT_TIME
#define	MAX_WAIT_TIME	60	/* hang around for 60 seconds max */
#endif

/* headers */
struct headers {
	const char *h_header;
	const char *h_value;
	SIMPLEQ_ENTRY(headers)	h_next;
};

/* http_req */
typedef struct {
	int	hr_method;
#define	HTTP_GET	0x01
#define HTTP_POST	0x02
#define HTTP_HEAD	0x03
#define HTTP_OPTIONS	0x04	/* not supported */
#define HTTP_PUT	0x05	/* not supported */
#define HTTP_DELETE	0x06	/* not supported */
#define HTTP_TRACE	0x07	/* not supported */
#define HTTP_CONNECT	0x08	/* not supported */
	const char *hr_methodstr;
	char	*hr_url;
	const char *hr_proto;
	const char *hr_content_type;
	const char *hr_content_length;
	const char *hr_allow;
	const char *hr_host;		/* HTTP/1.1 Host: */
	const char *hr_referrer;
	const char *hr_remotehost;
	const char *hr_remoteaddr;
	const char *hr_serverport;
#ifdef DO_HTPASSWD
	const char *hr_authrealm;
	const char *hr_authuser;
	const char *hr_authpass;
#endif
	SIMPLEQ_HEAD(, headers)	hr_headers;
	int	hr_nheaders;
} http_req;

/* debug flags */
#define DEBUG_NORMAL	1
#define DEBUG_FAT	2
#define DEBUG_OBESE	3
#define DEBUG_EXPLODING	4

#define WRSZ	(64 * 1024)

/* variables and functions */

#ifndef NO_CGIBIN_SUPPORT
static	char	*cgibin;	/* cgi-bin directory */
static	int	Cflag;		/* added a cgi handler, always process_cgi() */

#define CGIBIN_PREFIX		"cgi-bin/"
#define CGIBIN_PREFIX_LEN	(sizeof(CGIBIN_PREFIX)-1)

static	void	process_cgi(http_req *);
static	void	finish_cgi_output(http_req *request, int, int);
static	const char *content_cgihandler(http_req *, const char *);
#endif /* NO_CGIBIN_SUPPORT */


#ifndef NO_USER_SUPPORT
static	int	uflag;		/* allow /~user/ translation */
static	const char *public_html	= PUBLIC_HTML;

static	char	*transform_user(http_req *, int *);
#endif /* NO_USER_SUPPORT */


#ifndef NO_DIRINDEX_SUPPORT
static	int	Xflag;		/* do directory indexing */
static	int	Hflag;		/* hide .* */

static	void	directory_index(http_req *, const char *);
#endif /* NO_DIRINDEX_SUPPORT */


static	int	bflag;		/* background; drop into daemon mode */
static	int	fflag;		/* keep daemon mode in foreground */
static	int	eflag;		/* don't clean environ; -t/-U only */
#ifndef NO_DAEMON_MODE
static	char	*iflag;		/* bind address; default INADDR_ANY */
static	const char *Iflag = "http";/* bind port; default "http" */
static	int	Iflag_set = 0;
static	int	*sock;		/* bound sockets */
static	int	nsock;		/* number of above */

static	void	sigchild(int);	/* SIGCHLD handler */
#endif /* NO_DAEMON_MODE */

static	int	dflag = 0;	/* debugging level */

#ifdef DEBUG
static	void	debug__(int, const char *, ...)
			__attribute__((__format__(__printf__, 2, 3)));
#define debug(x)	debug__ x
#else
#define	debug(x)	
#endif /* DEBUG */

#define	strornull(x)	((x) ? (x) : "<null>")

#ifndef LOG_FTP
#define LOG_FTP LOG_DAEMON
#endif

static	char	*tflag;		/* root directory */
static	char 	*Uflag;		/* user name to switch to */
static	int	nflag;		/* avoid gethostby*() */
static	int	Hflag;		/* avoid gethostby*() */
static	int	rflag;		/* make sure referrer = me unless url = / */
static	int	sflag = 0;	/* log to stderr even if it is not a TTY */
#ifndef NO_SSL_SUPPORT
static	int	Zflag = 0;	/* default SSL support off */
#endif /* NO_SSL_SUPPORT */
static	char	*myname;	/* my name */
static	char	*slashdir;	/* www slash directory */
static	char	*vpath;		/* virtual directory base */

static	char	date[40];	/* text date string */

#ifndef NO_SSL_SUPPORT
static	SSL_CTX		*ssl_context = NULL;
static	SSL_METHOD	*ssl_method = NULL;
static	SSL		*bozossl = NULL;		/* XXX initialized in child */
#endif /* NO_SSL_SUPPORT */

static	const char *server_software = SERVER_SOFTWARE;
static	const char *index_html = INDEX_HTML;
static	const char http_09[] = "HTTP/0.9";
static	const char http_10[] = "HTTP/1.0";
static	const char http_11[] = "HTTP/1.1";
static	const char text_plain[]	= "text/plain";

static	void	usage(void);
static	void	alarmer(int);
volatile sig_atomic_t	alarmhit;

static	void	parse_request(char *, char **, char **, char **);
static	http_req *read_request(void);
static	void	process_request(http_req *);
static	void	check_special_files(http_req *, const char *);
static	int	check_direct_access(http_req *request);
static	char	*transform_request(http_req *, int *);
static	void	handle_redirect(http_req *, const char *, int);
static	void	print_header(http_req *, struct stat *, const char *,
			     const char *);

static	void	check_virtual(http_req *);
static	void	check_bzredirect(http_req *);
static	void	fix_url_percent(http_req *);
static	void	process_method(http_req *, const char *);
static	void	process_proto(http_req *, const char *);
static	struct content_map *match_content_map(const char *, int);
static	const char *content_type(http_req *, const char *);
static	const char *content_encoding(http_req *, const char *);
static	void	escape_html(http_req *);
#ifndef NO_DYNAMIC_CONTENT
#ifndef NO_CGIBIN_SUPPORT
static	void	add_content_map_cgi(char *, char *);
#endif
static	void	add_content_map_mime(char *, char *, char *, char *);
static	struct content_map *get_content_map(const char *);
#endif

static	void	set_http_date(void);

static	void	warning(const char *, ...)
			__attribute__((__format__(__printf__, 1, 2)))
			;
static	void	error(int, const char *, ...)
			__attribute__((__format__(__printf__, 2, 3)))
			;

static	void	http_error(int, http_req *, const char *);
static	const char *http_errors_short(int);
static	const char *http_errors_long(int);

static	char	*dgetln(int, ssize_t *, ssize_t	(*)(int, void *, size_t));
static	char	*strnsep(char **, const char *, ssize_t *);

static	void	*bozomalloc(size_t);
static	void	*bozorealloc(void *, size_t);
static	char	*bozostrdup(const char *);

/* bozotic io */
static	int	(*bozoprintf)(const char *, ...) = printf;
static	ssize_t	(*bozoread)(int, void *, size_t) = read;
static	ssize_t	(*bozowrite)(int, const void *, size_t) = write;
static	int	(*bozoflush)(FILE *) = fflush;
#ifndef NO_SSL_SUPPORT
int     ssl_printf(const char *, ...);
ssize_t ssl_read(int, void *, size_t);
ssize_t ssl_write(int, const void *, size_t);
int	ssl_flush(FILE *);
#endif /* NO_SSL_SUPPORT */

#ifdef DO_HTPASSWD
static	void	check_auth(http_req *, const char *);
static	ssize_t	base64_decode(const unsigned char *, size_t,
			    unsigned char *, size_t);
#else /* DO_HTPASSWD */
#define		check_auth(x, y)	/* nothing */
#endif /* DO_HTPASSWD */

	char	*progname;

	int	main(int, char *[]);

static void
usage(void)
{

	warning("usage: %s [options] slashdir [myname]", progname);
	warning("options:");
#ifdef DEBUG
	warning("   -d\t\t\tenable debug support");
#endif
	warning("   -s\t\t\talways log to stderr");
#ifndef NO_USER_SUPPORT
	warning("   -u\t\t\tenable ~user/public_html support");
	warning("   -p dir\t\tchange `public_html' directory name]");
#endif
#ifndef NO_DYNAMIC_CONTENT
	warning("   -M arg t c c11\tadd this mime extenstion");
#endif
#ifndef NO_CGIBIN_SUPPORT
#ifndef NO_DYNAMIC_CONTENT
	warning("   -C arg prog\t\tadd this CGI handler");
#endif
	warning("   -c cgibin\t\tenable cgi-bin support in this directory");
	warning("   -a\t\t\tdo not send headers for CGI scripts");
#endif
#ifndef NO_DAEMON_MODE
	warning("   -b\t\t\tbackground and go into daemon mode");
	warning("   -f\t\t\tkeep daemon mode in the foreground");
	warning("   -i address\t\tbind on this address (daemon mode only)");
	warning("   -I port\t\tbind on this port (daemon mode only)");
#endif
	warning("   -S version\t\tset server version string");
	warning("   -t dir\t\tchroot to `dir'");
	warning("   -U username\t\tchange user to `user'");
	warning("   -e\t\t\tdon't clean the environment (-t and -U only)");
	warning("   -v virtualroot\tenable virtual host support in this directory");
	warning("   -r\t\t\tmake sure sub-pages come from this host via referrer");
#ifndef NO_DIRINDEX_SUPPORT
	warning("   -X\t\t\tenable automatic directory index support");
	warning("   -H\t\t\thide files starting with a period (.) in index mode");
#endif
	warning("   -x index\t\tchange default `index.html' file name");
#ifndef NO_SSL_SUPPORT
	warning("   -Z cert privkey\tspecify path to server certificate and private key file\n"
		"\t\t\tfile in pem format and enable bozohttpd in SSL mode");
	warning("   -z keyfile\t\tspecify path to private key file in pem format");
#endif /* NO_SSL_SUPPORT */
	error(1, "%s failed to start", progname);
}

int
main(int argc, char *argv[])
{
	http_req *http_request;
	extern	char **environ;
	char	*cleanenv[1];
	uid_t	uid;
	int	c;
#ifndef NO_SSL_SUPPORT
	char	*certificate_file = NULL;
	char	*privatekey_file = NULL;
#endif /* NO_SSL_SUPPORT */

	if ((progname = strrchr(argv[0], '/')) != NULL)
		progname++;
	else
		progname = argv[0];

	openlog(progname, LOG_PID|LOG_NDELAY, LOG_FTP);
	set_http_date();

	while ((c = getopt(argc, argv, "C:HI:M:S:U:XZ:bc:defhi:np:rst:uv:x:z:")) != -1) {
		switch(c) {

		case 'M':
#ifndef NO_DYNAMIC_CONTENT
			/* make sure there's four arguments */
			if (argc - optind < 3)
				usage();
			add_content_map_mime(optarg, argv[optind],
			    argv[optind+1], argv[optind+2]);
			optind += 3;
			break;
#else
			error(1, "dynmic mime content support is not enabled");
			/* NOTREACHED */
#endif /* NO_DYNAMIC_CONTENT */

		case 'n':
			nflag = 1;
			break;

		case 'r':
			rflag = 1;
			break;

		case 's':
			sflag = 1;
			break;

		case 'S':
			server_software = optarg;
			break;
		case 'Z':
#ifndef NO_SSL_SUPPORT
			/* make sure there's two arguments */
			if (argc - optind < 1)
				usage();
			certificate_file = optarg;
			privatekey_file = argv[optind++];
			warning("using certificate file: %s", certificate_file);
			warning("using private key file: %s", privatekey_file);
			Zflag++;
			if (Iflag_set == 0)
				Iflag = "https";
			bozoprintf = ssl_printf;
			bozoread = ssl_read;
			bozowrite = ssl_write;
			bozoflush = ssl_flush;
			break;
#else
			error(1, "ssl support is not enabled");
			/* NOT REACHED */
#endif /* NO_SSL_SUPPORT */
		case 'U':
			Uflag = optarg;
			break;

		case 'v':
			vpath = optarg;
			break;

		case 'x':
			index_html = optarg;
			break;

#ifndef NO_DAEMON_MODE
		case 'b':
			bflag = 1;
			break;

		case 'e':
			eflag = 1;
			break;

		case 'f':
			fflag = 1;
			break;

		case 'i':
			iflag = optarg;
			break;

		case 'I':
			Iflag_set = 1;
			Iflag = optarg;
			break;
#else /* NO_DAEMON_MODE */
		case 'b':
		case 'i':
		case 'I':
			error(1, "Daemon mode is not enabled");
			/* NOTREACHED */
#endif /* NO_DAEMON_MODE */

#ifndef NO_CGIBIN_SUPPORT
		case 'c':
			cgibin = optarg;
			debug((DEBUG_OBESE, "cgibin (cgi-bin directory) is %s", cgibin));
			break;

		case 'C':
#ifndef NO_DYNAMIC_CONTENT
			Cflag = 1;
			/* make sure there's two arguments */
			if (argc - optind < 1)
				usage();
			add_content_map_cgi(optarg, argv[optind++]);
			break;
#else
			error(1, "dynmic CGI handler support is not enabled");
			/* NOTREACHED */
#endif /* NO_DYNAMIC_CONTENT */

#else
		case 'c':
		case 'C':
			error(1, "CGI is not enabled");
			/* NOTREACHED */
#endif /* NO_CGIBIN_SUPPORT */

		case 'd':
			dflag++;
#ifndef DEBUG
			if (dflag == 1)
				warning("Debugging is not enabled");
#endif /* !DEBUG */
			break;

#ifndef NO_USER_SUPPORT
		case 'p':
			public_html = optarg;
			break;

		case 't':
			tflag = optarg;
			break;

		case 'u':
			uflag = 1;
			break;
#else
		case 'u':
		case 'p':
			error(1, "User support is not enabled");
			/* NOTREACHED */
#endif /* NO_USER_SUPPORT */

#ifndef NO_DIRINDEX_SUPPORT
		case 'H':
			Hflag = 1;
			break;

		case 'X':
			Xflag = 1;
			break;

#else
		case 'H':
		case 'X':
			error(1, "directory indexing is not enabled");
			/* NOTREACHED */
#endif /* NO_DIRINDEX_SUPPORT */

		default:
			usage();
			/* NOTREACHED */
		}
	}
	argc -= optind;
	argv += optind;

	if (argc == 1) {
		myname = bozomalloc(MAXHOSTNAMELEN+1);
		/* XXX we do not check for FQDN here */
		if (gethostname(myname, MAXHOSTNAMELEN+1) < 0)
			error(1, "gethostname");
		myname[MAXHOSTNAMELEN] = '\0';
	} else if (argc == 2)
		myname = argv[1];
	else
		usage();
	
	slashdir = argv[0];
	debug((DEBUG_OBESE, "myname is %s, slashdir is %s", myname, slashdir));

#ifndef NO_SSL_SUPPORT
	/*
	* If we're running in SSL mode intialize SSL support.
	*/
	if (Zflag) {
		SSL_library_init();
		SSL_load_error_strings();
	
		ssl_method = SSLv23_server_method();
		ssl_context = SSL_CTX_new(ssl_method);

		/* XXX we need to learn how to check the SSL stack for more info */
		if (ssl_context == NULL)
			error(1, "SSL context initialization failed.");

		SSL_CTX_use_certificate_file(ssl_context, certificate_file,
		    SSL_FILETYPE_PEM);
		SSL_CTX_use_PrivateKey_file(ssl_context, privatekey_file,
		    SSL_FILETYPE_PEM);

		/* check consistency of key vs certificate */
		if (!SSL_CTX_check_private_key(ssl_context))
			error(1, "check private key failed");
	}
#endif /* NO_SSL_SUPPORT */

#ifndef NO_DAEMON_MODE
	/*
	 * deal with daemon mode.
	 */
	if (bflag) {
		struct addrinfo h, *r, *r0;
		int e, i, on = 1;

		if (fflag == 0)
			daemon(1, 0);

		warning("started in daemon mode as `%s' port `%s' root `%s'",
		    myname, Iflag, slashdir);
		
#if 0
		if (iflag == NULL)
			iflag = myname;
#endif
		memset(&h, 0, sizeof h);
		h.ai_family = PF_UNSPEC;
		h.ai_socktype = SOCK_STREAM;
		h.ai_flags = AI_PASSIVE;
		e = getaddrinfo(iflag, Iflag, &h, &r0);
		if (e)
			error(1, "getaddrinfo([%s]:%s): %s",
			    iflag ? iflag : "*", Iflag, gai_strerror(e));
		for (r = r0; r != NULL; r = r->ai_next)
			nsock++;
		sock = bozomalloc(nsock * sizeof *sock);
		for (i = 0, r = r0; r != NULL; r = r->ai_next) {
			sock[i] = socket(r->ai_family, SOCK_STREAM, 0);
			if (sock[i] == -1)
				continue;
			if (setsockopt(sock[i], SOL_SOCKET, SO_REUSEADDR, &on,
			    sizeof(on)) == -1)
				warning("setsockopt SO_REUSEADDR: %s",
				    strerror(errno));
			if (bind(sock[i], r->ai_addr, r->ai_addrlen) == -1)
				continue;
			if (listen(sock[i], SOMAXCONN) == -1)
				continue;
			i++;
		}
		if (i == 0)
			error(1, "could not find any addresses to bind");
		nsock = i;
		freeaddrinfo(r0);

		signal(SIGCHLD, sigchild);
	}
#endif /* NO_DAEMON_MODE */

	/*
	 * prevent info leakage between different compartments.
	 * some PATH values in the environment would be invalided
	 * by chroot. cross-user settings might result in undesirable
	 * effects.
	 */
	if ((tflag != NULL || Uflag != NULL) && !eflag) {
		cleanenv[0] = NULL;
		environ = cleanenv;
	}
	
	/*
	 * look up user/group information.
	 */
	if (Uflag != NULL) {
		struct passwd *pw;

		if ((pw = getpwnam(Uflag)) == NULL)
			error(1, "getpwnam(%s): %s", Uflag, strerror(errno));
		if (initgroups(pw->pw_name, pw->pw_gid) == -1)
			error(1, "initgroups: %s", strerror(errno));
		if (setgid(pw->pw_gid) == -1)
			error(1, "setgid(%u): %s", pw->pw_gid, strerror(errno));
		uid = pw->pw_uid;
	}

	/*
	 * handle chroot.
	 */
	if (tflag != NULL) {
		if (chdir(tflag) == -1)
			error(1, "chdir(%s): %s", tflag, strerror(errno));
		if (chroot(tflag) == -1)
			error(1, "chroot(%s): %s", tflag, strerror(errno));
	}

	if (Uflag != NULL)
		if (setuid(uid) == -1)
			error(1, "setuid(%d): %s", uid, strerror(errno));

	/*
	 * be sane, don't start serving up files from a
	 * hierarchy we don't have permission to get to.
	 */
	if (tflag != NULL)
		if (chdir("/") == -1)
			error(1, "chdir /: %s", strerror(errno));

	/*
	 * read and process the HTTP request.
	 */
	do {
		http_request = read_request();
		if (http_request) {
			process_request(http_request);
			return (0);
		}
	} while (bflag);

	return (0);
}

static void
set_http_date()
{
	struct	tm *tm;
	time_t	now;

	/* Sun, 06 Nov 1994 08:49:37 GMT */
	now = time(NULL);
	tm = gmtime(&now);	/* HTTP/1.1 spec rev 06 sez GMT only */
	strftime(date, sizeof date, "%a, %d %b %Y %H:%M:%S GMT", tm);
}

/*
 * convert "in" into the three parts of a request (first line)
 */
static void
parse_request(in, method, url, proto)
	char	*in;
	char	**method;
	char	**url;
	char	**proto;
{
	ssize_t	len;
	char	*val;
	
	*method = *url = *proto = NULL;		/* set them up */

	len = (ssize_t)strlen(in);
	val = strnsep(&in, " \t\n\r", &len);
	if (len < 1 || val == NULL)
		return;
	*method = val;
	while (*in == ' ' || *in == '\t')
		in++;
	val = strnsep(&in, " \t\n\r", &len);
	if (len < 1) {
		if (len == 0)
			*url = val;
		else
			*url = in;
		return;
	}
	*url = val;
	if (in) {
		while (*in && (*in == ' ' || *in == '\t'))
			in++;
		if (*in)
			*proto = in;
	}
}

/*
 * send a HTTP/1.1 408 response if we timeout.
 */
/* ARGSUSED */
static void
alarmer(sig)
	int sig;
{

	alarmhit = 1;
}

/*
 * This function reads a http request from stdin, returning a pointer to a
 * http_req structure, describing the request.
 */
static http_req *
read_request()
{
	struct	sigaction	sa;
	char	*str, *val, *method, *url, *proto;
	char	*host, *addr, *port;
	char	bufport[10];
	char	hbuf[NI_MAXHOST], abuf[NI_MAXSERV];
	struct	sockaddr_storage ss;
	ssize_t	len;
	int	line = 0;
	socklen_t slen;
	http_req *request;
#ifndef NO_DAEMON_MODE
	static struct pollfd *fds;

	while (bflag) {
		int fd;
		int i;

#ifndef POLLRDNORM
#define POLLRDNORM 0
#endif
#ifndef POLLRDBAND
#define POLLRDBAND 0
#endif
#ifndef INFTIM
#define INFTIM -1
#endif
		if (fds == NULL) {
			fds = bozomalloc(nsock * sizeof *fds);
			for (i = 0; i < nsock; i++) {
				fds[i].events = POLLIN | POLLPRI | POLLRDNORM |
						POLLRDBAND | POLLERR;
				fds[i].fd = sock[i];
			}
		}

		/*
		 * wait for a connection, then fork() and return NULL in
		 * the parent, who will come back here waiting for another
		 * connection.  read the request in in the child, and return
		 * it, for processing.
		 */
again:
		if (poll(fds, nsock, INFTIM) == -1) {
			if (errno != EINTR)
				error(1, "poll: %s", strerror(errno));
			goto again;
		}

		for (i = 0; i < nsock; i++) {
			if (fds[i].revents & (POLLNVAL|POLLERR|POLLHUP)) {
				warning("poll on fd %d: %s", fds[i].fd,
				    strerror(errno));
				continue;
			}
			if (fds[i].revents == 0)
				continue;

			fd = accept(sock[i], (struct sockaddr *)&ss, &slen);
			set_http_date();	/* we were sleeping in accept() */
			if (fd == -1) {
				if (errno != EAGAIN)
					error(1, "accept: %s", strerror(errno));
				return NULL;
			}
			switch (fork()) {
			case -1: /* eep, failure */
				warning("fork() failed, sleeping for 10 seconds: %s",
				    strerror(errno));
				sleep(10);
				return NULL;

			case 0: /* child */
				/* setup stdin/stdout/stderr */
				dup2(fd, 0);
				dup2(fd, 1);
				/*dup2(fd, 2);*/
				close(fd);
				goto child;

			default: /* parent */
				close(fd);
				continue;
			}
		}
	}
child:
#endif /* NO_DAEMON_MODE */

#ifndef NO_SSL_SUPPORT
	/*
	 * If in SSL mode setup SSL structure w/ default context.
	 */
	if (Zflag) {
		bozossl = SSL_new(ssl_context); /* XXX global sucks */
		SSL_set_rfd(bozossl, 0);
		SSL_set_wfd(bozossl, 1);
		SSL_accept(bozossl);
	}
#endif /* NO_SSL_SUPPORT */

	request = bozomalloc(sizeof *request);
	memset(request, 0, sizeof *request);
	request->hr_allow = request->hr_host = NULL;
	request->hr_content_type = request->hr_content_length = NULL;

	slen = sizeof(ss);
	if (getpeername(0, (struct sockaddr *)&ss, &slen) < 0)
		host = addr = NULL;
	else {
		if (getnameinfo((struct sockaddr *)&ss, slen,
		    abuf, sizeof abuf, NULL, 0, NI_NUMERICHOST) == 0)
			addr = abuf;
		else
			addr = NULL;
		if (nflag == 0 && getnameinfo((struct sockaddr *)&ss, slen,
		    hbuf, sizeof hbuf, NULL, 0, 0) == 0)
			host = hbuf;
		else
			host = NULL;
	}
	if (host != NULL)
		request->hr_remotehost = bozostrdup(host);
	if (addr != NULL)
		request->hr_remoteaddr = bozostrdup(addr);
	slen = sizeof(ss);
	if (getsockname(0, (struct sockaddr *)&ss, &slen) < 0)
		port = NULL;
	else {
		if (getnameinfo((struct sockaddr *)&ss, slen, NULL, 0,
		    bufport, sizeof bufport, NI_NUMERICSERV) == 0)
			port = bufport;
		else
			port = NULL;
	}
	if (port != NULL)
		request->hr_serverport = bozostrdup(port);

	/*
	 * setup a timer to make sure the request is not hung
	 */
	sa.sa_handler = alarmer;
	sigemptyset(&sa.sa_mask);
	sigaddset(&sa.sa_mask, SIGALRM);
	sa.sa_flags = 0;
	(void)sigaction(SIGALRM, &sa, NULL);	/* XXX */

	alarm(MAX_WAIT_TIME);
	while ((str = dgetln(STDIN_FILENO, &len, bozoread)) != NULL) {
		alarm(0);
		if (alarmhit)
			http_error(408, NULL, "request timed out");
		line++;
		str = bozostrdup(str);	/* we use this copy */

		if (line == 1) {
			if (len < 1)
				http_error(404, NULL, "null method");
			warning("got request ``%s'' from host %s to port %s",
			    str,
			    host ? host : addr ? addr : "<local>",
			    port ? port : "<stdin>");
			debug((DEBUG_FAT, "read_req, getting request: ``%s''",
			    str));

			parse_request(str, &method, &url, &proto);
			if (method == NULL)
				http_error(404, NULL, "null method");
			if (url == NULL)
				http_error(404, NULL, "null url");

			/*
			 * note that we parse the proto first, so that we
			 * can more properly parse the method and the url.
			 */
			request->hr_url = url;
			process_proto(request, proto);
			process_method(request, method);

			/* http/0.9 has no header processing */
			if (request->hr_proto == http_09)
				break;
		} else {		/* incoming headers */
			struct	headers *hdr;

			if (*str == '\0')
				break;

			val = strnsep(&str, ":", &len);
			debug((DEBUG_EXPLODING,
			    "read_req2: after strnsep: str ``%s'' val ``%s''",
			    str, val));
			if (val == NULL || len == -1)
				http_error(404, request, "no header");
			while (*str == ' ' || *str == '\t')
				len--, str++;

#ifdef DO_HTPASSWD
			if (strcasecmp(val, "authorization") == 0 &&
			    strncasecmp(str, "Basic ", 6) == 0) {
				char	authbuf[BUFSIZ];
				char	*pass;
				ssize_t	alen;

				alen = base64_decode(str + 6, len - 6,
				    authbuf, sizeof(authbuf) - 1);
				if (alen != -1)
					authbuf[alen] = '\0';
				if (alen == -1 ||
				    (pass = strchr(authbuf, ':')) == NULL)
					http_error(400, request,
					    "bad authorization field");
				*pass++ = '\0';
				request->hr_authuser = bozostrdup(authbuf);
				request->hr_authpass = bozostrdup(pass);
				debug((DEBUG_FAT,
				    "decoded authorization `%s' as `%s':`%s'",
				    str, request->hr_authuser, request->hr_authpass));
					/* don't store in request->headers */
				goto nextheader;
			}
#endif /* DO_HTPASSWD */

			/*
			 * XXX: http apparently requires that headers like
			 *	foo: bar
			 *	foo: baz
			 * be parsed as if they were
			 *	foo: bar baz
			 * including having other headers in beteen.  FIX ME.
			 */
			hdr = bozomalloc(sizeof *hdr);
			hdr->h_header = val;
			if (str && *str)
				hdr->h_value = str;
			else
				hdr->h_value = " ";

			SIMPLEQ_INSERT_TAIL(&request->hr_headers, hdr, h_next);
			request->hr_nheaders++;

			if (strcasecmp(hdr->h_header, "content-type") == 0)
				request->hr_content_type = hdr->h_value;
			else if (strcasecmp(hdr->h_header, "content-length") == 0)
				request->hr_content_length = hdr->h_value;
			else if (strcasecmp(hdr->h_header, "host") == 0)
				request->hr_host = hdr->h_value;
			/* HTTP/1.1 rev06 draft spec: 14.20 */
			else if (strcasecmp(hdr->h_header, "expect") == 0)
				http_error(417, request, "we don't support Expect:");
			else if (strcasecmp(hdr->h_header, "referrer") == 0 ||
			         strcasecmp(hdr->h_header, "referer") == 0)
				request->hr_referrer = hdr->h_value;

			debug((DEBUG_FAT, "adding header %s: %s",
			    hdr->h_header, hdr->h_value));
		}
#ifdef DO_HTPASSWD
nextheader:
#endif /* DO_HTPASSWD */
		alarm(MAX_WAIT_TIME);
	}

	/* now, clear it all out */
	alarm(0);
	(void)signal(SIGALRM, SIG_DFL);

	/* RFC1945, 8.3 */
	if (request->hr_method == HTTP_POST && request->hr_content_length == NULL)
		http_error(400, request, "missing content length");

	/* HTTP/1.1 draft rev-06, 14.23 & 19.6.1.1 */
	if (request->hr_proto == http_11 && request->hr_host == NULL)
		http_error(400, request, "missing Host header");

	debug((DEBUG_FAT, "read_request returns url %s in request", request->hr_url));
	return (request);
}

/*
 * process_request does the following:
 *	- check the request is valid
 *	- process cgi-bin if necessary
 *	- transform a filename if necesarry
 *	- return the HTTP request
 */
static void
process_request(request)
	http_req *request;
{
	struct	stat sb;
	char	*file;
	const char *type, *encoding;
	int	fd, isindex;

	/*
	 * note that transform_request chdir()'s if required.  also note
	 * that cgi is handed here, and a cgi request will never return
	 * back here.
	 */
	file = transform_request(request, &isindex);
	if (file == NULL)
		http_error(404, request, "empty file after transform");

	fd = open(file, O_RDONLY);
	if (fd < 0) {
		debug((DEBUG_FAT, "open failed: %s", strerror(errno)));
		if (errno == EPERM)
			http_error(403, request, "no permission to open file");
		else if (errno == ENOENT) {
#ifndef NO_DIRINDEX_SUPPORT
			if (isindex && Xflag) {
				const char *name;

				if (strlen(file) <= strlen(index_html))
					name = ".";
				else {
					file[strlen(file) - strlen(index_html)] = '\0';
					name = file;
				}
				debug((DEBUG_FAT, "trying dir index (%s)",
				    name));
				directory_index(request, name);
				return;
			}
#endif /* NO_DIRINDEX_SUPPORT */
			http_error(404, request, "no file");
		} else
			http_error(500, request, "open file");
	}
	if (fstat(fd, &sb) < 0)
		http_error(500, request, "can't fstat");
	if (S_ISDIR(sb.st_mode))
		handle_redirect(request, NULL, 0);
		/* NOTREACHED */
	/* XXX RFC1945 10.9 If-Modified-Since (http code 304) */

	(void)bozoprintf("%s 200 OK\r\n", request->hr_proto);

	if (request->hr_proto != http_09) {
		type = content_type(request, file);
		encoding = content_encoding(request, file);

		print_header(request, &sb, type, encoding);
		(void)bozoprintf("\r\n");
	}
	(void)bozoflush(stdout);

	if (request->hr_method != HTTP_HEAD) {
		char *addr;
		void *oaddr;
		off_t sz = sb.st_size;

		oaddr = addr = mmap(0, (size_t)sz, PROT_READ,
		    MAP_SHARED, fd, 0);
		if (addr == (char *)-1)
			error(1, "mmap failed: %s", strerror(errno));

#ifdef MADV_SEQUENTIAL
		(void)madvise(addr, sz, MADV_SEQUENTIAL);
#endif
		while (sz > WRSZ) {
			if (bozowrite(STDOUT_FILENO, addr, WRSZ) != WRSZ)
				error(1, "write failed: %s", strerror(errno));
			sz -= WRSZ;
			addr += WRSZ;
		}
		if (sz && bozowrite(STDOUT_FILENO, addr, sz) != sz)
			error(1, "final write failed: %s", strerror(errno));
		if (munmap(oaddr, (size_t)sb.st_size) < 0)
			warning("munmap failed");
	}
#ifndef NO_SSL_SUPPORT
	/* If SSL enabled cleanup SSL structure. */
	if (Zflag) 
		SSL_free(bozossl);
#endif /* NO_SSL_SUPPORT */
	(void)close(fd);
}

#ifndef NO_DIRINDEX_SUPPORT
/*
 * output a directory index.
 */
static void
directory_index(request, dirname)
	http_req *request;
	const char *dirname;
{
	struct stat sb;
	struct dirent *de;
	struct tm *tm;
	DIR *dp;
	char buf[MAXPATHLEN];
	int l;

	debug((DEBUG_FAT, "directory_index: dirname ``%s''", dirname));
	if (stat(dirname, &sb) < 0 ||
	    (dp = opendir(dirname)) == NULL) {
		if (errno == EPERM)
			http_error(403, request,
			    "no permission to open directory");
		else if (errno == ENOENT)
			http_error(404, request, "no file");
		else
			http_error(500, request, "open directory");
		return;
	}

	(void)bozoprintf("%s 200 OK\r\n", request->hr_proto);

	if (request->hr_proto != http_09) {
		print_header(request, NULL, "text/html", "");
		bozoprintf("\r\n");
	}
	(void)bozoflush(stdout);

	if (request->hr_method == HTTP_HEAD) {
		closedir(dp);
		return;
	}

	(void)bozoprintf("<html><head><title>Index of %s</title></head>\r\n",
	    request->hr_url);
	(void)bozoprintf("<body><h1>Index of %s</h1>\r\n", request->hr_url);
	(void)bozoprintf("<pre>\r\n");
	(void)bozoprintf("Name                             "
	    "Last modified          "
	    "Size\n");
	(void)bozoprintf("<hr noshade align=\"left\" width=\"80%%\">\r\n\r\n");

	while ((de = readdir(dp)) != NULL) {
		int nostat = 0;
		char *name = de->d_name;

		if (strcmp(name, ".") == 0 || 
		    (strcmp(name, "..") != 0 && Hflag && name[0] == '.'))
			continue;

		(void)snprintf(buf, sizeof buf, "%s/%s", dirname, name);
		if (stat(buf, &sb))
			nostat = 1;

		l = 0;

		if (strcmp(name, "..") == 0) {
			(void)bozoprintf("<a href=\"../\">");
			l += bozoprintf("Parent Directory");
		} else if (S_ISDIR(sb.st_mode)) {
			(void)bozoprintf("<a href=\"%s/\">", name);
			l += bozoprintf("%s/", name);
		} else {
			(void)bozoprintf("<a href=\"%s\">", name);
			l += bozoprintf("%s", name);
		}
		(void)bozoprintf("</a>");

		/* XXX fixme, i'm not efficient */
		(void)bozoprintf(" ");
		for (; l < 30; l++)
			(void)bozoprintf(" ");

		if (nostat)
			(void)bozoprintf("?                         ?");
		else {
			tm = gmtime(&sb.st_mtime);
			(void)strftime(buf, sizeof buf, "%d-%b-%Y %R", tm);
			l += bozoprintf("%s", buf);

			(void)bozoprintf(" ");
			for (; l < 44; l++)
				(void)bozoprintf(" ");

			(void)bozoprintf("%7ukB", ((unsigned int)(sb.st_size >> 10)));
		}
		(void)bozoprintf("\r\n");
	}

	closedir(dp);
	(void)bozoprintf("</pre><hr></body></html>\r\n");
	(void)bozoflush(stdout);
}
#endif /* NO_DIRINDEX_SUPPORT */

#ifndef NO_DAEMON_MODE 
/* ARGSUSED */
static void
sigchild(signo)
	int signo;
{

	while (waitpid(-1, NULL, WNOHANG) > 0)
		;
}
#endif /* NO_DAEMON_MODE */

/*
 * deal with virtual host names; we do this:
 *	if we have a virtual path root (vpath), and we are given a
 *	virtual host spec (Host: ho.st or http://ho.st/), see if this
 *	directory exists under vpath.  if it does, use this as the
 #	new slashdir.
 */
static void
check_virtual(request)
	http_req *request;
{
	char *url = request->hr_url, *s;
	struct dirent **list;
	size_t len;
	int i;

	if (!vpath)
		goto out;

	/*
	 * convert http://virtual.host/ to request->hr_host
	 */
	debug((DEBUG_OBESE, "checking for http:// virtual host in ``%s''", url));
	if (strncasecmp(url, "http://", 7) == 0) {
		/* we would do virtual hosting here? */
		url += 7;
		s = strchr(url, '/');
		/* HTTP/1.1 draft rev-06, 5.2: URI takes precedence over Host: */
		request->hr_host = url;
		request->hr_url = bozostrdup(s ? s : "/");
		debug((DEBUG_OBESE, "got host ``%s'' url is now ``%s''",
		    request->hr_host, request->hr_url));
	} else if (!request->hr_host)
		goto out;


	/*
	 * ok, we have a virtual host, use scandir(3) to find a case
	 * insensitive match for the virtual host we are asked for.
	 * note that if the virtual host is the same as the master,
	 * we don't need to do anything special.
	 */
	len = strlen(request->hr_host);
	debug((DEBUG_OBESE,
	    "check_virtual: checking host `%s' under vpath `%s' for url `%s'",
	    request->hr_host, vpath, request->hr_url));
	if (strncasecmp(myname, request->hr_host, len) != 0) {
		s = 0;
		for (i = scandir(vpath, &list, 0, 0); i--; list++) {
			debug((DEBUG_OBESE, "looking at dir``%s''",
			    (*list)->d_name));
			if (strncasecmp((*list)->d_name, request->hr_host,
			    len) == 0) {
				/* found it, punch it */
				myname = (*list)->d_name;
				if (asprintf(&s, "%s/%s", vpath, myname) < 0)
					error(1, "asprintf");
				break;
			}
		}
		if (s == 0)
			http_error(404, request, "unknown URL");
	} else
out:
		s = slashdir;

	/*
	 * ok, nailed the correct slashdir, chdir to it
	 */
	if (chdir(s) < 0)
		error(1, "can't chdir %s: %s", s, strerror(errno));
}

/* make sure we're not trying to access special files */
static void
check_special_files(http_req *request, const char *name)
{

	/* ensure basename(name) != special files */
	if (strcmp(name, DIRECT_ACCESS_FILE) == 0)
		http_error(403, request,
		    "no permission to open direct access file");
	if (strcmp(name, REDIRECT_FILE) == 0)
		http_error(403, request,
		    "no permission to open redirect file");
	if (strcmp(name, ABSREDIRECT_FILE) == 0)
		http_error(403, request,
		    "no permission to open redirect file");
#ifdef DO_HTPASSWD
	if (strcmp(name, AUTH_FILE) == 0)
		http_error(403, request,
		    "no permission to open authfile");
#endif
}

/*
 * checks to see if this request has a valid .bzredirect file.  returns
 * 0 on failure and 1 on success.
 */
static void
check_bzredirect(http_req *request)
{
	struct stat sb;
	char dir[MAXPATHLEN], redir[MAXPATHLEN], redirpath[MAXPATHLEN];
	char *basename, *finalredir;
	int rv, absolute;

	/*
	 * if this pathname is really a directory, but doesn't end in /,
	 * use it as the directory to look for the redir file.
	 */
	snprintf(dir, sizeof(dir), "%s", request->hr_url + 1);
	debug((DEBUG_FAT, "check_bzredirect: dir %s", dir));
	basename = strrchr(dir, '/');

	if ((!basename || basename[1] != '\0') &&
	    lstat(dir, &sb) == 0 && S_ISDIR(sb.st_mode))
		/* nothing */;
	else if (basename == NULL)
		strcpy(dir, ".");
	else {
		*basename++ = '\0';
		check_special_files(request, basename);
	}

	snprintf(redir, sizeof(redir), "%s/%s", dir, REDIRECT_FILE);
	if (lstat(redir, &sb) == 0) {
		if (S_ISLNK(sb.st_mode) == 0)
			return;
		absolute = 0;
	} else {
		snprintf(redir, sizeof(redir), "%s/%s", dir, ABSREDIRECT_FILE);
		if (lstat(redir, &sb) < 0 || S_ISLNK(sb.st_mode) == 0)
			return;
		absolute = 1;
	}
	debug((DEBUG_FAT, "check_bzredirect: calling readlink"));
	rv = readlink(redir, redirpath, sizeof redirpath);
	if (rv == -1) {
		debug((DEBUG_FAT, "readlink failed"));
		return;
	}
	if (rv - 1 < sizeof redirpath)
		redirpath[rv] = '\0';
	debug((DEBUG_FAT, "readlink returned \"%s\"", redirpath));
	
	/* now we have the link pointer, redirect to the real place */
	if (absolute)
		finalredir = redirpath;
	else
		snprintf(finalredir = redir, sizeof(redir), "/%s/%s", dir,
			 redirpath);

	debug((DEBUG_FAT, "check_bzredirect: new redir %s", finalredir));
	handle_redirect(request, finalredir, absolute);
}

/*
 * checks to see if this request has a valid .bzdirect file.  returns
 * 0 on failure and 1 on success.
 */
static int
check_direct_access(http_req *request)
{
	FILE *fp;
	struct stat sb;
	char dir[MAXPATHLEN], dirfile[MAXPATHLEN], *basename;

	snprintf(dir, sizeof(dir), "%s", request->hr_url + 1);
	debug((DEBUG_FAT, "check_bzredirect: dir %s", dir));
	basename = strrchr(dir, '/');

	if ((!basename || basename[1] != '\0') &&
	    lstat(dir, &sb) == 0 && S_ISDIR(sb.st_mode))
		/* nothing */;
	else if (basename == NULL)
		strcpy(dir, ".");
	else {
		*basename++ = '\0';
		check_special_files(request, basename);
	}

	snprintf(dirfile, sizeof(dirfile), "%s/%s", dir, DIRECT_ACCESS_FILE);
	if (stat(dirfile, &sb) < 0 ||
	    (fp = fopen(dirfile, "r")) == NULL)
		return 0;
	fclose(fp);
	return 1;
}

/*
 * transform_request does this:
 *	- ``expand'' %20 crapola
 *	- punt if it doesn't start with /
 *	- check rflag / referrer
 *	- look for "http://myname/" and deal with it.
 *	- maybe call process_cgi() 
 *	- check for ~user and call transform_user() if so
 *	- if the length > 1, check for trailing slash.  if so,
 *	  add the index.html file
 *	- if the length is 1, return the index.html file
 *	- disallow anything ending up with a file starting
 *	  at "/" or having ".." in it.
 *	- anything else is a really weird internal error
 */
static char *
transform_request(request, isindex)
	http_req *request;
	int *isindex;
{
	char	*file;
	char	*url;
	size_t	len;

	file = NULL;	/* XXX gcc */
	*isindex = 0;
	debug((DEBUG_FAT, "tf_req: url %s", request->hr_url));
	fix_url_percent(request);
	check_virtual(request);
	url = request->hr_url;

	if (url[0] != '/')
		http_error(404, request, "unknown URL");

	check_bzredirect(request);

	if (rflag) {
		int to_indexhtml = 0;

#define TOP_PAGE(x)	(strcmp((x), "/") == 0 || \
			 strcmp((x) + 1, index_html) == 0 || \
			 strcmp((x) + 1, "favicon.ico") == 0) 
		
		debug((DEBUG_EXPLODING, "checking rflag"));
		/*
		 * first check that this path isn't allowed via .bzdirect file,
		 * and then check referrer; make sure that people come via the
		 * real name... otherwise if we aren't looking at / or
		 * /index.html, redirect...  we also special case favicon.ico.
		 */
		if (check_direct_access(request))
			/* nothing */;
		else if (request->hr_referrer) {
			const char *r = request->hr_referrer;

			debug((DEBUG_FAT,
			   "checking referrer \"%s\" vs myname %s", r, myname));
			if (strncmp(r, "http://", 7) != 0 ||
			    (strncasecmp(r + 7, myname, strlen(myname)) != 0 &&
			     !TOP_PAGE(url)))
				to_indexhtml = 1;
		} else {
			const char *h = request->hr_host;

			debug((DEBUG_FAT, "url has no referrer at all"));
			/* if there's no referrer, let / or /index.html past */
			if (!TOP_PAGE(url) ||
			    (h && strncasecmp(h, myname, strlen(myname)) != 0))
				to_indexhtml = 1;
		}

		if (to_indexhtml) {
			char *slashindexhtml;

			if (asprintf(&slashindexhtml, "/%s", index_html) < 0)
				error(1, "asprintf");
			debug((DEBUG_FAT, "rflag: redirecting %s to %s", url, slashindexhtml));
			handle_redirect(request, slashindexhtml, 0);
			/* NOTREACHED */
		}
	}

#ifndef NO_CGIBIN_SUPPORT
	if (cgibin || Cflag)
		process_cgi(request);
#endif /* NO_CGIBIN_SUPPORT */

	len = strlen(url);
	if (0) {
#ifndef NO_USER_SUPPORT
	} else if (len > 1 && uflag && url[1] == '~') {
		if (url[2] == '\0')
			http_error(404, request, "missing username");
		if (strchr(url + 2, '/') == NULL)
			handle_redirect(request, NULL, 0);
			/* NOTREACHED */
		debug((DEBUG_FAT, "calling transform_user"));
		return (transform_user(request, isindex));
#endif /* NO_USER_SUPPORT */
	} else if (len > 1) {
		debug((DEBUG_FAT, "url[len-1] == %c", url[len-1]));
		if (url[len-1] == '/') {	/* append index.html */
			*isindex = 1;
			debug((DEBUG_FAT, "appending index.html"));
			file = bozomalloc(len + strlen(index_html) + 1);
			strcpy(file, url + 1);
			strcat(file, index_html);
		} else
			file = bozostrdup(url + 1);
	} else if (len == 1) {
		debug((DEBUG_EXPLODING, "tf_req: len == 1"));
		file = bozostrdup(index_html);
		*isindex = 1;
	} else		/* len == 0 ? */
		http_error(500, request, "request->hr_url is nul?");

	if (file == NULL)
		http_error(500, request, "internal failure");

	/*
	 * look for "http://myname/" and deal with it as necessary.
	 */

	/*
	 * stop traversing outside our domain 
	 *
	 * XXX true security only comes from our parent using chroot(2)
	 * before execve(2)'ing us.  or our own built in chroot(2) support.
	 */
	if (*file == '/' || strstr(file, ".."))
		http_error(403, request, "illegal request");

	check_auth(request, file);

	debug((DEBUG_FAT, "transform_request returned: %s", file));
	return (file);
}

#ifndef NO_USER_SUPPORT
/*
 * transform_user does this:
 *	- chdir's /~user/public_html
 *	- returns the rest of the file, index.html appended if required
 *
 * transform_request() is supposed to check that we have user support
 * enabled.
 */
static char *
transform_user(request, isindex)
	http_req *request;
	int *isindex;
{
	char	c, *s, *file = NULL;
	struct	passwd *pw;

	*isindex = 0;

	if ((s = strchr(request->hr_url + 2, '/')) != NULL) {
		*s++ = '\0';
		c = s[strlen(s)-1];
		*isindex = (c == '/' || c == '\0');
	}

	debug((DEBUG_OBESE, "looking for user %s", request->hr_url + 2));
	pw = getpwnam(request->hr_url + 2);
	/* fix this up immediately */
	if (s)
		s[-1] = '/';
	if (pw == NULL)
		http_error(404, request, "no such user");

	debug((DEBUG_OBESE, "user %s home dir %s uid %d gid %d", pw->pw_name,
	    pw->pw_dir, pw->pw_uid, pw->pw_gid));

	if (chdir(pw->pw_dir) < 0) {
		warning("chdir1 error: %s: %s", pw->pw_dir, strerror(errno));
		http_error(403, request, "can't chdir to homedir");
	}
	if (chdir(public_html) < 0) {
		warning("chdir2 error: %s: %s", public_html, strerror(errno));
		http_error(403, request, "can't chdir to public_html");
	}
	if (s == NULL || *s == '\0') {
		file = bozostrdup(index_html);
	} else {
		file = bozomalloc(strlen(s) +
		    (*isindex ? strlen(index_html) + 1 : 1));
		strcpy(file, s);
		if (*isindex)
			strcat(file, index_html);
	}

	/* see above */
	if (*file == '/' || strstr(file, ".."))
		http_error(403, request, "illegal request");

	check_auth(request, file);

	debug((DEBUG_FAT, "transform_user returning %s under %s", file,
	    pw->pw_dir));
	return (file);
}
#endif /* NO_USER_SUPPORT */

/*
 * do automatic redirection
 */
static void
handle_redirect(request, url, absolute)
	http_req *request;
	const char *url;
	int absolute;
{
	char *urlbuf;
	char portbuf[20];
	
	if (url == NULL) {
		if (asprintf(&urlbuf, "%s/", request->hr_url) < 0)
			error(1, "asprintf");
		url = urlbuf;
	}
	if (request->hr_serverport)
		snprintf(portbuf, sizeof(portbuf), ":%s",
		    request->hr_serverport);
	else
		portbuf[0] = '\0';
	warning("redirecting %s%s%s", myname, portbuf, url);
	debug((DEBUG_FAT, "redirecting %s", url));
	(void)bozoprintf("%s 301 Document Moved\r\n", request->hr_proto);
	if (request->hr_proto != http_09) 
		print_header(request, NULL, "text/html", NULL);
	if (request->hr_proto != http_09) {
		(void)bozoprintf("Location: http://");
		if (absolute == 0)
			(void)bozoprintf("%s%s", myname, portbuf);
		(void)bozoprintf("%s\r\n", url);
	}
	(void)bozoprintf("\r\n");
	if (request->hr_method == HTTP_HEAD)
		goto head;
	(void)bozoprintf("<html><head><title>Document Moved</title></head>\n");
	(void)bozoprintf("<body><h1>Document Moved</h1>\n");
	(void)bozoprintf("This document had moved <a href=\"http://");
	if (absolute)
		(void)bozoprintf("%s", url);
	else
		(void)bozoprintf("%s%s%s", myname, portbuf, url);
	(void)bozoprintf("\">here</a>\n");
	(void)bozoprintf("</body></html>\n");
head:
	bozoflush(stdout);
	exit(0);
}

/* generic header printing routine */
static void
print_header(request, sbp, type, encoding)
	http_req *request;
	struct stat *sbp;
	const char *type;
	const char *encoding;
{

	(void)bozoprintf("Date: %s\r\n", date);
	(void)bozoprintf("Server: %s\r\n", server_software);
	if (sbp) {
		char filedate[40];
		struct	tm *tm;

		tm = gmtime(&sbp->st_mtime);
		strftime(filedate, sizeof filedate,
		    "%a, %d %b %Y %H:%M:%S GMT", tm);
		(void)bozoprintf("Last-Modified: %s\r\n", filedate);
	}
	if (type && *type)
		(void)bozoprintf("Content-Type: %s\r\n", type);
	if (encoding && *encoding)
		(void)bozoprintf("Content-Encoding: %s\r\n", encoding);
	if (sbp)
		(void)bozoprintf("Content-Length: %qd\r\n", (long long)sbp->st_size);
	if (request && request->hr_proto == http_11)
		(void)bozoprintf("Connection: close\r\n");
	bozoflush(stdout);
}

#ifndef NO_CGIBIN_SUPPORT
/* help build up the environ pointer */
static __inline__ void
spsetenv(const char *env, const char *val, char **envp)
{
	char *s1 = bozomalloc(strlen(env) + strlen(val) + 2);

	strcpy(s1, env);
	strcat(s1, "=");
	strcat(s1, val);
	debug((DEBUG_OBESE, "spsetenv: %s", s1));
	*envp = s1;
}

/*
 * Checks if the request has asked for a cgi-bin.  Should only be called if
 * cgibin is set.  If it starts CGIBIN_PREFIX or has a ncontent handler,
 * process the cgi, otherwise just return.
 */
static void
process_cgi(request)
	http_req *request;
{
	char	buf[WRSZ];
	struct	headers *headp;
	const char *type, *clen, *info, *cgihandler;
	char	*query, *s, *t, *path, *env, *command, *url;
	char	**envp, **curenvp, *argv[4];
	size_t	len;
	ssize_t rbytes;
	pid_t	pid;
	int	envpsize, ix, nph;
	int	sv[2];

	debug((DEBUG_NORMAL, "process_cgi: url `%s'", request->hr_url));

	url = bozostrdup(request->hr_url);
	if ((s = strchr(url, '?')) != NULL) {
		*s++ = '\0';
		query = s;
	} else
		query = NULL;
	path = NULL;
	envp = NULL;
	cgihandler = NULL;
	command = NULL;
	info = NULL;

	len = strlen(url);
	if (len == 0 || url[len - 1] == '/') {	/* append index.html */
		debug((DEBUG_FAT, "appending index.html"));
		url = bozorealloc(url, len + strlen(index_html) + 1);
		strcat(url, index_html);
		debug((DEBUG_NORMAL, "process_cgi: url adjusted to `%s'", url));
	}

	check_auth(request, url + 1);

	if (!cgibin || strncmp(url + 1, CGIBIN_PREFIX, CGIBIN_PREFIX_LEN) != 0) {
		cgihandler = content_cgihandler(request, url + 1);
		if (cgihandler == NULL) {
			free(url);
			return;
		}
		debug((DEBUG_NORMAL, "process_cgi: cgihandler `%s'",
		    cgihandler));
	}

	ix = 0;
	if (cgihandler) {
		command = url + 1;
		path = bozostrdup(cgihandler);
		argv[ix++] = path;
			/* argv[] = [ path, command, query, NULL ] */
	} else {
		command = url + CGIBIN_PREFIX_LEN + 1;
		if ((s = strchr(command, '/')) != NULL) {
			info = bozostrdup(s);
			*s = '\0';
		}
		path = bozomalloc(strlen(cgibin) + 1 + strlen(command) + 1);
		strcpy(path, cgibin);
		strcat(path, "/");
		strcat(path, command);
			/* argv[] = [ command, query, NULL ] */
	}
	argv[ix++] = command;
	argv[ix++] = query;
	argv[ix++] = NULL;

	debug((DEBUG_FAT,
	    "process_cgi: path `%s' cmd `%s' info `%s' query `%s'",
	    path, command, strornull(info), strornull(query)));

	type = request->hr_content_type;
	clen = request->hr_content_length;

	envpsize = 13 + request->hr_nheaders + 
	    (info && *info ? 1 : 0) +
	    (query && query ? 1 : 0) +
	    (type && *type ? 1 : 0) +
	    (clen && *clen ? 1 : 0) +
	    (request->hr_remotehost && *request->hr_remotehost ? 1 : 0) +
	    (request->hr_remoteaddr && *request->hr_remoteaddr ? 1 : 0) +
#ifdef DO_HTPASSWD
	    (request->hr_authuser && *request->hr_authuser ? 2 : 0) +
#endif /* DO_HTPASSWD */
	    (request->hr_serverport && *request->hr_serverport ? 1 : 0);

	envp = bozomalloc(sizeof(*envp) * envpsize);
	for (ix = 0; ix < envpsize; ix++)
		envp[ix] = NULL;
	curenvp = envp;
	for (headp = SIMPLEQ_FIRST(&request->hr_headers); headp;
	    headp = SIMPLEQ_NEXT(headp, h_next)) {
		const char *s2;
		env = bozomalloc(6 + strlen(headp->h_header) + 1 +
		    strlen(headp->h_value));

		t = env;
		strcpy(t, "HTTP_");
		t += strlen(t);
		for (s2 = headp->h_header; *s2; t++, s2++)
			if (islower(*s2))
				*t = toupper(*s2);
			else if (*s2 == '-')
				*t = '_';
			else
				*t = *s2;
		*t = '\0';
		debug((DEBUG_OBESE, "setting header %s as %s = %s",
		    headp->h_header, env, headp->h_value));
		spsetenv(env, headp->h_value, curenvp++);
		free(env);
	}
		
#ifndef _PATH_DEFPATH
#define _PATH_DEFPATH "/usr/bin:/bin"
#endif

	spsetenv("PATH", _PATH_DEFPATH, curenvp++);
	spsetenv("IFS", " \t\n", curenvp++);
	spsetenv("SERVER_NAME", myname, curenvp++);
	spsetenv("GATEWAY_INTERFACE", "CGI/1.1", curenvp++);
	spsetenv("SERVER_PROTOCOL", request->hr_proto, curenvp++);
	spsetenv("REQUEST_METHOD", request->hr_methodstr, curenvp++);
	spsetenv("SCRIPT_NAME", url, curenvp++);
	spsetenv("SCRIPT_FILENAME", url + 1, curenvp++);
	spsetenv("SERVER_SOFTWARE", server_software, curenvp++);
	spsetenv("REQUEST_URI", request->hr_url, curenvp++);
	spsetenv("DATE_GMT", date, curenvp++);
	if (query && *query)
		spsetenv("QUERY_STRING", query, curenvp++);
	if (info && *info)
		spsetenv("PATH_INFO", info, curenvp++);
	if (type && *type)
		spsetenv("CONTENT_TYPE", type, curenvp++);
	if (clen && *clen)
		spsetenv("CONTENT_LENGTH", clen, curenvp++);
	if (request->hr_serverport && *request->hr_serverport)
		spsetenv("SERVER_PORT", request->hr_serverport, curenvp++);
	if (request->hr_remotehost && *request->hr_remotehost)
		spsetenv("REMOTE_HOST", request->hr_remotehost, curenvp++);
	if (request->hr_remoteaddr && *request->hr_remoteaddr)
		spsetenv("REMOTE_ADDR", request->hr_remoteaddr, curenvp++);
#ifdef DO_HTPASSWD
	if (request->hr_authuser && *request->hr_authuser) {
		spsetenv("AUTH_TYPE", "Basic", curenvp++);
		spsetenv("REMOTEUSER", request->hr_authuser, curenvp++);
	}
#endif /* DO_HTPASSWD */

	debug((DEBUG_FAT, "process_cgi: going exec %s, %s %s %s",
	    path, argv[0], strornull(argv[1]), strornull(argv[2])));

	if (-1 == socketpair(AF_UNIX, SOCK_STREAM, PF_UNSPEC, sv))
		error(1, "child socketpair failed: %s", strerror(errno));

	/*
	 * We create 2 procs: one to become the CGI, one read from
	 * the CGI and output to the network, and this parent will
	 * continue reading from the network and writing to the
	 * CGI procsss.
	 */
	switch (fork()) {
	case -1: /* eep, failure */
		error(1, "child fork failed: %s", strerror(errno));
	case 0:
		close(sv[0]);
		dup2(sv[1], STDIN_FILENO);
		dup2(sv[1], STDOUT_FILENO);

		if (-1 == execve(path, argv, envp))
			error(1, "child exec failed: %s", path);
		/* NOT REACHED */
		error(1, "child execve returned?!");
	}

	close(sv[1]);

	/* parent: read from stdin (bozoread()) write to sv[0] */
	/* child: read from sv[0] (bozowrite()) write to stdout */
	pid = fork();
	if (pid == -1)
		error(1, "io child fork failed: %s", strerror(errno));
	else if (pid == 0) {
		/* child reader/writer */
		close(STDIN_FILENO);
		finish_cgi_output(request, sv[0], nph);
		/* if we're done output, our parent is useless... */
		kill(getppid(), SIGKILL);
		debug((DEBUG_FAT, "done processing cgi output"));
		_exit(0);
	}
	close(STDOUT_FILENO);

	/* XXX we should have some goo that times us out
	 */
	while ((rbytes = bozoread(STDIN_FILENO, buf, sizeof buf)) > 0) {
		ssize_t wbytes;
		char *bp = buf;

		while (rbytes) {
			wbytes = write(sv[0], buf , rbytes);
			if (wbytes > 0) {
				rbytes -= wbytes;
				bp += wbytes;
			} else
				error(1, "write failed: %s", strerror(errno));
		}		
	}
	debug((DEBUG_FAT, "done processing cgi input"));
	exit(0);
}

/*
 * handle parsing a CGI header output, transposing a Status: header
 * into the HTTP reply (ie, instead of "200 OK").
 */
static void
finish_cgi_output(http_req *request, int in, int nph)
{
	char	buf[WRSZ];
	char	*str, *val;
	size_t	len;
	ssize_t rbytes;
	SIMPLEQ_HEAD(, headers)	headers;
	struct	headers *hdr;
	int	write_header, nheaders = 0, write_str = 0;

	/* much of this code is like read_request()'s header loop. hmmm... */
	SIMPLEQ_INIT(&headers);
	write_header = nph == 0;
	while (nph && (str = dgetln(in, &len, read)) != NULL) {
		str = bozostrdup(str);	/* we use this copy */

		if (*str == '\0') {
			write_str = 1;
			break;
		}

		val = strnsep(&str, ":", &len);
		debug((DEBUG_EXPLODING,
		    "read_req2: after strnsep: str ``%s'' val ``%s''",
		    str, val));
		if (val == NULL || len == -1) {
			write_str = 1;
			break;
		}
		while (*str == ' ' || *str == '\t')
			len--, str++;

		if (strcasecmp(val, "status") == 0) {
			debug((DEBUG_OBESE, "process_cgi:  writing HTTP header "
					    "from status %s ..", str));
			(void)bozoprintf("%s %s\r\n", request->hr_proto, str);
			bozoflush(stdout);
			write_header = 0;
			break;
		}

		hdr = bozomalloc(sizeof *hdr);
		hdr->h_header = val;
		hdr->h_value = str;
		SIMPLEQ_INSERT_TAIL(&headers, hdr, h_next);
		nheaders++;
	}

	if (write_header) {
		debug((DEBUG_OBESE, "process_cgi:  writing HTTP header .."));
		(void)bozoprintf("%s 200 OK\r\n", request->hr_proto);
		bozoflush(stdout);
	}

	if (nheaders) {
		debug((DEBUG_OBESE, "process_cgi:  writing delayed HTTP "
				    "headers .."));
		for (hdr = SIMPLEQ_FIRST(&headers); hdr;
		    hdr = SIMPLEQ_NEXT(hdr, h_next)) {
			(void)bozoprintf("%s: %s\r\n", hdr->h_header,
					 hdr->h_value);
		}
		bozoflush(stdout);
	}

	/* XXX we should have some goo that times us out
	 */
	while ((rbytes = read(in, buf, sizeof buf)) > 0) {
		ssize_t wbytes;
		char *bp = buf;

		while (rbytes) {
			wbytes = bozowrite(STDOUT_FILENO, buf, rbytes);
			if (wbytes > 0) {
				rbytes -= wbytes;
				bp += wbytes;
			} else
				error(1, "cgi output write failed: %s",
				    strerror(errno));
		}		
	}
}
#endif /* NO_CGIBIN_SUPPORT */

/* this escape HTML tags */
static void
escape_html(request)
	http_req *request;
{
	int	i, j;
	char	*url = request->hr_url, *tmp;

	for (i = 0, j = 0; url[i]; i++) {
		switch (url[i]) {
		case '<':
		case '>':
			j += 4;
			break;
		case '&':
			j += 5;
			break;
		}
	}

	if (j == 0)
		return;

	if ((tmp = (char *) malloc(strlen(url) + j)) == 0)
		/*
		 * ouch, but we are only called from an error context, and
		 * most paths here come from malloc(3) failures anyway...
		 * we could completely punt and just exit, but isn't returning
		 * an not-quite-correct error better than nothing at all?
		 */
		return;

	for (i = 0, j = 0; url[i]; i++) {
		switch (url[i]) {
		case '<':
			memcpy(tmp + j, "&lt;", 4);
			j += 4;
			break;
		case '>':
			memcpy(tmp + j, "&gt;", 4);
			j += 4;
			break;
		case '&':
			memcpy(tmp + j, "&amp;", 5);
			j += 5;
			break;
		default:
			tmp[j++] = url[i];
		}
	}
	tmp[j] = 0;

	/*
	 * XXX original "url" is a substring of an allocation, so we
	 * can't touch it.  so, ignore it and replace the request.
	 */
	request->hr_url = tmp;
}

/* this fixes the %HH hack that RFC2396 requires.  */
static void
fix_url_percent(request)
	http_req *request;
{
	char	*s, *t, buf[3], *url;
	char	*end;	/* if end is not-zero, we don't translate beyond that */

	url = request->hr_url;

	/* make sure we don't translate *too* much */
	end = strchr(request->hr_url, '?');

	/* fast forward to the first % */
	if ((s = strchr(url, '%')) == NULL)
		return;

	t = s;
	do {
		if (end && s >= end) {
			debug((DEBUG_EXPLODING, "fu_%%: past end, filling out.."));
			while (*s)
				*t++ = *s++;
			break;
		}
		debug((DEBUG_EXPLODING, "fu_%%: got s == %%, s[1]s[2] == %c%c",
		    s[1], s[2]));
		if (s[1] == '\0' || s[2] == '\0')
			http_error(400, request,
			    "percent hack missing two chars afterwards");
		if (s[1] == '0' && s[2] == '0')
			http_error(404, request, "percent hack was %00");
		if (s[1] == '2' && s[2] == 'f')
			http_error(404, request, "percent hack was %2f (/)");
			
		buf[0] = *++s;
		buf[1] = *++s;
		buf[2] = '\0';
		s++;
		*t = (char)strtol(buf, NULL, 16);
		debug((DEBUG_EXPLODING, "fu_%%: strtol put %c into *t", *t));
		if (*t++ == '\0')
			http_error(400, request, "percent hack got a 0 back");

		while (*s && *s != '%')
			*t++ = *s++;
	} while (*s);
	*t = '\0';
	debug((DEBUG_FAT, "fix_url_percent returns %s in url", request->hr_url));
}

/*
 * process each type of HTTP method, setting this HTTP requests
 # method type.
 */
static struct method_map {
	const char *name;
	int	type;
} method_map[] = {
	{ "GET", 	HTTP_GET, },
	{ "POST",	HTTP_POST, },
	{ "HEAD",	HTTP_HEAD, },
#if 0	/* other non-required http/1.1 methods */
	{ "OPTIONS",	HTTP_OPTIONS, },
	{ "PUT",	HTTP_PUT, },
	{ "DELETE",	HTTP_DELETE, },
	{ "TRACE",	HTTP_TRACE, },
	{ "CONNECT",	HTTP_CONNECT, },
#endif
	{ NULL,		0, },
};

static void
process_method(request, method)
	http_req *request;
	const char *method;
{
	struct	method_map *mmp;

	for (mmp = method_map; mmp->name; mmp++)
		if (strcasecmp(method, mmp->name) == 0) {
			request->hr_method = mmp->type;
			request->hr_methodstr = mmp->name;
			return;
		}

	if (request->hr_proto == http_11)
		request->hr_allow = "GET, HEAD, POST";
	http_error(404, request, "unknown method");
}

/*
 * as the prototype string is not constant (eg, "HTTP/1.1" is equivalent
 * to "HTTP/001.01"), we MUST parse this.
 */
static void
process_proto(request, proto)
	http_req *request;
	const char *proto;
{
	char	majorstr[16], *minorstr;
	int	majorint, minorint;

	if (proto == NULL) {
got_proto_09:
		request->hr_proto = http_09;
		debug((DEBUG_FAT, "request %s is http/0.9", request->hr_url));
		return;
	}

	if (strncasecmp(proto, "HTTP/", 5) != 0)
		goto bad;
	strncpy(majorstr, proto + 5, sizeof majorstr);
	majorstr[sizeof(majorstr)-1] = 0;
	minorstr = strchr(majorstr, '.');
	if (minorstr == NULL)
		goto bad;
	*minorstr++ = 0;

	majorint = atoi(majorstr);
	minorint = atoi(minorstr);

	switch (majorint) {
	case 0:
		if (minorint != 9)
			break;
		goto got_proto_09;
	case 1:
		if (minorint == 0)
			request->hr_proto = http_10;
		else if (minorint == 1)
			request->hr_proto = http_11;
		else
			break;

		debug((DEBUG_FAT, "request %s is %s", request->hr_url,
		    request->hr_proto));
		SIMPLEQ_INIT(&request->hr_headers);
		request->hr_nheaders = 0;
		return;
	}
bad:
	http_error(404, NULL, "unknown prototype");
}

/*
 * this map and the functions below map between filenames and the
 * content type and content encoding definitions.  this should become
 * a configuration file, perhaps like apache's mime.types (but that
 * has less info per-entry).
 */
struct content_map {
	const char *name;	/* postfix of file */
	const char *type;	/* matching content-type */
	const char *encoding;	/* matching content-encoding */
	const char *encoding11;	/* matching content-encoding (HTTP/1.1) */
	const char *cgihandler;	/* optional CGI handler */
};

static struct content_map content_map[] = {
	{ ".html",	"text/html",			"",		"", NULL },
	{ ".htm",	"text/html",			"",		"", NULL },
	{ ".gif",	"image/gif",			"",		"", NULL },
	{ ".jpeg",	"image/jpeg",			"",		"", NULL },
	{ ".jpg",	"image/jpeg",			"",		"", NULL },
	{ ".jpe",	"image/jpeg",			"",		"", NULL },
	{ ".png",	"image/png",			"",		"", NULL },
	{ ".mp3",	"audio/mpeg",			"",		"", NULL },
	{ ".css",	"text/css",			"",		"", NULL },
	{ ".txt",	"text/plain",			"",		"", NULL },
	{ ".swf",	"application/x-shockwave-flash","",		"", NULL },
	{ ".dcr",	"application/x-director",	"",		"", NULL },
	{ ".pac",	"application/x-ns-proxy-autoconfig", "",	"", NULL },
	{ ".pa",	"application/x-ns-proxy-autoconfig", "",	"", NULL },
	{ ".tar",	"multipart/x-tar",		"",		"", NULL },
	{ ".gtar",	"multipart/x-gtar",		"",		"", NULL },
	{ ".tar.Z",	"multipart/x-tar",		"x-compress",	"compress", NULL },
	{ ".tar.gz",	"multipart/x-tar",		"x-gzip",	"gzip", NULL },
	{ ".taz",	"multipart/x-tar",		"x-gzip",	"gzip", NULL },
	{ ".tgz",	"multipart/x-tar",		"x-gzip",	"gzip", NULL },
	{ ".tar.z",	"multipart/x-tar",		"x-pack",	"x-pack", NULL },
	{ ".Z",		"application/x-compress",	"x-compress",	"compress", NULL },
	{ ".gz",	"application/x-gzip",		"x-gzip",	"gzip", NULL },
	{ ".z",		"unknown",			"x-pack",	"x-pack", NULL },
	{ ".bz2",	"application/x-bzip2",		"x-bzip2",	"x-bzip2", NULL },
	{ ".ogg",	"application/x-ogg",		"",		"", NULL },
	{ ".xbel",	"text/xml",			"",		"", NULL },
	{ ".xml",	"text/xml",			"",		"", NULL },
	{ ".xsl",	"text/xml",			"",		"", NULL },
	{ ".hqx",	"application/mac-binhex40",	"",		"", NULL },
	{ ".cpt",	"application/mac-compactpro",	"",		"", NULL },
	{ ".doc",	"application/msword",		"",		"", NULL },
	{ ".bin",	"application/octet-stream",	"",		"", NULL },
	{ ".dms",	"application/octet-stream",	"",		"", NULL },
	{ ".lha",	"application/octet-stream",	"",		"", NULL },
	{ ".lzh",	"application/octet-stream",	"",		"", NULL },
	{ ".exe",	"application/octet-stream",	"",		"", NULL },
	{ ".class",	"application/octet-stream",	"",		"", NULL },
	{ ".oda",	"application/oda",		"",		"", NULL },
	{ ".pdf",	"application/pdf",		"",		"", NULL },
	{ ".ai",	"application/postscript",	"",		"", NULL },
	{ ".eps",	"application/postscript",	"",		"", NULL },
	{ ".ps",	"application/postscript",	"",		"", NULL },
	{ ".ppt",	"application/powerpoint",	"",		"", NULL },
	{ ".rtf",	"application/rtf",		"",		"", NULL },
	{ ".bcpio",	"application/x-bcpio",		"",		"", NULL },
	{ ".torrent",	"application/x-bittorrent",	"",		"", NULL },
	{ ".vcd",	"application/x-cdlink",		"",		"", NULL },
	{ ".cpio",	"application/x-cpio",		"",		"", NULL },
	{ ".csh",	"application/x-csh",		"",		"", NULL },
	{ ".dir",	"application/x-director",	"",		"", NULL },
	{ ".dxr",	"application/x-director",	"",		"", NULL },
	{ ".dvi",	"application/x-dvi",		"",		"", NULL },
	{ ".hdf",	"application/x-hdf",		"",		"", NULL },
	{ ".cgi",	"application/x-httpd-cgi",	"",		"", NULL },
	{ ".skp",	"application/x-koan",		"",		"", NULL },
	{ ".skd",	"application/x-koan",		"",		"", NULL },
	{ ".skt",	"application/x-koan",		"",		"", NULL },
	{ ".skm",	"application/x-koan",		"",		"", NULL },
	{ ".latex",	"application/x-latex",		"",		"", NULL },
	{ ".mif",	"application/x-mif",		"",		"", NULL },
	{ ".nc",	"application/x-netcdf",		"",		"", NULL },
	{ ".cdf",	"application/x-netcdf",		"",		"", NULL },
	{ ".patch",	"application/x-patch",		"",		"", NULL },
	{ ".sh",	"application/x-sh",		"",		"", NULL },
	{ ".shar",	"application/x-shar",		"",		"", NULL },
	{ ".sit",	"application/x-stuffit",	"",		"", NULL },
	{ ".sv4cpio",	"application/x-sv4cpio",	"",		"", NULL },
	{ ".sv4crc",	"application/x-sv4crc",		"",		"", NULL },
	{ ".tar",	"application/x-tar",		"",		"", NULL },
	{ ".tcl",	"application/x-tcl",		"",		"", NULL },
	{ ".tex",	"application/x-tex",		"",		"", NULL },
	{ ".texinfo",	"application/x-texinfo",	"",		"", NULL },
	{ ".texi",	"application/x-texinfo",	"",		"", NULL },
	{ ".t",		"application/x-troff",		"",		"", NULL },
	{ ".tr",	"application/x-troff",		"",		"", NULL },
	{ ".roff",	"application/x-troff",		"",		"", NULL },
	{ ".man",	"application/x-troff-man",	"",		"", NULL },
	{ ".me",	"application/x-troff-me",	"",		"", NULL },
	{ ".ms",	"application/x-troff-ms",	"",		"", NULL },
	{ ".ustar",	"application/x-ustar",		"",		"", NULL },
	{ ".src",	"application/x-wais-source",	"",		"", NULL },
	{ ".zip",	"application/zip",		"",		"", NULL },
	{ ".au",	"audio/basic",			"",		"", NULL },
	{ ".snd",	"audio/basic",			"",		"", NULL },
	{ ".mpga",	"audio/mpeg",			"",		"", NULL },
	{ ".mp2",	"audio/mpeg",			"",		"", NULL },
	{ ".aif",	"audio/x-aiff",			"",		"", NULL },
	{ ".aiff",	"audio/x-aiff",			"",		"", NULL },
	{ ".aifc",	"audio/x-aiff",			"",		"", NULL },
	{ ".ram",	"audio/x-pn-realaudio",		"",		"", NULL },
	{ ".rpm",	"audio/x-pn-realaudio-plugin",	"",		"", NULL },
	{ ".ra",	"audio/x-realaudio",		"",		"", NULL },
	{ ".wav",	"audio/x-wav",			"",		"", NULL },
	{ ".pdb",	"chemical/x-pdb",		"",		"", NULL },
	{ ".xyz",	"chemical/x-pdb",		"",		"", NULL },
	{ ".ief",	"image/ief",			"",		"", NULL },
	{ ".tiff",	"image/tiff",			"",		"", NULL },
	{ ".tif",	"image/tiff",			"",		"", NULL },
	{ ".ras",	"image/x-cmu-raster",		"",		"", NULL },
	{ ".pnm",	"image/x-portable-anymap",	"",		"", NULL },
	{ ".pbm",	"image/x-portable-bitmap",	"",		"", NULL },
	{ ".pgm",	"image/x-portable-graymap",	"",		"", NULL },
	{ ".ppm",	"image/x-portable-pixmap",	"",		"", NULL },
	{ ".rgb",	"image/x-rgb",			"",		"", NULL },
	{ ".xbm",	"image/x-xbitmap",		"",		"", NULL },
	{ ".xpm",	"image/x-xpixmap",		"",		"", NULL },
	{ ".xwd",	"image/x-xwindowdump",		"",		"", NULL },
	{ ".rtx",	"text/richtext",		"",		"", NULL },
	{ ".tsv",	"text/tab-separated-values",	"",		"", NULL },
	{ ".etx",	"text/x-setext",		"",		"", NULL },
	{ ".sgml",	"text/x-sgml",			"",		"", NULL },
	{ ".sgm",	"text/x-sgml",			"",		"", NULL },
	{ ".mpeg",	"video/mpeg",			"",		"", NULL },
	{ ".mpg",	"video/mpeg",			"",		"", NULL },
	{ ".mpe",	"video/mpeg",			"",		"", NULL },
	{ ".qt",	"video/quicktime",		"",		"", NULL },
	{ ".mov",	"video/quicktime",		"",		"", NULL },
	{ ".avi",	"video/x-msvideo",		"",		"", NULL },
	{ ".movie",	"video/x-sgi-movie",		"",		"", NULL },
	{ ".ice",	"x-conference/x-cooltalk",	"",		"", NULL },
	{ ".wrl",	"x-world/x-vrml",		"",		"", NULL },
	{ ".vrml",	"x-world/x-vrml",		"",		"", NULL },
	{ NULL,		NULL,				NULL,		NULL, NULL, },
};

static struct content_map *dynamic_content_map;
static int dynamic_content_map_size;

/* call with stage == 0 for full search, stage == 1 for dynamic only */
static struct content_map *
match_content_map(name, stage)
	const char	*name;
	int	stage;
{
	size_t	len = strlen(name), nlen;
	struct	content_map	*map;

	
	for (map = dynamic_content_map; map && map->name; map++) {
again:
		nlen = strlen(map->name);
		if (nlen > len || strcmp(map->name, name + (len - nlen)) != 0)
			continue;
		return (map);
	}
	if (stage++ == 0) {
		map = content_map;
		goto again;
	}
		
	return (NULL);
}

#ifndef NO_DYNAMIC_CONTENT
static struct content_map *
get_content_map(name)
	const char	*name;
{
	struct	content_map	*map;

	if ((map = match_content_map(name, 1)))
		return (map);
	
	dynamic_content_map_size++;
	dynamic_content_map = bozorealloc(dynamic_content_map,
	    (dynamic_content_map_size + 1) * sizeof *map);
	map = &dynamic_content_map[dynamic_content_map_size];
	map->name = map->type = map->encoding = map->encoding11 =
	    map->cgihandler = NULL;
	map--;

	return (map);
}

#ifndef NO_CGIBIN_SUPPORT
/* cgi maps are simple ".postfix /path/to/prog" */
static void
add_content_map_cgi(arg, cgihandler)
	char	*arg;
	char	*cgihandler;
{
	struct content_map *map;

	debug((DEBUG_NORMAL, "add_content_map_cgi: name %s cgi %s", arg, cgihandler));

	map = get_content_map(arg);
	map->name = arg;
	map->type = map->encoding = map->encoding11 = NULL;
	map->cgihandler = cgihandler;
}
#endif /* NO_CGIBIN_SUPPORT */

/*
 * mime content maps look like:
 *	".name type encoding encoding11"
 * where any of type, encoding or encoding11 a dash "-" means "".
 * eg the .gtar, .tar.Z from above  could be written like:
 *	".gtar multipart/x-gtar - -"
 *	".tar.Z multipart/x-tar x-compress compress"
 * or
 *	".gtar multipart/x-gtar"
 *	".tar.Z multipart/x-tar x-compress compress"
 * NOTE: we destroy 'arg'
 */
static void
add_content_map_mime(char *cmap0, char *cmap1, char *cmap2, char *cmap3)
{
	struct content_map *map;

	debug((DEBUG_FAT, "add_content_map: name %s type %s enc %s enc11 %s ",
		cmap0, cmap1, cmap2, cmap3));

	map = get_content_map(cmap0);
#define CHECKMAP(s)	(!s || ((s)[0] == '-' && (s)[1] == '\0') ? "" : (s))
	map->name = CHECKMAP(cmap0);
	map->type = CHECKMAP(cmap1);
	map->encoding = CHECKMAP(cmap2);
	map->encoding11 = CHECKMAP(cmap3);
#undef CHECKMAP
	map->cgihandler = NULL;
}
#endif /* NO_DYNAMIC_CONTENT */

/*
 * given the file name, return a valid Content-Type: value.
 */
/* ARGSUSED */
static const char *
content_type(request, file)
	http_req *request;
	const char	*file;
{
	struct	content_map	*map;

	map = match_content_map(file, 0);
	if (map)
		return (map->type);
	return (text_plain);
}

/*
 * given the file name, return a valid Content-Encoding: value.
 */
static const char *
content_encoding(request, file)
	http_req *request;
	const char	*file;
{
	struct	content_map	*map;

	map = match_content_map(file, 0);
	if (map)
		return (request->hr_proto == http_11 ?
		    map->encoding11 : map->encoding);
	return (NULL);
}

#ifndef NO_CGIBIN_SUPPORT
/*
 * given the file name, return a CGI interpreter
 */
static const char *
content_cgihandler(request, file)
	http_req *request;
	const char	*file;
{
	struct	content_map	*map;

	map = match_content_map(file, 0);
	if (map)
		return (map->cgihandler);
	return (NULL);
}
#endif /* NO_CGIBIN_SUPPORT */

/*
 * XXX
 */
#ifdef DEBUG
static void
debug__(int level, const char *fmt, ...)
{
	va_list	ap;
	int savederrno;
	
	/* only log if the level is low enough */
	if (dflag < level)
		return;

	savederrno = errno;
	va_start(ap, fmt);
	if (sflag) {
		vfprintf(stderr, fmt, ap);
		fputs("\n", stderr);
	} else
		vsyslog(LOG_DEBUG, fmt, ap);
	va_end(ap);
	errno = savederrno;
}
#endif /* DEBUG */

/* these are like warn() and err(), except for syslog not stderr */
static void
warning(const char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	if (sflag || isatty(STDERR_FILENO)) {
		vfprintf(stderr, fmt, ap);
		fputs("\n", stderr);
	} else
		vsyslog(LOG_INFO, fmt, ap);
	va_end(ap);
}

static void
error(int code, const char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	if (sflag || isatty(STDERR_FILENO)) {
		vfprintf(stderr, fmt, ap);
		fputs("\n", stderr);
	} else
		vsyslog(LOG_ERR, fmt, ap);
	va_end(ap);
	exit(code);
}

/* the follow functions and variables are used in handling HTTP errors */
/* ARGSUSED */
static void
http_error(code, request, msg)
	int	code;
	http_req *request;
	const char *msg;
{
	static	char buf[BUFSIZ];
	char portbuf[20];
	const char *header = http_errors_short(code);
	const char *reason = http_errors_long(code);
	const char *proto = (request && request->hr_proto) ? request->hr_proto : http_11;
	int	size;

	debug((DEBUG_FAT, "http_error %d: %s", code, msg));
	if (header == NULL || reason == NULL)
		error(1, "http_error() failed (short = %p, long = %p)",
		    header, reason);

	if (request && request->hr_serverport)
		snprintf(portbuf, sizeof(portbuf), ":%s", request->hr_serverport);
	else
		portbuf[0] = '\0';

	if (request && request->hr_url) {
		escape_html(request);
		size = snprintf(buf, sizeof buf,
		    "<html><head><title>%s</title></head>\n"
		    "<body><h1>%s</h1>\n"
		    "%s: <pre>%s</pre>\n"
 		    "<hr><address><a href=\"http://%s%s/\">%s%s</a></address>\n"
		    "</body></html>\n",
		    header, header, request->hr_url, reason,
		    myname, portbuf, myname, portbuf);
		if (size >= sizeof buf)
			warning("http_error buffer too small, truncated");
	} else
		size = 0;

	(void)bozoprintf("%s %s\r\n", proto, header);
#ifdef DO_HTPASSWD
	if (code == 401)
		(void)bozoprintf("WWW-Authenticate: Basic realm=\"%s\"\r\n",
		    request && request->hr_authrealm ? request->hr_authrealm :
		    "default realm");
#endif /* DO_HTPASSWD */
	(void)bozoprintf("Content-Type: text/html\r\n");
	(void)bozoprintf("Content-Length: %d\r\n", size);
	(void)bozoprintf("Server: %s\r\n", server_software);
	if (request && request->hr_allow)
		(void)bozoprintf("Allow: %s\r\n", request->hr_allow);
	(void)bozoprintf("\r\n");
	if (size)
		(void)bozoprintf("%s", buf);
	bozoflush(stdout);

	exit(1);
}

/* short map between error code, and short/long messages */
static struct errors_map {
	int	code;			/* HTTP return code */
	const char *shortmsg;		/* short version of message */
	const char *longmsg;		/* long version of message */
} errors_map[] = {
	{ 400,	"400 Bad Request",	"The request was not valid", },
	{ 401,	"401 Unauthorized",	"No authorization", },
	{ 403,	"403 Forbidden",	"Access to this item has been denied", },
	{ 404, 	"404 Not Found",	"This item has not been found", },
	{ 408, 	"408 Request Timeout",	"This request took too long", },
	{ 417,	"417 Expectation Failed","Expectations not available", },
	{ 500,	"500 Internal Error",	"An error occured on the server", },
	{ 501,	"501 Not Implemented",	"This request is not available", },
	{ 0,	NULL,			NULL, },
};

static const char *help = "DANGER! WILL ROBINSON! DANGER!";

static const char *
http_errors_short(code)
	int	code;
{
	struct errors_map *ep;

	for (ep = errors_map; ep->code; ep++)
		if (ep->code == code)
			return (ep->shortmsg);
	return (help);
}

static const char *
http_errors_long(code)
	int	code;
{
	struct errors_map *ep;

	for (ep = errors_map; ep->code; ep++)
		if (ep->code == code)
			return (ep->longmsg);
	return (help);
}

#ifndef NO_SSL_SUPPORT
int
ssl_printf(const char * fmt, ...)
{
	int nbytes;
	char *buf;
	va_list ap;

	/* XXX we need more elegant/proper handling of SSL_write return */
	va_start(ap, fmt);
	if ((nbytes = vasprintf(&buf, fmt, ap)) != -1) 
		SSL_write(bozossl, buf, nbytes);
	va_end(ap);

	return nbytes;
}

ssize_t
ssl_read(int fd, void *buf, size_t nbytes)
{
	ssize_t rbytes;

	/* XXX we need elegant/proper handling of SSL_read return */
	rbytes = (ssize_t)SSL_read(bozossl, buf, nbytes);
	if (1 > rbytes) {
		if (SSL_get_error(bozossl, rbytes) == SSL_ERROR_WANT_READ)
			warning("SSL_ERROR_WANT_READ");
		else
			warning("SSL_ERROR OTHER");
	}

	return rbytes;
}

ssize_t
ssl_write(int fd, const void *buf, size_t nbytes)
{
	ssize_t wbytes;

	/* XXX we need elegant/proper handling of SSL_write return */
	wbytes = (ssize_t)SSL_write(bozossl, buf, nbytes);

	return wbytes;
}

int
ssl_flush(FILE *fp)
{
	/* nothing to see here, move right along */
	return 0;
}
#endif /* NO_SSL_SUPPORT */

/* Below are various modified libc functions */

/*
 * returns -1 in lenp if the string ran out before finding a delimiter,
 * but is otherwise the same as strsep.  Note that the length must be
 * correctly passed in.
 */
static char *
strnsep(strp, delim, lenp)
	char	**strp;
	const char *delim;
	ssize_t	*lenp;
{
	char	*s;
	const	char *spanp;
	int	c, sc;
	char	*tok;

	if ((s = *strp) == NULL)
		return (NULL);
	for (tok = s;;) {
		if (lenp && --(*lenp) == -1)
			return (NULL);
		c = *s++;
		spanp = delim;
		do {
			if ((sc = *spanp++) == c) {
				if (c == 0)
					s = NULL;
				else
					s[-1] = '\0';
				*strp = s;
				return (tok);
			}
		} while (sc != 0);
	}
	/* NOTREACHED */
}

/*
 * inspired by fgetln(3), but works for fd's.  should work identically
 * except it, however, does *not* return the newline, and it does nul
 * terminate the string.
 */
static char *
dgetln(fd, lenp, readfn)
	int	fd;
	ssize_t	*lenp;
	ssize_t	(*readfn)(int, void *, size_t);
{
	static	char *buffer;
	static	ssize_t buflen = 0;
	ssize_t	len;
	int	got_cr = 0;
	char	c, *nbuffer;

	/* initialise */
	if (buflen == 0) {
		buflen = 128;	/* should be plenty for most requests */
		buffer = malloc(buflen);
		if (buffer == NULL) {
			buflen = 0;
			return NULL;
		}
	}
	len = 0;

	/*
	 * we *have* to read one byte at a time, to not break cgi
	 * programs (for we pass stdin off to them).  could fix this
	 * by becoming a fd-passing program instead of just exec'ing
	 * the program
	 */
	for (; readfn(fd, &c, 1) == 1; ) {
		debug((DEBUG_EXPLODING, "dgetln read %c", c));

		if (len == buflen) {
			buflen *= 2;
			debug((DEBUG_EXPLODING,
			    "dgetln: reallocating buffer to buflen %d", buflen));
			nbuffer = realloc(buffer, buflen);
			if (nbuffer == NULL) {
				free(buffer);
				buflen = 0;
				buffer = NULL;
				return NULL;
			}
			buffer = nbuffer;
		}

		buffer[len++] = c;
		if (c == '\r') {
			got_cr = 1;
			continue;
		} else if (c == '\n') {
			/*
			 * HTTP/1.1 spec says to ignore CR and treat
			 * LF as the real line terminator.  even though
			 * the same spec defines CRLF as the line
			 * terminator, it is recommended in section 19.3
			 * to do the LF trick for tolerance.
			 */
			if (got_cr)
				len -= 2;
			else
				len -= 1;
			break;
		}

	}
	buffer[len] = '\0';
	debug((DEBUG_OBESE, "dgetln returns: ``%s'' with len %d", buffer, len));
	*lenp = len;
	return (buffer);
}

static void *
bozorealloc(ptr, size)
	void	*ptr;
	size_t	size;
{
	void	*p;

	p = realloc(ptr, size);
	if (p == NULL)
		http_error(500, NULL, "memory allocation failure");
	return (p);
}

static void *
bozomalloc(size)
	size_t	size;
{
	void	*p;

	p = malloc(size);
	if (p == NULL)
		http_error(500, NULL, "memory allocation failure");
	return (p);
}

static char *
bozostrdup(str)
	const char	*str;
{
	char	*p;

	p = strdup(str);
	if (p == NULL)
		http_error(500, NULL, "memory allocation failure");
	return (p);
}

#ifdef DO_HTPASSWD
/*
 * Check if HTTP authentication is required
 */
static void
check_auth(request, file)
	http_req *request;
	const char *file;
{
	struct stat sb;
	char dir[MAXPATHLEN], authfile[MAXPATHLEN], *basename;
	char user[BUFSIZ], *pass;
	FILE *fp;
	int len;

			/* get dir=dirname(file) */
	snprintf(dir, sizeof(dir), "%s", file);
	if ((basename = strrchr(dir, '/')) == NULL)
		strcpy(dir, ".");
	else {
		*basename++ = '\0';
			/* ensure basename(file) != AUTH_FILE */
		check_special_files(request, basename);
	}
	request->hr_authrealm = dir;

	snprintf(authfile, sizeof(authfile), "%s/%s", dir, AUTH_FILE);
	if (stat(authfile, &sb) < 0) {
		debug((DEBUG_NORMAL,
		    "check_auth realm `%s' dir `%s' authfile `%s' missing",
		    dir, file, authfile));
		return;
	}
	if ((fp = fopen(authfile, "r")) == NULL)
		http_error(403, request, "no permission to open authfile");
	debug((DEBUG_NORMAL,
	    "check_auth realm `%s' dir `%s' authfile `%s' open",
	    dir, file, authfile));
	if (request->hr_authuser && request->hr_authpass) {
		while (fgets(user, sizeof(user), fp) != NULL) {
			len = strlen(user);
			if (user[len-1] == '\n')
				user[--len] = '\0';
			if ((pass = strchr(user, ':')) == NULL)
				continue;
			*pass++ = '\0';
			debug((DEBUG_NORMAL,
			    "check_auth authfile `%s':`%s' client `%s':`%s'",
			    user, pass, request->hr_authuser,
			    request->hr_authpass));
			if (strcmp(request->hr_authuser, user) != 0)
				continue;
			if (strcmp(crypt(request->hr_authpass, pass), pass))
				break;
			fclose(fp);
			return;
		}
	}
	fclose(fp);
	http_error(401, request, "bad auth");
}

/*
 * Decode len bytes starting at in using base64 encoding into out.
 * Result is *not* NUL terminated.
 * Written by Luke Mewburn <lukem@NetBSD.org>
 */
const unsigned char decodetable[] = {
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,  62, 255, 255, 255,  63, 
	 52,  53,  54,  55,  56,  57,  58,  59,  60,  61, 255, 255, 255,   0, 255, 255, 
	255,   0,   1,   2,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14, 
	 15,  16,  17,  18,  19,  20,  21,  22,  23,  24,  25, 255, 255, 255, 255, 255, 
	255,  26,  27,  28,  29,  30,  31,  32,  33,  34,  35,  36,  37,  38,  39,  40, 
	 41,  42,  43,  44,  45,  46,  47,  48,  49,  50,  51, 255, 255, 255, 255, 255, 
};

static ssize_t
base64_decode(in, ilen, out, olen)
	const unsigned char *in;
	size_t ilen;
	unsigned char *out;
	size_t olen;
{
	unsigned char *cp;
	size_t	 i;

	cp = out;
	for (i = 0; i < ilen; i += 4) {
		if (cp + 3 > out + olen)
			return (-1);
#define IN_CHECK(x) \
		if ((x) > sizeof(decodetable) || decodetable[(x)] == 255) \
			    return(-1)

		IN_CHECK(in[i + 0]);
		*(cp++) = decodetable[in[i + 0]] << 2
			| decodetable[in[i + 1]] >> 4;
		IN_CHECK(in[i + 1]);
		*(cp++) = decodetable[in[i + 1]] << 4
			| decodetable[in[i + 2]] >> 2;
		IN_CHECK(in[i + 2]);
		*(cp++) = decodetable[in[i + 2]] << 6
			| decodetable[in[i + 3]];
#undef IN_CHECK
	}
	while (in[i - 1] == '=')
		cp--,i--;
	return (cp - out);
}
#endif /* DO_HTPASSWD */
