/*	$NetBSD: btdevctl.c,v 1.7 2008/07/21 13:36:57 lukem Exp $	*/

/*-
 * Copyright (c) 2006 Itronix Inc.
 * All rights reserved.
 *
 * Written by Iain Hibbert for Itronix Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of Itronix Inc. may not be used to endorse
 *    or promote products derived from this software without specific
 *    prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ITRONIX INC. ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL ITRONIX INC. BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/ioctl.h>

#include <bluetooth.h>
#include <ctype.h>
#include <err.h>
#include <fcntl.h>
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <dev/bluetooth/btdev.h>

#include "btdevctl.h"

#define BTHUB_PATH		"/dev/bthub0"

int main(int, char *[]);
void usage(void);
char *uppercase(const char *);
int bthub_devioctl(unsigned long, btdevctl_db_t);

int
main(int argc, char *argv[])
{
	btdevctl_db_t dev;
	bdaddr_t laddr, raddr;
	const char *service;
	int ch, query, verbose, attach, detach, set, mode;

	bdaddr_copy(&laddr, BDADDR_ANY);
	bdaddr_copy(&raddr, BDADDR_ANY);
	service = NULL;
	mode = NULL;
	query = false;
	verbose = false;
	attach = false;
	detach = false;
	set = false;
	mode = -1;

	while ((ch = getopt(argc, argv, "Aa:Dd:hm:qs:v")) != -1) {
		switch (ch) {
		case 'A': /* Attach device */
			attach = true;
			break;

		case 'a': /* remote address */
			if (!bt_aton(optarg, &raddr)) {
				struct hostent  *he = NULL;

				if ((he = bt_gethostbyname(optarg)) == NULL)
					errx(EXIT_FAILURE, "%s: %s",
						optarg, hstrerror(h_errno));

				bdaddr_copy(&raddr, (bdaddr_t *)he->h_addr);
			}
			break;

		case 'D': /* Detach device */
			detach = true;
			break;

		case 'd': /* local device address */
			if (!bt_devaddr(optarg, &laddr))
				err(EXIT_FAILURE, "%s", optarg);
			break;

		case 'm': /* link mode */
			if (strcasecmp(optarg, "none") == 0)
				mode = BTDEV_MODE_NONE;
			else if (strcasecmp(optarg, "auth") == 0)
				mode = BTDEV_MODE_AUTH;
			else if (strcasecmp(optarg, "encrypt") == 0)
				mode = BTDEV_MODE_ENCRYPT;
			else if (strcasecmp(optarg, "secure") == 0)
				mode = BTDEV_MODE_SECURE;
			else
				errx(EXIT_FAILURE, "%s: unknown mode", optarg);
			break;

		case 'q':
			query = true;
			break;

		case 's': /* service */
			service = uppercase(optarg);
			break;

		case 'v': /* verbose */
			verbose = true;
			break;

		case 'h':
		default:
			usage();
		}
	}

	argc -= optind;
	argv += optind;

	if (argc > 0
	    || (attach == true && detach == true)
	    || bdaddr_any(&laddr)
	    || bdaddr_any(&raddr)
	    || service == NULL)
		usage();

	if (attach == false && detach == false)
		verbose = true;

	dev = db_get(&laddr, &raddr, service);
	if (dev == NULL || query == true) {
		if (verbose == true)
			printf("Performing SDP query for service '%s'..\n", service);

		dev = cfg_query(&laddr, &raddr, service);
		if (dev == NULL)
			errx(EXIT_FAILURE, "%s/%s not found", bt_ntoa(&raddr, NULL), service);

		set = true;
	}

	if (mode != -1) {
		dev->bd_mode = mode;
		set = true;
	}

	if (set == true && !db_set(dev, &laddr, &raddr, service))
		errx(EXIT_FAILURE, "service store failed");

	/* add binary local-bdaddr */
	bdaddr_copy(&dev->bd_laddr, &laddr);

	/* add binary remote-bdaddr */
	bdaddr_copy(&dev->bd_raddr, &raddr);

	/* add service name */
	dev->bd_type = db_gettype(service);

	if (verbose == true)
		cfg_print(dev);

	if (attach == true)
		bthub_devioctl(BTDEV_ATTACH, dev);

	if (detach == true)
		bthub_devioctl(BTDEV_DETACH, dev);

	exit(EXIT_SUCCESS);
}

void
usage(void)
{
	extern const char *__progname;

	fprintf(stderr,
		"usage: %s [-A | -D] [-qv] [-m mode] -a address -d device -s service\n"
		"Where:\n"
		"\t-A           attach device\n"
		"\t-a address   remote device address\n"
		"\t-D           detach device\n"
		"\t-d device    local device address\n"
		"\t-m mode      link mode\n"
		"\t-q           force SDP query\n"
		"\t-s service   remote service\n"
		"\t-v           verbose\n"
		"", __progname);

	exit(EXIT_FAILURE);
}

char *
uppercase(const char *arg)
{
	char *str, *ptr;

	str = strdup(arg);
	if (str == NULL)
		err(EXIT_FAILURE, "strdup");

	for (ptr = str ; *ptr ; ptr++)
		*ptr = (char)toupper((int)*ptr);

	return str;
}

int
bthub_devioctl(unsigned long cmd, btdevctl_db_t dev)
{
	struct btdev_attach_args *bd = dev;
	int fd;

	fd = open(BTHUB_PATH, O_WRONLY, 0);
	if (fd < 0)
		err(EXIT_FAILURE, "%s", BTHUB_PATH);

	if (ioctl(fd, cmd, bd) != 0)
		err(EXIT_FAILURE, "%s", BTHUB_PATH);

	close(fd);
	return EXIT_SUCCESS;
}
