/* Bluefish HTML Editor
 * configure_cb.c
 *
 * Copyright (C) 1999 Olivier Sessink
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "default_include.h"

#include <string.h>
#include <locale.h>

#include "bluefish.h"
#include "init.h"
#include "gtk_easy.h"
#include "menu.h"
#include "interface.h"
#include "highlight.h"
#include "document.h"			/* set fonts */

typedef enum { none, file, font } Textra_but;

GtkWidget *font_e, *browser_e, *weblint_e, *tabfont_e, *tabpos_c;
GtkWidget *width_s, *height_s, *tab_s, *thumbtype_c, *thumbstring_e;
GtkWidget *full_li_c, *full_p_c, *lowercase_tags_c, *word_wrap_c, *line_wrap_c, *fontset_c, *link_management_c, *defaulthighlight_c;
GtkWidget *cont_high_up_c, *cont_highlight_full_c, *default_charset_c;

#ifdef WITH_SPC
GtkWidget *spc_command_e, *spc_lang_e, *spc_acc_comp_c, *spc_use_esc_c, *spc_use_pers_dict_c, *spc_pers_dict_e;
#endif

/**************************************************************************/
void configure_cb(GtkWidget * widget, gpointer data);
static GtkWidget *combo_entry2(gchar * labeltext, gchar * which_config_string, GtkWidget * box, GList * poplist);
static GtkWidget *check_entry2(gchar * labeltext, gint which_config_int, GtkWidget * box);
static void apply_choose_font(GtkWidget * widget, gpointer data);
static void ok_choose_font(GtkWidget * widget, gpointer data);
static void delete_choose_font(GtkWidget * widget, GdkEventAny * event, gpointer data);
static void close_choose_font(GtkWidget * widget, gpointer data);
static void choose_font(GtkWidget * widget, gpointer data);
static void integer_apply(gint * config_var, GtkWidget * spinbutton);
static void check_apply(gint * config_var, GtkWidget * check_button);

/**************************************************************************/

/*************************
 * 
 * start of functions
 *
 *************************/

static GtkWidget *string_entry2(gchar * labeltext, gchar * which_config_string, GtkWidget * box, Textra_but with_but)
{
	GtkWidget *hbox, *return_widget, *font_b=NULL;

	hbox = gtk_hbox_new(FALSE,3);
	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(labeltext), FALSE, FALSE, 3);
	return_widget = boxed_entry_with_text(which_config_string, 1023, hbox);
	gtk_box_pack_start(GTK_BOX(box), hbox, FALSE, FALSE, 3);
	if (with_but == file) {
		gtk_box_pack_start(GTK_BOX(hbox), file_but_new(return_widget, gtk_widget_get_toplevel(box), 1), FALSE, FALSE, 3);
	} else if (with_but == font) {
		font_b = gtk_button_new_with_label(_("Font..."));
		gtk_signal_connect(GTK_OBJECT(font_b), "clicked", GTK_SIGNAL_FUNC(choose_font), return_widget);
		gtk_box_pack_start(GTK_BOX(hbox), font_b, FALSE, FALSE, 0);
	}
	return return_widget;
}

static GtkWidget *combo_entry2(gchar * labeltext, gchar * which_config_string, GtkWidget * box, GList * poplist)
{
	GtkWidget *return_widget;
	GtkWidget *hbox;

	hbox = gtk_hbox_new(FALSE,3);
	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(labeltext), FALSE, FALSE, 3);
	return_widget = boxed_combo_with_popdown(which_config_string, poplist, 0, hbox);
	gtk_box_pack_start(GTK_BOX(box), hbox, FALSE, FALSE, 3);
	return return_widget;
}

static GtkWidget *check_entry2(gchar * labeltext, gint which_config_int, GtkWidget * box)
{
	return boxed_checkbut_with_value(labeltext, which_config_int, box);
}

static GtkWidget *spin_button2(gchar *labeltext, gint which_config_int, GtkWidget * box, gfloat lower, gfloat upper) {
	GtkWidget *return_widget;
	GtkWidget *hbox;
	GtkObject *adjust;
	gfloat step_increment, page_increment;

	step_increment = (upper - lower)/1000;
	if (step_increment < 1) {
		step_increment = 1;
	}
	page_increment = (upper - lower)/20;
	if (page_increment < 10) {
		page_increment = 10;
	}
	hbox = gtk_hbox_new(FALSE,3);
	adjust = gtk_adjustment_new((1.0 * which_config_int), lower, upper, step_increment ,page_increment , 0);
	return_widget = gtk_spin_button_new(GTK_ADJUSTMENT(adjust), 0.1, 0);
	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(labeltext), FALSE, FALSE, 3);
	gtk_box_pack_start(GTK_BOX(hbox), return_widget, TRUE, TRUE, 3);
	gtk_box_pack_start(GTK_BOX(box), hbox, FALSE, FALSE, 3);
	return return_widget;
}

/*
 */
static void apply_choose_font(GtkWidget * widget, gpointer data)
{
	gchar *font;
	GtkWidget *font_dialog;

	font_dialog = gtk_widget_get_toplevel(widget);
	font = gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG(font_dialog));
	DEBUG_MSG("apply_choose_font, font=%s\n", font);
	gtk_entry_set_text(GTK_ENTRY(data), font);
};

/*
 */
static void ok_choose_font(GtkWidget * widget, gpointer data)
{
	apply_choose_font(widget, data);
	delete_choose_font(widget, NULL, data);
};

/*
 */
static void delete_choose_font(GtkWidget * widget, GdkEventAny * event, gpointer data)
{
	GtkWidget *win;
	DEBUG_MSG("delete_choose_font, data=%p\n", data);
	win = gtk_widget_get_toplevel(widget);
	gtk_grab_remove(GTK_WIDGET(win));
	gtk_widget_destroy(GTK_WIDGET(win));
};

/*
 */
static void close_choose_font(GtkWidget * widget, gpointer data)
{
	delete_choose_font(widget, NULL, data);
};

/*
 * Function: choose_font
 * Arguments:
 * 	widget	- called widget
 * 	data	- pointer to entry widget
 * Return value:
 * 	void
 * Desciption:
 * 	Callback function for calling GtkFontSelection and install
 * 	choosed font string in entry 8-)
 */
static void choose_font(GtkWidget * widget, gpointer data)
{
	GtkWidget *font_dialog;
	gchar *font;

	font_dialog = gtk_font_selection_dialog_new(_("Font Selection"));

	/* Set font in font selection dialog */
	font = gtk_editable_get_chars(GTK_EDITABLE(data), 0, -1);
	gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(font_dialog), font);
	g_free(font);

	/* connect all needing signals */
	gtk_signal_connect(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(font_dialog)->ok_button),
					   "clicked", GTK_SIGNAL_FUNC(ok_choose_font), (gpointer) data);

	gtk_signal_connect(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(font_dialog)->apply_button),
					   "clicked", GTK_SIGNAL_FUNC(apply_choose_font), (gpointer) data);

	gtk_signal_connect(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(font_dialog)->cancel_button),
					   "clicked", GTK_SIGNAL_FUNC(close_choose_font), (gpointer) data);

	gtk_signal_connect(GTK_OBJECT(font_dialog), "delete_event", GTK_SIGNAL_FUNC(delete_choose_font), (gpointer) data);

	DEBUG_MSG("font_dialog=%p\n", font_dialog);
	gtk_widget_show_all(font_dialog);

	/* make modal */
	gtk_grab_add(font_dialog);
};


static void save_n_apply_config()
{
	GList *tmplist;
	Tdocument *tmpdoc;

	save_config_files();
	
	/* actually try to apply these config values */
	gtk_window_set_policy(GTK_WINDOW(main_v->main_window), FALSE, FALSE, FALSE);
	gtk_widget_set_usize(GTK_WIDGET(main_v->main_window), main_v->props.main_window_w, main_v->props.main_window_h);
	gtk_window_set_policy(GTK_WINDOW(main_v->main_window), TRUE, TRUE, FALSE);
	tmplist = g_list_first(main_v->documentlist);
	while (tmplist) {
		tmpdoc = (Tdocument *) tmplist->data;
		if (tmpdoc) {
			document_set_fonts(main_v->props.cfg_editor_font, main_v->props.cfg_tab_font, tmpdoc);
			document_set_wrap(main_v->props.word_wrap, main_v->props.line_wrap, tmpdoc);
			GTK_TEXT(tmpdoc->textbox)->default_tab_width = main_v->props.cfg_editor_tabwidth;
		}						/* if (tmpdoc) */
		tmplist = g_list_next(tmplist);
	}
	notebook_set_tab_pos(main_v->props.cfg_tab_pos);
	make_syntax_struct_list();
	make_external_menu_entries(1);
	make_external_menu_entries(2);

	statusbar_message(_("Configuration saved and applied succesfully"), 1000);
	DEBUG_MSG("apply_config, finished applying config values\n");
}


static void configure_ok_lcb(GtkWidget * button, gpointer entry) {

	statusbar_message(_("Saving and applying configuration"), 1000);
	string_apply(&main_v->props.cfg_browser_cline, browser_e);
	string_apply(&main_v->props.cfg_weblint_cline, weblint_e);
	string_apply(&main_v->props.cfg_editor_font, font_e);
	string_apply(&main_v->props.cfg_tab_font, tabfont_e);
	string_apply(&main_v->props.cfg_tab_pos, GTK_COMBO(tabpos_c)->entry);
	string_apply(&main_v->props.cfg_thumbnailtype, GTK_COMBO(thumbtype_c)->entry);
	string_apply(&main_v->props.cfg_thumbnailstring, thumbstring_e);
	integer_apply(&main_v->props.main_window_w, width_s);
	integer_apply(&main_v->props.main_window_h, height_s);
	integer_apply(&main_v->props.cfg_editor_tabwidth, tab_s);
	check_apply(&main_v->props.full_p, full_p_c);
	check_apply(&main_v->props.full_li, full_li_c);
	check_apply(&main_v->props.word_wrap, word_wrap_c);
	check_apply(&main_v->props.line_wrap, line_wrap_c);
	check_apply(&main_v->props.fontset, fontset_c);
	check_apply(&main_v->props.link_management, link_management_c);
	check_apply(&main_v->props.defaulthighlight, defaulthighlight_c);
	check_apply(&main_v->props.lowercase_tags, lowercase_tags_c);
	check_apply(&main_v->props.cont_highlight_update, cont_high_up_c);
	check_apply(&main_v->props.cont_highlight_full, cont_highlight_full_c);
	string_apply(&main_v->props.default_charset, GTK_COMBO(default_charset_c)->entry);
#ifdef WITH_SPC
	string_apply(&main_v->props.cfg_spc_cline, spc_command_e);
	string_apply(&main_v->props.cfg_spc_lang, spc_lang_e);
	check_apply(&main_v->props.spc_accept_compound, spc_acc_comp_c);
	check_apply(&main_v->props.spc_use_esc_chars, spc_use_esc_c);
	check_apply(&main_v->props.spc_use_pers_dict, spc_use_pers_dict_c);
	string_apply(&main_v->props.spc_pers_dict, spc_pers_dict_e);
#endif
	if (main_v->props.syntax_configstrings == NULL) {
		reset_syntax_highlighting_lcb(NULL, NULL);
	}
	gtk_main_quit();

	save_n_apply_config();
}




void configure_cb(GtkWidget * widget, gpointer data)
{
	GtkWidget *frame, *dialog;
	GtkWidget *ok;
	GtkWidget *cancel;
	GtkWidget *vbox;
	GtkWidget *notebook;
	GList *poplist = NULL;
	gint count;

	/* create the widgets */
	DEBUG_MSG("configure_cb, started\n");
	count = bf_statusbar_message(_("Configuration Window"));
	dialog = gtk_dialog_new();
	gtk_signal_connect(GTK_OBJECT(dialog), "delete_event", GTK_SIGNAL_FUNC(gtk_main_quit), NULL);
	gtk_signal_connect(GTK_OBJECT(dialog), "destroy", GTK_SIGNAL_FUNC(gtk_main_quit), NULL);
	/* give the dialog a name */
	gtk_widget_realize(dialog);

	notebook = gtk_notebook_new();
	gtk_notebook_set_homogeneous_tabs(GTK_NOTEBOOK(notebook), TRUE);
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_BOTTOM);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), notebook, TRUE, TRUE, 7);

	/* editor */
	frame = gtk_frame_new(_("Editor options"));
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	font_e = string_entry2(_("Default font"), main_v->props.cfg_editor_font, vbox, font);

	tab_s = spin_button2(_("Tab width"), main_v->props.cfg_editor_tabwidth, vbox, 1.0, 20.0);

	word_wrap_c = check_entry2(_("Word wrap"), main_v->props.word_wrap, vbox);
	line_wrap_c = check_entry2(_("Line wrap"), main_v->props.line_wrap, vbox);
	defaulthighlight_c = check_entry2(_("Highlight by default"), main_v->props.defaulthighlight, vbox);
	cont_high_up_c = check_entry2(_("Update syntax highlighting continuous"), main_v->props.cont_highlight_update, vbox);
	cont_highlight_full_c = check_entry2(_("Update full widget instead of the current line (slower)"), main_v->props.cont_highlight_full, vbox);
	link_management_c = check_entry2(_("Automatic link management"), main_v->props.link_management, vbox);

	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, gtk_label_new(_("Editor")));
	/* HTML options */
	frame = gtk_frame_new(_("HTML options"));
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	full_li_c = check_entry2(_("Insert closing list-item tag </LI>"), main_v->props.full_li, vbox);
	full_p_c = check_entry2(_("Insert closing paragraph tag </P>"), main_v->props.full_p, vbox);
	lowercase_tags_c = check_entry2(_("Insert lowercase HTML tags (Needed for WML!)"), main_v->props.lowercase_tags, vbox);

	poplist = NULL;
	poplist = g_list_append(poplist, "other");
	poplist = g_list_append(poplist, "iso8859-1");
	poplist = g_list_append(poplist, "iso8859-2");
	poplist = g_list_append(poplist, "iso8859-3");
	poplist = g_list_append(poplist, "iso8859-4");
	poplist = g_list_append(poplist, "iso8859-5");
	poplist = g_list_append(poplist, "iso8859-6");
	poplist = g_list_append(poplist, "iso8859-7");
	poplist = g_list_append(poplist, "iso8859-8");
	poplist = g_list_append(poplist, "iso8859-9");
	poplist = g_list_append(poplist, "iso8859-10");
	poplist = g_list_append(poplist, "iso8859-11");
	poplist = g_list_append(poplist, "iso8859-12");
	poplist = g_list_append(poplist, "iso8859-13");
	poplist = g_list_append(poplist, "iso8859-14");
	poplist = g_list_append(poplist, "iso8859-15");

	default_charset_c = combo_entry2(_("Default charset"), main_v->props.default_charset, vbox, poplist);
	g_list_free(poplist);

	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, gtk_label_new(_("HTML")));

	/* main GUI */
	frame = gtk_frame_new(_("Main GUI options"));
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	tabfont_e = string_entry2(_("Notebook tab fontset"), main_v->props.cfg_tab_font, vbox, font);
	fontset_c = check_entry2(_("Font is a fontset"), main_v->props.fontset, vbox);

	poplist = NULL;
	poplist = g_list_append(poplist, "top");
	poplist = g_list_append(poplist, "bottom");
	poplist = g_list_append(poplist, "left");
	poplist = g_list_append(poplist, "right");

	tabpos_c = combo_entry2(_("Notebook position"), main_v->props.cfg_tab_pos, vbox, poplist);
	g_list_free(poplist);

	width_s = spin_button2(_("Initial width"), main_v->props.main_window_w, vbox, 50.0, 2000.0);
	height_s = spin_button2(_("Initial height"), main_v->props.main_window_h, vbox, 50.0, 2000.0);

	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, gtk_label_new(_("GUI")));

	/* image dialog */
	frame = gtk_frame_new(_("Image dialog options"));
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	poplist = NULL;
	poplist = g_list_append(poplist, "png");
	poplist = g_list_append(poplist, "jpg");
	poplist = g_list_append(poplist, "gif");
	thumbtype_c = combo_entry2(_("Thumbnail type"), main_v->props.cfg_thumbnailtype, vbox, poplist);
	g_list_free(poplist);

	thumbstring_e = string_entry2(_("Thumbnail extension string"), main_v->props.cfg_thumbnailstring, vbox, none);

	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, gtk_label_new(_("Image dialog")));

	/* external programs */
	frame = gtk_frame_new(_("External program options"));
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	browser_e = string_entry2(_("Browser command"), main_v->props.cfg_browser_cline, vbox, file);
	weblint_e = string_entry2(_("Weblint command"), main_v->props.cfg_weblint_cline, vbox, file);
#ifdef WITH_SPC
	spc_command_e = string_entry2(_("Ispell command"), main_v->props.cfg_spc_cline, vbox, file );
	spc_lang_e = string_entry2(_("Default language"), main_v->props.cfg_spc_lang, vbox, none);
	spc_acc_comp_c = check_entry2(_("Accept compound words"), main_v->props.spc_accept_compound, vbox);
	spc_use_esc_c = check_entry2(_("Specify additional characters that might form a word"), main_v->props.spc_use_esc_chars, vbox);
	spc_use_pers_dict_c = check_entry2(_("Use personal dictionary"), main_v->props.spc_use_pers_dict, vbox);
	spc_pers_dict_e = string_entry2(_("Personal dictionary"), main_v->props.spc_pers_dict, vbox, file);
#endif
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, gtk_label_new(_("External")));

	ok = bf_stock_button(_(" Save & close "), GTK_SIGNAL_FUNC(configure_ok_lcb), dialog);
	cancel = bf_stock_cancel_button(GTK_SIGNAL_FUNC(gtk_main_quit), NULL);

	/* pack the widgets */
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), ok, TRUE, TRUE, 0);
	gtk_box_pack_end(GTK_BOX(GTK_DIALOG(dialog)->action_area), cancel, TRUE, TRUE, 0);

	/* show the widgets */
	gtk_widget_show_all(dialog);

	gtk_grab_add(dialog);
	gtk_main();
	DEBUG_MSG("configure_cb, just after gtk_main()\n");
  	statusbar_remove(GINT_TO_POINTER(count));
	window_destroy(dialog);
}

static void integer_apply(gint * config_var, GtkWidget * spinbutton)
{

	*config_var = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spinbutton));
}

static void check_apply(gint * config_var, GtkWidget * check_button)
{

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check_button))) {
		*config_var = 1;
	} else {
		*config_var = 0;
	}
}

