/* Bluefish HTML Editor
 * stringlist.c - functions that deal with stringlists
 *
 * Copyright (C) 1999 Olivier Sessink and Hylke van der Schaaf
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtk/gtk.h>

#include <stdio.h>

#include <locale.h>
#include <libintl.h>
#include "intl.h"


#include "bluefish.h"
#include "html.h"
#include "stringlist.h"
#include "gtk_easy.h"
#include "debug.h"

#define STRING_MAX_SIZE 1024

typedef struct {
	GtkWidget *window;
	GtkWidget *ins_upd_entry;
	GtkWidget *clist;
	gint last_row;
	GList *the_list;
} Teditstringlist;

/************************************************************************/
GList *edit_stringlist(GList * which_list, gchar * title, gint what_list);
void edit_stringlist_select_row(GtkWidget * widget, gint row, gint column, GdkEventButton * event, gpointer data);
gint free_stringlist(GList * which_list);
GList *get_stringlist(gchar * filename, GList * which_list);
gint put_stringlist(gchar * filename, GList * which_list);
GList *add_to_stringlist(GList * which_list, gchar * string);
static void edit_stringlist_unselect_row(GtkWidget * widget, gint row, gint column, GdkEventButton * event, gpointer data);
static void edit_stringlist_up_lcb(GtkWidget * widget, gpointer data);
static void edit_stringlist_down_lcb(GtkWidget * widget, gpointer data);
static void edit_stringlist_add(GtkWidget * widget, gpointer data);
static void edit_stringlist_update(GtkWidget * widget, gpointer data);
static void edit_stringlist_delete(GtkWidget * widget, gpointer data);
static void edit_stringlist_sort(GtkWidget * widget, gpointer data);
static void edit_stringlist_ok(GtkWidget * widget, Teditstringlist * data);
static void edit_stringlist_close(GtkWidget * widget, gpointer data);


/************************************************************************/

GList *edit_stringlist(GList * which_list, gchar * title, gint what_list)
{

	Teditstringlist *edits;
	GList *tmplist;
	GtkWidget *but_add, *but_update, *but_del, *but_up, *but_down, *but_cancel, *but_ok, *but_var = NULL, *but_sort;
	GtkWidget *table, *scrolwin, *vbox;
	gchar *tmplistentry[1];

	edits = g_malloc(sizeof(Teditstringlist));
	edits->the_list = NULL;
	edits->window = window_with_title(title, GTK_WIN_POS_MOUSE, GTK_WINDOW_TOPLEVEL, 10);
	/* gtk_signal_connect(GTK_OBJECT(edits->window), "destroy",
	   GTK_SIGNAL_FUNC(edit_stringlist_cancel), edits); */
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(edits->window), vbox);

	switch (what_list) {
	case 1:
		/* files */
		but_var = file_but_new(edits->ins_upd_entry, edits->window, 1);
		break;
	case 2:
		/* color */
		but_var = color_but_new(edits->ins_upd_entry, edits->window);
		break;
	case 4:
		/* url */
		but_var = file_but_new(edits->ins_upd_entry, edits->window, 0);
		break;
	case 7:
		/* external filters and commands */
		gtk_box_pack_start(GTK_BOX(vbox), gtk_label_new(_("The syntax is \"label/ command\", so not slash in the label! ")), FALSE,
						   FALSE, 0);
		break;
	case 8:
		/* syntax highlighting patterns */
		gtk_box_pack_start(GTK_BOX(vbox), gtk_label_new(_("The syntax is \"#?????? regular-expression\", only one space between them! ")),
						   FALSE, FALSE, 0);
		break;
	default:
		/* targets(5), fonts(3) meta_cb (6) */
		but_var = NULL;
		break;
	}

	table = gtk_table_new(8, 4, 1);
	DEBUG_MSG("edit_stringlist, edits(%p), which_list(%p), what_list=%d\n", edits, which_list, what_list);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

	scrolwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolwin), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_table_attach_defaults(GTK_TABLE(table), scrolwin, 0, 4, 1, 7);

	edits->clist = gtk_clist_new(1);
	gtk_clist_set_reorderable(GTK_CLIST(edits->clist), TRUE);
	/* gtk_clist_set_selection_mode(GTK_CLIST(edits->clist), GTK_SELECTION_MULTIPLE); */

	gtk_signal_connect(GTK_OBJECT(edits->clist), "select_row", GTK_SIGNAL_FUNC(edit_stringlist_select_row), edits);
	gtk_signal_connect(GTK_OBJECT(edits->clist), "unselect_row", GTK_SIGNAL_FUNC(edit_stringlist_unselect_row), edits);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolwin), edits->clist);
	gtk_clist_set_reorderable(GTK_CLIST(edits->clist), TRUE);
	gtk_clist_freeze(GTK_CLIST(edits->clist));
	tmplist = g_list_first(which_list);
	while (tmplist != NULL) {
		DEBUG_MSG("edit_stringlist, tmplist->data(%p)=%s\n", tmplist->data, (gchar *) tmplist->data);
		tmplistentry[0] = (gchar *) tmplist->data;
		gtk_clist_append(GTK_CLIST(edits->clist), tmplistentry);
		tmplist = g_list_next(tmplist);
	}
	gtk_clist_thaw(GTK_CLIST(edits->clist));
	DEBUG_MSG("edit_stringlist, entry to go\n");
	edits->ins_upd_entry = gtk_entry_new_with_max_length(900);
	gtk_table_attach_defaults(GTK_TABLE(table), edits->ins_upd_entry, 0, 4, 0, 1);
	gtk_widget_realize(GTK_WIDGET(edits->window));
	if (but_var != NULL) {
		gtk_table_attach_defaults(GTK_TABLE(table), but_var, 4, 5, 0, 1);
		GTK_WIDGET_SET_FLAGS(but_var, GTK_CAN_DEFAULT);
	}

	but_add = gtk_button_new_with_label(_(" Add "));
	but_update = gtk_button_new_with_label(_(" Update "));
	but_del = gtk_button_new_with_label(_(" Delete "));
	but_up = gtk_button_new_with_label(_(" Up "));
	but_down = gtk_button_new_with_label(_(" Down "));
	but_sort = gtk_button_new_with_label(_(" Sort "));

	gtk_signal_connect(GTK_OBJECT(but_add), "clicked", edit_stringlist_add, edits);
	gtk_signal_connect(GTK_OBJECT(but_update), "clicked", edit_stringlist_update, edits);
	gtk_signal_connect(GTK_OBJECT(but_del), "clicked", edit_stringlist_delete, edits);
	gtk_signal_connect(GTK_OBJECT(but_up), "clicked", edit_stringlist_up_lcb, edits);
	gtk_signal_connect(GTK_OBJECT(but_down), "clicked", edit_stringlist_down_lcb, edits);
	gtk_signal_connect(GTK_OBJECT(but_sort), "clicked", edit_stringlist_sort, edits);

	GTK_WIDGET_SET_FLAGS(but_add, GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(but_update, GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(but_del, GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(but_up, GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(but_down, GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(but_sort, GTK_CAN_DEFAULT);

	gtk_table_attach_defaults(GTK_TABLE(table), but_add, 4, 5, 1, 2);
	gtk_table_attach_defaults(GTK_TABLE(table), but_update, 4, 5, 2, 3);
	gtk_table_attach_defaults(GTK_TABLE(table), but_del, 4, 5, 3, 4);
	gtk_table_attach_defaults(GTK_TABLE(table), but_up, 4, 5, 4, 5);
	gtk_table_attach_defaults(GTK_TABLE(table), but_down, 4, 5, 5, 6);
	gtk_table_attach_defaults(GTK_TABLE(table), but_sort, 4, 5, 6, 7);

	but_cancel = bf_stock_cancel_button(edit_stringlist_close, edits);
	but_ok = bf_stock_ok_button(edit_stringlist_ok, edits);
	GTK_WIDGET_SET_FLAGS(but_cancel, GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(but_ok, GTK_CAN_DEFAULT);
	gtk_table_attach_defaults(GTK_TABLE(table), but_cancel, 2, 4, 7, 8);
	gtk_table_attach_defaults(GTK_TABLE(table), but_ok, 0, 2, 7, 8);

	gtk_widget_show_all(edits->window);
	gtk_grab_add(GTK_WIDGET(edits->window));
	gtk_window_set_default(GTK_WINDOW(edits->window), but_ok);
	gtk_window_set_transient_for(GTK_WINDOW(edits->window), GTK_WINDOW(main_v->main_window));
	gtk_main();


	if (edits->the_list != NULL) {
		free_stringlist(which_list);
		which_list = edits->the_list;
	}
	gtk_signal_handlers_destroy(GTK_OBJECT(edits->window));
	gtk_widget_destroy(edits->window);
	g_free(edits);

	DEBUG_MSG("edit_stringlist, about to return %p\n", which_list);
	return which_list;
}


void edit_stringlist_select_row(GtkWidget * widget, gint row, gint column, GdkEventButton * event, gpointer data)
{
	gchar *text;

	gtk_clist_get_text(GTK_CLIST(((Teditstringlist *) data)->clist), row, column, &text);
	gtk_entry_set_text(GTK_ENTRY(((Teditstringlist *) data)->ins_upd_entry), text);
	((Teditstringlist *) data)->last_row = row;
	DEBUG_MSG("edit_stringlist_select_row, row=%d, column=%d\n", row, column);

}

static void edit_stringlist_unselect_row(GtkWidget * widget, gint row, gint column, GdkEventButton * event, gpointer data)
{
	((Teditstringlist *) data)->last_row = -1;
}

static void edit_stringlist_up_lcb(GtkWidget * widget, gpointer data)
{

	Teditstringlist *edits;

	edits = (Teditstringlist *) data;
	DEBUG_MSG("edit_stringlist_up_lcb, started, last_row=%d\n", edits->last_row);
	if (edits->last_row >= 1) {
		gtk_clist_swap_rows(GTK_CLIST(edits->clist), edits->last_row, edits->last_row - 1);
		edits->last_row--;
	}

}

static void edit_stringlist_down_lcb(GtkWidget * widget, gpointer data)
{

	Teditstringlist *edits;

	edits = (Teditstringlist *) data;
	DEBUG_MSG("edit_stringlist_down_lcb, started, last_row=%d, clist->rows=%d\n", edits->last_row, GTK_CLIST(edits->clist)->rows);
	if (edits->last_row < GTK_CLIST(edits->clist)->rows - 1) {
		gtk_clist_swap_rows(GTK_CLIST(edits->clist), edits->last_row, edits->last_row + 1);
		edits->last_row++;
	}

}

static void edit_stringlist_add(GtkWidget * widget, gpointer data)
{
	gchar *tmplistentry[1];

	tmplistentry[0] = gtk_entry_get_text(GTK_ENTRY(((Teditstringlist *) data)->ins_upd_entry));;
	gtk_clist_append(GTK_CLIST(((Teditstringlist *) data)->clist), tmplistentry);

}

static void edit_stringlist_update(GtkWidget * widget, gpointer data)
{


	gtk_clist_set_text(GTK_CLIST(((Teditstringlist *) data)->clist)
					   , ((Teditstringlist *) data)->last_row, 0, gtk_entry_get_text(GTK_ENTRY(((Teditstringlist *) data)->ins_upd_entry)));

}

static void edit_stringlist_delete(GtkWidget * widget, gpointer data)
{
	gtk_clist_remove(GTK_CLIST(((Teditstringlist *) data)->clist), ((Teditstringlist *) data)->last_row);
}

static void edit_stringlist_sort(GtkWidget * widget, gpointer data)
{
	gtk_clist_set_auto_sort(GTK_CLIST(((Teditstringlist *) data)->clist), 1);
}

static void edit_stringlist_ok(GtkWidget * widget, Teditstringlist * data)
{

	gchar *tmpstring, *tmpstring2;
	gint row, retval;

	row = 0;
	retval = gtk_clist_get_text(GTK_CLIST(data->clist), row, 0, &tmpstring);
	while (retval == 1) {
		DEBUG_MSG("edit_stringlist_ok, retval=%d\n", retval);
		tmpstring2 = g_strdup(tmpstring);
		g_assert(tmpstring2);
		data->the_list = g_list_append(data->the_list, tmpstring2);
		row++;
		retval = gtk_clist_get_text(GTK_CLIST(data->clist), row, 0, &tmpstring);
	}
	edit_stringlist_close(NULL, NULL);

}

static void edit_stringlist_close(GtkWidget * widget, gpointer data)
{
	gtk_main_quit();
}

gint free_stringlist(GList * which_list)
{
	GList *tmplist;

	DEBUG_MSG("free_stringlist, started\n");

	tmplist = g_list_first(which_list);
	while (tmplist != NULL) {
		DEBUG_MSG("free_stringlist, tmplist->data(%p)\n", tmplist->data);
		DEBUG_MSG("free_stringlist, tmplist->data=%s\n", (gchar *) tmplist->data);
		g_free(tmplist->data);
		tmplist = g_list_next(tmplist);
	}
	DEBUG_MSG("free_stringlist, strings free-ed, about to free list itself\n");
	g_list_free(which_list);
	which_list = NULL;
	return 1;
}

GList *get_stringlist(gchar * filename, GList * which_list)
{
	char *tempstr, *tmpbuf;
	FILE *fd;

	DEBUG_MSG("get_stringlist, started with filename=%s\n", filename);
	if (filename == NULL) {
		return 0;
	}
	tmpbuf = g_malloc(STRING_MAX_SIZE);
	if (file_exists_and_readable(filename)) {
		DEBUG_MSG("get_stringlist, opening %s\n", filename);
		fd = fopen(filename, "r");
		if (fd == NULL) {
			return NULL;
		}
		while (fgets(tmpbuf, STRING_MAX_SIZE, fd) != NULL) {
			tmpbuf = trunc_on_char(tmpbuf, '\n');
			tempstr = g_strdup(tmpbuf);
			DEBUG_MSG("get_stringlist, adding string \"%s\" to the list=%p\n", tempstr, which_list);
			which_list = g_list_append(which_list, tempstr);
		}
		fclose(fd);
	}
	g_free(tmpbuf);
	return which_list;
}

gint put_stringlist(gchar * filename, GList * which_list)
{
	char *tmpstr;
	FILE *fd;
	GList *tmplist;

	DEBUG_MSG("put_stringlist, started with filename=%s\n", filename);

	DEBUG_MSG("put_stringlist, opening %s for saving list(%p)\n", filename, which_list);
	fd = fopen(filename, "w");
	if (fd == NULL) {
		return 0;
	}

	tmplist = g_list_first(which_list);
	while (tmplist != NULL) {
		tmpstr = g_strndup((char *) tmplist->data, STRING_MAX_SIZE - 1);
		DEBUG_MSG("put_stringlist, tmplist(%p), adding string(%p)=%s (strlen=%d)the file\n", tmplist, tmpstr, tmpstr, strlen(tmpstr));
		fputs(tmpstr, fd);
		g_free(tmpstr);
		fputs("\n", fd);
		tmplist = g_list_next(tmplist);
	}
	fclose(fd);
	DEBUG_MSG("put_stringlist, finished, filedescriptor closed\n");
	return 1;
}


/* designed for adding strings to colorlist, urllist, fontlist and targetlist */
GList *add_to_stringlist(GList * which_list, gchar * string)
{
	gchar *tempstr2;
	gint add;
	GList *tmplist;

	tempstr2 = g_strdup(string);
	g_assert(tempstr2);
	add = 1;
	tmplist = g_list_first(which_list);
	while (tmplist) {
		if (strcmp((gchar *) tmplist->data, tempstr2) == 0) {
			add = 0;
			DEBUG_MSG("add_to_stringlist, strings are the same, don't add!!\n");
		}
		tmplist = g_list_next(tmplist);
	}
	if (add) {
		DEBUG_MSG("add_to_stringlist, adding string=%s, listlenght=%d\n", string, g_list_length(which_list));
		which_list = g_list_append(which_list, tempstr2);
	} else {
		DEBUG_MSG("add_to_stringlist, string is already in there, freeing\n");
		g_free(tempstr2);
	}
	return which_list;
}
