/* Bluefish HTML Editor
 *
 * Copyright (C) 1999-2000 Olivier Sessink
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtk/gtk.h>

#include <locale.h>
#include <libintl.h>
#include "intl.h"

#include "debug.h"
#include "bluefish.h"


/*
 * Function: setup_toggle_item
 * Arguments:
 * 	ifactory - the itemfactory where the toggle item is in defined
 * 	path - the path in the itemfactory to find the toggle item
 * 	state - the state to put it in (0 or 1)
 * Return value:
 * 	void
 * Description:
 * 	change the setting of a toggle item in a menu
 */
void setup_toggle_item(GtkItemFactory * ifactory, gchar * path, gint state)
{

	GtkWidget *toggle;

	toggle = gtk_item_factory_get_widget(ifactory, path);

	if ((GTK_CHECK_MENU_ITEM(toggle)->active) && state == 0) {
		GTK_CHECK_MENU_ITEM(toggle)->active = state;
		return;
	}
	if ((!GTK_CHECK_MENU_ITEM(toggle)->active) && state == 1) {
		GTK_CHECK_MENU_ITEM(toggle)->active = state;
		return;
	}
}

/*
 * Function: string_apply
 * Arguments:
 * 	config_var - a pointer to a gchar pointer with where it should be placed
 * 	entry - the entry to read
 * Return value:
 * 	void
 * Description:
 * 	free the old string, and replace it with the new string
 */
void string_apply(gchar ** config_var, GtkWidget * entry)
{
	gchar *tmpstring;

	tmpstring = gtk_entry_get_text(GTK_ENTRY(entry));
	DEBUG_MSG("string_apply, tmpstring=%s\n", tmpstring);
	if (config_var != NULL) {
		g_free(*config_var);
	}
	*config_var = g_strdup(tmpstring);
}

/*
 * Function: combo_with_popdown
 * Arguments:
 * 	setstring - string to setb in textbox, if NULL set ""
 * 	which_list - GList to set in popdown widget
 * 	editable - if editable (1 or 0)
 * Return value:
 * 	Pointer to create combo
 * Description:
 * 	Create new combo and preset some values
 */
GtkWidget *combo_with_popdown(gchar * setstring, GList * which_list, gint editable)
{

	GtkWidget *returnwidget;

	returnwidget = gtk_combo_new();
	if (which_list != NULL) {
		gtk_combo_set_popdown_strings(GTK_COMBO(returnwidget), which_list);
	}
	if (setstring != NULL) {
		gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(returnwidget)->entry), setstring);
	} else {
		gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(returnwidget)->entry), "");
	}
	if (editable == 0) {
		gtk_editable_set_editable(GTK_EDITABLE(GTK_ENTRY(GTK_COMBO(returnwidget)->entry)), FALSE);
	}
	return returnwidget;
}

/*
 * Function: entry_with_text
 * Arguments:
 * 	setstring - if !NULL set this text
 * 	max_lenght - max. characters in the entry
 * Return value:
 * 	Pointer to the new entry widget
 * Description:
 * 	Create new entry with some preset values
 */

GtkWidget *entry_with_text(gchar * setstring, gint max_lenght)
{

	GtkWidget *returnwidget;

	if (max_lenght) {
		returnwidget = gtk_entry_new_with_max_length(max_lenght);
	} else {
		returnwidget = gtk_entry_new();
	}
	if (setstring) {
		gtk_entry_set_text(GTK_ENTRY(returnwidget), setstring);
	}
	return returnwidget;

}

/*
 * Function: window_with_title
 * Arguments:
 * 	title - the title string
 * 	position - the gtk window position
 * 	type - the gtk window type
 * Return value:
 * 	Pointer to created windpw
 * Description:
 * 	Create new window with title and some more settings
 */
GtkWidget *window_with_title(gchar * title, GtkWindowPosition position, GtkWindowType type, gint borderwidth)
{

	GtkWidget *returnwidget;

	returnwidget = gtk_window_new(type);
	if (title) {
		gtk_window_set_title(GTK_WINDOW(returnwidget), title);
	}
	if (borderwidth) {
		gtk_container_set_border_width(GTK_CONTAINER(returnwidget), borderwidth);
	}
	if (position) {
		gtk_window_set_position(GTK_WINDOW(returnwidget), position);
	}
	return returnwidget;
}

/*
 * Function: apply_font_style
 * Arguments:
 * 	this_widget - the widget to apply the new fontstyle
 * 	fontstring - the fontset which will be loaded
 * Return value:
 * 	this_widget
 * Description:
 * 	Change the font of a widget with a certain font in a string
 */

GtkWidget *apply_font_style(GtkWidget * this_widget, gchar * fontstring)
{
	GdkFont *tmpfont;
	GtkStyle *style;

	style = gtk_style_new();
	tmpfont = style->font;

	if (main_v->fontset) {
		style->font = gdk_fontset_load(fontstring);
	} else {
		style->font = gdk_font_load(fontstring);
	}
	if (style->font) {
		gdk_font_unref(tmpfont);
		gtk_widget_push_style(style);
		gtk_widget_set_style(GTK_WIDGET(this_widget), style);
		gtk_widget_pop_style();
		DEBUG_MSG("apply_font_style, style applied\n");
	} else {
		DEBUG_MSG("apply_font_style, font = NULL, fontstring=%s\n", fontstring);
		style->font = tmpfont;
		gtk_style_unref(style);
	}
	return this_widget;
}

/*
 * Function: bf_stock_button
 * Arguments:
 * 	Text - label string
 * 	func - pointer to signal handler
 * 	func_data - data for signal handler
 * Return value:
 * 	Pointer to create button
 * Description:
 * 	Create new button and link button with "clicked" signal handler
 */
GtkWidget *bf_stock_button(const gchar * Text, GtkSignalFunc func, gpointer func_data)
{
	GtkWidget *button;

	button = gtk_button_new_with_label(Text);
	g_return_val_if_fail(button, NULL);
	gtk_signal_connect(GTK_OBJECT(button), "clicked", func, func_data);
	DEBUG_MSG("bf_stock_button, func_data=%p\n", func_data);
	return button;
};

/*
 * Function: bf_stock_ok_button
 * Arguments:
 * 	func - pointer to signal handler
 * 	func_data - data for signal handler
 * Return value:
 * 	Pointer to create button
 * Description:
 * 	Create new "Ok" button
 */
GtkWidget *bf_stock_ok_button(GtkSignalFunc func, gpointer func_data)
{
	return bf_stock_button(_(" Ok "), func, func_data);
};

/*
 * Function: bf_stock_cancel_button
 * Arguments:
 * 	func - pointer to signal handler
 * 	func_data - data for signal handler
 * Return value:
 * 	Pointer to create button
 * Description:
 * 	Create new "Cancel" button
 */
GtkWidget *bf_stock_cancel_button(GtkSignalFunc func, gpointer func_data)
{
	return bf_stock_button(_(" Cancel "), func, func_data);
};
