/********************************************************************
 * Copyright (C) Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of Beesoft Differ.
 *
 * Beesoft Differ is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * Beesoft Differ is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Beesoft Differ; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 *-------------------------------------------------------------------
 * Project      : Beesoft Differ
 * File         : QBtWorkspace.cpp
 * Author       : Piotr Pszczolkowski
 * Contact      : piotr@beesoft.org
 * Creation date: 28.01.2008
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "QBtWorkspace.h"
#include "QBtPointsButton.h"
#include "QBtBrowser.h"
#include "QBtIndicator.h"
#include "QBtSeparator.h"
#include "QBtSettings.h"
#include "QBtConfig.h"
#include "QBtShared.h"
#include "QBtDiffProcess.h"
#include "QBtEventsController.h"
#include "QBtOperator.h"
#include "QBtSaveQuestion.h"
#include "QBtLineData.h"
#include <QApplication>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QComboBox>
#include <QFileDialog>
#include <QScrollBar>
#include <QFile>
#include <QMessageBox>
#include <QTextDocumentFragment>
#include <QtDebug>

/*------- local constants:
-------------------------------------------------------------------*/
const char* const QBtWorkspace::NOT_FILE    = QT_TR_NOOP( "The file: %1,\nis not a regular file." );
const char* const QBtWorkspace::NO_SELECTED = QT_TR_NOOP( "Not selected" );
const char* const QBtWorkspace::FILES_EQUAL = QT_TR_NOOP( "Both files are the same." );


//*******************************************************************
// QBtWorkspace                                          CONSTRUCTOR
//*******************************************************************
QBtWorkspace::QBtWorkspace( QWidget* const in_parent ) : QWidget( in_parent )
, lft_btn_       ( new QBtPointsButton )
, rgt_btn_       ( new QBtPointsButton )
, lft_fpath_     ( new QComboBox )
, rgt_fpath_     ( new QComboBox )
, lft_sbar_      ( new QScrollBar )
, rgt_sbar_      ( new QScrollBar )
, lft_browser_   ( new QBtBrowser  ( true , lft_sbar_ ) )
, rgt_browser_   ( new QBtBrowser  ( false, rgt_sbar_ ) )
, lft_indicator_ ( new QBtIndicator( lft_browser_ ) )
, rgt_indicator_ ( new QBtIndicator( rgt_browser_ ) )
, lft_operator_  ( new QBtOperator ( lft_browser_, rgt_browser_ ) )
, rgt_operator_  ( new QBtOperator ( rgt_browser_, lft_browser_ ) )
, separator_     ( new QBtSeparator( lft_browser_, rgt_browser_ ) )
, lft_dir_       ( QDir::homePath() )
, rgt_dir_       ( lft_dir_ )
, can_scroll_    ( true )
{
   QBtDiffProcess::instance();
   QBtDiffProcess::instance()->setParent( this );

   lft_fpath_->setDuplicatesEnabled( false );
   rgt_fpath_->setDuplicatesEnabled( false );

   // Lewy browser ze scrollbarem.
   QHBoxLayout* const lft_layout = new QHBoxLayout;
   lft_layout->setSpacing( 0 );
   lft_layout->setMargin( 0 );
   lft_layout->addWidget( lft_sbar_ );
   lft_layout->addWidget( lft_browser_ );
   lft_layout->addWidget( lft_operator_ );
   QFrame* const lft_container = new QFrame;
   lft_container->setFrameShadow( QFrame::Sunken );
   lft_container->setFrameShape( QFrame::StyledPanel );
   lft_container->setLineWidth( 1 );
   lft_container->setLayout( lft_layout );

   // Prawy browser ze scrollbarem.
   QHBoxLayout* const rgt_layout = new QHBoxLayout;
   rgt_layout->setSpacing( 0 );
   rgt_layout->setMargin( 0 );
   rgt_layout->addWidget( rgt_operator_ );
   rgt_layout->addWidget( rgt_browser_ );
   rgt_layout->addWidget( rgt_sbar_ );
   QFrame* const rgt_container = new QFrame;
   rgt_container->setFrameShadow( QFrame::Sunken );
   rgt_container->setFrameShape( QFrame::StyledPanel );
   rgt_container->setLineWidth( 1 );
   rgt_container->setLayout( rgt_layout );

   QGridLayout* const main_layout = new QGridLayout;
   main_layout->addWidget( lft_fpath_    , 0, 1 );
   main_layout->addWidget( lft_btn_      , 0, 2 );
   main_layout->addWidget( rgt_fpath_    , 0, 4 );
   main_layout->addWidget( rgt_btn_      , 0, 5 );
   main_layout->addWidget( lft_indicator_, 1, 0 );
   main_layout->addWidget( lft_container , 1, 1, 1, 2 );
   main_layout->addWidget( separator_    , 1, 3 );
   main_layout->addWidget( rgt_container , 1, 4, 1, 2 );
   main_layout->addWidget( rgt_indicator_, 1, 6 );
   
   main_layout->setMargin( 10 );
   main_layout->setSpacing( 0 );
   setLayout( main_layout );

   restore_config();

   connect( lft_fpath_  , SIGNAL( activated ( const QString& ) ),
            this        , SLOT  ( lft_fpath_activated( const QString& ) ) );
   connect( rgt_fpath_  , SIGNAL( activated ( const QString& ) ),
            this        , SLOT  ( rgt_fpath_activated( const QString& ) ) );
   connect( lft_btn_    , SIGNAL( clicked             () ),
            this        , SLOT  ( lft_file_selection  () ) );
   connect( rgt_btn_    , SIGNAL( clicked             () ),
            this        , SLOT  ( rgt_file_selection  () ) );
   connect( lft_browser_, SIGNAL( scroll_request      ( int ) ),
            this        , SLOT  ( lft_browser_scrolled( int ) ) );
   connect( rgt_browser_, SIGNAL( scroll_request      ( int ) ),
            this        , SLOT  ( rgt_browser_scrolled( int ) ) );
   connect( lft_browser_, SIGNAL( update_request      () ),
            this        , SLOT  ( update_request      () ) );
   connect( rgt_browser_, SIGNAL( update_request      () ),
            this        , SLOT  ( update_request      () ) );

   QBtEventsController* const ec = QBtEventsController::instance();
   ec->append( this, QBtEvent::BROWSER_CFG_CHANGED );
   ec->append( this, QBtEvent::DIFF_CFG_CHANGED );
   ec->append( this, QBtEvent::REMOVE_FROM_LEFT );
   ec->append( this, QBtEvent::REMOVE_FROM_RIGHT );
   ec->append( this, QBtEvent::MOVE_FROM_LEFT );
   ec->append( this, QBtEvent::MOVE_FROM_RIGHT );
   ec->append( this, QBtEvent::CHANGE_FROM_LEFT );
   ec->append( this, QBtEvent::CHANGE_FROM_RIGHT );
}
// end of QBtWorkspace

//*******************************************************************
// ~QBtWorkspace                                          DESTRUCTOR
//*******************************************************************
QBtWorkspace::~QBtWorkspace()
{
   QBtEventsController::instance()->remove( this );
      
   remove_no_selected( lft_fpath_ );
   remove_no_selected( rgt_fpath_ );
   save_config();
}
// end of ~QBtWorkspace

//*******************************************************************
// customEvent                                     PRIVATE inherited
//*******************************************************************
void QBtWorkspace::customEvent( QEvent* const in_event )
{
   QBtEvent* const event = dynamic_cast< QBtEvent* >( in_event );
   const int type = static_cast< int >( event->type() );

   switch( type ) {
      case QBtEvent::BROWSER_CFG_CHANGED:
         update_looks();
         break;
      case QBtEvent::DIFF_CFG_CHANGED:
         update_request();
         break;
      case QBtEvent::REMOVE_FROM_LEFT:
         remove_from_left( event->data( 0 ).toInt() );
         break;
      case QBtEvent::REMOVE_FROM_RIGHT:
         remove_from_right( event->data( 0 ).toInt() );
         break;
      case QBtEvent::MOVE_FROM_LEFT:
         move_from_left( event->data( 0 ).toInt() );
         break;
      case QBtEvent::MOVE_FROM_RIGHT:
         move_from_right( event->data( 0 ).toInt() );
         break;
      case QBtEvent::CHANGE_FROM_LEFT:
         change_from_left( event->data( 0 ).toInt() );
         break;
      case QBtEvent::CHANGE_FROM_RIGHT:
         change_from_right( event->data( 0 ).toInt() );
         break;
   }
}
// end of customEvent

//*******************************************************************
// save_config                                               PRIVATE
//*******************************************************************
void QBtWorkspace::save_config() const
{
   QStringList lft_data = QStringList();
   QStringList rgt_data = QStringList();
   
   {
      const int n = lft_fpath_->count();
      for( int i = 0; i < n; ++i ) {
         lft_data << lft_fpath_->itemText( i );
      }
   }
   {
      const int n = rgt_fpath_->count();
      for( int i = 0; i < n; ++i ) {
         rgt_data << rgt_fpath_->itemText( i );
      }
   }
   
   QBtSettings stt;
   stt.save( QBtConfig::MAIN_WINDOW_GROUP + QBtConfig::LFT_FPATH_HISTORY_KEY, lft_data );
   stt.save( QBtConfig::MAIN_WINDOW_GROUP + QBtConfig::RGT_FPATH_HISTORY_KEY, rgt_data );
}
// end of save_config

//*******************************************************************
// restore_config                                            PRIVATE
//*******************************************************************
void QBtWorkspace::restore_config()
{
   QStringList lft_data = QStringList();
   QStringList rgt_data = QStringList();
   
   QBtSettings stt;
   QVariant data;
   
   if( stt.read( QBtConfig::MAIN_WINDOW_GROUP + QBtConfig::LFT_FPATH_HISTORY_KEY, data ) ) {
      lft_data = data.toStringList();
   }
   if( stt.read( QBtConfig::MAIN_WINDOW_GROUP + QBtConfig::RGT_FPATH_HISTORY_KEY, data ) ) {
      rgt_data = data.toStringList();
   }

   lft_fpath_->addItems( lft_data );
   rgt_fpath_->addItems( rgt_data );
   
   lft_fpath_->insertItem( 0, tr( NO_SELECTED ) );
   lft_fpath_->setCurrentIndex( 0 );
   rgt_fpath_->insertItem( 0, tr( NO_SELECTED ) );
   rgt_fpath_->setCurrentIndex( 0 );
   
}
// end of restore_config

//*******************************************************************
// lft_fpath_activated                                  PRIVATE slot
//-------------------------------------------------------------------
// Wybrano nowa pozycje z listy wczesniej uzywanych plikow w lewym
// combobox.
//*******************************************************************
void QBtWorkspace::lft_fpath_activated( const QString& in_fpath )
{
   if( in_fpath != tr( NO_SELECTED ) ) {
      if( in_fpath != lft_browser_->fpath() ) {
         if( lft_browser_->read_file( in_fpath ) ) {
            remove_no_selected( lft_fpath_ );
         }
         else {
            update_lft_cbox( false );
         }
      }
      update_request();
   }
}
// end of lft_fpath_activated

//*******************************************************************
// rgt_fpath_activated                                  PRIVATE slot
//-------------------------------------------------------------------
// Wybrano nowa pozycje z listy wczesniej uzywanych plikow w prawym
// combobox.
//*******************************************************************
void QBtWorkspace::rgt_fpath_activated( const QString& in_fpath )
{
   if( in_fpath != tr( NO_SELECTED ) ) {
      if( in_fpath != rgt_browser_->fpath() ) {
         if( rgt_browser_->read_file( in_fpath ) ) {
            remove_no_selected( rgt_fpath_ );
         }
         else {
            update_rgt_cbox( false );
         }
      }
      update_request();
   }
}
// end of rgt_fpath_activated

//*******************************************************************
// lft_file_selection                                   PRIVATE slot
//-------------------------------------------------------------------
// Uzytkownik nacisnal lewy kropki-przycisk.
// Czyli chce za pomoca dialogu wskazac, ktory plik chce wczytac
// do lewego okna.
//*******************************************************************
void QBtWorkspace::lft_file_selection()
{
   const QString fpath = QFileDialog::getOpenFileName( this, QString(), lft_dir_ );
   if( !fpath.isEmpty() ) {
      lft_dir_ = QFileInfo( fpath ).absolutePath();
      lft_read_file( fpath );
   }
}
// end of lft_file_selection

//*******************************************************************
// lft_read_file                                              PUBLIC
//-------------------------------------------------------------------
// Uzytkownik wybral plik w dialog.
//*******************************************************************
void QBtWorkspace::lft_read_file( const QString& in_fpath )
{
   if( !in_fpath.isEmpty() ) {
      if( !lft_browser_->read_file( in_fpath ) ) {
         update_lft_cbox( false );
      }
   }
   update_request();
}
// end of lft_read_file

//*******************************************************************
// rgt_file_selection                                   PRIVATE slot
//-------------------------------------------------------------------
// Uzytkownik nacisnal prawy kropki-przycisk.
// Czyli chce za pomoca dialogu wskazac, ktory plik chce wczytac
// do prawego okna.
//*******************************************************************
void QBtWorkspace::rgt_file_selection()
{
   const QString fpath = QFileDialog::getOpenFileName( this, QString(), rgt_dir_ );
   if( !fpath.isEmpty() ) {
      rgt_dir_ = QFileInfo( fpath ).absolutePath();
      rgt_read_file( fpath );
   }
}
// end of rgt_file_selection

//*******************************************************************
// rgt_read_file                                              PUBLIC
//-------------------------------------------------------------------
// Uzytkownik wybral plik w dialogu.
//*******************************************************************
void QBtWorkspace::rgt_read_file( const QString& in_fpath )
{
   if( !in_fpath.isEmpty() ) {
      if( !rgt_browser_->read_file( in_fpath ) ) {
         update_rgt_cbox( false );
      }
   }
   update_request();
}
// rgt_read_file

//*******************************************************************
// update_lft_cbox                                           PRIVATE
//*******************************************************************
void QBtWorkspace::update_lft_cbox( const bool in_ok )
{
   remove_no_selected( lft_fpath_ );
   const int idx = lft_fpath_->findText( lft_browser_->fpath() );
   if( idx != -1 ) lft_fpath_->removeItem( idx );
   
   if( in_ok ) {
      lft_fpath_->insertItem( 0, lft_browser_->fpath() );
      lft_fpath_->setCurrentIndex( 0 );
      check_fpath_history( lft_fpath_ );
   }
   else {
      check_fpath_history( lft_fpath_ );
      lft_fpath_->insertItem( 0, tr( NO_SELECTED ) );
      lft_fpath_->setCurrentIndex( 0 );
   }
}
// end of update_lft_cbox

//*******************************************************************
// update_rgt_cbox                                           PRIVATE
//*******************************************************************
void QBtWorkspace::update_rgt_cbox( const bool in_ok )
{
   remove_no_selected( rgt_fpath_ );
   const int idx = rgt_fpath_->findText( rgt_browser_->fpath() );
   if( idx != -1 ) rgt_fpath_->removeItem( idx );   
   
   if( in_ok ) {      
      rgt_fpath_->insertItem( 0, rgt_browser_->fpath() );
      rgt_fpath_->setCurrentIndex( 0 );
      check_fpath_history( rgt_fpath_ );
   }
   else {
      check_fpath_history( rgt_fpath_ );
      rgt_fpath_->insertItem( 0, tr( NO_SELECTED ) );
      rgt_fpath_->setCurrentIndex( 0 );
   }
}
// end of update_rgt_cbox

//*******************************************************************
// check_fpath_history                                       PRIVATE
//-------------------------------------------------------------------
// Funkcja kontroluje liczbe plikow na liscie wskazanego combobox'a.
// Nadmiarowe pliki sa usuwane (od konca).
//*******************************************************************
void QBtWorkspace::check_fpath_history( QComboBox* const inout_cbox )
{
   const int max = QBtConfig::instance()->fpath_history_size();
   int n = inout_cbox->count();
   if( n <= max ) return;
   
   while( n > max ) {
      inout_cbox->removeItem( n - 1 );
      n = inout_cbox->count();
   }
}
// end of check_fpath_history

//*******************************************************************
// save                                                       PUBLIC
//*******************************************************************
void QBtWorkspace::save()
{
   lft_browser_->save_file();
   rgt_browser_->save_file();
   QBtEventsController::instance()->send_event( QBtEvent::FILES_UNCHANGED );
}
// end of save

//*******************************************************************
// save_on_exit                                              PRIVATE
//*******************************************************************
bool QBtWorkspace::save_on_exit()
{
   const int lft = lft_browser_->modified();
   const int rgt = rgt_browser_->modified();
   
   if( lft | rgt ) {
      QBtSaveQuestion dialog( lft, rgt, this );
      const int answer = dialog.exec();
      if( answer ) {
         switch( answer ) {
            case 1:
               lft_browser_->save_file();
               break;
            case 2:
               rgt_browser_->save_file();
               break;
            case 3:
               lft_browser_->save_file();
               rgt_browser_->save_file();
               break;
         }
         QBtEventsController::instance()->send_event( QBtEvent::FILES_UNCHANGED );
         return true;
      }
      return false;
   }
   return true; 
}
// end of save_on_exit

//*******************************************************************
// remove_no_selected                                        PRIVATE
//*******************************************************************
void QBtWorkspace::remove_no_selected( QComboBox* const inout_cbox )
{
   const QString text = tr( NO_SELECTED );
   int idx = inout_cbox->findText( text );

   while( idx != -1 ) {
      inout_cbox->removeItem( idx );
      idx = inout_cbox->findText( text );
   }
}
// end of remove_no_selected

//*******************************************************************
// update_request                                            PRIVATE
//*******************************************************************
void QBtWorkspace::update_request()
{
   if( !lft_browser_->fpath().isEmpty() && !rgt_browser_->fpath().isEmpty() ) {
      QBtDiffProcess::instance()->diff( lft_browser_->fpath(), rgt_browser_->fpath() );
   }
}
// end of update_request

//*******************************************************************
// ready                                                PRIVATE slot
//*******************************************************************
void QBtWorkspace::ready()
{
   update_lft_cbox();
   update_rgt_cbox();
   update_all();
   lft_browser_->rehighlight();
   rgt_browser_->rehighlight();

   are_the_same();
}
// end of ready

//*******************************************************************
// update_looks                                              PRIVATE         
//*******************************************************************
void QBtWorkspace::update_looks()
{
   lft_browser_->update_looks();
   rgt_browser_->update_looks();
   lft_operator_->update_looks();
   rgt_operator_->update_looks();
   update_all();   
}
// end of update_looks

//*******************************************************************
// lft_browser_scrolled                                 PRIVATE slot
//*******************************************************************
void QBtWorkspace::lft_browser_scrolled( const int in_dy )
{
   if( can_scroll_ ) {
      can_scroll_ = false;
      rgt_browser_->scroll_by( in_dy );
      const int nr  = ( in_dy > 0 ) ? lft_browser_->first_selected()
                                    : lft_browser_->last_selected();
      if( nr != -1 ) {
         const int snr = QBtDiffProcess::instance()->first_to_second( nr );
         if( snr != -1 ) rgt_browser_->goto_line( snr ? snr : 1 );
      }
      can_scroll_ = true;
   }

   lft_operator_->update();
   separator_->update();
}
// end of lft_browser_scrolled

//*******************************************************************
// rgt_browser_scrolled                                 PRIVATE slot
//*******************************************************************
void QBtWorkspace::rgt_browser_scrolled( const int in_dy )
{
   if( can_scroll_ ) {
      can_scroll_ = false;
      lft_browser_->scroll_by( in_dy );
      const int nr  = ( in_dy > 0 ) ? rgt_browser_->first_selected()
                                    : rgt_browser_->last_selected();
      if( nr != -1 ) {
         const int fnr = QBtDiffProcess::instance()->second_to_first( nr );
         if( fnr != -1 ) lft_browser_->goto_line( fnr ? fnr : 1 );
      }
      can_scroll_ = true;
   }

   rgt_operator_->update();
   separator_->update();
}
// end of rgt_browser_scrolled

//*******************************************************************
// befor_automation                                          PRIVATE
//*******************************************************************
void QBtWorkspace::befor_automation()
{
   QApplication::setOverrideCursor( Qt::WaitCursor );
   lft_browser_->save_pos();
   rgt_browser_->save_pos();
}
// end of befor_automation

//*******************************************************************
// after_automation                                          PRIVATE
//*******************************************************************
void QBtWorkspace::after_automation()
{
   lft_browser_->rehighlight();
   rgt_browser_->rehighlight();
   lft_browser_->adjust_size();
   rgt_browser_->adjust_size();
   lft_browser_->restore_pos();
   rgt_browser_->restore_pos();

   update_all();
   QApplication::restoreOverrideCursor();
   are_the_same();
}
// end of after_automation

//*******************************************************************
// del_on_left                                                PUBLIC
//*******************************************************************
void QBtWorkspace::del_on_left()
{
   befor_automation();

   QTextBlock block = lft_browser_->document()->begin();
   if( block.isValid() && block.userData() ) {
      QBtDiffProcess* const dp = QBtDiffProcess::instance();
      while( block.isValid() ) {
         const QBtLineData* const info = dynamic_cast<QBtLineData*>( block.userData() );
         const int opc = info->get_status();
         if( QBtShared::DELETE == opc ) {
            QBtDiffInfo di = dp->info_for_nr_in_first( info->get_number(), true );
            if( di.is_valid() ) {
               const QTextBlock prv_block = block.previous();
               const int nr1 = di.first_range().nr1();
               const int nr2 = di.first_range().nr2();
               dp->decrement_left_after( nr1, nr2 - nr1 + 1 );
               lft_browser_->remove_range( nr1, nr2, false );
               lft_browser_->renew( false );
               block = ( prv_block.isValid() ) ? prv_block.next() : lft_browser_->document()->begin();
               continue;
            }
         }
         block = block.next();
      }
   }
   after_automation();
}
// end of del_on_left

//*******************************************************************
// del_on_right                                                PUBLIC
//*******************************************************************
void QBtWorkspace::del_on_right()
{
   befor_automation();

   QTextBlock block = rgt_browser_->document()->begin();
   if( block.isValid() && block.userData() ) {
      QBtDiffProcess* const dp = QBtDiffProcess::instance();
      while( block.isValid() ) {
         const QBtLineData* const info = dynamic_cast<QBtLineData*>( block.userData() );
         const int opc = info->get_status();
         if( QBtShared::APPEND == opc ) {
            QBtDiffInfo di = dp->info_for_nr_in_second( info->get_number(), true );
            if( di.is_valid() ) {
               const QTextBlock prv_block = block.previous();
               const int nr1 = di.second_range().nr1();
               const int nr2 = di.second_range().nr2();
               dp->decrement_right_after( nr1, nr2 - nr1 + 1 );
               rgt_browser_->remove_range( nr1, nr2, false );
               rgt_browser_->renew( false );
               block = ( prv_block.isValid() ) ? prv_block.next() : rgt_browser_->document()->begin();
               continue;
            }
         }
         block = block.next();
      }
   }
   after_automation();
}
// end of del_on_right

//*******************************************************************
// merge_to_left                                              PUBLIC
//*******************************************************************
void QBtWorkspace::merge_to_left()
{
   befor_automation();
   
   QTextBlock block = rgt_browser_->document()->begin();
   if( block.isValid() && block.userData() ) {
      QBtDiffProcess* const dp = QBtDiffProcess::instance();
      while( block.isValid() ) {
         const QBtLineData* const info = dynamic_cast<QBtLineData*>( block.userData() );
         const int opc = info->get_status();
         if( ( QBtShared::CHANGE == opc ) || ( QBtShared::APPEND == opc ) ) {
            QBtDiffInfo di = dp->info_for_nr_in_second( info->get_number(), true );
            if( di.is_valid() ) {
               const int nr1 = di.second_range().nr1();
               const int nr2 = di.second_range().nr2();
               const int nr3 = di.first_range().nr1();
               const int nr4 = di.first_range().nr2();
               const int d1  = nr2 - nr1 + 1;
               const int d2  = nr4 - nr3 + 1;
               const QTextDocumentFragment tdf = rgt_browser_->get_fragment( nr1, nr2 );
               if( QBtShared::CHANGE == opc ) {
                  dp->increment_left_after( nr3, d1 - d2 );
                  lft_browser_->replace_fragment( nr3, nr4, tdf, false );
               }
               else {
                  dp->increment_left_after( nr3, d1 );
                  lft_browser_->set_fragment( nr3, tdf, false );
               }
               lft_browser_->renew( false );
            }
         }
         block = block.next();
      }
   }

   after_automation();   
}
// end of merge_to_left

//*******************************************************************
// merge_to_rigth                                             PUBLIC
//*******************************************************************
void QBtWorkspace::merge_to_right()
{
   befor_automation();
   
   QTextBlock block = lft_browser_->document()->begin();
   if( block.isValid() && block.userData() ) {
      QBtDiffProcess* const dp  = QBtDiffProcess::instance();
      while( block.isValid() ) {
         const QBtLineData* const info = dynamic_cast<QBtLineData*>( block.userData() );
         const int opc = info->get_status();
         if( ( QBtShared::CHANGE == opc ) || ( QBtShared::DELETE == opc ) ) {
            QBtDiffInfo  di  = dp->info_for_nr_in_first( info->get_number(), true );
            if( di.is_valid() ) {
               const int nr1 = di.first_range().nr1();
               const int nr2 = di.first_range().nr2();
               const int nr3 = di.second_range().nr1();
               const int nr4 = di.second_range().nr2();
               const int d1  = nr2 - nr1 + 1;
               const int d2  = nr4 - nr3 + 1;

               const QTextDocumentFragment tdf = lft_browser_->get_fragment( nr1, nr2 );
               if( QBtShared::CHANGE == opc ) {
                  dp->increment_right_after( nr3, d1 - d2 );
                  rgt_browser_->replace_fragment( nr3, nr4, tdf, false );
               }
               else {
                  rgt_browser_->set_fragment( nr3, tdf, false );
                  dp->increment_right_after( nr3, d1 );
               }
               rgt_browser_->renew( false );
            }
         }
         block = block.next();
      }
   }

   after_automation();
}
// end of merge_to_rigth

//*******************************************************************
// remove_from_left                                          PRIVATE
//*******************************************************************
void QBtWorkspace::remove_from_left( const int in_nr, const bool in_update )
{
   QBtDiffProcess* const dp  = QBtDiffProcess::instance();
   QBtDiffInfo           di  = dp->info_for_nr_in_first( in_nr, true );
   const int             nr1 = di.first_range().nr1();
   const int             nr2 = di.first_range().nr2();
   
   dp->decrement_left_after( nr1, nr2 - nr1 + 1 );
   lft_browser_->remove_range( nr1, nr2, in_update );
   
   if( in_update ) {
      update_all();
      are_the_same();
   }
}
// end of remove_from_left

//*******************************************************************
// remove_from_right                                         PRIVATE
//*******************************************************************
void QBtWorkspace::remove_from_right( const int in_nr, const bool in_update )
{
   QBtDiffProcess* const dp  = QBtDiffProcess::instance();
   QBtDiffInfo           di  = dp->info_for_nr_in_second( in_nr, true );
   const int             nr1 = di.second_range().nr1();
   const int             nr2 = di.second_range().nr2();
   
   dp->decrement_right_after( nr1, nr2 - nr1 + 1 );
   rgt_browser_->remove_range( nr1, nr2, in_update );
   
   if( in_update ) {
      update_all();
      are_the_same();
   }
}
// end of remove_from_right

//*******************************************************************
// move_from_left                                            PRIVATE
//-------------------------------------------------------------------
// Lewy browser posiada kod, ktorego nie ma w prawym browserze.
// Kopiujemy z lewej na prawo.
//*******************************************************************
void QBtWorkspace::move_from_left( const int in_nr, const bool in_update )
{
   QBtDiffProcess* const dp  = QBtDiffProcess::instance();
   QBtDiffInfo           di  = dp->info_for_nr_in_first( in_nr, true );
   const int             nr1 = di.first_range().nr1();
   const int             nr2 = di.first_range().nr2();
   const int             nr  = di.second_range().nr1();
   
   dp->increment_right_after( nr, nr2 - nr1 + 1 );
   const QTextDocumentFragment tdf = lft_browser_->get_fragment( nr1, nr2 );
   rgt_browser_->set_fragment( nr, tdf, in_update );
   
   if( in_update ) {
      update_all();
      are_the_same();
   }
}
// end of move_from_left

//*******************************************************************
// move_from_rigth                                           PRIVATE
//*******************************************************************
void QBtWorkspace::move_from_right( const int in_nr, const bool in_update )
{
   QBtDiffProcess* const dp  = QBtDiffProcess::instance();
   QBtDiffInfo           di  = dp->info_for_nr_in_second( in_nr, true );
   const int             nr1 = di.second_range().nr1();
   const int             nr2 = di.second_range().nr2();
   const int             nr  = di.first_range().nr2();
   
   dp->increment_left_after( nr, nr2 - nr1 + 1 );
   const QTextDocumentFragment tdf = rgt_browser_->get_fragment( nr1, nr2 );
   lft_browser_->set_fragment( nr, tdf, in_update );
   
   if( in_update ) {
      update_all();
      are_the_same();
   }
}
// end of move_from_right

//*******************************************************************
// change_from_left                                          PRIVATE
//-------------------------------------------------------------------
// Blok tekstu z lewego browsera ma zastapic odpowiedni blok
// tekstu w prawym browserze.
//-------------------------------------
// in_nr : numer pierwszej linii bloku zrodlowego w lewym browserze.
//*******************************************************************
void QBtWorkspace::change_from_left( const int in_nr, const bool in_update )
{
   QBtDiffProcess* const dp  = QBtDiffProcess::instance();
   QBtDiffInfo           di  = dp->info_for_nr_in_first( in_nr, true );
   const int             nr1 = di.first_range().nr1();
   const int             nr2 = di.first_range().nr2();
   const int             nr3 = di.second_range().nr1();
   const int             nr4 = di.second_range().nr2();
   const int             d1  = nr2 - nr1 + 1;
   const int             d2  = nr4 - nr3 + 1;
   
   dp->increment_right_after( nr3, d1 - d2 );
   const QTextDocumentFragment tdf = lft_browser_->get_fragment( nr1, nr2 );
   rgt_browser_->replace_fragment( nr3, nr4, tdf, in_update );

   if( in_update ) {
      if( ( nr1 == nr2 ) && ( nr3 == nr4 ) ) {
         lft_browser_->rehighlight();
         rgt_browser_->rehighlight();
      }
      update_all();
      are_the_same();
   }
}
// end of change_from_left

//*******************************************************************
// change_from_rigth                                         PRIVATE
//-------------------------------------------------------------------
// Blok tekstu z prawego browsera ma zastapic odpowiedni blok
// tekstu w lewym browserze.
//-------------------------------------
// in_nr : numer pierwszej linii bloku zrodlowego w prawym browserze.
//*******************************************************************
void QBtWorkspace::change_from_right( const int in_nr, const bool in_update )
{
   QBtDiffProcess* const dp = QBtDiffProcess::instance();
   QBtDiffInfo           di = dp->info_for_nr_in_second( in_nr, true );
   const int             nr1 = di.second_range().nr1();
   const int             nr2 = di.second_range().nr2();
   const int             nr3 = di.first_range().nr1();
   const int             nr4 = di.first_range().nr2();
   const int             d1  = nr2 - nr1 + 1;
   const int             d2  = nr4 - nr3 + 1;
   
   dp->increment_right_after( nr3, d1 - d2 );
   const QTextDocumentFragment tdf = rgt_browser_->get_fragment( nr1, nr2 );
   lft_browser_->replace_fragment( nr3, nr4, tdf, in_update );

   if( in_update ) {
      if( ( nr1 == nr2 ) && ( nr3 == nr4 ) ) {
         rgt_browser_->rehighlight();
         lft_browser_->rehighlight();
      }
      update_all();
      are_the_same();
   }
}
// end of change_from_right

//*******************************************************************
// update_all                                                PRIVATE
//*******************************************************************
void QBtWorkspace::update_all()
{
   QBtDiffProcess* const dp = QBtDiffProcess::instance();
   
   lft_browser_->renew();
   rgt_browser_->renew();
   
   dp->update( lft_browser_, rgt_browser_ );
   
   lft_browser_->update();
   lft_operator_->update();
   lft_indicator_->update();
   
   rgt_browser_->update();
   rgt_operator_->update();
   rgt_indicator_->update();

   separator_->update();

   int nchg, nadd, ndel;
   dp->statistic( nchg, nadd, ndel );
   emit stat_total( nchg + nadd + ndel );
   emit stat_chg( nchg );
   emit stat_add( nadd );
   emit stat_del( ndel );
}
// end of update_all

//*******************************************************************
// are_the_same                                              PRIVATE
//*******************************************************************
void QBtWorkspace::are_the_same()
{
   QApplication::processEvents();
   
   int nchg, nadd, ndel;
   QBtDiffProcess::instance()->statistic( nchg, nadd, ndel );

   if( ( 0 == nchg ) && ( 0 == nadd ) && ( 0 == ndel ) ) {
      QBtEventsController::instance()->send_event( QBtEvent::CANT_MERGE );
      QMessageBox::information( this, QBtShared::program_name(), tr( FILES_EQUAL ) );
      return;
   }
   QBtEventsController::instance()->send_event( QBtEvent::READY_TO_MERGE );
   return;
}
// end of are_the_same
