/********************************************************************
 * Copyright (C) Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of Beesoft Differ.
 *
 * Beesoft Differ is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * Beesoft Differ is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Beesoft Differ; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 *-------------------------------------------------------------------
 * Project      : Beesoft Differ
 * File         : QBtBrowser.cpp
 * Author       : Piotr Pszczolkowski
 * Contact      : piotr@beesoft.org
 * Creation date: 04.02.2008
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "QBtBrowser.h"
#include "QBtLineData.h"
#include "QBtDiffProcess.h"
#include "QBtConfig.h"
#include "QBtShared.h"
#include "QBtEventsController.h"
#include "QBtOperator.h"
#include <QFile>
#include <QFileInfo>
#include <QFont>
#include <QTextStream>
#include <QApplication>
#include <QScrollBar>
#include <QTextBlock>
#include <QTextLayout>
#include <QPainter>
#include <QMessageBox>
#include <QUrl>
#include <QtDebug>

/*------- constnats:
-------------------------------------------------------------------*/
const char* const QBtBrowser::READING     = QT_TR_NOOP( "File reading" );
const char* const QBtBrowser::NOT_FILE    = QT_TR_NOOP( "This is not a file:\n%1" );
const char* const QBtBrowser::NOT_EXISTS  = QT_TR_NOOP( "This file not exists:\n%1" );
const char* const QBtBrowser::READ_ERROR  = QT_TR_NOOP( "Error on read the file:\n%1\nError: %2" );
const char* const QBtBrowser::SAVING      = QT_TR_NOOP( "File saving" );
const char* const QBtBrowser::READ_ONLY   = QT_TR_NOOP( "This file is read only:\n%1" );
const char* const QBtBrowser::SAVE_ERROR  = QT_TR_NOOP( "Error on save the file:\n%1\nError: %2" );


//*******************************************************************
// QBtBrowser                                            CONSTRUCTOR
//*******************************************************************
QBtBrowser::QBtBrowser( const int      in_is_left,
                        QScrollBar* const in_scroll,
                        QWidget* const    in_parent )
: QTextBrowser ( in_parent )
, is_left_     ( in_is_left )
, scroll_      ( in_scroll )
, syntax_      ( new QBtSyntax( document() ) )
, operator_    ( 0 )
, fpath_       ( QString() )
, drop_fpath_  ( QString() )
, is_writable_ ( true )
, saved_sbpos_ ( int() )
, is_changed_  ( int() )
{
   setAcceptDrops( true );
   
   setFrameShadow( Plain );
   setFrameShape( NoFrame );
   setLineWidth( 0 );
   
   setLineWrapMode( NoWrap );
   setAutoFillBackground( true );
   update_looks();
   
   init_scrollbar();
}
// end of QBtBrowser

//*******************************************************************
// ~QBtBrowser                                           DESTRUCTOR
//*******************************************************************
QBtBrowser::~QBtBrowser()
{}
// end of ~QBtBrowser

//*******************************************************************
// update_looks                                              PRIVATE
//*******************************************************************
void QBtBrowser::update_looks()
{
   QPalette p = viewport()->palette();
   p.setColor( viewport()->backgroundRole(), QBtConfig::instance()->bkg_color() );
   viewport()->setPalette( p );
   syntax_->renew();
   document()->adjustSize();
   renew();
}
// end of update_looks

//*******************************************************************
// read_file                                                  PUBLIC
//*******************************************************************
bool QBtBrowser::read_file( const QString& in_fpath, const bool in_is_drag )
{
   const QFileInfo fi( in_fpath );
   const QString fpath = fi.absoluteFilePath();

   if( !fi.exists() ) {
      QMessageBox::critical( this, tr( READING ), tr( NOT_EXISTS ).arg( fpath ) );
      return false;
   }
   if( !fi.isFile() ) {
      QMessageBox::critical( this, tr( READING ), tr( NOT_FILE ).arg( fpath ) );
      return false;
   }

   if( !fi.isWritable() ) {
      QMessageBox::information( this, tr( SAVING ), tr( READ_ONLY ).arg( fpath ) );
   }
   QFile file( fpath );
   if( file.open( QIODevice::ReadOnly ) ) {
      clear();
      QApplication::setOverrideCursor( Qt::WaitCursor );
      const QByteArray data = file.readAll();
      setPlainText( QString::fromLocal8Bit( data ) );
      QApplication::restoreOverrideCursor();
      
      file.close();
      fpath_ = fpath;
      is_writable_ = fi.isWritable();
      document()->adjustSize();
   }
   else {
      QMessageBox::critical( this, tr( READING ),
         tr( READ_ERROR ).arg( fpath ).arg( file.errorString() ) );
      return false;
   }

   if( in_is_drag ) {
      emit update_request();
   }
   return true;
}
// end of read_file

//*******************************************************************
// save_file                                                  PUBLIC
//*******************************************************************
void QBtBrowser::save_file()
{
   if( fpath_.isEmpty() ) return;
   if( !is_changed_ ) return;
   
   const QFileInfo fi( fpath_ );
   if( !fi.isWritable() ) {
      return;
   }

   // Tworzymy kopie zapasowa starego pliku.
   const QString tmp_fpath = fpath_ + ".bak";
   QFile::remove( tmp_fpath );
   QFile::rename( fpath_, tmp_fpath );

   // Zapisujemy plik na dysk.
   QFile file( fpath_ );
   if( file.open( QIODevice::Truncate | QIODevice::WriteOnly ) ) {
      QApplication::setOverrideCursor( Qt::WaitCursor );
      file.write( toPlainText().toLocal8Bit() );
      QApplication::restoreOverrideCursor();
      file.close();
      is_changed_ = 0;
   }
   else {
      QMessageBox::critical( this, tr( SAVING ),
         tr( SAVE_ERROR ).arg( fpath_ ).arg( file.errorString() ) );
   }
}
// end of save_file

//*******************************************************************
// renew                                                      PUBLIC
//*******************************************************************
void QBtBrowser::renew( const bool in_repaint )
{
   QBtDiffProcess* const dp = QBtDiffProcess::instance();
   int n = 1;
   
   QTextBlock block = document()->begin();
   while( block.isValid() ) {
      const int status = is_left_ ? dp->in_first( n ) : dp->in_second( n );

      if( block.userData() ) {
         dynamic_cast<QBtLineData*>( block.userData() )->set_data( n, status, is_left_ );
      }
      else {
         block.setUserData( new QBtLineData( n, status, is_left_ ) );
      }
      
      block = block.next();
      ++n;
   }
   
   if( in_repaint ) viewport()->repaint();
}
// end of renew

//*******************************************************************
// fragment_position                                         PRIVATE
//*******************************************************************
void QBtBrowser::fragment_position( const int  in_nr1,
                                    const int  in_nr2,
                                          int& out_pos1,
                                          int& out_pos2 ) const
{
   QTextBlock block = document()->begin();
   while( block.isValid() ) {
      const QBtLineData* const info = dynamic_cast<QBtLineData*>( block.userData() );
      const int nr = info->get_number();
      if( nr < in_nr1 ) {
         block = block.next();
         continue;
      }
      if( nr == in_nr1 ) {
         out_pos1 = block.position();
      }
      if( nr == in_nr2 ) {
         out_pos2 = block.position() + block.length();
         break;
      }
      block = block.next();
   }
}
// end of fragment_position

//*******************************************************************
// remove_range                                               PUBLIC
//*******************************************************************
void QBtBrowser::remove_range( const int in_nr1,
                               const int in_nr2,
                               const bool   in_update )
{
   if( in_update ) save_pos();
   
   int pos1 = -1;
   int pos2 = -1;
   fragment_position( in_nr1, in_nr2, pos1, pos2 );

   if( ( pos1 != -1 ) && ( pos2 != -1 ) ) {
      QTextCursor cursor = textCursor();
      cursor.beginEditBlock();
      cursor.setPosition( pos1 );
      cursor.setPosition( pos2, QTextCursor::KeepAnchor );
      cursor.removeSelectedText();
      cursor.endEditBlock();
   }
   if( in_update ) {
      document()->adjustSize();
      restore_pos();
   }
   is_changed_ = 1;
}
// end of remove_range

//*******************************************************************
// get_fragment                                               PUBLIC
//*******************************************************************
QTextDocumentFragment QBtBrowser::get_fragment( const int in_nr1,
                                                const int in_nr2 )
{
   QTextDocumentFragment retval = QTextDocumentFragment();
   
   int pos1 = -1;
   int pos2 = -1;
   fragment_position( in_nr1, in_nr2, pos1, pos2 );

   if( ( pos1 != -1 ) && ( pos2 != -1 ) ) {
      QTextCursor cursor = textCursor();
      cursor.setPosition( pos1 );
      cursor.setPosition( pos2, QTextCursor::KeepAnchor );
      retval = cursor.selection();
   }
   return retval;
}
// end of get_fragment

//*******************************************************************
// set_fragment                                               PUBLIC
//*******************************************************************
void QBtBrowser::set_fragment( const int in_nr,
                               const QTextDocumentFragment& in_fragment,
                               const bool in_update )
{
   if( in_update ) save_pos();
   
   int pos1 = -1;
   int pos2 = -1;

   if( 0 == in_nr ) {
      // Jest to sytuacja specjalna.
      // Normalnie nie ma linii 0. Linia 0 to pseudo-linia.
      // Przyslanie numeru zero oznacza, ze tekst nalezy wcisnac
      // przed 1-sza linia tekstu. Na poczatek tekstu.
      pos1 = pos2 = 0;
   }
   else {
      // To jest sytuacja normalna.
      // Przyslana 'porzadny' numer linii.
      fragment_position( in_nr, in_nr, pos1, pos2 );
   }

   if( pos2 != -1 ) {
      QTextCursor cursor = textCursor();
      cursor.beginEditBlock();
      cursor.setPosition( pos2 );
      cursor.insertFragment( in_fragment );
      cursor.endEditBlock();
   }
   if( in_update ) {
      document()->adjustSize();
      restore_pos();
   }
   is_changed_ = 1;
}
// end of set_fragment

//*******************************************************************
// replace_fragment                                           PUBLIC
//*******************************************************************
void QBtBrowser::replace_fragment( const int in_nr1,
                                   const int in_nr2,
                                   const QTextDocumentFragment& in_fragment,
                                   const bool   in_update )
{
   if( in_update ) save_pos();
   
   int pos1 = -1;
   int pos2 = -1;
   fragment_position( in_nr1, in_nr2, pos1, pos2 );

   if( ( pos1 != -1 ) && ( pos2 != -1 ) ) {
      QTextCursor cursor = textCursor();
      cursor.beginEditBlock();
      cursor.setPosition( pos1 );
      cursor.setPosition( pos2, QTextCursor::KeepAnchor );
      cursor.insertFragment( in_fragment );
      cursor.endEditBlock();
   }
   if( in_update ) {
      document()->adjustSize();
      restore_pos();
   }
   is_changed_ = 1;
}
// end of replace_fragment

//*******************************************************************
// get_y                                                      PUBLIC
//*******************************************************************
QString QBtBrowser::get_y( const int in_nr1,
                           const int in_nr2,
                           int&      out_yt,
                           int&      out_yb ) const
{
   QString retval = QString();
   
   QTextBlock block = document()->begin();
   while( block.isValid() ) {
      const QBtLineData* info = dynamic_cast< QBtLineData* >( block.userData() );
      const int nr = info->get_number();
      if( nr < in_nr1 ) {
         block = block.next();
         continue;
      }
      if( nr > in_nr2 ) {
         break;
      }

      if( in_nr1 == nr ) {
         out_yt = qRound( block.layout()->position().y() );
         retval = block.text();
      }
      if( in_nr2 == nr ) {
         const int block_yt = qRound( block.layout()->position().y() );
         const int block_h  = block.layout()->boundingRect().toRect().height();
         out_yb = block_yt + block_h - 1;
         break;
      }
      block = block.next();
   }
   if( 0 == in_nr1 ) out_yt = 2;
   if( 0 == in_nr2 ) out_yb = 2;

   return retval;
}
// end of get_y

//*******************************************************************
// paintEvent                                                PRIVATE
//*******************************************************************
void QBtBrowser::paintEvent( QPaintEvent* const in_event )
{
   setTextColor( QBtConfig::instance()->font_color() );
   QPainter p( viewport() );

   const QBtConfig* const cfg = QBtConfig::instance();
   
   const int view_yt    = verticalScrollBar()->value();
   const int view_yb    = view_yt + viewport()->height() - 1;
   const int width      = viewport()->width();
   bool         is_painted = false;

   QTextBlock block = document()->begin();
   while( block.isValid() ) {
      const int block_yt = qRound( block.layout()->position().y() );
      const int block_h  = block.layout()->boundingRect().toRect().height();
      const int block_yb = block_yt + block_h - 1;

      if( !is_inside( view_yt - 1, view_yb + 1, block_yt, block_yb ) ) {
         if( is_painted ) break;
         block = block.next();
         continue;
      }

      is_painted = true;
      QBtLineData* const info = dynamic_cast<QBtLineData*>( block.userData() );
      if( info ) {
         const int block_yt = qRound( block.layout()->position().y() );
         const int block_h  = block.layout()->boundingRect().toRect().height();
         QBrush brush          = cfg->bkg_brush();

         if( info->get_status() ) {
            bool only_line = false;
            switch( info->get_status() ) {
               case QBtShared::CHANGE:
                  brush = cfg->chg_brush();
                  break;   
               case QBtShared::DELETE:
                  brush = cfg->del_brush();
                  if( !is_left_ ) only_line = true;
                  break;
               case QBtShared::APPEND:
                  brush = cfg->add_brush();
                  if( is_left_ ) only_line = true;
                  break;
            }
            if( only_line ) {
               const int yt = block_yt + block_h - 3;
               const int h = 2;
               p.fillRect( QRect( 0, yt - view_yt, width, h ), brush );
            }
            else {
               const int yt = block_yt - 1;
               const int h = block_h;
               p.fillRect( QRect( 0, yt - view_yt, width, h ), brush );
            }
         }
         else {
            const int yt = block_yt - 1;
            const int h = block_h;
            p.fillRect( QRect( 0, yt - view_yt, width, h ), brush );
         }

         if( 1 == info->get_number() ) {
            // Poniewaz kreske malujemy wzdluz gornej krawedzi linii nr. 1
            // malujemy ja tylko i wylacznie wtedy, gdy cala linia
            // jest juz widoczna.
            if( view_yt < 2 ) {
               QBtDiffProcess* const dp = QBtDiffProcess::instance();
               QBtDiffInfo di = is_left_ ? dp->info_for_nr_in_first( 0 )
                                         : dp->info_for_nr_in_second( 0 );
               if( di.is_valid() ) {
                  const QBrush brush = is_left_ ? cfg->add_brush() : cfg->del_brush();
                  p.fillRect( QRect( 0, 0, width, 2 ), brush );
               }
            }
         }
      }
      block = block.next();
   }
   p.end();
   
   QTextBrowser::paintEvent( in_event );
}
// end of renew

//*******************************************************************
// is_inside                                                 PRIVATE
//*******************************************************************
bool QBtBrowser::is_inside( const int in_view_yt,
                            const int in_view_yb,
                            const int in_block_yt,
                            const int in_block_yb ) const
{
   if( ( in_block_yt >= in_view_yt ) && ( in_block_yt <= in_view_yb ) ) return true;
   if( ( in_block_yb >= in_view_yt ) && ( in_block_yb <= in_view_yb ) ) return true;
   return false;
}
// end of in_inside

//*******************************************************************
// goto_line                                                  PUBLIC
//*******************************************************************
void QBtBrowser::goto_line( const int in_nr )
{
   QTextBlock block = document()->begin();
   while( block.isValid() ) {
      const QBtLineData* const info = dynamic_cast< QBtLineData* >( block.userData() );
      if( info ) {
         if( info->get_number() == in_nr  ) {
            set_cursor_position( block.position() );
            if( 1 == in_nr ) {
               verticalScrollBar()->setValue( 0 );
            }
            return;
         }
      }
      block = block.next();
   }
}
// end of goto_line

//*******************************************************************
// set_cursor_position                                       PRIVATE
//*******************************************************************
void QBtBrowser::set_cursor_position( const int in_position )
{
   if( in_position >= 0 ) {
      QTextCursor cursor = textCursor();
      cursor.setPosition( in_position );
      setTextCursor( cursor );
      ensureCursorVisible();
   }
}
// end of set_cursor_position

//*******************************************************************
// first_selected                                             PUBLIC
//-------------------------------------------------------------------
// Zwraca pierwsza WYBRANA linie. Wybrana tzn. taka ktora jest
// wskazana przez diff'a.
//*******************************************************************
int QBtBrowser::first_selected() const
{
   const int view_yt = verticalScrollBar()->value();
   const int view_yb = view_yt + viewport()->height() - 1;
   
   QTextBlock block = document()->begin();
   while( block.isValid() && block.userData() ) {
      const QBtLineData* const info = dynamic_cast< QBtLineData* >( block.userData() );
      const int block_yt = qRound( block.layout()->position().y() );
      const int block_h  = block.layout()->boundingRect().toRect().height();
      const int block_yb = block_yt + block_h - 1;

      if( ( block_yt > view_yt ) && (  block_yb < view_yb ) ) {
         if( info->get_status() != QBtShared::NO_OPER ) {
            return info->get_number();
         }
      }
      block = block.next();
   }
   return -1;
}
// end of first_selected

//*******************************************************************
// last_selected                                              PUBLIC
//*******************************************************************
int QBtBrowser::last_selected() const
{
   const int view_yt = verticalScrollBar()->value();
   const int view_yb = view_yt + viewport()->height() - 1;
   
   QTextBlock block = document()->end().previous();
   while( block.isValid() ) {
      const QBtLineData* const info = dynamic_cast< QBtLineData* >( block.userData() );
      if( !info ) return -1;
      
      const int block_yt = qRound( block.layout()->position().y() );
      const int block_h  = block.layout()->boundingRect().toRect().height();
      const int block_yb = block_yt + block_h - 1;
      
      if( ( block_yt > view_yt ) && ( block_yb < view_yb ) ) {
         if( info->get_status() != QBtShared::NO_OPER ) {
            return info->get_number();
         }
      }
      block = block.previous();
   }
   return -1;
}
// end of last_selected

//*******************************************************************
// first_visible                                              PUBLIC
//*******************************************************************
int QBtBrowser::first_visible() const
{
   const int view_yt = verticalScrollBar()->value();
   
   QTextBlock block = document()->begin();
   while( block.isValid() ) {
      const QBtLineData* const info = dynamic_cast< QBtLineData* >( block.userData() );
      if( !info ) return -1;

      const int block_yt = qRound( block.layout()->position().y() );
      if( block_yt >= view_yt ) {
         return info->get_number();
      }
      block = block.next();
   }
   return -1;
}
// end of first_visible

//*******************************************************************
// last_visible                                               PUBLIC
//*******************************************************************
int QBtBrowser::last_visible() const
{
   const int view_yt = verticalScrollBar()->value();
   const int view_yb = view_yt + viewport()->height() - 1;
   
   QTextBlock block = document()->end().previous();
   while( block.isValid() ) {
      const QBtLineData* const info = dynamic_cast< QBtLineData* >( block.userData() );
      if( !info ) return -1;
      
      const int block_yt = qRound( block.layout()->position().y() );
      const int block_h  = block.layout()->boundingRect().toRect().height();
      const int block_yb = block_yt + block_h - 1;
      if( block_yb <= view_yb ) {
         return info->get_number();
      }
      block = block.previous();
   }
   return -1;
}
// end of last_visible


//###################################################################
//#                                                                 #
//#                       MY SCROLL BAR                             #
//#                                                                 #
//###################################################################


//*******************************************************************
// init_scrollbar                                            PRIVATE
//*******************************************************************
void QBtBrowser::init_scrollbar()
{
   scroll_->setMinimum ( verticalScrollBar()->minimum() );
   scroll_->setMaximum ( verticalScrollBar()->maximum() );
   scroll_->setPageStep( verticalScrollBar()->pageStep() );
   scroll_->setValue   ( verticalScrollBar()->value() );
      
   connect( verticalScrollBar(), SIGNAL( rangeChanged( int, int ) ),
            this               , SLOT  ( set_range   ( int, int ) ) );
   connect( scroll_            , SIGNAL( valueChanged( int ) ),
            this               , SLOT  ( set_value   ( int ) ) );
   connect( scroll_            , SIGNAL( sliderMoved ( int ) ),
            this               , SLOT  ( set_value   ( int ) ) );
               
   setVerticalScrollBarPolicy( Qt::ScrollBarAlwaysOff );
   scroll_->setVisible( scroll_->maximum() != 0 );
}
// end of init_scrollbar

//*******************************************************************
// scrollContentsBy                                PRIVATE inherited
//-------------------------------------------------------------------
// Funkcje dziedziczymy od QTextBrowser po to, aby moc uaktualniac
// nasz scrollbar. Dotyczy to sytuacji gdy skrolowana jest
// zawartosc edytora przez uzycie kolka myszy lub klawiszy strzalek.
//*******************************************************************
void QBtBrowser::scrollContentsBy( int in_dx, int in_dy )
{
   QTextEdit::scrollContentsBy( in_dx, in_dy );
   scroll_->setValue( verticalScrollBar()->value() );
   if( in_dy ) emit scroll_request( in_dy );
}
// end of scrollContentsBy

//*******************************************************************
// scroll_by                                                  PUBLIC
//*******************************************************************
void QBtBrowser::scroll_by( const int in_dy ) const
{
   int pos = verticalScrollBar()->value() - in_dy;
   verticalScrollBar()->setValue( pos );
}
// end of scroll_by

//*******************************************************************
// set_range                                            PRIVATE slot
//*******************************************************************
void QBtBrowser::set_range( int in_min, int in_max ) const
{
   scroll_->setRange( in_min, in_max );
   scroll_->setPageStep( verticalScrollBar()->pageStep() );
   scroll_->setVisible( in_max != 0 );
}
// end of set_range

//*******************************************************************
// set_value                                            PRIVATE slot
//*******************************************************************
void QBtBrowser::set_value( const int in_value ) const
{
   verticalScrollBar()->setValue( in_value );
}
// end of set_value

//*******************************************************************
// dragEnterEvent                                  PRIVATE inherited
//*******************************************************************
void QBtBrowser::dragEnterEvent( QDragEnterEvent* const in_event )
{
   if( in_event ) {
      const QMimeData* const md = in_event->mimeData();
      if( md ) {
         if( md->hasText() ) in_event->accept();
      }
   }
}
// end of dragEnterEvent

//*******************************************************************
// dragMoveEvent                                   PRIVATE inherited
//*******************************************************************
void QBtBrowser::dragMoveEvent( QDragMoveEvent* const in_event )
{
   if( in_event ) {
      const QMimeData* const md = in_event->mimeData();
      if( md ) {
         if( md->hasText() ) in_event->accept();
      }
   }
}
// end of dragMoveEvent

//*******************************************************************
// dropEvent                                       PRIVATE inherited
//*******************************************************************
void QBtBrowser::dropEvent( QDropEvent* const in_event )
{
   const QMimeData* const mdata = in_event->mimeData();
   if( !mdata ) return;
   if( !mdata->hasText() ) return;

   const QFileInfo fi( QUrl( mdata->text().trimmed() ).toLocalFile() );
   if( fi.isFile() ) read_file( drop_fpath_ = fi.absoluteFilePath(), true );
}
// end of dropEvent
