/*  $Id: BatrachiansEngine.h,v 1.20 2006/01/26 02:47:02 sarrazip Exp $
    BatrachiansEngine.h - Main engine

    batrachians - A fly-eating frog game.
    Copyright (C) 2004 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#ifndef _H_BatrachiansEngine
#define _H_BatrachiansEngine

#include <flatzebra/GameEngine.h>
#include <flatzebra/RSprite.h>
#include <flatzebra/KeyState.h>
#include <flatzebra/SoundMixer.h>

#include <math.h>
#include <string>
#include <list>
#include <stdexcept>

using namespace std;
using namespace flatzebra;


class BatrachiansEngine : public GameEngine
/*  A hamburger-smashing video game.
*/
{
public:

    enum { FPS = 40 };  // frames per second (nominal value)

    BatrachiansEngine(const string &windowManagerCaption,
			int millisecondsPerFrame,
			bool useSound,
			bool fullScreen)
				    throw(int,
					string,
					PixmapLoadError,
					SoundMixer::Error);
    /*
	See base class.

	'windowManagerCaption' must contain the title to give to the window.

	Throws a string or an integer code if an error occurs.
    */

    virtual ~BatrachiansEngine();
    /*
	Nothing interesting.
    */

    virtual void processKey(SDLKey keysym, bool pressed);
    /*
	Inherited.
    */

    virtual bool tick();
    /*
	Inherited.
    */

private:

    class LilyPad
    {
    public:
	static int y;
	static int height;

	int x;
	int width;

	LilyPad(int xx, int ww) : x(xx), width(ww) {}
	int xMiddle() const { return x + width / 2; }
	int xRight() const { return x + width; }
	bool isPointOverPad(RCouple p) const
	{
	    return (x <= p.x && p.x < x + width);
	}
    };

    enum  // frog pixmap indexes
    {
	FROG_L_STAND,
	FROG_L_SWIM,
	FROG_R_STAND,
	FROG_R_SWIM
    };

    typedef RSpriteList::iterator Iter;
    typedef RSpriteList::const_iterator CIter;

    enum { NUM_SHADES = 7 };  // must be odd (actually, must be 7...)

    enum { WATER_HEIGHT_ABOVE_LILYPADS = 50 };

    static const double PI;

    bool firstTime;
    int tickCount;
    bool gameOver;
    time_t gameStartTime;
    long secondsPlayed;

    Uint32 blackColor;
    Uint32 redColor;
    Uint32 waterColors[NUM_SHADES];
    Uint32 skyColors[NUM_SHADES];
    size_t colorIndex;

    SDL_Surface *lilyPadSurfaces[NUM_SHADES];
    Couple lilyPadSurfacePos;
    LilyPad leftLilyPad;
    LilyPad rightLilyPad;

    #if 0
    SDL_Surface *cloudSurfaces[2];
    #endif

    PixmapArray starPA;
    RSpriteList stars;

    PixmapArray frogPA;

    RSprite *userFrog;
    long userScore;

    RSprite *computerFrog;
    long computerScore;
    int ticksBeforeNextComputerJump;

    PixmapArray splashPA;
    RSprite *userSplash;
    RSprite *computerSplash;

    PixmapArray tonguePA;
    RSprite *userTongue;
    int userTongueTicksLeft;
    RSprite *computerTongue;
    int computerTongueTicksLeft;

    PixmapArray crosshairsPA;
    RSprite *crosshairs;

    PixmapArray fly0PA;
    PixmapArray fly1PA;
    PixmapArray fly2PA;
    PixmapArray fly3PA;
    RSpriteList flies;
    int ticksBeforeNextFly;

    PixmapArray userDigitPA;
    PixmapArray computerDigitPA;
    RSpriteList scoreSprites;

    KeyState startKS;
    KeyState quitKS;
    KeyState jumpKS;
    KeyState tongueKS;
    KeyState leftKS;
    KeyState rightKS;
    KeyState upKS;
    KeyState downKS;

    SoundMixer *theSoundMixer;
    bool useSound;
    struct Sound
    {
	SoundMixer::Chunk gameStarts;
	SoundMixer::Chunk frogJumps;
	SoundMixer::Chunk flyEaten;
	SoundMixer::Chunk tongueOut;
	SoundMixer::Chunk splash;
	SoundMixer::Chunk gameEnds;

	Sound() {}
    } sounds;

    const Couple fontDim;


    void setTicksBeforeNextComputerJump()
    {
	ticksBeforeNextComputerJump = rand() % (FPS * 25 / 10);
    }

    void setTicksBeforeNextFly()
    {
	ticksBeforeNextFly = 1 + 0 * FPS + rand() % (2 * FPS);
    }

    bool isFrogSwimming(RSprite &s) const
    {
	return (s.currentPixmapIndex % 4 == FROG_L_SWIM
		|| s.currentPixmapIndex % 4 == FROG_R_SWIM);
    }

    void rectangle(int x, int y, int width, int height, Uint32 color)
    {
	SDL_Rect rect = { Sint16(x), Sint16(y), Sint16(width), Sint16(height) };
	SDL_FillRect(theSDLScreen, &rect, color);
    }

    void rectangle(double x, double y, double width, double height,
    								Uint32 color)
    {
	rectangle(int(x), int(y), int(width), int(height), color);
    }

    void putSprite(const RSprite *s)
    {
	copySpritePixmap(*s, s->currentPixmapIndex, s->getPos());
    }

    static RCouple getCoupleFromAngle(double radius, double angle)
    {
	return RCouple(radius * cos(angle), - radius * sin(angle));
    }

    static RCouple chooseRandomSpeed(double radius)
    {
	double angle = rand() * PI / RAND_MAX;
	return RCouple(radius * cos(angle), - radius * sin(angle));
    }

    static double getAngleFromCouple(RCouple c)
    {
	return atan2(-c.y, c.x);
    }

    void putSpriteList(RSpriteList &sl)
    {
	for (Iter it = sl.begin(); it != sl.end(); it++)
	    putSprite(*it);
    }

    void addToScore(long n, bool forUser)
    {
	if (forUser)
	    userScore += n;
	else
	    computerScore += n;
    }

    void changeStarColors()
    {
	for (Iter it = stars.begin(); it != stars.end(); it++)
	    (*it)->currentPixmapIndex = rand() % 2;
    }

    int getYWater() const
    {
	return lilyPadSurfacePos.y - WATER_HEIGHT_ABOVE_LILYPADS;
    }


    void initFrogPositions();
    void initGame();
    void moveFrog(RSprite &aFrog, RSprite &itsTongue,
					RSprite *&itsSplash, bool isUser);
    void animateSplash(RSprite *&splash);
    void moveFlies();
    void setTonguePosition(const RSprite &frog,
				size_t piOffset, RSprite &tongue);
    bool draw();
    void detectCollisions();
    size_t detectTongueFlyCollisions(RSprite &tongue, bool isUser);
    void controlCrosshairs();
    void controlUserFrogJump();
    void controlUserFrogTongue();
    RSprite *findNearestFly(RCouple pos);
    void controlComputerFrogJump();
    void controlComputerFrogTongue();
    RCouple computeJumpSpeed(RCouple start, RCouple target, double g)
						throw(invalid_argument);
    void playSoundEffect(SoundMixer::Chunk &chunk);
    void createScoreSprites(long n, RCouple center, bool forUser);
    void animateTemporarySprites(RSpriteList &sl) const;
    int getFlyType(const RSprite &fly) const;

    /*	Forbidden operations:
    */
    BatrachiansEngine(const BatrachiansEngine &);
    BatrachiansEngine &operator = (const BatrachiansEngine &);
};


#endif  /* _H_BatrachiansEngine */
