/*  $Id: BatrachiansEngine.cpp,v 1.22 2006/01/26 02:47:02 sarrazip Exp $
    BatrachiansEngine.cpp - Main engine

    batrachians - A fly-eating frog game.
    Copyright (C) 2004-2006 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#include "BatrachiansEngine.h"

#include <flatzebra/PixmapLoadError.h>

#include <assert.h>
#include <iostream>
#include <sstream>
#include <algorithm>

#include <time.h>

#include "images/lilypads0.xpm"
#include "images/lilypads1.xpm"
#include "images/lilypads2.xpm"
#include "images/lilypads3.xpm"
#include "images/lilypads4.xpm"
#include "images/lilypads5.xpm"
#include "images/lilypads6.xpm"
#if 0
#include "images/cloud0.xpm"
#include "images/cloud1.xpm"
#endif
#include "images/frog0_r_stand.xpm"
#include "images/frog0_r_swim.xpm"
#include "images/frog0_l_stand.xpm"
#include "images/frog0_l_swim.xpm"
#include "images/frog1_r_stand.xpm"
#include "images/frog1_r_swim.xpm"
#include "images/frog1_l_stand.xpm"
#include "images/frog1_l_swim.xpm"
#include "images/crosshairs.xpm"
#include "images/fly0_r0.xpm"
#include "images/fly1_r0.xpm"
#include "images/fly2_r0.xpm"
#include "images/fly3_r0.xpm"
#include "images/tongue0_r2.xpm"
#include "images/splash0.xpm"
#include "images/splash1.xpm"
#include "images/star0.xpm"
#include "images/star1.xpm"

#include "images/digit_user_0.xpm"
#include "images/digit_user_1.xpm"
#include "images/digit_user_2.xpm"
#include "images/digit_user_3.xpm"
#include "images/digit_user_4.xpm"
#include "images/digit_user_5.xpm"
#include "images/digit_user_6.xpm"
#include "images/digit_user_7.xpm"
#include "images/digit_user_8.xpm"
#include "images/digit_user_9.xpm"

#include "images/digit_computer_0.xpm"
#include "images/digit_computer_1.xpm"
#include "images/digit_computer_2.xpm"
#include "images/digit_computer_3.xpm"
#include "images/digit_computer_4.xpm"
#include "images/digit_computer_5.xpm"
#include "images/digit_computer_6.xpm"
#include "images/digit_computer_7.xpm"
#include "images/digit_computer_8.xpm"
#include "images/digit_computer_9.xpm"

using namespace std;
using namespace flatzebra;


///////////////////////////////////////////////////////////////////////////////


const int PAUL = 12;


ostream &
operator << (ostream &out, const RCouple &c)
{
    return out << '(' << c.x << ", " << c.y << ')';
}

inline
string
doubleToString(double x)
{
    char temp[512];
    snprintf(temp, sizeof(temp), "%f", x);
    return temp;
}

inline
string
exprToString(const char *name, double value)
{
    return string("{") + name + " = " + doubleToString(value) + "} ";
}

inline
string
exprToString(const char *name, int value)
{
    return string("{") + name + " = " + doubleToString(value) + "} ";
}

inline
string
exprToString(const char *name, size_t value)
{
    return string("{") + name + " = " + doubleToString(value) + "} ";
}

inline
string
exprToString(const char *name, RCouple value)
{
    return string("{") + name + " = ("
		+ doubleToString(value.x)
		+ ", " + doubleToString(value.y) + ")} ";
}

inline
string
exprToString(const char *name, Couple value)
{
    return string("{") + name + " = ("
		+ doubleToString(value.x)
		+ ", " + doubleToString(value.y) + ")} ";
}

#define SHOW(expr) (exprToString(#expr, expr))


static void
removeNulls(RSpriteList &sl)
{
    RSpriteList::iterator it = remove(sl.begin(), sl.end(), (RSprite *) NULL);
    sl.erase(it, sl.end());
}

static void
deleteSprite(RSprite *p)
{
    delete p;
}

template <class Container>
static void
deleteSprites(Container &c)
{
    for_each(c.begin(), c.end(), deleteSprite);
    c.clear();
}

static
string
getDir(const char *defaultValue, const char *envVarName)
/*
    Makes sure that the returned directory name ends with a slash.
*/
{
    string dir;
    const char *s = getenv(envVarName);
    if (s != NULL)
	dir = s;
    else
	dir = defaultValue;
    
    if (!dir.empty() && dir[dir.length() - 1] != '/')
	dir += '/';

    return dir;
}


enum { SCRNWID = 800, SCRNHT = 540 };


const double BatrachiansEngine::PI = 4.0 * atan(1.0);

int BatrachiansEngine::LilyPad::y = SCRNHT - 50;
int BatrachiansEngine::LilyPad::height = 8;

static const double g = 1.4;  // vertical gravitational acceleration


BatrachiansEngine::BatrachiansEngine(const string &windowManagerCaption,
					int millisecondsPerFrame,
					bool useSound,
					bool fullScreen)
						    throw(int,
							string,
							PixmapLoadError,
							SoundMixer::Error)
  : GameEngine(Couple(SCRNWID, SCRNHT), windowManagerCaption, fullScreen),
						// may throw string exception
    firstTime(true),
    tickCount(0),
    gameOver(true),
    gameStartTime(0),
    secondsPlayed(0),

    blackColor(SDL_MapRGB(theSDLScreen->format,   0,   0,   0)),
    redColor  (SDL_MapRGB(theSDLScreen->format, 255,   0,   0)),
    waterColors(),
    skyColors(),
    colorIndex(0),

    lilyPadSurfaces(),
    leftLilyPad (SCRNWID * 14 / 100, SCRNWID * 32 / 100),
    rightLilyPad(SCRNWID * 59 / 100, SCRNWID * 27 / 100),

    #if 0
    cloudSurfaces(),
    #endif

    starPA(2),
    stars(),

    frogPA(8),

    userFrog(NULL),
    userScore(0),

    computerFrog(NULL),
    computerScore(0),
    ticksBeforeNextComputerJump(0),

    splashPA(2),
    userSplash(NULL),
    computerSplash(NULL),

    tonguePA(6),
    userTongue(NULL),
    userTongueTicksLeft(0),
    computerTongue(NULL),
    computerTongueTicksLeft(0),

    crosshairsPA(1),
    crosshairs(NULL),

    fly0PA(4),
    fly1PA(4),
    fly2PA(4),
    fly3PA(4),
    flies(),
    ticksBeforeNextFly(1),

    userDigitPA(10),
    computerDigitPA(10),
    scoreSprites(),

    startKS(SDLK_SPACE),
    quitKS(SDLK_ESCAPE),
    jumpKS(SDLK_LCTRL),
    tongueKS(SDLK_LSHIFT),
    leftKS(SDLK_LEFT),
    rightKS(SDLK_RIGHT),
    upKS(SDLK_UP),
    downKS(SDLK_DOWN),

    theSoundMixer(NULL),
    sounds(),
    fontDim(getFontDimensions())
{
    this->useSound = useSound;

    int i;
    for (i = 0; i < NUM_SHADES; i++)
    {
	int num = NUM_SHADES - 1 - i;
	int den = NUM_SHADES - 1;
	waterColors[i] = SDL_MapRGB(theSDLScreen->format,
			      0 * num / den,
			     44 * num / den,
			    255 * num / den);
	skyColors[i] = SDL_MapRGB(theSDLScreen->format,
			     33 * num / den,
			    186 * num / den,
			    255 * num / den);
    }


    /*  Lily pads:
    */
    lilyPadSurfaces[0] = IMG_ReadXPMFromArray(lilypads0_xpm);
    if (lilyPadSurfaces[0] == NULL)
	throw PixmapLoadError(PixmapLoadError::UNKNOWN, NULL);
    lilyPadSurfaces[1] = IMG_ReadXPMFromArray(lilypads1_xpm);
    if (lilyPadSurfaces[1] == NULL)
	throw PixmapLoadError(PixmapLoadError::UNKNOWN, NULL);
    lilyPadSurfaces[2] = IMG_ReadXPMFromArray(lilypads2_xpm);
    if (lilyPadSurfaces[2] == NULL)
	throw PixmapLoadError(PixmapLoadError::UNKNOWN, NULL);
    lilyPadSurfaces[3] = IMG_ReadXPMFromArray(lilypads3_xpm);
    if (lilyPadSurfaces[3] == NULL)
	throw PixmapLoadError(PixmapLoadError::UNKNOWN, NULL);
    lilyPadSurfaces[4] = IMG_ReadXPMFromArray(lilypads4_xpm);
    if (lilyPadSurfaces[4] == NULL)
	throw PixmapLoadError(PixmapLoadError::UNKNOWN, NULL);
    lilyPadSurfaces[5] = IMG_ReadXPMFromArray(lilypads5_xpm);
    if (lilyPadSurfaces[5] == NULL)
	throw PixmapLoadError(PixmapLoadError::UNKNOWN, NULL);
    lilyPadSurfaces[6] = IMG_ReadXPMFromArray(lilypads6_xpm);
    if (lilyPadSurfaces[6] == NULL)
	throw PixmapLoadError(PixmapLoadError::UNKNOWN, NULL);

    Couple size(lilyPadSurfaces[0]->w, lilyPadSurfaces[0]->h);
    lilyPadSurfacePos = Couple((SCRNWID - size.x) / 2, SCRNHT - size.y);
    int yWater = getYWater();


    #if 0
    /*	Clouds:
    */
    cloudSurfaces[0] = IMG_ReadXPMFromArray(cloud0_xpm);
    cloudSurfaces[1] = IMG_ReadXPMFromArray(cloud1_xpm);
    #endif


    /*  Stars:
    */
    loadPixmap(star0_xpm, starPA, 0);
    loadPixmap(star1_xpm, starPA, 1);
    for (i = 20; i > 0; i--)
	stars.push_back(new RSprite(starPA,
			RCouple(rand() % (SCRNWID - 200) + 100,
				rand() % (yWater - 60) + 30),
			RCouple(), RCouple(), RCouple(), RCouple()));
    changeStarColors();


    /*  User and computer frogs:
    */
    loadPixmap(frog0_l_stand_xpm, frogPA, FROG_L_STAND);
    loadPixmap(frog0_l_swim_xpm,  frogPA, FROG_L_SWIM);
    loadPixmap(frog0_r_stand_xpm, frogPA, FROG_R_STAND);
    loadPixmap(frog0_r_swim_xpm,  frogPA, FROG_R_SWIM);
    loadPixmap(frog1_l_stand_xpm, frogPA, 4 + FROG_L_STAND);
    loadPixmap(frog1_l_swim_xpm,  frogPA, 4 + FROG_L_SWIM);
    loadPixmap(frog1_r_stand_xpm, frogPA, 4 + FROG_R_STAND);
    loadPixmap(frog1_r_swim_xpm,  frogPA, 4 + FROG_R_SWIM);

    Couple frogSize = frogPA.getImageSize();

    userFrog = new RSprite(frogPA,
		    RCouple(), RCouple(), RCouple(0, +2), RCouple(), frogSize);

    computerFrog = new RSprite(frogPA,
		    RCouple(), RCouple(), RCouple(0, +2), RCouple(), frogSize);


    loadPixmap(splash0_xpm, splashPA, 0);
    loadPixmap(splash1_xpm, splashPA, 1);


    loadPixmap(tongue0_r2_xpm, tonguePA, 5);

    RCouple collBoxPos(0, 0);
    RCouple collBoxSize = tonguePA.getImageSize() + RCouple(0, 2);
    userTongue = new RSprite(tonguePA,
			    RCouple(), RCouple(), RCouple(),
			    collBoxPos, collBoxSize);

    computerTongue = new RSprite(tonguePA,
			    RCouple(), RCouple(), RCouple(),
			    collBoxPos, collBoxSize);


    loadPixmap(crosshairs_xpm, crosshairsPA, 0);

    loadPixmap(fly0_r0_xpm, fly0PA, 2);
    loadPixmap(fly1_r0_xpm, fly1PA, 2);
    loadPixmap(fly2_r0_xpm, fly2PA, 2);
    loadPixmap(fly3_r0_xpm, fly3PA, 2);

    crosshairs = new RSprite(crosshairsPA,
			RCouple(), RCouple(), RCouple(), RCouple(), RCouple());


    /*	Digits:
    */
    loadPixmap(digit_user_0_xpm, userDigitPA, 0);
    loadPixmap(digit_user_1_xpm, userDigitPA, 1);
    loadPixmap(digit_user_2_xpm, userDigitPA, 2);
    loadPixmap(digit_user_3_xpm, userDigitPA, 3);
    loadPixmap(digit_user_4_xpm, userDigitPA, 4);
    loadPixmap(digit_user_5_xpm, userDigitPA, 5);
    loadPixmap(digit_user_6_xpm, userDigitPA, 6);
    loadPixmap(digit_user_7_xpm, userDigitPA, 7);
    loadPixmap(digit_user_8_xpm, userDigitPA, 8);
    loadPixmap(digit_user_9_xpm, userDigitPA, 9);

    loadPixmap(digit_computer_0_xpm, computerDigitPA, 0);
    loadPixmap(digit_computer_1_xpm, computerDigitPA, 1);
    loadPixmap(digit_computer_2_xpm, computerDigitPA, 2);
    loadPixmap(digit_computer_3_xpm, computerDigitPA, 3);
    loadPixmap(digit_computer_4_xpm, computerDigitPA, 4);
    loadPixmap(digit_computer_5_xpm, computerDigitPA, 5);
    loadPixmap(digit_computer_6_xpm, computerDigitPA, 6);
    loadPixmap(digit_computer_7_xpm, computerDigitPA, 7);
    loadPixmap(digit_computer_8_xpm, computerDigitPA, 8);
    loadPixmap(digit_computer_9_xpm, computerDigitPA, 9);


    /*  Sound effects:
    */
    if (useSound)
    {
	try
	{
	    theSoundMixer = NULL;
	    theSoundMixer = new SoundMixer(16);  // may throw string
	}
	catch (const SoundMixer::Error &)
	{
	    return;
	}

	string d = getDir(PKGSOUNDDIR, "PKGSOUNDDIR");

	try
	{
	    sounds.gameStarts.init(d + "game-starts.wav");
	    sounds.frogJumps.init(d + "frog-jumps.wav");
	    sounds.flyEaten.init(d + "fly-eaten.wav");
	    sounds.tongueOut.init(d + "tongue-out.wav");
	    sounds.splash.init(d + "splash.wav");
	    sounds.gameEnds.init(d + "game-ends.wav");
	}
	catch (const SoundMixer::Error &e)
	{
	    throw e.what();
	}
    }


    initGame();
    gameOver = true;
}


BatrachiansEngine::~BatrachiansEngine()
{
    delete theSoundMixer;
    delete crosshairs;
    delete computerFrog;
    delete userFrog;
    for (size_t i = 0; i < NUM_SHADES; i++)
	SDL_FreeSurface(lilyPadSurfaces[i]);
}


///////////////////////////////////////////////////////////////////////////////


/*virtual*/
void
BatrachiansEngine::processKey(SDLKey keysym, bool pressed)
{
    startKS.check(keysym, pressed);
    quitKS.check(keysym, pressed);
    jumpKS.check(keysym, pressed);
    tongueKS.check(keysym, pressed);
    leftKS.check(keysym, pressed);
    rightKS.check(keysym, pressed);
    upKS.check(keysym, pressed);
    downKS.check(keysym, pressed);
}


/*virtual*/
bool
BatrachiansEngine::tick()
{
    if (quitKS.isPressed())
	return false;

    tickCount++;

    if (!gameOver)
    {
	moveFrog(*userFrog, *userTongue, userSplash, true);
	moveFrog(*computerFrog, *computerTongue, computerSplash, false);
    }

    moveFlies();
    animateTemporarySprites(scoreSprites);


    // If at night, make stars sparkle:
    if (colorIndex == NUM_SHADES - 1 && tickCount % (3 * FPS) == 1)
	changeStarColors();


    bool timeIsUp = draw();

    if (!gameOver && timeIsUp)
    {
	gameOver = true;
	initFrogPositions();
	playSoundEffect(sounds.gameEnds);
    }

    if (!gameOver)
    {
	detectCollisions();

	controlCrosshairs();
	controlUserFrogJump();
	controlUserFrogTongue();
	controlComputerFrogJump();
	controlComputerFrogTongue();
    }
    else
    {
	if (startKS.justPressed())
	{
	    firstTime = false;
	    initGame();

	    playSoundEffect(sounds.gameStarts);
	}

	startKS.remember();
    }

    return true;
}


void
BatrachiansEngine::initFrogPositions()
{
    Couple frogSize = frogPA.getImageSize();
    userFrog->setPos(RCouple(leftLilyPad.xMiddle() - frogSize.x / 2,
						LilyPad::y - frogSize.y));
    userFrog->setSpeed(RCouple());
    userFrog->setAccel(RCouple());
    userFrog->currentPixmapIndex = FROG_R_STAND;

    computerFrog->currentPixmapIndex = 4 + FROG_L_STAND;
    computerFrog->setPos(RCouple(rightLilyPad.xMiddle() - frogSize.x / 2,
						LilyPad::y - frogSize.y));
    computerFrog->setSpeed(RCouple());
    computerFrog->setAccel(RCouple());
}


void
BatrachiansEngine::initGame()
{
    gameOver = false;

    colorIndex = 0;

    initFrogPositions();

    userScore = 0;
    computerScore = 0;

    setTicksBeforeNextComputerJump();

    delete userSplash;
    userSplash = NULL;
    delete computerSplash;
    computerSplash = NULL;

    userTongue->currentPixmapIndex = 5;
    userTongueTicksLeft = 0;

    computerTongue->currentPixmapIndex = 5;
    computerTongueTicksLeft = 0;

    Couple crosshairsSize = crosshairsPA.getImageSize();
    crosshairs->setPos(RCouple(
    			userFrog->getCenterPos().x - crosshairsSize.x / 2,
			SCRNHT / 2 - crosshairsSize.y / 2));

    setTicksBeforeNextFly();

    deleteSprites(flies);

    gameStartTime = time(NULL);
    secondsPlayed = 0;
}


void
BatrachiansEngine::moveFrog(RSprite &aFrog, RSprite &itsTongue,
					RSprite *&itsSplash, bool isUser)
{
    RSprite *frog = &aFrog;

    if (frog->getSpeed().isZero())
	return;

    bool swimming = isFrogSwimming(*frog);

    if (swimming && itsSplash != NULL)
    {
	animateSplash(itsSplash);
	return;  // wait for splash animation to end before swimming to a pad
    }

    RCouple &pos = frog->getPos();
    RCouple &speed = frog->getSpeed();
    RCouple &accel = frog->getAccel();
    Couple frogSize = frog->getSize();

    frog->addAccelToSpeed();
    frog->addSpeedToPos();

    RCouple lrPos = frog->getLowerRightPos();

    const int N = 15;
    const int M = 10;

    RCouple left(pos.x + N, lrPos.y + 1);
    RCouple right(lrPos.x - 1 - N, lrPos.y + 1);
    bool onLeftPad = (leftLilyPad.isPointOverPad(left)
			|| leftLilyPad.isPointOverPad(right));
    bool onRightPad = (rightLilyPad.isPointOverPad(left)
			|| rightLilyPad.isPointOverPad(right));

    if (swimming)
	assert(speed.isNonZero());

    size_t piOffset = (isUser ? 0 : 4);

    if (!swimming && lrPos.y > LilyPad::y)
    {
	/*  The frog's bottom is now lower than the surface of the
	    lily pad.
	    If one of the bottom corners of the frog is over a lily pad,
	    then the frog stops moving and now stands still on the pad.
	    If the whole bottom of the frog is over water, then the frog
	    is allowed to go a little lower, and then it changes state
	    to the "swimming" pose.
	*/

	if (onLeftPad || onRightPad)
	{
	    // Frog is over a lily pad.
	    pos.y = LilyPad::y - frogSize.y;
	    speed.zero();
	    setTicksBeforeNextComputerJump();
	}
	else
	{
	    if (lrPos.y > LilyPad::y + LilyPad::height)
	    {
		pos.y = LilyPad::y + LilyPad::height - frogSize.y / 2 + 1;

		// Choose a horiz. direction towards the nearest pad:
		int dx = 0;
		if (lrPos.x < leftLilyPad.xRight())
		    dx = +1;
		else if (pos.x > rightLilyPad.x)
		    dx = -1;
		else
		{
		    double start = frog->getCenterPos().x;
		    double toLeft = fabs(leftLilyPad.xRight() - 1 - start);
		    double toRight = fabs(rightLilyPad.x - start);
		    dx = (toLeft <= toRight ? -1 : +1);
		}
		assert(dx != 0);

		speed = RCouple(dx * 1.5, 0);
		accel.zero();
		frog->currentPixmapIndex = piOffset + FROG_R_SWIM;
		assert(isFrogSwimming(*frog));
		swimming = true;

		// Show a splash:
		assert(itsSplash == NULL);
		itsSplash = new RSprite(splashPA,
					RCouple(frog->getPos()),
					RCouple(), RCouple(),
					RCouple(), RCouple());
		itsSplash->currentPixmapIndex = 0;
		itsSplash->values = new long[1];
		itsSplash->setTimeToLive(35);

		// Hide tongue:
		if (isUser)
		    userTongueTicksLeft = 0;
		else
		    computerTongueTicksLeft = 0;

		playSoundEffect(sounds.splash);
	    }
	}
    }
    else if (swimming)
    {
	if (onRightPad)
	{
	    if (pos.x > rightLilyPad.x)
	    {
		pos = RCouple(rightLilyPad.xRight() + M, LilyPad::y) - frogSize;
		frog->currentPixmapIndex = piOffset + FROG_L_STAND;
	    }
	    else
	    {
		pos = RCouple(rightLilyPad.x - M, LilyPad::y - frogSize.y);
		frog->currentPixmapIndex = piOffset + FROG_R_STAND;
	    }
	}
	else if (onLeftPad)
	{
	    if (pos.x < leftLilyPad.x)
	    {
		pos = RCouple(leftLilyPad.x - M, LilyPad::y - frogSize.y);
		frog->currentPixmapIndex = piOffset + FROG_R_STAND;
	    }
	    else
	    {
		pos = RCouple(leftLilyPad.xRight() + M, LilyPad::y) - frogSize;
		frog->currentPixmapIndex = piOffset + FROG_L_STAND;
	    }
	}

	if (onLeftPad || onRightPad)
	{
	    speed.zero();
	    setTicksBeforeNextComputerJump();
	}
    }

    if (!swimming)
    {
	if (pos.x < 0)
	{
	    pos.x = 0;
	    speed.x = 0;
	}
	if (lrPos.x > SCRNWID)
	{
	    pos.x = SCRNWID - frogSize.x;
	    speed.x = 0;
	}

	setTonguePosition(*frog, piOffset, itsTongue);
    }

    assert(frog->currentPixmapIndex < frogPA.getNumImages());
}


void
BatrachiansEngine::setTonguePosition(const RSprite &frog,
					size_t piOffset,
					RSprite &tongue)
{
    Couple frogSize = frog.getSize();
    Couple tongueSize = tongue.getSize();
    int yTongue = (frogSize.y - tongueSize.y) / 2;

    if (frog.currentPixmapIndex == piOffset + FROG_R_STAND)
	tongue.setPos(frog.getPos() + RCouple(frogSize.x, yTongue));
    else
	tongue.setPos(frog.getPos() + RCouple(- tongueSize.x, yTongue));
}


void
BatrachiansEngine::animateSplash(RSprite *&splash)
{
    assert(splash != NULL);
    unsigned long ticksLeft = splash->getTimeToLive();
    if (ticksLeft > 0)
    {
	if (ticksLeft % 2 == 0)
	    splash->currentPixmapIndex ^= 1;
	splash->decTimeToLive();
	return;
    }

    delete splash;
    splash = NULL;
}


bool
BatrachiansEngine::draw()
{
    // Background:
    {
	int yWater = getYWater();
	rectangle(0, 0, SCRNWID, yWater, skyColors[colorIndex]);
	rectangle(0, yWater, SCRNWID, SCRNHT - yWater, waterColors[colorIndex]);
	assert(colorIndex < NUM_SHADES);
	copyPixmap(lilyPadSurfaces[colorIndex], lilyPadSurfacePos);

	if (colorIndex == NUM_SHADES - 1)  // if at night
	    putSpriteList(stars);


	#if 0
    	copyPixmap(cloudSurfaces[0], Couple(SCRNWID * 20 / 100, 50));
    	copyPixmap(cloudSurfaces[1], Couple(SCRNWID * 60 / 100, SCRNHT / 4));
	#endif
    }


    bool timeIsUp = false;

    // Score, etc:
    {
	char s[256];
	snprintf(s, sizeof(s), " %3ld ", userScore);
	writeString(s, Couple(15, 15));
	snprintf(s, sizeof(s), " %3ld ", computerScore);
	writeStringRightJustified(s, Couple(SCRNWID - 15, 15));

	{
	    long secondsPlayedBefore = secondsPlayed;
	    if (!gameOver)
	    {
		time_t now = time(NULL);
		secondsPlayed = long(now) - long(gameStartTime) - 1;
	    }
	    if (secondsPlayed < 0)
		secondsPlayed = 0;

	    bool secondsPlayedChanged = (secondsPlayed != secondsPlayedBefore);

	    const long max = 3 * 60;
	    long secondsToPlay = max - secondsPlayed;
	    long minutes = secondsToPlay / 60;
	    long seconds = secondsToPlay % 60;
	    snprintf(s, sizeof(s), " %lu:%02lu ", minutes, seconds);
	    writeStringXCentered(s, Couple(SCRNWID / 2, 15));


	    #if 0
	    size_t numSlices = 2 * (NUM_SHADES - 1);
	    size_t sliceLength = max / numSlices;  // in seconds
	    size_t currentSliceNo = size_t(secondsPlayed) / sliceLength;
	    assert(currentSliceNo <= numSlices);
	    if (currentSliceNo < numSlices / 2)
		colorIndex = 0;
	    else if (currentSliceNo < numSlices)
		colorIndex = currentSliceNo - numSlices / 2 + 1;
	    else
		colorIndex = NUM_SHADES - 1;
	    #endif

	    assert(NUM_SHADES == 7);
	    if (secondsPlayedChanged)
		switch (secondsPlayed)
		{
		    case 100:
		    case 110:
		    case 120:
		    case 130:
		    case 140:
		    case 150:
			colorIndex++;
			break;
		}

	    //colorIndex = NUM_SHADES - 1;
	    assert(colorIndex < NUM_SHADES);

	    timeIsUp = (secondsToPlay <= 0);
	}
    }


    // Flies:
    for (Iter it = flies.begin(); it != flies.end(); it++)
    {
	if (colorIndex < NUM_SHADES - 1 || rand() % 6 == 0)
	    putSprite(*it);
    }


    // Frogs:
    assert(computerFrog->currentPixmapIndex < frogPA.getNumImages());
    putSprite(computerFrog);
    putSprite(userFrog);

    if (!gameOver)
    {
	if (computerTongueTicksLeft > 0)
	{
	    assert(computerTongue->getPos().isNonZero());
	    putSprite(computerTongue);
	    computerTongueTicksLeft--;
	}

	if (userTongueTicksLeft > 0)
	{
	    assert(userTongue->getPos().isNonZero());
	    putSprite(userTongue);
	    userTongueTicksLeft--;
	}

	// Splashes:
	if (userSplash != NULL)
	    putSprite(userSplash);
	if (computerSplash != NULL)
	    putSprite(computerSplash);


	// Misc.:
	putSprite(crosshairs);
    }


    // Scores:
    putSpriteList(scoreSprites);


    Couple center = Couple(SCRNWID, SCRNHT) / 2;
    if (gameOver)
    {
	if (!firstTime)
	    writeStringXCentered(" GAME OVER ", center + Couple(0, 50));

	//writeStringXCentered(" Press SPACE to start ", center + Couple(0, 50));

	static const char *lines[] =
	{
	    "                                ",
	    "      " PACKAGE_FULL_NAME_EN " " PACKAGE_VERSION "         ",
	    "      By Pierre Sarrazin        ",
	    "                                ",
	    " Your frog is the red one.      ",
	    " You control the crosshairs.    ",
	    " Move them with the ARROW keys. ",
	    " Jump with the LEFT CTRL key.   ",
	    " Stick out the tongue with the  ",
	    " LEFT SHIFT key.                ",
	    "                                ",
	    " Press SPACE to start a game.   ",
	    "                                ",
	    NULL
	};

	size_t numLines = 0;
	size_t i;
	for (i = 0; lines[i] != NULL; i++)
	    numLines++;

	Couple pos = center - Couple(0, fontDim.y * numLines);
	for (i = 0; lines[i] != NULL; i++)
	{
	    writeStringXCentered(lines[i], pos);
	    pos.y += fontDim.y;
	}
    }

    return timeIsUp;
}


void
BatrachiansEngine::detectCollisions()
{
    if (userTongueTicksLeft > 0)
	detectTongueFlyCollisions(*userTongue, true);

    if (computerTongueTicksLeft > 0)
	detectTongueFlyCollisions(*computerTongue, false);
}


size_t
BatrachiansEngine::detectTongueFlyCollisions(RSprite &tongue, bool isUser)
{
    size_t numFliesEaten = 0;
    for (Iter it = flies.begin(); it != flies.end(); it++)
    {
	RSprite &fly = **it;
	if (tongue.collidesWithRSprite(fly))
	{
	    numFliesEaten++;

	    size_t type = getFlyType(fly);
	    assert(type < 4);
	    int score = 0;
	    switch (type)
	    {
		case 0: score = 35; break;
		case 1: score = 15; break;
		case 2: score = 10; break;
		case 3: score =  5; break;
		default: assert(false);
	    }

	    createScoreSprites(score, fly.getCenterPos(), isUser);

	    delete *it;
	    *it = NULL;

	    playSoundEffect(sounds.flyEaten);
	}
    }

    removeNulls(flies);

    return numFliesEaten;
}


void
BatrachiansEngine::moveFlies()
{
    double radius = 4.5;

    if (flies.size() < 2 && --ticksBeforeNextFly == 0)
    {
	setTicksBeforeNextFly();

	size_t type = rand() % 4;
	PixmapArray *pa = NULL;
	switch (type)
	{
	    case 0: pa = &fly0PA; break;
	    case 1: pa = &fly1PA; break;
	    case 2: pa = &fly2PA; break;
	    case 3: pa = &fly3PA; break;
	    default: assert(false);
	}
	Couple flySize = pa->getImageSize();

	int dx = (rand() % 2 == 0 ? -1 : +1);
	RCouple pos(dx < 0 ? SCRNWID : - flySize.x,
				rand() % (SCRNHT / 2));
	double angle = (rand() * PI / 2 / RAND_MAX)
			+ (dx < 0 ? 0.75 * PI : -0.25 * PI);
	RCouple speed = getCoupleFromAngle(radius, angle);

	RSprite *fly = new RSprite(*pa, pos, speed, RCouple(),
				    RCouple(5, 5), flySize - RCouple(5, 5));
	fly->currentPixmapIndex = 2;

	flies.push_back(fly);
    }

    for (Iter it = flies.begin(); it != flies.end(); it++)
    {
	RSprite &fly = **it;
	int type = getFlyType(fly);

	if (rand() % (5 + 15 * type) == 0)  // if time to change direction
	{
	    double angle = getAngleFromCouple(fly.getSpeed());
	    double max = PI / 180 * (rand() % (3 + type) == 0 ? 360 : 4);
	    double change = rand() * max / RAND_MAX - (max / 2);
	    fly.setSpeed(getCoupleFromAngle(radius, angle + change));
	}

	fly.addSpeedToPos();
	
	RCouple &pos = fly.getPos();
	RCouple lrPos = fly.getLowerRightPos();
	RCouple &speed = fly.getSpeed();

	if (pos.x >= SCRNWID || lrPos.x <= 0 || lrPos.y < 0)
	{
	    delete *it;
	    *it = NULL;
	}
	else if (pos.y > LilyPad::y - 100)
	{
	    fly.subSpeedFromPos();
	    speed.y = - speed.y;
	    fly.addSpeedToPos();
	}
    }

    removeNulls(flies);
}


void
BatrachiansEngine::controlCrosshairs()
{
    const RCouple chPos = crosshairs->getPos();
    const RCouple chLRPos = crosshairs->getLowerRightPos();
    const Couple chSize = crosshairs->getSize();
    RCouple newCHPos = crosshairs->getPos();

    enum { CROSSHAIRS_SPEED = FPS * 3 / 4 };

    if (leftKS.isPressed())
	newCHPos.x -= CROSSHAIRS_SPEED;
    if (rightKS.isPressed())
	newCHPos.x += CROSSHAIRS_SPEED;
    if (upKS.isPressed())
	newCHPos.y -= CROSSHAIRS_SPEED;
    if (downKS.isPressed())
	newCHPos.y += CROSSHAIRS_SPEED;

    if (newCHPos.x < 0)
	newCHPos.x = 0;
    else if (newCHPos.x + chSize.x > SCRNWID)
	newCHPos.x = SCRNWID - chSize.x;

    int yWater = getYWater();
    if (newCHPos.y < 0)
	newCHPos.y = 0;
    else if (newCHPos.y + chSize.y > yWater)
	newCHPos.y = yWater - chSize.y;

    RCouple newDelta = newCHPos - userFrog->getCenterPos();
    double newDist = newDelta.length();
    double maxDist = SCRNHT * 0.75;
    if (newDist > maxDist)
    {
	newCHPos = userFrog->getCenterPos() + newDelta * maxDist / newDist;
	if (newCHPos.y + chSize.y > yWater)
	    newCHPos.y = yWater - chSize.y;
    }

    crosshairs->setPos(newCHPos);
}


void
BatrachiansEngine::controlUserFrogJump()
{
    if (jumpKS.justPressed() && userFrog->getSpeed().isZero())
    {
	try
	{
	    RCouple speed = computeJumpSpeed(
		    userFrog->getPos(),
		    crosshairs->getCenterPos() - userFrog->getSize() / 2,
		    g);
	    userFrog->setSpeed(speed);
	    userFrog->setAccel(RCouple(0, g));

	    if (speed.x < 0)
		userFrog->currentPixmapIndex = FROG_L_STAND;
	    else
		userFrog->currentPixmapIndex = FROG_R_STAND;

	    playSoundEffect(sounds.frogJumps);
	}
	catch (invalid_argument &e)
	{
	    cerr << "logic_error: " << e.what() << endl;
	    abort();
	}
    }

    jumpKS.remember();
}


void
BatrachiansEngine::controlUserFrogTongue()
{
    if (tongueKS.justPressed()
    		&& userTongueTicksLeft == 0
		&& !isFrogSwimming(*userFrog))
    {
	RCouple frogSize = userFrog->getSize();
	userTongueTicksLeft = FPS * 3 / 4;
	setTonguePosition(*userFrog, 0, *userTongue);
	playSoundEffect(sounds.tongueOut);
    }

    tongueKS.remember();
}


RSprite *
BatrachiansEngine::findNearestFly(RCouple pos)
{
    RSprite *nearestFly = NULL;
    double shortestDist = HUGE_VAL;
    for (Iter it = flies.begin(); it != flies.end(); it++)
    {
	RSprite *fly = *it;
	double dist = (fly->getPos() - pos).length();
	if (dist < shortestDist)
	{
	    shortestDist = dist;
	    nearestFly = fly;
	}
    }
    return nearestFly;
}


void
BatrachiansEngine::controlComputerFrogJump()
{
    if (computerFrog->getSpeed().isNonZero())
	return;
    if (isFrogSwimming(*computerFrog))
	return;
    if (ticksBeforeNextComputerJump > 0)
    {
	ticksBeforeNextComputerJump--;
	return;
    }

    RSprite *nearestFly = findNearestFly(computerFrog->getPos());
    if (nearestFly == NULL)
	return;
    RCouple nfPos = nearestFly->getPos();
    if (nfPos.x < 100 || nearestFly->getLowerRightPos().x > SCRNWID - 100)
	return;
    double dist = (nfPos - computerFrog->getPos()).length();
    double maxDist = SCRNHT * 0.70;
    if (dist > maxDist)
	return;

    try
    {
	RCouple target = nearestFly->getCenterPos()
					- computerFrog->getSize() / 2;

	// Introduce a clumsiness coefficient...
	target += RCouple(rand() % 50 - 25, rand() % 50 - 25);

	// Move the target in the expected direction of the fly:
	target += 10 * nearestFly->getSpeed();

	RCouple speed = computeJumpSpeed(computerFrog->getPos(), target, g);
	computerFrog->setSpeed(speed);
	computerFrog->setAccel(RCouple(0, g));

	if (speed.x < 0)
	    computerFrog->currentPixmapIndex = 4 + FROG_L_STAND;
	else
	    computerFrog->currentPixmapIndex = 4 + FROG_R_STAND;

	playSoundEffect(sounds.frogJumps);
    }
    catch (invalid_argument &e)
    {
	cerr << "logic_error: " << e.what() << endl;
	abort();
    }
}


void
BatrachiansEngine::controlComputerFrogTongue()
{
    if (computerTongueTicksLeft == 0
		&& computerFrog->getSpeed().isNonZero()
		&& !isFrogSwimming(*computerFrog))
    {
	RSprite *nearestFly = findNearestFly(computerFrog->getPos());
	if (nearestFly != NULL)
	{
	    double dist = (nearestFly->getPos()
					- computerFrog->getPos()).length();
	    if (dist <= 120)
	    {
		computerTongueTicksLeft = FPS * 3 / 4;
		setTonguePosition(*computerFrog, 4, *computerTongue);
		playSoundEffect(sounds.tongueOut);
	    }
	}
    }
}


/** Calculates the initial speed vector for a jump to a target.

    The parabola of the jump is defined by these parametric equations:

	x = vx * t
	y = vy * t - g * t^2 / 2

    where g is the vertical gravitational acceleration.  Let (a, b) be
    the peak of the jump.  Let T be the time when the peak is reached.
    Then y'(T) = 0, thus vy - g*T = 0, so T = vy / g.  By using this T
    for t in the two first equations, we get

	a = vx * vy / g
	b = vy^2 / g - g * vy^2 / g^2 / 2

    From the last equation, we get vy = sqrt(2 * g * b), and then
    we have vx = a * g / vy.

    THANK YOU PAUL GUERTIN

    @param	start		point whence the jump starts
    @param	target		highest point of the jump
    @param	g		vertical gravitational acceleration
				(positive number of pixels)
    @returns			the speed vector to use at the start
    				of the jump; the acceleration vector
				RCouple(0, g) should be used to animate
				the jump
    @throws	invalid_argument	if 'g' is non-positive or if the
    					target is not above the start
*/
RCouple
BatrachiansEngine::computeJumpSpeed(RCouple start, RCouple target, double g)
							throw(invalid_argument)
{
    if (g <= 0)
	throw invalid_argument("invalid gravitational constant");

    double a = target.x - start.x;
    double b = target.y - start.y;
    if (b >= 0)
	throw invalid_argument("target is below frog");

    double vy = sqrt(2 * g * -b) + 1;
		// +1 as patch to compensate for accumulated round off errors
    double vx = a * g / vy;
    return RCouple(vx, -vy);
}


void
BatrachiansEngine::playSoundEffect(SoundMixer::Chunk &chunk)
{
    if (theSoundMixer != NULL)
    {
	try
	{
	    theSoundMixer->playChunk(chunk);
	}
	catch (const SoundMixer::Error &e)
	{
	    fprintf(stderr, "playSoundEffect: %s (chunk at %p)\n",
			e.what().c_str(), &chunk);
	}
    }
}


void
BatrachiansEngine::createScoreSprites(long n, RCouple center, bool forUser)
{
    if (n < 0)
	n = -n;

    addToScore(n, forUser);

    char number[64];
    snprintf(number, sizeof(number), "%ld", n);
    size_t numDigits = strlen(number);

    RCouple digitSize = userDigitPA.getImageSize();
    RCouple totalSize((digitSize.x + 2) * numDigits - 2, digitSize.y);
    RCouple scorePos = center - totalSize / 2;

    PixmapArray &pa = forUser ? userDigitPA : computerDigitPA;

    for (size_t i = 0; i < numDigits; i++)
    {
        int digit = number[i] - '0';
        RSprite *s = new RSprite(pa,
			    scorePos + i * RCouple(digitSize.x + 2, 0),
			    RCouple(0, -1), RCouple(),
			    RCouple(), RCouple());
        s->setTimeToLive(FPS);
	s->currentPixmapIndex = digit;
        scoreSprites.push_back(s);
    }
}


void
BatrachiansEngine::animateTemporarySprites(RSpriteList &sl) const
/*  'slist' must be a list of sprites that die when their "time to live"
    expires.  This method removes sprites from 'slist' when they die.
    Sprites that live are advanced by adding their speed to their position.
*/
{
    for (Iter it = sl.begin(); it != sl.end(); it++)
    {
        RSprite *s = *it;
        assert(s != NULL);
        if (s->getTimeToLive() == 0)
        {
            delete s;
            *it = NULL;  // mark list element for deletion
        }
        else
        {
            s->decTimeToLive();
            s->addSpeedToPos();
        }
    }

    removeNulls(sl);
}


int
BatrachiansEngine::getFlyType(const RSprite &fly) const
{
    int type = 3;
    if (fly.getPixmapArray() == &fly0PA)
	type = 0;
    else if (fly.getPixmapArray() == &fly1PA)
	type = 1;
    else if (fly.getPixmapArray() == &fly2PA)
	type = 2;
    else if (fly.getPixmapArray() == &fly3PA)
	type = 3;
    else
	assert(false);
    return type;
}
