
// balance - a balancing tcp proxy
// $Revision: 2.24 $
//
// Copyright (c) 2000,2001 by Thomas Obermair (obermair@acm.org) 
// and Inlab Software GmbH (info@inlab.de), Gruenwald, Germany. 
// All rights reserved.
//
// Thanks to Bernhard Niederhammer for the initial idea and heavy
// testing on *big* machines ... 
//
// For license terms, see the file COPYING in this directory.
// 
// This program is dedicated to Richard Stevens... 

// Changes:
//
// 2.24:
//   'channel 2 overload' problem fixed, thanks to Ed "KuroiNeko"
//

#include <stdio.h>
#include <strings.h>
#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <malloc.h>

#ifdef __FreeBSD__
#define MYBSD 1
#endif 

#ifdef bsdi
#define MYBSD 1
#endif

#ifdef BSD
#define MYBSD 1
#endif

#ifdef MYBSD
#include <sys/wait.h>
#else
#include <sys/resource.h>
#endif

#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <sys/shm.h>

#define MAXTXSIZE 	(32*1024)
#define FILENAMELEN 	1024
#define SHMDIR 		"/tmp/"

#define MAXCHANNELS 	16	// max channels in group 
#define MAXGROUPS   	16	// max groups
#define MAXINPUTLINE 	128	// max line in input mode 
#define DEFAULTTIMEOUT  5	// default-timeout for non reachable hosts

static char* rcsid="$Id: balance.c,v 2.24 2000/12/25 12:48:37 tommy Exp $";
static char* revision="$Revision: 2.24 $";

static char rendezvousfile[FILENAMELEN];
static int  rendezvousfd;	

typedef struct {
  int status;
  int port;
  struct in_addr ipaddr;
  int c;			// current # of connections
  int maxc;			// max # of connections, 0 = no limit
  int bsent;			// bytes sent
  int breceived;		// bytes received
} CHANNEL;

typedef struct {
  int nchannels;		// number of channels in group
  int current;			// current channel in group
  CHANNEL channels[MAXCHANNELS];
} GROUP;

typedef struct {
  GROUP groups[MAXGROUPS];
} COMMON;

// Macros to access various elements of struct GROUP and struct CHANNEL 
// within COMMON array
//
// a       pointer to variable of type COMMON
// g       group index
// i       channel index

#define cmn_group(a,g)       	 ((a)->groups[(g)])
#define grp_nchannels(a,g) 	 (cmn_group((a),(g)).nchannels)
#define grp_current(a,g)    	 (cmn_group((a),(g)).current)
#define grp_channel(a,g,i)  	 (cmn_group((a),(g)).channels[(i)])
#define chn_status(a,g,i)   	 (grp_channel((a),(g),(i)).status)
#define chn_port(a,g,i)		 (grp_channel((a),(g),(i)).port)
#define chn_ipaddr(a,g,i)	 (grp_channel((a),(g),(i)).ipaddr)
#define chn_c(a,g,i)		 (grp_channel((a),(g),(i)).c)
#define chn_maxc(a,g,i)		 (grp_channel((a),(g),(i)).maxc)
#define chn_bsent(a,g,i)	 (grp_channel((a),(g),(i)).bsent)
#define chn_breceived(a,g,i)	 (grp_channel((a),(g),(i)).breceived)

static err_dump(char *text) {
  fprintf(stderr,"balance: %s\n",text);
  fflush(stderr);
  exit(1);
}

COMMON *common;

static int debugflag;
static int foreground;
static int packetdump;
static int interactive;		

static int connect_timeout;

// locking ...

a_readlock(off_t start, off_t len) {
  int rc;
  struct flock fdata;
  fdata.l_type=F_RDLCK;
  fdata.l_whence=SEEK_SET;
  fdata.l_start=0;
  fdata.l_len=0;
  // fdata.l_sysid=0;
  // fdata.l_pid=0;
repeat:
  if((rc=fcntl(rendezvousfd, F_SETLKW, &fdata)) < 0) {
    if(errno==EINTR) {
      goto repeat; // 8-)
    } else {
      perror("readlock");
      exit(2);
    }
  }
  return(rc);
}

b_readlock() { a_readlock(0,0); }

c_readlock(int group, int channel) { 
  a_readlock(((char*)&(grp_channel(common,group,channel)))-(char*)common,sizeof(CHANNEL)); 
}

a_writelock(off_t start, off_t len) {
  int rc;
  struct flock fdata;
  fdata.l_type=F_WRLCK;
  fdata.l_whence=SEEK_SET;
  fdata.l_start=0;
  fdata.l_len=0;
  // fdata.l_sysid=0;
  // fdata.l_pid=0;
repeat:
  if((rc=fcntl(rendezvousfd, F_SETLKW, &fdata)) < 0) {
    if(errno==EINTR) {
      goto repeat; // 8-)
    } else {
      perror("a_writelock");
      exit(2);
    }
  }
  return(rc);
}

b_writelock() { a_writelock(0,0); }

c_writelock(int group, int channel) { 
  a_writelock(((char*) &(grp_channel(common,group,channel)))-(char*)common,sizeof(CHANNEL)); 
}

a_unlock(off_t start, off_t len) {
  int rc;
  struct flock fdata;
  fdata.l_type=F_UNLCK;
  fdata.l_whence=SEEK_SET;
  fdata.l_start=0;
  fdata.l_len=0;
  // fdata.l_sysid=0;
  // fdata.l_pid=0;
repeat:
  if((rc=fcntl(rendezvousfd, F_SETLK, &fdata)) < 0) {
    if(errno==EINTR) {
      goto repeat; // 8-)
    } else {
      perror("a_unlock");
      exit(2);
    }
  }
  return(rc);
}

b_unlock() { a_unlock(0,0); }

c_unlock(int group, int channel) { 
  a_unlock(((char*) &(grp_channel(common,group,channel)))-(char*)common,sizeof(CHANNEL)); 
}

//

void* shm_malloc(char* file, int size) {
  key_t key;
  int shmid;
  char *data;
  int mode;

  if ((key = ftok(file, 'x')) == -1) {
    perror("ftok");
    exit(1);
  }

  if ((shmid = shmget(key, size, 0644 | IPC_CREAT)) == -1) {
    perror("shmget");
    exit(1);
  }

  data = shmat(shmid, (void*) 0, 0);
  if (data == (char *)(-1)) {
    perror("shmat");
    exit(1);
  }
  return(data);
}

// readable output of a packet (-p)

void print_packet(unsigned char*s, int l) {
  int i,cc;
  cc=0;
  for(i=0; i<l; i++) {
    if(isprint(s[i]) && isascii(s[i])) {
      if(s[i]=='\\') {
	printf("\\\\");
	cc+=2;
      } else {
        printf("%c",s[i]);
	cc++;
      }
    } else {
      printf("\\%02X",s[i]);
      cc+=3;
      if(s[i] == '\n') {
	printf("\n");
	cc=0;
      }
    }
    if(cc > 80) {
      printf("\n");
      cc=0;
    }
  }
  printf("\n");
}

//

int getport(char* port) {
  struct servent* sp;
  sp=getservbyname(port,"tcp");
  if(sp == NULL) {
    return(atoi(port));
  } else {
    return(ntohs(sp->s_port));
  }
}

int setipaddress(struct in_addr *ipaddr, char* string) {
  struct hostent *hent;
  hent = gethostbyname(string);
  if(hent == NULL) {
    if((ipaddr->s_addr = inet_addr(string))== -1) {
      fprintf(stderr,"unknown or invalid address [%s]\n",string);
      exit(1);
    }
  } else {
    memcpy(ipaddr, hent->h_addr, hent->h_length);
  }
}

int setaddress(struct in_addr *ipaddr, int *port, char* string, int default_port, int *maxc) {
  char *host_string;
  char *port_string;
  char *maxc_string;

  char *p, *q;

  struct hostent *hent;

  host_string=string;

  if((p=index(string,':')) != NULL) {
    *p='\000';
    port_string=p+1;
    if((q=index(port_string,':')) != NULL) {
      *q='\000';
      maxc_string=q+1;
    } else {
      maxc_string="";
    }
  } else {
    port_string="";
  }

  if(!strcmp(port_string,"")) 
    port_string=NULL; 
  if(!strcmp(maxc_string,"")) 
    maxc_string=NULL; 
  
  hent = gethostbyname(string);
  if(hent == NULL) {
    if((ipaddr->s_addr = inet_addr(string))== -1) {
      fprintf(stderr,"unknown or invalid address [%s]\n",string);
      exit(1);
    }
  } else {
    memcpy(ipaddr, hent->h_addr, hent->h_length);
  }

  if(port_string != NULL) {
    *port = getport(port_string);
  } else {
    *port = default_port;
  }

  if(maxc_string != NULL) {
    *maxc=atoi(maxc_string);
  }
}

int setaddress_noexitonerror(struct in_addr *ipaddr, int *port, char* string, int default_port) {
  char *host_string;
  char *port_string;
  struct hostent *hent;
  host_string=strtok(string,":");
  port_string=strtok(NULL,":");
  hent = gethostbyname(string);
  if(hent == NULL) {
    if((ipaddr->s_addr = inet_addr(string))== -1) {
      return(0);
    }
  } else {
    memcpy(ipaddr, hent->h_addr, hent->h_length);
  }

  if(port_string != NULL) {
    *port = getport(port_string);
  } else {
    *port = default_port;
  }
  return(1);
}

// readline 

int readline(int fd, char *ptr,int  maxlen) {
  int	n, rc;
  char	c;

  for (n = 1; n < maxlen; n++) {
    if ( (rc = read(fd, &c, 1)) == 1) {
      *ptr++ = c;
      if (c == '\n') {
	break;
      }
    } else if (rc == 0) {
      if (n == 1) {
	return(0);			// EOF, no data read 
      } else {
	break;				// EOF, some data was read 
      }
    } else {
      return(-1);			// error 
    }
  }
  *ptr = 0;
  return(n);
}

int writen(int fd, char* ptr, int nbytes) {
  int     nleft, nwritten;

  nleft = nbytes;

  while (nleft > 0) {
    nwritten = write(fd, ptr, nleft);
    if (nwritten <= 0) {
      return(nwritten);               	// error 
    }
    nleft -= nwritten;
    ptr   += nwritten;
  }

  return(nbytes - nleft);
}

hin(int von, int nach, int groupindex, int channelindex) {
  char c;
  int rc;

  char buffer[MAXTXSIZE];

  rc=read(von,buffer,MAXTXSIZE);

  if(packetdump) {
    printf("-> %d\n",rc);
    print_packet(buffer, rc);
  }

  if (rc <= 0) {
    return(-1); 
  } else {
    if (writen(nach, buffer, rc) != rc) {
       return(-1);
    }
    c_writelock(groupindex, channelindex);
    chn_bsent(common,groupindex,channelindex) += rc; 
    c_unlock(groupindex, channelindex);
  }
  return(0);
}

zurueck(int von, int nach, int groupindex, int channelindex) {
  char c;
  int rc;

  char buffer[MAXTXSIZE];

  rc=read(von,buffer,MAXTXSIZE);

  if(packetdump) {
    printf("-< %d\n",rc);
    print_packet(buffer, rc);
  }

  if (rc <= 0) {
    return(-1); 
  } else {
    if (writen(nach, buffer, rc) != rc) {
       return(-1);
    }
    c_writelock(groupindex, channelindex);
    chn_breceived(common,groupindex,channelindex) += rc; 
    c_unlock(groupindex, channelindex);
  }
  return(0);
}

// the connection is really established, let's transfer the data
// as efficient as possible :-) 

stream2(int clientfd, int serverfd, int groupindex, int channelindex) {
  fd_set readfds;
  int    fdset_width;
  int    sr;

  fdset_width = ((clientfd > serverfd) ? clientfd : serverfd) + 1;

  for(;;) {

    FD_ZERO(&readfds);
    FD_SET(clientfd,&readfds);
    FD_SET(serverfd,&readfds);

    for(;;) {
      sr=select(fdset_width, &readfds, NULL, NULL, NULL); 
      if (sr < 0 && errno != EINTR) {
        err_dump("select error");
      }
      if(sr > 0)
        break;
    }

    if(FD_ISSET(clientfd, &readfds)) {
      if(hin(clientfd, serverfd, groupindex, channelindex) < 0) {
	break;
      }
    } else {
      if(zurueck(serverfd, clientfd, groupindex, channelindex) < 0) {
	break;
      }
    }
  }
  c_writelock(groupindex, channelindex);
  chn_c(common,groupindex,channelindex)-= 1; 
  c_unlock(groupindex, channelindex);
  exit(1);
}

void alarm_handler() {
}

// a channel in a group is selected and we try to establish a connection 

void *stream(int arg, int groupindex, int index) {
  int			  startindex;
  int                     sockfd;
  int			  clientfd;
  struct sockaddr_in      serv_addr;
  char id[64];

  startindex=index;	  // lets keep where we start...
  clientfd= arg;

  for(;;) {
    
    if(debugflag) {
      fprintf(stderr,"trying group %d channel %d ... ", groupindex, index);
      fflush(stderr);
    }

    if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
      err_dump("can't open stream socket");
    }

    b_readlock();
      bzero((char *) &serv_addr, sizeof(serv_addr));
      serv_addr.sin_family      = AF_INET;
      serv_addr.sin_addr.s_addr = chn_ipaddr(common,groupindex,index).s_addr;
      serv_addr.sin_port        = htons(chn_port(common,groupindex,index));
    b_unlock();

    signal(SIGALRM, alarm_handler);
    alarm(connect_timeout);
  
    if (connect(sockfd, (struct sockaddr *) &serv_addr, sizeof(serv_addr)) < 0) {
      if(debugflag) {
        if(errno == EINTR) {
          fprintf(stderr, "timeout group %d channel %d\n", groupindex,  index);
        } else {
          fprintf(stderr, "connection refused group %d channel %d\n", groupindex, index);
        }
      }

      // here we've received an error (either 'timeout' or 'connection refused')
      // let's start some magical failover mechanisms 

      c_writelock(groupindex, index);
      chn_c(common,groupindex,index)--;
      c_unlock(groupindex, index);

      b_readlock();
        for(;;) {
	  for(;;) {
            index++;
	    if(index >= grp_nchannels(common,groupindex)) {
	      index=0;
	    }
	    if(index == startindex) {
	      index = -1;	// Giveup 
	      break;
	    }
            if(chn_status(common,groupindex,index) == 1 &&
             (chn_maxc(common,groupindex,index) == 0 ||
              (chn_c(common, groupindex, index) < chn_maxc(common, groupindex, index)))) {
	      break;		// new index found 
	    } else {
	      continue;
	    }
	  }

          if(index >= 0) {
            // neuer index in groupindex-group found...
            break;
          } else {
again:
            groupindex++;
            if(groupindex >= MAXGROUPS) {
              // giveup, index=-1.
              break;
            } else {
              if(grp_nchannels(common,groupindex) > 0) {
                index=grp_current(common,groupindex);
              } else {
                goto again;
              }
              break; 
            }
          }
        }
        // we drop out here with a new index

      b_unlock();

      if(index>=0) {
	// lets try it again 
	close(sockfd);
        c_writelock(groupindex, index);
        chn_c(common,groupindex,index)+= 1; 
        c_unlock(groupindex, index);
	continue;
      } else {
	break;
      }

    } else {
      if(debugflag) {
	fprintf(stderr, "connect to channel %d succesful\n", index);
      }

      // this prevents the 'channel 2 overload problem'

      b_writelock();
      grp_current(common,groupindex)=index;
      grp_current(common,groupindex)++;
      if(grp_current(common,groupindex) >= grp_nchannels(common,groupindex)) {
        grp_current(common,groupindex)=0;
      }
      b_unlock();

      // everything's fine ... 

      stream2(clientfd, sockfd, groupindex, index); // Stream 2 bekommt den Channel-Index mit
      break;
    }
  }

  close(sockfd);
  exit(0);
}


static usage() {
  char* version;
  char* revision_copy;
  char* token;

  if((revision_copy=(char*) malloc(strlen(revision)+1))==NULL) {
    fprintf(stderr,"malloc problem in usage()\n");
  } else {
    strcpy(revision_copy, revision);
    token=strtok(revision_copy," ");
    token=strtok(NULL," ");
    version=token!=NULL?token:"*undefined*";
    fprintf(stderr,"\n");

    fprintf(stderr,"balance %s - a balancing tcp proxy\n\n",version);

    fprintf(stderr,"Copyright (c) 2000,2001 by Thomas Obermair (obermair@acm.org) \n"); 
    fprintf(stderr,"and Inlab Software GmbH (info@inlab.de), Gruenwald, Germany.\n"); 
    fprintf(stderr,"All rights reserved.\n");
    fprintf(stderr,"\n");
    free(revision_copy);
  }

  fprintf(stderr,"usage:\n");
  fprintf(stderr,"  balance [-b host] [-t sec] [-dfp] \\\n");
  fprintf(stderr,"          port [h1[:p1[:maxc1]] [!] [ ... hN[:pN[:maxcN]]]]\n");
  fprintf(stderr,"  balance [-b host] [-id] port\n");
  fprintf(stderr,"\n");

  fprintf(stderr,"  -b host   bind to specific host address\n");
  fprintf(stderr,"  -d        debugging on\n");
  fprintf(stderr,"  -f        stay in foregound\n");
  fprintf(stderr,"  -i        interactive control\n");
  fprintf(stderr,"  -p        packetdump\n");
  fprintf(stderr,"  -t sec    specify connect timeout in seconds (default=%d)\n",DEFAULTTIMEOUT);
  fprintf(stderr,"   !        separates channelgroups\n");

  fprintf(stderr,"\n");
  fprintf(stderr,"example:\n");
  fprintf(stderr,"  balance smtp mailhost1:smtp mailhost2:25 mailhost3\n");
  fprintf(stderr,"  balance -i smtp\n");
  fprintf(stderr,"\n");

  exit(1);
}

// detach from controlling tty ...

hintergrund() {
  int childpid;
  if ((childpid = fork()) < 0) {
    fprintf(stderr,"cannot fork\n");
    exit(1);
  } else {
    if (childpid > 0) {
      exit(0);  /* parent */
    }
  }
#ifdef MYBSD
  setpgid(getpid(),0);
#else
  setpgrp();
  signal(SIGCHLD,SIG_IGN);
#endif
}

COMMON* makecommon(int argc, char** argv, int source_port) {
  int i;
  int group;
  int channel;
  COMMON* mycommon;

  if(argc-1 >= MAXCHANNELS) {
    fprintf(stderr,"MAXCHANNELS exceeded...\n");
    exit(2);
  }

  if((rendezvousfd=open(rendezvousfile,O_RDWR,0)) < 0) {
    perror("open");
    exit(2);
  }

  if(argc >= MAXCHANNELS) {
    fprintf(stderr,"MAXCHANNELS exceeded\n");
    exit(2);
  }

  b_writelock();

    if((mycommon=(COMMON*) shm_malloc(rendezvousfile,sizeof(COMMON))) == NULL) {
      fprintf(stderr,"cannot alloc COMMON struct\n");
      exit(1);
    }

    for(group=0; group<MAXGROUPS; group++) {
      grp_nchannels(mycommon,group)=0;
      grp_current(mycommon,group)=0;
    }

    group=0;
    channel=0;

    for(i=1; i<argc; i++) {
      if(!strcmp(argv[i],"!")) {
        group++;
        channel=0;
        if(group>=MAXGROUPS) {
          err_dump("groups exceed MAXGROUPS");
        }
        if(channel>=MAXCHANNELS) {
          err_dump("to many channels in one group");
        }
      } else {
        chn_status(mycommon,group,channel)=1;
        chn_c(mycommon,group,channel)=0;				// connections...
        chn_maxc(mycommon,group,channel)=0;				// maxconnections...
        setaddress(&chn_ipaddr(mycommon,group,channel), 
	           &chn_port(mycommon,group,channel), 
	           argv[i], 
	           source_port,
                   &chn_maxc(mycommon,group,channel)); 
        chn_bsent(mycommon,group,channel)=0;
        chn_breceived(mycommon,group,channel) = 0; 
        
        grp_nchannels(mycommon,group)+=1;
        channel++;
      }
    }

    if(debugflag) {
      fprintf(stderr,"the following channels are active:\n");
      for(group=0; group<=MAXGROUPS; group++) {
        for(i=0; i<grp_nchannels(mycommon,group); i++) {
          fprintf(stderr,"%3d %2d %s:%d:%d\n", 
                 group,
	         i, 
	         inet_ntoa(chn_ipaddr(mycommon,group,i)), 
	         chn_port(mycommon,group,i),
		 chn_maxc(mycommon,group,i));
        }
      }
    }

  b_unlock();
  return(mycommon);
}

mycmp(char* s1, char* s2) {
  int l;
  l=strlen(s1)<strlen(s2)?strlen(s1):strlen(s2);
  if(strlen(s1) > strlen(s2)) {
    return(!1);
  } else {
    return(!strncmp(s1, s2, l));
  }
}

shell() {
  int i;
  int currentgroup=0;
  char line[MAXINPUTLINE];
  char *command;

  for(;;) {
    
    printf("balance[%d] ",currentgroup);
    if (fgets(line, MAXINPUTLINE, stdin) == NULL) {
      printf("\n");
      exit(0);
    }

    if((command=strtok(line," \t\n")) != NULL) {
      if (mycmp(command,"quit")) {
        exit(0);
      } else if(mycmp(command,"show")) {
        b_readlock();
          {
            int group;

	    printf("%3s %2s %3s %16s %5s %4s %4s %10s %10s\n",
		   "GRP", "#", "S","ip-address","port","c","maxc","sent","rcvd"); 
            for (group=0; group<=MAXGROUPS; group++) {
	      for(i=0; i<grp_nchannels(common,group); i++) {
	        printf("%3d %2d %3s %16s %5d %4d %4d %10d %10d\n", 
                       group,
		       i, 
		       chn_status(common,group,i)==1?"ENA":"dis",
		       inet_ntoa(chn_ipaddr(common,group,i)), 
                       chn_port(common,group,i),
                       chn_c(common,group,i),
                       chn_maxc(common,group,i),
		       chn_bsent(common,group,i),
            	       chn_breceived(common,group,i)
		       );
	      }
            }
          }
        b_unlock();
      } else if(mycmp(command,"help") || mycmp(command,"?")){
	printf("available commands:\n");

	printf("  create <host> <port>  creates a channel in the current group\n");
	printf("  disable <channel>     disables specified channel in current group\n");
	printf("  enable <channel>      enables channel in current group\n");
	printf("  group <group>         changes current group to <group>\n");
	printf("  help                  prints this message\n");
	printf("  maxc <channel> <maxc> specifies new maxc for channel of current group\n");
	printf("  quit                  quit interactive mode\n");
	printf("  reset <channel>       reset all counters of channel in current group\n");
	printf("  show                  show all channels in all groups\n");
	printf("  version               show version id\n");

      } else if(mycmp(command,"disable")) {
        char* arg, n; 
        if((arg=strtok(NULL," \t\n")) != NULL) {
	  n=atoi(arg);
	  if(n<0 || n >= grp_nchannels(common,currentgroup)) {
	    printf("no such channel %d\n", n);
	  } else {
	    c_writelock(0, n);
	      if(chn_status(common,currentgroup,n) == 0) {
		printf("channel %d already disabled\n",n);
	      } else {
		chn_status(common,currentgroup,n) = 0;
	      }
	    c_unlock(0, n);
	  }
	} else {
	  printf("syntax error\n");
	}
      } else if(mycmp(command,"group")) {
        char* arg, n; 
        if((arg=strtok(NULL," \t\n")) != NULL) {
	  n=atoi(arg);
          if(n>=MAXGROUPS || n<0) {
            printf("value out of range\n");
          } else{
            currentgroup=n;
          }
	} else {
	  printf("syntax error\n");
	}

      } else if(mycmp(command,"reset")) {	// reset channel counters
        char* arg, n; 

        if((arg=strtok(NULL," \t\n")) != NULL) {
	  n=atoi(arg);
	  if(n<0 || n >= grp_nchannels(common,currentgroup)) {
	    printf("no such channel %d\n", n);
	  } else {
	    c_writelock(currentgroup, n);
		chn_breceived(common, currentgroup, n)=0;
		chn_bsent(common, currentgroup, n)=0;
	    c_unlock(currentgroup, n);
	  }
	} else {
	  printf("syntax error\n");
	}

      } else if(mycmp(command,"enable")) {

        char* arg, n; 
        if((arg=strtok(NULL," \t\n")) != NULL) {
	  n=atoi(arg);
	  if(n<0 || n >= grp_nchannels(common,currentgroup)) {
	    printf("no such channel %d\n", n);
	  } else {
	    c_writelock(currentgroup, n);
	      if(chn_status(common,currentgroup,n) == 1) {
		printf("channel %d already enabled\n",n);
	      } else {
		chn_status(common,currentgroup,n)=1;
	      }
	    c_unlock(currentgroup, n);
	  }
	} else {
          printf("syntax error\n");
	}

      } else if(mycmp(command,"create")) {
        char *arg1, *arg2;
        b_writelock();
        if(grp_nchannels(common,currentgroup) >= MAXCHANNELS) {
          printf("no channel slots available\n");
	} else {
          if((arg1=strtok(NULL," \t\n")) != NULL) {
            if((arg2=strtok(NULL," \t\n")) != NULL) {
              chn_status(common,currentgroup,grp_nchannels(common,currentgroup))=0;
              if(setaddress_noexitonerror(&chn_ipaddr(common,currentgroup,grp_nchannels(common,currentgroup)), 
                         &chn_port(common,currentgroup,grp_nchannels(common,currentgroup)),
	                 arg1, 
	                 getport(arg2))) {
                chn_bsent(common,currentgroup,grp_nchannels(common,currentgroup))=0;
                chn_breceived(common,currentgroup,grp_nchannels(common,currentgroup))=0;
                grp_nchannels(common,currentgroup)++;
	      } else {
		printf("invalid address\n");
	      }
            } else {
              printf("syntax error\n");
	    }
	  } else {
            printf("syntax error\n");
	  }
        }
        b_unlock();

      } else if(mycmp(command,"maxc")) {
        char *arg1, *arg2;
        b_writelock();

        if((arg1=strtok(NULL," \t\n")) != NULL) {
          if((arg2=strtok(NULL," \t\n")) != NULL) {
            if(atoi(arg1) < 0 || atoi(arg1) >= MAXCHANNELS || atoi(arg1)+1 > grp_nchannels(common,currentgroup)) { 
	      printf("unknown channel\n");
            } else {
	      chn_maxc(common,currentgroup,atoi(arg1))=atoi(arg2);;
            } 
          } else {
	    printf("syntax error\n");
          }
        } else {
          printf("syntax error\n");
        }

        b_unlock();

      } else if(mycmp(command,"version")) {
	printf("  $Id: balance.c,v 2.24 2000/12/25 12:48:37 tommy Exp $\n");
        printf("  MAXGROUPS=%d\n", MAXGROUPS);
        printf("  MAXCHANNELS=%d\n", MAXCHANNELS);
      } else {
        printf("syntax error\n");
      }
    }
  }
}

#ifdef MYBSD
void sig_chld(int signo) 
{
    int status, child_val;
    if (waitpid(-1, &status, WNOHANG) < 0) 
    {
        fprintf(stderr, "waitpid failed\n");
        return;
    }
}
#endif

char bindhost_address[FILENAMELEN];

main(argc, argv)
int	argc;
char	*argv[];
{
  int			sockfd, newsockfd, clilen, childpid;
  struct sockaddr_in	cli_addr, serv_addr;
  char*			bindhost=NULL;

#ifdef MYBSD
  struct sigaction act;
  pid_t pid;
#else
  struct rlimit 	r;
#endif
  int			c;
  int			source_port;

  connect_timeout=DEFAULTTIMEOUT;

  while((c = getopt(argc,argv,"b:t:dfpi")) != EOF) {
    switch(c) {
    case 'b':
      bindhost=optarg;
      break;
    case 't':
      connect_timeout=atoi(optarg);
      if(connect_timeout < 1) {
	usage;
      }
      break;
    case 'f':
      foreground=1;
      break;
    case 'd':
      debugflag=1;
      break;
    case 'p':
      packetdump=1;
      break;
    case 'i':
      interactive=1;
      foreground=1;
      packetdump=0;
      break;
    case '?':
    default:
      usage();
    }
  }

  if(debugflag) {
    printf("argv[0]=%s\n", argv[0]);
    printf("bindhost=%s\n",bindhost==NULL?"NULL":bindhost);
  }

  if(interactive) {
    foreground=1;
    packetdump=0;
  }

  argc -= optind;
  argv += optind;

  if(!interactive) {
    if(argc < 1) {
      usage();
    }
  } else {
    if(argc != 1) {
      usage();
    }
  }

  // really dump core if something fails... 

#ifdef MYBSD
#else
  getrlimit(RLIMIT_CORE, &r);
  r.rlim_cur=r.rlim_max;
  setrlimit(RLIMIT_CORE, &r);
#endif

  // Ermittlung des Source-Ports 

  if((source_port =  getport(argv[0])) == 0) {
    fprintf(stderr, "invalid port [%s], exiting.\n", argv[0]);
    exit(1);
  }

  if(debugflag) {
    fprintf(stderr,"source port %d\n", source_port);
  }

  // Bind our local address so that the client can send to us.
  // -b !

  bzero((char *) &serv_addr, sizeof(serv_addr));
  serv_addr.sin_family      = AF_INET;
  if(bindhost != NULL) {
    setipaddress(&serv_addr.sin_addr, bindhost);
    sprintf(bindhost_address, inet_ntoa(serv_addr.sin_addr));
  } else {
    serv_addr.sin_addr.s_addr=htonl(INADDR_ANY);
    sprintf(bindhost_address,"0.0.0.0");
  }
  serv_addr.sin_port=htons(source_port);

  // evtl. Anlage des dummy-Files fuer ftok() und fctl()

  {
     struct stat buffer;
     int fd;

     sprintf(rendezvousfile,"%sbalance.%d.%s",SHMDIR,source_port,bindhost_address);

     if(stat(rendezvousfile, &buffer) == -1) {
       // File existiert (noch) nicht ...
       if((fd = open(rendezvousfile, O_CREAT | O_RDWR, 0666)) == -1) {
	 fprintf(stderr,"cannot create rendevouz file %s\n", rendezvousfile);
	 exit(1);
       } else {
	 if(debugflag) 
	   fprintf(stderr,"file %s created\n",rendezvousfile);
	 close(fd);
       }
     } else {
       if(debugflag) 
	 fprintf(stderr,"file %s already there\n",rendezvousfile);
     }
  }

  // --

  signal(SIGCHLD, SIG_IGN);

  if(!foreground) {
    hintergrund();
  }

  if(interactive) {
    // Kommandomodus !

    if((rendezvousfd=open(rendezvousfile,O_RDWR,0)) < 0) {
      perror("open");
      exit(2);
    }
    if((common=(COMMON*) shm_malloc(rendezvousfile,sizeof(COMMON))) == NULL) {
      fprintf(stderr,"cannot alloc COMMON struct\n");
      exit(1);
    }
    shell();
  }

  // Open a TCP socket (an Internet stream socket).

  if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    err_dump("can't open stream socket");
  }

  if(debugflag) {
    fprintf(stderr,"bindhost_adress=[%s]\n",bindhost_address);
  }

  if (bind(sockfd, (struct sockaddr *) &serv_addr, sizeof(serv_addr)) < 0) {
    err_dump("can't bind local address");
  }

  // init of common (*after* bind()) 

  common=makecommon(argc,argv,source_port);

  listen(sockfd, 5);

#ifdef MYBSD
  act.sa_handler = sig_chld;
  sigemptyset(&act.sa_mask);
  act.sa_flags = SA_NOCLDSTOP;
  if (sigaction(SIGCHLD, &act, NULL) < 0) {
    fprintf(stderr, "sigaction failed\n");
    return 1;
  }
#else
#endif

  for (;;) {
    int index;
    int groupindex=0;	// always start at groupindex 0

    // ... richards original comment left intact:  
    // Wait for a connection from a client process.
    // This is an example of a concurrent server.

    clilen = sizeof(cli_addr);

    newsockfd = accept(sockfd, (struct sockaddr *) &cli_addr, &clilen);
    if (newsockfd < 0) {
      if(debugflag) {
        fprintf(stderr,"accept error %d\n", errno);
      }
      continue;
    }

    // the balancing itself ...

    b_writelock();

      for(;;) {

        index = grp_current(common,groupindex);
        for(;;) {
          if(chn_status(common,groupindex,index) == 1 && 
             (chn_maxc(common,groupindex,index) == 0 ||
              (chn_c(common,groupindex,index) < chn_maxc(common,groupindex,index)))
             ) {
	    break; 						// channel found
	  } else {
	    index++;
	    if(index >= grp_nchannels(common,groupindex)) {
	      index=0;
	    }
	    if(index == grp_current(common,groupindex)) {
	      index =-1;					// no channel available in this group
	      break;
	    }
	  }
        }
        grp_current(common,groupindex)=index;
        grp_current(common,groupindex)++;
        if(grp_current(common,groupindex) >= grp_nchannels(common,groupindex)) {
          grp_current(common,groupindex)=0;
        }

        if(index >=0) {
          chn_c(common,groupindex,index)++; 			// we promise a succesful connection 
								// c++ 
          break;						// index in this group found 
        } else {
          groupindex++;
          if(groupindex >= MAXGROUPS) {
            break;	// end of groups...
          }
        }  
      }

    b_unlock();
    
    if(index >= 0) {
      if ((childpid = fork()) < 0) {

	// the connection is rejected if fork() returns error, 
        // but main process stays alive !

	if(debugflag) {
          fprintf(stderr,"fork error\n");
	}
      } else if (childpid == 0) {				// child process 
        close(sockfd);						// close original socket 
        stream(newsockfd, groupindex, index);			// process the request 
        exit(0);
      }
    }

    close(newsockfd);						// parent process 
  }
}

