// Copyright 2000 by Kevin Atkinson under the terms of the LGPL

#include <strstream>

#include <pspell/app_string.hh>

#include "simple_fstream.hh"
#include "data.hh"
#include "language.hh"
#include "language_exceps.hh"
#include "file_util.hh"
#include "config.hh"
#include "manager.hh"

#include "data_id.hh"

namespace aspell {

  DataSet::Id::Id(DataSet * p, const FileName & fn)
    : ptr(p)
  {
    file_name = fn.name;
#ifdef USE_FILE_INO
    struct stat s;
    // the file ,i
    if (file_name[0] != '\0' && stat(fn.path.c_str(), &s) == 0) {
      ino = s.st_ino;
      dev = s.st_dev;
    } else {
      ino = 0;
      dev = 0;
    }
#endif
  }

  bool operator==(const DataSet::Id & rhs, const DataSet::Id & lhs)
  {
    if (rhs.ptr == 0 || lhs.ptr == 0) {
      if (rhs.file_name == 0 || lhs.file_name == 0)
	return false;
#ifdef USE_FILE_INO
      return rhs.ino == lhs.ino && rhs.dev == lhs.dev;
#else
      return strcmp(rhs.file_name, lhs.file_name) == 0;
#endif
    } else {
      return rhs.ptr == lhs.ptr;
    }
  }

  void DataSet::attach(const Language &l) {
    if (lang_ && strcmp(l.name(),lang_->name()) != 0)
      throw MismatchedLang(lang_->name(), l.name());
    if (!lang_) lang_.reset(new Language(l));
    ++attach_count_;
  }

  void DataSet::detach() {
    --attach_count_;
  }

  DataSet::DataSet()
    : lang_(), attach_count_(0), id_(), basic_type(no_type) 
  {
    id_.reset(new Id(this));
  }

  DataSet::~DataSet() {
  }

  bool DataSet::is_attached() const {
    return attach_count_;
  }

  const char * DataSet::lang_name() const {
    return lang_->name();
  }

  void DataSet::check_lang(const string &l) {
    if (l != lang_->name()) 
      throw MismatchedLang(lang_->name(), l);
  }

  void DataSet::set_check_lang (const string & l, Config * config)
  {
    if (lang_ == 0) {
      lang_.reset(new Language(l,*config));
      if (!lang_)
	throw UnknownLang(l);
      set_lang_hook(config);
    } else {
      if (l != lang_->name()) 
	throw MismatchedLang(lang_->name(), l);
    }
  }

  void DataSet::FileName::copy(const FileName & other) 
  {
    const_cast<string &      >(path) = other.path;
    const_cast<const char * &>(name) = path.c_str() + (other.name - other.path.c_str());
  }

  void DataSet::FileName::clear()
  {
    const_cast<string &      >(path) = "";
    const_cast<const char * &>(name) = path.c_str();
  }

  void DataSet::FileName::set(const string & str) 
  {
    const_cast<string &>(path) = str;
    int s = path.size();
    int i = s;
    while (i >= 0) {
      if (path[i] == '/' || path[i] == '\\') {
	++i;
	break;
      }
      --i;
    }
    int i0 = i;
    while (i < s) {
      const_cast<string &>(path)[i] = tolower(path[i]);
      ++i;
    }
    const_cast<const char * &>(name) = path.c_str() + i0;
  }
  

  void LoadableDataSet::set_file_name(const string & fn) 
  {
    file_name_.set(fn);
    *id_ = Id(this, file_name_);
  }

  void LoadableDataSet::update_file_info(SimpleFstream & f) 
  {
#ifdef USE_FILE_INO
    struct stat s;
    int ok = fstat(filedesc(f), &s);
    assert(ok == 0);
    id_->ino = s.st_ino;
    id_->dev = s.st_dev;
#endif
  }

  const char * CompoundInfo::read(const char * i, 
				  const Language & lang)
  {
    if (*i == '\0') return i;
    // FIXME: properally handle ignoring parm 
    // if not a compound parameter
    d = 0;
    const char * i0 = i;
    if (*i == ':') ++i;
    bool use_compound = false;
    if (*i == 'C' || *i == 'c') {++i; use_compound = true;}
    if (*i == '1') {++i; beg(true);}
    if (*i == '2') {++i; mid(true);}
    if (*i == '3') {++i; end(true);}
    if (!any())  
    {
      if (!use_compound)
	return i0;
      beg(true); mid(true); end(true);
    }
    int m = 0;
    const char * nm = lang.mid_chars();
    while (nm[m] != '\0' && nm[m] != lang.to_lower(*i))
      ++m;
    mid_char(m);
    if (nm[m] != '\0') {
      mid_required(lang.is_upper(*i));
      ++i;
    }
    return i;
  }

  void CompoundInfo::write (PspellAppendableString & o,
			    const Language & l) const
  {
    if (!any()) return;
    o += ":c";
    if (!beg() || !mid() || !end()) {
      if (beg()) o += "1";
      if (mid()) o += "2";
      if (end()) o += "3";
    }
    if (l.mid_chars()[mid_char()]) {
      char c;
      if (mid_required())
	c = l.to_upper(l.mid_chars()[mid_char()]);
      else
	c = l.to_upper(l.mid_chars()[mid_char()]);
      o.append(&c, 1);
    }
  }

  template <class O>
  class AppOstream : public PspellAppendableString 
  {
    O * o_;
  public:
    AppOstream(O & o) : o_(&o) {}
    void append (const char * str) 
    {
      *o_ << str;
    }
    void append (const char * str, unsigned int size) 
    {
      o_->write(str, size);
    }
  };

  SimpleFstream & CompoundInfo::write(SimpleFstream & o, const Language & l) const
  {
    AppOstream<SimpleFstream> ao(o);
    write(ao, l);
    return o;
  }

  ostream & CompoundInfo::write(ostream & o, const Language & l) const
  {
    AppOstream<ostream> ao(o);
    write(ao, l);
    return o;
  }

  CompoundInfo::Position new_position (CompoundInfo::Position unsplit_word, 
				       CompoundInfo::Position pos) 
  {
    switch (unsplit_word) {
    case CompoundInfo::Orig:
      return pos;
    case CompoundInfo::Beg:
      if (pos == CompoundInfo::End) 
	return CompoundInfo::Mid;
      else
	return pos;
    case CompoundInfo::Mid:
      return CompoundInfo::Mid;
    case CompoundInfo::End:
      if (pos == CompoundInfo::Beg)
	return CompoundInfo::Mid;
      else
	return pos;
    }
    abort();
  }

  bool CompoundInfo::compatible(Position pos) {
    switch (pos) {
    case Beg:
      return beg();
    case Mid:
      return mid();
    case End:
      return end();
    case Orig:
      abort();
    }
    abort();
  }

  void SingleWordInfo::append_word(string & w, const Language &, 
				   const ConvertWord & c) const
  {
    c.convert(word,w);
    if (middle_char != '\0')
      w += middle_char;
  }

  void WordInfo::get_word(string & word, const Language & l, 
			  const ConvertWord & c) const
  {
    word = "";
    for (const SingleWordInfo * i = words; *i; ++i) {
      i->append_word(word,l,c);
    }
  }

  SimpleFstream & WordInfo::write(SimpleFstream & o, const Language & l, 
				  const ConvertWord & c) const
  {
    string word;
    get_word(word, l, c);
    o << word;
    return o;
  }

  SimpleFstream & BasicWordInfo::write (SimpleFstream & o,
					const Language & l,
					const ConvertWord & c) const
  {
    string w;
    c.convert(word, w);
    o << w;
    compound.write(o,l);
    return o;
  }

  ostream & WordInfo::write(ostream & o, const Language & l, 
			    const ConvertWord & c) const
  {
    string word;
    get_word(word, l, c);
    o << word;
    return o;
  }

  ostream & BasicWordInfo::write (ostream & o,
				  const Language & l,
				  const ConvertWord & c) const
  {
    string w;
    c.convert(word, w);
    o << w;
    compound.write(o,l);
    return o;
  }

  LoadableDataSet * add_data_set(const string & fn,
				 Config & config,
				 Manager * manager,
				 const LocalWordSetInfo * local_info,
				 DataType allowed)
  {
    static const char * suffix_list[] = {"", ".multi", ".pws", ".prepl"};
    SimpleFstream in;
    const char * * suffix = suffix_list;
    const char * * suffix_end 
      = suffix_list + sizeof(suffix_list)/sizeof(const char *);
    string dict_dir = config.retrieve("dict-dir");
    string true_file_name;
    DataSet::FileName file_name(fn);
    do {
      true_file_name = add_possible_dir(dict_dir, file_name.path+*suffix);
      in.open(true_file_name, "r");
      ++suffix;
    } while (!in && suffix != suffix_end);
    if (!in) {
      true_file_name = add_possible_dir(dict_dir, file_name.path);
      throw CantReadFile(true_file_name);
    }

    DataType actual_type;
    if (true_file_name.size() > 6 
	&& true_file_name.substr(true_file_name.size() - 6, 6) == ".multi") {
      
      actual_type = DT_Multi;

    } else {
      
      char head[16];
      in.read(head, 16);
      head[15] = '\0';

      if      (strncmp(head, "aspell rowl", 11) ==0)
	actual_type = DT_ReadOnly;
      else if (strncmp(head, "personal_repl", 13) == 0)
	actual_type = DT_WritableRepl;
      else if (strncmp(head, "personal_ws", 11) == 0)
	actual_type = DT_Writable;
      else
	throw BadFileFormat(true_file_name);

    }
    
    if (actual_type & ~allowed)
      throw BadFileFormat(true_file_name, "is not one of the allowed types");

    LoadableDataSet * ws = 0;
    if (manager != 0)
      ws = manager
	->locate(DataSet::Id(0,DataSet::FileName(true_file_name))).word_set;

    if (ws != 0) {
      cerr << "Warning: " << true_file_name << " already exists!" << endl;      
      return ws;
    }

    switch (actual_type) {
    case DT_ReadOnly: 
      ws = new_default_readonly_word_set();
      break;
    case DT_Multi:
      ws = new_default_multi_word_set();
      break;
    case DT_Writable: 
      ws = new_default_writable_word_set(); 
      break;
    case DT_WritableRepl:
      ws = new_default_writable_replacement_set();
      break;
    default:
      abort();
    }

    ws->load(true_file_name, &config, manager, local_info);
    if (manager != 0)
      manager->steal(ws, local_info);

    return ws;
    
  }
  
  void LocalWordSetInfo::set_language(const Language * l)
  {
    compare.lang = l;
    convert.lang = l;
  }

  void LocalWordSetInfo::set(const Language * l, const Config * c, bool strip)
  {
    if (c->have("strip-accents"))
      strip = c->retrieve_bool("strip-accents");

    compare.lang = l;
    compare.case_insensitive = c->retrieve_bool("ignore-case");
    compare.ignore_accents   = c->retrieve_bool("ignore-accents");
    compare.strip_accents    = strip;
    convert.lang = l;
    convert.strip_accents = strip;
  }
  
}

