/*
 * Argus Software
 * Copyright (c) 2000-2008 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * rafilteraddr - filter records based on an address list.  bypasses
 *                standard filter compiler.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */

/* 
 * $Id: $
 * $DateTime: $
 * $Change: $
 */

#if defined(CYGWIN)
#define USE_IPV6
#endif

#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <ctype.h>

#if defined(HAVE_SOLARIS)
#include <strings.h>
#include <string.h>
#endif

#include <math.h>

#include <rabins.h>
#include <argus_util.h>
#include <argus_labeler.h>
#include <argus_client.h>
#include <argus_filter.h>
#include <argus_main.h>
#include <argus_cluster.h>


int RaProcessAddress (struct ArgusParserStruct *, struct ArgusRecordStruct *, unsigned int *, int);
struct ArgusLabelerStruct *ArgusNewLabeler (struct ArgusParserStruct *);
struct ArgusLabelerStruct *ArgusLabeler = NULL;

int RaReadAddressConfig (struct ArgusParserStruct *, struct ArgusLabelerStruct *, char *);

/*
   IANA style address label configuration file syntax is:
      addr "label"

      where addr is:
         %d[[[.%d].%d].%d]/%d   CIDR address
         CIDR - CIDR            Address range

   The Regional Internet Registries (RIR) database support allows for
   country codes to be associated with address prefixes.  We'll treat
   them as simple labels.   The file syntax is:

      rir|co|[asn|ipv4|ipv6]|#allocatable|[allocated | assigned]

   so if we find '|', we know the format.
   
*/


#define ARGUS_EXACT_MATCH	0
#define ARGUS_LONGEST_MATCH	1
#define ARGUS_ANY_MATCH		2

int ArgusRmonMode = 0;

void RaMapLabelMol (struct ArgusLabelerStruct *, struct RaAddressStruct *, int, int, int, int);
void RaPrintLabelMol (struct ArgusLabelerStruct *, struct RaAddressStruct *, int, int, int, int);
void RaPrintLabelTree (struct ArgusLabelerStruct *, struct RaAddressStruct *, int, int);


int RaPrintLabelTreeMode = 0;

void
ArgusClientInit (struct ArgusParserStruct *parser)
{
   struct ArgusLabelerStruct *labeler = NULL;
   struct ArgusModeStruct *mode = NULL;
   parser->RaWriteOut = 0;

   if (!(parser->RaInitialized)) {
      (void) signal (SIGHUP,  (void (*)(int)) RaParseComplete);

      if ((parser->ArgusLabeler = ArgusNewLabeler(parser)) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit: ArgusNewLabeler error");

      if (parser->ArgusFlowModelFile)
         RaReadAddressConfig (parser, parser->ArgusLabeler, parser->ArgusFlowModelFile);
      else
         ArgusLog (LOG_ERR, "ArgusClientInit: no address list, use -f");
         



      labeler = parser->ArgusLabeler;

      if ((parser->ArgusAggregator = ArgusNewAggregator(parser, NULL)) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit: ArgusNewAggregator error");

      if ((mode = parser->ArgusModeList) != NULL) {
         while (mode) {
            if (!(strncasecmp (mode->mode, "debug.mol", 9))) {
               RaPrintLabelTreeMode = ARGUS_MOL;

               RaMapLabelMol (ArgusLabeler, labeler->ArgusAddrTree[AF_INET], 0, 0, 0, 0);
               RaPrintLabelMol (ArgusLabeler, labeler->ArgusAddrTree[AF_INET], 0, 0, 0, 0);
               exit(0);
            }
            if ((!(strncasecmp (mode->mode, "debug.tree", 10))) ||
                (!(strncasecmp (mode->mode, "debug", 5)))) {
               RaPrintLabelTreeMode = ARGUS_TREE;
               RaPrintLabelTree (ArgusLabeler, labeler->ArgusAddrTree[AF_INET], 0, 0);
               exit(0);
            }

            mode = mode->nxt;
         }
      }

      parser->RaInitialized++;
   }
}

void RaArgusInputComplete (struct ArgusInput *input) { return; }


int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   if ((sig >= 0) && (!RaParseCompleting)) {
      RaParseCompleting++;

      if ((ArgusParser->ArgusWfileList != NULL) && (!(ArgusListEmpty(ArgusParser->ArgusWfileList)))) {
         struct ArgusWfileStruct *wfile = NULL, *start = NULL;
 
         if ((wfile = (struct ArgusWfileStruct *) ArgusFrontList(ArgusParser->ArgusWfileList)) != NULL) {
            start = wfile;
            fflush(wfile->fd);
            ArgusPopFrontList(ArgusParser->ArgusWfileList, ARGUS_NOLOCK);
            ArgusPushBackList(ArgusParser->ArgusWfileList, (struct ArgusListRecord *) wfile, ARGUS_NOLOCK);
            wfile = (struct ArgusWfileStruct *) ArgusFrontList(ArgusParser->ArgusWfileList);
         } while (wfile != start);
      } 
   }

   fflush(stdout);
   exit(0);

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaParseComplete (%d) returning\n", sig);
#endif
}

void
ArgusClientTimeout ()
{

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusClientTimeout: returning\n");
#endif
}

void
parse_arg (int argc, char**argv)
{ 

#ifdef ARGUSDEBUG
   ArgusDebug (6, "parse_arg (%d, 0x%x) returning\n", argc, argv);
#endif
}


void
usage ()
{
   extern char version[];
   fprintf (stderr, "rafilteraddr Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusParser->ArgusProgramName);
   fprintf (stderr, "usage: %s -f address.file [raoptions] -S remoteServer  [- filter-expression]\n", ArgusParser->ArgusProgramName);
   fprintf (stderr, "usage: %s -f address.file [raoptions] -r argusDataFile [- filter-expression]\n", ArgusParser->ArgusProgramName);
   exit(1);
}

extern struct RaAddressStruct *RaFindAddress (struct ArgusParserStruct *, struct RaAddressStruct *, struct RaAddressStruct *, int);

int
RaProcessAddress (struct ArgusParserStruct *parser, struct ArgusRecordStruct *argus, unsigned int *addr, int type)
{
   struct ArgusLabelerStruct *labeler = NULL;
   struct RaAddressStruct *raddr;
   int retn = 0;

   if ((labeler = parser->ArgusLabeler) == NULL)
      ArgusLog (LOG_ERR, "RaProcessAddress: No labeler\n");

   switch (type) {
      case ARGUS_TYPE_IPV4: {
         struct RaAddressStruct node;
         bzero ((char *)&node, sizeof(node));

         node.addr.type = AF_INET;
         node.addr.len = 4;
         node.addr.addr[0] = *addr;
         node.addr.masklen = 32;

         if ((raddr = RaFindAddress (parser, labeler->ArgusAddrTree[AF_INET], &node, ARGUS_EXACT_MATCH)) != NULL) {
            retn++;
         }
         break;
      }

      case ARGUS_TYPE_IPV6:
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaProcessAddress (0x%x, 0x%x, 0x%x, %d) returning %d\n", parser, argus, addr, type, retn);
#endif

   return (retn);
}

void
RaProcessRecord (struct ArgusParserStruct *parser, struct ArgusRecordStruct *argus)
{
   struct ArgusFlow *flow = (struct ArgusFlow *) argus->dsrs[ARGUS_FLOW_INDEX];
   int retn = 0;

   if (flow && (!(argus->hdr.type & ARGUS_MAR))) {
      switch (flow->hdr.subtype & 0x3F) {
         case ARGUS_FLOW_CLASSIC5TUPLE:
         case ARGUS_FLOW_LAYER_3_MATRIX: {
            switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
               case ARGUS_TYPE_IPV4:
                  if ((!retn && parser->ArgusAggregator->mask & ARGUS_MASK_SADDR_INDEX))
                     retn = RaProcessAddress(parser, argus, &flow->ip_flow.ip_src, ARGUS_TYPE_IPV4);
                  if (!retn && (parser->ArgusAggregator->mask & ARGUS_MASK_DADDR_INDEX))
                     retn = RaProcessAddress(parser, argus, &flow->ip_flow.ip_dst, ARGUS_TYPE_IPV4);
                  break;
               case ARGUS_TYPE_IPV6:
                  if (!retn && (parser->ArgusAggregator->mask & ARGUS_MASK_SADDR_INDEX))
                     retn = RaProcessAddress(parser, argus, (unsigned int *) &flow->ipv6_flow.ip_src, ARGUS_TYPE_IPV6);
                  if (!retn && (parser->ArgusAggregator->mask & ARGUS_MASK_DADDR_INDEX))
                     retn = RaProcessAddress(parser, argus, (unsigned int *) &flow->ipv6_flow.ip_dst, ARGUS_TYPE_IPV6);
                  break;
            }
            break; 
         }
      }
   }

   if (parser->vflag)
      retn = (retn) ? 0 : 1;

   if (parser->ArgusWfileList != NULL) {
      struct ArgusWfileStruct *wfile = NULL;
      struct ArgusListObjectStruct *lobj = NULL;
      int i, count = parser->ArgusWfileList->count;

      if ((lobj = parser->ArgusWfileList->start) != NULL) {
         for (i = 0; i < count; i++) {
            if ((wfile = (struct ArgusWfileStruct *) lobj) != NULL) {
               struct ArgusRecord *argusrec = NULL;
               static char sbuf[0x10000];

               if ((argusrec = ArgusGenerateRecord (argus, 0L, sbuf)) != NULL) {
#ifdef _LITTLE_ENDIAN
                  ArgusHtoN(argusrec);
#endif
                  if (parser->exceptfile != NULL) {
                     if (retn && strcmp(wfile->filename, parser->exceptfile))
                        ArgusWriteNewLogfile (parser, argus->input, wfile, argusrec);
                     else
                        if (!retn && !strcmp(wfile->filename, parser->exceptfile))
                           ArgusWriteNewLogfile (parser, argus->input, wfile, argusrec);

                  } else {
                     if (retn)
                        ArgusWriteNewLogfile (parser, argus->input, wfile, argusrec);
                  }
               }
            }

            lobj = lobj->nxt;
         }
      }

   } else {
      if (retn) {
         char buf[MAXSTRLEN];
         if (!parser->qflag) {
            if (parser->Lflag) {
               if (parser->RaLabel == NULL)
                  parser->RaLabel = ArgusGenerateLabel(parser, argus);
    
               if (!(parser->RaLabelCounter++ % parser->Lflag))
                  printf ("%s\n", parser->RaLabel);
    
               if (parser->Lflag < 0)
                  parser->Lflag = 0;
            }

            *(int *)&buf = 0;
            ArgusPrintRecord(parser, buf, argus, MAXSTRLEN);
            fprintf (stdout, "%s ", buf);
         }
         if (parser->ArgusWfileList == NULL)
            fprintf (stdout, "\n");
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaProcessRecord (0x%x) returning\n", argus);
#endif
}


int
RaSendArgusRecord(struct ArgusRecordStruct *argus)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaSendArgusRecord (0x%x) returning\n", argus);
#endif
   return 1;
}

void ArgusWindowClose(void) { } 

int RaLabelItemNum = 0;

float xBaseValue = 30.0;
float yBaseValue = 100.0;

float yDelta = -2.0;
float xDelta = -12.0;

void
RaMapLabelMol (struct ArgusLabelerStruct *labeler, struct RaAddressStruct *node, int level, int x, int y, int dir)
{
   if (node != NULL) {
      x += (xDelta * 16.0/node->addr.masklen);
      if (node->r) RaMapLabelMol(labeler, node->r, level + 1, x, y, dir);
      node->x = x;
      node->y = yBaseValue + (RaLabelItemNum++ * yDelta);
      if (node->l) RaMapLabelMol(labeler, node->l, level + 1, x, y, dir);
   }
}

void
RaPrintLabelMol (struct ArgusLabelerStruct *labeler, struct RaAddressStruct *node, int level, int x, int y, int dir)
{
   char strbuf[256];
   float xl, yl, zl;
   int slen = 0;

   if (node != NULL) {
      float size = 0.2;

      if (node->addr.masklen);
         size = 32.0/node->addr.masklen;

      if (node->r) {
         printf ("draw arrow {%f %f %f} {%f %f %f}\n", node->x, node->y, 0.0, node->r->x, node->r->y, 0.0); 
         RaPrintLabelMol(labeler, node->r, level + 1, x, y, RA_SRV_RIGHT);
      }

      if (!(node->r || node->l))
         printf ("draw color green\n");

      printf ("draw sphere {%f %f %f} radius %f resolution 32\n", node->x, node->y, 0.0, size); 

      snprintf (strbuf, 256, "%s/%d ", intoa(node->addr.addr[0] & (0xFFFFFFFF << (32 - node->addr.masklen))), node->addr.masklen);
      printf ("draw color white\n"); 
      slen = strlen(strbuf);

      if (node->label) {
         char *ptr;
         if ((ptr = strchr (node->label, '\n')) != NULL) *ptr = '\0';
         snprintf (&strbuf[slen], 256 - slen, "%s", node->label);
         xl = node->x; yl = node->y; zl = (size*2 + 0.25);

      } else {
         snprintf (&strbuf[slen], 256 - slen,  "\"");
         xl = node->x; yl = node->y; zl = (size*2 + 0.25);
      }

      printf ("draw text {%f %f %f} \"%s size %f\n", xl, yl, zl, strbuf, size/4); 
      printf ("draw color blue\n"); 

      if (node->l) {
         printf ("draw arrow {%f %f %f} {%f %f %f}\n", node->x, node->y, 0.0, node->l->x, node->l->y, 0.0); 
         RaPrintLabelMol(labeler, node->l, level + 1, x, y, RA_SRV_LEFT);
      }
   }
}



char RaAddrTreeArray[MAXSTRLEN];

void
RaPrintLabelTree (struct ArgusLabelerStruct *labeler, struct RaAddressStruct *node, int level, int dir)
{
   int i = 0, length, len;
   int olen = strlen(RaAddrTreeArray);
   char str[MAXSTRLEN], chr = ' ';

   bzero(str, MAXSTRLEN);

   if (node != NULL) {
      if (dir == RA_SRV_LEFT) {
         strcat (str, "   |");
         strcat (RaAddrTreeArray, str);
         printf ("%s\n", RaAddrTreeArray);
      }

      length = strlen(RaAddrTreeArray);
      if ((len = length) > 0) {
         chr = RaAddrTreeArray[len - 1];
         if (node->r != NULL) {
            if (dir == RA_SRV_RIGHT)
               RaAddrTreeArray[len - 1] = ' ';
         }
      }

      strcat (RaAddrTreeArray, "   |");

      RaPrintLabelTree(labeler, node->r, level + 1, RA_SRV_RIGHT);

      for (i = length, len = strlen(RaAddrTreeArray); i < len; i++)
         RaAddrTreeArray[i] = '\0';

      if ((len = length) > 0)
         RaAddrTreeArray[len - 1] = chr;
      
      printf ("%s+", RaAddrTreeArray);

      if (node->label) {
         if (node->addr.str)
            printf ("%s ", node->addr.str);
         else
            printf ("%s/%d ", intoa(node->addr.addr[0] & (0xFFFFFFFF << (32 - node->addr.masklen))), node->addr.masklen);

         printf ("%s ", intoa((0xFFFFFFFF << (32 - node->addr.masklen))));

         if (node->label)
            printf ("%s", node->label);
      }

      printf ("\n");

      len = strlen(RaAddrTreeArray);
      if (len > 0) {
         chr = RaAddrTreeArray[len - 1];
         if (node->l != NULL) {
            if (dir == RA_SRV_LEFT)
               RaAddrTreeArray[len - 1] = ' ';
         }
      }

      RaPrintLabelTree(labeler, node->l, level + 1, RA_SRV_LEFT);

      if (dir == RA_SRV_RIGHT) {
         printf (RaAddrTreeArray);
         putchar ('\n');
      }

      for (i = olen, len = strlen(RaAddrTreeArray); i < len; i++)
         RaAddrTreeArray[i] = '\0';
   }
}
