/*
 * Raw memory allocation interface
 */

#include <stdlib.h>
#include <string.h>

#include "stdlib.h"

/*
 * Private wrapper structure for allocated memory
 */
typedef struct {
  int   size;
  int 	rw;
  void  *data;
} memblock;

/*
 * Initialize memory block structure
 */
static memblock *mem_init(int size, void *data)
{
  memblock *mem;
  
  mem = oom(malloc(sizeof(memblock)));
  mem->size = size;
  mem->data = data;
  mem->rw   = 1;
  
  return mem;
}

/*
 * Clean up memory block structure
 */
static void mem_cleanup(void *data)
{
  memblock *mem = data;
  
  if (mem) free(mem->data);
  free(mem);
}

/*
 * Clean up memory block structure put not contained pointer
 */
static void mem_halfcleanup(void *data)
{
  free(data);
}

/*
 * Allocate a given number of bytes
 */
value *mem_malloc(arena_state *s, unsigned int argc, value **argv)
{
  int size = INT_OF(argv[0]);
  void *res;

  res = malloc(size);
  if (!res) {
    return value_make_void();
  }
  return value_make_resource(RESOURCE_MEM, mem_init(size, res), mem_cleanup);
}

/*
 * Allocate a given number of zero-filled bytes
 */
value *mem_calloc(arena_state *s, unsigned int argc, value **argv)
{
  int size = INT_OF(argv[0]) * INT_OF(argv[1]);
  void *res;
  
  res = calloc(INT_OF(argv[0]), INT_OF(argv[1]));
  if (!res) {
    return value_make_void();
  }
  return value_make_resource(RESOURCE_MEM, mem_init(size, res), mem_cleanup);
}

/*
 * Free a memory resource
 */
value *mem_free(arena_state *s, unsigned int argc, value **argv)
{
  memblock *mem = RESDATA_OF(argv[0]);
  
  if (RESTYPE_OF(argv[0]) == RESOURCE_MEM && mem) {
    mem_cleanup(mem);
    RESDATA_OF(argv[0]) = NULL;
  }
  return value_make_void();
}

/*
 * Create resource for C NULL pointer
 */
value *mem_cnull(arena_state *s, unsigned int argc, value **argv)
{
  return value_make_resource(RESOURCE_MEM, NULL, mem_cleanup);
}

/*
 * Create resource from string value
 */
value *mem_cstring(arena_state *s, unsigned int argc, value **argv)
{
  char *str = argv[0]->value_u.string_val.value;
  int len;
  char *res;

  if (str && !value_str_compat(argv[0])) {
    return value_make_void();
  }

  if (!str) {
    str = "";
  }
  
  len = strlen(str) + 1;
  res = oom(malloc(len));
  strcpy(res, str);
  
  return value_make_resource(RESOURCE_MEM, mem_init(len, res), mem_cleanup);
}

/*
 * Put character at offset in memory resource
 */
value *mem_put_char(arena_state *s, unsigned int argc, value **argv)
{
  memblock *mem = RESDATA_OF(argv[0]);
  char *put;
  int offset = INT_OF(argv[1]);
  char val = INT_OF(argv[2]);
  
  if (RESTYPE_OF(argv[0]) != RESOURCE_MEM || !mem || !mem->rw) {
    return value_make_bool(0);
  }

  if (mem->size < offset + sizeof(char)) {
    mem->data = oom(realloc(mem->data, offset + sizeof(char)));
    mem->size = offset + sizeof(char);
  }

  put = mem->data;
  put += offset;
  *put = val;

  return value_make_bool(1);
}

/*
 * Put short integer at offset in memory resource
 */
value *mem_put_short(arena_state *s, unsigned int argc, value **argv)
{
  memblock *mem = RESDATA_OF(argv[0]);
  char *pos;
  short *put;
  int offset = INT_OF(argv[1]);
  short val = INT_OF(argv[2]);
  
  if (RESTYPE_OF(argv[0]) != RESOURCE_MEM || !mem || !mem->rw) {
    return value_make_bool(0);
  }

  if (mem->size < offset + sizeof(short)) {
    mem->data = oom(realloc(mem->data, offset + sizeof(short)));
    mem->size = offset + sizeof(short);
  }

  pos = mem->data;
  pos += offset;
  put = (short *) pos;
  *put = val;

  return value_make_bool(1);
}

/*
 * Put integer at offset in memory resource
 */
value *mem_put_int(arena_state *s, unsigned int argc, value **argv)
{
  memblock *mem = RESDATA_OF(argv[0]);
  char *pos;
  int *put;
  int offset = INT_OF(argv[1]);
  int val = INT_OF(argv[2]);
  
  if (RESTYPE_OF(argv[0]) != RESOURCE_MEM || !mem || !mem->rw) {
    return value_make_bool(0);
  }

  if (mem->size < offset + sizeof(int)) {
    mem->data = oom(realloc(mem->data, offset + sizeof(int)));
    mem->size = offset + sizeof(int);
  }

  pos = mem->data;
  pos += offset;
  put = (int *) pos;
  *put = val;

  return value_make_bool(1);
}

/*
 * Put pointer to another resource at offset in resource
 */
value *mem_put_pointer(arena_state *s, unsigned int argc, value **argv)
{
  memblock *mem = RESDATA_OF(argv[0]);
  memblock *ptr = RESDATA_OF(argv[2]);
  char *pos;
  void **put;
  int offset = INT_OF(argv[1]);
  
  if (RESTYPE_OF(argv[0]) != RESOURCE_MEM || !mem || !mem->rw ||
      RESTYPE_OF(argv[2]) != RESOURCE_MEM) {
    return value_make_bool(0);
  }

  if (mem->size < offset + sizeof(void *)) {
    mem->data = oom(realloc(mem->data, offset + sizeof(void *)));
    mem->size = offset + sizeof(void *);
  }
  
  pos = mem->data;
  pos += offset;
  put = (void **) pos;
  if (ptr) {
    *put = ptr->data;
  } else {
    *put = NULL;
  }
  
  return value_make_bool(1);
}

/*
 * Get character from offset in memory resource
 */
value *mem_get_char(arena_state *s, unsigned int argc, value **argv)
{
  memblock *mem = RESDATA_OF(argv[0]);
  char *get;
  int offset = INT_OF(argv[1]);
  
  if (RESTYPE_OF(argv[0]) != RESOURCE_MEM || !mem ||
      (mem->size && offset + sizeof(char) > mem->size)) {
    return value_make_void();
  }
  
  get = mem->data;
  get += offset;
  
  return value_make_int((int) *get);
}

/*
 * Get short integer from offset in memory resource
 */
value *mem_get_short(arena_state *s, unsigned int argc, value **argv)
{
  memblock *mem = RESDATA_OF(argv[0]);
  char *pos;
  short *get;
  int offset = INT_OF(argv[1]);
  
  if (RESTYPE_OF(argv[0]) != RESOURCE_MEM || !mem ||
      (mem->size && offset + sizeof(short) > mem->size)) {
    return value_make_void();
  }
  
  pos = mem->data;
  pos += offset;
  get = (short *) pos;
  
  return value_make_int((int) *get);
}

/*
 * Get integer from offset in memory resource
 */
value *mem_get_int(arena_state *s, unsigned int argc, value **argv)
{
  memblock *mem = RESDATA_OF(argv[0]);
  char *pos;
  int *get;
  int offset = INT_OF(argv[1]);
  
  if (RESTYPE_OF(argv[0]) != RESOURCE_MEM || !mem ||
      (mem->size && offset + sizeof(int) > mem->size)) {
    return value_make_void();
  }
  
  pos = mem->data;
  pos += offset;
  get = (int *) pos;
  
  return value_make_int(*get);
}

/*
 * Create memory resource from direct pointer
 */
value *mem_make_pointer(void *data, int free)
{
  memblock *mem;
  value *res;
  
  res = value_make_resource(RESOURCE_MEM, mem_init(0, data),
    free ? mem_cleanup : mem_halfcleanup);
  mem = RESDATA_OF(res);
  mem->rw = 0;
  
  return res;
}

/*
 * Get memory resource from offset in memory resource
 */
value *mem_get_pointer(arena_state *s, unsigned int argc, value **argv)
{
  memblock *mem = RESDATA_OF(argv[0]);
  memblock *resmem;
  char *pos;
  void **get;
  int offset = INT_OF(argv[1]);
  int cleanup = BOOL_OF(argv[2]);
  value *res;

  if (RESTYPE_OF(argv[0]) != RESOURCE_MEM || !mem ||
      (mem->size && offset + sizeof(void *) > mem->size)) {
    return value_make_void();
  }
  
  pos = mem->data;
  pos += offset;
  get = (void **) pos;
  
  res = value_make_resource(RESOURCE_MEM, mem_init(0, *get),
    cleanup ? mem_cleanup : mem_halfcleanup);
  resmem = RESDATA_OF(res);
  resmem->rw = 0;
  
  return res;
}

/*
 * Get string value from offset in memory resource
 */
value *mem_string(arena_state *s, unsigned int argc, value **argv)
{
  memblock *mem = RESDATA_OF(argv[0]);
  char *get;
  int offset = INT_OF(argv[1]);
  
  if (RESTYPE_OF(argv[0]) != RESOURCE_MEM || !mem ||
      (mem->size && offset + sizeof(char) > mem->size)) {
    return value_make_void();
  }
  
  get = mem->data;
  get += offset;
  
  return value_make_string(get);
}

/*
 * Check whether memory resource is writeable
 */
value *mem_is_rw(arena_state *s, unsigned int argc, value **argv)
{
  memblock *mem = RESDATA_OF(argv[0]);
  
  if (RESTYPE_OF(argv[0]) != RESOURCE_MEM || !mem || !mem->rw) {
    return value_make_bool(0);
  }
  return value_make_bool(1);
}

/*
 * Return current allocation size of memory resource (0 = unknown)
 */
value *mem_size(arena_state *s, unsigned int argc, value **argv)
{
  memblock *mem = RESDATA_OF(argv[0]);
  
  if (RESTYPE_OF(argv[0]) != RESOURCE_MEM) {
    return value_make_void();
  }
  if (!mem) {
    return value_make_int(0);
  }
  return value_make_int(mem->size);
}
