/*
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL TOM KNIENIEDER BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF TOM
 * KNIENIEDER HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * TOM KNIENIEDER SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND TOM KNIENIEDER HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

/* 
 * UPS daemon
 * -----------------------------------------------------------
 * Tom Knienieder, OpenBSD
 * changed the name to apc-upsd tue to a naming conflict with
 * an exitsting package.
 * -----------------------------------------------------------
 * Tom Knienieder, OpenBSD
 * added -USR1 according to Chris Capuccio.
 * -----------------------------------------------------------
 * Tom Knienieder, OpenBSD
 * works with APC BackUPS Pro with the APC "gray" cable.
 * -----------------------------------------------------------
 * Tom Knienieder, OpenBSD
 * works with APC Smart-UPS with the APC "black" cable.
 * (part nr. 940-0024C)
 * -----------------------------------------------------------
 * Copyright 1997 Slavik Terletsky. All rights reserved.
 * Author: Slavik Terletsky <ts@polynet.lviv.ua>
 * System: FreeBSD
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <syslog.h>
#include <unistd.h>
#include <varargs.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/uio.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/ttycom.h>
#include <sys/ioctl.h>
#include <sys/termios.h>

#include "apc_upsd.h"

/* ------------------------------------------------------------------ */

UPS     *ups = NULL;

/* ------------------------------------------------------------------ */

void    upsusage()
{
        fprintf(stderr, "usage: %s <filename>\n", ups->myname );
}

/* ------------------------------------------------------------------ */

int main(int argc, char *argv[]) 
{
        if ( ( ups = (UPS *)calloc(1, sizeof(UPS))) == NULL )
        {
                fprintf(stderr, "out of core.\n");
                exit(1);
        }
        ups->myname     = * argv;
        ups->pow                = 1;
        ups->filename   = UPS_DEFAULT_CONFIG_FILENAME;

        if ( -- argc )
        {
                ups->filename = * ++ argv;
                if ( -- argc )
                {
                        upsusage();
                        exit(1);
                }
        }

        if ( ! parser())
                exit(3);

        /* check if script exists */

        if(!(ups->fd = fopen(ups->exec, "r"))) 
        {
                fprintf(stderr, "open: %s: %s\n", ups->exec, 
                        sys_errlist[errno]);
                exit(1);
        }
        fclose(ups->fd);

        /* check if upsd is already running */

        if((ups->fd = fopen(ups->pidfile, "r")) != NULL ) 
        {
                fprintf(stderr, "open: %s: File already exists\n", ups->pidfile);
                exit(1);
        }

        /* become a daemon */

        if ( ! ups->debug )
                upsfork(); 
        else
                printf("\n\nnot forking in debug mode ...\n");

        /* create the pid file */

        if ( ! ups->debug )
                upspidfile();

        /* open ... */

        if ( ! upsopen())
                exit(2);

        /* daemon is alive */

        openlog("upsd", LOG_PID, LOG_DAEMON);
        syslog(LOG_INFO, "daemon started");
        syslog(LOG_INFO, "device at %s, time %d", ups->device, ups->wait );

        /* signal reaction */

        (void)signal(SIGTERM, upsterm);
        (void)signal(SIGUSR1, upstest);

        upsloop();

        /* not reached */
        
        return 0;
}

/* ------------------------------------------------------------------ */

void upstest() 
{
        /* log test message */
        syslog(LOG_INFO, "test called.");

        ups->testflag = 1;

        return;
}

/* ------------------------------------------------------------------ */

void upsterm() 
{
        /* log termination message */
        syslog(LOG_INFO, "daemon terminated");

        /* remove pid file */

        unlink(ups->pidfile);
        exit(0);
}

/* ------------------------------------------------------------------ */

void upsdown() 
{
        /* log shutdown message */
        syslog(LOG_ALERT, "system is going down");

        /* remove pid file */
        unlink(ups->pidfile);

        /* save our filesystem */
        system("/bin/sync");
        system("/bin/sync");
        system("/bin/sync");

        /* shutdown ups */

        if ( ups->smartmode )
                apc_smart_UPS_off();
        else
                apc_dumb_UPS_off();

        system(ups->exec);
}


/* ------------------------------------------------------------------ */


int     upsopen()
{
        /* open monitor device */

        if ( ups->smartmode )
                return  apc_smart_open();
        else
                return  apc_dumb_open();

        /* notreached */

        return  0;
}


/* ------------------------------------------------------------------ */

int     upsloop()
{
        if ( ups->smartmode )
                return  apc_smart_loop();
        else
                return  apc_dumb_loop();

        /* notreached */ 

        return  0;
}
        
/* ------------------------------------------------------------------ */

void    upsfork()
{
        switch(fork()) 
        {
        case -1:       /* error */
                fprintf(stderr, "fork: %s\n", sys_errlist[errno]);
                exit(1);
        case  0:       /* child */
                break;
        default:       /* parent */
                exit(0);
        }
}

/* ------------------------------------------------------------------ */


void    upspidfile()
{
        /* save the pid */
        if(!(ups->fd = fopen(ups->pidfile, "w"))) 
        {
                fprintf(stderr, "open: %s: %s\n", 
                        ups->pidfile, sys_errlist[errno]);
                exit(1);
        }
        fprintf(ups->fd, "%d\n", (int)getpid());
        fclose(ups->fd);
}

