/*
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL TOM KNIENIEDER BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF TOM
 * KNIENIEDER HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * TOM KNIENIEDER SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND TOM KNIENIEDER HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */


#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <syslog.h>
#include <unistd.h>
#include <varargs.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/uio.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/ttycom.h>
#include <sys/ioctl.h>
#include <sys/termios.h>

#include "apc_upsd.h"

/* ------------------------------------------------------------ */

#define MAXLINE         100

#define UPS_ON_BATT     '!'
#define UPS_ON_LINE     '$'
#define BATT_LOW        '%'
#define BATT_OK         '+'

/* ------------------------------------------------------------ */

static  void    apc_smart_info();
static  int     apc_ask_smart();
static  void    apc_smart_test();

/* ------------------------------------------------------------ */


int     apc_smart_open  ()
{
        int     oldflag;

        /* open monitor device */

        if((ups->pd = open( ups->device, O_RDWR | O_NONBLOCK )) < 0)
        {
                fprintf(stderr, "open: %s %s\n", ups->device,  sys_errlist[errno]);
                return  0;
        }
        tcgetattr( ups->pd,&ups->oldtio); /* Save old settings */

        ups->newtio.c_cflag = CS8 | CRTSCTS | CREAD;

        ups->newtio.c_iflag = IGNBRK | IGNPAR;
        ups->newtio.c_oflag = 0; 
        ups->newtio.c_lflag = 0; 

        ups->newtio.c_cc[VMIN] = 1;
        ups->newtio.c_cc[VTIME] = 50;

        cfsetspeed(& ups->newtio , 2400);

        tcsetattr( ups->pd, TCSADRAIN, & ups->newtio);


        ioctl(ups->pd, TIOCSDTR, 0);
        ioctl(ups->pd, TIOCSCTTY, 0);


        oldflag = fcntl(ups->pd, F_GETFL, 0);
        fcntl(ups->pd, F_SETFL, oldflag & ~O_NONBLOCK);

        return  1;
}

/* ------------------------------------------------------------------ */


int getline(int fd, char *s)
{
        int     i       = 0;
        int     ending  = 0;
        char    c;

        while ( ! ending ) 
        {
                if ( i >= ( MAXLINE -1 ))
                {
                        syslog(LOG_INFO,"maximum answer width exceeded.");
                        break;
                }

                read(fd,&c,1);

                switch(c) 
                {
                case UPS_ON_BATT: 
                        syslog(LOG_INFO,"going on battery");
                        ups->battery =1;
                        break;
                
                case UPS_ON_LINE: 
                        syslog(LOG_INFO,"going on-line");
                        ups->battery =0;
                        break;

                case BATT_LOW: 
                        ups->battlow=1;
                        break;

                case BATT_OK: 
                        ups->battlow=0;
                        break;

                case '\n': 
                        ending=1;
                    break;

                default: 
                        s[i++]=c;
                        break;
                }
        }
        s[i]='\0';

        return 0;
}

/* ------------------------------------------------------------------ */

static  void apc_send_ups( int ch )
{
        int     q = ch;

        write( ups->pd, &q, 1 ); 
}

/* ------------------------------------------------------------------ */

static  void    apc_smart_test()
{
        char    answer[MAXLINE];

        apc_send_ups( 'Y' );getline( ups->pd, answer ); 
        apc_send_ups( 'A' );getline( ups->pd, answer ); 

        syslog( LOG_INFO, "test started.");
}

/* ------------------------------------------------------------------ */

static  int     apc_ask_smart()
{
        char    answer[MAXLINE];

        apc_send_ups( 'Y' );getline( ups->pd, answer ); 

        if ( ups->extended )
        {
                apc_send_ups( 'f' ); getline( ups->pd, answer ); 
                ups->BatLoad  = atof(answer);  

                apc_send_ups( 'N' ); getline( ups->pd, answer ); 
                ups->LineMin  = atof(answer);

                apc_send_ups( 'M' ); getline( ups->pd, answer ); 
                ups->LineMax  = atof(answer);

                apc_send_ups( 'P'); getline( ups->pd, answer ); 
                ups->UPSLoad  = atof(answer);

                apc_send_ups('F'); getline( ups->pd, answer ); 
                ups->LineFreq = atof(answer);

                apc_send_ups( 'L'); getline( ups->pd, answer ); 
                ups->LineVoltage = atof(answer);

                apc_send_ups( 'O'); getline( ups->pd, answer ); 
                ups->OutputVoltage = atof(answer);

                apc_send_ups( 'C' ); getline( ups->pd, answer ); 
                ups->UPSTemp  = atof(answer);

                apc_send_ups('B'); getline( ups->pd, answer ); 
                ups->BattVoltage = atof(answer);
        }

        apc_send_ups('Q'); getline( ups->pd, answer ); 
        ups->Status    = atoi(answer);

        /* write info to syslog */   

        ups->log_counter ++;
        if ( ups->log_counter >= ups->log_timer )
        {
                ups->log_counter = 0;
                apc_smart_info();
        }

        return  1;
}
                
/* ------------------------------------------------------------------ */


static void apc_smart_info()
{
        if ( ups->extended )
        {
                syslog( LOG_INFO, "battery load  : %6.2f %%",   ups->BatLoad       );
                syslog( LOG_INFO, "voltage       : %6.2f volt", ups->LineVoltage   );
                syslog( LOG_INFO, "line min.     : %6.2f volt", ups->LineMin       );
                syslog( LOG_INFO, "line max.     : %6.2f volt", ups->LineMax       );
                syslog( LOG_INFO, "load          : %6.2f %%",   ups->UPSLoad       );
                syslog( LOG_INFO, "line freq     : %6.2f HZ",   ups->LineFreq      );
                syslog( LOG_INFO, "output voltage: %6.2f volt", ups->OutputVoltage );
                syslog( LOG_INFO, "temperature   : %6.2f degrees celsius", 
                ups->UPSTemp       );
                syslog( LOG_INFO, "battery volt. : %6.2f volt", ups->BattVoltage   );
        }
        switch( ups->Status )
        {
        case    2:
                syslog( LOG_INFO, "status        : %d (IN SHUTDOWN STATE)", 
                        ups->Status );
                break;

        case    8:
                syslog( LOG_INFO, "status        : %d (ON-LINE)", 
                        ups->Status );
                break;

        case    10:
                syslog( LOG_INFO, "status        : %d (ON BATTERY)", 
                        ups->Status );
                break;

        case    18:
                syslog( LOG_INFO, "status        : %d (BOTH LEDS ON)", 
                        ups->Status );
                break;

         default:
                syslog( LOG_INFO, "status        : %d", 
                        ups->Status );
                break;
         }
}

/* ------------------------------------------------------------------ */

static void apc_debug_smart_info()
{
        if ( ups->extended )
        {
                printf( "battery load  : %6.2f %%\n",   ups->BatLoad       );
                printf( "voltage       : %6.2f volt\n", ups->LineVoltage   );
                printf( "line min.     : %6.2f volt\n", ups->LineMin       );
                printf( "line max.     : %6.2f volt\n", ups->LineMax       );
                printf( "load          : %6.2f %%\n",   ups->UPSLoad       );
                printf( "line freq     : %6.2f HZ\n",   ups->LineFreq      );
                printf( "output voltage: %6.2f volt\n", ups->OutputVoltage );
                printf( "temperature   : %6.2f degrees celsius\n", ups->UPSTemp       );
                printf( "battery volt. : %6.2f volt\n", ups->BattVoltage   );
        }
        printf( "status        : %d\n"    ,     ups->Status        );
}

/* ------------------------------------------------------------------ */

int     apc_smart_loop   ()
{
        /* monitor device */

        ups->log_counter = 
        ups->log_timer   = 3600; /* Once a hour, show an info */
        ups->pow         = 1;    /* We have power when we start */

        if ( ups->startuptest )
                ups->testflag = 1;

        if ( ups->debug )
        {
                apc_ask_smart();
                apc_debug_smart_info();
        }

        while(1) 
        {
                if ( ups->testflag )
                {
                        apc_smart_test();
                        ups->testflag = 0;
                        sleep(1);
                }

                apc_ask_smart();

                if ( ! ups->battery )
                {
                        /* power up or back */

                        if(!ups->pow)
                        {
                                syslog(LOG_CRIT, "POWER UP");
                                ups->pow = 1;
                                ups->seconds = 0;
                                apc_smart_info();
                        }
                }
                else 
                {
                        /* power down */

                        if(ups->pow)
                        {
                                syslog(LOG_CRIT, "POWER DOWN");
                                ups->pow = 0;
                                apc_smart_info();
                        }

                        ups->seconds ++;
                }
                
                if ( ups->seconds > ups->wait )
                {
                        upsdown();
                }
                sleep(1);
        }
}

/* ------------------------------------------------------------------ */

void    apc_smart_UPS_off()
{
        int     q;
        char    *str = "-p\n600\n";

        /* set timeout to max ... */

        write( ups->pd, str, 7 ); sleep(1);

        /* K pause K ... switch off after delay */

        apc_send_ups( 'K' ); write( ups->pd, &q, 1 ); sleep(2);
        apc_send_ups( 'K' ); write( ups->pd, &q, 1 ); 
}


