/*
 * word2text.c
 * Copyright (C) 1998,1999 A.J. van Os
 *
 * Description:
 * MS Word to text functions
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#if defined(__riscos)
#include "visdelay.h"
#endif /* __riscos */
#include "antiword.h"


#define INITIAL_SIZE		40
#define EXTENTION_SIZE		20


/* Macros to make sure all such statements will be identical */
#define OUTPUT_LINE()		\
	do {\
		vAlign2Window(pDiag, pAnchor, iWidthMax, ucAlignment);\
		pAnchor = pStartNewOutput(pAnchor, NULL);\
		pOutput = pAnchor;\
	} while(0)

#define RESET_LINE()		\
	do {\
		pAnchor = pStartNewOutput(pAnchor, NULL);\
		pOutput = pAnchor;\
	} while(0)

/* Length of the document in characters */
static unsigned int	uiDocumentLength;
/* Number of characters processed so far */
static unsigned int	uiCharCounter;
static int		iCurrPct, iPrevPct;
/* Special treatment for files from the Apple Macintosh */
static BOOL		bMacFile = FALSE;
/* Needed for reading a complete table row */
static row_block_type	tRowInfo;
static BOOL		bRowInfo = FALSE;
static BOOL	bStartRow = FALSE;
static BOOL	bEndRow = FALSE;
static BOOL	bIsTableRow = FALSE;
/* Needed for finding the start of a style */
static const style_block_type	*pStyleInfo = NULL;
static BOOL	bStartStyle = FALSE;
/* Needed for finding the start of a font */
static const font_block_type	*pFontInfo = NULL;
static BOOL	bStartFont = FALSE;


/*
 * vUpdateCounters - Update the counters for the hourglass
 */
static void
vUpdateCounters(void)
{
#if defined(__riscos)
	uiCharCounter++;
	iCurrPct = (int)((uiCharCounter * 100) / uiDocumentLength);
	if (iCurrPct != iPrevPct) {
		visdelay_percent(iCurrPct);
		iPrevPct = iCurrPct;
	}
#endif /* __riscos */
} /* end of vUpdateCounters */

/*
 * bOutputContainsText - see if the output contains more than white-space
 */
static BOOL
bOutputContainsText(output_type *pAnchor)
{
	output_type	*pTmp;
	int	iIndex;

	fail(pAnchor == NULL);

	for (pTmp = pAnchor; pTmp != NULL; pTmp = pTmp->pNext) {
		fail(pTmp->iStringWidth < 0);
		for (iIndex = 0; iIndex < pTmp->iNextFree; iIndex++) {
			if (isspace(pTmp->szStorage[iIndex])) {
				continue;
			}
#if defined(DEBUG)
			if ((unsigned char)pTmp->szStorage[iIndex] ==
			    FILLER_CHAR) {
				continue;
			}
#endif /* DEBUG */
			return TRUE;
		}
	}
	return FALSE;
} /* end of bOutputContainsText */

/*
 * iTotalStringWidth - compute the total width of the output string
 */
static int
iTotalStringWidth(output_type *pAnchor)
{
	output_type	*pTmp;
	int		iTotal;

	iTotal = 0;
	for (pTmp = pAnchor; pTmp != NULL; pTmp = pTmp->pNext) {
		fail(pTmp->iStringWidth < 0);
		iTotal += pTmp->iStringWidth;
	}
	return iTotal;
} /* end of iTotalStringWidth */

/*
 * vStoreCharacter - store one character
 */
static void
vStoreCharacter(int iChar, output_type *pOutput)
{
	fail(pOutput == NULL);

	if (iChar == 0) {
		pOutput->szStorage[pOutput->iNextFree] = '\0';
		return;
	}
	while (pOutput->iNextFree + 2 > pOutput->iStorageSize) {
		pOutput->iStorageSize += EXTENTION_SIZE;
		pOutput->szStorage = xrealloc(pOutput->szStorage,
					pOutput->iStorageSize);
	}
	pOutput->szStorage[pOutput->iNextFree] = (char)iChar;
	pOutput->szStorage[pOutput->iNextFree+1] = '\0';
	pOutput->iStringWidth += iComputeStringWidth(
				pOutput->szStorage + pOutput->iNextFree,
				1,
				pOutput->tFontRef,
				pOutput->ucFontsize);
	pOutput->iNextFree++;
} /* end of vStoreCharacter */

/*
 * vStoreString - store a string
 */
static void
vStoreString(const char *szString, int iStringLength, output_type *pOutput)
{
	int	iIndex;

	fail(szString == NULL || iStringLength < 0 || pOutput == NULL);

	for (iIndex = 0; iIndex < iStringLength; iIndex++) {
		vStoreCharacter(szString[iIndex], pOutput);
	}
} /* end of vStoreString */

/*
 * vStoreIntegerAsDecimal - store an integer as a decimal number
 */
static void
vStoreIntegerAsDecimal(int iNumber, output_type *pOutput)
{
	int	iLen;
	char	szString[15];

	fail(pOutput == NULL);

	iLen = sprintf(szString, "%d", iNumber);
	vStoreString(szString, iLen, pOutput);
} /* end of vStoreIntegerAsDecimal */

/*
 * vStoreIntegerAsRoman - store an integer as a roman numerical
 */
static void
vStoreIntegerAsRoman(int iNumber, output_type *pOutput)
{
	int	iLen;
	char	szString[15];

	fail(iNumber <= 0);
	fail(pOutput == NULL);

	iLen = iInteger2Roman(iNumber, FALSE, szString);
	vStoreString(szString, iLen, pOutput);
} /* end of vStoreIntegerAsRoman */

/*
 * vStoreStyle - store a style
 */
static void
vStoreStyle(const style_block_type *pStyleInfo, output_type *pOutput)
{
	int	iLen;
	char	szString[120];

	fail(pOutput == NULL);
	iLen = iStyle2Window(szString, pStyleInfo);
	vStoreString(szString, iLen, pOutput);
} /* end of vStoreStyle */

/*
 * vPutIndentation - output the given amount of indentation
 */
static void
vPutIndentation(diagram_type *pDiag, output_type *pOutput, BOOL bUnmarked,
	int iListNumber, unsigned char ucListType, char cListCharacter,
	int iLeftIndent)
{
	int	iLeftIndentation, iWidth, iNextFree;
	char	szLine[30];

	fail(pDiag == NULL || pOutput == NULL);
	fail(iListNumber < 0);
	fail(iLeftIndent < 0);

	if (iLeftIndent <= 0) {
		return;
	}
	iLeftIndentation = iTwips2MilliPoints(iLeftIndent);
	if (bUnmarked) {
		vSetLeftIndentation(pDiag, iLeftIndentation);
		return;
	}
	fail(iListNumber <= 0 || iscntrl(cListCharacter));

	switch (ucListType) {
	case LIST_BULLETS:
		iNextFree = 0;
		break;
	case LIST_ROMAN_NUM_UPPER:
	case LIST_ROMAN_NUM_LOWER:
		iNextFree = iInteger2Roman(iListNumber,
			ucListType == LIST_ROMAN_NUM_UPPER, szLine);
		break;
	case LIST_UPPER_ALPHA:
	case LIST_LOWER_ALPHA:
		iNextFree = iInteger2Alpha(iListNumber,
			ucListType == LIST_UPPER_ALPHA, szLine);
		break;
	case LIST_ARABIC_NUM:
	default:
		iNextFree = sprintf(szLine, "%d", iListNumber);
	}
	szLine[iNextFree++] = cListCharacter;
	szLine[iNextFree++] = ' ';
	szLine[iNextFree] = '\0';
	iWidth = iComputeStringWidth(szLine, iNextFree,
				pOutput->tFontRef, pOutput->ucFontsize);
	iLeftIndentation -= iWidth;
	if (iLeftIndentation > 0) {
		vSetLeftIndentation(pDiag, iLeftIndentation);
	}
	vStoreString(szLine, iNextFree, pOutput);
} /* end of vPutIndentation */

/*
 * vPutNoteSeparator - output a note separator
 *
 * A note separator is a horizontal line two inches long.
 * Two inches equals 144000 milliponts.
 */
static void
vPutNoteSeparator(output_type *pOutput)
{
	int	iCounter, iChars, iCharWidth;
	char	szOne[2];

	fail(pOutput == NULL);

	szOne[0] = OUR_EM_DASH;
	szOne[1] = '\0';
	iCharWidth = iComputeStringWidth(szOne, 1,
				pOutput->tFontRef, pOutput->ucFontsize);
	DBG_DEC(iCharWidth);
	iChars = (144000 + iCharWidth / 2) / iCharWidth;
	DBG_DEC(iChars);
	for (iCounter = 0; iCounter < iChars; iCounter++) {
		vStoreCharacter(OUR_EM_DASH, pOutput);
	}
} /* end of vPutNoteSeparator */

/*
 *
 */
static output_type *
pStartNextOutput(output_type *pCurrent)
{
	output_type	*pNew;

	if (pCurrent->iNextFree == 0) {
		/* The current record is empty, re-use */
		fail(pCurrent->szStorage[0] != '\0');
		fail(pCurrent->iStringWidth != 0);
		return pCurrent;
	}
	/* The current record is in use, make a new one */
	pNew = xmalloc(sizeof(*pNew));
	pCurrent->pNext = pNew;
	pNew->iStorageSize = INITIAL_SIZE;
	pNew->szStorage = xmalloc(pNew->iStorageSize);
	pNew->szStorage[0] = '\0';
	pNew->iNextFree = 0;
	pNew->iStringWidth = 0;
	pNew->iColour = 0;
	pNew->ucFontstyle = FONT_REGULAR;
	pNew->tFontRef = 0;
	pNew->ucFontsize = DEFAULT_FONT_SIZE;
	pNew->pPrev = pCurrent;
	pNew->pNext = NULL;
	return pNew;
} /* end of pStartNextOutput */

/*
 * pStartNewOutput
 */
static output_type *
pStartNewOutput(output_type *pAnchor, output_type *pLeftOver)
{
	output_type	*pCurr, *pNext;
	int		iColour;
	draw_fontref	tFontRef;
	unsigned char	ucFontstyle, ucFontsize;

	iColour = FONT_COLOUR_DEFAULT;
	ucFontstyle = FONT_REGULAR;
	tFontRef = 0;
	ucFontsize = DEFAULT_FONT_SIZE;
	/* Free the old output space */
	pCurr = pAnchor;
	while (pCurr != NULL) {
		pNext = pCurr->pNext;
		pCurr->szStorage = xfree(pCurr->szStorage);
		if (pCurr->pNext == NULL) {
			iColour = pCurr->iColour;
			ucFontstyle = pCurr->ucFontstyle;
			tFontRef = pCurr->tFontRef;
			ucFontsize = pCurr->ucFontsize;
		}
		pCurr = xfree(pCurr);
		pCurr = pNext;
	}
	if (pLeftOver == NULL) {
		/* Create new output space */
		pLeftOver = xmalloc(sizeof(*pLeftOver));
		pLeftOver->iStorageSize = INITIAL_SIZE;
		pLeftOver->szStorage = xmalloc(pLeftOver->iStorageSize);
		pLeftOver->szStorage[0] = '\0';
		pLeftOver->iNextFree = 0;
		pLeftOver->iStringWidth = 0;
		pLeftOver->iColour = iColour;
		pLeftOver->ucFontstyle = ucFontstyle;
		pLeftOver->tFontRef = tFontRef;
		pLeftOver->ucFontsize = ucFontsize;
		pLeftOver->pPrev = NULL;
		pLeftOver->pNext = NULL;
	}
	fail(!bCheckDoubleLinkedList(pLeftOver));
	return pLeftOver;
} /* end of pStartNewOutput */

/*
 * iGetChar - get the next character from the given list
 */
static int
iGetChar(FILE *pFile, list_id_enum eListID)
{
	const font_block_type *pCurr;
	int	iChar, iFileOffset;
	BOOL	bSkip;

	fail(pFile == NULL);

	pCurr = pFontInfo;
	bSkip = FALSE;
	for (;;) {
		iChar = iNextChar(pFile, eListID, &iFileOffset);
		if (iChar == EOF) {
			return EOF;
		}

		vUpdateCounters();

		if (!bStartRow) {
			bStartRow = bRowInfo &&
				iFileOffset == tRowInfo.iOffsetStart;
			NO_DBG_HEX_C(bStartRow, tRowInfo.iOffsetStart);
		}
		if (!bEndRow) {
			bEndRow = bRowInfo &&
				iFileOffset == tRowInfo.iOffsetEnd;
			NO_DBG_HEX_C(bStartRow, tRowInfo.iOffsetEnd);
		}
		if (!bStartStyle) {
			bStartStyle = pStyleInfo != NULL &&
				iFileOffset == pStyleInfo->iOffset;
			NO_DBG_HEX_C(bStartStyle, iFileOffset);
		}
		if (pCurr != NULL && iFileOffset == pCurr->iOffset) {
			bStartFont = TRUE;
			NO_DBG_HEX(iFileOffset);
			pFontInfo = pCurr;
			pCurr = pGetNextFontInfoListItem(pCurr);
		}

		/* Skip embedded characters */
		if (iChar == START_EMBEDDED) {
			bSkip = TRUE;
			continue;
		}
		if (iChar == END_IGNORE || iChar == END_EMBEDDED) {
			bSkip = FALSE;
			continue;
		}
		if (bSkip) {
			continue;
		}
		iChar = iTranslateCharacters(iChar, iFileOffset, bMacFile);
		if (iChar == EOF) {
			continue;
		}
		return iChar;
	}
} /* end of iGetChar */

/*
 * vWord2Text
 */
void
vWord2Text(diagram_type *pDiag, const char *szFilename)
{
	options_type	tOptions;
	FILE	*pFile;
	output_type	*pAnchor, *pOutput, *pLeftOver;
	list_id_enum 	eListID;
	int	iFontsize, iDefaultTabWidth;
	int	iFootnoteNumber, iEndnoteNumber, iLeftIndent;
	int	iChar, iListNumber, iWidthCurr, iWidthMax, iTmp;
	BOOL	bWasTableRow, bTableFontClosed, bUnmarked;
	BOOL	bAllCapitals, bHiddenText;
	unsigned char	ucFontnumber, ucFontcolour, ucTmp;
	unsigned char	ucFontstyle, ucFontstyleMinimal;
	unsigned char	ucListType, ucAlignment;
	char	cListChar;

	fail(pDiag == NULL || szFilename == NULL || szFilename[0] == '\0');

	DBG_MSG("vWord2Text");

	pFile = pOpenDocument(szFilename);
	if (pFile == NULL) {
		return;
	}
	vAddFonts2Diagram(pDiag);

	/* Initialisation */
	uiCharCounter = 0;
	iCurrPct = 0;
	iPrevPct = -1;
	uiDocumentLength = uiGetDocumentLength();
	bMacFile = bFileFromTheMac();
	iDefaultTabWidth = iGetDefaultTabWidth();
	DBG_DEC_C(iDefaultTabWidth != 36000, iDefaultTabWidth);
	bRowInfo = bGetNextRowInfoListItem(&tRowInfo);
	DBG_HEX_C(bRowInfo, tRowInfo.iOffsetStart);
	DBG_HEX_C(bRowInfo, tRowInfo.iOffsetEnd);
	bStartRow = FALSE;
	bEndRow = FALSE;
	bIsTableRow = FALSE;
	bWasTableRow = FALSE;
	vResetStyles();
	pStyleInfo = pGetNextStyleInfoListItem();
	bStartStyle = FALSE;
	pAnchor = NULL;
	pFontInfo = pGetNextFontInfoListItem(NULL);
	DBG_HEX_C(pFontInfo != NULL, pFontInfo->iOffset);
	DBG_MSG_C(pFontInfo == NULL, "No fonts at all");
	bStartFont = FALSE;
	ucFontnumber = 0;
	ucFontstyleMinimal = FONT_REGULAR;
	ucFontstyle = FONT_REGULAR;
	iFontsize = DEFAULT_FONT_SIZE;
	ucFontcolour = 0;
	pAnchor = pStartNewOutput(pAnchor, NULL);
	pOutput = pAnchor;
	pOutput->iColour = ucFontcolour;
	pOutput->ucFontstyle = ucFontstyle;
	pOutput->tFontRef =
		tOpenFont(pDiag, ucFontnumber, ucFontstyle, iFontsize);
	pOutput->ucFontsize = iFontsize;
	bTableFontClosed = TRUE;
	iLeftIndent = 0;
	bUnmarked = TRUE;
	ucListType = LIST_BULLETS;
	cListChar = OUR_BULLET;
	iListNumber = 0;
	ucAlignment = ALIGNMENT_LEFT;
	bAllCapitals = FALSE;
	bHiddenText = FALSE;
	vGetOptions(&tOptions);
	fail(tOptions.iParagraphBreak < 0);
	if (tOptions.iParagraphBreak == 0) {
		iWidthMax = INT_MAX;
	} else if (tOptions.iParagraphBreak < MIN_SCREEN_WIDTH) {
		iWidthMax = iChar2MilliPoints(MIN_SCREEN_WIDTH);
	} else if (tOptions.iParagraphBreak > MAX_SCREEN_WIDTH) {
		iWidthMax = iChar2MilliPoints(MAX_SCREEN_WIDTH);
	} else {
		iWidthMax = iChar2MilliPoints(tOptions.iParagraphBreak);
	}
	NO_DBG_DEC(iWidthMax);

	visdelay_begin();

	iFootnoteNumber = 0;
	iEndnoteNumber = 0;
	eListID = text_list;
	for(;;) {
		iChar = iGetChar(pFile, eListID);
		if (iChar == EOF) {
			if (bOutputContainsText(pAnchor)) {
				OUTPUT_LINE();
			} else {
				RESET_LINE();
			}
			switch (eListID) {
			case text_list:
				eListID = footnote_list;
				if (iFootnoteNumber > 0) {
					vPutNoteSeparator(pAnchor);
					OUTPUT_LINE();
					iFootnoteNumber = 0;
				}
				break;
			case footnote_list:
				eListID = endnote_list;
				if (iEndnoteNumber > 0) {
					vPutNoteSeparator(pAnchor);
					OUTPUT_LINE();
					iEndnoteNumber = 0;
				}
				break;
			case endnote_list:
			default:
				eListID = end_of_lists;
				break;
			}
			if (eListID == end_of_lists) {
				break;
			}
			continue;
		}

		if (iChar == UNKNOWN_NOTE_CHAR) {
			switch (eListID) {
			case footnote_list:
				iChar = FOOTNOTE_CHAR;
				break;
			case endnote_list:
				iChar = ENDNOTE_CHAR;
				break;
			default:
				break;
			}
		}

		if (bStartRow) {
			/* Begin of a tablerow found */
			if (bOutputContainsText(pAnchor)) {
				OUTPUT_LINE();
			} else {
				RESET_LINE();
			}
			fail(pAnchor != pOutput);
			if (bTableFontClosed) {
				/* Start special table font */
				vCloseFont();
				/*
				 * Compensate for the fact that Word uses
				 * proportional fonts for its tables and we
				 * only one fixed-width font
				 */
				pOutput->ucFontsize =
					iFontsize <= DEFAULT_FONT_SIZE ?
					(DEFAULT_FONT_SIZE * 5 + 3) / 6 :
					(iFontsize * 5 + 3) / 6;
				pOutput->tFontRef =
					tOpenTableFont(pDiag,
						pOutput->ucFontsize);
				pOutput->ucFontstyle = FONT_REGULAR;
				pOutput->iColour = FONT_COLOUR_BLACK;
				bTableFontClosed = FALSE;
			}
			bIsTableRow = TRUE;
			bStartRow = FALSE;
		}

		if (bWasTableRow &&
		    !bIsTableRow &&
		    iChar != PAR_END &&
		    iChar != HARD_RETURN &&
		    iChar != FORM_FEED &&
		    iChar != COLUMN_FEED) {
			/*
			 * The end of a table should be followed by an
			 * empty line, like the end of a paragraph
			 */
			OUTPUT_LINE();
			vEndOfParagraph2Diagram(pDiag,
						pOutput->tFontRef,
						pOutput->ucFontsize);
		}

		switch (iChar) {
		case FORM_FEED:
		case COLUMN_FEED:
			if (bIsTableRow) {
				vStoreCharacter('\n', pOutput);
				break;
			}
			if (bOutputContainsText(pAnchor)) {
				OUTPUT_LINE();
			} else {
				RESET_LINE();
			}
			if (iChar == FORM_FEED) {
				vEndOfPage2Diagram(pDiag,
						pOutput->tFontRef,
						pOutput->ucFontsize);
			} else {
				vEndOfParagraph2Diagram(pDiag,
						pOutput->tFontRef,
						pOutput->ucFontsize);
			}
			break;
		default:
			break;
		}

		if (bStartFont) {
			/* Begin of a font found */
			fail(pFontInfo == NULL);
			bAllCapitals = bIsCapitals(pFontInfo->ucFontstyle);
			bHiddenText = bIsHidden(pFontInfo->ucFontstyle);
			ucTmp = pFontInfo->ucFontstyle &
			(FONT_BOLD|FONT_ITALIC|FONT_UNDERLINE|FONT_STRIKE);
			if (!bIsTableRow &&
			    (iFontsize != pFontInfo->ucFontsize ||
			    ucFontnumber != pFontInfo->ucFontnumber ||
			    ucFontstyleMinimal != ucTmp ||
			    ucFontcolour != pFontInfo->ucFontcolour)) {
				pOutput = pStartNextOutput(pOutput);
				vCloseFont();
				pOutput->iColour = pFontInfo->ucFontcolour;
				pOutput->ucFontstyle = pFontInfo->ucFontstyle;
				pOutput->ucFontsize = pFontInfo->ucFontsize;
				pOutput->tFontRef = tOpenFont(
						pDiag,
						pFontInfo->ucFontnumber,
						pFontInfo->ucFontstyle,
						pFontInfo->ucFontsize);
				fail(!bCheckDoubleLinkedList(pAnchor));
			}
			ucFontnumber = pFontInfo->ucFontnumber;
			iFontsize = pFontInfo->ucFontsize;
			ucFontcolour = pFontInfo->ucFontcolour;
			ucFontstyle = pFontInfo->ucFontstyle;
			ucFontstyleMinimal = ucTmp;
			pFontInfo = pGetNextFontInfoListItem(pFontInfo);
			NO_DBG_HEX_C(pFontInfo != NULL, pFontInfo->iOffset);
			DBG_MSG_C(pFontInfo == NULL, "No more fonts");
			bStartFont = FALSE;
		}

		if (bStartStyle) {
			/* Begin of a style found */
			fail(pStyleInfo == NULL);
			if (!bIsTableRow) {
				vStoreStyle(pStyleInfo, pOutput);
			}
			iLeftIndent = pStyleInfo->sLeftIndent;
			bUnmarked = !pStyleInfo->bInList ||
						pStyleInfo->bUnmarked;
			ucListType = pStyleInfo->ucListType;
			cListChar = pStyleInfo->ucListCharacter;
			ucAlignment = pStyleInfo->ucAlignment;
			if (pStyleInfo->bInList) {
				if (!pStyleInfo->bUnmarked) {
					iListNumber++;
				}
			} else {
				iListNumber = 0;
			}
			pStyleInfo = pGetNextStyleInfoListItem();
			NO_DBG_HEX_C(pStyleInfo != NULL,pStyleInfo->iOffset);
			bStartStyle = FALSE;
		}

		if (!bIsTableRow &&
		    iTotalStringWidth(pAnchor) == 0) {
			vPutIndentation(pDiag, pAnchor, bUnmarked,
					iListNumber, ucListType, cListChar,
					iLeftIndent);
			/* One mark per paragraph will do */
			bUnmarked = TRUE;
		}

		switch (iChar) {
		case PICTURE:
			vStoreString("[pic]", 5, pOutput);
			break;
		case FOOTNOTE_CHAR:
			iFootnoteNumber++;
			vStoreCharacter('[', pOutput);
			vStoreIntegerAsDecimal(iFootnoteNumber, pOutput);
			vStoreCharacter(']', pOutput);
			break;
		case ENDNOTE_CHAR:
			iEndnoteNumber++;
			vStoreCharacter('[', pOutput);
			vStoreIntegerAsRoman(iEndnoteNumber, pOutput);
			vStoreCharacter(']', pOutput);
			break;
		case UNKNOWN_NOTE_CHAR:
			vStoreString("[?]", 3, pOutput);
			break;
		case PAR_END:
			if (bIsTableRow) {
				vStoreCharacter('\n', pOutput);
				break;
			}
			if (bOutputContainsText(pAnchor)) {
				OUTPUT_LINE();
			} else {
				RESET_LINE();
			}
			vEndOfParagraph2Diagram(pDiag,
						pOutput->tFontRef,
						pOutput->ucFontsize);
			/*
			 * End of paragraph seen, reset indentation,
			 * marking and alignment
			 */
			iLeftIndent = 0;
			bUnmarked = TRUE;
			ucAlignment = ALIGNMENT_LEFT;
			break;
		case HARD_RETURN:
			if (bIsTableRow) {
				vStoreCharacter('\n', pOutput);
				break;
			}
			if (bOutputContainsText(pAnchor)) {
				OUTPUT_LINE();
			}
			vEmptyLine2Diagram(pDiag,
					pOutput->tFontRef,
					pOutput->ucFontsize);
			break;
		case FORM_FEED:
		case COLUMN_FEED:
			/* Already dealt with */
			break;
		case TABLE_SEPARATOR:
			if (bIsTableRow) {
				vStoreCharacter(iChar, pOutput);
				break;
			}
			vStoreCharacter(' ', pOutput);
			vStoreCharacter(TABLE_SEPARATOR_CHAR, pOutput);
			break;
		case TAB:
			if (bIsTableRow) {
				vStoreCharacter(' ', pOutput);
				break;
			}
			if (tOptions.iParagraphBreak == 0 &&
			    !tOptions.bUseOutlineFonts) {
				/* No logical lines, so no tab expansion */
				vStoreCharacter(TAB, pOutput);
				break;
			}
			iTmp = iTotalStringWidth(pAnchor);
			iTmp += iDrawUnits2MilliPoints(pDiag->iXleft);
			iTmp /= iDefaultTabWidth;
			do {
				vStoreCharacter(FILLER_CHAR, pOutput);
				iWidthCurr = iTotalStringWidth(pAnchor);
				iWidthCurr +=
					iDrawUnits2MilliPoints(pDiag->iXleft);
			} while (iTmp == iWidthCurr / iDefaultTabWidth &&
				 iWidthCurr < iWidthMax);
			break;
		default:
			if (bHiddenText && tOptions.bHideHiddenText) {
				continue;
			}
			if (bAllCapitals && iChar < UCHAR_MAX) {
				iChar = iToUpper(iChar);
			}
			vStoreCharacter(iChar, pOutput);
			break;
		}

		if (bWasTableRow && !bIsTableRow) {
			/* End of a table, resume normal font */
			NO_DBG_MSG("End of table font");
			vCloseFont();
			bTableFontClosed = TRUE;
			pOutput->iColour = ucFontcolour;
			pOutput->ucFontstyle = ucFontstyle;
			pOutput->ucFontsize = iFontsize;
			pOutput->tFontRef = tOpenFont(pDiag, ucFontnumber,
						ucFontstyle, iFontsize);
		}
		bWasTableRow = bIsTableRow;

		if (bIsTableRow) {
			fail(pAnchor != pOutput);
			if (!bEndRow) {
				continue;
			}
			/* End of a table row */
			fail(!bRowInfo);
			vTableRow2Window(pDiag, pAnchor, &tRowInfo);
			/* Reset */
			pAnchor = pStartNewOutput(pAnchor, NULL);
			pOutput = pAnchor;
			bRowInfo = bGetNextRowInfoListItem(&tRowInfo);
			bIsTableRow = FALSE;
			bEndRow = FALSE;
			NO_DBG_HEX_C(bRowInfo, tRowInfo.iOffsetStart);
			NO_DBG_HEX_C(bRowInfo, tRowInfo.iOffsetEnd);
			continue;
		}
		iWidthCurr = iTotalStringWidth(pAnchor);
		iWidthCurr += iDrawUnits2MilliPoints(pDiag->iXleft);
		if (iWidthCurr < iWidthMax) {
			continue;
		}
		pLeftOver = pSplitList(pAnchor);
		vJustify2Window(pDiag, pAnchor, iWidthMax, ucAlignment);
		pAnchor = pStartNewOutput(pAnchor, pLeftOver);
		for (pOutput = pAnchor;
		     pOutput->pNext != NULL;
		     pOutput = pOutput->pNext)
			;	/* EMPTY */
		fail(pOutput == NULL);
		if (iTotalStringWidth(pAnchor) > 0) {
			vSetLeftIndentation(pDiag,
					iTwips2MilliPoints(iLeftIndent));
		}
	}

	pAnchor = pStartNewOutput(pAnchor, NULL);
	pAnchor->szStorage = xfree(pAnchor->szStorage);
	pAnchor = xfree(pAnchor);
	vCloseFont();
	vCloseDocument(pFile);
	visdelay_end();
} /* end of vWord2Text */
