/*
 * options.c
 * Copyright (C) 1998,1999 A.J. van Os
 *
 * Description:
 * Read and write the options
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#if defined(__riscos)
#include "wimpt.h"
#else
#include <stdlib.h>
#include <unistd.h>
#endif /* __riscos */
#include "antiword.h"

#define PARAGRAPH_BREAK		"set paragraph_break=%d"
#define AUTOFILETYPE		"set autofiletype_allowed=%d"
#define USE_OUTLINEFONTS	"set use_outlinefonts=%d"
#define HIDE_HIDDEN_TEXT	"set hide_hidden_text=%d"

/* Current values for options */
static options_type	tOptionsCurr;
#if defined(__riscos)
/* Temporary values for options */
static options_type	tOptionsTemp;
#else
static struct papersize_tag {
	char	szName[16];
	int	iWidth;		/* In points */
	int	iHeight;	/* In points */
} atPaperSizes[] = {
	{	"10x14",	 720,	1008	},
	{	"a3",		 842,	1191	},
	{	"a4",		 595,	 842	},
	{	"a5",		 420,	 595	},
	{	"b4",		 729,	1032	},
	{	"b5",		 516,	 729	},
	{	"executive",	 540,	 720	},
	{	"folio",	 612,	 936	},
	{	"legal",	 612,	1008	},
	{	"letter",	 612,	 792	},
	{	"note",		 540,	 720	},
	{	"quarto",	 610,	 780	},
	{	"statement",	 396,	 612	},
	{	"tabloid",	 792,	1224	},
	{	"",		  -1,	  -1	},
};
#endif /* __riscos */
/* Default values for options */
static const options_type	tOptionsDefault = {
	DEFAULT_SCREEN_WIDTH,
	TRUE,
#if defined(__riscos)
	TRUE,
#else
	FALSE,
#endif /* __riscos */
	TRUE,
	TRUE,
	INT_MAX,
};


/*
 * iReadOptions - read options
 *
 * returns:	-1: error
 *		 0: help
 *		>0: index first file argument
 */
int
iReadOptions(int argc, char **argv)
{
#if !defined(__riscos)
	extern	char	*optarg;
	extern int	optind, optopt;
	struct papersize_tag	*pPaperSize;
	char	*pcChar, *szTmp;
	int	iChar;
#endif /* __riscos */
	FILE	*pFile;
	int	iTmp;
	char	szLine[81];

	DBG_MSG("iReadOptions");

/* Defaults */
	tOptionsCurr = tOptionsDefault;

/* Environment */
#if !defined(__riscos)
	szTmp = getenv("COLUMNS");
	if (szTmp != NULL) {
		DBG_MSG(szTmp);
		iTmp = (int)strtol(szTmp, &pcChar, 10);
		if (pcChar == '\0') {
			iTmp -= 4;	/* This is the edge */
			if (iTmp < MIN_SCREEN_WIDTH) {
				iTmp = MIN_SCREEN_WIDTH;
			} else if (iTmp > MAX_SCREEN_WIDTH) {
				iTmp = MAX_SCREEN_WIDTH;
			}
			tOptionsCurr.iParagraphBreak = iTmp;
			DBG_DEC(tOptionsCurr.iParagraphBreak);
		}
	}
#endif /* __riscos */

/* Choices file */
	pFile = fopen(OPTIONS_FILE_R, "r");
	DBG_MSG_C(pFile == NULL, "Choices file not found");
	DBG_HEX_C(pFile != NULL, pFile);
	if (pFile != NULL) {
		while (fgets(szLine, sizeof(szLine), pFile) != NULL) {
			DBG_MSG(szLine);
			if (szLine[0] == OPTIONS_COMMENT_CHAR) {
				continue;
			}
			if (sscanf(szLine, PARAGRAPH_BREAK, &iTmp) == 1 &&
			    (iTmp == 0 ||
			    (iTmp >= MIN_SCREEN_WIDTH &&
			     iTmp <= MAX_SCREEN_WIDTH))) {
				tOptionsCurr.iParagraphBreak = iTmp;
				DBG_DEC(tOptionsCurr.iParagraphBreak);
			} else if (sscanf(szLine, AUTOFILETYPE, &iTmp)
								== 1) {
				tOptionsCurr.bAutofiletypeAllowed =
								iTmp != 0;
				DBG_DEC(tOptionsCurr.bAutofiletypeAllowed);
			} else if (sscanf(szLine, USE_OUTLINEFONTS, &iTmp)
								== 1) {
				tOptionsCurr.bUseOutlineFonts = iTmp != 0;
				DBG_DEC(tOptionsCurr.bUseOutlineFonts);
			} else if (sscanf(szLine, HIDE_HIDDEN_TEXT, &iTmp)
								== 1) {
				tOptionsCurr.bHideHiddenText = iTmp != 0;
				DBG_DEC(tOptionsCurr.bHideHiddenText);
			}
		}
		(void)fclose(pFile);
	}

/* Command line */
#if defined(__riscos)
	return 1;
#else
	while ((iChar = getopt(argc, argv, "cghp:stw:")) != -1) {
		switch (iChar) {
		case 'c':
			tOptionsCurr.bUseColour = TRUE;
			break;
		case 'g':
			tOptionsCurr.bUseColour = FALSE;
			break;
		case 'h':
			return 0;
		case 'p':
			tOptionsCurr.bUseOutlineFonts = TRUE;
			for (pPaperSize = atPaperSizes;
			     pPaperSize->szName[0] != '\0';
			     pPaperSize++) {
				if (StrEq(pPaperSize->szName,  optarg)) {
					tOptionsCurr.iPageHeight =
						iMilliPoints2DrawUnits(
						pPaperSize->iHeight * 1000);
					DBG_DEC(pPaperSize->iHeight);
					DBG_DEC(tOptionsCurr.iPageHeight);
					tOptionsCurr.iParagraphBreak =
						iMilliPoints2Char(
						pPaperSize->iWidth * 1000 -
						iDrawUnits2MilliPoints(
						PS_LEFT_MARGIN +
						PS_RIGHT_MARGIN));
					DBG_DEC(pPaperSize->iWidth);
					DBG_DEC(tOptionsCurr.iParagraphBreak);
				}
			}
		case 's':
			tOptionsCurr.bHideHiddenText = FALSE;
			break;
		case 't':
			tOptionsCurr.bUseOutlineFonts = FALSE;
			break;
		case 'w':
			iTmp = (int)strtol(optarg, &pcChar, 10);
			if (*pcChar == '\0' &&
			    (iTmp == 0 ||
			     (iTmp >= MIN_SCREEN_WIDTH &&
			      iTmp <= MAX_SCREEN_WIDTH))) {
				tOptionsCurr.iParagraphBreak = iTmp;
				DBG_DEC(tOptionsCurr.iParagraphBreak);
			}
			break;
		default:
			return -1;
		}
	}
	return optind;
#endif /* __riscos */
} /* end of iReadOptions */

/*
 * vGetOptions - get a copy of the current option values
 */
void
vGetOptions(options_type *ptOptions)
{
	fail(ptOptions == NULL);

	*ptOptions = tOptionsCurr;
} /* end of vGetOptions */

#if defined(__riscos)
/*
 * vWriteOptions - write the current options to the Options file
 */
static void
vWriteOptions(void)
{
	FILE	*pFile;
	char	*szOptionsFile;

	DBG_MSG("vWriteOptions");

	szOptionsFile = getenv(OPTIONS_FILE_W);
	if (szOptionsFile == NULL) {
		werr(0, "Warning: Name of the Choices file not found");
		return;
	}
	if (!bMakeDirectory(szOptionsFile)) {
		werr(0,
		"Warning: I can't make a directory for the Choices file");
		return;
	}
	pFile = fopen(szOptionsFile, "w");
	if (pFile == NULL) {
		werr(0, "Warning: I can't write the Choices file");
		return;
	}
	(void)fprintf(pFile, PARAGRAPH_BREAK"\n",
		tOptionsCurr.iParagraphBreak);
	(void)fprintf(pFile, AUTOFILETYPE"\n",
		tOptionsCurr.bAutofiletypeAllowed);
	(void)fprintf(pFile, USE_OUTLINEFONTS"\n",
		tOptionsCurr.bUseOutlineFonts);
	(void)fprintf(pFile, HIDE_HIDDEN_TEXT"\n",
		tOptionsCurr.bHideHiddenText);
	(void)fclose(pFile);
} /* end of vWriteOptions */

/*
 * vChoicesOpenAction - action to be taken when the Choices window opens
 */
void
vChoicesOpenAction(wimp_w tWindow)
{
	DBG_MSG("vChoicesOpenAction");

	tOptionsTemp = tOptionsCurr;
	if (tOptionsTemp.iParagraphBreak == 0) {
		vUpdateRadioButton(tWindow, CHOICES_BREAK_BUTTON, FALSE);
		vUpdateRadioButton(tWindow, CHOICES_NO_BREAK_BUTTON, TRUE);
		vUpdateWriteableNumber(tWindow, CHOICES_BREAK_WRITEABLE,
					DEFAULT_SCREEN_WIDTH);
	} else {
		vUpdateRadioButton(tWindow, CHOICES_BREAK_BUTTON, TRUE);
		vUpdateRadioButton(tWindow, CHOICES_NO_BREAK_BUTTON, FALSE);
		vUpdateWriteableNumber(tWindow,
			CHOICES_BREAK_WRITEABLE,
			tOptionsTemp.iParagraphBreak);
	}
	vUpdateRadioButton(tWindow, CHOICES_AUTOFILETYPE_BUTTON,
					tOptionsTemp.bAutofiletypeAllowed);
	vUpdateRadioButton(tWindow, CHOICES_HIDDEN_TEXT_BUTTON,
					tOptionsTemp.bHideHiddenText);
	if (tOptionsTemp.bUseOutlineFonts) {
		vUpdateRadioButton(tWindow,
			CHOICES_OUTLINEFONT_BUTTON, TRUE);
		vUpdateRadioButton(tWindow,
			CHOICES_SYSTEMFONT_BUTTON, FALSE);
	} else {
		vUpdateRadioButton(tWindow,
			CHOICES_OUTLINEFONT_BUTTON, FALSE);
		vUpdateRadioButton(tWindow,
			CHOICES_SYSTEMFONT_BUTTON, TRUE);
	}
} /* end of vChoicesOpenAction */

/*
 * vDefaultButtonAction - action when the default button is clicked
 */
static void
vDefaultButtonAction(wimp_w tWindow)
{
	DBG_MSG("vDefaultButtonAction");

	tOptionsTemp = tOptionsDefault;
	vUpdateRadioButton(tWindow, CHOICES_BREAK_BUTTON, TRUE);
	vUpdateRadioButton(tWindow, CHOICES_NO_BREAK_BUTTON, FALSE);
	vUpdateWriteableNumber(tWindow, CHOICES_BREAK_WRITEABLE,
				tOptionsTemp.iParagraphBreak);
	vUpdateRadioButton(tWindow, CHOICES_AUTOFILETYPE_BUTTON,
				tOptionsTemp.bAutofiletypeAllowed);
	vUpdateRadioButton(tWindow, CHOICES_HIDDEN_TEXT_BUTTON,
				tOptionsTemp.bHideHiddenText);
	vUpdateRadioButton(tWindow, CHOICES_OUTLINEFONT_BUTTON, TRUE);
	vUpdateRadioButton(tWindow, CHOICES_SYSTEMFONT_BUTTON, FALSE);
} /* end of vDefaultButtonAction */

/*
 * vApplyButtonAction - action to be taken when the OK button is clicked
 */
static void
vApplyButtonAction(void)
{
	DBG_MSG("vApplyButtonAction");

	tOptionsCurr = tOptionsTemp;
} /* end of vApplyButtonAction */

/*
 * vSaveButtonAction - action to be taken when the save button is clicked
 */
static void
vSaveButtonAction(void)
{
	DBG_MSG("vSaveButtonAction");

	vApplyButtonAction();
	vWriteOptions();
} /* end of vSaveButtonAction */

/*
 * vSetParagraphBreak - set the paragraph break to the given number
 */
static void
vSetParagraphBreak(wimp_w tWindow, int iNumber)
{
	tOptionsTemp.iParagraphBreak = iNumber;
	if (tOptionsTemp.iParagraphBreak == 0) {
		return;
	}
	vUpdateWriteableNumber(tWindow,
			CHOICES_BREAK_WRITEABLE,
			tOptionsTemp.iParagraphBreak);
} /* end of vSetParagraphBreak */

/*
 * vChangeParagraphBreak - change the paragraph break with the given number
 */
static void
vChangeParagraphBreak(wimp_w tWindow, int iNumber)
{
	int	iTmp;

	iTmp = tOptionsTemp.iParagraphBreak + iNumber;
	if (iTmp < MIN_SCREEN_WIDTH || iTmp > MAX_SCREEN_WIDTH) {
		return;
	}
	tOptionsTemp.iParagraphBreak = iTmp;
	vUpdateWriteableNumber(tWindow,
			CHOICES_BREAK_WRITEABLE,
			tOptionsTemp.iParagraphBreak);
} /* end of vChangeParagraphBreak */

/*
 * vChangeAutofiletype - invert the permission to autofiletype
 */
static void
vChangeAutofiletype(wimp_w tWindow)
{
	tOptionsTemp.bAutofiletypeAllowed =
				!tOptionsTemp.bAutofiletypeAllowed;
	vUpdateRadioButton(tWindow,
			CHOICES_AUTOFILETYPE_BUTTON,
			tOptionsTemp.bAutofiletypeAllowed);
} /* end of vChangeAutofiletype */

/*
 * vChangeHiddenText - invert the hide/show hidden text
 */
static void
vChangeHiddenText(wimp_w tWindow)
{
	tOptionsTemp.bHideHiddenText = !tOptionsTemp.bHideHiddenText;
	vUpdateRadioButton(tWindow,
			CHOICES_HIDDEN_TEXT_BUTTON,
			tOptionsTemp.bHideHiddenText);
} /* end of vChangeHiddenText */

/*
 * vUseOutlineFonts - use outline fonts
 */
static void
vUseOutlineFonts(BOOL bUseOutlineFonts)
{
	tOptionsTemp.bUseOutlineFonts = bUseOutlineFonts;
} /* end of vUseOutlineFonts */

/*
 * bChoicesMouseClick - handle a mouse click in the Choices window
 */
void
vChoicesMouseClick(wimp_mousestr *m)
{
	wimp_i	tAction;
	BOOL	bCloseWindow, bLeft, bRight;

	bLeft = (m->bbits & wimp_BLEFT) == wimp_BLEFT;
	bRight = (m->bbits & wimp_BRIGHT) == wimp_BRIGHT;
	if (!bLeft && !bRight) {
		DBG_HEX(m->bbits);
		return;
	}

	/* Which action should be taken */
	if (bRight && m->i == CHOICES_BREAK_UP_BUTTON) {
		tAction = CHOICES_BREAK_DOWN_BUTTON;
	} else if (bRight && m->i == CHOICES_BREAK_DOWN_BUTTON) {
		tAction = CHOICES_BREAK_UP_BUTTON;
	} else {
		tAction = m->i;
	}

	/* Actions */
	bCloseWindow = FALSE;
	switch (tAction) {
	case CHOICES_DEFAULT_BUTTON:
		vDefaultButtonAction(m->w);
		break;
	case CHOICES_SAVE_BUTTON:
		vSaveButtonAction();
		break;
	case CHOICES_CANCEL_BUTTON:
		bCloseWindow = TRUE;
		break;
	case CHOICES_APPLY_BUTTON:
		vApplyButtonAction();
		bCloseWindow = TRUE;
		break;
	case CHOICES_BREAK_BUTTON:
		vSetParagraphBreak(m->w, DEFAULT_SCREEN_WIDTH);
		break;
	case CHOICES_BREAK_UP_BUTTON:
		vChangeParagraphBreak(m->w, 1);
		break;
	case CHOICES_BREAK_DOWN_BUTTON:
		vChangeParagraphBreak(m->w, -1);
		break;
	case CHOICES_NO_BREAK_BUTTON:
		vSetParagraphBreak(m->w, 0);
		break;
	case CHOICES_AUTOFILETYPE_BUTTON:
		vChangeAutofiletype(m->w);
		break;
	case CHOICES_HIDDEN_TEXT_BUTTON:
		vChangeHiddenText(m->w);
		break;
	case CHOICES_OUTLINEFONT_BUTTON:
		vUseOutlineFonts(TRUE);
		break;
	case CHOICES_SYSTEMFONT_BUTTON:
		vUseOutlineFonts(FALSE);
		break;
	default:
		DBG_DEC(m->i);
		break;
	}
	if (bCloseWindow) {
		wimpt_noerr(wimp_close_wind(m->w));
	}
} /* end of vChoicesMouseClick */

void
vChoicesKeyPressed(wimp_caretstr *c)
{
	wimp_icon	tIcon;
	char		*pcChar;
	int		iNumber;

	DBG_MSG("vChoicesKeyPressed");

	fail(c == NULL);

	wimpt_noerr(wimp_get_icon_info(c->w, c->i, &tIcon));
	if ((tIcon.flags & (wimp_ITEXT|wimp_INDIRECT)) !=
	    (wimp_ITEXT|wimp_INDIRECT)) {
		werr(1, "Icon %d must be indirected text", (int)c->i);
		return;
	}
	iNumber = (int)strtol(tIcon.data.indirecttext.buffer, &pcChar, 10);
	if (*pcChar != '\0') {
		DBG_DEC(*pcChar);
		iNumber = DEFAULT_SCREEN_WIDTH;
	} else if (iNumber < MIN_SCREEN_WIDTH) {
		iNumber = MIN_SCREEN_WIDTH;
	} else if (iNumber > MAX_SCREEN_WIDTH) {
		iNumber = MAX_SCREEN_WIDTH;
	}
	vSetParagraphBreak(c->w, iNumber);
} /* end of vChoicesKeyPressed */
#endif /* __riscos */
