/* $Id: aguri_ip.c,v 1.7 2003/03/12 14:09:32 kjc Exp kjc $ */
/*
 * Copyright (C) 2001-2003 WIDE Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <sys/param.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#ifdef INET6
#include <netinet/ip6.h>
#include <netinet/icmp6.h>
#endif
#include <arpa/inet.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <err.h>

#include "aguri.h"
#include "aguri_tree.h"
#include "aguri_ip.h"
#include "aguri_pcap.h"
#include "aguri_plot.h"
#ifdef XTREE
#include "xtree.h"
#endif

int xflags = CHECK_SRCADDR | CHECK_DSTADDR | CHECK_SRCPROTO | CHECK_DSTPROTO |
	     CHECK_IPV4 | CHECK_IPV6;
int ip_thresh = 10;	/* 10/1000 (1%) */

struct tree *addr_src  = NULL;
struct tree *addr6_src = NULL;
struct tree *addr_dst  = NULL;
struct tree *addr6_dst = NULL;
struct tree *proto_src = NULL;
struct tree *proto_dst = NULL;

#define IP_TREES	6	/* number of trees */

static struct tree trees[IP_TREES];
static struct tree null_tree;

static void ip_nodename(struct tree_node *np, char *buf, size_t len);
static u_int64_t ip_addrparser(char *buf, void *key,
			       size_t *prefixlen, void *arg);
static u_int64_t ip_protoparser(char *buf, void *key,
				size_t *prefixlen, void *arg);
static void ip_counter(struct tree *tp, void *key, size_t prefixlen,
		       void *arg, u_int64_t cnt);
static void ipplot_counter(struct tree *tp, void *key, size_t prefixlen,
			   void *arg, u_int64_t cnt);
static void ipplot_printnames(struct tree *tp);
static u_int64_t null_parser(char *buf, void *key, size_t *prefixlen,
			     void *arg);
static void null_counter(struct tree *tp, void *key, size_t prefixlen,
			 void *arg, u_int64_t cnt);


static void
ip_nodename(struct tree_node *np, char *buf, size_t len)
{
	struct tree *tp;
	char prefixstr[8];

	tp = np->tn_tree;
	if (tp == addr_src || tp == addr_dst) {
		inet_ntop(AF_INET, np->tn_key, buf, len);
		if (np->tn_prefixlen < 32) {
			snprintf(prefixstr, sizeof(prefixstr), "/%u",
				 (u_int)np->tn_prefixlen);
			strncat(buf, prefixstr, len - strlen(buf));
		}
	} else if (tp == addr6_src || tp == addr6_dst) {
		inet_ntop(AF_INET6, np->tn_key, buf, len);
		if (np->tn_prefixlen < 128) {
			snprintf(prefixstr, sizeof(prefixstr), "/%u",
				 (u_int)np->tn_prefixlen);
			strncat(buf, prefixstr, len - strlen(buf));
		}
	} else if (tp == proto_src || tp == proto_dst) {
		if (np->tn_prefixlen < 8) {
			snprintf(buf, len, "%u/%d:%u:%u",
				 np->tn_key[0],
				 (int)np->tn_prefixlen,
				 np->tn_key[1],
				 ntohs(*(u_short *)&np->tn_key[2]));
		} else if (np->tn_prefixlen < 16) {
			snprintf(buf, len, "%u:%u/%d:%u",
				 np->tn_key[0],
				 np->tn_key[1],
				 (int)np->tn_prefixlen - 8,
				 ntohs(*(u_short *)&np->tn_key[2]));
		} else {
			snprintf(buf, len, "%u:%u:%u",
				 np->tn_key[0], np->tn_key[1],
				 ntohs(*(u_short *)&np->tn_key[2]));
			if (np->tn_prefixlen < 16+16) {
				snprintf(prefixstr, sizeof(prefixstr), "/%d",
					 (int)np->tn_prefixlen - 16);
				strncat(buf, prefixstr, len - strlen(buf));
			}
		}
	} else
		buf[0] = '\0';
}

static int
print_ipnode(struct tree_node *np, void *arg)
{
	char buf[INET_ADDRSTRLEN + 8];
	u_int64_t cnt, *total;
	int i;

	cnt = np->tn_count;
	if (cnt == 0)
		return (0);
	total = arg;
	ip_nodename(np, buf, sizeof(buf));
	for (i=0; i<(int)np->tn_prefixlen; i+=2)
		printf(" ");
	printf("%s\t%llu (%.2f%%", buf, (ull)cnt,
	       (double)cnt/(*total)*100);
	if (np->tn_prefixlen != np->tn_tree->tr_keylen)
		printf("/%.2f%%",
		       (double)subtree_sum(np)/(*total)*100);
	printf(")\n");
	return (0);
}

static int
print_ip6node(struct tree_node *np, void *arg)
{
	char buf[INET6_ADDRSTRLEN + 8];
	u_int64_t *total;
	int i;

	if (np->tn_count == 0)
		return (0);
	total = arg;
	ip_nodename(np, buf, sizeof(buf));
	for (i=0; i<(int)np->tn_prefixlen; i+=8)
		printf(" ");
	printf("%s\t%llu (%.2f%%",
	       buf, (ull)np->tn_count,
	       (double)np->tn_count/(*total)*100);
	if (np->tn_prefixlen != np->tn_tree->tr_keylen)
		printf("/%.2f%%",
		       (double)subtree_sum(np)/(*total)*100);
	printf(")\n");
	return (0);
}

static int
print_protonode(struct tree_node *np, void *arg)
{
	char buf[INET6_ADDRSTRLEN + 8];
	u_int64_t *total;
	int i;

	if (np->tn_count == 0)
		return (0);
	total = arg;
	ip_nodename(np, buf, sizeof(buf));
	if (np->tn_prefixlen >= 16)
		for (i=0; i<(int)np->tn_prefixlen - 16; i++)
			printf(" ");
	printf("%s\t%llu (%.2f%%",
	       buf, (ull)np->tn_count,
	       (double)np->tn_count/(*total)*100);
	if (np->tn_prefixlen != np->tn_tree->tr_keylen)
		printf("/%.2f%%",
		       (double)subtree_sum(np)/(*total)*100);
	printf(")\n");
	return (0);
}

int
ipinfo_init(void)
{
	if (xflags & CHECK_SRCADDR) {
		if (xflags & CHECK_IPV4)
			addr_src  = &trees[0];
		if (xflags & CHECK_IPV6)
			addr6_src = &trees[1];
	}
	if (xflags & CHECK_DSTADDR) {
		if (xflags & CHECK_IPV4)
			addr_dst  = &trees[2];
		if (xflags & CHECK_IPV6)
			addr6_dst = &trees[3];
	}
	if (xflags & CHECK_SRCPROTO)
		proto_src = &trees[4];
	if (xflags & CHECK_DSTPROTO)
		proto_dst = &trees[5];

	if (addr_src != NULL)
		tree_init(addr_src, 32, lru_size);
	if (addr6_src != NULL)
		tree_init(addr6_src, 128, lru_size);
	if (addr_dst != NULL)
		tree_init(addr_dst, 32, lru_size);
	if (addr6_dst != NULL)
		tree_init(addr6_dst, 128, lru_size);
	if (proto_src != NULL)
		tree_init(proto_src, 8+8+16, lru_size);
	if (proto_dst != NULL)
		tree_init(proto_dst, 8+8+16, lru_size);

#ifdef XTREE
	if (xtree) {
		if (addr_src != NULL)
			xtree_init(addr_src, "src addr", xtree_geometry);
		if (addr6_src != NULL)
			xtree_init(addr6_src, "src addr ipv6", xtree_geometry);
		if (addr_dst != NULL)
			xtree_init(addr_dst, "dst addr", xtree_geometry);
		if (addr6_dst != NULL)
			xtree_init(addr6_dst, "dst addr ipv6", xtree_geometry);
		if (proto_src != NULL)
			xtree_init(proto_src, "src proto", xtree_geometry);
		if (proto_dst != NULL)
			xtree_init(proto_dst, "dst proto", xtree_geometry);

		xtree_pause();
	}
#endif

	return (0);
}

int
print_summary(void)
{
	struct tree *tp;
	u_int64_t thresh, total;
	time_t t;
	double sec, avg;
	char buf[128];

#ifdef XTREE
	if (xtree) {
		xtree_pause();

		for (tp = &trees[0]; tp < &trees[IP_TREES]; tp++)
			if (tp->tr_top != NULL)
				xtree_erase(tp->tr_top);
	}
#endif

	printf("%s\n", fmt_string);
	t = start_time.tv_sec;
	if (t > 0) {
		strftime(buf, sizeof(buf), "%a %b %d %T %Y", localtime(&t));
		printf("%%%%StartTime: %s ", buf);
		strftime(buf, sizeof(buf), "%Y/%m/%d %T", localtime(&t));
		printf("(%s)\n", buf);
	}
	t = end_time.tv_sec;
	if (t > 0) {
		strftime(buf, sizeof(buf), "%a %b %d %T %Y", localtime(&t));
		printf("%%%%EndTime:   %s ", buf);
		strftime(buf, sizeof(buf), "%Y/%m/%d %T", localtime(&t));
		printf("(%s)\n", buf);
	}

	total = 0;
	if (addr_src != NULL || addr6_src != NULL) {
		if (addr_src != NULL)
			total += addr_src->tr_count;
		if (addr6_src != NULL)
			total += addr6_src->tr_count;
	} else if (addr_dst != NULL || addr6_dst != NULL) {
		if (addr_dst != NULL)
			total += addr_dst->tr_count;
		if (addr6_dst != NULL)
			total += addr6_dst->tr_count;
	} else if (proto_src != NULL)
		total = proto_src->tr_count;
	else if (proto_dst != NULL)
		total = proto_dst->tr_count;

	sec = (double)end_time.tv_sec - start_time.tv_sec
	    + (end_time.tv_usec - start_time.tv_usec) / 1000000;
	if (sec != 0.0) {
		avg = (double)total * 8 / sec;
		if (avg > 1000000.0)
			printf("%%AvgRate: %.2fMbps\n", avg/1000000.0);
		else
			printf("%%AvgRate: %.2fKbps\n", avg/1000.0);
	}

	print_pcapstat(stdout);
	printf("\n");

	if (addr_src != NULL || addr6_src != NULL) {
		total = 0;
		if (addr_src != NULL)
			total += addr_src->tr_count;
		if (addr6_src != NULL)
			total += addr6_src->tr_count;
		thresh = total * ip_thresh / 1000;

		if (addr_src != NULL)
			tree_aggregate(addr_src, thresh);
		if (addr6_src != NULL)
			tree_aggregate(addr6_src, thresh);

		printf("[src address] %llu (%.2f%%)\n", (ull)total, 100.0);
		if (addr_src != NULL)
			tree_walk(addr_src, print_ipnode, &total);
		if (addr6_src != NULL)
			tree_walk(addr6_src, print_ip6node, &total);
#ifdef AGURI_STATS
		{
			u_int all = 0, hits = 0, reclaimed = 0;
			if (addr_src != NULL) {
				all += addr_src->tr_stats.total;
				hits += addr_src->tr_stats.hits;
				reclaimed += addr_src->tr_stats.reclaimed;
			}
			if (addr6_src != NULL) {
				all += addr6_src->tr_stats.total;
				hits += addr6_src->tr_stats.hits;
				reclaimed += addr6_src->tr_stats.reclaimed;
			}
			if (all > 0)
				printf("%%LRU hits: %.2f%% (%u/%u)  "
				    "reclaimed: %u\n",
				    (double)hits / all * 100, hits, all,
				    reclaimed);
		}
#endif
	}
	if (addr_dst != NULL || addr6_dst != NULL) {
		total = 0;
		if (addr_dst != NULL)
			total += addr_dst->tr_count;
		if (addr6_dst != NULL)
			total += addr6_dst->tr_count;
		thresh = total * ip_thresh / 1000;

		if (addr_dst != NULL)
			tree_aggregate(addr_dst, thresh);
		if (addr6_dst != NULL)
			tree_aggregate(addr6_dst, thresh);

		printf("[dst address] %llu (%.2f%%)\n", (ull)total, 100.0);
		if (addr_dst != NULL)
			tree_walk(addr_dst, print_ipnode, &total);
		if (addr6_dst != NULL)
			tree_walk(addr6_dst, print_ip6node, &total);
#ifdef AGURI_STATS
		{
			u_int all = 0, hits = 0, reclaimed = 0;
			if (addr_dst != NULL) {
				all += addr_dst->tr_stats.total;
				hits += addr_dst->tr_stats.hits;
				reclaimed += addr_dst->tr_stats.reclaimed;
			}
			if (addr6_dst != NULL) {
				all += addr6_dst->tr_stats.total;
				hits += addr6_dst->tr_stats.hits;
				reclaimed += addr6_dst->tr_stats.reclaimed;
			}
			if (all > 0)
				printf("%%LRU hits: %.2f%% (%u/%u)  "
				    "reclaimed: %u\n",
				    (double)hits / all * 100, hits, all,
				    reclaimed);
		}
#endif
	}
	if (proto_src != NULL) {
		total = proto_src->tr_count;
		thresh = total * ip_thresh / 1000;

		tree_aggregate(proto_src, thresh);
		printf("[ip:proto:srcport] %llu (%.2f%%)\n",
		       (ull)total, 100.0);
		tree_walk(proto_src, print_protonode, &total);
#ifdef AGURI_STATS
		tp = proto_src;
		if (tp->tr_stats.total > 0)
			printf("%%LRU hits: %.2f%% (%u/%u)  "
			    "reclaimed: %u\n",
			    (double)tp->tr_stats.hits /
			    tp->tr_stats.total * 100,
			    tp->tr_stats.hits, tp->tr_stats.total,
			    tp->tr_stats.reclaimed);
#endif
	}
	if (proto_dst != NULL) {
		total = proto_dst->tr_count;
		thresh = total * ip_thresh / 1000;

		tree_aggregate(proto_dst, thresh);
		printf("[ip:proto:dstport] %llu (%.2f%%)\n",
		       (ull)total, 100.0);
		tree_walk(proto_dst, print_protonode, &total);
#ifdef AGURI_STATS
		tp = proto_dst;
		if (tp->tr_stats.total > 0)
			printf("%%LRU hits: %.2f%% (%u/%u)  "
			    "reclaimed: %u\n",
			    (double)tp->tr_stats.hits /
			    tp->tr_stats.total * 100,
			    tp->tr_stats.hits, tp->tr_stats.total,
			    tp->tr_stats.reclaimed);
#endif
	}
	printf("\n");
	fflush(stdout);
	return (0);
}

int
ipinfo_finish(void)
{
	struct tree *tp;

	if (interval == 0)
		print_summary();

#ifdef XTREE
	if (xtree) {
		xtree_pause();

		for (tp = &trees[0]; tp < &trees[IP_TREES]; tp++)
			if (tp->tr_top != NULL)
				xtree_destroy(tp);
	}
#endif

	for (tp = &trees[0]; tp < &trees[IP_TREES]; tp++)
		if (tp->tr_top != NULL)
			tree_destroy(tp);
	return (0);
}

struct tree *
ip_parsetype(char *buf, size_t len,
	     u_int64_t (**parser)(char *, void *, size_t *, void *),
	     void (**counter)(struct tree *, void *, size_t, void *, u_int64_t))
{
	struct tree *tp;

	if (strncmp("[src address]", buf, 13) == 0) {
		tp = addr_src;
		*parser = ip_addrparser;
	} else if (strncmp("[dst address]", buf, 13) == 0) {
		tp = addr_dst;
		*parser = ip_addrparser;
	} else if (strncmp("[ip:proto:srcport]", buf, 18) == 0) {
		tp = proto_src;
		*parser = ip_protoparser;
	} else if (strncmp("[ip:proto:dstport]", buf, 18) == 0) {
		tp = proto_dst;
		*parser = ip_protoparser;
	} else
		return (NULL);

	*counter = ip_counter;

	if (tp == NULL) {
		/* ignore lines for this type */
		tp = &null_tree;
		*parser = null_parser;
		*counter = null_counter;
	}

	return (tp);
}

static u_int64_t
ip_addrparser(char *buf, void *key, size_t *prefixlen, void *arg)
{
	int af;
	char *cp, *ap;
	size_t len;
	u_int64_t cnt;

	cp = buf;
	/* skip leading white space */
	while (isspace(*cp))
		cp++;
	if (*cp == '\0' || *cp == '%')
		return (0);
	ap = cp;

	if ((cp = strchr(ap, '/')) != NULL) {
		*cp++ = '\0';
		len = strtol(cp, NULL, 0);
	} else {
		cp = ap;
		len = 128;
	}
	cp = strpbrk(cp, " \t");  /* move to the next white space char */
	if (cp == NULL)
		return (0);
	*cp++ = '\0';
	while (isspace(*cp))	/* skip white space */
		cp++;
	cnt = strtouq(cp, NULL, 0);

	if (strchr(ap, ':') != NULL)
		af = AF_INET6;
	else if (strchr(ap, '.') != NULL) {
		af = AF_INET;
		if (len == 128)
			len = 32;
	} else {
		warnx("not ip: %s", buf);
		return (0);
	}

	if (inet_pton(af, ap, key) != 1) {
		warnx("inet_pton: %s", buf);
		return (0);
	}
	*prefixlen = len;
	if (arg != NULL) {
		if (af == AF_INET6)
			*(int *)arg = 1;
		else
			*(int *)arg = 0;
	}
	return (cnt);
}

static u_int64_t
ip_protoparser(char *buf, void *key, size_t *prefixlen, void *arg)
{
	struct proto *pdata;
	char *cp, *ap, *pp;
	size_t len;
	u_int64_t cnt;

	pdata = key;
	len = 999;
	cp = buf;
	/* skip leading white space */
	while (isspace(*cp))
		cp++;
	if (*cp == '\0' || *cp == '%')
		return(0);

	/* ip version */
	ap = cp;
	if ((cp = strchr(ap, ':')) == NULL)
		goto bad;
	*cp++ = '\0';
	if ((pp = strchr(ap, '/')) != NULL) {
		*pp++ = '\0'; 
		len = strtoul(pp, NULL, 0);
	}
	pdata->p_ipver = (u_char)strtoul(ap, NULL, 0);

	/* proto */
	while (isspace(*cp))	/* skip white space */
		cp++;
	ap = cp;
	if ((cp = strchr(ap, ':')) == NULL)
		goto bad;
	*cp++ = '\0';
	if ((pp = strchr(ap, '/')) != NULL) {
		*pp++ = '\0'; 
		len = 8 + strtoul(pp, NULL, 0);
	}
	pdata->p_proto = (u_char)strtoul(ap, NULL, 0);

	/* port */
	while (isspace(*cp))	/* skip white space */
		cp++;
	ap = cp;
	cp = strpbrk(cp, " \t");  /* move to the next white space char */
	if (cp == NULL)
		return (0);
	*cp++ = '\0';
	if ((pp = strchr(ap, '/')) != NULL) {
		*pp++ = '\0'; 
		len = 8 + 8 + strtoul(pp, NULL, 0);
	} else if (len == 999)
		len = 8 + 8 + 16;
	pdata->p_port = htons((u_short)strtoul(ap, NULL, 0));

	/* count */
	while (isspace(*cp))	/* skip white space */
		cp++;
	cnt = strtouq(cp, NULL, 0);

	*prefixlen = len;
	if (arg != NULL)
		*(int *)arg = 0;

	return (cnt);

  bad:
	warnx("parseproto: %s", buf);
	return (0);
}

static void
ip_counter(struct tree *tp, void *key, size_t prefixlen,
	   void *arg, u_int64_t cnt)
{
	/* arg points to offset of the tree array (used for addr6 trees). */
	if (arg != NULL)
		tp += *(int *)arg;

	(void)tnode_addcount(tp, key, prefixlen, cnt);
}

/*
 * plot format support
 */
int
ipplot_phase1(int nentries)
{
	if (addr_src != NULL)
		plot_phase1(addr_src, 2, nentries);
	if (addr_dst != NULL)
		plot_phase1(addr_dst, 2, nentries);
	if (proto_src != NULL)
		plot_phase1(proto_src, 1, nentries);
	if (proto_dst != NULL)
		plot_phase1(proto_dst, 1, nentries);
	return (0);
}

int
ipplot_phase2(void)
{
	if (addr_src != NULL) {
		printf("#[src address]\n");
		ipplot_printnames(addr_src);
		plot_phase2(addr_src);
	}
	if (addr_dst != NULL) {
		printf("#[dst address]\n");
		ipplot_printnames(addr_dst);
		plot_phase2(addr_dst);
	}
	if (proto_src != NULL) {
		printf("#[ip:proto:srcport]\n");
		ipplot_printnames(proto_src);
		plot_phase2(proto_src);
	}
	if (proto_dst != NULL) {
		printf("#[ip:proto:dstport]\n");
		ipplot_printnames(proto_dst);
		plot_phase2(proto_dst);
	}
	return (0);
}

/*
 * plot phase1 parser is the same as normal case.
 */
struct tree *
ipplot_parse1(char *buf, size_t len,
	     u_int64_t (**parser)(char *, void *, size_t *, void *),
	     void (**counter)(struct tree *, void *, size_t, void *, u_int64_t))
{
	return (ip_parsetype(buf, len, parser, counter));
}

/*
 * plot phase2 uses a different counter
 */
struct tree *
ipplot_parse2(char *buf, size_t len,
	     u_int64_t (**parser)(char *, void *, size_t *, void *),
	     void (**counter)(struct tree *, void *, size_t, void *, u_int64_t))
{
	struct	tree *tp;

	tp = ip_parsetype(buf, len, parser, counter);

	/* override the counter */
	if (tp != NULL)
		*counter = ipplot_counter;
	return (tp);
}

static void
ipplot_counter(struct tree *tp, void *key, size_t prefixlen,
	       void *arg, u_int64_t cnt)
{
	struct tree_node *np;
	struct plot_list *pl;
	struct plot_entry *ep;

	if (plot_timestamps[time_slot] == 0)
		plot_timestamps[time_slot] = end_time.tv_sec;

	/* arg points to offset of the tree array (used for addr6 trees). */
	if (arg != NULL)
		tp += *(int *)arg;

	/* need to call tnode_addcount without cnt to maintain the tree */
	np = tnode_addcount(tp, key, prefixlen, 0);

	/* add count to the plot list entry */
	pl = tree2plotlist(tp);
	while ((ep = tnode2pentry(np)) == NULL) {
		np = np->tn_parent;
		if (np == NULL)
			return;
	}
	ep->pe_counts[time_slot] += cnt;
}

static void
ipplot_printnames(struct tree *tp)
{
	struct plot_list *pl;
	struct plot_entry *ep;
	struct tree_node *np;
	char buf[INET6_ADDRSTRLEN + 8];

	if ((pl = tree2plotlist(tp)) == NULL)
		return;
	if (pl->pl_num == 0)
		/* no entries */
		return;
	printf("#time total ");
	for (ep = TAILQ_FIRST(&pl->pl_head); ep != NULL;
	     ep = TAILQ_NEXT(ep, pe_chain)) {
		np = ep->pe_node;
		ip_nodename(np, buf, sizeof(buf));
		printf("%s ", buf);
	}
	printf("\n");
}

static u_int64_t
null_parser(char *buf, void *key, size_t *prefixlen, void *arg)
{
	return (0);
}

static void
null_counter(struct tree *tp, void *key, size_t prefixlen,
	       void *arg, u_int64_t cnt)
{
}

