/*  $Id: SoundPlayer.cpp,v 1.2 2001/12/31 03:14:53 sarrazip Exp $
    SoundPlayer.cpp - SDL Sound effect player

    afternoonstalker - A robot-killing video game.
    Copyright (C) 2001 Pierre Sarrazin <sarrazip@iname.com>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#include "SoundPlayer.h"

#include <limits.h>


SoundPlayer::SoundPlayer()
{
    numCurrentIterators = 0;

    memset(&theDesiredAudioSpec, '\0', sizeof(theDesiredAudioSpec));

    theDesiredAudioSpec.freq = 11025;
    theDesiredAudioSpec.format = AUDIO_U8;  // unsigned 8-bit samples
    theDesiredAudioSpec.channels = 1;  // mono (SDL doc says 0; it seems wrong)
    theDesiredAudioSpec.samples = 512;  // low value, for short response time
    theDesiredAudioSpec.callback = sdlAudioCallback;
    theDesiredAudioSpec.userdata = this;

    if (SDL_OpenAudio(&theDesiredAudioSpec, NULL) != 0)
	throw string("SDL_OpenAudio failed");

    SDL_PauseAudio(1);  // start pause
}


SoundPlayer::~SoundPlayer()
{
    SDL_CloseAudio();
}


/*static*/
void
SoundPlayer::sdlAudioCallback(void *userdata, Uint8 *stream, int len)
{
    SoundPlayer *theSoundPlayer = (SoundPlayer *) userdata;
    theSoundPlayer->callbackMethod(stream, len);
}


inline
void
SoundPlayer::mute()
/*  CAUTION: this function is made to be called from the audio callback,
    and thus it does not lock and unlock the audio critical region.
    If it did, it would create a deadlock.
*/
{
    SDL_PauseAudio(1);
    numCurrentIterators = 0;
}


void
SoundPlayer::callbackMethod(Uint8 *stream, int len)
{
    /*  Determine how many bytes we can play.
	Eliminate empty buffers as they are discovered.
    */
    Uint32 minAvail = INT_MAX;
    for (size_t i = 0; i < numCurrentIterators; i++)
    {
	Uint32 buflen = currentIterators[i].wavBuffer->getLength();
	Uint32 offset = currentIterators[i].offset;
	Uint32 remaining = buflen - offset;
	Uint32 qty = (Uint32(len) <= remaining ? Uint32(len) : remaining);

	if (qty == 0)
	{
	    eraseSlot(i);
	    i--;
	    continue;
	}

	// Record the smallest quantity available:
	if (qty < minAvail)
	    minAvail = qty;
    }

    /*  If all buffers were empty, do nothing.
    */
    if (numCurrentIterators == 0)
    {
	mute();
	return;
    }

    /*  For each byte to send, compute a byte that is an average of
	the bytes that come for the several buffers.
    */
    for (Uint32 w = 0; w < minAvail; w++)
    {
	Uint32 acc = 0;
	for (size_t i = 0; i < numCurrentIterators; i++)
	{
	    Uint32 offset = currentIterators[i].offset++;
	    acc += currentIterators[i].wavBuffer->buf[offset];
	}
	acc /= numCurrentIterators;
	stream[w] = (Uint8) acc;
    }
}


void
SoundPlayer::eraseSlot(size_t slotIndex)
{
    for (size_t i = slotIndex + 1; i < numCurrentIterators; i++)
	currentIterators[i - 1] = currentIterators[i];
    numCurrentIterators--;
}


void
SoundPlayer::playWavBuffer(WavBuffer &wb)
{
    SDL_LockAudio();

    if (numCurrentIterators == MAX_NUM_SOUNDS)
	eraseSlot(0);
    currentIterators[numCurrentIterators].wavBuffer = &wb;
    currentIterators[numCurrentIterators].offset = 0;
    numCurrentIterators++;

    SDL_PauseAudio(0);
    SDL_UnlockAudio();
}


SoundPlayer::WavBuffer::WavBuffer()
  : len(0),
    buf(NULL)
{
}


SoundPlayer::WavBuffer::WavBuffer(const string &wavFilename) throw(int)
{
    if (init(wavFilename) != 0)
	throw -1;
}


int
SoundPlayer::WavBuffer::init(const string &wavFilename)
{
    len = 0;
    buf = NULL;
    SDL_AudioSpec wavSpec; 
    if (SDL_LoadWAV(wavFilename.c_str(), &wavSpec, &buf, &len) == NULL)
    {
	buf = NULL;  // to make sure destructor is safe
	return -1;
    }
    return 0;
}


SoundPlayer::WavBuffer::~WavBuffer()
{
    if (buf != NULL)
	SDL_FreeWAV(buf);
}
