/* aewm - a minimalist X11 window mananager. vim:sw=4:ts=4:et
 * Copyright 1998-2004 Decklin Foster <decklin@red-bean.com>
 * This program is free software; see LICENSE for details. */

#include "aewm.h"
#include "atom.h"

static void init_position(client_t *, strut_t *);
static void reparent(client_t *, strut_t *);
#ifdef MWM_HINTS
static int get_mwm_hints(Window, PropMwmHints *);
#endif

/* Set up a client structure for the new (not-yet-mapped) window. The
 * confusing bit is that we have to ignore 2 unmap events if the
 * client was already mapped but has IconicState set (for instance,
 * when we are the second window manager in a session).  That's
 * because there's one for the reparent (which happens on all viewable
 * windows) and then another for the unmapping itself. */

void make_new_client(Window w)
{
    client_t *c;
    XWindowAttributes attr;
    XWMHints *hints;
#ifdef MWM_HINTS
    PropMwmHints mhints;
#endif
    Atom win_type;
    strut_t s;
    long dummy;

    c = malloc(sizeof *c);
    c->next = head_client;
    head_client = c;

    XGrabServer(dpy);

    XGetTransientForHint(dpy, w, &c->trans);
    XFetchName(dpy, w, &c->name);
    XGetWindowAttributes(dpy, w, &attr);

    c->frame = None;
    c->window = w;
    c->ignore_unmap = 0;
    c->x = attr.x;
    c->y = attr.y;
    c->width = attr.width;
    c->height = attr.height;
    c->cmap = attr.colormap;
    c->has_title = 1;
    c->has_border = 1;
    c->shaded = 0;
    c->max_vert = 0;
    c->max_horz = 0;

    c->size = XAllocSizeHints();
    XGetWMNormalHints(dpy, c->window, c->size, &dummy);

#ifdef SHAPE
    c->has_been_shaped = 0;
#endif

#ifdef XFT
    c->xftdraw = NULL;
#endif

#ifdef DEBUG
    dump_title(c, "creating", 'w');
    dump_geom(c, "initial");
#endif

#ifdef MWM_HINTS
    if (get_mwm_hints(c->window, &mhints)) {
        if (mhints.flags & MWM_HINTS_DECORATIONS
            && !(mhints.decorations & MWM_DECOR_ALL)) {
            c->has_title  = mhints.decorations & MWM_DECOR_TITLE;
            c->has_border = mhints.decorations & MWM_DECOR_BORDER;
        }
    }
#endif

    if (get_atom(c->window, net_wm_window_type, XA_ATOM, &win_type)) {
        if (win_type == net_wm_window_type_dock) {
            c->has_title = 0;
            c->has_border = 0;
        }
    }

    atom_foreach(c->window, net_wm_state, XA_ATOM,
        handle_net_wm_state_item, (void *)c);

    if (get_atom(c->window, net_wm_desktop, XA_CARDINAL, &c->desktop)) {
        if (c->desktop >= opt_ndesks)
            c->desktop = opt_ndesks - 1;
    } else {
        set_atom(c->window, net_wm_desktop, XA_CARDINAL, cur_desk);
        c->desktop = cur_desk;
    }

    accumulate_struts(c->desktop, &s);

    if (attr.map_state == IsViewable) {
        c->ignore_unmap++;
    } else {
        init_position(c, &s);
        set_wm_state(c, NormalState);
        if ((hints = XGetWMHints(dpy, w))) {
            if (hints->flags & StateHint) set_wm_state(c, hints->initial_state);
            XFree(hints);
        }
    }

#ifdef DEBUG
    dump_geom(c, "set to");
    dump_info(c);
#endif

    gravitate(c, GRAV_APPLY);
    reparent(c, &s);

#ifdef XFT
    c->xftdraw = XftDrawCreate(dpy, (Drawable)c->frame,
        DefaultVisual(dpy, DefaultScreen(dpy)),
        DefaultColormap(dpy, DefaultScreen(dpy)));
#endif

    if (attr.map_state == IsViewable) {
        if (get_wm_state(c) == IconicState) {
            c->ignore_unmap++;
            XUnmapWindow(dpy, c->window);
        } else {
            XMapWindow(dpy, c->window);
            if (ON_CUR_DESK(c)) XMapRaised(dpy, c->frame);
            set_wm_state(c, NormalState);
        }
    } else {
        if (get_wm_state(c) == NormalState) {
            XMapWindow(dpy, c->window);
            if (ON_CUR_DESK(c)) XMapRaised(dpy, c->frame);
        }
    }

    append_to_atom(root, net_client_list, XA_WINDOW, c->window);

    XSync(dpy, False);
    XUngrabServer(dpy);
}

#ifdef MWM_HINTS
static int get_mwm_hints(Window w, PropMwmHints *h)
{
    Atom real_type;
    int real_format =0;
    unsigned long items_read = 0;
    unsigned long bytes_left = 0;
    unsigned char *data;

    XGetWindowProperty(dpy, w, mwm_hints,
        0, PROP_MOTIF_WM_HINTS_ELEMENTS, False, mwm_hints,
        &real_type, &real_format, &items_read, &bytes_left, &data);

    if (real_format == 32 && items_read >= PROP_MOTIF_WM_HINTS_ELEMENTS) {
        /* Look ma, no cast! */
        memcpy(h, data, sizeof *h);
        XFree(data);
        return 1;
    } else {
        return 0;
    }
}
#endif

/* Figure out where to map the window. c->x, c->y, c->width, and
 * c->height actually start out with values in them (whatever the
 * client passed to XCreateWindow).  Program-specified hints will
 * override these, but anything set by the program will be
 * sanity-checked before it is used. PSize is ignored completely,
 * because GTK sets it to 200x200 for almost everything. User-
 * specified hints will of course override anything the program says.
 *
 * If we can't find a reasonable position hint, we make up a position
 * using the mouse co-ordinates and window size. If the mouse is in
 * the center, we center the window; if it's at an edge, the window
 * goes on the edge. To account for window gravity while doing this,
 * we add theight into the calculation and then degravitate. Don't
 * think about it too hard, or your head will explode.
 *
 * We also subtract any struts (accumulated in s) from the range of x
 * and y values, and offset by the top/left of the strut, since we
 * don't want to map over a strut. */

static void init_position(client_t *c, strut_t *s)
{
    int xmax = DisplayWidth(dpy, screen) - s->left - s->right;
    int ymax = DisplayHeight(dpy, screen) - s->top - s->bottom;
    Atom win_type;

    if (c->size->flags & (USSize)) {
        if (c->size->width) c->width = c->size->width;
        if (c->size->height) c->height = c->size->height;
    } else {
        /* make sure it's big enough to click at... this is somewhat
         * arbitrary */
        if (c->width < 2 * theight(c)) c->width = 2 * theight(c);
        if (c->height < theight(c)) c->height = theight(c);
    }

    if (c->size->flags & USPosition) {
        c->x = c->size->x;
        c->y = c->size->y;
    } else {
        if (c->size->flags & PPosition) {
            c->x = c->size->x;
            c->y = c->size->y;
        }
        if (c->x < 0) c->x = 0;
        if (c->y < 0) c->y = 0;
        if (c->x > xmax) c->x = xmax - theight(c);
        if (c->y > ymax) c->y = ymax - theight(c);

        /* If it's -not- a dock, but it is at 0,0, we want to place it.
         * Docks are the the only types of windows that we let get
         * away with placing themselves at 0,0. */

        if (!(get_atom(c->window, net_wm_window_type, XA_ATOM, &win_type) &&
                win_type == net_wm_window_type_dock) &&
                c->x == 0 && c->y == 0) {
            int mouse_x, mouse_y;
            get_mouse_position(&mouse_x, &mouse_y);
            if (c->width < xmax)
                c->x = (mouse_x < xmax ? (mouse_x / (float)xmax) : 1)
                    * (xmax - c->width - 2*BW(c));
            if (c->height + theight(c) < ymax)
                c->y = (mouse_y < ymax ? (mouse_y / (float)ymax) : 1)
                    * (ymax - c->height - theight(c) - 2*BW(c));
            c->y += theight(c);
            gravitate(c, GRAV_UNDO);
        }
    }

    c->x += s->left;
    c->y += s->top;
}

/* Stick the client's window into our frame. */

static void reparent(client_t *c, strut_t *s)
{
    XSetWindowAttributes pattr;
    int fx, fy, fw, fh;

    if (c->max_vert && c->max_horz) {
        c->save_x = c->x;
        c->save_y = c->y;
        c->save_width = c->width;
        c->save_height = c->height;
        c->x = s->left;
        c->y = s->top + theight(c);
        c->width = DisplayWidth(dpy, screen) - 2*BW(c) -
            s->left - s->right;
        c->height = DisplayHeight(dpy, screen) - 2*BW(c) - theight(c) -
            s->top - s->bottom;
    }

    if (c->shaded) {
        fx = c->x;
        fy = c->y - theight(c);
        fw = c->width;
        fh = theight(c) - BW(c);
    } else {
        fx = c->x;
        fy = c->y - theight(c);
        fw = c->width;
        fh = c->height + theight(c);
    }

    pattr.override_redirect = True;
    pattr.background_pixel = bg.pixel;
    pattr.border_pixel = bd.pixel;
    pattr.event_mask = ChildMask|ButtonPressMask|ExposureMask|EnterWindowMask;
    c->frame = XCreateWindow(dpy, root, fx, fy, fw, fh, BW(c),
        DefaultDepth(dpy, screen), CopyFromParent, DefaultVisual(dpy, screen),
        CWOverrideRedirect|CWBackPixel|CWBorderPixel|CWEventMask, &pattr);

#ifdef SHAPE
    if (shape) {
        XShapeSelectInput(dpy, c->window, ShapeNotifyMask);
        set_shape(c);
    }
#endif

    XAddToSaveSet(dpy, c->window);
    XSelectInput(dpy, c->window, ColormapChangeMask|PropertyChangeMask);
    XSetWindowBorderWidth(dpy, c->window, 0);
    XResizeWindow(dpy, c->window, c->width, c->height);
    XReparentWindow(dpy, c->window, c->frame, 0, theight(c));

    send_config(c);
}
