/* aewm - a minimalist X11 window mananager. vim:sw=4:ts=4:et
 * Copyright 1998-2004 Decklin Foster <decklin@red-bean.com>
 * This program is free software; see LICENSE for details. */

#include <X11/Xmd.h>
#include "aewm.h"
#include "atom.h"

client_t *find_client(Window w, int mode)
{
    client_t *c;

    if (mode == MATCH_FRAME) {
        for (c = head_client; c; c = c->next)
            if (c->frame == w) return c;
    } else /* mode == MATCH_WINDOW */ {
        for (c = head_client; c; c = c->next)
            if (c->window == w) return c;
    }

    return NULL;
}

/* For a regular window, c->trans is None (false), and we include
 * enough space to draw the title. For a transient window we just make
 * a tiny strip. */

int theight(client_t *c)
{
    if (!c) return 0;

#ifdef MWM_HINTS
    if (!c->has_title) return 0;
#endif
#ifdef XFT
    return (c->trans ? 0 : xftfont->ascent + xftfont->descent) +
        2*opt_pad + BW(c);
#else
    return (c->trans ? 0 : font->ascent + font->descent) +
        2*opt_pad + BW(c);
#endif
}

/* If we can't find a WM_STATE we're going to have to assume
 * Withdrawn. This is not exactly optimal, since we can't really
 * distinguish between the case where no WM has run yet and when the
 * state was explicitly removed (clients are allowed to either set the
 * atom to Withdrawn or just remove it... yuck.)
 *
 * WM_STATE actually has a second element, the old ICCCM-style icon,
 * but we don't care about it so we can just use get_atom here. */

int get_wm_state(client_t *c)
{
    long state;

    if (get_atom(c->window, wm_state, wm_state, &state))
        return state;
    else
        return WithdrawnState;
}

int set_wm_state(client_t *c, int state)
{
    return set_atom(c->window, wm_state, wm_state, state);
}

/* This can be called before we map, so check c->frame. If c->frame is
 * still None, we merely set the variables, and then later (in
 * reparent) adjust window and/or frame sizes based on them. */

void handle_net_wm_state_item(unsigned char *data, void *cb_data)
{
    Atom prop = *(Atom *)data;
    client_t *c = cb_data;

    if (prop == net_wm_state_shaded && !c->shaded) {
        if (c->frame) shade_win(c);
        else c->shaded = 1;
    } else if (prop == net_wm_state_max_vert && !c->max_vert) {
        c->max_vert = 1;
        if (c->frame && c->max_vert && c->max_horz) maximize_win(c);
    } else if (prop == net_wm_state_max_horz && !c->max_horz) {
        c->max_horz = 1;
        if (c->frame && c->max_vert && c->max_horz) maximize_win(c);
    }
}

/* This will need to be called whenever we update our client_t stuff.
 * Yeah, yeah, stop yelling at me about OO. */

void send_config(client_t *c)
{
    XConfigureEvent ce;

    ce.type = ConfigureNotify;
    ce.event = c->window;
    ce.window = c->window;
    ce.x = c->x;
    ce.y = c->y;
    ce.width = c->width;
    ce.height = c->height;
    ce.border_width = 0;
    ce.above = None;
    ce.override_redirect = 0;

    XSendEvent(dpy, c->window, False, StructureNotifyMask, (XEvent *)&ce);
}

/* After pulling my hair out trying to find some way to tell if a
 * window is still valid, I've decided to instead carefully ignore any
 * errors raised by this function. We know that the X calls are, and
 * we know the only reason why they could fail -- a window has removed
 * itself completely before the Unmap and Destroy events get through
 * the queue to us. It's not absolutely perfect, but it works.
 *
 * The 'withdrawing' argument specifes if the client is actually
 * (destroying itself||being destroyed by us) or if we are merely
 * cleaning up its data structures when we exit mid-session. */

void remove_client(client_t *c, int mode)
{
    client_t *p;

    XGrabServer(dpy);
    XSetErrorHandler(ignore_xerror);

#ifdef DEBUG
    dump_title(c, "removing", 'r');
    dump_removal(c, mode);
#endif

    if (mode == REMOVE_WITHDRAW) {
        set_wm_state(c, WithdrawnState);
    } else /* mode == REMOVE_REMAP */ {
        if (c->max_vert && c->max_horz) {
            c->x = c->save_x;
            c->y = c->save_y;
            c->width = c->save_width;
            c->height = c->save_height;
            XResizeWindow(dpy, c->window,
                c->width, c->height);
        }
        XMapWindow(dpy, c->window);
    }

    remove_from_atom(root, net_client_list, XA_WINDOW, c->window);

    gravitate(c, GRAV_UNDO);
    XReparentWindow(dpy, c->window, root, c->x, c->y);
#ifdef MWM_HINTS
    if (c->has_border) XSetWindowBorderWidth(dpy, c->window, 1);
#else
    XSetWindowBorderWidth(dpy, c->window, 1);
#endif
#ifdef XFT
    if (c->xftdraw) XftDrawDestroy(c->xftdraw);
#endif
    XRemoveFromSaveSet(dpy, c->window);
    XDestroyWindow(dpy, c->frame);

    if (head_client == c) head_client = c->next;
    else for (p = head_client; p && p->next; p = p->next)
        if (p->next == c) p->next = c->next;

    if (c->name) XFree(c->name);
    if (c->size) XFree(c->size);
    free(c);

    XSync(dpy, False);
    XSetErrorHandler(handle_xerror);
    XUngrabServer(dpy);
}

/* I've changed this to just clear the window every time. The amount
 * of 'flicker' is basically imperceptable. Also, we might be drawing
 * an anti-aliased font with Xft, in which case we always have to
 * clear to draw the text properly. This allows us to simplify
 * handle_property_change as well. */

void redraw(client_t *c)
{
    int innerbox_x = 0;
    int innerbox_y = 0;
    int innerbox_w = 0;
    int innerbox_h = 0;

#ifdef MWM_HINTS
    if (!c->has_title) return;
#endif

    XClearWindow(dpy, c->frame);

    if (!c->trans && c->name) {
#ifdef XFT
        XftDrawString8(c->xftdraw, &xft_fg,
            xftfont, opt_pad, opt_pad + xftfont->ascent,
            c->name, strlen(c->name));
#else
        XDrawString(dpy, c->frame, string_gc,
            opt_pad, opt_pad + font->ascent,
            c->name, strlen(c->name));
#endif
    }

    XDrawLine(dpy, c->frame, border_gc,
        0, theight(c) - BW(c) + BW(c)/2,
        c->width, theight(c) - BW(c) + BW(c)/2);
    XDrawLine(dpy, c->frame, border_gc,
        c->width - theight(c) + BW(c)/2, 0,
        c->width - theight(c) + BW(c)/2, theight(c));
    XDrawLine(dpy, c->frame, border_gc,
        c->width - 2*theight(c) + BW(c)/2, 0,
        c->width - 2*theight(c) + BW(c)/2, theight(c));

    /* Ugh. What a mess. Don't ask me how I figured these numbers out.
     * I hate graphics programming. */

    if ((c->max_vert && c->max_horz) || c->desktop == DESK_ALL) {
        innerbox_x = c->width - 2*theight(c) + BW(c) + (theight(c)-BW(c))/4;
        innerbox_y = (theight(c)-BW(c))/4;
        innerbox_w = (theight(c)-BW(c)) - (theight(c)-BW(c))/4*2 - BW(c)%2;
        innerbox_h = (theight(c)-BW(c)) - (theight(c)-BW(c))/4*2 - BW(c)%2;
    }

    if (c->max_vert && c->max_horz) {
        XDrawRectangle(dpy, c->frame, border_gc,
            innerbox_x, innerbox_y, innerbox_w, innerbox_h);
    }
    if (c->desktop == DESK_ALL) {
        XDrawLine(dpy, c->frame, border_gc,
            innerbox_x + BW(c)%2,
            innerbox_y + BW(c)%2,
            innerbox_x + innerbox_w,
            innerbox_y + innerbox_h);
        XDrawLine(dpy, c->frame, border_gc,
            innerbox_x + BW(c)%2,
            innerbox_y + innerbox_h - 1,
            innerbox_x + innerbox_w,
            innerbox_y - (BW(c)+1)%2);
    }
}

/* Window gravity is a mess to explain, but we don't need to do much
 * about it since we're using X borders. For NorthWest et al, the top
 * left corner of the window when there is no WM needs to match up
 * with the top left of our fram once we manage it, and likewise with
 * SouthWest and the bottom right (these are the only values I ever
 * use, but the others should be obvious.) Our titlebar is on the top
 * so we only have to adjust in the first case. */

void gravitate(client_t *c, int mode)
{
    int dy = 0;
    int gravity = (c->size->flags & PWinGravity) ?
        c->size->win_gravity : NorthWestGravity;

    switch (gravity) {
        case NorthWestGravity:
        case NorthEastGravity:
        case NorthGravity: dy = theight(c); break;
        case CenterGravity: dy = theight(c)/2; break;
    }

    if (mode == GRAV_APPLY)
        c->y += dy;
    else /* mode == GRAV_UNDO */
        c->y -= dy;
}

/* Well, the man pages for the shape extension say nothing, but I was
 * able to find a shape.PS.Z on the x.org FTP site. What we want to do
 * here is make the window shape be a boolean OR (or union, if you
 * prefer) of the client's shape and our titlebar. The titlebar
 * requires both a bound and a clip because it has a border -- the X
 * server will paint the border in the region between the two. (I knew
 * that using X borders would get me eventually... ;-)) */

#ifdef SHAPE
void set_shape(client_t *c)
{
    int n, order;
    XRectangle temp, *dummy;

    dummy = XShapeGetRectangles(dpy, c->window, ShapeBounding, &n, &order);
    if (n > 1) {
        XShapeCombineShape(dpy, c->frame, ShapeBounding,
            0, theight(c), c->window, ShapeBounding, ShapeSet);
        temp.x = -BW(c);
        temp.y = -BW(c);
        temp.width = c->width + 2*BW(c);
        temp.height = theight(c) + BW(c);
        XShapeCombineRectangles(dpy, c->frame, ShapeBounding,
            0, 0, &temp, 1, ShapeUnion, YXBanded);
        temp.x = 0;
        temp.y = 0;
        temp.width = c->width;
        temp.height = theight(c) - BW(c);
        XShapeCombineRectangles(dpy, c->frame, ShapeClip,
            0, theight(c), &temp, 1, ShapeUnion, YXBanded);
        c->has_been_shaped = 1;
    } else if (c->has_been_shaped) {
        /* I can't find a 'remove all shaping' function... */
        temp.x = -BW(c);
        temp.y = -BW(c);
        temp.width = c->width + 2*BW(c);
        temp.height = c->height + theight(c) + 2*BW(c);
        XShapeCombineRectangles(dpy, c->frame, ShapeBounding,
            0, 0, &temp, 1, ShapeSet, YXBanded);
    }
    XFree(dummy);
}
#endif
