/* aewm - a minimalistic X11 window manager. ------- vim:sw=4:et
 * Copyright (c) 1998-2001 Decklin Foster <decklin@red-bean.com>
 * Free software! Please see README for details and license.  */

#include "aewm.h"

static void init_position(Client *);
static void reparent(Client *);
#ifdef MWM_HINTS
static PropMwmHints *get_mwm_hints(Window);
#endif

/* Set up a client structure for the new (not-yet-mapped) window. The
 * confusing bit is that we have to ignore 2 unmap events if the
 * client was already mapped but has IconicState set (for instance,
 * when we are the second window manager in a session).  That's
 * because there's one for the reparent (which happens on all viewable
 * windows) and then another for the unmapping itself. */

void make_new_client(Window w)
{
    Client *c;
    XWindowAttributes attr;
    XWMHints *hints;
#ifdef MWM_HINTS
    PropMwmHints *mhints;
#endif
    long dummy;

    c = malloc(sizeof *c);
    c->next = head_client;
    head_client = c;

    XGrabServer(dpy);

    XGetTransientForHint(dpy, w, &c->trans);
    XFetchName(dpy, w, &c->name);
    XGetWindowAttributes(dpy, w, &attr);

    c->window = w;
    c->ignore_unmap = 0;
#ifdef SHAPE
    c->has_been_shaped = 0;
#endif
    c->x = attr.x;
    c->y = attr.y;
    c->width = attr.width;
    c->height = attr.height;
    c->cmap = attr.colormap;
    c->size = XAllocSizeHints();
    XGetWMNormalHints(dpy, c->window, c->size, &dummy);

#ifdef MWM_HINTS
    c->has_title  = 1;
    c->has_border = 1;

    if ((mhints = get_mwm_hints(c->window))) {
        if (mhints->flags & MWM_HINTS_DECORATIONS
                && !(mhints->decorations & MWM_DECOR_ALL)) {
            c->has_title  = mhints->decorations & MWM_DECOR_TITLE;
            c->has_border = mhints->decorations & MWM_DECOR_BORDER;
        }
        XFree(mhints);
    }
#endif

    if (attr.map_state == IsViewable) {
        c->ignore_unmap++;
    } else {
        init_position(c);
        set_wm_state(c, NormalState);
        if ((hints = XGetWMHints(dpy, w))) {
            if (hints->flags & StateHint) set_wm_state(c, hints->initial_state);
            XFree(hints);
        }
    }

    gravitate(c, APPLY_GRAVITY);
    reparent(c);

#ifdef XFT
    c->xftdraw = XftDrawCreate(dpy, (Drawable) c->frame,
        DefaultVisual(dpy, DefaultScreen(dpy)),
        DefaultColormap(dpy, DefaultScreen(dpy)));
#endif

    if (attr.map_state == IsViewable) {
        if (get_wm_state(c) == IconicState) {
            c->ignore_unmap++;
            XUnmapWindow(dpy, c->window);
        } else {
            XMapWindow(dpy, c->window);
            XMapRaised(dpy, c->frame);
            set_wm_state(c, NormalState);
        }
    } else {
        if (get_wm_state(c) == NormalState) {
            XMapWindow(dpy, c->window);
            XMapRaised(dpy, c->frame);
        }
    }

#ifdef DEBUG
    dump(c);
#endif

    XSync(dpy, False);
    XUngrabServer(dpy);
}

/* This one does -not- free the data coming back from Xlib; it just
 * sends back the pointer to what was allocated. */

#ifdef MWM_HINTS
static PropMwmHints *get_mwm_hints(Window w)
{
    Atom real_type; int real_format;
    unsigned long items_read, items_left;
    PropMwmHints *data;

    if (XGetWindowProperty(dpy, w, mwm_hints, 0L, 20L, False,
            mwm_hints, &real_type, &real_format, &items_read, &items_left,
            (unsigned char **) &data) == Success
            && items_read >= PROP_MOTIF_WM_HINTS_ELEMENTS) {
        return data;
    } else {
        return NULL;
    }
}
#endif

/* All toolkits suck. GTK sucks in particular, because if you even
 * -look- at it funny, it will put a PSize hint on your window, and
 * then gleefully leave it at the default setting of 200x200 until you
 * change it. So PSize is pretty useless for us these days.
 *
 * Note that c->x, c->y, c->width, and c->height actually start out
 * with values in them (whatever the client passed to XCreateWindow).
 * Program-specified hints will override these, but anything set by
 * the program will be sanity-checked before it is used. User-
 * specified hints will of course override anything the program says.
 *
 * If we can't find a reasonable position hint, we make up a position
 * using the relative mouse co-ordinates and window size. To account
 * for window gravity while doing this, we add theight into the
 * calculation and then degravitate. Don't think about it too hard, or
 * your head will explode. */

static void init_position(Client *c)
{
    int xmax = DisplayWidth(dpy, screen);
    int ymax = DisplayHeight(dpy, screen);
    int mouse_x, mouse_y;

    if (c->size->flags & (USSize)) {
        c->width = c->size->width;
        c->height = c->size->height;
    } else {
        /* we would check PSize here, if GTK didn't blow goats */
        /* make sure it's big enough to click at */
       if (c->width < 2 * theight(c)) c->width = 2 * theight(c);
       if (c->height < theight(c)) c->height = theight(c);
    }

    if (c->size->flags & USPosition) {
        c->x = c->size->x;
        c->y = c->size->y;
    } else {
        if (c->size->flags & PPosition) {
            c->x = c->size->x;
            c->y = c->size->y;
        }
        if (c->x < 0) c->x = 0;
        if (c->y < 0) c->y = 0;
        if (c->x > xmax) c->x = xmax - 5;
        if (c->y > ymax) c->y = ymax - 5;
    }

    if (c->x == 0 && c->y == 0) {
        get_mouse_position(&mouse_x, &mouse_y);

        if (c->width < xmax)
            c->x = (mouse_x / (float)xmax) * (xmax - c->width);
        if (c->height + theight(c) < ymax)
            c->y = (mouse_y / (float)ymax) * (ymax - c->height - theight(c));

        c->y += theight(c);
        gravitate(c, REMOVE_GRAVITY);
    }
}

static void reparent(Client *c)
{
    XSetWindowAttributes pattr;

    pattr.override_redirect = True;
    pattr.background_pixel = bg.pixel;
    pattr.border_pixel = bd.pixel;
    pattr.event_mask = ChildMask|ButtonPressMask|ExposureMask|EnterWindowMask;
    c->frame = XCreateWindow(dpy, root,
        c->x, c->y - theight(c), c->width, c->height + theight(c), BW(c),
        DefaultDepth(dpy, screen), CopyFromParent, DefaultVisual(dpy, screen),
        CWOverrideRedirect|CWBackPixel|CWBorderPixel|CWEventMask, &pattr);

#ifdef SHAPE
    if (shape) {
        XShapeSelectInput(dpy, c->window, ShapeNotifyMask);
        set_shape(c);
    }
#endif

    XAddToSaveSet(dpy, c->window);
    XSelectInput(dpy, c->window, ColormapChangeMask|PropertyChangeMask);
    XSetWindowBorderWidth(dpy, c->window, 0);
    XResizeWindow(dpy, c->window, c->width, c->height);
    XReparentWindow(dpy, c->window, c->frame, 0, theight(c));

    send_config(c);
}
