/********************************************************************
This file is part of the abs 0.8 distribution.  abs is a spreadsheet
with graphical user interface.

Copyright (C) 1998-2000  Andr Bertin (Andre.Bertin@pi.be) 

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version if in the same spirit as version 2.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

Concact: abs@ping.be or abs@pi.be
         http://www.ping.be/bertin/abs.shtml
         http://www.pi.be/bertin/abs.shtml

*********************************************************************/

#include "callback.h"
#include "callback_decl.h"
#include "inputbox.h"
#include "gr_interf.h"
#include "main.h"
#include "mainwin.h"
#include "memory.h"

#include "cursor.h"
#include "application.h"

extern void setscrollv (double v);
extern void setscrollh (double v);

#include "newplot.h"

extern int rebuild_thumb ();

#define AW ActiveWorksheet
#define CW worksheet_getcolw
#define LH worksheet_getrowh
#define SW worksheet_setcolw
#define SH worksheet_setrowh

static int nocursor = 0;
static int selectismoving = 0;
static int Shift = 0;
static int Control = 0;

static int EXTEND_DIR = UNDEFINE;

static Cell *EditedCell = NULL;

int
DoUndoFormula (int r, int c, char *formula)
{
  char *old = (char *) get_formula (r, c);
  char redo[256];
  char undo[256];

  if (old == NULL)
    sprintf (undo, "Cells(%d,%d).Formula = \"\"\n", r, c);
  else
    {
      sprintf (undo, "Cells(%d,%d).Formula = \"%s\"\n", r, c, old);
    }
  if (formula != NULL)
    sprintf (redo, "Cells(%d,%d).Formula = \"%s\"\n", r, c, formula);
  else
    sprintf (redo, "Cells(%d,%d).Formula = \"\"\n", r, c);

  newmacroline (redo);

  return 0;
}

int
cb_changesheet (int sheet)
{

  if (textismanaged && texthaschanged && nocursor == 0)
    {
      EditedCell = ActiveCell;
      nocursor = 1;
      popdownactivecell ();
    }

  if (application_changesheet (sheet))
    {
      return -1;
    }

  return 0;
}

int
repaint_activesheet ()
{
  isselectinggra = 0;
  isselectingdra = 0;
  isselectingbtn = 0;
  resetselect (0, 0, 0);
  xrepaint ();
  return 0;
}

int
selectingto0 ()
{
  isselectinggra = 0;
  isselectingdra = 0;
  isselectingbtn = 0;
  return 0;
}

void
a_update ()
{
  char *buf;
  char *str;
  int len;
  int second = 0;

  if (ActiveWorksheet != NULL)
    {
      if (textismanaged && texthaschanged)
	{
	  str = GetEntryString (Letext);
	  if (str == NULL)
	    return;
	  len = strlen (str);
	  if (len > 1)
	    if ('0' <= str[1] && str[1] <= '9')
	      {
		second = 1;
	      }

	  buf =
	    (char *) absmalloc (sizeof (char) * (len + 3), "a_update:buf");
	  if (str[0] == '=' || str[0] == '+' || str[0] == '-'
	      || str[0] == '\'' || ('0' <= str[0] && str[0] <= '9')
	      || (len > 1 && str[0] == '.' && second == 1))
	    {
	      strcpy (buf, str);
	    }
	  else
	    {
	      buf[0] = '\'';
	      strcpy (buf + 1, str);
	    }

	  cell_activate ((Cell *) applicationcell (AW->cur_r, AW->cur_c, 1));
	  DoUndoFormula (AW->cur_r, AW->cur_c, buf);
	  set_formula (AW->cur_r, AW->cur_c, buf);

	  absfree (buf, "a_update:buf");
	}
    }

  if (textismanaged)
    {
      xdrawcell (AW->cur_r, AW->cur_c, 4);
      setstring (Letext, "");
      LetextLooseFocus ();
      textismanaged = 0;
      texthaschanged = 0;
      resetselect (0, 0, 0);
      popdownactivecell ();
    }
  return;

}

void
echofct (char *buf)
{
  long p = XawTextGetInsertionPoint (Letext);
  if (p == 0)
    insertstring (Letext, "=");
  insertstring (Letext, buf);
  moveinsertionpoint (Letext, -1);

}

int
setcursor (int r, int c)
{
  int oldr, oldc;
  char *label;

  if (AW == NULL)
    return -1;

  if (is_selecting () || isselectinggra || isselectingdra)
    {
      resetselect (0, 0, 0);
      isselectinggra = 0;
      isselectingdra = 0;
      isselectingbtn = 0;
      xrepaint ();
    }
  if (texthaschanged)
    a_update ();

  LetextGetFocus ();
  textismanaged = 1;
  texthaschanged = 0;

  oldr = AW->cur_r;
  oldc = AW->cur_c;
  worksheet_setcursor (AW, r, c);
  xdrawlincell (oldr, 0, 0);
  xdrawcolcell (0, oldc, 0);
  xdrawlincell (AW->cur_r, 0, 0);
  xdrawcolcell (0, AW->cur_c, 0);
  clearwin (linindex);
  clearwin (colindex);

  setselection (AW->cur_r, AW->cur_c, AW->cur_r, AW->cur_c);

  set_visible (AW->cur_r, AW->cur_c);
  label = (char *) get_formula (AW->cur_r, AW->cur_c);
  if (label != NULL)
    setstring (Letext, label);
  else
    setstring (Letext, "");

  xdrawcursor ();
  return 0;
}

int
movecursor (int dr, int dc)
{
  int oldr, oldc;
  char *label;

  if (AW == NULL)
    return -1;

  if (is_selecting () || isselectinggra || isselectingdra)
    {
      resetselect (0, 0, 0);
      isselectinggra = 0;
      isselectingdra = 0;
      isselectingbtn = 0;
      xrepaint ();
    }
  if (texthaschanged)
    a_update ();

  LetextGetFocus ();
  textismanaged = 1;
  texthaschanged = 0;

  oldr = AW->cur_r;
  oldc = AW->cur_c;
  worksheet_movecursor (AW, dr, dc);
  xdrawlincell (oldr, 0, 0);
  xdrawcolcell (0, oldc, 0);
  xdrawlincell (AW->cur_r, 0, 0);
  xdrawcolcell (0, AW->cur_c, 0);
  clearwin (linindex);
  clearwin (colindex);

  setselection (AW->cur_r, AW->cur_c, AW->cur_r, AW->cur_c);
  set_visible (AW->cur_r, AW->cur_c);

  label = (char *) get_formula (AW->cur_r, AW->cur_c);
  if (label != NULL)
    setstring (Letext, label);
  else
    setstring (Letext, "");

  xdrawcursor ();
  return 0;
}

void
cb_TextUp (Widget w, XEvent * event, String * params, Cardinal * num_params)
{
  movecursor (-1, 0);
  arrowmove = 1;
}

void
cb_TextDown (Widget w, XEvent * event, String * params, Cardinal * num_params)
{
  movecursor (1, 0);
  arrowmove = 1;
}

void
cb_TextLeft (Widget w, XEvent * event, String * params, Cardinal * num_params)
{
  long p;

  if (texthaschanged)
    {
      p = XawTextGetInsertionPoint (Letext);
      if (p > 0)
	{
	  p--;
	  XawTextSetInsertionPoint (Letext, p);
	  return;
	}

    }
  movecursor (0, -1);
  arrowmove = 1;

}

void
cb_TextRight (Widget w, XEvent * event, String * params,
	      Cardinal * num_params)
{
  long p;
  int l;

  if (texthaschanged)
    {
      l = strlen (GetEntryString (Letext));
      p = XawTextGetInsertionPoint (Letext);
      if (p < l)
	{
	  p++;
	  XawTextSetInsertionPoint (Letext, p);
	  return;
	}
    }
  movecursor (0, 1);
  arrowmove = 1;
}

void
cb_Delete (Widget w, XEvent * event, String * params, Cardinal * num_params)
{
  long p;
  int l, i;
  char *str;
  char tmp[256];
  Widget t;

  if (textismanaged)
    t = Letext;
  else
    t = w;

  str = GetEntryString (t);
  if (str == NULL)
    return;
  strcpy (tmp, str);

  l = strlen (str);
  p = XawTextGetInsertionPoint (t);
  if (p == l)
    return;

  for (i = p; i < l - 1; i++)
    tmp[i] = tmp[i + 1];
  tmp[l - 1] = '\0';

  setstring (t, tmp);
  XawTextSetInsertionPoint (t, p);
  return;
}

void
cb_BackSpace (Widget w, XEvent * event, String * params,
	      Cardinal * num_params)
{
  long p;
  int l, i;
  char *str;
  char tmp[256];
  Widget t;

  if (textismanaged)
    t = Letext;
  else
    t = w;

  str = GetEntryString (t);
  if (str == NULL)
    return;
  strcpy (tmp, str);

  l = strlen (str);
  p = XawTextGetInsertionPoint (t);
  if (p == 0)
    return;

  for (i = p - 1; i < l - 1; i++)
    tmp[i] = tmp[i + 1];

  tmp[l - 1] = '\0';
  setstring (t, tmp);
  XawTextSetInsertionPoint (t, p - 1);
  return;
}

void
cb_DocMotion (Widget w, XEvent * event, String * params,
	      Cardinal * num_params)
{

  Drawing *dr;
  int x, y;
  int xmin, xmax, ymin, ymax;
  int i1, i2, j1, j2, k;

  if (BTN1_STATE != UP || texthaschanged)
    return;
  x = event->xbutton.x * 100 / zoom;
  y = event->xbutton.y * 100 / zoom;

  if (drawingmode)
    {

      movex = x;
      movey = y;
      absmovex = movex + CW (AW, AW->cc);
      absmovey = movey + LH (AW, AW->ll);

      if (drawingmode == READYEDITDRAWING)
	{
	  for (k = 0; k < getnumdra (); k++)
	    {
	      dr = (Drawing *) getdra (k);
	      if (drawing_pointed (dr, absmovex, absmovey, 5, 5)
		  && ActiveDrawing != dr)
		{
		  ActivateDrawing (dr);
		  worksheet_select_drawing (AW, dr);
		  clearnflush ();
		  drawselect ();
		}
	    }
	}
      return;
    }

  if (isselectinggra)
    {
      if (xa - 4 <= x && x <= xa + 4 && yb - 4 <= y && y <= yb + 4)
	{
	  if (CURSOR != BOTTOM_LEFT)
	    TurnOnBottomLeft (draw);
	  CURSOR = BOTTOM_LEFT;
	}
      else if (CURSOR == BOTTOM_LEFT)
	{
	  TurnOffCursor (draw);
	  CURSOR = ARROW;
	}
      if (xb - 4 <= x && x <= xb + 4 && yb - 4 <= y && y <= yb + 4)
	{
	  if (CURSOR != BOTTOM_RIGHT)
	    TurnOnBottomRight (draw);
	  CURSOR = BOTTOM_RIGHT;
	}
      else if (CURSOR == BOTTOM_RIGHT)
	{
	  TurnOffCursor (draw);
	  CURSOR = ARROW;
	}
      if (xa - 4 <= x && x <= xa + 4 && ya - 4 <= y && y <= ya + 4)
	{
	  if (CURSOR != TOP_LEFT)
	    TurnOnTopLeft (draw);
	  CURSOR = TOP_LEFT;
	}
      else if (CURSOR == TOP_LEFT)
	{
	  TurnOffCursor (draw);
	  CURSOR = ARROW;
	}
      if (xb - 4 <= x && x <= xb + 4 && ya - 4 <= y && y <= ya + 4)
	{
	  if (CURSOR != TOP_RIGHT)
	    TurnOnTopRight (draw);
	  CURSOR = TOP_RIGHT;
	}
      else if (CURSOR == TOP_RIGHT)
	{
	  TurnOffCursor (draw);
	  CURSOR = ARROW;
	}
      if ((xa + xb) / 2 - 4 <= x && x <= (xa + xb) / 2 + 4 && yb - 4 <= y
	  && y <= yb + 4)
	{
	  if (CURSOR != BOTTOM_SIDE)
	    TurnOnBottomSide (draw);
	  CURSOR = BOTTOM_SIDE;
	}
      else if (CURSOR == BOTTOM_SIDE)
	{
	  TurnOffCursor (draw);
	  CURSOR = ARROW;
	}
      if (xb - 4 <= x && x <= xb + 4 && (yb + ya) / 2 - 4 <= y
	  && y <= (yb + ya) / 2 + 4)
	{
	  if (CURSOR != RIGHT_SIDE)
	    TurnOnRightSide (draw);
	  CURSOR = RIGHT_SIDE;
	}
      else if (CURSOR == RIGHT_SIDE)
	{
	  TurnOffCursor (draw);
	  CURSOR = ARROW;
	}
      if ((xa + xb) / 2 - 4 <= x && x <= (xa + xb) / 2 + 4 && ya - 4 <= y
	  && y <= ya + 4)
	{
	  if (CURSOR != TOP_SIDE)
	    TurnOnTopSide (draw);
	  CURSOR = TOP_SIDE;
	}
      else if (CURSOR == TOP_SIDE)
	{
	  TurnOffCursor (draw);
	  CURSOR = ARROW;
	}
      if (xa - 4 <= x && x <= xa + 4 && (yb + ya) / 2 - 4 <= y
	  && y <= (yb + ya) / 2 + 4)
	{
	  if (CURSOR != LEFT_SIDE)
	    TurnOnLeftSide (draw);
	  CURSOR = LEFT_SIDE;
	}
      else if (CURSOR == LEFT_SIDE)
	{
	  TurnOffCursor (draw);
	  CURSOR = ARROW;
	}
    }
  if (w == colindex)
    {
      TurnOnHand (colindex);
    }
  if (w == linindex)
    {
      TurnOnHand (linindex);
    }

  get_select (&i1, &j1, &i2, &j2);
  if (i1 == 0 && j1 == 0 && ActiveCell != NULL)
    {
      i1 = i2 = ActiveCell->r;
      j1 = j2 = ActiveCell->c;
    }

  ijtoxy (i1, j1, &xmin, &ymin);
  ijtoxy (i2 + 1, j2 + 1, &xmax, &ymax);
  if (xmin < x && x < xmax && ymin - 10 < y && y < ymin + 5)
    {
      if (CURSOR != HAND && CURSOR != BRUSH)
	{
	  TurnOnHand (draw);
	  CURSOR = HAND;
	}
    }
  else if (xmax - 3 < x && x < xmax + 3 && ymax - 3 < y && y < ymax + 3)
    {
      if (CURSOR != EXTEND_SELECT && CURSOR != BRUSH)
	{
	  TurnOnBottomRight (draw);
	  CURSOR = EXTEND_SELECT;
	}
    }
  else if (CURSOR == HAND || CURSOR == EXTEND_SELECT)
    {
      TurnOffCursor (draw);
      CURSOR = ARROW;
    }

}

void
cb_DocBtn2Down (Widget w, XEvent * event, String * params,
		Cardinal * num_params)
{
  downx = event->xbutton.x * 100 / zoom;
  downy = event->xbutton.y * 100 / zoom;

}

void
cb_DocBtn2Motion (Widget w, XEvent * event, String * params,
		  Cardinal * num_params)
{
  int i0, j0, i1, j1;

  movex = event->xbutton.x * 100 / zoom;
  movey = event->xbutton.y * 100 / zoom;

  if (movex != downx || movey != downy)
    {
      xytoij (downx, downy, &i0, &j0);
      xytoij (movex, movey, &i1, &j1);
      i0 = AW->ll - (i1 - i0);
      j0 = AW->cc - (j1 - j0);
      if (i0 < 1)
	i0 = 1;
      if (j0 < 1)
	j0 = 1;
      cbrepaint (i0, j0, Width, Height);
    }

  downx = movex;
  downy = movey;

}

void
cb_DocBtn2Up (Widget w, XEvent * event, String * params,
	      Cardinal * num_params)
{
  int i0, j0, i1, j1;

  upx = event->xbutton.x * 100 / zoom;
  upy = event->xbutton.y * 100 / zoom;

  if (upx != downx || upy != downy)
    {
      xytoij (downx, downy, &i0, &j0);
      xytoij (upx, upy, &i1, &j1);
      i0 = AW->ll - (i1 - i0);
      j0 = AW->cc - (j1 - j0);

      if (i0 < 1)
	i0 = 1;
      if (j0 < 1)
	j0 = 1;
      cbrepaint (i0, j0, Width, Height);
    }

}

void
cb_PageHome (Widget w, XEvent * event, String * params, Cardinal * num_params)
{
  cbrepaint (1, 1, Width, Height);
  if (!texthaschanged)
    {
      setcursor (1, 1);
      arrowmove = 1;
    }
}

void
cb_PageUp (Widget w, XEvent * event, String * params, Cardinal * num_params)
{
  int i1, j1, i2, j2;
  int h;
  get_visible_cells (&i1, &j1, &i2, &j2);
  h = i2 - i1 - 1;
  if (i1 - h < 1)
    h = i1 + 1;
  cbrepaint (i1 - h, -1, Width, Height);
  if (!texthaschanged)
    {
      movecursor (-h, 0);
      arrowmove = 1;
    }
}

void
cb_PageDown (Widget w, XEvent * event, String * params, Cardinal * num_params)
{
  int i1, j1, i2, j2;
  get_visible_cells (&i1, &j1, &i2, &j2);

  cbrepaint (i2 - 1, -1, Width, Height);

  if (!texthaschanged)
    {
      movecursor ((i2 - i1 - 1), 0);
      arrowmove = 1;
    }
}

void
cb_PageLeft (Widget w, XEvent * event, String * params, Cardinal * num_params)
{
  int i1, j1, i2, j2;
  int h;
  if (texthaschanged)
    return;
  get_visible_cells (&i1, &j1, &i2, &j2);
  h = j2 - j1;
  j1 -= h + 1;
  if (j1 < 1)
    j1 = 1;

  cbrepaint (-1, j1, Width, Height);
  movecursor (0, -(j2 - j1 - 2));
  arrowmove = 1;

}

void
cb_PageRight (Widget w, XEvent * event, String * params,
	      Cardinal * num_params)
{
  int i1, j1, i2, j2;
  if (texthaschanged)
    return;

  get_visible_cells (&i1, &j1, &i2, &j2);

  cbrepaint (-1, j2 - 1, Width, Height);
  movecursor (0, (j2 - j1 - 2));
  arrowmove = 1;

}

void
cb_DocBtn1Up (Widget w, XEvent * event, String * params,
	      Cardinal * num_params)
{
  char *label;
  char buf[64];
  char buf1[64];
  int x, y;
  int i, j;
  Graph *gr;
  Button *btn;

  if (Shift)
    {
      BTN1_STATE = UP;
      BTN1_WHAT = 0;
      WHAT = 0;
      return;
    }

  if (!arrowmove)
    {
      x = event->xbutton.x * 100 / zoom;
      y = event->xbutton.y * 100 / zoom;
      upx = x;
      upy = y;
      absupx = upx + CW (AW, AW->cc);
      absupy = upy + LH (AW, AW->ll);

      xytoij (x, y, &i, &j);
      upi = i;
      upj = j;
    }

  if (drawingmode)
    {
      if (drawingmode == EDITINGDRAWING)
	{
	  drawingmode = READYEDITDRAWING;
	  TurnOnPencil (draw);
	  BTN1_STATE = UP;
	  xrepaint ();
	  return;
	}

      if (drawingmode == CREATINGDRAWING)
	{
	  drawingmode = 0;
	  BTN1_STATE = UP;
	  drplot ();
	  return;
	}
      drawingmode = READYEDITDRAWING;
      TurnOnPencil (draw);
      BTN1_STATE = UP;
      return;
    }

  switch (BTN1_WHAT)
    {
    case CB_BUTTON:
      {

	btn = (Button *) getbtn (isselectingbtn - 1);

	if (upx > btn->x1 - CW (AW, AW->cc) && upy > btn->y1 - LH (AW, AW->ll)
	    && upx < btn->x2 - CW (AW, AW->cc)
	    && upy < btn->y2 - LH (AW, AW->ll))
	  {
	    button_click (btn);
	  }

	redrawbt (btn);
	copypix ();
	BTN1_STATE = UP;
	return;
      }
    case CB_CREATE_GRAPH:
      {
	CURSOR = ARROW;
	TurnOffCursor (draw);
	BTN1_STATE = UP;
	drplot ();
	break;
      }
    case CB_GRAPH:
      {
	if (CURSOR == CROSS)
	  {
	    CURSOR = ARROW;
	    TurnOffCursor (draw);
	    gr = (Graph *) getlastgra ();
	    gr->x = grx;
	    gr->y = gry;
	    gr->h = grh;
	    gr->w = grw;
	    plot ();
	    xrepaint ();
	  }
	if (CURSOR == BOTTOM_LEFT ||
	    CURSOR == BOTTOM_RIGHT ||
	    CURSOR == TOP_RIGHT ||
	    CURSOR == TOP_SIDE ||
	    CURSOR == RIGHT_SIDE ||
	    CURSOR == BOTTOM_SIDE ||
	    CURSOR == LEFT_SIDE || CURSOR == TOP_LEFT)
	  {
	    CURSOR = ARROW;
	    TurnOffCursor (draw);
	    gr = (Graph *) getlastgra ();
	    gr->x = grx;
	    gr->y = gry;
	    gr->h = grh;
	    gr->w = grw;
	    resizegra ();
	    plot ();
	    xrepaint ();
	  }
	break;
      }
    case CB_LININDEX:
      {
	if (BTN1_STATE == DOWN)
	  {
	    selectlin (i, Shift);
	    li = 0;
	    cj = 0;
	    break;
	  }
	else
	  {
	    xrepaint ();
	    cj = 0;
	    li = 0;
	    break;
	  }
      }
    case CB_COLINDEX:
      {
	if (BTN1_STATE == DOWN)
	  {
	    selectcol (j, Shift);
	    li = 0;
	    cj = 0;
	    break;
	  }
	else
	  {
	    xrepaint ();
	    cj = 0;
	    li = 0;
	    break;
	  }
      }
    case CB_CORNER:
      {
	if (BTN1_STATE == DOWN)
	  {
	    selectall ();
	    li = 0;
	    cj = 0;
	    break;
	  }
	else
	  {
	    xrepaint ();
	    cj = 0;
	    li = 0;
	    break;
	  }
      }
    case CB_GRAP:
      {
	nocursor = 0;
	numtoalpha (buf, downi, downj);
	if (TEXT_WIDGET == Letext)
	  {
	    if (ActiveCell != NULL)
	      {
		if (ActiveCell->worksheet != NULL)
		  {
		    if (EditedCell != NULL)
		      {
			if (ActiveWorksheet != EditedCell->worksheet)
			  {
			    sprintf (buf1, "%s!%s", ActiveWorksheet->Name,
				     buf);
			    ActivateWorksheet (EditedCell->worksheet);
			    insertstring (Letext, buf1);
			    xdrawcursor ();
			  }
			else

			  {
			    insertstring (Letext, buf);
			  }
		      }
		    else

		      {
			insertstring (Letext, buf);
		      }
		  }
	      }
	  }
	else

	  {
	    if (DIALOG_WIDGET != NULL)
	      XtManageChild (DIALOG_WIDGET);
	    setstring (TEXT_WIDGET, buf);
	  }

	if (downi != i || downj != j)
	  {
	    numtoalpha (buf1, i, j);

	    if (TEXT_WIDGET != Letext)
	      {
		sprintf (buf, "%s:%s", buf, buf1);
		setstring (TEXT_WIDGET, buf);
	      }
	    else
	      {
		sprintf (buf, ":%s", buf1);
		insertstring (Letext, buf);
	      }
	  }
	copypix ();
	TEXT_WIDGET = Letext;

	BTN1_WHAT = CB_CELL;
	BTN1_STATE = UP;
	nocursor = 0;
	break;
      }
    case CB_SELAREA:
      {
	TurnOffMove (draw);
	CURSOR = ARROW;
	if (downi - i || downj - j)
	  {
	    selectismoving = 0;
	    moveselect (downi, downj, i, j, 1);

	    upi += i - downi;
	    upj += j - downj;
	    downi = i;
	    downj = j;
	    break;
	  }
	else
	  {
	    resetselect (downi, downj, 1);
	    BTN1_WHAT = CB_CELL;
	  }

      }
    default:
      {
	if (CURSOR == EXTEND_SELECT)
	  {
	    int di, dj;
	    EXTEND_DIR = UNDEFINE;
	    di = i - downi;
	    dj = j - downj;
	    if (di > 0)
	      {
		copyd ();
	      }
	    else if (dj > 0)
	      {
		copyr ();
	      }

	    TurnOffCursor (draw);
	    CURSOR = ARROW;
	    break;
	  }
	ijtoxy (i, j, &x, &y);
	if (is_selecting ())
	  {
	    makebrush ();
	    TurnOffCursor (draw);
	    CURSOR = ARROW;
	    break;
	  }
	if (w == draw)
	  {
	    label = (char *) get_formula (i, j);
	    if (label != NULL)
	      setstring (Letext, label);
	    else
	      setstring (Letext, "");

	    left = 0;
	    set_visible (i, j);
	    if (is_selecting () || isselectinggra || isselectingdra)
	      {
		resetselect (0, 0, 0);
		isselectinggra = 0;
		isselectingdra = 0;
		isselectingbtn = 0;
		xrepaint ();
	      }
	    if (AW->cur_r == i && AW->cur_c == j && !texthaschanged)
	      {
		LetextGetFocus ();
		textismanaged = 1;
		texthaschanged = 1;
		xdrawcursor ();
	      }
	    else
	      {
		setcursor (i, j);
	      }
	  }
      }
    }
  BTN1_STATE = UP;
  BTN1_WHAT = 0;
  WHAT = 0;

}

static int hnumover = 0;
static int vnumover = 0;

int
popupactivecell ()
{
  int cheight, cwidth;
  int width, height;
  int i = AW->cur_r;
  int j = AW->cur_c;
  char *str = GetEntryString (Letext);
  int font = cell_getfont (ActiveCell);
  int fontw = cell_getfontw (ActiveCell);
  int fonts = cell_getfonts (ActiveCell);

  int x1 = CW (AW, j) - CW (AW, AW->cc) - 1;
  int y1 = LH (AW, i) - LH (AW, AW->ll) - 1;
  int x2, y2;

  x2 = CW (AW, j + 1 + hnumover) - CW (AW, AW->cc);
  y2 = LH (AW, i + 1 + vnumover) - LH (AW, AW->ll);
  width = x2 - x1;
  height = y2 - y1;

  cheight = gettexth (str, strlen (str), font, fontw, fonts);
  cwidth = gettextw (str, strlen (str), font, fontw, fonts) + 5;
  while (width < cwidth && hnumover < 50)
    {
      hnumover++;
      popdowncell ();
      x2 = CW (AW, j + 1 + hnumover) - CW (AW, AW->cc);
      width = x2 - x1;
    }
  while (height < cheight && vnumover < 50)
    {
      vnumover++;
      popdowncell ();
      y2 = LH (AW, i + 1 + vnumover) - LH (AW, AW->ll);
      height = y2 - y1;
    }

  setstring (ActiveMainwin->cell, GetEntryString (Letext));
  popupcell (x1 * zoom / 100, y1 * zoom / 100, width, height);
  return 0;
}

int
popdownactivecell ()
{
  hnumover = 0;
  vnumover = 0;
  popdowncell ();
  return 0;
}

void
cb_TextChange ()
{
  KeyPressed ();
}

int
KeyPressed ()
{
  if (!textismanaged)
    {
      popupactivecell ();
      textismanaged = 1;
    }

  if (texthaschanged)
    {
      set_visible (AW->cur_r, AW->cur_c);
    }
  if (textismanaged && !arrowmove)
    {

      popupactivecell ();

      texthaschanged = 1;
    }
  arrowmove = 0;
  return 0;
}

void
cb_TextReturn (Widget w, XEvent * event, String * params,
	       Cardinal * num_params)
{
  movecursor (1, 0);
  arrowmove = 1;
}

void
cb_BtnOk ()
{

  a_update ();
}

void
cb_BtnCancel ()
{
  if (textismanaged)
    {
      xdrawcell (AW->cur_r, AW->cur_c, 4);
      setstring (Letext, "");
      LetextLooseFocus ();
      textismanaged = 0;
      texthaschanged = 0;
      resetselect (0, 0, 0);
      xdrawcursor ();
      return;
    }
}

void
cb_DocBtn1Motion (Widget w, XEvent * event, String * params,
		  Cardinal * num_params)
{
  int i, j;
  int x, y;
  int dx, dy;
  Drawing *dr;

  x = event->xbutton.x * 100 / zoom;
  y = event->xbutton.y * 100 / zoom;

  movex = x;
  movey = y;
  absmovex = movex + CW (AW, AW->cc);
  absmovey = movey + LH (AW, AW->ll);
  xytoij (x, y, &i, &j);
  movei = i;
  movej = j;

  if (Shift)
    return;

  if (drawingmode)
    {
      if (drawingmode == STARTCREATEDRAWING)
	drawingmode = CREATINGDRAWING;

      if (drawingmode == CREATINGDRAWING)
	{
	  set_visible (i, j);
	  drplot ();
	  return;
	}

      if (drawingmode == EDITINGDRAWING)
	{
	  set_visible (i, j);
	  dx = x - lx;
	  dy = y - ly;

	  if (CURSOR == CROSS)
	    {
	      dr = ActiveDrawing;
	      dr->x1 += dx;
	      dr->y1 += dy;
	      dr->x2 += dx;
	      dr->y2 += dy;
	    }
	  else if (CURSOR == BOTTOM_RIGHT)
	    {
	      dr = ActiveDrawing;
	      dr->x2 += dx;
	      dr->y2 += dy;
	    }
	  else if (CURSOR == TOP_LEFT)
	    {
	      dr = ActiveDrawing;
	      dr->x1 += dx;
	      dr->y1 += dy;
	    }
	  lx = x;
	  ly = y;
	  drplot ();
	  copypix ();
	}
      return;
    }

  switch (BTN1_WHAT)
    {
    case CB_CREATE_GRAPH:
      {
	set_visible (i, j);
	drplot ();
	break;
      }
    case CB_GRAPH:
      {
	set_visible (i, j);
	dx = x - lx;
	dy = y - ly;

	if (CURSOR == ARROW)
	  {
	    TurnOnMove (draw);
	    CURSOR = CROSS;
	  }
	if (CURSOR == CROSS)
	  {
	    grx += dx;
	    gry += dy;
	    lx = x;
	    ly = y;

	    copypix ();
	    break;
	  }
	if (CURSOR == BOTTOM_LEFT)
	  {
	    grx += dx;
	    grw -= dx;
	    grh += dy;
	  }
	if (CURSOR == BOTTOM_RIGHT)
	  {
	    grw += dx;
	    grh += dy;
	  }
	if (CURSOR == TOP_LEFT)
	  {
	    grx += dx;
	    gry += dy;
	    grw -= dx;
	    grh -= dy;
	  }
	if (CURSOR == TOP_RIGHT)
	  {
	    gry += dy;
	    grw += dx;
	    grh -= dy;
	  }
	if (CURSOR == LEFT_SIDE)
	  {
	    grx += dx;
	    grw -= dx;
	  }
	if (CURSOR == RIGHT_SIDE)
	  {
	    grw += dx;
	  }
	if (CURSOR == TOP_SIDE)
	  {
	    gry += dy;
	    grh -= dy;
	  }
	if (CURSOR == BOTTOM_SIDE)
	  {
	    grh += dy;
	  }
	if (grw < 50)
	  grw = 50;
	if (grh < 40)
	  grh = 40;
	lx = x;
	ly = y;
	copypix ();

	break;
      }
    case CB_LININDEX:
      {
	if ((ly - y) * (ly - y) < 9)
	  break;
	dy = y - ly;
	ly = y;

	if (dy < 0 && LH (AW, li) + dy - LH (AW, li - 1) < 10)
	  break;

	SH (AW, li - 1, LH (AW, li) + dy - LH (AW, li - 1));
	border_repaint (CB_LININDEX);
	break;
      }

    case CB_COLINDEX:
      {
	if ((lx - x) * (lx - x) < 9)
	  break;
	dx = x - lx;
	lx = x;

	if (dx < 0 && CW (AW, cj) + dx - CW (AW, cj - 1) < 10)
	  break;

	SW (AW, cj - 1, CW (AW, cj) + dx - CW (AW, cj - 1));
	border_repaint (CB_COLINDEX);
	break;
      }
    case CB_SELAREA:
      {
	int i1, j1, i2, j2;
	int di = i - lasti;
	int dj = j - lastj;
	if (!(di) && !(dj))
	  break;
	TurnOnMove (draw);
	selectismoving = 1;
	copypix ();
	worksheet_movecursor (AW, di, dj);
	getselection (&i1, &j1, &i2, &j2);
	di = i - downi;
	dj = j - downj;
	set_visible (i, j);
	drawselectbox (i1 + di, j1 + dj, i2 + di, j2 + dj);
	break;
      }
    case CB_GRAP:
      {
	if (BTN1_STATE == DOWN)
	  {
	    copypix ();
	    drawselectbox (downi, downj, downi, downj);
	  }
	else
	  {
	    xytoij (x, y, &i, &j);
	    if (i != 0 && j != 0)

	      set_visible (i, j);
	    copypix ();
	    drawselectbox (downi, downj, i, j);
	  }
	break;
      }
    case CB_CELL:
      {

	isselectinggra = 0;
	isselectingdra = 0;
	isselectingbtn = 0;
	if (BTN1_STATE == DOWN)
	  {
	    if (CURSOR != EXTEND_SELECT)
	      {
		setcursor (downi, downj);
	      }
	    else
	      {
		int i1, j1, i2, j2, di, dj;
		di = i - downi;
		dj = j - downj;
		getselection (&i1, &j1, &i2, &j2);

		if (i1 == 0 && j1 == 0)
		  {
		    i1 = i2 = ActiveCell->r;
		    j1 = j2 = ActiveCell->c;
		  }
		if (EXTEND_DIR == DOWN && di <= 0)
		  {
		    i2 = i1;
		    di = 0;
		    EXTEND_DIR = UNDEFINE;
		  };

		if (EXTEND_DIR == RIGHT && dj <= 0)
		  {
		    j2 = j1;
		    dj = 0;
		    EXTEND_DIR = UNDEFINE;
		  };

		if (EXTEND_DIR == UNDEFINE)
		  {
		    if (di > 0)
		      {
			EXTEND_DIR = DOWN;
		      }
		    else if (dj > 0)
		      {
			EXTEND_DIR = RIGHT;
		      }
		  }

		if (EXTEND_DIR == DOWN)
		  {
		    i2 = i1 + di;
		  }
		else if (EXTEND_DIR == RIGHT)
		  {
		    j2 = j1 + dj;
		  }

		set_visible (i2, j2);
		setselection (i1, j1, i2, j2);
		return;
	      }
	  }
	else
	  {
	    xytoij (x, y, &i, &j);
	    if (i != 0 && j != 0)
	      setselection (downi, downj, i, j);
	    set_visible (i, j);
	  }
      }
    }

  BTN1_STATE = MOVE;
  lasti = i;
  lastj = j;

}

void
cb_Shift (int shift)
{
  Shift = shift;
}

void
cb_Control (int control)
{
  Control = control;
}

void
cb_DocBtn1Down (Widget w, XEvent * event, String * params,
		Cardinal * num_params)
{
  int i, j, k;
  int x, y;
  Graph *gr;
  Button *btn;

  x = event->xbutton.x * 100 / zoom;
  y = event->xbutton.y * 100 / zoom;

  downx = x;
  downy = y;
  absdownx = downx + CW (AW, AW->cc);
  absdowny = downy + LH (AW, AW->ll);

  xytoij (downx, downy, &i, &j);

  downi = i;
  downj = j;
  lasti = i;
  lastj = j;

  BTN1_STATE = DOWN;

  if (drawingmode)
    {
      if (textismanaged)
	a_update ();
      if (drawingmode == READYEDITDRAWING && ActiveDrawing != NULL)
	{
	  drawingmode = EDITINGDRAWING;
	  lx = x;
	  ly = y;
	  if (abs (ActiveDrawing->x1 - absdownx) <= 5
	      && abs (ActiveDrawing->y1 - absdowny) <= 5)
	    {
	      TurnOnTopLeft (draw);
	      CURSOR = TOP_LEFT;
	    }
	  else if (abs (ActiveDrawing->x2 - absdownx) <= 5
		   && abs (ActiveDrawing->y2 - absdowny) <= 5)
	    {
	      TurnOnBottomRight (draw);
	      CURSOR = BOTTOM_RIGHT;
	    }
	  else
	    {
	      TurnOnMove (draw);
	      CURSOR = CROSS;
	    }
	}
      return;
    }

  if (BTN1_WHAT == CB_CREATE_GRAPH)
    return;

  if (BTN1_WHAT == CB_GRAP)
    {
      nocursor = 1;
      copypix ();
      drawselectbox (downi, downj, downi, downj);
      return;
    }
  if (CURSOR == HAND)
    {
      BTN1_WHAT = CB_SELAREA;
      isselectinggra = 0;
      isselectingdra = 0;
      isselectingbtn = 0;
      return;
    }
  if (CURSOR == EXTEND_SELECT)
    {
      BTN1_WHAT = CB_CELL;
      return;
    }
  if (CURSOR == BOTTOM_LEFT ||
      CURSOR == BOTTOM_RIGHT ||
      CURSOR == TOP_RIGHT ||
      CURSOR == TOP_SIDE ||
      CURSOR == RIGHT_SIDE ||
      CURSOR == BOTTOM_SIDE || CURSOR == LEFT_SIDE || CURSOR == TOP_LEFT)
    {
      lx = downx;
      ly = downy;
      if (isselectinggra)
	BTN1_WHAT = CB_GRAPH;
      if (isselectingdra)
	BTN1_WHAT = CB_DRAWING;
      return;
    }
  if (w == colindex)
    {
      BTN1_WHAT = CB_COLINDEX;
      xytoclotherij (downx, downy, &i, &j);
      cj = j;
      lx = downx;
      isselectinggra = 0;
      isselectingdra = 0;
      isselectingbtn = 0;
      return;
    }
  if (w == linindex)
    {
      BTN1_WHAT = CB_LININDEX;
      xytoclotherij (downx, downy, &i, &j);
      li = i;
      ly = downy;
      isselectinggra = 0;
      isselectingdra = 0;
      isselectingbtn = 0;
      return;
    }
  if (w == corner)
    {
      BTN1_WHAT = CB_CORNER;
      isselectinggra = 0;
      isselectingdra = 0;
      isselectingbtn = 0;
      return;
    }
  if (textismanaged && texthaschanged)
    {

      BTN1_WHAT = CB_GRAP;

      isselectinggra = 0;
      isselectingdra = 0;
      isselectingbtn = 0;
      return;
    }
  if (textismanaged && !texthaschanged)
    {
      a_update ();
    }
  for (k = 0; k < getnumgra (); k++)
    {
      gr = (Graph *) getgra (k);
      grx = gr->x;
      gry = gr->y;
      grw = gr->w;
      grh = gr->h;
      if (downx > grx - CW (AW, AW->cc) && downy > gry - LH (AW, AW->ll) &&
	  downx < (grx - CW (AW, AW->cc)) + (grw)
	  && downy < (gry - LH (AW, AW->ll)) + (grh))
	{
	  lx = downx;
	  ly = downy;
	  resetselect (i, j, 1);
	  if (isselectingdra)
	    {
	      isselectingdra = 0;
	      xrepaint ();
	    };
	  if (isselectinggra != k + 1)
	    {
	      worksheet_select_graph (AW, gr);
	      isselectinggra = k + 1;
	      drawselect ();
	    }
	  setactivegra (k);
	  BTN1_WHAT = CB_GRAPH;
	  return;
	}
    }
  for (k = 0; k < getnumbtn (); k++)
    {
      btn = (Button *) getbtn (k);

      if (downx > btn->x1 - CW (AW, AW->cc)
	  && downy > btn->y1 - LH (AW, AW->ll)
	  && downx < btn->x2 - CW (AW, AW->cc)
	  && downy < btn->y2 - LH (AW, AW->ll))
	{

	  lx = downx;
	  ly = downy;
	  isselectingbtn = k + 1;
	  setactivebtn (k);
	  BTN1_WHAT = CB_BUTTON;
	  redrawbtdown (btn);
	  copypix ();
	  return;
	}
    }

  isselectingdra = 0;
  if (isselectinggra)
    {
      isselectinggra = 0;
      xrepaint ();
    };

  if (Shift)
    {
      extendselection (downi, downj);
    }
  else
    {
      resetselect (0, 0, 1);
      BTN1_WHAT = CB_CELL;
    }
}

void
m_paste (Widget w, XEvent * event, String * params, Cardinal * num_params)
{
  if (textismanaged && texthaschanged)
    a_update ();
  if (CLIPBOARD == CB_CELL)
    {
      paste ();
    }
  if (CLIPBOARD == CB_GRAPH)
    {
      paste ();
    }
  if (CLIPBOARD == CB_DRAWING)
    {
      paste ();
    }

  xrepaint ();
}

void
m_brush (Widget w, XEvent * event, String * params, Cardinal * num_params)
{
  if (textismanaged && texthaschanged)
    a_update ();
  setbrush (applicationcell (downi, downj, 1));
  TurnOnSpray (draw);
  CURSOR = BRUSH;

}

void
m_sum (Widget w, XEvent * event, String * params, Cardinal * num_params)
{
  char buf[128];
  char buf2[8];
  int col, startr, endr;
  char *txt;

  if (!textismanaged)
    return;
  if (AW->cur_r < 2)
    return;

  col = AW->cur_c;
  endr = AW->cur_r - 1;
  startr = endr;

  txt = (char *) get_text (startr, col);
  if (txt == NULL)
    return;

  while (startr > 0 && txt != NULL)
    {
      startr--;
      txt = (char *) get_text (startr, col);

    }
  startr++;

  sprintf (buf, "+sum (");
  numtoalpha (buf2, startr, col);
  strcat (buf, buf2);
  strcat (buf, ":");
  numtoalpha (buf2, endr, col);
  strcat (buf, buf2);
  strcat (buf, ")");

  insertstring (Letext, buf);

}

void
m_sorta (Widget w, XEvent * event, String * params, Cardinal * num_params)
{
  if (textismanaged && texthaschanged)
    a_update ();
  makesorta ();
}

void
m_sortd (Widget w, XEvent * event, String * params, Cardinal * num_params)
{
  if (textismanaged && texthaschanged)
    a_update ();
  makesortd ();
}

void
m_copy (Widget w, XEvent * event, String * params, Cardinal * num_params)
{
  if (textismanaged && texthaschanged)
    a_update ();
  if (isselectinggra == 1)
    {
      copy ();
      CLIPBOARD = CB_GRAPH;
    }
  else if (drawingmode)
    {
      copy ();
      CLIPBOARD = CB_DRAWING;
    }
  else
    {
      copy ();
      CLIPBOARD = CB_CELL;
    }
}

void
m_cut ()
{
  if (textismanaged && texthaschanged)
    a_update ();
  if (isselectinggra == 1)
    {
      cut ();
      isselectinggra = 0;
      CLIPBOARD = CB_GRAPH;
    }
  else if (drawingmode)
    {
      cut ();
      isselectingdra = 0;
      CLIPBOARD = CB_DRAWING;
    }
  else
    {
      cut ();
      CLIPBOARD = CB_CELL;
    }

  if (drawingmode && AW->nbdrawing > 0)
    worksheet_select_drawing (AW, AW->Drawings[0]);

  xrepaint ();
}

void
cb_DocDelete ()
{
  if (textismanaged && texthaschanged)
    a_update ();
  m_cut ();
}

void
m_incr ()
{
  if (textismanaged && texthaschanged)
    a_update ();
  incr ();
}

void
m_color ()
{
  if (textismanaged && texthaschanged)
    a_update ();
  color ();
}

void
m_fit ()
{
  if (textismanaged && texthaschanged)
    a_update ();
  fit ();
  xrepaint ();
}

void
m_openfac (Widget w, XtPointer client_data, XtPointer call_data)
{
}

void
m_openfile (char *filename)
{
  if (textismanaged && texthaschanged)
    a_update ();
  openfile (filename);
  setcursor (1, 1);
}

void
m_newfile (Widget w, XtPointer client_data, XtPointer call_data)
{
  if (textismanaged && texthaschanged)
    a_update ();
  new_doc (NULL);
  setcursor (1, 1);
}

void
m_addsheet (Widget w, XtPointer client_data, XtPointer call_data)
{
  if (textismanaged && texthaschanged)
    a_update ();
  workbook_addsheet (ActiveWorkbook, NULL);
  setcursor (1, 1);
}

void
m_delsheet (Widget w, XtPointer client_data, XtPointer call_data)
{
  if (textismanaged && texthaschanged)
    a_update ();
  workbook_delactivesheet (ActiveWorkbook);
}

void
m_delline (Widget w, XtPointer client_data, XtPointer call_data)
{
  if (textismanaged && texthaschanged)
    a_update ();
  delline ();
}

void
m_find (Widget w, XtPointer client_data, XtPointer call_data)
{
}
void
m_replace (Widget w, XtPointer client_data, XtPointer call_data)
{
}
void
m_goto (Widget w, XtPointer client_data, XtPointer call_data)
{
}

void
m_delcolumn (Widget w, XtPointer client_data, XtPointer call_data)
{
  if (textismanaged && texthaschanged)
    a_update ();
  delcolumn ();
}

void
m_insertline (Widget w, XtPointer client_data, XtPointer call_data)
{
  if (textismanaged && texthaschanged)
    a_update ();
  insertline ();
}

void
m_insertcolumn (Widget w, XtPointer client_data, XtPointer call_data)
{
  if (textismanaged && texthaschanged)
    a_update ();
  insertcolumn ();
}

void
m_changewin (Widget w, XtPointer client_data, XtPointer call_data)
{
  if (textismanaged && texthaschanged)
    a_update ();
  resetselect (0, 0, 0);
  Activate_workbook2 (XtName (w));

}

void
m_savefile (Widget w, XtPointer client_data, XtPointer call_data)
{
  if (textismanaged && texthaschanged)
    a_update ();
  savefile ();
}

void
m_saveasfile (Widget w, XtPointer client_data, XtPointer call_data)
{
  char *buf;
  if (textismanaged && texthaschanged)
    a_update ();
  saveasfile (buf);
}

void
m_printfile (Widget w, XtPointer client_data, XtPointer call_data)
{

}

void
m_update ()
{
}

void
m_copyd ()
{

  if (textismanaged && texthaschanged)
    a_update ();
  copyd ();
}

void
m_copyr ()
{

  if (textismanaged && texthaschanged)
    a_update ();
  copyr ();
}

void
m_createplot ()
{
  if (textismanaged && texthaschanged)
    a_update ();
  WHAT = CB_CREATE_GRAPH;
  BTN1_WHAT = CB_CREATE_GRAPH;
  drawingmode = 0;
  isselectinggra = 0;
  TurnOnCross (draw);
  return;
}

void
m_plotline ()
{

  if (textismanaged && texthaschanged)
    a_update ();
  WHAT = CB_LINE;
  TurnOnCross (draw);
  drawingmode = STARTCREATEDRAWING;
  isselectinggra = 0;
  resetselect (0, 0, 0);
  return;
}

void
m_plotrect ()
{
  if (textismanaged && texthaschanged)
    a_update ();
  WHAT = CB_RECT;
  TurnOnCross (draw);
  drawingmode = STARTCREATEDRAWING;
  isselectinggra = 0;
  resetselect (0, 0, 0);
  return;
}

void
m_plotarc ()
{
  if (textismanaged && texthaschanged)
    a_update ();
  WHAT = CB_ARC;
  TurnOnCross (draw);
  drawingmode = STARTCREATEDRAWING;
  isselectinggra = 0;
  resetselect (0, 0, 0);
  return;
}

void
m_plotcircle ()
{
  if (textismanaged && texthaschanged)
    a_update ();
  WHAT = CB_CIRCLE;
  TurnOnCross (draw);
  drawingmode = STARTCREATEDRAWING;
  isselectinggra = 0;
  resetselect (0, 0, 0);
  return;
}

void
m_plotbutton ()
{
  if (textismanaged && texthaschanged)
    a_update ();
  WHAT = CB_BUTTON;
  TurnOnCross (draw);
  drawingmode = STARTCREATEDRAWING;
  isselectinggra = 0;
  resetselect (0, 0, 0);
  return;
}

void
m_editdraw ()
{
  if (textismanaged && texthaschanged)
    a_update ();
  drawingmode = READYEDITDRAWING;
  CURSOR = EDIT_DRA;
  TurnOnPencil (draw);
  isselectinggra = 0;
  resetselect (0, 0, 0);
  worksheet_select_drawing (AW, AW->activedrawing);
}

void
m_exitdraw ()
{
  if (textismanaged && texthaschanged)
    a_update ();
  CURSOR = ARROW;
  TurnOffCursor (draw);
  drawingmode = 0;
}

int
drplot ()
{
  Graph *gr;
  Drawing *dr;
  Button *btn;
  int centerx, centery, width, height, a1, a2;
  int pi = 90 * 64;

  if (textismanaged && texthaschanged)
    a_update ();
  if (BTN1_STATE == UP || drawingmode == ENDCREATEDRAWING)
    {
      plotx = absdownx;
      ploty = absdowny;
      ploth = absupy - absdowny;
      if (ploth < 0)
	{
	  ploth = -ploth;
	  ploty = absupy;
	}
      plotw = absupx - absdownx;
      if (plotw < 0)
	{
	  plotw = -plotw;
	  plotx = absupx;
	}
      switch (WHAT)
	{
	case CB_CREATE_GRAPH:
	  {
	    gr = (Graph *) newgra ();
	    if (plotw < 50)
	      plotw = 50;
	    if (ploth < 40)
	      ploth = 40;
	    if (plotx < 0)
	      plotx = 0;
	    if (ploty < 0)
	      ploty = 0;

	    gr->x = plotx;
	    gr->y = ploty;
	    gr->w = plotw;
	    gr->h = ploth;
	    gr->pixmap = createpixmap (draw, plotw, ploth);
	    fillrectangle (dpy, gr->pixmap, 0, 0, 0, plotw, ploth, 0, 0, 0);
	    plot ();
	    copypix ();
	    break;
	  }
	case CB_LINE:
	  {
	    dr = (Drawing *) newdra ();
	    dr->x1 = absdownx;
	    dr->x2 = absupx;
	    dr->y1 = absdowny;
	    dr->y2 = absupy;
	    dr->type = CB_LINE;
	    drawline (dpy, pixmap, 1, absdownx - CW (AW, AW->cc),
		      absdowny - LH (AW, AW->ll), upx, upy, 0, 0);
	    worksheet_select_drawing (AW, dr);
	    break;
	  }
	case CB_RECT:
	  {
	    dr = (Drawing *) newdra ();
	    dr->x1 = absdownx;
	    dr->x2 = absupx;
	    dr->y1 = absdowny;
	    dr->y2 = absupy;
	    dr->type = CB_RECT;
	    drawline (dpy, pixmap, 1, absdownx - CW (AW, AW->cc),
		      absdowny - LH (AW, AW->ll), upx, absdowny - LH (AW,
								      AW->ll),
		      0, 0);
	    drawline (dpy, pixmap, 1, upx, absdowny, upx, upy, 0, 0);
	    drawline (dpy, pixmap, 1, upx, upy, absdownx - CW (AW, AW->cc),
		      upy, 0, 0);
	    drawline (dpy, pixmap, 1, absdownx - CW (AW, AW->cc), upy,
		      absdownx - CW (AW, AW->cc), absdowny - LH (AW, AW->ll),
		      0, 0);
	    worksheet_select_drawing (AW, dr);
	    break;
	  }
	case CB_BUTTON:
	  {
	    char *label;
	    btn = (Button *) newbtn ();
	    label = inputbox ("Button label:", "Button Label", "button");
	    button_setlabel (btn, label);
	    label = inputbox ("Button macro:", "Button Macro", "macro1");
	    button_setmacro (btn, label);

	    btn->x1 = absdownx;
	    btn->x2 = absupx;
	    btn->y1 = absdowny;
	    btn->y2 = absupy;
	    fillrectangle (dpy, pixmap, 3, absdownx - CW (AW, AW->cc),
			   absdowny - LH (AW, AW->ll),
			   upx - (absdownx - CW (AW, AW->cc)),
			   upy - (absdowny - LH (AW, AW->ll)), 0, 0, 0);
	    worksheet_select_button (AW, btn);
	    break;
	  }
	case CB_ARC:
	  {
	    dr = (Drawing *) newdra ();
	    dr->x1 = absdownx;
	    dr->x2 = absupx;
	    dr->y1 = absdowny;
	    dr->y2 = absupy;
	    dr->type = CB_ARC;
	    centerx = absdownx;
	    centery = absdowny;
	    width = 2 * abs (absdownx - absupx);
	    height = 2 * abs (absdowny - absupy);

	    if (absdownx <= absupx && absdowny <= absupy)
	      {
		a1 = 0;
		a2 = pi;
		centery += height / 2;
	      }
	    if (absdownx >= absupx && absdowny <= absupy)
	      {
		a1 = -pi;
		a2 = pi;
		centerx -= width / 2;
	      }
	    if (absdownx >= absupx && absdowny >= absupy)
	      {
		a1 = 2 * pi;
		a2 = pi;
		centery -= height / 2;
	      }
	    if (absdownx <= absupx && absdowny >= absupy)
	      {
		a1 = pi;
		a2 = pi;
		centerx += width / 2;
	      }
	    drawarc
	      (dpy, pixmap, 1, (centerx - width / 2) - CW (AW, AW->cc),
	       (centery - height / 2) - LH (AW, AW->ll), width, height, a1,
	       a2, 0, 0);
	    worksheet_select_drawing (AW, dr);
	    break;
	  }
	case CB_CIRCLE:
	  {
	    dr = (Drawing *) newdra ();
	    dr->x1 = absdownx;
	    dr->x2 = absupx;
	    dr->y1 = absdowny;
	    dr->y2 = absupy;
	    dr->type = CB_CIRCLE;
	    centerx = (absdownx + absupx) / 2;
	    centery = (absdowny + absupy) / 2;
	    width = abs (absdownx - absupx);
	    height = abs (absdowny - absupy);
	    drawarc
	      (dpy, pixmap, 1, (centerx - width / 2) - CW (AW, AW->cc),
	       (centery - height / 2) - LH (AW, AW->ll), width, height, 0,
	       360 * 64, 0, 0);
	    worksheet_select_drawing (AW, dr);
	    break;
	  }
	}
      WHAT = -1;
      TurnOffCross (draw);
      return 0;
    }
  else

    {
      clearnflush ();

      if (WHAT == CB_CREATE_GRAPH || WHAT == CB_GRAPH)
	{
	  drawline (dpy, XtWindow (draw), 1, absdownx - CW (AW, AW->cc),
		    absdowny - LH (AW, AW->ll), movex, absdowny - LH (AW,
								      AW->ll),
		    0, 0);
	  drawline (dpy, XtWindow (draw), 1, movex, absdowny, movex, movey, 0,
		    0);
	  drawline (dpy, XtWindow (draw), 1, movex, movey,
		    absdownx - CW (AW, AW->cc), movey, 0, 0);
	  drawline (dpy, XtWindow (draw), 1, absdownx - CW (AW, AW->cc),
		    movey, absdownx - CW (AW, AW->cc), absdowny - LH (AW,
								      AW->ll),
		    0, 0);
	}

    }

  if (drawingmode == CREATINGDRAWING || drawingmode == EDITINGDRAWING)
    {
      clearnflush ();

      switch (WHAT)
	{
	case CB_LINE:
	  {
	    drawline (dpy, XtWindow (draw), 1, absdownx - CW (AW, AW->cc),
		      absdowny - LH (AW, AW->ll), movex, movey, 0, 0);
	    break;
	  }
	case CB_RECT:
	  {
	    drawline (dpy, XtWindow (draw), 1, absdownx - CW (AW, AW->cc),
		      absdowny - LH (AW, AW->ll), movex, absdowny - LH (AW,
									AW->
									ll),
		      0, 0);
	    drawline (dpy, XtWindow (draw), 1, movex,
		      absdowny - LH (AW, AW->ll), movex, movey, 0, 0);
	    drawline (dpy, XtWindow (draw), 1, movex, movey,
		      absdownx - CW (AW, AW->cc), movey, 0, 0);
	    drawline (dpy, XtWindow (draw), 1, absdownx - CW (AW, AW->cc),
		      movey, absdownx - CW (AW, AW->cc), absdowny - LH (AW,
									AW->
									ll),
		      0, 0);
	    break;
	  }
	case CB_BUTTON:
	  {
	    fillrectangle (dpy, XtWindow (draw), 3,
			   absdownx - CW (AW, AW->cc), absdowny - LH (AW,
								      AW->ll),
			   movex - (absdownx - CW (AW, AW->cc)),
			   movey - (absdowny - LH (AW, AW->ll)), 0, 0, 0);
	    break;
	  }
	case CB_ARC:
	  {
	    absupx = movex + CW (AW, AW->cc);
	    absupy = movey + LH (AW, AW->ll);
	    centerx = absdownx;
	    centery = absdowny;
	    width = 2 * abs (absdownx - absupx);
	    height = 2 * abs (absdowny - absupy);
	    if (absdownx <= absupx && absdowny <= absupy)
	      {
		a1 = 0;
		a2 = pi;
		centery += height / 2;
	      }
	    if (absdownx >= absupx && absdowny <= absupy)
	      {
		a1 = -pi;
		a2 = pi;
		centerx -= width / 2;
	      }
	    if (absdownx >= absupx && absdowny >= absupy)
	      {
		a1 = 2 * pi;
		a2 = pi;
		centery -= height / 2;
	      }
	    if (absdownx <= absupx && absdowny >= absupy)
	      {
		a1 = pi;
		a2 = pi;
		centerx += width / 2;
	      }
	    drawarc
	      (dpy, XtWindow (draw), 1,
	       (centerx - width / 2) - CW (AW, AW->cc),
	       (centery - height / 2) - LH (AW, AW->ll), width, height, a1,
	       a2, 0, 0);

	    break;
	  }
	case CB_CIRCLE:
	  {
	    absupx = movex + CW (AW, AW->cc);
	    absupy = movey + LH (AW, AW->ll);
	    centerx = (absdownx + absupx) / 2;
	    centery = (absdowny + absupy) / 2;
	    width = abs (absdownx - absupx);
	    height = abs (absdowny - absupy);
	    drawarc
	      (dpy, XtWindow (draw), 1,
	       (centerx - width / 2) - CW (AW, AW->cc),
	       (centery - height / 2) - LH (AW, AW->ll), width, height, 0,
	       360 * 64, 0, 0);
	    break;
	  }
	}
    }
  return 0;
}

int
drawselect ()
{

  int i1, j1, i2, j2;
  Drawing *dr;

  if (selectismoving)
    return 0;

  if (drawingmode == READYEDITDRAWING || drawingmode == EDITINGDRAWING)
    {
      dr = ActiveDrawing;
      if (dr == NULL || !getnumdra ())
	{
	  return -1;
	}
      xa = dr->x1 - CW (AW, AW->cc);
      ya = dr->y1 - LH (AW, AW->ll);
      xb = dr->x2 - CW (AW, AW->cc);
      yb = dr->y2 - LH (AW, AW->ll);
      fillrectangle (dpy, XtWindow (draw), 1, xa - 4, ya - 4, 8, 8, 0, 0, 0);
      fillrectangle (dpy, XtWindow (draw), 1, xb - 4, yb - 4, 8, 8, 0, 0, 0);
      return 0;
    }

  if (isselectinggra)
    {

      xa = grx - CW (AW, AW->cc) - 4;
      ya = gry - LH (AW, AW->ll) - 4;
      xb = grx - CW (AW, AW->cc) + grw + 4;
      yb = gry - LH (AW, AW->ll) + grh + 4;
      drawline (dpy, XtWindow (draw), 1, xa, ya, xb, ya, 0, 0);
      drawline (dpy, XtWindow (draw), 1, xb, ya, xb, yb, 0, 0);
      drawline (dpy, XtWindow (draw), 1, xb, yb, xa, yb, 0, 0);
      drawline (dpy, XtWindow (draw), 1, xa, yb, xa, ya, 0, 0);
      fillrectangle (dpy, XtWindow (draw), 1, xa - 4, ya - 4, 8, 8, 0, 0, 0);
      fillrectangle (dpy, XtWindow (draw), 1, xa - 4, yb - 4, 8, 8, 0, 0, 0);
      fillrectangle (dpy, XtWindow (draw), 1, xb - 4, ya - 4, 8, 8, 0, 0, 0);
      fillrectangle (dpy, XtWindow (draw), 1, xb - 4, yb - 4, 8, 8, 0, 0, 0);
      fillrectangle (dpy, XtWindow (draw), 1, (xa + xb) / 2, ya - 4, 8, 8, 0,
		     0, 0);
      fillrectangle (dpy, XtWindow (draw), 1, (xa + xb) / 2, yb - 4, 8, 8, 0,
		     0, 0);
      fillrectangle (dpy, XtWindow (draw), 1, xa - 4, (ya + yb) / 2, 8, 8, 0,
		     0, 0);
      fillrectangle (dpy, XtWindow (draw), 1, xb - 4, (ya + yb) / 2, 8, 8, 0,
		     0, 0);

      return 0;
    }

  if (!get_select (&i1, &j1, &i2, &j2))
    return 0;

  drawselectbox (i1, j1, i2, j2);
  return 0;
}

int
drawselectbox (int i1, int j1, int i2, int j2)
{
  int x1, y1, x2, y2;
  double fact = 100.0 / zoom;
  if (i2 >= i1 && j2 >= j1)
    {
      i2++;
      j2++;
    }
  if (i2 < i1 && j2 >= j1)
    {
      j2++;
      i1++;
    }
  if (i2 >= i1 && j2 < j1)
    {
      i2++;
      j1++;
    }
  if (i2 < i1 && j2 < j1)
    {
      i1++;
      j1++;
    }
  ijtoxy (i1, j1, &x1, &y1);
  ijtoxy (i2, j2, &x2, &y2);

  if (x1 < 0)
    x1 = 0;
  if (x2 < 0)
    x2 = 0;
  if (y1 < 0)
    y1 = 0;
  if (y2 < 0)
    y2 = 0;
  if (x1 > Width * fact)
    x1 = Width * fact;
  if (x2 > Width * fact)
    x2 = Width * fact;
  if (y1 > Height * fact)
    y1 = Height * fact;
  if (y2 > Height * fact)
    y2 = Height * fact;

  setlineattributes (dpy, 1, 3, LineDoubleDash, CapButt, JoinBevel);
  drawline (dpy, XtWindow (draw), 1, x1, y1, x2, y1, 0, 0);
  drawline (dpy, XtWindow (draw), 1, x2, y1, x2, y2 - 4, 0, 0);
  drawline (dpy, XtWindow (draw), 1, x2 - 4, y2, x1, y2, 0, 0);
  drawline (dpy, XtWindow (draw), 1, x1, y2, x1, y1, 0, 0);
  setlineattributes (dpy, 1, 3, LineSolid, CapButt, JoinBevel);
  drawline (dpy, XtWindow (draw), 1, x2 - 2, y2 + 2, x2 + 2, y2 + 2, 0, 0);
  drawline (dpy, XtWindow (draw), 1, x2 + 2, y2 - 2, x2 + 2, y2 + 4, 0, 0);
  setlineattributes (dpy, 1, 1, LineSolid, CapButt, JoinBevel);

  xdrawcursor ();
  selX0 = x1;
  selY0 = y1;
  return 0;
}

int
cbsetup (Widget _toplevel, Widget drawzone, Dimension width, Dimension height,
	 Widget _colindex, Widget _linindex, Widget _text, Widget _info,
	 Widget _corner)
{
  int D;

  Height = height;
  Width = width;

  win = XtWindow (drawzone);
  dpy = XtDisplay (drawzone);
  screen = DefaultScreen (dpy);
  draw = drawzone;

  setXparam (dpy, win, _toplevel);

  Letext = _text;
  TEXT_WIDGET = Letext;
  linindex = _linindex;
  colindex = _colindex;
  toplevel = _toplevel;
  info = _info;
  corner = _corner;

  D = DefaultDepth (dpy, DefaultScreen (dpy));
  pixmap = createpixmap (draw, width, height);
  setbackpix (draw, pixmap);
  setzoom (100);
  linpixmap = createpixmap (linindex, 40, height);
  colpixmap = createpixmap (colindex, width, 20);
  cornerpixmap = createpixmap (corner, 40, 20);
  setbackpix (linindex, linpixmap);
  setbackpix (colindex, colpixmap);
  setbackpix (corner, cornerpixmap);

  init_application ();
  xrepaint ();
  setbackpix (draw, pixmap);
  XtSetKeyboardFocus (toplevel, draw);
  return 0;
}

int
cbrepaint (int l, int c, Dimension width, Dimension height)
{
  double L, C;

  if (c < 1 || c > AW->nbcol)
    c = AW->cc;;
  if (l < 1 || l > AW->nblin)
    l = AW->ll;
  if (AW->cc == c && AW->ll == l)
    return 0;
  lx -= CW (AW, c) - CW (AW, AW->cc);
  ly -= LH (AW, l) - LH (AW, AW->ll);

  AW->cc = c;
  AW->ll = l;

  C = c;
  L = l;
  setscrollv (L);
  setscrollh (C);

  xrepaint ();
  return 0;
}

int
setstring (Widget w, char *str)
{

  if (w == Letext)
    {
      setstring (ActiveMainwin->cell, str);
    }

  SetEntryString (w, str);
  return 0;
}

int
inform (char *str)
{
  return mainwin_inform (str);
}

int
insertstring (Widget w, char *str)
{
  XawTextBlock text;
  long p = XawTextGetInsertionPoint (w);
  text.firstPos = 0;
  text.length = strlen (str);
  text.ptr = str;
  XawTextReplace (w, p, p, &text);
  XawTextSetInsertionPoint (w, p + text.length);

  KeyPressed ();
  return 0;
}

int
moveinsertionpoint (Widget w, int delta)
{
  XawTextPosition p = XawTextGetInsertionPoint (w);
  XawTextSetInsertionPoint (w, p + delta);
  return 0;
}

int
xytoij (int x, int y, int *i, int *j)
{
  int jj = 0;
  int x1, x2, y1, y2;
  x += CW (AW, AW->cc);
  y += LH (AW, AW->ll);
  get_visible (&x1, &x2, &y1, &y2);
  while (x > CW (AW, jj))
    {
      jj++;
    }
  *j = jj - 1;
  jj = 0;
  while (y > LH (AW, jj))
    {
      jj++;
    }
  *i = jj - 1;

  if (*i < 1)
    *i = 1;
  if (*j < 1)
    *j = 1;
  if (*i > AW->nblin - 1)
    *i = AW->nblin - 1;
  if (*j > AW->nbcol - 1)
    *j = AW->nbcol - 1;

  if (x < x1 + CW (AW, 1))
    *j = 1;
  if (y < y1 + LH (AW, 1))
    *i = 1;

  return 1;
}

int
ijtoxy (int i, int j, int *x, int *y)
{
  if (AW == NULL)
    return -1;
  if (i < 0 || i > AW->nblin || j < 0 || j > AW->nbcol)
    return -1;

  *x = CW (AW, j);
  *y = LH (AW, i);

  *x -= CW (AW, AW->cc);
  *y -= LH (AW, AW->ll);

  return 1;
}

int
absijtoxy (int i, int j, int *x, int *y)
{
  if (AW == NULL)
    return -1;
  if (i < 0 || i > AW->nblin || j < 0 || j > AW->nbcol)
    return -1;

  *x = CW (AW, j);
  *y = LH (AW, i);

  return 1;
}

int
xytoclotherij (int x, int y, int *i, int *j)
{
  int jj = 0;
  x += CW (AW, AW->cc);
  y += LH (AW, AW->ll);
  while (x > CW (AW, jj))
    {
      jj++;
    }
  *j = jj - 1;
  if (CW (AW, jj) - x < x - CW (AW, jj - 1))
    *j = jj;
  jj = 0;
  while (y > LH (AW, jj))
    {
      jj++;
    }
  *i = jj - 1;
  if (LH (AW, jj) - y < y - LH (AW, jj - 1))
    *i = jj;

  if (*i < 1)
    *i = 1;
  if (*j < 1)
    *j = 1;
  if (*i > AW->nblin - 1)
    *i = AW->nblin - 1;
  if (*j > AW->nbcol - 1)
    *j = AW->nbcol - 1;

  return 1;
}

int
xfit (int ii, int jj, char *name)
{
  int cheight, cwidth;
  int i, j;

  if (AW == NULL)
    return -1;
  if (i < 0 || i > AW->nblin || j < 0 || j > AW->nbcol)
    return -1;

  if (LH (AW, ii + 1) - LH (AW, ii) < cheight)
    {
      SH (AW, ii, cheight);
    }
  if (CW (AW, jj + 1) - CW (AW, jj) < cwidth)
    {
      SW (AW, jj, cwidth);
    }
  return 0;
}

int
xfitcolumn (int ii, int jj)
{
  int cwidth;

  int font, fontw, fonts;
  char *name;
  Cell *cell;

  if (AW == NULL)
    return -1;
  if (ii < 0 || ii > AW->nblin || jj < 0 || jj > AW->nbcol)
    return -1;

  cell = (Cell *) applicationcell (ii, jj, -1);
  if (cell == NULL)
    return -1;
  name = cell_gettext (cell);
  if (name == NULL)
    return 0;
  if (strlen (name) < 1)
    return 0;

  font = cell_getfont (cell);
  fontw = cell_getfontw (cell);
  fonts = cell_getfonts (cell);

  cwidth = gettextw (name, strlen (name), font, fontw, fonts) + 2;

  if (CW (AW, jj + 1) - CW (AW, jj) < cwidth)
    {
      SW (AW, jj, cwidth);
    }
  return 0;
}

int
xfitline (int ii, int jj)
{
  int cheight;

  int font, fontw, fonts;
  char *name;
  Cell *cell;

  if (AW == NULL)
    return -1;
  if (ii < 0 || ii > AW->nblin || jj < 0 || jj > AW->nbcol)
    return -1;

  cell = (Cell *) applicationcell (ii, jj, -1);
  if (cell == NULL)
    return -1;

  name = cell_gettext (cell);
  if (name == NULL)
    return 0;
  if (strlen (name) < 1)
    return 0;

  font = cell_getfont (cell);
  fontw = cell_getfontw (cell);
  fonts = cell_getfonts (cell);

  cheight = gettexth (name, strlen (name), font, fontw, fonts) + 2;

  if (LH (AW, ii + 1) - LH (AW, ii) < cheight)
    {
      SW (AW, ii, cheight);
    }
  return 0;
}

Widget getdrawzone ()
{
  return draw;
}

int
get_visible (int *x1, int *x2, int *y1, int *y2)
{
  *x1 = CW (AW, AW->cc);
  *y1 = LH (AW, AW->ll);
  *x2 = *x1 + Width * 100 / zoom;
  *y2 = *y1 + Height * 100 / zoom;
  return 0;
}

int
get_visible_cells (int *i1, int *j1, int *i2, int *j2)
{
  *i1 = AW->ll;
  *j1 = AW->cc;
  *i2 = *i1;
  *j2 = *j1;

  while (LH (AW, *i2) < LH (AW, AW->ll) + Height * 100 / zoom
	 && *i2 < AW->nblin)
    {
      (*i2)++;
    }

  while (CW (AW, *j2) < CW (AW, AW->cc) + Width * 100 / zoom
	 && *j2 < AW->nbcol)
    {
      (*j2)++;
    }
  return 0;
}

int
set_visible (int i, int j)
{
  int c, l;
  double fact = 100.0 / zoom;

  if (AW == NULL)
    return -1;
  if (i < 0 || i > AW->nblin || j < 0 || j > AW->nbcol)
    return -1;

  c = AW->cc;
  l = AW->ll;

  if (i >= l && j >= c && (CW (AW, c) + Width * fact) > (CW (AW, j + 1))
      && (LH (AW, l) + Height * fact) > (LH (AW, i + 1)))
    return 0;

  if (c > j)
    c = j;
  else
    while ((CW (AW, c) + Width * fact) < (CW (AW, j + 1)) && c < AW->nbcol)
      c++;

  if (l > i)
    l = i;
  else
    while ((LH (AW, l) + Height * fact) < (LH (AW, i + 1)) && l < AW->nblin)
      l++;

  if (l < 1)
    l = 1;
  if (c < 1)
    c = 1;

  cbrepaint (l, c, Width, Height);
  return 0;
}

int
coldrag (double c)
{
  int C;
  C = c;

  cbrepaint (-1, C, Width, Height);
  return 0;
}

int
lindrag (double l)
{
  int L;
  L = l;

  cbrepaint (L, -1, Width, Height);
  return 0;
}

void
cb_ZoomChange (Widget w, XtPointer cd, XEvent * event, Boolean * ctd)
{

}

int
zoomin ()
{
  zoom += 10;
  setzoom (zoom);
  xrepaint ();
  return 0;
}

int
zoomout ()
{
  zoom -= 10;
  setzoom (zoom);
  xrepaint ();
  return 0;
}

int
cb_setzoom (int z)
{
  Graph *gr;
  int i;

  if (zoom == z)
    return 0;
  setzoom (z);
  ActiveWorksheet->zoom = z;
  zoom = z;

  for (i = 0; i < getnumgra (); i++)
    {
      gr = (Graph *) getgra (i);
      gr->zoom = z;
      resizegra (i);
    }
  redrawgra ();
  xrepaint ();
  return 0;
}

int
resizegra ()
{
  Graph *gr;
  gr = (Graph *) getlastgra ();
  freepixmap (draw, gr->pixmap);
  gr->pixmap = createpixmap (draw,
			     gr->w * gr->zoom / 100, gr->h * gr->zoom / 100);
  fillrectangle (dpy, gr->pixmap, 0, 0, 0, gr->w, gr->h, 0, 0, 0);
  return 0;
}

int
redrawbt (Button * btn)
{
  int x1, x2, y1, y2;

  x1 = btn->x1 - CW (AW, AW->cc);
  y1 = btn->y1 - LH (AW, AW->ll);
  x2 = btn->x2 - CW (AW, AW->cc);
  y2 = btn->y2 - LH (AW, AW->ll);
  drawbuttonup (dpy, pixmap, x1, y1, x2, y2, button_getlabel (btn),
		CW (AW, AW->cc), LH (AW, AW->ll));
  return 0;
}

int
redrawbtdown (Button * btn)
{
  int x1, x2, y1, y2;
  char *label;

  x1 = btn->x1 - CW (AW, AW->cc);
  y1 = btn->y1 - LH (AW, AW->ll);
  x2 = btn->x2 - CW (AW, AW->cc);
  y2 = btn->y2 - LH (AW, AW->ll);

  drawline (dpy, pixmap, 2, x1, y1, x2 - 1, y1, CW (AW, AW->cc),
	    LH (AW, AW->ll));
  drawline (dpy, pixmap, 2, x1, y1, x1, y2, CW (AW, AW->cc), LH (AW, AW->ll));

  drawline (dpy, pixmap, 1, x1 + 1, y1 + 1, x2 - 1, y1 + 1, CW (AW, AW->cc),
	    LH (AW, AW->ll));
  drawline (dpy, pixmap, 1, x1 + 1, y1 + 2, x1 + 1, y2 - 1, CW (AW, AW->cc),
	    LH (AW, AW->ll));

  drawline (dpy, pixmap, 1, x2, y1, x2, y2, CW (AW, AW->cc), LH (AW, AW->ll));
  drawline (dpy, pixmap, 1, x1, y2, x2, y2, CW (AW, AW->cc), LH (AW, AW->ll));
  drawline (dpy, pixmap, 0, x2 - 1, y1, x2 - 1, y2 - 1, CW (AW, AW->cc),
	    LH (AW, AW->ll));
  drawline (dpy, pixmap, 0, x1 + 1, y2 - 1, x2 - 1, y2 - 1, CW (AW, AW->cc),
	    LH (AW, AW->ll));
  fillrectangle (dpy, pixmap, 2, x1 + 2, y1 + 2, x2 - x1 - 3, y2 - y1 - 3,
		 CW (AW, AW->cc), LH (AW, AW->ll), 1);
  label = button_getlabel (btn);
  drawstring (dpy, pixmap, 1, x1 + 10, (y1 + y2) / 2, label, strlen (label),
	      CW (AW, AW->cc), LH (AW, AW->ll));
  return 0;
}

int
redrawdr (Drawing * dr)
{
  int centerx, centery, width, height, a1, a2;
  int pi = 90 * 64;
  int x1, x2, y1, y2;

  x1 = dr->x1 - CW (AW, AW->cc);
  y1 = dr->y1 - LH (AW, AW->ll);
  x2 = dr->x2 - CW (AW, AW->cc);
  y2 = dr->y2 - LH (AW, AW->ll);

  switch (dr->type)
    {

    case CB_LINE:
      {

	drawline (dpy, pixmap, 1, x1, y1, x2, y2, CW (AW, AW->cc),
		  LH (AW, AW->ll));
	break;
      }
    case CB_RECT:
      {
	drawline (dpy, pixmap, 1, x1, y1, x2, y1, CW (AW, AW->cc),
		  LH (AW, AW->ll));
	drawline (dpy, pixmap, 1, x2, y1, x2, y2, CW (AW, AW->cc),
		  LH (AW, AW->ll));
	drawline (dpy, pixmap, 1, x2, y2, x1, y2, CW (AW, AW->cc),
		  LH (AW, AW->ll));
	drawline (dpy, pixmap, 1, x1, y2, x1, y1, CW (AW, AW->cc),
		  LH (AW, AW->ll));
	break;
      }
    case CB_BUTTON:
      {
	fillrectangle (dpy, pixmap, 3, x1, y1, x2 - x1, y2 - y1,
		       CW (AW, AW->cc), LH (AW, AW->ll), 0);
	break;
      }
    case CB_ARC:
      {
	centerx = x1;
	centery = y1;
	width = 2 * abs (x1 - x2);
	height = 2 * abs (y1 - y2);

	if (x1 <= x2 && y1 <= y2)
	  {
	    a1 = 0;
	    a2 = pi;
	    centery += height / 2;
	  }
	if (x1 >= x2 && y1 <= y2)
	  {
	    a1 = -pi;
	    a2 = pi;
	    centerx -= width / 2;
	  }
	if (x1 >= x2 && y1 >= y2)
	  {
	    a1 = 2 * pi;
	    a2 = pi;
	    centery -= height / 2;
	  }
	if (x1 <= x2 && y1 >= y2)
	  {
	    a1 = pi;
	    a2 = pi;
	    centerx += width / 2;
	  }
	drawarc
	  (dpy, pixmap, 1, centerx - width / 2, centery - height / 2, width,
	   height, a1, a2, CW (AW, AW->cc), LH (AW, AW->ll));
	break;
      }
    case CB_CIRCLE:
      {
	centerx = (x1 + x2) / 2;
	centery = (y1 + y2) / 2;
	width = abs (x1 - x2);
	height = abs (y1 - y2);
	drawarc
	  (dpy, pixmap, 1, centerx - width / 2, centery - height / 2, width,
	   height, 0, 360 * 64, CW (AW, AW->cc), LH (AW, AW->ll));
	break;
      }
    }
  return 0;
}

void
cb_grap (Widget w, Widget dialog)
{
  TEXT_WIDGET = w;
  DIALOG_WIDGET = dialog;
  BTN1_WHAT = CB_GRAP;
}

Pixmap createpixmapforgraph (Graph * gr)
{
  gr->pixmap = createpixmap (draw, gr->w, gr->h);
  fillrectangle (dpy, gr->pixmap, 0, 0, 0, gr->w, gr->h, 0, 0, 0);
  return gr->pixmap;

}

Pixmap getpixmap (Graph * gr, int *x, int *y, int *w, int *h)
{
  *x = gr->x;
  *y = gr->y;

  *w = gr->w;
  *h = gr->h;
  return gr->pixmap;
}

int
border_repaint (int n)
{
  int i, j;
  if (BatchMode)
    return 0;

  setfont2 (0, 1, 0, 3);
  {
    desactivate_zoom ();
    fillrectangle (XtDisplay (corner), cornerpixmap, 2, 0, 0, 40, 20, 0, 0,
		   0);
    drawline (XtDisplay (corner), cornerpixmap, 3, 0, 0, 0, 20, 0, 0);
    drawline (XtDisplay (corner), cornerpixmap, 1, 1, 0, 1, 20, 0, 0);
    drawline (XtDisplay (corner), cornerpixmap, 3, 0, 0, 40, 0, 0, 0);
    drawline (XtDisplay (corner), cornerpixmap, 1, 1, 1, 40, 1, 0, 0);

    drawline (XtDisplay (corner), cornerpixmap, 0, 2, 2, 39, 2, 0, 0);
    drawline (XtDisplay (corner), cornerpixmap, 0, 2, 2, 2, 18, 0, 0);
    drawline (XtDisplay (corner), cornerpixmap, 3, 2, 19, 39, 19, 0, 0);
    drawline (XtDisplay (corner), cornerpixmap, 3, 39, 2, 39, 18, 0, 0);
    reactivate_zoom ();
    clearwin (corner);
  }
  if (n == 0 || n == CB_LININDEX)
    {
      desactivate_zoom ();
      fillrectangle (XtDisplay (linindex), linpixmap, 2, 0, 0, 40, Height, 0,
		     0, 0);
      drawline (XtDisplay (linindex), linpixmap, 3, 0, 0, 0, Height, 0, 0);
      drawline (XtDisplay (linindex), linpixmap, 1, 1, 0, 1, Height, 0, 0);

      reactivate_zoom ();
      for (i = AW->ll; LH (AW, i) - LH (AW, AW->ll) < Height * 100 / zoom;
	   i++)
	xdrawlincell (i, 0, 0);
      clearwin (linindex);
    }
  if (n == 0 || n == CB_COLINDEX)
    {
      desactivate_zoom ();
      fillrectangle (XtDisplay (colindex), colpixmap, 2, 0, 0, Width, 20, 0,
		     0, 0);
      drawline (XtDisplay (colindex), colpixmap, 3, 0, 0, Width, 0, 0, 0);
      drawline (XtDisplay (colindex), colpixmap, 1, 0, 1, Width, 1, 0, 0);

      reactivate_zoom ();
      for (j = AW->cc; CW (AW, j) - CW (AW, AW->cc) < Width * 100 / zoom; j++)
	xdrawcolcell (0, j, 0);
      clearwin (colindex);
    }
  if (n == CB_COLINDEX || n == CB_LININDEX)
    {
      xdrawcursor ();
    }
  return 0;
}

int
xrepaint ()
{
  int i, j;
  border_repaint (0);
  paint = 1;

  fillrectangle (dpy, pixmap, 0, 0, 0, Width * 100 / zoom,
		 Height * 100 / zoom, 0, 0, 0);

  for (i = AW->ll + 1; LH (AW, i) - LH (AW, AW->ll) < Height * 100 / zoom;
       i++)
    drawline (dpy, pixmap, 2, 0, LH (AW, i) - LH (AW, AW->ll),
	      Width * 100 / zoom, LH (AW, i) - LH (AW, AW->ll), 0, 0);
  for (j = AW->cc + 1; CW (AW, j) - CW (AW, AW->cc) < Width * 100 / zoom; j++)
    drawline (dpy, pixmap, 2, CW (AW, j) - CW (AW, AW->cc), 0,
	      CW (AW, j) - CW (AW, AW->cc), Height * 100 / zoom, 0, 0);

  if (AW->maxcol != NULL)
    for (i = AW->ll;
	 i <= AW->maxrow
	 && LH (AW, i) - LH (AW, AW->ll) < Height * 100 / zoom; i++)
      {
	for (j = AW->cc;
	     j <= AW->maxcol[i]
	     && CW (AW, j) - CW (AW, AW->cc) < Width * 100 / zoom; j++)
	  xdrawcell (i, j, 4);
      }

  redrawdra ();
  redrawbtn ();
  copypix ();
  paint = 0;

  xdrawcursor ();
  return 0;
}

void
event_Repaint (Widget w, XtPointer cd, XEvent * event, Boolean * ctd)
{
  xrepaint ();
}

void
cb_Repaint (Widget w, XtPointer data, XEvent * event)
{
  xrepaint ();
}

void
cb_redrawdra ()
{
  redrawdra ();
}

int
redrawdra ()
{
  Drawing *gr;
  int i;

  int numdra;

  if (ActiveWorksheet == NULL)
    {
      fprintf (stderr, "ActiveWorksheet =NULL\n");
      return -1;
    }

  numdra = ActiveWorksheet->nbdrawing;
  for (i = 0; i < numdra; i++)
    {
      gr = (Drawing *) getdra (i);
      if (gr != NULL)
	redrawdr (gr);
      else
	fprintf (stderr, "dr =NULL\n");

    }
  return 0;
}

void
cb_redrawgra ()
{
  redrawgra ();
}

int
redrawgra ()
{
  Graph *gr;
  int i;
  int numgra;

  if (ActiveWorksheet == NULL)
    {
      fprintf (stderr, "ActiveWorksheet =NULL\n");
      return -1;
    }
  numgra = ActiveWorksheet->nbgraph;

  for (i = 0; i < numgra; i++)
    {
      gr = (Graph *) getgra (i);
      if (gr != NULL)
	newplot (gr);

    }
  return 0;

}

int
redrawbtn ()
{
  Button *gr;
  int i;
  int numbtn;

  if (ActiveWorksheet == NULL)
    {
      fprintf (stderr, "ActiveWorksheet =NULL\n");
      return -1;
    }
  numbtn = ActiveWorksheet->nbbutton;

  for (i = 0; i < numbtn; i++)
    {

      gr = (Button *) getbtn (i);
      if (gr != NULL)
	redrawbt (gr);
      else
	fprintf (stderr, "btn =NULL\n");

    }
  paint = 0;
  return 0;
}

static int incopypix = 0;

void
copypix ()
{
  Graph *gr;
  int i;
  int numgra;
  if (incopypix)
    return;
  incopypix = 1;

  if (ActiveWorksheet == NULL)
    {
      fprintf (stderr, "ActiveWorksheet =NULL\n");
      return;
    }

  numgra = ActiveWorksheet->nbgraph;

  for (i = 0; i < numgra; i++)
    {
      gr = (Graph *) getgra (i);
      if (gr != NULL)
	{
	  if (gr->pixmap == 0)
	    {
	      creategraphpixmap (gr);
	    }
	  if (gr->pixmap != 0)
	    {
	      copyarea (gr->pixmap, pixmap, 1, 0, 0,
			gr->w * gr->zoom / 100, gr->h * gr->zoom / 100,
			(gr->x - CW (AW, AW->cc)) * gr->zoom / 100,
			(gr->y - LH (AW, AW->ll)) * gr->zoom / 100);
	    }
	}
    }

  clearnflush ();
  drawselect ();
  xdrawcursor ();
  incopypix = 0;
}

int
cb_resize (int width, int height)
{
  freepixmap (draw, pixmap);
  freepixmap (linindex, linpixmap);
  freepixmap (colindex, colpixmap);

  Height = height - 20;
  Width = width - 40;
  pixmap = createpixmap (draw, width - 40, height - 20);
  setbackpix (draw, pixmap);

  linpixmap = createpixmap (linindex, 40, height - 20);
  colpixmap = createpixmap (colindex, width - 40, 20);

  setbackpix (linindex, linpixmap);
  setbackpix (colindex, colpixmap);
  xrepaint ();
  return 0;
}

static int numover = 0;

int
checkwidth (char *name, int width, Cell * cell, int r, int c, int type)
{
  int cwidth;
  int nchar;
  int i, j;
  int color, just, font, fontw, fonts;
  int delta;
  char *formula = NULL;
  Cell *cl;

  if (name == NULL)
    return 0;
  nchar = strlen (name);
  if (nchar < 1)
    return 0;

  color = cell_getfg (cell);
  just = cell_getjust (cell);
  font = cell_getfont (cell);
  fontw = cell_getfontw (cell);
  fonts = cell_getfonts (cell);

  cwidth = gettextw (name, nchar, font, fontw, fonts);
  if (type == 2 || type == 3)
    delta = CW (AW, c + 1 + numover) - CW (AW, c + 1);
  else
    delta = 0;

  if (cwidth > width - 2 + delta && (type != 2 && type != 3))
    {

      i = r;
      j = c;
      cl = (Cell *) applicationcell (i, j + 1, -1);
      if (cl != NULL)
	formula = cl->formula;
      else
	formula = NULL;
      while (((cwidth > width - 2 + delta && formula == NULL) ||
	      (cwidth > width - 2 + delta && (type == 2 || type == 3))) &&
	     (width < 1000))
	{
	  j++;
	  if (type == 2 || type == 3)
	    numover++;
	  delta = CW (AW, j + 1) - CW (AW, c + 1);
	  cl = (Cell *) applicationcell (i, j + 1, -1);
	  if (cl != NULL)
	    formula = cl->formula;
	  else
	    formula = NULL;
	}

    }

  return width + delta;

}

int
xdrawstring (char *name, int x, int y, int width, int widthmax, int height,
	     int type, Cell * cell)
{
  int textx, texty;

  int cheight, cwidth;
  int nchar;
  int color, just, font, fontw, fonts;

  if (name == NULL)
    return 0;
  nchar = strlen (name);
  if (nchar < 1)
    return 0;

  font = cell_getfont (cell);
  fontw = cell_getfontw (cell);
  fonts = cell_getfonts (cell);

  cheight = gettexth (name, nchar, font, fontw, fonts);
  cwidth = gettextw (name, nchar, font, fontw, fonts);
  if (height < (cheight + 2))
    return width;

  while (cwidth > widthmax - 2 && nchar > 0)
    {
      nchar--;
      cwidth = gettextw (name, nchar, font, fontw, fonts);
    }
  if (nchar < 1)
    return 0;

  color = cell_getfg (cell);
  just = cell_getjust (cell);

  if (type == 3)
    just = 0;

  if (just == 0)
    textx = x + 1;
  if (just == 1)
    textx = x + width / 2 - cwidth / 2;
  if (just == 2)
    textx = x + width - cwidth;
  if (textx < x + 1)
    textx = x + 1;
  texty = y + height / 2 + cheight / 2 + 1;

  setfont2 (color, font, fontw, fonts);
  drawstring (XtDisplay (draw), pixmap, color, textx, texty, name, nchar,
	      CW (AW, AW->cc), LH (AW, AW->ll));

  return 0;
}

int
xdrawrrect (int x, int y, int width, int height, int type, Cell * cell)
{

  int bg = cell_getbg (cell);
  int fg = cell_getfg (cell);

  if (bg < 0)
    bg = 0;
  if (fg < 0)
    fg = 5;
  switch (type)
    {
    case 0:
      {
	fillrectangle (dpy, pixmap, bg, x + 1, y + 1, width - 1, height - 1,
		       CW (AW, AW->cc), LH (AW, AW->ll), 0);
	break;
      }
    case 4:
      {
	fillrectangle (dpy, pixmap, 2, x, y, width, height, CW (AW, AW->cc),
		       LH (AW, AW->ll), 0);
	fillrectangle (dpy, pixmap, bg, x + 1, y + 1, width - 1, height - 1,
		       CW (AW, AW->cc), LH (AW, AW->ll), 1);
	break;
      }
    case 7:
      {
	fillrectangle (dpy, pixmap, 2, x, y, width, height, 0, 0, 0);
	fillrectangle (dpy, pixmap, bg, x + 1, y + 1, width - 1, height - 1,
		       CW (AW, AW->cc), LH (AW, AW->ll), 0);
	break;
      }
    case 5:
      {
	if (bg != 0 && bg != 45)
	  fillrectangle (dpy, pixmap, bg, x + 1, y + 1, width - 1, height - 1,
			 CW (AW, AW->cc), LH (AW, AW->ll), 5);
	break;
      }
    case 2:
      {
	fillrectangle (dpy, pixmap, bg, x + 2, y + 2, width - 3, height - 3,
		       CW (AW, AW->cc), LH (AW, AW->ll), 0);
	break;
      }
    case 3:
      {
	fillrectangle (dpy, pixmap, bg, x + 2, y + 2, width - 3, height - 3,
		       CW (AW, AW->cc), LH (AW, AW->ll), 0);
	break;
      }
    case 6:
      {
	fillrectangle (dpy, pixmap, bg, x + 2, y + 2, width - 3, height - 3,
		       CW (AW, AW->cc), LH (AW, AW->ll), 0);
	break;
      }
    }

  if (cell_gettop (cell))
    {
      drawline (dpy, pixmap, fg, x, y, x + width, y, CW (AW, AW->cc),
		LH (AW, AW->ll));
    }
  if (cell_getbot (cell))
    {
      drawline (dpy, pixmap, fg, x + width, y + height, x, y + height,
		CW (AW, AW->cc), LH (AW, AW->ll));
    }
  if (cell_getrig (cell))
    {
      drawline (dpy, pixmap, fg, x + width, y, x + width, y + height,
		CW (AW, AW->cc), LH (AW, AW->ll));
    }
  if (cell_getlef (cell))
    {
      drawline (dpy, pixmap, fg, x, y, x, y + height, CW (AW, AW->cc),
		LH (AW, AW->ll));
    }
  return 0;
}

int
xdrawcursor ()
{
  int x1, x2, y1, y2;
  int i, j;
  int d = 4;
  if (ActiveWorksheet == NULL)
    return 0;
  if (ActiveCell == NULL)
    return 0;
  if (BatchMode)
    return 0;

  if (nocursor)
    return 0;
  i = ActiveWorksheet->cur_r;
  j = ActiveWorksheet->cur_c;

  x1 = CW (AW, j) - CW (AW, AW->cc);

  y1 = LH (AW, i) - LH (AW, AW->ll);

  x2 = CW (AW, j + 1) - CW (AW, AW->cc);
  y2 = LH (AW, i + 1) - LH (AW, AW->ll);

  if (texthaschanged)
    {
      popupactivecell ();
    }
  else
    {
      popdownactivecell ();
      if (!selectismoving)
	copypix ();

      if (is_selecting ())
	d = 0;
      setlineattributes (dpy, 1, 2, LineSolid, CapProjecting, JoinBevel);
      drawline (dpy, XtWindow (draw), 5, x1, y1, x2, y1, CW (AW, AW->cc),
		LH (AW, AW->ll));
      drawline (dpy, XtWindow (draw), 5, x2 + 1, y1, x2 + 1, y2 - d,
		CW (AW, AW->cc), LH (AW, AW->ll));
      drawline (dpy, XtWindow (draw), 5, x2 - d, y2 + 1, x1, y2 + 1,
		CW (AW, AW->cc), LH (AW, AW->ll));
      drawline (dpy, XtWindow (draw), 5, x1, y2 + 1, x1, y1, CW (AW, AW->cc),
		LH (AW, AW->ll));
      {
	if (!is_selecting ())
	  {
	    drawline (dpy, XtWindow (draw), 1, x2 - 1, y2 + 2, x2 + 2, y2 + 2,
		      0, 0);
	    drawline (dpy, XtWindow (draw), 1, x2 + 2, y2 - 1, x2 + 2, y2 + 1,
		      0, 0);
	  }
      }
      setlineattributes (dpy, 1, 1, LineSolid, CapButt, JoinBevel);
    }

  return 0;
}

int
xdrawlincell (int i, int j, int color)
{
  char label[8];
  int height;
  int top;
  int act = 0;

  if (i <= 0)
    return -1;
  if (i > ActiveWorksheet->nblin)
    return -1;
  height = (LH (AW, i + 1) - LH (AW, i)) * zoom / 100 - 1;
  top = (LH (AW, i) - LH (AW, AW->ll)) * zoom / 100 + 1;
  sprintf (label, "%d", i);
  desactivate_zoom ();

/*haut */ drawline (XtDisplay (linindex), linpixmap, 0, 2, top, 38, top, 0,
		      0);

  if (AW != NULL)
    if (AW->cur_r == i)
      act = 1;

  fillrectangle (XtDisplay (linindex), linpixmap, 2, 2, top + 1, 37,
		 height - 1, 0, 0, 0);

  if (act)
    {
      drawline (XtDisplay (linindex), linpixmap, 1, 2, (top + height), 38,
		(top + height), 0, 0);
      /*bas-in */ drawline (XtDisplay (linindex), linpixmap, 3, 3,
			    (top + height) - 1, 37, (top + height) - 1, 0, 0);
      /*gauch */ drawline (XtDisplay (linindex), linpixmap, 0, 2, 1 + top, 2,
			   (top + height) - 1, 0, 0);
      /*droi */ drawline (XtDisplay (linindex), linpixmap, 1, 39, top, 39,
			  (top + height), 0, 0);
      /*droi-in */ drawline (XtDisplay (linindex), linpixmap, 3, 38, top + 1,
			     38, (top + height) - 1, 0, 0);
      /*droi-in-top */ drawline (XtDisplay (linindex), linpixmap, 2, 38, top,
				 38, top, 0, 0);
    }
  else
    {
      drawline (XtDisplay (linindex), linpixmap, 3, 2, (top + height), 38,
		(top + height), 0, 0);
      /*gauch */ drawline (XtDisplay (linindex), linpixmap, 2, 2, top + 1, 2,
			   (top + height) - 1, 0, 0);
      /*droi */ drawline (XtDisplay (linindex), linpixmap, 3, 39, top, 39,
			  (top + height), 0, 0);
    }
  reactivate_zoom ();

  setfont2 (0, 1, 0, 3);
  drawstring (XtDisplay (linindex), linpixmap, 1, 1 + 3 * 100 / zoom,
	      (top + height - 4) * 100 / zoom, label, strlen (label), 0, 0);
  return 0;
}

int
xdrawcolcell (int i, int j, int color)
{
  char label[4];
  int width;
  int left;
  int act = 0;
  if (j <= 0)
    return -1;
  if (j > ActiveWorksheet->nbcol)
    return -1;
  width = CW (AW, j + 1) - CW (AW, j);
  left = CW (AW, j) - CW (AW, AW->cc);

  if (j <= 26)
    {
      label[0] = 'A' + j - 1;
      label[1] = '\0';
    }
  else
    {
      label[0] = 'A' + (j - 1) / 26 - 1;
      label[1] = 'A' + j - 26 * ((j - 1) / 26) - 1;
      label[2] = '\0';
    }

  if (AW != NULL)
    if (AW->cur_c == j)
      act = 1;

  desactivate_zoom ();
  fillrectangle (XtDisplay (colindex), colpixmap, 2, left * zoom / 100 + 1, 2,
		 width * zoom / 100 - 1, 17, 0, 0, 0);

  if (act)
    {
      drawline (XtDisplay (colindex), colpixmap, 0, left * zoom / 100 + 1, 2,
		left * zoom / 100 + 1, 19, 0, 0);
      drawline (XtDisplay (colindex), colpixmap, 1,
		(left + width) * zoom / 100, 2, (left + width) * zoom / 100,
		19, 0, 0);
      drawline (XtDisplay (colindex), colpixmap, 1, left * zoom / 100 + 1, 19,
		(left + width) * zoom / 100 - 1, 19, 0, 0);
      drawline (XtDisplay (colindex), colpixmap, 0, left * zoom / 100 + 1, 2,
		(left + width) * zoom / 100 - 1, 2, 0, 0);
      drawline (XtDisplay (colindex), colpixmap, 3, left * zoom / 100 + 1, 18,
		(left + width) * zoom / 100 - 1, 18, 0, 0);
      drawline (XtDisplay (colindex), colpixmap, 2, left * zoom / 100 + 1, 18,
		left * zoom / 100 + 1, 18, 0, 0);
      drawline (XtDisplay (colindex), colpixmap, 3,
		(left + width) * zoom / 100 - 1, 3,
		(left + width) * zoom / 100 - 1, 18, 0, 0);
    }
  else
    {
      drawline (XtDisplay (colindex), colpixmap, 0, left * zoom / 100 + 1, 2,
		left * zoom / 100 + 1, 19, 0, 0);
      drawline (XtDisplay (colindex), colpixmap, 3,
		(left + width) * zoom / 100, 2, (left + width) * zoom / 100,
		19, 0, 0);
      drawline (XtDisplay (colindex), colpixmap, 3, left * zoom / 100 + 1, 19,
		(left + width) * zoom / 100 - 1, 19, 0, 0);
      drawline (XtDisplay (colindex), colpixmap, 2, left * zoom / 100 + 2, 2,
		(left + width) * zoom / 100 - 1, 2, 0, 0);
    }
  reactivate_zoom ();

  setfont2 (0, 1, 0, 3);
  drawstring (XtDisplay (colindex), colpixmap, 1, left + width / 2 - 5,
	      15 * 100 / zoom, label, strlen (label), 0, 0);
  return 0;
}

int
xdrawcell (int i, int j, int type)
{
  Cell *cell = (Cell *) applicationcell (i, j, -1);
  return xdrawcell2 (cell, type);
}

int
xdrawcell2 (Cell * cell, int type)
{
  int left;
  int top;
  int width;
  int widthmax;
  int height;
  char *buf = NULL;
  int i, j;

  if (cell == NULL)
    return -1;

  i = cell->r;
  j = cell->c;

  if (type == 2)
    numover = 0;

  if (cell == NULL && (type == 4 || type == 5))
    return 0;

  if (type == DRAW_FORCED)
    numover = 0;

  width = CW (AW, j + 1) - CW (AW, j);
  left = CW (AW, j) - CW (AW, AW->cc);
  height = LH (AW, i + 1) - LH (AW, i);
  top = LH (AW, i) - LH (AW, AW->ll);

  if (cell != NULL)
    buf = (char *) cell_gettext (cell);

  if (type == 2 && cell != NULL)
    buf = (char *) cell_getformula (cell);

  if (type == 3)
    {
      buf = (char *) GetEntryString (Letext);
    }

  widthmax = width;
  if (buf != NULL)
    {
      if (type == 3 || type == 2 || (cell_getrig (cell) == 0))
	widthmax = checkwidth (buf, width, cell, i, j, type);
      if (type == 2)
	{
	  left += 2;
	  top += 2;
	  width -= 4;
	  height -= 4;
	}

    }
  xdrawrrect (left, top, widthmax, height, type, cell);

  if (buf != NULL)
    xdrawstring (buf, left, top, width, widthmax, height, type, cell);

  if (!paint && BTN1_STATE != MOVE)
    cleararea (dpy, win, left, top, widthmax + 1, height + 1, False);

  return 0;
}

int
creategraphpixmap (Graph * gr)
{
  deletegraphpixmap (gr);
  gr->pixmap = createpixmap (draw, gr->w, gr->h);
  fillrectangle (dpy, gr->pixmap, 0, 0, 0, gr->w, gr->h, 0, 0, 0);
  return 0;
}

int
deletegraphpixmap (Graph * gr)
{
  if (gr)
    if (gr->pixmap)
      {
	freepixmap (draw, gr->pixmap);
	gr->pixmap = 0;
      }
  return 0;
}
