/*
    Copyright (C) 2000  Ulric Eriksson <ulric@siag.nu>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston,
    MA 02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <limits.h>
#include <sys/stat.h>
#include <unistd.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xmu/Xmu.h>

#include "xutil.h"

#define TEST_MONO 0

/* this is needed if we include vanilla Athena headers */
#ifndef XtNshadowWidth
#define XtNshadowWidth "shadowWidth"
#endif


/* color allocation stuff snarfed from the xmagick library */
#ifndef USHRT_MAX
#define USHRT_MAX 65535
#endif

static int rgb_shift = 10;

typedef struct xmagick_color {
	XColor color;
	unsigned long rgb;
} xmagick_color;

static xmagick_color known_colors[128];
static int ncolors, maxcolors = 128, lastcolor;

void xawm_free_colors(void)
{
	free(known_colors);
/*	known_colors = NULL;
*/
	ncolors = 0;
	lastcolor = -1;
}

void xawm_alloc_color(Display *display, Colormap colormap, XColor *color)
{
	int i, j, lower, upper;
	int cr, cg, cb, d;
	unsigned long rgb;
	cr = color->red >> rgb_shift;
	cg = color->green >> rgb_shift;
	cb = color->blue >> rgb_shift;
	rgb = (cr << 16) + (cg << 8) + cb;

	if (lastcolor != -1 && known_colors[lastcolor].rgb == rgb) {
		color->pixel = known_colors[lastcolor].color.pixel;
		return;
	}
	lower = 0;
	upper = ncolors-1;
	while (lower <= upper) {
		i = (lower+upper)/2;
		d = rgb-known_colors[i].rgb;
		if (d == 0) {
			lastcolor = i;
			color->pixel = known_colors[i].color.pixel;
			return;
		} else if (d > 0) {
			lower = i+1;
		} else {
			upper = i-1;
		}
	}
	if (ncolors >= maxcolors) {
		color->pixel = 0;
		return;
	}
	color->flags = DoRed|DoGreen|DoBlue;
	if (XAllocColor(display, colormap, color) == 0) {
		int dr, dg, db, bd = INT_MAX, best = 0;
		for (j = 0; j < ncolors; j++) {
			XColor c = known_colors[j].color;
			dr = cr - (c.red >> rgb_shift);
			dg = cg - (c.green >> rgb_shift);
			db = cb - (c.blue >> rgb_shift);
			d = dr*dr+dg*dg+db*db;
			if (d < bd) {
				bd = d;
				best = j;
			}
		}
		color->pixel = known_colors[best].color.pixel;
	}
	if (known_colors[upper].rgb > rgb) i = upper;
	else i = lower;
	if (i < 0) i = 0;
	if (i > ncolors) i = ncolors;
	for (j = ncolors; j > i; j--) {
		known_colors[j] = known_colors[j-1];
	}
	known_colors[i].color = *color;
	known_colors[i].rgb = rgb;
	known_colors[i].color.pixel = color->pixel;
	lastcolor = i;
	ncolors++;
}

void xawm_query_color(Display *display, Colormap colormap, XColor *color)
{
	int i;

	for (i = 0; i < ncolors; i++) {
		if (known_colors[i].color.pixel == color->pixel) {
			color->red = known_colors[i].color.red;
			color->green = known_colors[i].color.green;
			color->blue = known_colors[i].color.blue;
			return;
		}
	}
	if (i >= maxcolors) {
		color->red = 0;
		color->green = 0;
		color->blue = 0;
		return;
	}
	XQueryColor(display, colormap, color);
	known_colors[i].color = *color;
	ncolors++;
}

        /* General Utility:  Allocate a foreground GC */
GC
_XawAllocFgGC(Widget w, Pixel fg, Font font)
{
    XGCValues   gcv ;
    XtGCMask    valuemask, dontcare;
 
    gcv.foreground = fg ;
    gcv.font = font ;
    gcv.graphics_exposures = False ;
 
    valuemask = GCForeground|GCGraphicsExposures ;
    dontcare =  GCBackground|GCStipple|GCTile|
                GCGraphicsExposures|GCDashOffset|GCDashList|GCArcMode|
                GCTileStipXOrigin|GCTileStipYOrigin ;
 
    if( font != None ) valuemask |= GCFont ;
    else dontcare |= GCFont ;
 
    return XtAllocateGC(w, 0, valuemask, &gcv, 0L, dontcare) ;
}
 
 
 
 
        /* General Utility:  Allocate an "insensitive" GC */
GC
_XawAllocGreyGC(Widget w, Pixel fg, Font font, int cnt, int be_nice_to_cmap)
/* cnt is contrast: 0-100 */
{
    XGCValues   gcv ;
    XtGCMask    valuemask, dontcare;
 
    gcv.graphics_exposures = False ;
 
    if( TEST_MONO || w->core.depth == 1 || be_nice_to_cmap ||
        !_XawAllocGrayPixel(w, fg, cnt, &gcv.foreground) ) {
      gcv.foreground = fg ;
      gcv.background = w->core.background_pixel ;
      gcv.fill_style = FillOpaqueStippled ;
      gcv.stipple = XmuCreateStippledPixmap(XtScreen(w), 1L,0L,1) ;
 
      valuemask = GCForeground|GCBackground|GCFillStyle|
                  GCStipple|GCGraphicsExposures ;
      dontcare =  GCTile|GCDashOffset|GCDashList|GCArcMode|
                  GCTileStipXOrigin|GCTileStipYOrigin ;
    } else {
      valuemask = GCForeground|GCGraphicsExposures ;
      dontcare =  GCBackground|GCStipple|GCTile|
                  GCDashOffset|GCDashList|GCArcMode|
                  GCTileStipXOrigin|GCTileStipYOrigin ;
    }
 
    gcv.font = font ;
 
    if( font != None ) valuemask |= GCFont ;
    else dontcare |= GCFont ;
 
    return XtAllocateGC(w, 0, valuemask, &gcv, 0L, dontcare) ;
}
 
 
/* cnt is contrast: 0-100 */
/* pixel is returned pixel value */
 
Status _XawAllocGrayPixel(Widget w, Pixel fg, int cnt, Pixel *pixel)
{
    Colormap cmap = w->core.colormap ;
    XColor fgc, bgc, greyc ;
 
    fgc.pixel = fg ;
    bgc.pixel = w->core.background_pixel ;
    XQueryColor(XtDisplay(w),cmap, &fgc) ;
    XQueryColor(XtDisplay(w),cmap, &bgc) ;
    greyc.red =   (fgc.red * cnt +   bgc.red * (100-cnt)) / 100 ;
    greyc.green = (fgc.green * cnt + bgc.green * (100-cnt)) / 100 ;
    greyc.blue =  (fgc.blue * cnt +  bgc.blue * (100-cnt)) / 100 ;
    greyc.flags = DoRed|DoGreen|DoBlue ;
    if( !XAllocColor(XtDisplay(w),cmap, &greyc) )
      return False ;
 
    *pixel = greyc.pixel ;
    return True ;
}
