# Copyright (c) 2001-2004 Twisted Matrix Laboratories.
# See LICENSE for details.


"""
Lookup a name using multiple resolvers.

Future Plans: This needs someway to specify which resolver answered
the query, or someway to specify (authority|ttl|cache behavior|more?)

@author: U{Jp Calderone<mailto:exarkun@twistedmatrix.com>}
"""

from twisted.internet import defer, interfaces
from twisted.names import dns
from zope.interface import implements
import common

class FailureHandler:
    def __init__(self, resolver, query, timeout):
        self.resolver = resolver
        self.query = query
        self.timeout = timeout


    def __call__(self, failure):
        # AuthoritativeDomainErrors should halt resolution attempts
        failure.trap(dns.DomainError, defer.TimeoutError, NotImplementedError)
        return self.resolver(self.query, self.timeout)


class ResolverChain(common.ResolverBase):
    """Lookup an address using multiple C{IResolver}s"""

    implements(interfaces.IResolver)


    def __init__(self, resolvers):
        common.ResolverBase.__init__(self)
        self.resolvers = resolvers


    def _lookup(self, name, cls, type, timeout):
        q = dns.Query(name, type, cls)
        d = self.resolvers[0].query(q, timeout)
        for r in self.resolvers[1:]:
            d = d.addErrback(
                FailureHandler(r.query, q, timeout)
            )
        return d


    def lookupAllRecords(self, name, timeout = None):
        d = self.resolvers[0].lookupAllRecords(name, timeout)
        for r in self.resolvers[1:]:
            d = d.addErrback(
                FailureHandler(r.lookupAllRecords, name, timeout)
            )
        return d
