<?php
/*********************************************************************************
 * The contents of this file are subject to the TimeTrex Public License Version
 * 1.1.0 ("License"); You may not use this file except in compliance with the
 * License. You may obtain a copy of the License at http://www.TimeTrex.com/TPL
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.  See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * All copies of the Covered Code must include on each user interface screen:
 *    (i) the "Powered by TimeTrex" logo and
 *    (ii) the TimeTrex copyright notice
 * in the same form as they appear in the distribution.  See full license for
 * requirements.
 *
 * The Original Code is: TimeTrex Open Source
 * The Initial Developer of the Original Code is TimeTrex Payroll Services
 * Portions created by TimeTrex are Copyright (C) 2004-2007 TimeTrex Payroll Services;
 * All Rights Reserved.
 *
 ********************************************************************************/
/*
 * $Revision: 1246 $
 * $Id: TTPDF.class.php 1246 2007-09-14 23:47:42Z ipso $
 * $Date: 2007-09-14 16:47:42 -0700 (Fri, 14 Sep 2007) $
 */
require_once(Environment::getBasePath() .'/classes/tcpdf/tcpdf.php');
//require_once(Environment::getBasePath() .'/classes/tcpdf/barcode/barcode.php');
//require_once(Environment::getBasePath() .'/classes/tcpdf/barcode/c128aobject.php');

class TTPDF extends tcpdf {

	/**
	 * This method allows printing text with line breaks. They can be automatic (as soon as the text reaches the right border of the cell) or explicit (via the \n character). As many cells as necessary are output, one below the other.<br />
	 * Text can be aligned, centered or justified. The cell block can be framed and the background painted.
	 * @param float $w Width of cells. If 0, they extend up to the right margin of the page.
	 * @param float $h Height of cells.
	 * @param string $txt String to print
	 * @param mixed $border Indicates if borders must be drawn around the cell block. The value can be either a number:<ul><li>0: no border (default)</li><li>1: frame</li></ul>or a string containing some or all of the following characters (in any order):<ul><li>L: left</li><li>T: top</li><li>R: right</li><li>B: bottom</li></ul>
	 * @param string $align Allows to center or align the text. Possible values are:<ul><li>L or empty string: left align</li><li>C: center</li><li>R: right align</li><li>J: justification (default value)</li></ul>
	 * @param int $fill Indicates if the cell background must be painted (1) or transparent (0). Default value: 0.
	 * @param int $ln Indicates where the current position should go after the call. Possible values are:<ul><li>0: to the right</li><li>1: to the beginning of the next line [DEFAULT]</li><li>2: below</li></ul>
	 * @param int $min_h Specifies the minimum vertical height of the cell (or cells).  Default value: 0
	 * @param int $max_h Specifies the maximum vertical height of the cell (or cells).  Default value: PHP_INT_MAX (usually 2147483647)
	 * @since 1.3
	 * @see SetFont(), SetDrawColor(), SetFillColor(), SetTextColor(), SetLineWidth(), Cell(), Write(), SetAutoPageBreak()
	 *
	 * @note This function overrides TCPDF::MultiCell and adds the parameters min_h and max_h.
	 */
	public function MultiCell($w, $h, $txt, $border=0, $align='J', $fill=0, $ln=1, $min_h = 0, $max_h = NULL) {

		// save current position
		$prevx = $this->x;
		$prevy = $this->y;
		$total_h = 0;

		if( $max_h === NULL ) {
			$max_h = defined( 'PHP_INT_MAX' ) ? PHP_INT_MAX : 2147483647;
		}

		//Output text with automatic or explicit line breaks
		$cw = &$this->CurrentFont['cw'];

		if($w == 0) {
			$w = $this->w - $this->rMargin - $this->x;
		}

		$wmax = ($w - 2 * $this->cMargin);

		$s = str_replace("\r", '', $txt); // remove carriage returns
		$nb = strlen($s);

		$b=0;
		if($border) {
			if($border==1) {
				$border='LTRB';
				$b='LRT';
				$b2='LR';
			}
			else {
				$b2='';
				if(strpos($border,'L')!==false) {
					$b2.='L';
				}
				if(strpos($border,'R')!==false) {
					$b2.='R';
				}
				$b=(strpos($border,'T')!==false) ? $b2.'T' : $b2;
			}
		}
		$sep=-1;
		$i=0;
		$j=0;
		$l=0;
		$ns=0;
		$nl=1;
		while($i<$nb && $total_h + $h <= $max_h) {
			//Get next character
			$c = $s{$i};
			if(preg_match("/[\n]/u", $c)) {
				//Explicit line break
				if($this->ws > 0) {
					$this->ws = 0;
					$this->_out('0 Tw');
				}
				if( $total_h + $h <= $max_h ) {
					$this->Cell($w, $h, substr($s, $j, $i-$j), $b, 2, $align, $fill);
					$total_h += $h;
				}
				$i++;
				$sep=-1;
				$j=$i;
				$l=0;
				$ns=0;
				$nl++;
				if($border and $nl==2) {
					$b = $b2;
				}
				continue;
			}
			if(preg_match("/[ ]/u", $c)) {
				$sep = $i;
				$ls = $l;
				$ns++;
			}

			$l = $this->GetStringWidth(substr($s, $j, $i-$j));

			if($l > $wmax) {
				//Automatic line break
				if($sep == -1) {
					if($i == $j) {
						$i++;
					}
					if($this->ws > 0) {
						$this->ws = 0;
						$this->_out('0 Tw');
					}
					if( $total_h + $h <= $max_h ) {
					  $this->Cell($w, $h, substr($s, $j, $i-$j), $b, 2, $align, $fill);
					  $total_h += $h;
					}
				}
				else {
					if($align=='J') {
						$this->ws = ($ns>1) ? ($wmax-$ls)/($ns-1) : 0;
						$this->_out(sprintf('%.3f Tw', $this->ws * $this->k));
					}
					if( $total_h + $h <= $max_h ) {
						$this->Cell($w, $h, substr($s, $j, $sep-$j), $b, 2, $align, $fill);
						$total_h += $h;
					}
					$i = $sep + 1;
				}
				$sep=-1;
				$j=$i;
				$l=0;
				$ns=0;
				$nl++;
				if($border AND ($nl==2)) {
					$b=$b2;
				}
			}
			else {
				$i++;
			}
		}
		//Last chunk
		if($this->ws>0) {
			$this->ws=0;
			$this->_out('0 Tw');
		}
		if($border and is_int(strpos($border,'B'))) {
			$b.='B';
		}
		$remaining_h = $total_h < $min_h ? $min_h - $total_h : $h;
		if( $total_h + $remaining_h >= $max_h ) {
			$remaining_h = $max_h - $total_h;
		}
		$this->Cell($w, $remaining_h, substr($s, $j, $i-$j), $b, 2, $align, $fill);
		$total_h += $remaining_h;

		// move cursor to specified position
		// since: 2007-03-03
		 if($ln == 1) {
			// go to the beginning of the next line
			$this->x = $this->lMargin;
		} elseif($ln == 0) {
			// go to the top-right of the cell
			$this->y = $prevy;
			$this->x = $prevx + $w;
		} elseif($ln == 2) {
			// go to the bottom-left of the cell
			$this->x = $prevx;
		}

	}

	function rotate($angle, $x = -1, $y = -1) {
		if ($x == -1) {
			$x = $this->x;
		}
		if ($y == -1) {
			$y = $this->y;
		}
		if ($this->angle != 0) {
			$this->_out('Q');
		}
		$this->angle = $angle;
		if ($angle != 0) {
			$angle = deg2rad(-$angle);
			$c = cos($angle);
			$s = sin($angle);
			$cx = $x * $this->k;
			$cy = ($this->h - $y) * $this->k;
			$this->_out(sprintf('q %.5f %.5f %.5f %.5f %.2f %.2f cm 1 0 0 1 %.2f %.2f cm',
								   $c,$s,-$s,$c,$cx,$cy,-$cx,-$cy));
		}
	}

	protected function _freadint($f) {
		//Read a 4-byte integer from file
		$a=unpack('Ni',fread($f,4));

		//Fixed bug in PHP v5.2.1 and less where it is returning a huge negative number.
		//See: http://ca3.php.net/manual/en/function.unpack.php
		//If you are trying to make unpack 'N' work with unsigned long on 64 bit machines, you should take a look to this bug:
		//http://bugs.php.net/bug.php?id=40543
		$b = sprintf("%b", $a['i']); // binary representation
		if(strlen($b) == 64){
			$new = substr($b, 33);
			$a['i'] = bindec($new);
		}
		return $a['i'];
	}

	//TCPDF oddly enough defines standard header/footers, instead of disabling them
	//in every script, just override them as blank here.
	function header() {
		return TRUE;
	}
	function footer() {
		return TRUE;
	}
}

?>
