/*
 * status.c: handles the status line updating, etc for IRCII 
 *
 * Written By Michael Sandrof
 *
 * Copyright(c) 1990 
 *
 * See the COPYRIGHT file, or do a HELP IRCII COPYRIGHT 
 * $Id: status.c,v 1.4 2000/07/01 14:03:05 edwards Exp $
 */


#include "teknap.h"
#include "struct.h"

#include "ircterm.h"
#include "list.h"
#include "server.h"
#include "status.h"
#include "vars.h"
#include "input.h"
#include "window.h"
#include "screen.h"
#include "output.h"
#include "ircaux.h"
#include "napster.h"
#define MY_BUFFER 120

extern char *ltoa (long);

static	char	*convert_format (Window *, char *, int);
static	char	*status_nickname (Window *);
static	char	*status_query_nick (Window *);
static	char	*status_right_justify (Window *);
static	char	*status_channel (Window *);
static	char	*status_server (Window *);
static	char	*status_insert_mode (Window *);
static	char	*status_overwrite_mode (Window *);
static	char	*status_user0 (Window *);
static	char	*status_user1 (Window *);
static	char	*status_user2 (Window *);
static	char	*status_user3 (Window *);
static	char	*status_user4 (Window *);
static	char	*status_user5 (Window *);
static	char	*status_user6 (Window *);
static	char	*status_user7 (Window *);
static	char	*status_user8 (Window *);
static	char	*status_user9 (Window *);
static	char	*status_user10 (Window *);
static	char	*status_user11 (Window *);
static	char	*status_user12 (Window *);
static	char	*status_user13 (Window *);
static	char	*status_user14 (Window *);
static	char	*status_user15 (Window *);
static	char	*status_user16 (Window *);
static	char	*status_user17 (Window *);
static	char	*status_user18 (Window *);
static	char	*status_user19 (Window *);
static	char	*status_version (Window *);
static	char	*status_clock (Window *);
static	char	*status_hold_lines (Window *);
static	char	*status_window (Window *);
static	char	*status_refnum (Window *);
static	char	*status_null_function (Window *);
static	char	*status_notify_windows (Window *);
static	char	*convert_sub_format (char *, char, char *);
static	char	*status_cpu_saver_mode (Window *);
static	char	*status_position (Window *);
static	char	*status_scrollback (Window *);
static	char	*status_percent (Window *);
static	char	*status_hold(Window *);
static	char	*status_server_stats(Window *);
static	char	*napster_updown(Window *);
static	char	*napster_shared(Window *);
static	char	*napster_download(Window *);
static	char	*napster_load_share(Window *);
static	char	*status_eta(Window *);
static	char	*status_usercount(Window *);
static	char	*status_cloak(Window *);
static	char	*status_lag(Window *);
static	char	*status_topic(Window *);
static	char	*status_sharedir(Window *);

/*
 * This is how we store status line expandos.
 */
struct status_formats {
	int	map;
	char 	key;
	char	*(*callback_function)(Window *);
	char	**format_var;
	int	format_set;
};


char	
	*hold_lines_format = NULL,
	*channel_format = NULL,
	*notify_format = NULL,
	*cpu_saver_format = NULL,
	*nick_format = NULL,
	*query_format = NULL,
	*server_format = NULL,
	*lag_format = NULL,
	*clock_format = NULL,
	*stats_format = NULL,
	*mode_format = NULL; /* not used yet */
	
/*
 * This is the list of possible expandos.  Note that you should not use
 * the '{' character, as it would be confusing.  It is already used for 
 * specifying the map.
 */
struct status_formats status_expandos[] = {
{ 0, 'A', napster_updown,	NULL,			-1 },
{ 0, 'B', status_hold_lines,     &hold_lines_format,	STATUS_HOLD_LINES_VAR },
{ 0, 'C', status_channel,        &channel_format,	STATUS_CHANNEL_VAR },
{ 0, 'D', napster_shared,	NULL,			-1 },
{ 0, 'E', status_scrollback,     NULL, 			-1 },
{ 0, 'F', status_notify_windows, &notify_format,	STATUS_NOTIFY_VAR },
{ 0, 'G', napster_load_share,	NULL,			-1 },
{ 0, 'H', status_hold,		 NULL,			-1 },
{ 0, 'I', status_insert_mode,    NULL,			-1 },
{ 0, 'J', status_cpu_saver_mode, &cpu_saver_format,	STATUS_CPU_SAVER_VAR },
{ 0, 'L', status_lag,		 NULL,			-1 },
{ 0, 'N', status_nickname,	 &nick_format,		STATUS_NICK_VAR },
{ 0, 'O', status_overwrite_mode, NULL,			-1 },
{ 0, 'P', status_position,       NULL,			-1 },
{ 0, 'Q', status_query_nick,     &query_format,		STATUS_QUERY_VAR },
{ 0, 'R', status_refnum,         NULL, 			-1 },
{ 0, 'S', status_server,         &server_format,     	STATUS_SERVER_VAR },
{ 0, 'T', status_clock,          &clock_format,      	STATUS_CLOCK_VAR },
{ 0, 'd', napster_download,	NULL,			-1 },
{ 0, 'c', status_usercount,	NULL,			-1 },
{ 0, 'e', status_eta,		NULL,			-1 },
{ 0, 'h', status_cloak,		NULL,			-1 },
{ 0, 's', status_server_stats,	 &stats_format,		STATUS_STATS_VAR },
{ 0, 't', status_topic,		NULL,			-1 },
{ 0, 'u', status_sharedir,	NULL,			-1 },
{ 0, 'U', status_user0,		 NULL, 			-1 },
{ 0, 'V', status_version,	 NULL, 			-1 },
{ 0, 'W', status_window,	 NULL, 			-1 },
{ 0, 'X', status_user1,		 NULL, 			-1 },
{ 0, 'Y', status_user2,		 NULL, 			-1 },
{ 0, 'Z', status_user3,		 NULL, 			-1 },
{ 0, '%', status_percent,	 NULL, 			-1 },
{ 0, '>', status_right_justify,	 NULL, 			-1 },
{ 0, '0', status_user0,		 NULL, 			-1 },
{ 0, '1', status_user1,		 NULL, 			-1 },
{ 0, '2', status_user2,		 NULL, 			-1 },
{ 0, '3', status_user3,		 NULL, 			-1 },
{ 0, '4', status_user4,		 NULL, 			-1 },
{ 0, '5', status_user5,		 NULL, 			-1 },
{ 0, '6', status_user6,		 NULL, 			-1 },
{ 0, '7', status_user7,		 NULL, 			-1 },
{ 0, '8', status_user8,		 NULL, 			-1 },
{ 0, '9', status_user9,		 NULL, 			-1 },
{ 1, '0', status_user10,	 NULL, 			-1 },
{ 1, '1', status_user11,	 NULL, 			-1 },
{ 1, '2', status_user12,	 NULL, 			-1 },
{ 1, '3', status_user13,	 NULL, 			-1 },
{ 1, '4', status_user14,	 NULL, 			-1 },
{ 1, '5', status_user15,	 NULL, 			-1 },
{ 1, '6', status_user16,	 NULL, 			-1 },
{ 1, '7', status_user17,	 NULL, 			-1 },
{ 1, '8', status_user18,	 NULL, 			-1 },
{ 1, '9', status_user19,	 NULL, 			-1 },
};

#define NUMBER_OF_EXPANDOS (sizeof(status_expandos) / sizeof(struct status_formats))


/*
 * convert_sub_format: This is used to convert the formats of the
 * sub-portions of the status line to a format statement specially designed
 * for that sub-portions.  convert_sub_format looks for a single occurence of
 * %c (where c is passed to the function). When found, it is replaced by "%s"
 * for use is a sprintf.  All other occurences of % followed by any other
 * character are left unchanged.  Only the first occurence of %c is
 * converted, all subsequence occurences are left unchanged.  This routine
 * mallocs the returned string. 
 */
static	char	* convert_sub_format(char *format, char c, char *padded)
{
	char	buffer[BIG_BUFFER_SIZE + 1];
	static	char	bletch[] = "%% ";
	char	*ptr = NULL;
	int	dont_got_it = 1;
	
	if (format == NULL)
		return (NULL);
	*buffer = (char) 0;
	memset(buffer, 0, sizeof(buffer));
	while (format)
	{
#if 0
/wset status_mode +#%+%
#endif
		if ((ptr = (char *) strchr(format, '%')) != NULL)
		{
			*ptr = (char) 0;
			strmcat(buffer, format, BIG_BUFFER_SIZE);
			*(ptr++) = '%';
			if ((*ptr == c)/* && dont_got_it*/)
			{
				dont_got_it = 0;
				if (*padded)
				{
					strmcat(buffer, "%", BIG_BUFFER_SIZE);
					strmcat(buffer, padded, BIG_BUFFER_SIZE);
					strmcat(buffer, "s", BIG_BUFFER_SIZE);
				}
				else
					strmcat(buffer, "%s", BIG_BUFFER_SIZE);
			}
			else if (*ptr == '<')
			{
				char *s = ptr + 1;
				while(*ptr && *ptr != '>') ptr++;				
				if (*ptr)
				{
					ptr++;
					if (!*ptr)
						continue;
					else if (*ptr == c)
					{

						strmcat(buffer, "%", BIG_BUFFER_SIZE);
						strmcat(buffer, s, ptr - s);
						strmcat(buffer, "s", BIG_BUFFER_SIZE);
					}
				}
			
			}
			else
			{
				bletch[2] = *ptr;
				strmcat(buffer, bletch, BIG_BUFFER_SIZE);
				if (!*ptr)
				{
					format = ptr;
					continue;
				}
			}
			ptr++;
		}
		else
			strmcat(buffer, format, BIG_BUFFER_SIZE);
		format = ptr;
	}
	malloc_strcpy(&ptr, buffer);
	return (ptr);
}

static	char	*convert_format(Window *win, char *format, int k)
{
	char	buffer[BIG_BUFFER_SIZE + 1];
	char	padded[BIG_BUFFER_SIZE + 1];
	int	pos = 0;
	int	*cp;
	int	map;
	char	key;
	int	i;

	cp = &win->func_cnt[k];
	while (*format && pos < BIG_BUFFER_SIZE - 4)
	{
		*padded = 0;
		if (*format != '%')
		{
			buffer[pos++] = *format++;
			continue;
		}

		/* Its a % */
		map = 0;

		/* Find the map, if neccesary */
		if (*++format == '{')
		{
			char	*endptr;

			format++;
			map = strtoul(format, &endptr, 10);
			if (*endptr != '}')
			{
				/* Unrecognized */
				continue;
			}
			format = endptr + 1;
		}
		else if (*format == '<')
		{
			char *p = padded;
			format++;
			while(*format && *format != '>') 
				*p++ = *format++;
			*p = 0;
			format++;
		}
		key = *format++;

		/* Choke once we get to the maximum number of expandos */
		if (*cp >= MAX_FUNCTIONS)
			continue;

		for (i = 0; i < NUMBER_OF_EXPANDOS; i++)
		{
			if (status_expandos[i].map != map ||
			    status_expandos[i].key != key)
				continue;

			if (status_expandos[i].format_var)
				new_free(status_expandos[i].format_var);
			if (status_expandos[i].format_set != -1)
				*(status_expandos[i].format_var) =
				     convert_sub_format(get_string_var(status_expandos[i].format_set), key, padded);
			buffer[pos++] = '%';
			buffer[pos++] = 's';

			win->status_func[k][(*cp)++] = 
				status_expandos[i].callback_function;
			break;
		}
	}

	win->func_cnt[k] = *cp;
	buffer[pos] = 0;
	return m_strdup(buffer);
}

char    *alias_special_char (char **, char *, char *, char *, int *);


void fix_status_buffer(Window *win, char *buffer, int in_status)
{
unsigned char	rhs_buffer[3*BIG_BUFFER_SIZE + 1];
unsigned char	lhs_buffer[3*BIG_BUFFER_SIZE + 1];
unsigned char	lhs_fillchar[6],
		rhs_fillchar[6],
		*fillchar = lhs_fillchar,
		*lhp = lhs_buffer,
		*rhp = rhs_buffer,
		*cp,
		*start_rhs = 0,
		*str = NULL, *ptr = NULL;
int		in_rhs = 0,
		pr_lhs = 0,
		pr_rhs = 0,
		*prc = &pr_lhs;

	lhs_buffer[0] = 0;
	rhs_buffer[0] = 0;
	if (!buffer || !*buffer)
		return;
	/*
	 * This converts away any ansi codes in the status line
	 * in accordance with the current settings.  This leaves us
	 * with nothing but logical characters, which are then easy
	 * to count. :-)
	 */
	str = strip_ansi(buffer);

	/*
	 * Count out the characters.
	 * Walk the entire string, looking for nonprintable
	 * characters.  We count all the printable characters
	 * on both sides of the %> tag.
	 */
	ptr = str;
	cp = lhp;
	lhs_buffer[0] = rhs_buffer[0] = 0;
	while (*ptr)
	{
		/*
		 * The FIRST such %> tag is used.
		 * Using multiple %>s is bogus.
		 */
		if (*ptr == '\f' && start_rhs == NULL)
		{
			ptr++;
			start_rhs = ptr;
			fillchar = rhs_fillchar;
			in_rhs = 1;
			*cp = 0;
			cp = rhp;
			prc = &pr_rhs;
		}
		/*
		 * Skip over color attributes if we're not
		 * doing color.
		 */
		else if (*ptr == '\003')
		{
			const u_char *end = skip_ctl_c_seq(ptr, NULL, NULL, 0);
			while (ptr < end)
				*cp++ = *ptr++;
		}
		/*
		 * If we have a ROM character code here, we need to
		 * copy it to the output buffer, as well as the fill
		 * character, and increment the printable counter by
		 * only 1.
		 */
		else if (*ptr == ROM_CHAR)
		{
			fillchar[0] = *cp++ = *ptr++;
			fillchar[1] = *cp++ = *ptr++;
			fillchar[2] = *cp++ = *ptr++;
			fillchar[3] = *cp++ = *ptr++;
			fillchar[4] = 0;
			*prc += 1;
		}
		/*
		 * Is it NOT a printable character?
		 */
		else if ((*ptr == REV_TOG) || (*ptr == UND_TOG) ||
			 (*ptr == ALL_OFF) || (*ptr == BOLD_TOG) ||
			 (*ptr == BLINK_TOG))
				*cp++ = *ptr++;
		/*
		 * So it is a printable character.
		 * Or maybe its a tab. ;-)
		 */
		else
		{
			*prc += 1;
			fillchar[0] = *cp++ = *ptr++;
			fillchar[1] = 0;
		}
		/*
		 * Dont allow more than CO printable characters
		 */
		if (pr_lhs + pr_rhs >= win->screen->co)
		{
			*cp = 0;
			break;
		}
	}
	*cp = 0;
	strcpy(buffer, lhs_buffer);
	strmcat(buffer, rhs_buffer, BIG_BUFFER_SIZE);
	new_free(&str);
}

void build_status(Window *win, char *format, int unused)
{
	int	i, k;
	char	*f = NULL;
	if (!win)
		return;
	for (k = 0; k < 2; k++) 
	{
		new_free(&win->status_format[k]);
		win->func_cnt[k] = 0;
		switch(k)
		{
			case 0:
				f = get_string_var(STATUS_FORMAT1_VAR);
				break;
			case 1:
				f = get_string_var(STATUS_FORMAT2_VAR);
				break;
		}			

		if (f)
			win->status_format[k] = convert_format(win, f, k);

		for (i = win->func_cnt[k]; i < MAX_FUNCTIONS; i++)
			win->status_func[k][i] = status_null_function;
	}
	update_all_status(win, NULL, 0);
}

void make_status(Window *win)
{
	u_char	buffer	    [BIG_BUFFER_SIZE + 1];
	u_char	lhs_buffer  [BIG_BUFFER_SIZE + 1];
	u_char	rhs_buffer  [BIG_BUFFER_SIZE + 1];
	char	*func_value[MAX_FUNCTIONS+10] = {NULL};
	u_char	*ptr;
	
	int	len = 1,
		status_line;

	/* The second status line is only displayed in the bottom window
	 * and should always be displayed, no matter what SHOW_STATUS_ALL
	 * is set to - krys
	 */
	for (status_line = 0 ; status_line < 1+win->double_status + win->status_lines; status_line++)
	{
		u_char	lhs_fillchar[6],
			rhs_fillchar[6],
			*fillchar = lhs_fillchar,
			*lhp = lhs_buffer,
			*rhp = rhs_buffer,
			*cp,
			*start_rhs = 0,
			*str;
		int	in_rhs = 0,
			pr_lhs = 0,
			pr_rhs = 0,
			line = status_line,
			*prc = &pr_lhs, 
			i;

		fillchar[0] = fillchar[1] = 0;

		if (!win->status_format[line])
			continue;
			
		for (i = 0; i < MAX_FUNCTIONS; i++)
			func_value[i] = (win->status_func[line][i]) (win);
		len = 1;
		
		if (get_int_var(REVERSE_STATUS_VAR))
			buffer[0] = get_int_var(REVERSE_STATUS_VAR) ? REV_TOG : '\0';
		else
			len = 0;
		str = &buffer[len];
		snprintf(str, BIG_BUFFER_SIZE - 1, 
			win->status_format[line],
			func_value[0], func_value[1], func_value[2],
			func_value[3], func_value[4], func_value[5],
			func_value[6], func_value[7], func_value[8],
			func_value[9], func_value[10], func_value[11],
			func_value[12], func_value[13], func_value[14],
			func_value[15], func_value[16], func_value[17],
			func_value[18], func_value[19], func_value[20],
			func_value[21], func_value[22], func_value[23],
			func_value[24], func_value[25], func_value[26],
			func_value[27], func_value[28], func_value[29],
			func_value[30], func_value[31],func_value[32],
			func_value[33], func_value[34],func_value[35],
			func_value[36], func_value[37],func_value[38]);

			/*  Patched 26-Mar-93 by Aiken
			 *  make_window now right-justifies everything 
			 *  after a %>
			 *  it's also more efficient.
			 */

		/*
		 * This converts away any ansi codes in the status line
		 * in accordance with the current settings.  This leaves us
		 * with nothing but logical characters, which are then easy
		 * to count. :-)
		 */
		str = strip_ansi(buffer);

		/*
		 * Count out the characters.
		 * Walk the entire string, looking for nonprintable
		 * characters.  We count all the printable characters
		 * on both sides of the %> tag.
		 */
		ptr = str;
		cp = lhp;
		lhs_buffer[0] = rhs_buffer[0] = 0;

		while (*ptr)
		{
			/*
			 * The FIRST such %> tag is used.
			 * Using multiple %>s is bogus.
			 */
			if (*ptr == '\f' && start_rhs == NULL)
			{
				ptr++;
				start_rhs = ptr;
				fillchar = rhs_fillchar;
				in_rhs = 1;
				*cp = 0;
				cp = rhp;
				prc = &pr_rhs;
			}

			/*
			 * Skip over color attributes if we're not
			 * doing color.
			 */
			else if (*ptr == '\003')
			{
				const u_char *end = skip_ctl_c_seq(ptr, NULL, NULL, 0);
				while (ptr < end)
					*cp++ = *ptr++;
			}

			/*
			 * If we have a ROM character code here, we need to
			 * copy it to the output buffer, as well as the fill
			 * character, and increment the printable counter by
			 * only 1.
			 */
			else if (*ptr == ROM_CHAR)
			{
				fillchar[0] = *cp++ = *ptr++;
				fillchar[1] = *cp++ = *ptr++;
				fillchar[2] = *cp++ = *ptr++;
				fillchar[3] = *cp++ = *ptr++;
				fillchar[4] = 0;
				*prc += 1;
			}

			/*
			 * Is it NOT a printable character?
			 */
			else if ((*ptr == REV_TOG) || (*ptr == UND_TOG) ||
				 (*ptr == ALL_OFF) || (*ptr == BOLD_TOG) ||
				 (*ptr == BLINK_TOG))
					*cp++ = *ptr++;
			/*
			 * So it is a printable character.
			 * Or maybe its a tab. ;-)
			 */
			else
			{
				*prc += 1;
				fillchar[0] = *cp++ = *ptr++;
				fillchar[1] = 0;
			}

			/*
			 * Dont allow more than CO printable characters
			 */
			if (pr_lhs + pr_rhs >= win->screen->co)
			{
				*cp = 0;
				break;
			}
		}
		*cp = 0;

		/* What will we be filling with? */
		if (get_int_var(STATUS_NO_REPEAT_VAR))
		{
			lhs_fillchar[0] = ' ';
			lhs_fillchar[1] = 0;
			rhs_fillchar[0] = ' ';
			rhs_fillchar[1] = 0;
		}

		/*
		 * Now if we have a rhs, then we have to adjust it.
		 */
		if (start_rhs)
		{
			int numf = 0;

			numf = win->screen->co - pr_lhs - pr_rhs  -1;
			while (numf-- >= 0)
				strmcat(lhs_buffer, lhs_fillchar, 
						BIG_BUFFER_SIZE);
		}

		/*
		 * No rhs?  If the user wants us to pad it out, do so.
		 */
		else if (get_int_var(FULL_STATUS_LINE_VAR))
		{
			int chars = win->screen->co - pr_lhs - 1;

			while (chars-- >= 0)
				strmcat(lhs_buffer, lhs_fillchar, 
						BIG_BUFFER_SIZE);
		}

		strcpy(buffer, lhs_buffer);
		strmcat(buffer, rhs_buffer, BIG_BUFFER_SIZE);
		strmcat(buffer, ALL_OFF_STR, BIG_BUFFER_SIZE);
		new_free(&str);

		if (!win->status_line[status_line] ||
			strcmp(buffer, win->status_line[status_line]))

		{
			malloc_strcpy(&win->status_line[status_line], buffer);
			output_screen = win->screen;

			if (win->status_lines && (line == win->double_status+win->status_lines) && win->status_split)
				term_move_cursor(0,win->top);
			else if (win->status_lines && !win->status_split)
				term_move_cursor(0,win->bottom+status_line-win->status_lines);
			else
				term_move_cursor(0,win->bottom+status_line);

			output_line(buffer);
#if 0
#if defined(SCO) || defined(HPUX) || defined(SOLARIS)
			term_set_foreground(7);    term_set_background(0);
#endif
#endif
			cursor_in_display(win);
			term_all_off();
		} 
	}
	cursor_to_input();
}


/* Some useful macros */
/*
 * This is used to get the current window on a window's screen
 */
#define CURRENT_WINDOW window->screen->current_window

/*
 * This tests to see if the window IS the current window on its screen
 */
#define IS_CURRENT_WINDOW (window->screen->current_window == window)

/*
 * This tests to see if all expandoes are to appear in all status bars
 */
#define SHOW_ALL_WINDOWS (get_int_var(SHOW_STATUS_ALL_VAR))

/*
 * "Current-type" window expandoes occur only on the current window for a 
 * screen.  However, if /set show_status_all is on, then ALL windows act as
 * "Current-type" windows.
 */
#define DISPLAY_ON_WINDOW (IS_CURRENT_WINDOW || SHOW_ALL_WINDOWS)

#define RETURN_EMPTY  return empty_string


static	char	*status_nickname(Window *window)
{
static char my_buffer[MY_BUFFER/2+1];
	snprintf(my_buffer, MY_BUFFER/2, nick_format, 
		get_server_nickname(window->server) ? 
		get_server_nickname(window->server) : 
		get_string_var(DEFAULT_NICKNAME_VAR));
	return my_buffer;
}

static	char	*status_server(Window *window)
{
	char	*name;
static	char	my_buffer[MY_BUFFER+1];
	if (window->server != -1)
	{
		if (server_format)
		{
			if (!(name = get_server_itsname(window->server)))
				name = get_server_name(window->server);
			snprintf(my_buffer, MY_BUFFER, server_format, name);
			return my_buffer;
		}
		else
			RETURN_EMPTY;
	}
	RETURN_EMPTY;
}

static	char	*status_query_nick(Window *window)
{
static	char my_buffer[BIG_BUFFER_SIZE+1];

	if (window->query_nick && query_format)
	{
		snprintf(my_buffer, BIG_BUFFER_SIZE, query_format, window->query_nick);
		return my_buffer;
	}
	else
		RETURN_EMPTY;
}

static	char	*status_right_justify(Window *window)
{
static	char	my_buffer[] = "\f";

	return my_buffer;
}

static	char	*status_notify_windows(Window *window)
{
	int	doneone = 0;
	char	buf2[MY_BUFFER+2];
static	char	my_buffer[MY_BUFFER/2+1];
	if (get_int_var(SHOW_STATUS_ALL_VAR) || window == window->screen->current_window)
	{
		*buf2='\0';
		window = NULL;
		while ((traverse_all_windows(&window)))
		{
			if (window->miscflags & WINDOW_NOTIFIED)
			{
				if (doneone++)
					strmcat(buf2, ",", MY_BUFFER/2);
				strmcat(buf2, ltoa(window->refnum), MY_BUFFER/2);
			}
		}
	}
	if (doneone && notify_format)
	{
		snprintf(my_buffer, MY_BUFFER/2, notify_format, buf2);
		return (my_buffer);
	}
	RETURN_EMPTY;
}

static	char	*status_clock(Window *window)
{
static	char	my_buf[MY_BUFFER+1];

	if ((get_int_var(CLOCK_VAR) && clock_format)  &&
	    (get_int_var(SHOW_STATUS_ALL_VAR) ||
	    (window == window->screen->current_window)))
		snprintf(my_buf, MY_BUFFER, clock_format, update_clock(GET_TIME));
	else
		RETURN_EMPTY;
	return my_buf;
}

static	char	*status_hold_lines(Window *window)
{
	int	num;
static  char	my_buffer[MY_BUFFER/2+1];
	
	if ((num = (window->lines_held /10) * 10))
	{
		snprintf(my_buffer, MY_BUFFER/2, hold_lines_format, ltoa(num));
		return(my_buffer);
	}
	RETURN_EMPTY;
}

static	char	*status_channel(Window *window)
{
	char	channel[IRCD_BUFFER_SIZE + 1];
static	char	my_buffer[IRCD_BUFFER_SIZE + 1];

	if (window->current_channel)
	{
		int num;
		strmcpy(channel, window->current_channel, IRCD_BUFFER_SIZE);

		if ((num = get_int_var(CHANNEL_NAME_WIDTH_VAR)) &&
		    ((int) strlen(channel) > num))
			channel[num] = (char) 0;
		snprintf(my_buffer, IRCD_BUFFER_SIZE, channel_format, channel);
		return my_buffer;
	}
	RETURN_EMPTY;
}

static	char	*status_insert_mode(Window *window)
{
char	*text;

	if (get_int_var(INSERT_MODE_VAR) && (get_int_var(SHOW_STATUS_ALL_VAR)
	    || (window->screen->current_window == window)))
		if ((text = get_string_var(STATUS_INSERT_VAR)))
			return text;
	RETURN_EMPTY;
}

static	char	*status_overwrite_mode(Window *window)
{
char	*text;

	if (!get_int_var(INSERT_MODE_VAR) && (get_int_var(SHOW_STATUS_ALL_VAR)
	    || (window->screen->current_window == window)))
	{
	    if ((text = get_string_var(STATUS_OVERWRITE_VAR)))
		return text;
	}
	RETURN_EMPTY;
}


/*
 * This is a generic status_userX variable.  All of these go to the
 * current-type window, although i think they should go to all windows.
 */
static char *status_user0 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER0_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user1 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER1_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user2 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER2_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user3 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER3_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user4 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER4_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user5 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER5_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user6 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER6_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user7 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER7_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user8 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER8_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user9 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER9_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user10 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER10_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user11 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER11_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user12 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER12_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user13 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER13_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user14 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER14_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user15 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER15_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user16 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER16_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user17 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER17_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user18 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER18_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

static char *status_user19 (Window *window)
{
	char *text;
	if ((text = get_string_var(STATUS_USER19_VAR)) &&
		DISPLAY_ON_WINDOW)
			return text;
	else
		RETURN_EMPTY;
}

#if 0
#define STATUS_VAR(x) \
static	char	*status_user ## x ## (Window *window)			\
{									\
	char	*text;							\
									\
	if ((text = get_string_var(STATUS_USER ## x ## _VAR)) && 	\
	     DISPLAY_ON_WINDOW)						\
		return text;						\
	else								\
		RETURN_EMPTY;					\
}

STATUS_VAR(0)
STATUS_VAR(1)
STATUS_VAR(2)
STATUS_VAR(3)
STATUS_VAR(4)
STATUS_VAR(5)
STATUS_VAR(6)
STATUS_VAR(7)
STATUS_VAR(8)
STATUS_VAR(9)
STATUS_VAR(10)
STATUS_VAR(11)
STATUS_VAR(12)
STATUS_VAR(13)
STATUS_VAR(14)
STATUS_VAR(15)
STATUS_VAR(16)
STATUS_VAR(17)
STATUS_VAR(18)
STATUS_VAR(19)
#endif


static	char	*status_hold(Window *window)
{
char	*text;

	if (window->holding_something && (text = get_string_var(STATUS_HOLD_VAR)))
		return(text);
	RETURN_EMPTY;
}

static	char	*status_window(Window *window)
{
char	*text;
	if ((number_of_windows_on_screen(window) > 1) && (window->screen->current_window == window) &&
	    (text = get_string_var(STATUS_WINDOW_VAR)))
		return(text);
	RETURN_EMPTY;
}

static	char	*status_refnum(Window *window)
{
static char my_buffer[MY_BUFFER/3+1];
	strmcpy(my_buffer, window->name ? window->name : ltoa(window->refnum), MY_BUFFER/3);
	return (my_buffer);
}

static	char	*status_version(Window *window)
{
	if (!get_int_var(SHOW_STATUS_ALL_VAR) && (window->screen->current_window != window))
		return(empty_string);
	return ((char *)nap_version);
}

static char *status_cpu_saver_mode (Window *window)
{
static char my_buffer[MY_BUFFER/2+1];
	if (cpu_saver && cpu_saver_format)
	{
		snprintf(my_buffer, MY_BUFFER/2, cpu_saver_format, "cpu");
		return my_buffer;
	}

	RETURN_EMPTY;
}


static	char	*status_null_function(Window *window)
{
	RETURN_EMPTY;
}


static char *status_position (Window *window)
{
static char my_buffer[MY_BUFFER/2+1];

	snprintf(my_buffer, MY_BUFFER/2, "(%d-%d)", window->lines_scrolled_back,
					window->distance_from_display);
	return my_buffer;
}
 
static char *status_scrollback(Window *win)
{
char *stuff;
	if (win->scrollback_point &&
	    (stuff = get_string_var(STATUS_SCROLLBACK_VAR)))
		return stuff;
	else
		RETURN_EMPTY;
}

static	char	*status_percent		(Window *window)
{
	static	char	percent[] = "%";
	return	percent;
}

static	char	*status_server_stats(Window *win)
{
static char my_buffer[MY_BUFFER+1];
N_STATS *stats;
	if ((stats = get_server_stats(win->server)) && stats_format)
	{
		int c;
		char gigs[20], songs[20], shares[20];
		c = charcount(stats_format, '%');
		*my_buffer = 0;
		strcpy(gigs, ltoa(stats->gigs));
		strcpy(songs, ltoa(stats->songs));
		strcpy(shares, ltoa(stats->libraries));
		switch(c)
		{
			case 0:
				snprintf(my_buffer, MY_BUFFER, stats_format);
				break;
			case 1:
				snprintf(my_buffer, MY_BUFFER, stats_format, shares);
				break;
			case 2:
				snprintf(my_buffer, MY_BUFFER, stats_format, shares, songs);
				break;
			case 3:
				snprintf(my_buffer, MY_BUFFER, stats_format, shares, songs, gigs);
				break;
		}
		return my_buffer;
	}
	RETURN_EMPTY;
}

static char *napster_shared(Window *win)
{
static char my_buffer[MY_BUFFER+1];
	if (!shared_stats.shared_files)
		RETURN_EMPTY;
	snprintf(my_buffer, MY_BUFFER, "[Sh:%lu/%3.2f%s] ", 
		shared_stats.shared_files, _GMKv(shared_stats.shared_filesize), 
		_GMKs(shared_stats.shared_filesize));
	return my_buffer;
}

static char *napster_updown(Window *win)
{
static char my_buffer[MY_BUFFER+1];
int upload = 0, download = 0, queued = 0;
GetFile *sg;
NickStruct *n;
extern Server *server_list;
	if (!transfer_struct || win->server < 0)
		RETURN_EMPTY;
	for (sg = transfer_struct; sg; sg = sg->next)
		if ((sg->flags & NAP_DOWNLOAD) == NAP_DOWNLOAD)
			download++;
		else
			upload++;
	for (n = server_list[win->server].users; n; n = n->next)
		for (sg = n->Queued; sg; sg = sg->next)
			queued++;
	snprintf(my_buffer, MY_BUFFER, " [U:%d/D:%d", upload, download);
	if (queued)
		snprintf(my_buffer+strlen(my_buffer), MY_BUFFER, "/Q:%d", queued);
	strcat(my_buffer, "]");
	return my_buffer;
}

static char *status_eta(Window *win)
{
static char buff[MY_BUFFER+1];
GetFile *tmp;
	*buff = 0;
	for (tmp = transfer_struct; tmp; tmp = tmp->next)
	{
		if (!tmp->filesize)
			continue;
		if (*buff)
			strlcat(buff, ",", MY_BUFFER);
		strlcat(buff, calc_eta(tmp), MY_BUFFER);
		
	}
	return buff;
}

static char *napster_download(Window *win)
{
int upload = 0;
int download = 0;
GetFile *tmp;
static char buff[MY_BUFFER+1];
char upbuffer[BIG_BUFFER_SIZE+1];
char dnbuffer[BIG_BUFFER_SIZE];
char tmpbuff[80];
double perc = 0.0;
	*upbuffer = 0;
	*dnbuffer = 0;
	for (tmp = transfer_struct; tmp; tmp = tmp->next)
	{
		if (!tmp->filesize)
			continue;
                perc = (100.0 * (((double)(tmp->received + tmp->resume)) / (double)tmp->filesize));
		sprintf(tmpbuff, "%4.1f%%", perc);
		if ((tmp->flags & NAP_DOWNLOAD) == NAP_DOWNLOAD)
		{
			if (download)
				strlcat(dnbuffer, ",", BIG_BUFFER_SIZE);
			else
				strlcat(dnbuffer, " [G:", BIG_BUFFER_SIZE);
			strlcat(dnbuffer, tmpbuff, BIG_BUFFER_SIZE);
			download++;
		}
		else
		{
			if (upload)
				strlcat(upbuffer, ",", BIG_BUFFER_SIZE);
			else
				strlcat(upbuffer, " [S:", BIG_BUFFER_SIZE);
			strlcat(upbuffer, tmpbuff, BIG_BUFFER_SIZE);
			upload++;
		}
	}
	if (download)
		strlcat(dnbuffer, "]", BIG_BUFFER_SIZE);
	if (upload)
		strlcat(upbuffer, "]", BIG_BUFFER_SIZE);
	strlcpy(buff, dnbuffer, MY_BUFFER-1);
	strlcat(buff, upbuffer, MY_BUFFER-1);
	return buff;
}

#ifdef WINNT
#define THREAD
#endif

static char *napster_load_share(Window *win)
{
#ifdef THREAD
static char my_buffer[MY_BUFFER];
extern int in_load;
	if (in_load)
	{
#if !defined(WINNT)
		if (pthread_mutex_trylock(&shared_count_mutex) == EBUSY)
			RETURN_EMPTY;
#endif
		if (shared_count)
		{
			sprintf(my_buffer, "%lu", shared_count);
#if !defined(WINNT)
			pthread_mutex_unlock(&shared_count_mutex);
#endif
			return my_buffer;
		}
#if !defined(WINNT)
		pthread_mutex_unlock(&shared_count_mutex);
#endif
	}
#endif
	RETURN_EMPTY;
}

static char *status_usercount(Window *win)
{
ChannelStruct *ch;
NickStruct *n;
int count = 0;
static char buff[30];

	if (!win || !win->current_channel)
		RETURN_EMPTY;
	if (!(ch = (ChannelStruct *)find_in_list((List **)&win->nchannels, win->current_channel, 0)))
		RETURN_EMPTY;
	for (n = ch->nicks; n; n = n->next)
		count++;
	sprintf(buff, "%u", count);
	return buff;
}

static char *status_cloak(Window *win)
{
static char buff[] = "*";
	if (win->server > -1)
		if (get_server_cloak(win->server))
			return buff;
	RETURN_EMPTY;
}

static	char	*status_lag (Window *window)
{
static  char	my_buffer[MY_BUFFER/2+1];
	if (get_server_admin(window->server) > USER_USER)
	{
		struct timeval td;		
		td = get_server_lag(window->server);
		snprintf(my_buffer, MY_BUFFER/2-1, "%1.3f", ((double)td.tv_sec + ((double)td.tv_usec / 1000000.0))/2);
		return(my_buffer);
	}
	RETURN_EMPTY;
}

static char	*status_topic(Window *window)
{
static char	my_buffer[MY_BUFFER+1];
	if (window->current_channel)
	{
		ChannelStruct *ch;
		if ((ch = (ChannelStruct *)find_in_list((List **)&window->nchannels, window->current_channel, 0)))
		{
			strncpy(my_buffer, ch->topic, MY_BUFFER);
			return my_buffer;
		}
	}
	RETURN_EMPTY;
}

static char	*status_sharedir(Window *window)
{
static char 	my_buffer[MY_BUFFER+1];
char *p;
extern char *return_current_share_dir();
	p = return_current_share_dir();
	if (*p)
		strncpy(my_buffer, p, MY_BUFFER);
	else
		*my_buffer = 0;
	return my_buffer;
}
