/* 
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 * 
 * The Original Code is the Sablotron XSLT Processor.
 * 
 * The Initial Developer of the Original Code is Ginger Alliance Ltd.
 * Portions created by Ginger Alliance are Copyright (C) 2000 Ginger
 * Alliance Ltd. All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL"), in which case the provisions of the GPL are applicable 
 * instead of those above.  If you wish to allow use of your 
 * version of this file only under the terms of the GPL and not to
 * allow others to use your version of this file under the MPL,
 * indicate your decision by deleting the provisions above and
 * replace them with the notice and other provisions required by
 * the GPL.  If you do not delete the provisions above, a recipient
 * may use your version of this file under either the MPL or the
 * GPL.
 */

#ifndef VertsHIncl
#define VertsHIncl


#include "base.h"
#include "arena.h"
#include "datastr.h"
#include "expr.h"
#include "hash.h"

class Context;
class Expression;

class AttList;

/****************************************
V e r t e x
****************************************/

class Vertex : public ArenaMember
{
public:
    Vertex(Processor* proc_, VTYPE avt = VT_VERTEX_WF)
        : vt(avt), parent(NULL), ordinal(0), proc(proc_)
    {
        stamp = lineno = 0;
    };
    virtual ~Vertex();
    virtual eFlag execute(Context *);
   	virtual eFlag value(DStr &, Context *);
    virtual eFlag startCopy();
    virtual eFlag endCopy();
    virtual eFlag copy();
    void old__output(DataLine &);
    virtual void speak(DStr &, SpeakMode);
    void setParent(Vertex *v);
    virtual const QName& getName() const;
    Vertex *parent;
    Processor* proc;
    int 
        ordinal,
        lineno;
    VTYPE vt;
    int 
        stamp;          // the vertex number in doc order (0 = root)

};

/****************************************
V e r t e x L i s t
****************************************/

// FIXME: changed PList to SList...

class VertexList: public SList<Vertex*>
{
public:
    VertexList(int logBlockSize_ = LIST_SIZE_SMALL);
    virtual ~VertexList();
    eFlag execute (Context*);
    eFlag value(DStr&, Context *);
    virtual void speak(DStr &, SpeakMode);
    virtual void append(Vertex *);
    void appendAndSetOrdinal(Vertex *v);
    void destructMembers();
    int strip();
    eFlag copy();
};

/****************************************
A r e n a V e r t e x L i s t
****************************************/

class ArenaVertexList : public VertexList, public ArenaMember
{
public:
    ArenaVertexList(Arena *arena__=NULL, int logBlockSize_ = LIST_SIZE_SMALL)
        : VertexList(logBlockSize_), arena_(arena__) 
    {};

    ~ArenaVertexList()
    {
        deppendall();
    }

protected:
    virtual Vertex** claimMemory( int nbytes ) const 
    { 
        return arena_ ? (Vertex**)(arena_ -> armalloc(nbytes)) : 
            (Vertex**)VertexList::claimMemory(nbytes); 
    }

    virtual Vertex** reclaimMemory( Vertex**p, int newbytes, int oldbytes ) const 
    {
        if (!arena_)
            return VertexList::reclaimMemory(p, newbytes, oldbytes);
        else if (newbytes > oldbytes)
        {
            Vertex **newpos = (Vertex**) (arena_ -> armalloc(newbytes));
            memcpy(newpos, p, oldbytes);
            return newpos;
        }
        else
            return p;
    }

    virtual void returnMemory( Vertex**&p ) const 
    {
        if (arena_)
            arena_ -> arfree(p);
        else
            if (p) VertexList::returnMemory(p); 
        p = NULL;
    }
    Arena *arena_;
};


/****************************************
D a d d y
****************************************/

class Daddy : public Vertex
{
public:
    Daddy(Processor* proc, VTYPE avt = VT_DADDY_WF)
        : Vertex(proc, avt), contents(proc -> getArena())
    {};
    virtual ~Daddy(void);
    virtual eFlag execute(Context *c);
   	virtual eFlag value(DStr &, Context *);
    virtual eFlag newChild(Vertex*);
    virtual eFlag checkChildren();
    virtual void speak(DStr &, SpeakMode);
    virtual int strip();
    //
    ArenaVertexList contents;
};

/****************************************
R o o t N o d e
****************************************/

class RootNode : public Daddy
{
public:
    RootNode(Processor* proc_)
        : Daddy(proc_, VT_ROOT_WF)
    {
    };
    virtual ~RootNode();
    virtual eFlag execute(Context *c);
    virtual eFlag newChild(Vertex*);
    virtual eFlag checkChildren();
    virtual void speak(DStr &, SpeakMode);
    virtual eFlag copy();
};

/****************************************
A t t r i b u t e
****************************************/

class Attribute : public Vertex
{
public:
    Attribute(const QName&, const Str&, XSL_ATT, Processor* proc_);
    virtual ~Attribute();
    eFlag buildExpr(Bool, ExType);
    virtual eFlag execute(Context *c);
    virtual void speak(DStr &, SpeakMode);
   	virtual eFlag value(DStr &, Context *);
    virtual eFlag startCopy();
    virtual const QName& getName() const;
    QName name;
    ArenaStr cont;
    Expression *expr;
    XSL_ATT op;
};

/****************************************
A t t L i s t
****************************************/

class AttList: public ArenaVertexList
{
public:
    AttList(Arena *arena_)
        : ArenaVertexList(arena_)
    {};
    Attribute *find(XSL_ATT);
};

/*****************************************************************
    N m S p a c e
*****************************************************************/

class NmSpace: public Vertex
{
public:
    NmSpace(Phrase prefix_, Phrase uri_, Processor* proc_);
    virtual ~NmSpace();
    Phrase
        prefix,
        uri;
    virtual eFlag execute(Context *);
    virtual void speak(DStr &, SpeakMode);
   	virtual eFlag value(DStr &, Context *);
    virtual eFlag startCopy();
};

/*****************************************************************
    N S L i s t
*****************************************************************/

class NSList : public ArenaVertexList
{
public:
    NSList (Processor* proc_, Arena* arena_ = NULL)
        : proc ( proc_ ), ArenaVertexList(arena_,0)
    { }

    ~NSList();
    NmSpace *find(Phrase) const;
    eFlag resolve(Phrase&, Bool) const;
    void unresolve(Phrase&) const;
    void giveCurrent(NSList &, Tree*) const;
    Processor* proc;
};

/****************************************
E l e m e n t
****************************************/

class Element : public Daddy
{
public:
    Element(QName&, Tree *_ownerT, Processor* proc_, VTYPE = VT_ELEMENT_WF);
    virtual ~Element();
    virtual eFlag execute(Context *c);
    virtual eFlag newChild(Vertex*);
    virtual void speak(DStr &, SpeakMode);
    void removeBindings();
    virtual eFlag startCopy();
    virtual eFlag endCopy();
    virtual eFlag copy();
    virtual const QName& getName() const;
    //
    NSList
        namespaces;
    AttList 
        atts;
    Tree
        *ownerT;
    QName
        name;
};

/****************************************
T e x t
****************************************/

class Text : public Vertex
{
public:
    ArenaStr cont;
    Text(char *acont, Processor* proc_, Arena* arena_, int alen=0)
        : Vertex(proc_, VT_TEXT_WF), cont(arena_)
    {
        if (alen)
            cont.nset(acont,alen);
        else cont = (char*) acont;
    };
    virtual ~Text();
    virtual eFlag execute(Context *);
    virtual void speak(DStr &, SpeakMode);
   	virtual eFlag value(DStr &, Context *);
    virtual eFlag startCopy();
};


/****************************************
C o m m e n t
****************************************/

class Comment: public Vertex
{
public:
    ArenaStr cont;
    Comment(const Str& cont_, Processor* proc_);
    virtual ~Comment();
    virtual eFlag execute(Context *);
    virtual void speak(DStr &, SpeakMode);
   	virtual eFlag value(DStr &, Context *);
    virtual eFlag startCopy();
};


/****************************************
P r o c I n s t r
****************************************/

class ProcInstr: public Vertex
{
public:
    ProcInstr(Phrase name_, const Str& cont_, Processor* proc_);
    virtual ~ProcInstr();
    virtual eFlag execute(Context *);
    virtual void speak(DStr &, SpeakMode);
   	virtual eFlag value(DStr &, Context *);
    virtual eFlag startCopy();
    virtual const QName& getName() const;
    ArenaStr cont;
    QName name;
};


/****************************************
X S L E l e m e n t
****************************************/

class XSLElement : public Element
{
public:
	eFlag checkAtts();       
    XSLElement(QName&, Tree *_ownerT, XSL_OP, Processor* proc_);
    virtual eFlag execute(Context *c);
    virtual eFlag newChild(Vertex*);
    eFlag checkToplevel();
    virtual eFlag checkChildren();
    void checkExtraChildren(int& k);
    Expression *getAttExpr(XSL_ATT);
    virtual int strip();
    XSL_OP op;
//    VertexList defs;
private:
    // applies only to XSL_APPLY_TEMPLATES and XSL_FOR_EACH:
    eFlag makeSortDefs(SortDefList &sortDefs, Context *c);
    // applies only to XSL_SORT:
    eFlag make1SortDef(SortDef *&def, Context *c);
};

#endif //ifndef VertsHIncl

