/* 
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 * 
 * The Original Code is the Sablotron XSLT Processor.
 * 
 * The Initial Developer of the Original Code is Ginger Alliance Ltd.
 * Portions created by Ginger Alliance are Copyright (C) 2000 Ginger
 * Alliance Ltd. All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL"), in which case the provisions of the GPL are applicable 
 * instead of those above.  If you wish to allow use of your 
 * version of this file only under the terms of the GPL and not to
 * allow others to use your version of this file under the MPL,
 * indicate your decision by deleting the provisions above and
 * replace them with the notice and other provisions required by
 * the GPL.  If you do not delete the provisions above, a recipient
 * may use your version of this file under either the MPL or the
 * GPL.
 */

#include "verts.h"
#include "tree.h"
#include "expr.h"
#include "proc.h"
#include "context.h"
#include "vars.h"
#include "parser.h"

/*****************************************************************
    Instruction tables
*****************************************************************/

struct AttTableItem
{
    XSL_ATT
        attCode;
    Bool 
        required,
        avtemplate;
    ExType
        exprType;
};

typedef AttTableItem AttTable[];

AttTable at_empty = 
{
    {XSLA_NONE, FALSE, FALSE, EX_NONE}
},
at_apply_templates  = 
{
    {XSLA_SELECT, FALSE, FALSE, EX_NODESET},
    {XSLA_MODE, FALSE, FALSE, EX_NONE}
},
at_attribute = 
{
    {XSLA_NAME, TRUE, TRUE, EX_STRING},
    {XSLA_NAMESPACE, FALSE, TRUE, EX_STRING}
},
at_attribute_set =
{
    {XSLA_NAME, TRUE, FALSE, EX_NONE},
    {XSLA_USE_ATTR_SETS, FALSE, FALSE, EX_NONE},
},
at_call_template = 
    {
        {XSLA_NAME, TRUE, FALSE, EX_NONE}
    }, 
        at_copy =
    {
        {XSLA_USE_ATTR_SETS, FALSE,  FALSE, EX_NONE}
    },
        at_copy_of =
    {
        {XSLA_SELECT, TRUE, FALSE, EX_UNKNOWN}
    },
        at_decimal_format =
    {
        {XSLA_NAME, FALSE, FALSE, EX_NONE},
        {XSLA_DECIMAL_SEPARATOR, FALSE, FALSE, EX_NONE},
        {XSLA_GROUPING_SEPARATOR, FALSE, FALSE, EX_NONE},
        {XSLA_INFINITY, FALSE, FALSE, EX_NONE},
        {XSLA_MINUS_SIGN, FALSE, FALSE, EX_NONE},
        {XSLA_NAN, FALSE, FALSE, EX_NONE},
        {XSLA_PERCENT, FALSE, FALSE, EX_NONE},
        {XSLA_PER_MILLE, FALSE, FALSE, EX_NONE},
        {XSLA_ZERO_DIGIT, FALSE, FALSE, EX_NONE},
        {XSLA_DIGIT, FALSE, FALSE, EX_NONE},
        {XSLA_PATTERN_SEPARATOR, FALSE, FALSE, EX_NONE}
    },
        at_element =
    {
        {XSLA_NAME, TRUE, TRUE, EX_STRING},
        {XSLA_NAMESPACE, FALSE, TRUE, EX_STRING},
        {XSLA_USE_ATTR_SETS, FALSE, FALSE, EX_NONE}
    },
        at_for_each =
    {
        {XSLA_SELECT, TRUE, FALSE, EX_NODESET}
    },
        at_if =
    {
        {XSLA_TEST, TRUE, FALSE, EX_BOOLEAN}
    },
        at_import =
    {
        {XSLA_HREF, TRUE, FALSE, EX_NONE}
    },
        at_include =
    {
        {XSLA_HREF, TRUE, FALSE, EX_NONE}
    },
        at_key =
    {
        {XSLA_NAME, TRUE, FALSE, EX_NONE},
        {XSLA_MATCH, TRUE, FALSE, EX_NODESET_PATTERN},
        {XSLA_USE, TRUE, FALSE, EX_UNKNOWN}
    },
        at_message =
    {
        {XSLA_TERMINATE, FALSE, FALSE, EX_NONE}
    },
        at_ns_alias =
    {
        {XSLA_STYLESHEET_PREFIX, TRUE, FALSE, EX_NONE},
        {XSLA_RESULT_PREFIX, TRUE, FALSE, EX_NONE}
    },
        at_number =
    {
        {XSLA_LEVEL, FALSE, FALSE, EX_NONE},
        {XSLA_COUNT, FALSE, FALSE, EX_NONE},
        {XSLA_FROM, FALSE, FALSE, EX_NONE},
        {XSLA_VALUE, FALSE, FALSE, EX_NONE},
        {XSLA_FORMAT, FALSE, TRUE, EX_STRING},
        {XSLA_LANG, FALSE, TRUE, EX_STRING},
        {XSLA_LETTER_VALUE, FALSE, TRUE, EX_STRING},
        {XSLA_GROUPING_SEPARATOR, FALSE, TRUE, EX_STRING},
        {XSLA_GROUPING_SIZE, FALSE, TRUE, EX_STRING}
    },
        at_output =
    {
        {XSLA_METHOD, FALSE, FALSE, EX_NONE},
        {XSLA_VERSION, FALSE, FALSE, EX_NONE},
        {XSLA_ENCODING, FALSE, FALSE, EX_NONE},
        {XSLA_OMIT_XML_DECL, FALSE, FALSE, EX_NONE},
        {XSLA_STANDALONE, FALSE, FALSE, EX_NONE},
        {XSLA_DOCTYPE_PUBLIC, FALSE, FALSE, EX_NONE},
        {XSLA_DOCTYPE_SYSTEM, FALSE, FALSE, EX_NONE},
        {XSLA_CDATA_SECT_ELEMS, FALSE, FALSE, EX_NONE},
        {XSLA_INDENT, FALSE, FALSE, EX_NONE},
        {XSLA_MEDIA_TYPE, FALSE, FALSE, EX_NONE}
    },
        at_param =
    {
        {XSLA_NAME, TRUE, FALSE, EX_NONE},
        {XSLA_SELECT, FALSE, FALSE, EX_UNKNOWN}
    },
        at_preserve_space =
    {
        {XSLA_ELEMENTS, TRUE, FALSE, EX_NONE}
    },
        at_pi =
    {
        {XSLA_NAME, TRUE, TRUE, EX_STRING}
    },
        at_sort =
    {
        {XSLA_SELECT, FALSE, FALSE, EX_STRING},
        {XSLA_LANG, FALSE, TRUE, EX_STRING},
        {XSLA_DATA_TYPE, FALSE, TRUE, EX_STRING},
        {XSLA_ORDER, FALSE, TRUE, EX_STRING},
        {XSLA_CASE_ORDER, FALSE, TRUE, EX_STRING}
    },
        at_strip_space =
    {
        {XSLA_ELEMENTS, TRUE, FALSE, EX_NONE}
    },
        at_template =
    {
        {XSLA_MATCH, FALSE, FALSE, EX_NODESET_PATTERN},
        {XSLA_NAME, FALSE, FALSE, EX_NONE},
        {XSLA_PRIORITY, FALSE, FALSE, EX_NONE},
        {XSLA_MODE, FALSE, FALSE, EX_NONE}
    },
        at_text =
    {
        {XSLA_DISABLE_OUTPUT_ESC, FALSE, FALSE, EX_NONE}
    },
        at_transform =
    {
        {XSLA_ID, FALSE, FALSE, EX_NONE},
        {XSLA_EXT_ELEM_PREFIXES, FALSE, FALSE, EX_NONE},
        {XSLA_EXCL_ELEM_PREFIXES, FALSE, FALSE, EX_NONE},
        {XSLA_VERSION, TRUE, FALSE, EX_NONE}
    },
        at_value_of =
    {
        {XSLA_SELECT, TRUE, FALSE, EX_STRING},
        {XSLA_DISABLE_OUTPUT_ESC, FALSE, FALSE, EX_NONE}
    },
        at_variable =
    {
        {XSLA_NAME, TRUE, FALSE, EX_NONE},
        {XSLA_SELECT, FALSE, FALSE, EX_UNKNOWN}
    },
        at_when =
    {
        {XSLA_TEST, TRUE, FALSE, EX_BOOLEAN}
    },
        at_with_param =
    {
        {XSLA_NAME, TRUE, FALSE, EX_NONE},
        {XSLA_SELECT, FALSE, FALSE, EX_UNKNOWN}
    };

#define instr(code, elemflg, req, max, table) \
{code, elemflg, req, max, (AttTableItem*) table}

enum ElemFlags
{
//    ELEM_EMPTY = 1,
    ELEM_TOPLEVEL = 2,
    ELEM_INSTR = 4,
    ELEM_EXTRA = 8,
    //
    ELEM_CONT_PCDATA = 0x10,
    ELEM_CONT_TOPLEVEL = 0x20,
    ELEM_CONT_INSTR = 0x40,
    ELEM_CONT_EXTRA = 0x80,
    //
    ELEM_SELF = ELEM_TOPLEVEL | ELEM_INSTR | ELEM_EXTRA,
    ELEM_CONT = ELEM_CONT_TOPLEVEL | ELEM_CONT_INSTR | ELEM_CONT_EXTRA,
    //
    ELEM_I0 = ELEM_INSTR,
    ELEM_IX = ELEM_INSTR | ELEM_CONT_EXTRA,
    ELEM_II = ELEM_INSTR | ELEM_CONT_INSTR,
    ELEM_IT_I = ELEM_INSTR | ELEM_TOPLEVEL | ELEM_CONT_INSTR,
    ELEM_I_XI = ELEM_INSTR | ELEM_CONT_EXTRA | ELEM_CONT_INSTR,
    ELEM_I__ = ELEM_INSTR | ELEM_CONT_PCDATA,
    //
    ELEM_T0 = ELEM_TOPLEVEL,
    ELEM_TX = ELEM_TOPLEVEL | ELEM_CONT_EXTRA,
    ELEM_TX_I = ELEM_TOPLEVEL | ELEM_EXTRA | ELEM_CONT_INSTR,
    ELEM_T_XI = ELEM_TOPLEVEL | ELEM_CONT_EXTRA | ELEM_CONT_INSTR,
    //
    ELEM_X0 = ELEM_EXTRA,
    ELEM_XI = ELEM_EXTRA | ELEM_CONT_INSTR,
    ELEM_X_XT = ELEM_EXTRA | ELEM_CONT_EXTRA | ELEM_CONT_TOPLEVEL,
    ELEM_NONE
};

struct InstrTableItem
{
    XSL_OP op;
    ElemFlags flags;
    int 
        reqAtts,
        maxAtts;
    AttTableItem *att;
} instrTable[]
=
{
    instr(XSL_APPLY_IMPORTS,    ELEM_I0,    0, 0, at_empty),
    instr(XSL_APPLY_TEMPLATES,  ELEM_IX,    0, 2, at_apply_templates),
    instr(XSL_ATTRIBUTE,        ELEM_II,    1, 2, at_attribute),
    instr(XSL_ATTRIBUTE_SET,    ELEM_TX,    1, 2, at_attribute_set),
    instr(XSL_CALL_TEMPLATE,    ELEM_IX,    1, 1, at_call_template),
    instr(XSL_CHOOSE,           ELEM_IX,    0, 0, at_empty),
    instr(XSL_COMMENT,          ELEM_II,    0, 0, at_empty),
    instr(XSL_COPY,             ELEM_II,    0, 1, at_copy),
    instr(XSL_COPY_OF,          ELEM_I0,    1, 1, at_copy_of),
    instr(XSL_DECIMAL_FORMAT,   ELEM_T0,    0, 11, at_decimal_format),
    instr(XSL_ELEMENT,          ELEM_II,    1, 3, at_element),
    instr(XSL_FALLBACK,         ELEM_II,    0, 0, at_empty),
    instr(XSL_FOR_EACH,         ELEM_I_XI,  1, 1, at_for_each),
    instr(XSL_IF,               ELEM_II,    1, 1, at_if),
    instr(XSL_IMPORT,           ELEM_X0,    1, 1, at_import),
    instr(XSL_INCLUDE,          ELEM_T0,    1, 1, at_include),
    instr(XSL_KEY,              ELEM_T0,    3, 3, at_key),
    instr(XSL_MESSAGE,          ELEM_II,    0, 1, at_message),
    instr(XSL_NAMESPACE_ALIAS,  ELEM_T0,    2, 2, at_ns_alias),
    instr(XSL_NUMBER,           ELEM_I0,    0, 9, at_number),
    instr(XSL_OTHERWISE,        ELEM_XI,    0, 0, at_empty),
    instr(XSL_OUTPUT,           ELEM_T0,    0, 10, at_output),
    instr(XSL_PARAM,            ELEM_TX_I,  1, 2, at_param),
    instr(XSL_PRESERVE_SPACE,   ELEM_T0,    1, 1, at_preserve_space),
    instr(XSL_PROCESSING_INSTR, ELEM_II,    1, 1, at_pi),
    instr(XSL_SORT,             ELEM_X0,    0, 5, at_sort),
    instr(XSL_STRIP_SPACE,      ELEM_T0,    1, 1, at_strip_space),
    instr(XSL_STYLESHEET,       ELEM_X_XT,  1, 4, at_transform),
    instr(XSL_TEMPLATE,         ELEM_T_XI,  0, 4, at_template),
    instr(XSL_TEXT,             ELEM_I__,   0, 1, at_text),
    instr(XSL_TRANSFORM,        ELEM_X_XT,  1, 4, at_transform),
    instr(XSL_VALUE_OF,         ELEM_I0,    1, 2, at_value_of),
    instr(XSL_VARIABLE,         ELEM_IT_I,  1, 2, at_variable),
    instr(XSL_WHEN,             ELEM_XI,    1, 1, at_when),
    instr(XSL_WITH_PARAM,       ELEM_XI,    1, 2, at_with_param)
};

/****************************************
V e r t e x
****************************************/

Vertex::~Vertex()
{
};

eFlag Vertex::execute(Context *c)
{
    assert(FALSE);
    return NOT_OK;
}

eFlag Vertex::value(DStr& ret, Context *c)
{
    assert(0);
    return NOT_OK;
}

eFlag Vertex::startCopy()
{
    return OK;
};

eFlag Vertex::endCopy()
{
    return OK;
};

eFlag Vertex::copy()
{
    startCopy();
    endCopy();
    return OK;
}

void Vertex::speak(DStr &ret, SpeakMode mode)
{
}

void Vertex::old__output(DataLine& outbox)
{
    DStr s;
    speak(s, (SpeakMode)(SM_NAME | SM_CONTENTS));
    if (!s.isEmpty())
        outbox.save(s, s.length());
}

void Vertex::setParent(Vertex *v)
{
    parent = v;
}

const QName& Vertex::getName() const
{
    return proc -> theEmptyQName;
}


/****************************************
V e r t e x L i s t
****************************************/

VertexList::VertexList(int logBlockSize_ /*=LIST_SIZE_SMALL*/)
: SList<Vertex*>(logBlockSize_)
{
}

VertexList::~VertexList()
{
    deppendall();
}

void VertexList::destructMembers()
{
    for (int i = 0; i < nItems; i++)
        (*this)[i] -> ~Vertex();
}


eFlag VertexList::execute(Context *c)
{
    int i;
    for (i=0; i<number(); i++)
        E( (*this)[i] -> execute(c) );
    return OK;
}

eFlag VertexList::value(DStr &ret, Context *c)
{
    int i;
    DStr temp;

    ret.empty();
    for (i=0; i<number(); i++)
    {
        E( (*this)[i] -> value(temp, c) );
        temp.appendSelf(ret);
    };
    return OK;
}

void VertexList::speak(DStr &ret, SpeakMode mode)
{
    int i;
    for (i = 0; i < number(); i++)
    {
        (*this)[i] -> speak(ret, mode);
        if ((mode & SM_INS_SPACES) && (i < number() - 1))
             ret += ' ';
    };
}

void VertexList::append(Vertex *v)
{
    List<Vertex*>::append(v);
}

void VertexList::appendAndSetOrdinal(Vertex *v)
{
    v -> ordinal = number();
    List<Vertex*>::append(v);
}

int VertexList::strip()
{
    int cnt = 0;
    for (int i = 0; i < number(); i++)
    {
        Vertex *node = (*this)[i];
        if ((node -> vt == VT_TEXT) && isAllWhite((char*)(toText(node) -> cont)))
        {
            cnt++;
            // not doing a freerm() since the node is allocated in the arena
            // freerm(i--,FALSE);
            rm(i--);
        }
    };
    return cnt;
}

eFlag VertexList::copy()
{
    for (int i = 0; i < number(); i++)
        (*this)[i] -> copy();
    return OK;
}


/****************************************
D a d d y
a vertex with contents
****************************************/

Daddy::~Daddy()
{
    // not doing freeall since it's all in the arena
    // contents.freeall(FALSE);
    contents.destructMembers();
};

eFlag Daddy::execute(Context *c)
{
    proc -> situation -> pushCurrV(this);
    E( contents.execute(c) );
    proc -> situation->popCurrent();
    return OK;
};

eFlag Daddy::value(DStr &ret, Context *c)
{
    E( contents.value(ret, c) );
    return OK;
}

eFlag Daddy::checkChildren()
{
    assert(!"Daddy::checkChildren()");
    return NOT_OK;
}

eFlag Daddy::newChild(Vertex *v)
{
    contents.appendAndSetOrdinal(v);
    v -> setParent(this);
    return OK;
};

void Daddy::speak(DStr &ret, SpeakMode mode)
{
    if (mode & SM_CONTENTS)
        contents.speak(ret, mode);
}

int Daddy::strip()
{
    return contents.strip();
}

/****************************************
R o o t N o d e
a vertex with prolog and contents 
****************************************/

RootNode::~RootNode()
{
}

eFlag RootNode::execute(Context *c)
{
    //! process prolog
    //
    E( Daddy::execute(c) );
    return OK;
};

eFlag RootNode::checkChildren()
{
    return OK;
}

eFlag RootNode::newChild(Vertex *v)
{
    //! insert prolog
    //
    E( Daddy::newChild(v) );
    return OK;
};

void RootNode::speak(DStr &s, SpeakMode mode)
{
    if (mode & SM_DESCRIBE)
        s += "[ROOT]";
    if (mode & SM_CONTENTS)
        Daddy::speak(s,mode);
}

eFlag RootNode::copy()
{
    E( startCopy() );
    E( contents.copy() );
    E( endCopy() );
    return OK;
}

/****************************************
E l e m e n t
a vertex with attributes, defs and contents
****************************************/

Element::Element(QName& aqname, Tree *_ownerT, Processor* proc, VTYPE avt /*= VT_ELEMENT_WF*/)
: Daddy(proc,avt), namespaces(proc, proc -> getArena()), name(proc), atts(proc -> getArena())
{
    name = aqname;
    ownerT = _ownerT;
};

Element::~Element()
{
    // not doing freeall since it's all in the arena
    // namespaces.freeall(FALSE);
    // atts.freeall(FALSE);
    namespaces.destructMembers();
    atts.destructMembers();
};

eFlag Element::execute(Context *c)
{
    E( proc -> outputter() -> eventElementStart(name));
    E( namespaces.execute(c) );
    E( atts.execute(c) );
    E( Daddy::execute(c) );
    removeBindings();
    E( proc -> outputter() -> eventElementEnd(name));
    return OK;
}

eFlag Element::newChild(Vertex *v)
{
    v -> setParent(this);
    if isAttr(v)
        atts.appendAndSetOrdinal(v);
    else
    {
        if (isNS(v))
            namespaces.appendAndSetOrdinal(v);
        else
            E( Daddy::newChild(v) );
    }
    return OK;
};

void Element::speak(DStr &ret, SpeakMode mode)
{
    if (mode & (SM_NAME | SM_CONTENTS))
    {
        ret += '<';
        name.speak(ret,mode);
        if (mode & SM_CONTENTS)
        {
            if (namespaces.number())
            {
                ret += ' ';
                namespaces.speak(ret, (SpeakMode)(mode | SM_INS_SPACES));
            }
            if (atts.number())
            {
                ret += ' ';
                atts.speak(ret, (SpeakMode)(mode | SM_INS_SPACES));
            };
/*
            commented this out as the closed empty tags
            confuse poor Navigator:

            if (!contents.number())
                ret += "/>";
            else
*/
            {
                ret += '>';
                contents.speak(ret,(SpeakMode)(mode & ~SM_INS_SPACES));
                //        ret += Str("</"); this caused errors!
                ret += "</";
                name.speak(ret,mode);
                ret += '>';
            };
        }
        else ret += '>';
    };
};

/*================================================================
removeBindings
    This removes all VARIABLE bindings for the variables
    that are children of this element. The param and with-param
    bindings are removed by the 'vars' list itself.
================================================================*/

void Element::removeBindings()
{
    Attribute *a;
    Vertex *v;
    for (int i = contents.number() - 1; i >= 0; i--)
    {
        v = contents[i];
        if (isXSLElement(v) && (cast(XSLElement*, v) -> op == XSL_VARIABLE))
        {
            QName q(proc);
            a = NZ(cast(XSLElement*, v) -> atts.find(XSLA_NAME));
            q.setLogical(a -> cont, &namespaces, FALSE);
            proc -> vars -> rmBinding(q);
        }
    }
}

eFlag Element::startCopy()
{
    E( proc -> outputter() -> eventElementStart(name) );
    E( namespaces.copy() );
    return OK;
}

eFlag Element::endCopy()
{
    E( proc -> outputter() -> eventElementEnd(name) );
    return OK;
}

eFlag Element::copy()
{
    E( startCopy() );
    E( atts.copy() );
    E( contents.copy() );
    E( endCopy() );
    return OK;
};

const QName& Element::getName() const
{
    return name;
}

/****************************************
A t t r i b u t e
****************************************/

Attribute::Attribute(const QName &aqname, const Str &acont, 
                     XSL_ATT aop, Processor* proc)
: Vertex(proc, (VTYPE) (VT_ATTRIBUTE_WF | 
    (aop != XSLA_NONE ? VT_XSL : 0))),
    name(proc), cont(proc -> getArena())
{
    expr = NULL;
    name = aqname;
    cont = acont;
    op = aop;
};

Attribute::~Attribute()
{
    if (expr)
        delete expr;
}

eFlag findAVTBrace(char *&p, char which, DStr &copybuf)
{
    char *p0 = p; // ,c;
    int len;
    copybuf.empty();
    while (*p)
    {
        /*
        if (((c = *p) == '"') || (c == '\''))
        {
            do p++; while (*p && (*p != c));
            SituationObj* situation = NULL;
            if (!*p) Err(situation, ET_INFINITE_LITERAL);
        };
        */
        if (*p != which) 
            p++;
        else
        {
            if (*(p+1) == which)
            {
                len = p+1 - p0;
                if (len)
                    copybuf.nadd(p0, len);
                p += 2;
                p0 = p;
            }
            else break;
        };
    };
    len = p - p0;
    if (len) copybuf.nadd(p0, len);
    return OK;
}

eFlag Attribute::buildExpr(Bool asTemplate, ExType ty)
{
    proc -> situation->pushCurrV(this);
    char *q;
    Expression *eadd;
    if (asTemplate)
    {
        DStr sadd;
        expr = new Expression(toE(parent),proc,EXF_STRINGSEQ);
        char *p = (char*) cont;
        while (*p)
        {
            E( findAVTBrace(q = p, '{', sadd) );
            if (!sadd.isEmpty())
            {
                eadd = new Expression(toE(parent),proc,EXF_ATOM);
                eadd -> setAtom(sadd);
                expr -> args.append(eadd);
            };
            if (!*q) break;
            //
            if (!*(p = q + 1)) break;
            E( findAVTBrace(q = p, '}',sadd) );
            if (!sadd.isEmpty())
            {
                eadd = new Expression(toE(parent),proc);
                E( eadd -> parse(sadd));
                expr -> args.append(eadd);
            };
            if (!*q) break;
            p = q + 1;
        }
    }
    else
    {
        expr = new Expression(toE(parent),proc);
        E( expr -> parse(cont, (Bool) (ty == EX_NODESET_PATTERN)) );
    }
    proc -> situation->popCurrent();
    return OK;
}

eFlag Attribute::execute(Context *c)
{
    assert(parent);
    if (isXSLElement(parent))
        return OK;
    proc -> situation->pushCurrV(this);
    E( proc -> outputter() -> eventAttributeStart(name) );
    DStr temp;
    E( value(temp,c) );
    E( proc -> outputter() -> eventData(temp) );
    E( proc -> outputter() -> eventAttributeEnd() );
    proc -> situation->popCurrent();
    return OK;
}

eFlag Attribute::value(DStr &ret, Context *c)
{
    if (expr)
    {
        Expression temp(toE(parent),proc);
        E( expr -> eval(temp, c) );
        ret = temp.tostring();
    }
    else
        ret = cont;
    return OK;
}

void Attribute::speak(DStr &ret, SpeakMode mode)
{
    if (mode & (SM_NAME | SM_CONTENTS))
        name.speak(ret,mode);
    if (mode & SM_CONTENTS)
    {
        ret += "=\"";
        // escape whitespace and quotes in value
        DStr escapedCont;
        const char* escNTQLG[] = 
            {escNewline, escTab, escQuote, escLess, escGreater, NULL};
        escapeChars(escapedCont, cont, "\n\t\"<>", escNTQLG);
        escapedCont.appendSelf(ret);
        ret += '\"';
    }
}

eFlag Attribute::startCopy()
{
    E( proc -> outputter() -> eventAttributeStart(name) );
    E( proc -> outputter() -> eventData(cont) );
    E( proc -> outputter() -> eventAttributeEnd() );
    return OK;
}

const QName& Attribute::getName() const
{
    return name;
}


/****************************************
A t t L i s t
****************************************/

Attribute *AttList::find(XSL_ATT what)
{
    int i;
    Attribute *a;
    for (i = 0; i < number(); i++)
    {
        a = cast(Attribute *, (*this)[i]);
        if (a -> op == what)
            return a;
    };
    return NULL;
}

/*****************************************************************
    N m S p a c e
*****************************************************************/

NmSpace::NmSpace(Phrase _prefix, Phrase _uri, Processor* proc)
: Vertex(proc, VT_NAMESPACE)
{
    prefix = _prefix;
    uri = _uri;
}

NmSpace::~NmSpace()
{
}

eFlag NmSpace::execute(Context *c)
{
    assert(parent);
/*
    made unnecessary by the shadowing inside OutputterObj:
    if (isXSLElement(parent) || 
        uri == proc -> stdPhrase(PHRASE_XSL_NAMESPACE) || 
        uri == proc -> stdPhrase(PHRASE_XML_NAMESPACE))
        return OK;
*/
    E( proc -> outputter() -> 
        eventNamespace(
	        proc -> dict().getKey(prefix), //was: proc -> aliasXL(prefix)),
            proc -> dict().getKey(uri)) );
    return OK;
}

void NmSpace::speak(DStr& s, SpeakMode mode)
{
    s += "xmlns";
    if (prefix != UNDEF_PHRASE)
    {
        s += ':';
        s += proc -> dict().getKey(prefix);
    }
    s += "=\"";
    s += proc -> dict().getKey(uri);
    s += '\"';
}

eFlag NmSpace::value(DStr &s, Context *c)
{
    s = proc -> dict().getKey(uri);
    return OK;
}

eFlag NmSpace::startCopy()
{
    E( proc -> outputter() -> eventNamespace(
        proc -> dict().getKey(prefix), 
        proc -> dict().getKey(uri)) );
    return OK;
}

/*****************************************************************
    N S L i s t
*****************************************************************/

NSList::~NSList()
{
    // what's this freeall doing here?? removing
    // freeall();
}

NmSpace *NSList::find(Phrase prefix) const
{
    NmSpace *nsitem;
    for (int i = 0; i < number(); i++)
        if ((nsitem = toNS((*this)[i])) -> prefix == prefix)
           return nsitem;
    return NULL;
}


eFlag NSList::resolve(Phrase &what, Bool defaultToo) const
{
    Bool emptystr = (what == UNDEF_PHRASE);
    if (emptystr && !defaultToo)
        return OK;
    NmSpace *p = find(what);
    if (!p)
    {
        if (emptystr)
            return OK;
        else
            Err1(proc -> situation, ET_BAD_PREFIX, (char*)(proc -> dict().getKey(what)));
    }
    else
        what = p -> uri;
    return OK;
}

void NSList::unresolve(Phrase &what) const
{
    assert(what != UNDEF_PHRASE);
    NmSpace *currNS;
    for (int i = 0; i < number(); i++)
    {
        currNS = toNS((*this)[i]);
        if (what == currNS -> uri)
        {
            what = currNS -> prefix;
            return;
        }
    };
    assert(0);
}

/*
appends all current namespace declarations as vertices to tree t
'other' is the list where they are being appended in t
*/

void NSList::giveCurrent(NSList &other, Tree *t) const
{
    const NmSpace *currNS;
    for (int i = number() - 1; i >= 0; i--)
    {
        currNS = toNS((*this)[i]);
        if (!other.find(currNS -> prefix))
            t -> appendVertex(
                new(proc -> getArena()) NmSpace(currNS -> prefix, currNS -> uri, proc));
    }
}

/****************************************
T e x t
****************************************/

Text::~Text()
{
}

eFlag Text::execute(Context *c)
{
    E( proc -> outputter() -> eventData(cont) );
    return OK;
}

eFlag Text::value(DStr& ret, Context *c)
{
    ret = cont;
    return OK;
}

void Text::speak(DStr &ret, SpeakMode mode)
{
    if (mode & SM_ESCAPE)
        cont.speakTerse(ret);
    else
        ret += cont;
};

eFlag Text::startCopy()
{
    E( proc -> outputter() -> eventData(cont) );
    return OK;
}


/****************************************
C o m m e n t
****************************************/

Comment::Comment(const Str& cont_, Processor* proc)
: 
Vertex(proc, VT_COMMENT), cont(proc -> getArena())
{
    cont = cont_;
}

Comment::~Comment()
{
}

eFlag Comment::execute(Context *c)
{
    return OK;
}

eFlag Comment::value(DStr& ret, Context *c)
{
    ret = cont;
    return OK;
}

void Comment::speak(DStr &ret, SpeakMode mode)
{
    ret += cont;
};

eFlag Comment::startCopy()
{
    E( proc -> outputter() -> eventCommentStart() );
    E( proc -> outputter() -> eventData(cont) );
    E( proc -> outputter() -> eventCommentEnd() );
    return OK;
}


/****************************************
P r o c I n s t r
****************************************/

ProcInstr::ProcInstr(Phrase name_, const Str& cont_, Processor* proc)
: 
Vertex(proc, VT_PI), cont(proc -> getArena()), name(proc)
{
    name.empty();
    name.local = name_;
    cont = cont_;
}

ProcInstr::~ProcInstr()
{
}

eFlag ProcInstr::execute(Context *c)
{
    return OK;
}

eFlag ProcInstr::value(DStr& ret, Context *c)
{
    ret = cont;
    return OK;
}

void ProcInstr::speak(DStr &ret, SpeakMode mode)
{
    ret += cont;
};

eFlag ProcInstr::startCopy()
{
    E( proc -> outputter() -> eventPIStart(name.getLocal()) );
    E( proc -> outputter() -> eventData(cont) );
    E( proc -> outputter() -> eventPIEnd() );
    return OK;
}

const QName& ProcInstr::getName() const
{
    return name;
}

/****************************************
X S L E l e m e n t
****************************************/

XSLElement::XSLElement(QName& aqname, Tree *_ownerT, XSL_OP code, Processor* proc)
:
Element(aqname, _ownerT, proc, VT_XSL_ELEMENT_WF)
{
    assert(code != XSL_NONE);
    op = code;
};


eFlag XSLElement::newChild(Vertex *v)
{
    // process defs
    //
    E( Element::newChild(v) );
    return OK;
};

eFlag XSLElement::execute(Context *c)
{
    Attribute *a;
    Bool didNotExecute = FALSE;
    // ???
    if (c -> isFinished())
        return OK;

    c->setCurrentNode(NULL);
    proc -> situation->pushCurrV(this);
    Vertex *v = c -> current();
    assert(v);

    switch(op)
    {
    case XSL_APPLY_TEMPLATES:
        {
            Attribute *aSelect = atts.find(XSLA_SELECT);
            Expression *e;
            if (aSelect)
                e = NZ(aSelect -> expr);
            else
            {
                e = new Expression(this, proc, EXF_LOCPATH);
                e -> setLS(
                    AXIS_CHILD, 
                    EXNODE_NODE);
            };

            // mode
            Bool addedMode = TRUE;
            a = atts.find(XSLA_MODE);
            if (a)
            {
                QName *m = new QName(proc);
                E( m -> setLogical(a -> cont,&namespaces,FALSE) );
                proc -> pushMode(m);
            }
            else if (proc -> getCurrentMode())
                proc -> pushMode(NULL);
            else addedMode = FALSE;
            
            Context *newc = c;
            E( e -> createContext( newc ) ); 
            if (!newc -> isVoid())
            {
                // process with-params
                E( contents.execute(c) );
                E( proc -> execute((Vertex*) NULL, newc) );
                // remove the prebindings introduced by with-param
                proc -> vars -> rmPrebindings();
            }
            else
                delete newc;
            if (addedMode)
                proc -> popMode();
            if (!aSelect)
                delete e;
        }; break;
    case XSL_IF:
        {
            a = NZ( atts.find(XSLA_TEST) );
            Expression boolexpr(this, proc);
            E( NZ(a -> expr) -> eval(boolexpr,c) );
            Bool boolval = boolexpr.tobool();
            if ((boolval) && (contents.number()))
//                E( proc -> execute(contents[0],c) );
                E( contents.execute(c) )
            else
                didNotExecute = TRUE;
        }; break;
    case XSL_CHOOSE:
        {
            Bool done = FALSE;
            for (int i = 0; i < contents.number() && !done; i++)
            {
                XSLElement *x = cast(XSLElement*,contents[i]);
                proc -> situation -> pushCurrV(x);
                if (x -> op == XSL_WHEN)
                {
                    E( NZ(NZ(x -> atts.find(XSLA_TEST)) 
                        -> expr) -> trueFor(c,done)     );
                    if (done)
                        E( x -> execute(c) );
                }
                else
                {
                    assert(x -> op == XSL_OTHERWISE);
                    E( x -> execute(c) );
                };
                proc -> situation -> popCurrent();
            };
        }; break;
    case XSL_WHEN: // condition tested by CHOOSE
        {
            E( contents.execute(c) );
        }; break;
    case XSL_OTHERWISE: // condition tested by CHOOSE
        {
            E( contents.execute(c) );
        }; break;
    case XSL_ELEMENT:
        {
            QName q(proc);
            DStr nameStr;
            E( NZ( atts.find(XSLA_NAME) ) -> value(nameStr, c) );
            E( q.setLogical(nameStr, &namespaces, TRUE) );
            E( proc -> outputter() -> eventElementStart(q) );
            E( Daddy::execute(c) );
            E( proc -> outputter() -> eventElementEnd(q) );
        }; break;

    case XSL_PROCESSING_INSTR:
        {
            QName q(proc);
            DStr nameStr;
            E( NZ( atts.find(XSLA_NAME) ) -> value(nameStr, c) );
            E( q.setLogical(nameStr, &namespaces, FALSE) );
            const Str& qloc = q.getLocal();
            if (( q.prefix != UNDEF_PHRASE ) || 
                strEqNoCase( qloc, "xml"))
                Err1(proc -> situation, E1_PI_TARGET, nameStr);
            E( proc -> outputter() -> eventPIStart( qloc ) );
            E( contents.execute(c) );
            E( proc -> outputter() -> eventPIEnd() ); 
        }; break;
    case XSL_COMMENT:
        {
            E( proc -> outputter() -> eventCommentStart() );
            E( contents.execute(c) );
            E( proc -> outputter() -> eventCommentEnd() ); 
        }; break;

    case XSL_ATTRIBUTE:
        {
            QName q(proc); 
            DStr nameStr;
            E( NZ( atts.find(XSLA_NAME) ) -> value(nameStr, c) );
            E( q.setLogical(nameStr, &namespaces, FALSE) );
            
            E( proc -> outputter() -> eventAttributeStart(q) );
            E( contents.execute(c) );
            E( proc -> outputter() -> eventAttributeEnd() );
        }; break;
    case XSL_TEXT:
        {
            Attribute *disableEsc = atts.find(XSLA_DISABLE_OUTPUT_ESC);
            if (disableEsc && disableEsc -> cont == (const char*)"yes")
                E( proc -> outputter() -> eventDisableEscapingForNext() );
            E( contents.execute(c) );
        }; break;
    case XSL_TEMPLATE:
        {
            proc -> vars -> startApplyOne();
            E( Daddy::execute(c) );
            proc -> vars -> endApplyOne();
        }; break;
    case XSL_STYLESHEET:
    case XSL_TRANSFORM:
        {
            Context *newc = c -> copy();

            // process globals and other top level elements
            for (int i = 0; i < contents.number(); i++)
            {
                Vertex *son = contents[i];
                if (isXSLElement(son) && (instrTable[toX(son) -> op].flags & ELEM_TOPLEVEL))
                {
                    switch(toX(son) -> op)
                    {
                    case XSL_TEMPLATE: break;
                    default: E( son -> execute(c) );
                    }
                }
            }

            proc -> vars -> startCall();
            E( proc -> execute((Vertex*) NULL, newc ));
            // newc has been deleted by now
            proc -> vars -> endCall();
        }; break;
    case XSL_VALUE_OF:
        {
            a = NZ( atts.find(XSLA_SELECT) );
            Expression temp(this, proc);
            E( NZ( a -> expr ) -> eval(temp, c) );
            // set output escaping
            Attribute *disableEsc = atts.find(XSLA_DISABLE_OUTPUT_ESC);
            if (disableEsc && disableEsc -> cont == (const char*)"yes")
                E( proc -> outputter() -> eventDisableEscapingForNext() );
            // dump contents
            Str cont = temp.tostring();
            E( proc -> outputter() -> eventData(cont) );
        };
        break;
    case XSL_COPY_OF:
        {
            a = NZ( atts.find(XSLA_SELECT) );
            Expression temp(this,proc);
            E( NZ( a -> expr ) -> eval(temp, c) );
            if (temp.type == EX_NODESET)
            {
                const Context& ctxt = temp.tonodesetRef();
                int k, kLimit = ctxt.getSize();
                for (k = 0; k < kLimit; k++)
                    E( ctxt[k] -> copy() );
            }
            else
            {
                Str cont = temp.tostring();
                E( proc -> outputter() -> eventData(cont) );
            }
        }; break;
    case XSL_COPY:
        {
            Vertex *curr = c -> current();
            E( curr -> startCopy() );
            E( contents.execute(c) );
            E( curr -> endCopy() );
        }; break;
    case XSL_FOR_EACH:
        {
            a = NZ( atts.find(XSLA_SELECT) );
            Context *newc = c;
            E( NZ( a -> expr ) -> createContext(newc));
            if (!newc -> isVoid() && contents.number())
            {
                E( proc -> execute(contents, newc) );
                // E( proc -> execute(contents[0],newc) );
                // proc disposes of the new context
            }
            else
                delete newc;
        };
        break;
    case XSL_CALL_TEMPLATE:
        {
            QName q(proc);
            a = NZ( atts.find(XSLA_NAME) );
            XSLElement *thatrule;
            E( q.setLogical(a -> cont, &namespaces, FALSE) );
            if (!(thatrule = proc -> rules.findByName(q)))
                Err(proc -> situation, ET_RULE_NOT_FOUND);

//            proc -> vars -> rmPrebindings();
            //process with-param
            E( contents.execute(c) );
            //execute the other rule
//            proc -> vars -> startLevel();
            E( thatrule -> execute(c) );
            proc -> vars -> rmPrebindings();
//            proc -> vars -> endLevel();
        }; break;
    case XSL_MESSAGE:
        {
           DStr msg;
           Expression *expr = new Expression(this, proc);
           if (contents.isEmpty())
           {
              expr -> setAtom(theEmptyString);
           } else {
              proc -> vars -> startNested();
              TreeConstructer *newTC;
              E( proc -> pushTreeConstructer(newTC, expr -> setFragment()) );
              E( contents.execute(c) );
              E( proc -> outputter() -> eventEndOutput() );
              E( proc -> popTreeConstructer(newTC) );
              proc -> vars -> endNested();
           };
            Expression *temp = new Expression(this, proc);
            E( expr -> eval(*temp,c) );
            temp -> pTree = expr -> pTree;
            expr -> pTree = NULL;
           a =  atts.find(XSLA_TERMINATE);
           if (a && a -> cont == (const char*) "yes")
           {
              Err1(proc -> situation, E1_XSL_MESSAGE_TERM, temp -> tostring());
           } else {
              Warn1(proc -> situation, W1_XSL_MESSAGE_NOTERM, temp -> tostring());
           };
           delete NZ(expr);
           delete NZ(temp);
        };
       break;
    case XSL_VARIABLE:
    case XSL_WITH_PARAM:
    case XSL_PARAM:
        {
            QName q(proc);

            Bool freeit = FALSE;
            Expression *expr = NULL;

            // there must be a 'name' attribute
            a = NZ(atts.find(XSLA_NAME));
            // stick the name into q referring to this element's namespace decl's
            q.setLogical(a -> cont, &namespaces, FALSE);

            // if there's a 'select', use the expression it gives
            a = atts.find(XSLA_SELECT);
            if (a)
                expr = NZ(a -> expr);

            // otherwise, construct a new expression; it may be an empty string
            // if this element has void content, or a result tree fragment otherwise 
            else
            {
                freeit = TRUE;
                expr = new Expression(this, proc);
                if (contents.isEmpty())
                    expr -> setAtom(theEmptyString);
                else // result tree fragment
                {
                    // starting a nesting will make the current prebindings
                    // invisible
                    proc -> vars -> startNested();
                    TreeConstructer *newTC;
                    E( proc -> pushTreeConstructer(newTC, expr -> setFragment()) );
                    // execute the inside to create the fragment
                    E( contents.execute(c) );
                    E( proc -> outputter() -> eventEndOutput() );
                    E( proc -> popTreeConstructer(newTC) );

                    // end the shadowing of preexisting bindings
                    proc -> vars -> endNested();
                };
            }
            //
            // evaluate the expression
            Expression *temp = new Expression(this, proc);
            E( expr -> eval(*temp,c) );
            temp -> pTree = expr -> pTree;
            expr -> pTree = NULL;

            // if the expression was newly constructed then drop it
            if (freeit)
                delete NZ(expr);

            // add the new binding
            switch(op)
            {
            case XSL_PARAM:
                {
                    E( proc -> vars -> addBinding(q, temp, FALSE) );
                }; break;
            case XSL_WITH_PARAM:
                {
                    E( proc -> vars -> addPrebinding(q, temp) );
                }; break;
            case XSL_VARIABLE:
                {
                    E( proc -> vars -> addBinding(q, temp, TRUE) );
                }; break;
            };
            // delete temp;  - deleted when removing the binding
        }; break;
    case XSL_OUTPUT:
    case XSL_NAMESPACE_ALIAS:
        // these were processed during parse
        break;
    // unsupported instructions that are not considered harmful
    case XSL_STRIP_SPACE:
    case XSL_PRESERVE_SPACE:
        Warn1(proc -> situation, W1_UNSUPP_XSL, xslOpNames[op]);
        break;
    default: 
        Err1(proc -> situation, E1_UNSUPP_XSL, xslOpNames[op]);
    };

    //remove the variable bindings that occured inside this element
    if ((op != XSL_TEMPLATE) && (op != XSL_TRANSFORM) && (op != XSL_STYLESHEET) && 
        (op != XSL_FOR_EACH) && !didNotExecute)
        removeBindings();
    proc -> situation->popCurrent();
    return OK;
}

Expression *XSLElement::getAttExpr(XSL_ATT code)
{
    Attribute *a = atts.find(code);
    return a ? (a -> expr) : NULL;
}

int XSLElement::strip()
{
    if (op != XSL_TEXT)
        return /* defs.strip() + */ contents.strip();
    else return 0;
}


/*................................................................
findAttNdx()
    returns the index of given attribute in the attribute table
    or -1 if not found
................................................................*/

int findAttNdx(InstrTableItem &iitem, Attribute *a)
{
    for (int i = 0; i < iitem.maxAtts; i++)
        if (iitem.att[i].attCode == a -> op) return i;
    return -1;
}

/*================================================================
checkAtts
    called when all attributes have been parsed in. Returns FALSE
    iff they are OK. Return value of TRUE means that the end tag handler
    signals error.
================================================================*/

eFlag XSLElement::checkAtts()
{
    InstrTableItem &instrData = instrTable[op];
    assert(instrData.op == op);
    int
        attNdx,
        reqCount = 0;
    Attribute *a;
    proc -> situation->pushCurrV(this);
    for (int i = 0; i < atts.number(); i++)
    {
        a = cast(Attribute*, atts[i]);
        if ((attNdx = findAttNdx(instrData,a)) == -1)
            Err1(proc -> situation, ET_BAD_ATTR,a -> name.getname());
        if (instrData.att[attNdx].required)
            reqCount++;
        if (instrData.att[attNdx].exprType != EX_NONE)
            E( a -> buildExpr(
                instrData.att[attNdx].avtemplate,
                instrData.att[attNdx].exprType) );
    };
    if (reqCount < instrData.reqAtts)
        Err(proc -> situation, ET_REQ_ATTR);
    proc -> situation->popCurrent();
    return OK;
};

//................................................................

void XSLElement::checkExtraChildren(int& k)
{
    Vertex *w;
    XSL_OP hisop;
    int status = 0;
    for (k = 0; k < contents.number(); k++)
    {
        w = contents[k];
        if (!isXSLElement(w)) return;
        hisop = toX(w) -> op;
        switch(op)
        {
        case XSL_APPLY_TEMPLATES:
            if ((hisop != XSL_SORT) && (hisop != XSL_WITH_PARAM)) return;
            break;
        case XSL_ATTRIBUTE_SET:
            if (hisop != XSL_ATTRIBUTE) return;
            break;
        case XSL_CALL_TEMPLATE:
            if (hisop != XSL_WITH_PARAM) return;
            break;
        case XSL_CHOOSE:
            switch(hisop)
            {
            case XSL_WHEN:
                {
                    if (status <= 1) status = 1;
                    else return;
                }; break;
            case XSL_OTHERWISE:
                {
                    if (status == 1) status = 2;
                    else return;
                }; break;
            default: return;
            }; 
            break;
        case XSL_FOR_EACH:
            if (hisop != XSL_SORT) return;
            break;
        case XSL_STYLESHEET:
        case XSL_TRANSFORM:
            if (hisop != XSL_WITH_PARAM) return;
            break;
        case XSL_TEMPLATE:
            if (hisop != XSL_PARAM) return;
            break;
        default: 
            return;
        }
    }
}

eFlag XSLElement::checkChildren()
{
    InstrTableItem &iData = instrTable[op];
    assert(iData.op == op);

    if (iData.flags & ELEM_CONT_PCDATA)
        return OK;

    proc -> situation->pushCurrV(this);
    if (!(iData.flags & ELEM_CONT) && contents.number())
        Err1(proc -> situation, E_ELEM_MUST_EMPTY, xslOpNames[op]);

    int firstAfter = 0;
    if (iData.flags & ELEM_CONT_EXTRA)
        checkExtraChildren(firstAfter);

    for (int k = firstAfter; k < contents.number(); k++)
    {
        Vertex *w = contents[k];
        if (isText(w) || (isElement(w) && !isXSLElement(w)))
        {
            if (!(iData.flags & ELEM_CONT_INSTR))
                Err1(proc -> situation, E_ELEM_CONT_TEXT_OR_LRE, xslOpNames[op]);
        }
        else
        {
            if (isXSLElement(w))
            {
                int hisflags = instrTable[toX(w) -> op].flags;
                if (!(((hisflags & ELEM_TOPLEVEL) && (iData.flags & ELEM_CONT_TOPLEVEL)) ||
                    ((hisflags & ELEM_INSTR) && (iData.flags & ELEM_CONT_INSTR))))
                    Err2(proc -> situation, E_ELEM_CONTAINS_ELEM, xslOpNames[op], xslOpNames[toX(w) -> op]);
            }
            else
                Err1(proc -> situation, E_BAD_ELEM_CONTENT, xslOpNames[op]);
        }
    };
    proc -> situation->popCurrent();
    return OK;
}

