/* 
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 * 
 * The Original Code is the Sablotron XSLT Processor.
 * 
 * The Initial Developer of the Original Code is Ginger Alliance Ltd.
 * Portions created by Ginger Alliance are Copyright (C) 2000 Ginger
 * Alliance Ltd. All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL"), in which case the provisions of the GPL are applicable 
 * instead of those above.  If you wish to allow use of your 
 * version of this file only under the terms of the GPL and not to
 * allow others to use your version of this file under the MPL,
 * indicate your decision by deleting the provisions above and
 * replace them with the notice and other provisions required by
 * the GPL.  If you do not delete the provisions above, a recipient
 * may use your version of this file under either the MPL or the
 * GPL.
 */

//
//  output.cpp
//

#include "output.h"
#include "uri.h"
#include "proc.h"

#define NONEMPTY_ELEMENT FALSE
#define EMPTY_ELEMENT TRUE


enum OutputterFlags
{
    HTML_ELEMENT = 1,
    HTML_SCRIPT_OR_STYLE = 2,
    CDATA_SECTION = 4
};

#define IF_SAX1( FUNC )\
    { if (mySAXHandler)\
        { mySAXHandler -> FUNC(mySAXUserData); }}
#define IF_SAX2( FUNC, ARG1 )\
    { if (mySAXHandler)\
        { mySAXHandler -> FUNC(mySAXUserData, (ARG1)); }}
#define IF_SAX3( FUNC, ARG1, ARG2 )\
    { if (mySAXHandler)\
        { mySAXHandler -> FUNC(mySAXUserData, (ARG1), (ARG2)); }}

#define IF_PH1( FUNC )\
    { if (physical)\
        { physical -> FUNC(); }}
#define IF_PH2( FUNC, ARG1)\
    { if (physical)\
        { physical -> FUNC(ARG1); }}
#define IF_PH3( FUNC, ARG1, ARG2 )\
    { if (physical)\
        { physical -> FUNC((ARG1), (ARG2)); }}

// STRING_ITEMS_COUNT (output.h) must be in sync with the size
// of the following table:

XSL_ATT outputStringAtts[STRING_ITEMS_COUNT + 1] = {
    XSLA_VERSION, XSLA_ENCODING, XSLA_OMIT_XML_DECL, XSLA_STANDALONE,
    XSLA_DOCTYPE_PUBLIC, XSLA_DOCTYPE_SYSTEM, XSLA_INDENT, XSLA_MEDIA_TYPE,
    XSLA_NONE
};

const char* theEmptyHTML40Tags[] = 
{
    "area", "base", "basefont", "br", "col", "frame", "hr", "img",
    "input", "isindex", "link", "meta", "param", NULL
};

Bool isEmptyHTMLTag(const Str& name)
{
    int ndx = lookupNoCase(name, theEmptyHTML40Tags);
    // return TRUE iff found
    return !!theEmptyHTML40Tags[ndx];
}

//
//  OutputDefinition
//

Bool checkYesNo(const Str& what)
{
    return(what == (const char*) "yes" || what == (const char*) "no");
}

OutputDefinition::OutputDefinition(Processor *proc_)
    : proc(proc_), method(proc_)
{
}

OutputDefinition::~OutputDefinition()
{
    cdataElems.freeall(FALSE);
}

int lookupAttCode(XSL_ATT* table, XSL_ATT what)
{
    int i;
    for (i = 0; table[i] != XSLA_NONE && table[i] != what; i++) {};
    return (table[i] == XSLA_NONE ? -1 : i);
}


eFlag OutputDefinition::setItemStr_(XSL_ATT itemId, const Str& value, Bool doCheck, Bool soft)
{
    switch(itemId)
    {
    case XSLA_OMIT_XML_DECL:
    case XSLA_STANDALONE:
    case XSLA_INDENT:
        {
            if (!checkYesNo(value))
                Err1(proc -> situation, E1_ATTR_YES_NO, xslAttNames[itemId]);
        }; break;
    };
    int index = lookupAttCode(outputStringAtts, itemId);
    assert(index >= 0);
    if (!stringItems[index].isEmpty())
    {
        if (doCheck)
            Warn1(proc -> situation, W1_OUTPUT_ATTR, xslAttNames[itemId]);
        if (!soft)
            stringItems[index] = value;
    }
    else
        stringItems[index] = value;
    return OK;
}


eFlag OutputDefinition::setItemStrCheck(XSL_ATT itemId, const Str& value)
{
    E( setItemStr_(itemId, value, TRUE, FALSE) );
    return OK;
}

eFlag OutputDefinition::setItemStrIfNot(XSL_ATT itemId, const Str& value)
{
    E( setItemStr_(itemId, value, FALSE, TRUE) );
    return OK;
}

eFlag OutputDefinition::setItemStrForce(XSL_ATT itemId, const Str& value)
{
    E( setItemStr_(itemId, value, FALSE, FALSE) );
    return OK;
}

eFlag OutputDefinition::setItemQName(XSL_ATT itemId, const QName& value, Bool doCheck)
{
    switch(itemId)
    {
    case XSLA_METHOD:
        {
            if (!method.isEmpty() && doCheck)
                Warn1(proc -> situation, W1_OUTPUT_ATTR, xslAttNames[itemId]);
            method = value;
        }; break;
    case XSLA_CDATA_SECT_ELEMS:
        cdataElems.append(new QName(value)); break;
    default: 
        assert(!"setItemQName()");
    };
    return OK;
}

const Str& OutputDefinition::getValueStr(XSL_ATT itemId) const
{
    int index = lookupAttCode(outputStringAtts, itemId);
    assert(index >= 0);
    return stringItems[index];
}

const QName& OutputDefinition::getValueQName(XSL_ATT itemId) const
{
    assert(itemId == XSLA_METHOD);
    return method;
};

Bool OutputDefinition::askQNameList(XSL_ATT itemId, const QName &what) const
{
    assert(itemId == XSLA_CDATA_SECT_ELEMS);
    return (cdataElems.find(what) != NULL);
}

OutputMethod OutputDefinition::getMethod() const
{
    const Str& method_ = getValueQName(XSLA_METHOD).getLocal();
    if (method_ == (char*) "html")
        return OUTPUT_HTML;
    else if (method_ == (char*) "text")
        return OUTPUT_TEXT;
    else if (method_ == (char*) "xml")
        return OUTPUT_XML;
    else return OUTPUT_UNKNOWN;
}

// FIXME: does it make sense to default to UTF-8 ?
// tom: changing == to eqNoCase
// tom: changing utf16 to utf-16 and ascii to us-ascii
Encoding OutputDefinition::getEncoding() const
{
    const Str& enc_ = getValueStr(XSLA_ENCODING);
    if (enc_.eqNoCase((char*) "utf-16"))
      return ENC_UTF16;
    else if (enc_.eqNoCase((char*) "us-ascii"))
      return ENC_ASCII;
    else if (enc_.eqNoCase((char*) "iso-8859-1"))
        return ENC_8859_1;
    else if (enc_.eqNoCase((char*) "iso-8859-2"))
        return ENC_8859_2;
    else if (enc_.eqNoCase((char*) "euc-jp"))
        return ENC_EUCJP;
    else if (enc_.eqNoCase((char*) "shift-jis"))
        return ENC_SJIS;
    else return ENC_UTF8;
}

eFlag OutputDefinition::setDefaults()
{
    OutputMethod meth = getMethod();
    assert( meth != OUTPUT_UNKNOWN );
    char strYes[] = "yes", strNo[] = "no";
    E( setItemStrIfNot(XSLA_ENCODING, "UTF-8") );
    switch(meth)
    {
    case OUTPUT_XML:
        {
            E( setItemStrIfNot(XSLA_VERSION, "1.0") );
            E( setItemStrIfNot(XSLA_INDENT, strNo) );
            E( setItemStrIfNot(XSLA_MEDIA_TYPE, "text/xml") );
            E( setItemStrIfNot(XSLA_OMIT_XML_DECL, strNo) );
        }; break;
    case OUTPUT_HTML:
        {
            E( setItemStrIfNot(XSLA_VERSION, "4.0") );
            E( setItemStrIfNot(XSLA_INDENT, strYes) );
            E( setItemStrIfNot(XSLA_MEDIA_TYPE, "text/html") );
            E( setItemStrIfNot(XSLA_OMIT_XML_DECL, strYes) );
        }; break;
    case OUTPUT_TEXT:
        {
            E( setItemStrIfNot(XSLA_INDENT, strNo) );
            E( setItemStrIfNot(XSLA_MEDIA_TYPE, "text/plain") );
            E( setItemStrIfNot(XSLA_OMIT_XML_DECL, strYes) );
        }; break;
    }
    return OK;
}



//
//
//  PhysicalOutputLayerObj
//
//

#define sendStrEsc(STRING,ESCAPING) sendOut((STRING), (STRING.length()), (ESCAPING))
#define sendStr(STRING)     sendStrEsc((STRING), ESCAPING_NONE)
#define sendLit(LITERAL)    sendOut((LITERAL), sizeof(LITERAL) - 1, ESCAPING_NONE)

PhysicalOutputLayerObj::PhysicalOutputLayerObj(Processor *proc_)
{
    curr = 0;
    proc = proc_;
}

PhysicalOutputLayerObj::~PhysicalOutputLayerObj()
{
}

eFlag PhysicalOutputLayerObj::setOptions(DataLine *targetDataLine_, OutputDefinition *outDef_)
{
    targetDataLine = targetDataLine_;
    outDef = outDef_;
    method = outDef -> getMethod();
    if (method != OUTPUT_UNKNOWN)
        E( outDef -> setDefaults() );
    enc = outDef -> getEncoding();
    const Str& encStr = outDef -> getValueStr(XSLA_ENCODING);
    if (!encStr.isEmpty() && !utf8CanRecodeTo(encStr))
    {
        Warn1(proc -> situation, W1_UNSUPP_OUT_ENCODING, encStr);
        E( outDef -> setItemStrForce(XSLA_ENCODING, "UTF-8") );
    }
    return OK;
}

eFlag PhysicalOutputLayerObj::outputDone()
{
    E( flushBuffer() );
    return OK;
}

eFlag PhysicalOutputLayerObj::outputElementStart(const Str& name,
    const NamespaceStack& namespaces, const int namespace_index,
    const QNameStrList& atts, Bool isEmpty)
{
    // begin output of start tag: output element name

    sendLit("<");
    E( sendStr(name) );

    // output namespace declarations

    int i;
    const Str* prefix;
    for (i = namespace_index; i < namespaces.number(); i++)
    {
        prefix = &(namespaces[i] -> prefix);
        if (! namespaces.isHidden(*prefix))
       {
          sendLit(" xmlns");
          if (!prefix -> isEmpty())
          {
             sendLit(":");
             E( sendStr(*prefix) );
          }
          sendLit("=\"");
          E( sendStrEsc(namespaces[i] -> uri, 
                        method == OUTPUT_HTML ?
                        ESCAPING_HTML_URI : ESCAPING_URI) );
          sendLit("\"");
       };
    };

    // output attributes

    for (i = 0; i < atts.number(); i++)
    {
        sendLit(" ");
        const QName& attQName = atts[i] -> key;
        if (attQName.prefix != UNDEF_PHRASE)
        {
            E( sendStr(attQName.getPrefix()) );
            sendLit(":");
        };
        E( sendStr(atts[i] -> key.getLocal()) );
        sendLit("=\"");
        E( sendStrEsc(atts[i] -> value, 
            method == OUTPUT_HTML ? ESCAPING_HTML_ATTR : ESCAPING_ATTR) );
        sendLit("\"");
    };

    // close the tag

    if (!isEmpty)
        sendLit(">");
    else
    {
        if (method != OUTPUT_HTML)
            sendLit("/>");
        else
        {
            if (!isEmptyHTMLTag(name))
            {
                sendLit("></");
                sendStr(name);
                sendLit(">");
            }
            else
                sendLit(">");
        }
    };
    return OK;
}



eFlag PhysicalOutputLayerObj::outputElementEnd(const Str& name, Bool isEmpty)
{
    if (!isEmpty)
    {
        sendLit("</");
        E( sendStr(name) );
        sendLit(">");
    };
    return OK;
}


eFlag PhysicalOutputLayerObj::outputComment(const Str& contents)
{
    sendLit("<!--");
    const char *p = contents, *p_was = p;
    int len = contents.length();
    Bool trailingHyphen = len ? (contents[len - 1] == '-') : FALSE;
    while (*p)
    {
        E( sendOutUntil(p, len - (p - p_was), ESCAPING_NONE, "--") );
        if (*p)
        {
            sendLit("- -");
            p += 2;
        }
    };
    if (trailingHyphen)
        sendLit(" ");
    sendLit("-->");
    return OK;
}


eFlag PhysicalOutputLayerObj::outputCDataSection(const Str& contents)
{         
    const char *p = contents, *p_was = p;
    if (!*p)
        return OK;
    sendLit("<![CDATA[");
    while (*p)
    {
        E( sendOutUntil(p, contents.length() -(int)(p - p_was), ESCAPING_NONE, "]]>") );
        if (*p)
        {
            sendLit("]]]]><![CDATA[>");
            p += 3;
        }
    };
    sendLit("]]>");
    return OK;
}

eFlag PhysicalOutputLayerObj::outputPI(const Str& target, const Str& data)
{
    sendLit("<?");
    E( sendStr(target) );
    sendLit(" ");
    E( sendStr(data) );
    if (method == OUTPUT_HTML)
        sendLit(">");
    else
        sendLit("?>"); 
    return OK;
}

eFlag PhysicalOutputLayerObj::outputDTD(
    const Str& name, const Str& publicId, const Str& systemId)
{
    if (method == OUTPUT_TEXT)
        return OK;
    sendLit("<!DOCTYPE ");
    switch(method)
    {
    case OUTPUT_XML:
        {
            E( sendStr(name) );
            if (!systemId.isEmpty())
            {
                if (!publicId.isEmpty())
                {
                    sendLit(" PUBLIC \"");
                    E( sendStrEsc(publicId, ESCAPING_URI) );
                    sendLit("\"");
                }
                else
                    sendLit(" SYSTEM");
                sendLit(" \"");
                E( sendStrEsc(systemId, ESCAPING_URI) );
                sendLit("\"");
            }
        }; break;
    case OUTPUT_HTML:
        {
            sendLit("html");
            if (!publicId.isEmpty())
            {
                sendLit(" PUBLIC \"");
                E( sendStrEsc(publicId, ESCAPING_URI) );
                sendLit("\"");
            }
            if (!systemId.isEmpty())
            {
                if (publicId.isEmpty())
                    sendLit(" SYSTEM");
                sendLit(" \"");
                E( sendStrEsc(systemId, ESCAPING_URI) );
                sendLit("\"");
            };
        }; break;
    };
    sendLit(">\xa");
    return OK;
}

eFlag PhysicalOutputLayerObj::outputText(const Str& contents, Bool disableEsc,
                                         Bool inHTMLSpecial)
{
    switch(method)
    {
    case OUTPUT_XML:
        {
            E( sendStrEsc(contents, disableEsc ? ESCAPING_NONE : ESCAPING_LT_AMP) );
        }; break;
    case OUTPUT_HTML:
        {
            E( sendStrEsc(contents, (disableEsc || inHTMLSpecial) ? ESCAPING_NONE : ESCAPING_LT_AMP) );
        }; break;
    case OUTPUT_TEXT:
        {
            E( sendStrEsc(contents, ESCAPING_NONE) );
        }; break;
    };
    return OK;
}


eFlag PhysicalOutputLayerObj::flushBuffer()
{
    E( targetDataLine -> save(buffer, curr) );
    curr = 0;
    return OK;
}

int PhysicalOutputLayerObj::writeCharacterRef(char* dest, const char *src, EscMode escapeMode)
{
    char *dest_was = dest;
    if (escapeMode == ESCAPING_URI || escapeMode == ESCAPING_HTML_URI)
    {
        int i, iLimit = utf8SingleCharLength(src);
        for (i = 0; i < iLimit; i++)
            dest += sprintf(dest, "%%%02x", src[i]);
        return (int)(dest - dest_was);
    }
    else
        return sprintf(dest, "&#%lu;", utf8CharCode(src));
}

// data is assumed null-terminated in the following

eFlag PhysicalOutputLayerObj::sendOutUntil(const char *& data, int length,
    EscMode escapeMode, const char* stoppingText)
{
    char *p = strstr(data, stoppingText);
    int sending = (p ? p - data : length);
    E( sendOut(data, sending, escapeMode) );
    data += sending;
    return OK;
}



#define minimum(a,b) (((a) < (b)) ? (a) : (b))

eFlag PhysicalOutputLayerObj::sendOut(const char* data, int length,
                                      EscMode escapeMode)
{
    int count = 0, 
        srcCharLen, destCharLen;
    Bool served;
    
    while (count < length)
    {
        srcCharLen = destCharLen = 1;   // default
        served = FALSE;
        switch(*data)
        {
        case '<':
            {
                switch(escapeMode)
                {
                case ESCAPING_URI:
                case ESCAPING_HTML_URI:
                    {
                        E( sendOut(smallBuf, 
                            writeCharacterRef(smallBuf, data, escapeMode),
                            ESCAPING_NONE) );
                        served = TRUE;
                    }; break;
                case ESCAPING_ATTR:
                case ESCAPING_LT_AMP:
                    {
                        E( sendLit("&lt;") );
                        served = TRUE;
                    }; break;
                }
            }; break;
        case '&':
            {
                switch(escapeMode)
                {
                case ESCAPING_HTML_URI:
                case ESCAPING_URI:
                    {
                        E( sendOut(smallBuf, 
                            writeCharacterRef(smallBuf, data, escapeMode),
                            ESCAPING_NONE) );
                        served = TRUE;
                    }; break;
                case ESCAPING_HTML_ATTR:
                    {
                        if (data[1] == '{')
                            break;
                    }; // no break
                case ESCAPING_ATTR:
                case ESCAPING_LT_AMP:
                    {
                        E( sendLit("&amp;") );
                        served = TRUE;
                    }; break;
                }
            }; break;
        case '\"':
            {
                switch(escapeMode)
                {
                case ESCAPING_HTML_ATTR:
                case ESCAPING_ATTR:
                    {
                        E( sendLit("&quot;") );
                        served = TRUE;
                    }; break;
                };
            }; break;
        case 9:
        case 10:
        case 13:
            {
                switch(escapeMode)
                {
                case ESCAPING_URI:
                case ESCAPING_HTML_URI:
                case ESCAPING_ATTR:
                case ESCAPING_HTML_ATTR:
                    {
                        E( sendOut(smallBuf, 
                            writeCharacterRef(smallBuf, data, escapeMode),
                            ESCAPING_NONE) );
                        served = TRUE;
                    }; break;
                }
            }; break;
        };
        if (!served)
        {
            if (*(const unsigned char*)data < 0x80)
            {
                if (enc == ENC_UTF16)
                {
                    *(unsigned short*)(buffer + curr) = 
                        (unsigned short)((unsigned char)*data);
                    destCharLen = 2;
                }
                else 
                    buffer[curr] = *data;
            }
            else
            {
                srcCharLen = utf8SingleCharLength(data);
                if (enc == ENC_UTF8)
                    memcpy(buffer + curr, data, destCharLen = srcCharLen);
                else
                {
                    destCharLen = utf8Recode(smallBuf, data, enc);
                    if (destCharLen)
                        memcpy(buffer + curr, smallBuf, destCharLen);
                    else
                    {
                        destCharLen = writeCharacterRef(smallBuf, data, escapeMode);
                        if (escapeMode == ESCAPING_NONE)
                            Err1(proc -> situation, E1_BAD_CHAR_IN_ENC, smallBuf)
                            else
                        {
                            E( sendOut(smallBuf, destCharLen, ESCAPING_NONE) );
                            served = TRUE;
                        }   // else
                    }       // else       
                }           // else
            }               // else
        }                   // if (!served)
    data += srcCharLen;
    count += srcCharLen;
    if (!served)
        curr += destCharLen;
    if (curr > OUTPUT_BUFFER_LIMIT)
        flushBuffer();
    }                       // while    
    return OK;
}


eFlag PhysicalOutputLayerObj::setMethodByDefault(OutputMethod method_)
{
    QName q(proc);
    assert(method == OUTPUT_UNKNOWN);
    switch( method = method_ )
    {
    case OUTPUT_XML:
        q.setLocal("xml"); break;
    case OUTPUT_HTML:
        q.setLocal("html"); break;
    default:
        assert(!"PhysicalOutputLayerObj::setMethod()");
    }
    E( NZ(outDef) -> setItemQName(XSLA_METHOD, q, TRUE) );
    E( outDef -> setDefaults() );
    return OK;
}

//
//
//      FrontMatter
//
//

eFlag FrontMatter::appendConstruct(FrontMatterKind kind, const Str& string1,
    const Str& string2, Bool disableEsc)
{
    FrontMatterItem* item;

    situation = NULL;

    M( situation, item = new FrontMatterItem );
    item -> kind = kind;
    item -> string1 = string1;
    item -> string2 = string2;
    item -> disableEsc = disableEsc;
    append(item);
    return OK;
};


//
//
//  OutputterObj
//
//


Str* nameForSAX(const QName& q)
{
    if (q.getUri().isEmpty())
        return new Str(q.getLocal());
    DStr temp = q.getUri();
    temp += THE_NAMESPACE_SEPARATOR;
    temp += q.getLocal();
    return new Str(temp);
};

OutputterObj::OutputterObj(Processor *proc_)
: proc(proc_), 
  currElement(proc_), currAttName (proc_), 
  currAtts(proc_)

{
    outputEscaping = TRUE;
    physical = NULL;
    mySAXHandler = NULL;
    method = OUTPUT_UNKNOWN;
    noElementYet = TRUE;
    noHeadYet = TRUE;       // to recognize the first HEAD element
    currNamespaces.appendConstruct("xsl",theXSLTNamespace,true);
    currNamespaces.appendConstruct("xml",theXMLNamespace,true);
}

OutputterObj::~OutputterObj()
{
    history.freeall(FALSE);
    front.freeall(FALSE);
    currNamespaces.freeall(FALSE);
    cdelete(physical);
}

eFlag OutputterObj::setOptions(DataLine *targetDataLine_, OutputDefinition *outDef_)
{
    if (targetDataLine_)
    {
        M( proc -> situation, physical = new PhysicalOutputLayerObj(proc) );
        E( physical -> setOptions(targetDataLine_, outDef_) ); 
    };
    outDef = NZ(outDef_);
    return OK;
}

eFlag OutputterObj::setOptionsSAX(SAXHandler *streaming_, void* userData_)
{
    mySAXHandler = streaming_;
    mySAXUserData = userData_;
    return OK;
}

eFlag OutputterObj::reportXMLDeclIfMust()
{
    if (!physical || method == OUTPUT_UNKNOWN || 
        outDef -> getValueStr(XSLA_OMIT_XML_DECL) == (const char*)"yes")
        return OK;
    DStr declText = "version=\"";
    declText += outDef -> getValueStr(XSLA_VERSION);
    declText += "\" encoding=\"";
    declText += outDef -> getValueStr(XSLA_ENCODING);
    declText += '\"';
    const Str &standaloneText = outDef -> getValueStr(XSLA_STANDALONE);
    if (!standaloneText.isEmpty())
    {
        declText += " standalone=\"";
        declText += standaloneText;
        declText += '\"';
    };
    E( physical -> outputPI(Str("xml"), declText) );
    return OK;
}

eFlag OutputterObj::reportDTDIfMust(const QName& docElementName)
// to be called only after the output method is determined
{
    assert(method == OUTPUT_XML || method == OUTPUT_HTML);
    if (!physical)
        return OK;
    const Str& 
        DTSystem = outDef -> getValueStr(XSLA_DOCTYPE_SYSTEM),
        DTPublic = outDef -> getValueStr(XSLA_DOCTYPE_PUBLIC);        
    Bool writeDTD;
    switch(method)
    {
    case OUTPUT_XML:
        writeDTD = !DTSystem.isEmpty();
        break;
    case OUTPUT_HTML:
        writeDTD = !DTSystem.isEmpty() || !DTPublic.isEmpty();
        break;
    default:
        writeDTD = FALSE;
    }
    if (writeDTD)
    {
        E( physical -> outputDTD( docElementName.getname(), DTPublic, DTSystem) );
    };
    return OK;
}


eFlag OutputterObj::reportFront()
{
    assert(method != OUTPUT_UNKNOWN);
    int i, frontCount = front.number();
    FrontMatterItem *item;
    for(i = 0; i < frontCount; i++)
    {
        item = front[i];
        switch(item -> kind)
        {
        case FM_TEXT:
            {
                if (item -> disableEsc)
                    E( eventDisableEscapingForNext() );
                E( eventData(item -> string1) );
            }; break;
        case FM_PI:
            {
                E( eventPIStart(item -> string1) );
                E( eventData(item -> string2) );
                E( eventPIEnd() );
            }; break;
        case FM_COMMENT:
            {
                E( eventCommentStart() );
                E( eventData(item -> string1) );
                E( eventCommentEnd() );
            };
        }
    }
    return OK;
}


eFlag OutputterObj::eventBeginOutput()
{
    if (physical)
    {
        method = outDef -> getMethod();
        if (method != OUTPUT_UNKNOWN)
            E( reportXMLDeclIfMust() );
    }
    // initialize the SAX interface
    IF_SAX1( startDocument ); 
    state = STATE_OUTSIDE;
    return OK;
}


eFlag OutputterObj::eventElementStart(const QName& name)
{
    if (noElementYet)
    {
        noElementYet = FALSE;
        if (physical)
        {
            if (method == OUTPUT_UNKNOWN)
            {
                if (name.getUri().isEmpty() && name.getLocal().eqNoCase("html"))
                    method = OUTPUT_HTML;
                else
                    method = OUTPUT_XML;
                E( physical -> setMethodByDefault(method) );
                E( reportXMLDeclIfMust() );
                E( reportFront() );
            };
            // If this is an included stylesheet, output method cd be TEXT here
            if (method == OUTPUT_XML || method == OUTPUT_HTML) {
                E( reportDTDIfMust( name ) );
            }
        };
    }

    switch(state)
    {
    case STATE_OUTSIDE:
    case STATE_IN_MARKUP:
    case STATE_IN_ELEMENT:
        {
            // reportStartTag also sends the event to SAX
            E( reportStartTag(NONEMPTY_ELEMENT) );
            E( reportCurrData() );
            pushLevel(name);
            // this sets state to STATE_IN_MARKUP
        };
        break;
    case STATE_IN_COMMENT:
    case STATE_IN_PI:
        Err( proc -> situation, E_ELEM_IN_COMMENT_PI );
    default:
        assert(!"eventElementStart");
    };

    // determine whether we're in a HEAD html element so a META tag
    // needs to be added

    return OK;
}

eFlag OutputterObj::throwInMeta()
{
    noHeadYet = FALSE;
    Str metaName("meta");
    if (physical)
    {
        QNameStrList metaAtts(proc);
        QName 
            httpEquivName(proc), 
            contentName(proc);
        httpEquivName.setLocal("http-equiv");
        contentName.setLocal("content");
        metaAtts.appendConstruct(httpEquivName, Str("Content-Type"));
        DStr contentValue = 
            NZ(outDef) -> getValueStr(XSLA_MEDIA_TYPE) + "; charset=" +
            outDef -> getValueStr(XSLA_ENCODING);
        metaAtts.appendConstruct(contentName, contentValue);
        E( physical -> outputElementStart(metaName, currNamespaces,
            getFirstOwnNS(), metaAtts, TRUE) );
        E( physical -> outputElementEnd(metaName, TRUE) );
        state = STATE_IN_ELEMENT;
        metaAtts.freeall(FALSE);
    };
    return OK;
}
    


eFlag OutputterObj::eventElementEnd(const QName& name)
{
    switch(state)
    {
    case STATE_IN_MARKUP:
        E( reportStartTag(EMPTY_ELEMENT) ); 
        break;
    case STATE_IN_ELEMENT:
        {
            E( reportCurrData() );
            IF_PH3( outputElementEnd,
                   proc -> getAliasedName(name, currNamespaces),
                   NONEMPTY_ELEMENT );
        }; break;
    default:
        assert(!"eventElementEnd");
    };

    // send the event to SAX
    Str* theSAXname = nameForSAX(name);
    IF_SAX2( endElement, (const char*) *theSAXname );
    delete theSAXname;
    // report namespace scope ends
    while (currNamespaces.number() > getFirstOwnNS())
    {
        IF_SAX2( endNamespace, currNamespaces.last() -> prefix );
        currNamespaces.freelast(FALSE);
    }

    // pop one history level
    history.freelast(FALSE);
    if (getLevel())
        state = STATE_IN_ELEMENT;
    else
        state = STATE_OUTSIDE;

    return OK;
}


eFlag OutputterObj::eventAttributeStart(const QName& name)
{
    switch(state)
    {
    case STATE_IN_MARKUP:
        {
            state = STATE_IN_ATTRIBUTE;
            currAttName = name;
        }; break;
    case STATE_IN_ELEMENT:
        Err1(proc -> situation, E1_ATTRIBUTE_TOO_LATE, name.getname());
        break;
    case STATE_OUTSIDE:
        Err1(proc -> situation, E1_ATTRIBUTE_OUTSIDE, name.getname());
        break;
    default:
        assert(!"eventAttributeStart");
    };
    return OK;
}


eFlag OutputterObj::eventAttributeEnd()
{
    assert(state == STATE_IN_ATTRIBUTE);
    if (currAtts.find(currAttName))
        Err1(proc -> situation, E1_ADDING_DUPLICIT_ATTRIBUTE, currAttName.getname());
    currAtts.appendConstruct(currAttName, currData);
    currData.empty();
    state = STATE_IN_MARKUP;
    return OK;
}


eFlag OutputterObj::eventCommentStart()
{
    switch(state)
    {
    case STATE_IN_MARKUP:
        E( reportStartTag(NONEMPTY_ELEMENT) );
        // no break
    case STATE_OUTSIDE:
    case STATE_IN_ELEMENT:
        {
            E( reportCurrData() );
            state = STATE_IN_COMMENT;
        }; break;
    default:
        assert(!"eventCommentStart");
    };
    return OK;
}

eFlag OutputterObj::eventCommentEnd()
{
    assert(state == STATE_IN_COMMENT);
    if (physical && method == OUTPUT_UNKNOWN)
        E( front.appendConstruct(FM_COMMENT, currData, theEmptyString, FALSE) )
    else
    {
        IF_PH2( outputComment, currData );
        IF_SAX2( comment, currData );
    }
    currData.empty();
    state = (getLevel()? STATE_IN_ELEMENT : STATE_OUTSIDE);
    return OK;
}

eFlag OutputterObj::eventPIStart(const Str& name)
{
    switch(state)
    {
    case STATE_IN_MARKUP:
        E( reportStartTag(NONEMPTY_ELEMENT) );
        // no break
    case STATE_IN_ELEMENT:
    case STATE_OUTSIDE:
        {
            E( reportCurrData() );
            state = STATE_IN_PI;
            currPIName = name;
        }; break;
    default:
        assert(!"eventPIStart");
    };
    return OK;
}

eFlag OutputterObj::eventPIEnd()
{
    assert(state == STATE_IN_PI);
    if (physical && method == OUTPUT_UNKNOWN)
        E( front.appendConstruct(FM_PI, currPIName, currData, FALSE) )
    else
    {
        IF_PH3( outputPI, currPIName, currData );
        IF_SAX3( processingInstruction, currPIName, currData );
    }
    currData.empty();
    currPIName.empty();
    state = (getLevel()? STATE_IN_ELEMENT : STATE_OUTSIDE);
    return OK;
}

eFlag OutputterObj::eventNamespace(const Str& prefix, const Str& uri)
{
    assert(state == STATE_IN_MARKUP);
    Str *existing_namespace = currNamespaces.getUri(prefix);
    if (!existing_namespace || !(*existing_namespace == uri))
        currNamespaces.appendConstruct(prefix, uri);
    return OK;
}

eFlag OutputterObj::eventDisableEscapingForNext()
{
    if (method != OUTPUT_TEXT)
    {
        switch(state)
        {
        case STATE_IN_ATTRIBUTE:
        case STATE_IN_PI:
        case STATE_IN_COMMENT:
            Warn(proc -> situation, W_DISABLE_OUTPUT_ESC);
        default:
            outputEscaping = FALSE;
        };
    }
    return OK;
}

eFlag OutputterObj::eventData(const Str& data)
{
    if (physical && method == OUTPUT_UNKNOWN && state == STATE_OUTSIDE)
    {
        E( front.appendConstruct(FM_TEXT, data, theEmptyString, !outputEscaping) );
        if (!isAllWhite((const char*) data))
        {
            E( physical -> setMethodByDefault(method = OUTPUT_XML) );
            E( reportXMLDeclIfMust() );
            E( reportFront() );
        }
        return OK;
    }

    switch(state)
    {
    case STATE_IN_MARKUP:
        E( reportStartTag(NONEMPTY_ELEMENT) );
        // no break
    case STATE_IN_ELEMENT:
    case STATE_OUTSIDE:
        {   // this is a text node
            if (!(getFlags() & CDATA_SECTION))
            {
                int htmlScriptOrStyle = 
                    ( method == OUTPUT_HTML && ( getFlags() & HTML_SCRIPT_OR_STYLE ));
                if (physical)
                    E( physical -> outputText(data, !outputEscaping, htmlScriptOrStyle) );
            }
            // reset outputEscaping to default after use
            // even if we just save a part of cdata section
            outputEscaping = TRUE;
            state = getLevel()? STATE_IN_ELEMENT : STATE_OUTSIDE;
        }   // no break
    case STATE_IN_COMMENT:
    case STATE_IN_PI:
    case STATE_IN_ATTRIBUTE:
        {
            currData += data;
        }; break;
    default:
        assert(!"eventData()");
    }
    return OK;
}

eFlag OutputterObj::eventEndOutput()
{
    assert(state == STATE_OUTSIDE);
    E( reportCurrData() );
    if (physical && method == OUTPUT_UNKNOWN)
    {
        E( physical -> setMethodByDefault(method = OUTPUT_XML) );
        E( reportXMLDeclIfMust() );
        E( reportFront() );
    }
    IF_PH1( outputDone );
    IF_SAX1( endDocument );
    state = STATE_DONE;

    return OK;
}

eFlag OutputterObj::reportCurrData()
{
    if (currData.isEmpty())
        return OK;
    switch(state)
    {
    case STATE_OUTSIDE:
    case STATE_IN_MARKUP:
    case STATE_IN_ELEMENT:
        {
            if (getFlags() & CDATA_SECTION)
            {
                // we might now call the SAX startCData handler (if there was one)
                IF_SAX3( characters, currData, currData.length() );
                IF_PH2( outputCDataSection, currData );
                // we might call the SAX endCData handler
            }
            else
            {
                // the text had been output to physical in parts
                IF_SAX3( characters, currData, currData.length());
            }
        }; break;
    default:
        assert(!"reportCurrData()");
    }
    currData.empty();
    return OK;
}


eFlag OutputterObj::reportStartTag(Bool isEmpty)
{
    if (state == STATE_OUTSIDE || currElement.isEmpty())
        return OK;
    if (physical)
       E( physical -> outputElementStart(
       proc -> getAliasedName(currElement, currNamespaces),
       currNamespaces, getFirstOwnNS(), currAtts, isEmpty) );
    if (mySAXHandler)
    {
        const char**
            attsTable = new const char*[(currAtts.number() << 1) + 1];
        int i;

        // report only the new namespaces
        for (i = getFirstOwnNS(); i < currNamespaces.number(); i++)
        {
            IF_SAX3( startNamespace, currNamespaces[i] -> prefix, 
                currNamespaces[i] -> uri );
        }

        // create the attribute table
        PList<Str*> stringsForDeletion;
        for (i = 0; i < currAtts.number(); i++)
        {
            Str *expatName = nameForSAX(currAtts[i] -> key);
            stringsForDeletion.append(expatName);
            attsTable[i << 1] = *expatName;
            attsTable[(i << 1) + 1] = currAtts[i] -> value;
        };
        attsTable[currAtts.number() << 1] = NULL;
        
        Str* theSAXname = nameForSAX(currElement); 
        stringsForDeletion.append(theSAXname);
        IF_SAX3(startElement, *theSAXname, attsTable);
        delete[] attsTable;
        stringsForDeletion.freeall(FALSE);
    }

    if (method == OUTPUT_HTML && noHeadYet && 
        currElement.getUri().isEmpty() && currElement.getLocal().eqNoCase("head"))
        E( throwInMeta() );

    currElement.empty();
    currAtts.freeall(FALSE);
    currData.empty();
    return OK;
}

void OutputterObj::pushLevel(const QName& name)
{
    currElement = name;
    OutputHistoryItem *newItem = new OutputHistoryItem;
    if (history.number())
        *newItem = *(history.last());
    else
        newItem -> flags = 0;
    if (physical)
    {
        if (outDef -> askQNameList(XSLA_CDATA_SECT_ELEMS, name))
            newItem -> flags |= CDATA_SECTION;
        else
            newItem -> flags &= ~CDATA_SECTION;
    };
    newItem -> firstOwnNS = currNamespaces.number();
    history.append(newItem);
    state = STATE_IN_MARKUP;
}

